/*
    "texture.cpp"   - OpenGL texture interface class
     
    DDS - Dureks DemoSystem
    Copyright (C)2001 dureks

    This source code is licensed under the GNU GPL.
    See the GNU General Public License for more details.
*/

#include <fstream.h>
#include "texture.h"
#include "glbase.h"
#include "surface.h"
#include "error.h"

namespace dds {

    /*  Texture class
        ------------- */

Texture::Texture(BaseSurface *texture, textureMip mipmap)
{
    if( glMaxTextureSize == 0 )
        throw Error("Texture::Texture()","Renderer not initialized");

    m_texture = texture;
    m_mipmap = mipmap;

    // avoid 8bpp and 16bpp textures (unsupported by OpenGL, I think...but what 'bout 3Dfx...?)
    if( m_texture->format().depth() < 24 )
        throw Error("Texture::Texture()","<24bpp textures are not supported");

    if( m_texture->format().pitch() == 4 )
        gl_format = GL_RGBA;
    else
        gl_format = GL_RGB;

    // scale texture if too large
    // FIX: this should probably check image ratio, e.g 630x360 -> 512x256)
    if( m_texture->width() > (int32)glMaxTextureSize ) {
        Surface *scaled = (Surface*)scale(m_texture,glMaxTextureSize,glMaxTextureSize);
        delete m_texture;
        m_texture = scaled;
    }

    // grab a free texture slot
    glGenTextures(1,&gl_texture_id);

    // use "best" options (according to OpenGL guides, that is)
    wrap(GL_CLAMP,GL_CLAMP);
    switch(m_mipmap) {
        case nomip:
            filter(GL_LINEAR,GL_LINEAR);
            break;
        case fastmip:
            filter(GL_NEAREST_MIPMAP_LINEAR,GL_NEAREST);
            break;
        case nicemip:
            filter(GL_LINEAR_MIPMAP_NEAREST,GL_LINEAR);
            break;
    }
}

Texture::~Texture()
{
    // free texture slot
    glDeleteTextures(1,&gl_texture_id);

    // this was just assigned to the texture
    m_texture = NULL;
}

void Texture::upload()
{
    try {
        glBindTexture(GL_TEXTURE_2D, gl_texture_id);    // select our texture slot
        glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_WRAP_S,gl_texture_wrap_s);
        glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_WRAP_T,gl_texture_wrap_t);
        glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER,gl_texture_min_filter);
        glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER,gl_texture_mag_filter);

        // store gl attributes
        GLint p_align,up_align;
        glGetIntegerv(GL_PACK_ALIGNMENT,&p_align);
        glGetIntegerv(GL_UNPACK_ALIGNMENT,&up_align);
        glPixelStorei(GL_PACK_ALIGNMENT,1);
        glPixelStorei(GL_UNPACK_ALIGNMENT,1);

        // upload texture/mipmaps
        if( m_mipmap == nomip ) {
            glTexImage2D(GL_TEXTURE_2D,0,m_texture->format().pitch(),(GLsizei)m_texture->width(),(GLsizei)m_texture->height(),0,gl_format,GL_UNSIGNED_BYTE,(char8*)m_texture->lock());
            m_texture->unlock();
        }
        if( m_mipmap == fastmip ) {
            glTexImage2D(GL_TEXTURE_2D,0,m_texture->format().pitch(),(GLsizei)m_texture->width(),(GLsizei)m_texture->height(),0,gl_format,GL_UNSIGNED_BYTE,(char8*)m_texture->lock());
            m_texture->unlock();
            buildmipmaps();
        }
        if( m_mipmap == nicemip ) {
            glTexImage2D(GL_TEXTURE_2D,0,m_texture->format().pitch(),(GLsizei)m_texture->width(),(GLsizei)m_texture->height(),0,gl_format,GL_UNSIGNED_BYTE,(char8*)m_texture->lock());
            m_texture->unlock();
            buildmipmaps();
        }

        glPixelStorei(GL_PACK_ALIGNMENT,p_align);
        glPixelStorei(GL_UNPACK_ALIGNMENT,up_align);
    }
    catch( Error &error ) {
        throw Error("Texture::upload()",error);
    }
}

void Texture::bind(int32 stage)
{
#ifdef GL_ARB_multitexture
    if( (stage < (int32)glMultiTexStages) && glActiveTextureARB_ext )
        glActiveTextureARB_ext((GLenum)(GL_TEXTURE0_ARB+stage));
#endif
    glBindTexture(GL_TEXTURE_2D,gl_texture_id);
}

void Texture::buildmipmaps()
{
    // FIX: this function is somewhat badly written (but works:)
    try {
        int level = 1;
        int p = m_texture->format().pitch();
        int w = m_texture->width();
        int h = m_texture->height();
        bool smooth = (bool)(m_mipmap == nicemip);
        Surface *prev = new Surface(w,h,m_texture->format(),SDL_SWSURFACE);
        prev->copy(m_texture);
        while( (w > 1) && (h > 1) ) {
            if( w > 1) w /= 2;
            if( h > 1) h /= 2;
            Surface *temp = (Surface*)scale(prev,w,h,smooth);

            delete prev;
            prev = temp;
            glTexImage2D(GL_TEXTURE_2D,level++,p,w,h,0,gl_format,
                         GL_UNSIGNED_BYTE,(char8*)prev->lock());
            prev->unlock();
        }
        delete prev;
    }
    catch( Error &error ) {
        throw Error("Texture::buildmipmaps()",error);
    }
}

BaseSurface* Texture::scale(BaseSurface *surface, int w, int h, bool smooth)
{
    int s_w = surface->width();
    int s_h = surface->height();
    int pitch = surface->format().pitch();
    float x_f = (float)(s_w/w);
    float y_f = (float)(s_h/h);

    Surface *scaled = new Surface(w,h,surface->format());

    // TODO: optimize this code (a_lot!)
    char8 *s_d = (char8*)surface->lock();
    char8 *d_d = (char8*)scaled->lock();
    for(int y=0;y<h;y++) {
        char8 *s_t = s_d+(int32)(y*s_w*y_f*pitch);
        for(int x=0;x<w;x++) {
            char8 *s_c = s_t+(int32)(x*x_f*pitch);
            for(int c=0;c<pitch;c++) {
                if( smooth == true )    // Texture::nicemip
                    *(d_d+c) = (*(s_c+c)
                               +*(s_c+c+pitch)
                               +*(s_c+c+pitch+s_w*pitch)
                               +*(s_c+c+s_w*pitch))>>2;
                else    // Texture::fastmip
                    *(d_d+c) = *(s_c+c);
            }
            d_d += pitch;
        }
    }

    scaled->unlock();
    surface->unlock();

    return(scaled);
}


}   // namespace dds