//############################################################################
// Definet ###################################################################
//############################################################################

// Ruutu
#define	SCREEN_W	800
#define	SCREEN_H	600
#define	SCREEN_B	32

// Aika
#define TIMER_FPS	60
#define TIMER_MS	(Uint32)(1000.0f / (float)TIMER_FPS)

//#define FLAG_FULLSCREEN
#define SDL_GFX_FLAGS	(SDL_HWSURFACE | SDL_OPENGL | SDL_HWSURFACE)
#ifdef FLAG_FULLSCREEN
#define SDL_GFX_FLAGS	(SDL_GFX_FLAGS | SDL_FULLSCREEN)
#endif
//#define REMOVE_STDLIB

//############################################################################
// Includet ##################################################################
//############################################################################

// Posix
#include <math.h>

// Windows
#ifdef FLAG_WINDOWS
#include "windows.h"
#endif

// 3rd party
#include "SDL.h"
#include "GL/gl.h"
#include "GL/glu.h"

#ifndef M_PI
#define M_PI 3.14159265358979323846f
#endif

//############################################################################
// Typedefit #################################################################
//############################################################################

/*
 * Katsomissuunnat.
 */

typedef struct
{
  int time;
  Sint16 data[6];
} LookAtUnit;

/*
 * Värit.
 */

typedef struct
{
  float f[4];
  Uint32 u;
} ColorUnit;

/*
 * Muodot.
 */

#define ARRAY_HEAP		10000

typedef struct
{
  float tex[2];
  Uint32 col;
  float ver[3];
} ShapeUnit;

typedef struct
{
  ShapeUnit data[ARRAY_HEAP];
  Uint32 index[ARRAY_HEAP];
  int ins_vertex, ins_index;
} ShapeDB;

/*
 * Merkit.
 */

#define CHAR_MAX_STROKES	10

typedef struct
{
  Sint8 stroke[CHAR_MAX_STROKES];
} Char2D;

/*
 * Bezier.
 */

#define BEZIER_HISTORY	5

typedef struct
{
  Sint16 pos[BEZIER_HISTORY * 3];
} BezierControl;

typedef struct
{
  int d[3], u[3];
} BezierLine;

/*
 * Valopilarit.
 */

typedef struct
{
  float height, width, radius;
  int lifetime, delay, col;
} LightPillarParams;

/*
 * Circles.
 */

typedef struct
{
  int radial, sector, dir, stime, etime;
} CirclesRunner;

/*
 * GitS.
 */

#define GITS_LINKSPER	5
#define GITS_PATH_LEN	2000

typedef struct
{
  float pos[3];
  int connections[GITS_LINKSPER], conncnt;
} GitsSphere;

// Polku
typedef struct
{
  float pos[3 * GITS_PATH_LEN], flen;
  int len;
} GitsPath;

// Efektit
typedef struct
{
  int start, end;
  void (*func)(int);
} EffectData;

//############################################################################
// Global variables for fun & profit #########################################
//############################################################################

float origo[3] = { 0.0f, 0.0f, 0.0f },
      camera_position[6];
ShapeDB shape_use;

//############################################################################
// Timer #####################################################################
//############################################################################

int var_timer;

Uint32 timer_func(Uint32 interval)
{
  var_timer++;
  return interval;
}

//############################################################################
// Helper funcs ##############################################################
//############################################################################

int modulo(int val, int base)
{
  if(val < 0)
    val += base;
  else if(val >= base)
    val -= base;
  return val;
}

void memclr(void *op, size_t bytes)
{
  Uint32 *dst = (Uint32*)op;
  for(; bytes; bytes -= 4, dst++)
    *dst = 0;
}

/*
 * Kahden pisteen etäisyys ilman neliöjuurta.
 */
float dist_sqr(float *a, float *b)
{
  float dx = a[0] - b[0],
	dy = a[1] - b[1],
	dz = a[2] - b[2];
  return dx * dx + dy * dy + dz * dz;
}

/*
 * Kahden pisteen etäisyys neliöjuuren kanssa.
 */
float dist_root(float *a, float *b)
{
  return sqrt(dist_sqr(a, b));
}

/*
 * Kaksiulotteinen etäisyys kokonaisluvuille.
 */
float dist_2d(int x1, int y1, int x2, int y2)
{
  int dx = x2 - x1,
      dy = y2 - y1;
  return sqrt((float)(dx * dx + dy * dy));
}

/*
 * Vaihtaa kaksi flotaria keskenään.
 */
void xchgf(float *a, float *b)
{
  float c = *a;
  *a = *b;
  *b = c;
}

/*
 * Rotatoi kolmen akselin yli.
 */
void glRotate3f(float x, float y, float z)
{
  glRotatef(x, 1.0f, 0.0f, 0.0f);
  glRotatef(y, 0.0f, 1.0f, 0.0f);
  glRotatef(z, 0.0f, 0.0f, 1.0f);
}

//############################################################################
// Kamera ####################################################################
//############################################################################

#define CAMERA_BACKCLIP		5000.0f

void camera_frustum(float fov)
{
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  gluPerspective(fov * (float)SCREEN_H / (float)SCREEN_W,
      (float)SCREEN_W / (float)SCREEN_H, 1.0, CAMERA_BACKCLIP);
  glMatrixMode(GL_MODELVIEW);
}

void camera_ortho(float fov)
{
  float ratio = (float)SCREEN_H / (float)SCREEN_W;
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  gluOrtho2D(-fov, fov, -fov * ratio, fov * ratio);
  glMatrixMode(GL_MODELVIEW);
}

void blend_mode(int op)
{
  if(op == 0)
    glBlendFunc(GL_ONE_MINUS_SRC_ALPHA, GL_ONE);
  else
    glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
}

//############################################################################
// Interpolation #############################################################
//############################################################################

/*
 * All interpolation functions here.
 */

void interpolate_linear_ff(float* target, float* from, float* to, float pos,
    int count)
{
  do
  {
    target[count] = (to[count] - from[count]) * pos + from[count];
  } while(count--);
}

void interpolate_linear_i16f(float* target, Sint16* from, Sint16* to,
    float pos, int count)
{
  do
  {
    target[count] = ((float)(to[count] - from[count])) * pos +
      (float)(from[count]);
  } while(count--);
}

float interpolate_sine(float base, float var, float currtime, float speed)
{
  return base + var * sin((float)currtime * speed);
}

float interpolate_count_se(int freeze, int start, int end)
{
  float pos = (float)(freeze - start) / (float)(end - start);
  float pos2 = 1.0f - fabs(0.5f - pos) * 2.0f;
  pos2 *= pos2 * 0.5f;
  if(pos < 0.5f)
    return pos2;
  return 1.0f - pos2;
}

float interpolate_count_lt(int freeze, int start, int life)
{
  return (float)(freeze - start) / (float)life;
}

//############################################################################
// Random ####################################################################
//############################################################################

/*
 * Random number generator: x[n + 1] = a * x[n] mod m
 * Made according to source by Glenn C Rhoads.
 * http://remus.rutgers.edu/~rhoads/Code/code.html
 */

static Uint32 var_rand_seed = 93186752, var_rand_a = 1588635695,
	      var_rand_q = 2, var_rand_r = 1117695901;

// Initialize random number generator
void rand_seed(int op)
{
  var_rand_seed = op;
}

// Random natural number [0, param[
int rand_int(int op)
{
  var_rand_seed = var_rand_a * (var_rand_seed % var_rand_q) -
    var_rand_r * (var_rand_seed / var_rand_q);
  return var_rand_seed % op;
}

// Random real number [0, number]
float rand_float(float op)
{
  return (float)(rand_int(32768)) / 32767.0f * op;
}

// Random real number [-number, number]
float rand_float_neg(float op)
{
  return (float)(rand_int(65534) - 32767) / 32767.0f * op;
}

//############################################################################
// Kontrolli #################################################################
//############################################################################

// Kusin joten tässä
#define MAX_EFFECT_TIME	15000

// Circles
#define CIRCLES_START	0
#define PILLAR_START	(CIRCLES_START + 3000)
#define PILLAR_DUR	1900
#define WINGS_START	(PILLAR_START + 600)
#define WINGS_END	(WINGS_START + 1000)
#define PILLAR_END	(PILLAR_START + PILLAR_DUR)
#define ORITEXT_START	(PILLAR_END + 100)
#define CIRCLES_END	(CIRCLES_START + 5300)
#define ORITEXT_END	CIRCLES_END
// GitS
#define GITS_START	CIRCLES_END
#define GITS_END	(GITS_START + 4700)
// City
#define CITY_START	GITS_END
#define CITY_END	(CITY_START + 6000)
// Loppu
#define TIME_END	GITS_END

LookAtUnit data_look_at_buf[50] =
{
  // Circles
  { CIRCLES_START, { 10, 30, 0, 50, 30, 0 } },
  { CIRCLES_START + 300, { 5, 40, 20, -10, 30, -40 } },
  { CIRCLES_START + 550, { -5, 80, 10, -50, 50, 10 } },
  { CIRCLES_START + 800, { 5, 140, -20, 10, 40, 40 } },
  { CIRCLES_START + 1200, { -45, 160, -9, 40, 60, 5 } },
  { CIRCLES_START + 1500, { 5, 180, -9, -30, 10, -10 } },
  { CIRCLES_START + 1700, { 5, 180, -30, -10, 20, -20 } },
  { CIRCLES_START + 2200, { 10, 190, 40, 0, 20, 0 } },
  { CIRCLES_START + 2400, { -5, 190, 30, 15, 20, -10 } },
  { PILLAR_START, { -40, 150, 190, 0, 50, 0 } },
  { WINGS_START, { -60, 140, 220, 0, 80, 0 } },
  { WINGS_END, { -120, 90, 350, 0, 130, 0 } },
  { PILLAR_END, { -80, 150, 110, 0, 50, 0 } },
  { CIRCLES_END, { -80, 130, 110, 0, 200, 0 } },
  // GitS
  { GITS_START, { -1020, 750, 7000, -1010, 750, -1000 } },
  { GITS_START + 180, { -1020, 750, 1510, -1010, 750, -1000 } },
  { GITS_START + 600, { -520, 750, 1510, -520, 750, -1000 } },
  { GITS_START + 670, { -80, 1220, 330, 10000, 750, 0 } },
  { GITS_START + 800, { -120, 580, 900, 1000, 450, 1400 } },
  { GITS_START + 850, { 420, 480, 1430, 500, 700, 500 } },
  { GITS_START + 920, { 430, 560, 1230, 1000, 700, 1000 } },
  { GITS_START + 1040, { 450, 800, 1030, 1000, 700, 1000 } },
  { GITS_START + 1240, { 640, 2000, 940, 1000, 1600, 700 } },
  { GITS_START + 1420, { 940, 1100, 1200, 2000, 1000, 700 } },
  { GITS_START + 1570, { 1520, 1250, 1180, 2000, 1500, 700 } },
  { GITS_START + 1660, { 1520, 1350, 540, 3000, 1100, 1200 } },
  { GITS_START + 2300, { 1530, 1150, 640, 3000, 1200, 1200 } },
  { GITS_START + 2400, { 2000, 940, 1200, 3000, 1100, 1200 } },
  { GITS_START + 2660, { 3000, 1150, 800, 4000, 800, 1100 } },
  { GITS_START + 2770, { 3010, 1290, 880, 4000, 1200, 1200 } },
  { GITS_START + 2890, { 2900, 750, 800, 12000, 900, 900 } },
  { GITS_START + 3900, { 12000, 750, 700, 14000, 700, 750 } },
  // City
  { CITY_START, { 0, 50, 350, 0, 90, 0 }, },
  { CITY_END, { 0, 50, 350, 0, 90, 0 } }
  /*{ CIRCLES_START, { 0, 180, 20, 0, 0, 0 } },
  { TIME_END, { 0, 180, 20, 0, 0, 0 } }*/
};

//############################################################################
// LookAt ####################################################################
//############################################################################

void look_at(void)
{
  LookAtUnit *to = &data_look_at_buf[1], *from = &data_look_at_buf[0];

  while(!((to->time > var_timer) && (var_timer >= from->time)))
  {
    to++;
    from++;
  }

  // Laske arvot
  interpolate_linear_i16f(camera_position, from->data, to->data,
      interpolate_count_se(var_timer, from->time, to->time), 5);

  // Katso
  gluLookAt(camera_position[0], camera_position[1], camera_position[2],
      camera_position[3], camera_position[4], camera_position[5],
      0.0, 1.0, 0.0);

}

//############################################################################
// Värit #####################################################################
//############################################################################

#define COLORDB_SIZE	150
#define FLASH_COUNT	1
#define FLASH_DURATION	100

ColorUnit var_colordb[COLORDB_SIZE] =
{
  { { 0.0f, 0.0f, 0.0f, 0.0f }, 0x00000000 },
  { { 0.0f, 0.0f, 0.0f, 0.0f }, 0x00003248 },
  { { 0.0f, 0.0f, 0.0f, 0.0f }, 0x00FFFFFF },
  { { 0.0f, 0.0f, 0.0f, 0.0f }, 0xF0FFFFFF },
  { { 0.0f, 0.0f, 0.0f, 0.0f }, 0xF0888888 },
  { { 0.0f, 0.0f, 0.0f, 0.0f }, 0xFFC05050 },
  { { 0.0f, 0.0f, 0.0f, 0.0f }, 0x995555FF },
  { { 0.0f, 0.0f, 0.0f, 0.0f }, 0xFF00FFFF },
  { { 0.0f, 0.0f, 0.0f, 0.0f }, 0xA010C020 },
  { { 0.0f, 0.0f, 0.0f, 0.0f }, 0x99A0C0FF },
  { { 0.0f, 0.0f, 0.0f, 0.0f }, 0x44FF4444 },
  { { 0.0f, 0.0f, 0.0f, 0.0f }, 0x882222FF },
  { { 0.0f, 0.0f, 0.0f, 0.0f }, 0x80000000 },
};

// Vapaat värit
int var_colordb_free = 20;

int var_flash[FLASH_COUNT * 2] =
{
  WINGS_END, 2
};

/*
 * Palauta väri (float)
 */
float* colordb_get_float(int op)
{
  return (var_colordb + op)->f;
}

/*
 * Palauta väri (uint).
 */
Uint32 colordb_get_uint(int op)
{
  return (var_colordb + op)->u;
}

/*
 * Konvertoi Uint-väri float-väriksi.
 */
void colordb_convert_u2f(int op)
{
  ColorUnit *cu = var_colordb + op;
  cu->f[0] = (float)(cu->u & 0x000000FF) / 255.0f;
  cu->f[1] = (float)((cu->u & 0x0000FF00) >> 8) / 255.0f;
  cu->f[2] = (float)((cu->u & 0x00FF0000) >> 16) / 255.0f;
  cu->f[3] = (float)((cu->u & 0xFF000000) >> 24) / 255.0f;
}

/*
 * Konvertoi float-väri Uint-väriksi.
 */
void colordb_convert_f2u(int op)
{
  ColorUnit *cu = var_colordb + op;
  cu->u = (Uint32)(cu->f[0] * 255.0f) |
    (((Uint32)(cu->f[1] * 255.0f)) << 8) |
    (((Uint32)(cu->f[2] * 255.0f)) << 16) |
    (((Uint32)(cu->f[3] * 255.0f)) << 24);
}

/*
 * ClearColor uusiksi.
 */
void glClearColorfv(float *op)
{
  glClearColor(op[0], op[1], op[2], op[3]);
}

/*
 * Tyhjennä GL-ruutu.
 */
void clearGL(int op)
{
  float *col = colordb_get_float(op);

  // Tsekataan väläys
  int i = 0, freezetime = var_timer;
  do
  {
    int flash = var_flash[i * 2];
    if((freezetime >= flash) && (freezetime < FLASH_DURATION + flash))
    {
      float intcol[4],
	    pos = interpolate_count_lt(freezetime, flash, FLASH_DURATION);
      interpolate_linear_ff(intcol, colordb_get_float(var_flash[i * 2 + 1]),
	  col, pos, 3);
      glClearColorfv(intcol);
      glClear(GL_DEPTH_BUFFER_BIT | GL_COLOR_BUFFER_BIT);
      return;
    }
  } while(i--);

  glClearColorfv(col);
  glClear(GL_DEPTH_BUFFER_BIT | GL_COLOR_BUFFER_BIT);
}

//############################################################################
// Shapet ####################################################################
//############################################################################

/*
 * Clear a shape database.
 */
void shapedb_clear(ShapeDB *op)
{
  op->ins_vertex = op->ins_index = 0;
}

/*
 * Functions to select the array pointers in databases.
 */
void shapedb_draw_elements(ShapeDB *op, int t, int element)
{
  // Specify the interleaevd array

  glInterleavedArrays(GL_T2F_C4UB_V3F, 0, op->data);

  // Tekstuuri päälle / pois
  if(t >= 0)
  {
    glEnable(GL_TEXTURE_2D);
    glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);
    glDisableClientState(GL_COLOR_ARRAY);
    glBindTexture(GL_TEXTURE_2D, t);
  }
  else
  {
    glDisable(GL_TEXTURE_2D);
  }

  // Remove later
#ifndef REMOVE_STDLIB
  if((op->ins_index >= ARRAY_HEAP) || (op->ins_vertex >= ARRAY_HEAP))
  {
    printf("ERROR: too large object.\n");
  }
#endif

  // Draw the elements
  glDrawElements(element, op->ins_index, GL_UNSIGNED_INT, op->index);
}

/*
 * Add a vertex and a color to a shape.
 */
void shapedb_add_cv(ShapeDB *op, float x, float y, float z, int c)
{
  // Shift the arrays
  ShapeUnit *dst = op->data + (op->ins_vertex++);
  // Vertex
  dst->ver[0] = x;
  dst->ver[1] = y;
  dst->ver[2] = z;
  dst->col = colordb_get_uint(c);
}

/*
 * Add a vertex and a texcoord to a shape.
 */
void shapedb_add_tv(ShapeDB *op, float x, float y, float z, float t,
    float u)
{
  // Shift the arrays
  ShapeUnit *dst = op->data + (op->ins_vertex++);
  // Vertex
  dst->ver[0] = x;
  dst->ver[1] = y;
  dst->ver[2] = z;
  // Texcoord
  dst->tex[0] = t;
  dst->tex[1] = u;
}

/*
 * Add a vertex that is an interpolation between two other vertices.
 */
void shapedb_interpolate_v(ShapeDB *op, int from, int to, float pos,
    Uint32 col)
{
  ShapeUnit *dst = op->data + (op->ins_vertex++);
  // Interpoloi
  interpolate_linear_ff(dst->ver, op->data[from].ver, op->data[to].ver, pos,
      3);
  // Color
  dst->col = colordb_get_uint(col);
}

/*
 * Insert an index to the array.
 */
void shapedb_add_index(ShapeDB *op, Uint32 idx)
{
  op->index[op->ins_index++] = idx;
}

/*
 * Return a certain vertex.
 */
float* shapedb_get_vertex(ShapeDB *op, int idx)
{
  return op->data[idx].ver;
}

/*
 * Tulosta shapedb.
 */
#ifndef REMOVE_STDLIB
void shapedb_print(ShapeDB *op)
{
  int i;
  for(i = 0; i < op->ins_vertex; i++)
  {
    ShapeUnit *dst = &(op->data[i]);
    printf("<%f, %f, %f> [%f, %f]: %x\n", dst->ver[0],
	dst->ver[1], dst->ver[2], dst->tex[0], dst->tex[1], dst->col);
  }
  for(i = 0; i < op->ins_index; i++)
    printf("%i, ", op->index[i]);
  printf("\r\r\n");
}
#endif

//############################################################################
// Textout ###################################################################
//############################################################################

// Fontti
Char2D var_characters[50] =
{
  { { 12, 103, 18, -1, 46, 50, -1, -1, -1, -1 } }, // A
  { { 100, 12, -1, 101, 82, 68, -1, 57, 38, 13 } }, // B
  { { 84, 104, 102, 78, 34, 14, 16, 40, -1, -1 } }, // C
  { { 100, 12, -1, 101, 60, 13, -1, -1, -1, -1 } }, // D
  { { 57, 59, 83, 103, 101, 89, 34, 14, 16, 28 } }, // E
  { { 100, 12, -1, 101, 104, -1, 57, 60, -1, -1 } }, // F
  { { 84, 104, 102, 78, 34, 14, 16, 40, 51, 49 } }, // G
  { { 100, 12, -1, 105, 17, -1, 57, 60, -1, -1 } }, // H
  { { 100, 102, -1, 90, 24, -1, 12, 14, -1, -1 } }, // I
  { { 100, 102, -1, 90, 35, 15, 39, -1, -1, -1 } }, // J
  { { 100, 12, -1, 57, 82, -1, 46, 16, -1, -1 } }, // K
  { { 100, 102, -1, 90, 24, -1, 12, 16, 27, -1 } }, // L
  { { 12, 100, 70, 106, 18, -1, -1, -1, -1, -1 } }, // M
  { { 12, 100, 17, 105, -1, -1, -1, -1, -1, -1 } }, // N
  { { 14, 34, 78, 102, 104, 84, 40, 16, 14, -1 } }, // O
  { { 100, 12, -1, 101, 82, 57, -1, -1, -1, -1 } }, // P
  { { 103, 67, 70, -1, 104, 16, -1, 37, 39, -1 } }, // Q
  { { 12, 100, 83, 57, 17, -1, -1, -1, -1, -1 } }, // R
  { { 83, 102, 78, 39, 15, 34, -1, -1, -1, -1 } }, // S
  { { 100, 105, -1, 91, 14, -1, -1, -1, -1, -1 } }, // T
  { { 100, 34, 13, 15, 38, 104, -1, -1, -1, -1 } }, // U
  { { 100, 15, 106, -1, -1, -1, -1, -1, -1, -1 } }, // V
  { { 100, 13, 37, 17, 106, -1, -1, -1, -1, -1 } }, // W
};

// Stringit
char string_gfxcode[14] = { 7, 18, 1, 16, 8, 9, 3, 19, 24, 3, 15, 4, 5, 0 };
char string_muscode[11] = { 13, 21, 19, 9, 3, 24, 3, 15, 4, 5, 0 };
char string_other[11] = { 15, 20, 8, 5, 18, 24, 3, 15, 4, 5, 0 };
char string_greets[7] = { 7, 18, 5, 5, 20, 19, 0 };
char string_juippi[7] = { 10, 21, 9, 16, 16, 9, 0 };
char string_trilkk[7] = { 20, 18, 9, 12, 11, 11, 0 };
char string_warma[6] = { 23, 1, 18, 13, 1, 0 };
char string_vv[3] = { 22, 22, 0 };

/*
 * Piirrä yksittäinen merkki.
 */
float text_draw_char(char op, float *pos, float x, float y, float scalex,
    float scaley, int col)
{
  Char2D *chr = var_characters + (op - 1);

  shapedb_clear(&shape_use);

  int i, last = -1;
  float lastcrd[2], maxx = 0.0f;
  for(i = 0; (i < CHAR_MAX_STROKES) && (*pos > 0.0f); i++)
  {
    Sint8 crd = chr->stroke[i];
    float newcrd[2];
    
    // Jos terminaattori
    if(crd < 0)
    {
      if(last < 0)
	break;
    }
    else
    {
      // Laske uudet pisteet
      newcrd[0] = (float)(crd % 11) / 10.0f * scalex;
      newcrd[1] = (float)(crd / 11) / 10.0f * scaley;
      //printf("Cords: %f, %f\n", newcrd[0], newcrd[1]);
      if(newcrd[0] > maxx)
	maxx = newcrd[0];
      newcrd[0] += x;
      newcrd[1] += y;

      if(last >= 0)
      {
	if(*pos < 1.0f)
	  interpolate_linear_ff(newcrd, lastcrd, newcrd, *pos, 1);

	int j = 10;
	do
	{
	  float xdev = rand_float_neg(scalex / 11.0f),
		ydev = rand_float_neg(scalex / 11.0f);
	  shapedb_add_cv(&shape_use, newcrd[0] + xdev, newcrd[1] + ydev, 0.0f,
	      col);
	  shapedb_add_cv(&shape_use, lastcrd[0] + xdev, lastcrd[1] + ydev,
	      0.0f, col);
	  shapedb_add_index(&shape_use, shape_use.ins_vertex - 2);
	  shapedb_add_index(&shape_use, shape_use.ins_vertex - 1);
	} while(--j);

	// Vähennä kuljettavaa matkaa
	*pos -= 1.0f;
	if(*pos <= 0.0f)
	  break;
      }
    }

    last = crd;
    lastcrd[0] = newcrd[0];
    lastcrd[1] = newcrd[1];
  }

  shapedb_draw_elements(&shape_use, -1, GL_LINES);
  //printf("max: %f\n", maxx);
  return maxx + scalex * 0.3f;
}

/*
 * Piirrä kokonainen string.
 */
void text_draw_string(char *op, float x, float y, float scalex,
    float scaley, int col, float op_pos)
{
  float pos = op_pos;

  while(*op)
  {
    if(*op == 24) // space
      x += scalex;
    else
      x += text_draw_char(*op, &pos, x, y, scalex, scaley, col);
    op++;
  }
}

/*
 * Monimutkainen textout-utilisaatio.
 */
void textout_complex(char *string, int start, int end, int freeze, int appear,
    float speed, float x, float y1, float y2, float scale, int col)
{
  // Jos ollaan ulkopuolella altogether
  if((freeze <= start) || (freeze >= end))
    return;

  // Tila
  camera_ortho(1.0f);
  glMatrixMode(GL_MODELVIEW);
  glPopMatrix();
  glLoadIdentity();
  blend_mode(1);

  // Muuttujat
  float base = -1.0f;
  if(x < 0)
    base = 1.0f;

  // Jos se on tulossa vasta
  float offset = x;
  if(freeze - start < appear)
    offset *= interpolate_count_lt(freeze, start, appear);
  else if(end - freeze < appear)
    offset *= interpolate_count_lt(end, freeze, appear);

  // Piirto
  //printf("Pos: %f, X: %f, %f\n", pos, x, base + pos * x);
  shapedb_clear(&shape_use);
  shapedb_add_cv(&shape_use, base + offset, y2, 0.0f, 12);
  shapedb_add_cv(&shape_use, base + offset, y1, 0.0f, 12);
  shapedb_add_cv(&shape_use, base, y1, 0.0f, 12);
  shapedb_add_cv(&shape_use, base, y2, 0.0f, 12);

  /*shapedb_add_cv(&shape_use, base + pos * x, y1, 0.0f, 12);
  shapedb_add_cv(&shape_use, base + pos * x, y2, 0.0f, 12);
  shapedb_add_cv(&shape_use, base, y2, 0.0f, 12);
  shapedb_add_cv(&shape_use, base, y1, 0.0f, 12);*/
  shapedb_add_index(&shape_use, 0);
  shapedb_add_index(&shape_use, 1);
  shapedb_add_index(&shape_use, 2);
  shapedb_add_index(&shape_use, 3);
  shapedb_draw_elements(&shape_use, -1, GL_QUADS);

  float scalepad = scale / 5.0f;
  text_draw_string(string,
      (x > 0) ? (-1.0f + scalepad - x + offset) : (1.0f + offset + scalepad),
      y1 + scalepad, scale, scale, col, (freeze - start - appear) * speed);

}

//############################################################################
// Spline ####################################################################
//############################################################################

/*
 * Anna piste kolmen pisteen bezier-käyrältä, t:[0,1]
 */
void qbezier(float *dst, float *a, float *b, float *c, float t)
{
  float m = 1.0f - t;
  float m2 = m * m,
	t2 = t * t;

  int i = 2;
  do
  {
    dst[i] = m2 * a[i] + 2.0f * t * m * b[i] + t2 * c[i];
  } while(i--);
}

/*
 * Piirrä bspline.
 */
void efu_qbezier(BezierControl *bc, BezierLine *bl, int *ba, int col,
    int freezetime)
{
  int from = 0,
      to = 1;
  float d[3 * 3], u[3 * 3];

  while(1)
  {
    if((freezetime >= ba[from]) && (freezetime < ba[to]))
      break;
    from++;
    to++;
  }

  float pos = interpolate_count_se(freezetime, ba[from], ba[to]);

  from *= 3;
  to *= 3;
  int i = 2;
  do
  {
    Sint16 *dl = bc[bl->d[i]].pos,
	   *ul = bc[bl->u[i]].pos;
    interpolate_linear_i16f(d + i * 3, dl + from, dl + to, pos, 2);
    interpolate_linear_i16f(u + i * 3, ul + from, ul + to, pos, 2);
  } while(i--);

  shapedb_clear(&shape_use);
  i = 11;
  pos = 0.0f;
  do
  {
    float tgt[3];
    qbezier(tgt, d + 0, d + 3, d + 6, pos);
    shapedb_add_cv(&shape_use, tgt[0], tgt[1], tgt[2], col);
    qbezier(tgt, u + 0, u + 3, u + 6, pos);
    shapedb_add_cv(&shape_use, tgt[0], tgt[1], tgt[2], col);
    shapedb_add_index(&shape_use, shape_use.ins_vertex - 2);
    shapedb_add_index(&shape_use, shape_use.ins_vertex - 1);
    pos += 0.1f;
  } while(--i);

  // Piirrä
  shapedb_draw_elements(&shape_use, -1, GL_TRIANGLE_STRIP);
}

//############################################################################
// Siivet ####################################################################
//############################################################################

/*
 * Siipien pisteet.
 */
BezierControl var_wingc[9] = 
{
  { // 0
    { 0, 0, 0,
      0, 0, 0,
      0, 0, 0,
      0, 0, 0,
      0, 0, 0
    }
  },
  { // 1
    {
      0, 50, 0,
      5, 50, -10,
      10, 50, -10,
      15, 50, -5,
      50, 80, -0
    }
  },
  { // 2
    {
      0, 100, 0,
      10, 100, 0,
      15, 110, 0,
      25, 115, 0,
      75, 140, 0
    }
  },
  { // 3
    {
      0, 70, 0,
      30, 80, 10,
      40, 90, 10,
      60, 120, 5,
      160, 180, 0
    }
  },
  { // 4
    {
      0, 20, 0,
      45, 30, 15,
      75, 40, 15,
      130, 180, 5,
      220, 220, 0
    }
  },
  { // 5
    {
      0, 60, 0,
      55, 70, 10,
      70, 80, 10,
      90, 110, 5,
      230, 200, 0
    }
  },
  { // 6
    {
      0, 90, 0,
      35, 90, 0,
      40, 125, -5,
      50, 105, 0,
      90, 110, 0
    }
  },
  { // 7
    {
      0, 40, 0,
      30, 55, -10,
      35, 50, -10,
      40, 50, -5,
      60, 70, 0      
    }
  },
  { // 8
    {
      0, 0, 0,
      10, 0, 0,
      10, 0, 0,
      10, 0, 0,
      10, 0, 0
    }
  }
};

/*
 * Siipien bezier-käyrät.
 */
BezierLine var_wingl[2] =
{
  {
    { 0, 1, 2 },
    { 8, 7, 6 }
  },
  {
    { 2, 3, 4 },
    { 6, 5, 4 }
  }
};

/*
 * Siipien ajoitus.
 */
int var_wingt[BEZIER_HISTORY] =
{
  0,
  100,
  300,
  500,
  800
};

/*
 * Piirrä siipi.
 */
void efu_wing(int freezetime, float x, float y, float z)
{
  rand_seed(0xdeadbeef);
  int i = 20;
  do
  {
    glPushMatrix();
    glScalef(x + rand_float_neg(1.0f) / 30.0f,
	y + rand_float_neg(1.0f) / 30.0f, z + rand_float_neg(1.0f) / 30.0f);
    efu_qbezier(var_wingc, var_wingl, var_wingt, 3, freezetime);
    efu_qbezier(var_wingc, var_wingl + 1, var_wingt, 3, freezetime);
    glPopMatrix();
  } while(--i);	
}

/*
 * Siipiefekti.
 */
void efu_wings(int freezetime)
{
  glPushMatrix();
  glRotate3f(0.0f, -20.0f, 0.0f);
  efu_wing(freezetime, 0.4f, 1.9f, 1.0f);
  efu_wing(freezetime, 1.0f, 1.2f, 1.0f);
  efu_wing(freezetime, 1.6f, 0.5f, 1.0f);
  efu_wing(freezetime, -0.4f, 1.9f, 1.0f);
  efu_wing(freezetime, -1.0f, 1.2f, 1.0f);
  efu_wing(freezetime, -1.6f, 0.5f, 1.0f);
  glPopMatrix();
}

//############################################################################
// Valopilari ################################################################
//############################################################################

#define LIGHTPILLAR_CNT		5
#define LIGHTPILLAR_WIDTH	5.0f
#define LIGHTPILLAR_HEIGHT	300.0f
#define LIGHTPILLAR_ROT		0.03
#define LIGHTPILLAR_SHARD_COMP	9
#define LIGHTPILLAR_SHARD_ARC	(M_PI / 1.8f)

// Valopilarien data
ShapeDB shape_pillar_draw, shape_pillar_wings;
LightPillarParams pillar_draw = { 300.0f, 2.0f, 5.0f, 90, 5, 4 };
LightPillarParams pillar_wings = { 400.0f, 16.0f, 100.0f, 300, 2, 5 };

// Paskat muistista, generoidaan yksi joka framelle!
float var_pillar_shard[MAX_EFFECT_TIME * TIMER_MS];

/*
 * Generoi valopilarin shard.
 */
void generate_light_pillar_shard(ShapeDB *shape, LightPillarParams *params)
{
  // Clear the shape
  shapedb_clear(shape);

  // Generoi verteksit
  int i = LIGHTPILLAR_SHARD_COMP;
  do
  {
    // Common
    float percent = (float)i / (float)LIGHTPILLAR_SHARD_COMP;
    // Arc and height
    float ang = (LIGHTPILLAR_SHARD_ARC / 2.0f) - LIGHTPILLAR_SHARD_ARC *
	    ((float)i / (float)LIGHTPILLAR_SHARD_COMP),
	  mul = sin((float)(i + 1) / (float)(LIGHTPILLAR_SHARD_COMP + 2) *
	      M_PI),
	  ca = (cos(ang) - 1.0f),
	  sa = sin(ang);
    // Color
    float brightness_mul = sin(percent * M_PI);
    interpolate_linear_ff(colordb_get_float(var_colordb_free),
	colordb_get_float(0), colordb_get_float(params->col), brightness_mul,
	3);
    colordb_convert_f2u(var_colordb_free);
    // Add
    shapedb_add_cv(shape, ca, 0.0f, sa, var_colordb_free);
    shapedb_add_cv(shape, ca, mul, sa, 0);
    var_colordb_free++;
  } while(i--);

  // Generoi quadit
  i = LIGHTPILLAR_SHARD_COMP - 1;
  do
  {
    shapedb_add_index(shape, i * 2 + 0);
    shapedb_add_index(shape, i * 2 + 1);
    shapedb_add_index(shape, i * 2 + 3);
    shapedb_add_index(shape, i * 2 + 2);
  } while(i--);
}

/*
 * Piirtää yhden valopilarin ennalta annetuilla asetuksilla.
 */
void efu_light_pillar(float *pos, int stime, int etime, int freezetime,
    ShapeDB *shape, LightPillarParams *params)
{
  int i = stime;

  // Perutaanko suoraan?
  if(freezetime > etime + params->lifetime)
    return;

  // Piirretään
  for(i = stime; (i <= freezetime); i += params->delay)
    if((i + params->lifetime >= freezetime) && (i < etime))
    {
      float mul = interpolate_count_lt(freezetime, i, params->lifetime),
	    rad = mul * params->radius,
	    ang = var_pillar_shard[i],
	    dx = cos(ang) * rad,
	    dz = sin(ang) * rad;
      // Piirrä matriisitempuilla
      glPushMatrix();
      glTranslatef(dx + pos[0], pos[1], dz + pos[2]);
      glRotate3f(0.0f, ang / M_PI * -180.0f, 0.0f);
      glScalef(params->width, (1.0 - mul) * params->height, params->width);
      shapedb_draw_elements(shape, -1, GL_QUADS);
      glPopMatrix();
    }
}

//############################################################################
// Ympyrät ###################################################################
//############################################################################

#define CIRCLES_RADIALS		13
#define CIRCLES_SECTORS		48
#define CIRCLES_RADIUS		10.0f
#define CIRCLES_HEIGHT_BASE	60.0f
#define CIRCLES_HEIGHT_VAR	40.0f
#define CIRCLES_HEIGHT_SPEED	1.0f
#define CIRCLES_SPEED		20
#define CIRCLES_RUNNER_MAX	100
#define CIRCLES_OFF		0
#define CIRCLES_CCW		1
#define CIRCLES_CW		2
#define CIRCLES_IN		3
#define CIRCLES_OUT		4
#define CIRCLESRUNNER_RCHANGE	2
#define CIRCLESRUNNER_NCHANGE	40
#define CIRCLESRUNNER_DIVIDE	25

/*
 * Variables of circle
 */
char var_circles[CIRCLES_RADIALS * CIRCLES_SECTORS];
ShapeDB shape_circles;
CirclesRunner var_runner[CIRCLES_RUNNER_MAX];

/*
 * Generate one runner to draw the circle.
 */
void efu_circles_create_runner(unsigned radial, unsigned sector, unsigned dir,
    int elapsed)
{
  int i = CIRCLES_RUNNER_MAX;
  CirclesRunner *runner = var_runner;

  do
  {
    if(runner->dir == 0)
    {
      runner->radial = radial;
      runner->sector = sector;
      runner->dir = dir;
      runner->stime = elapsed;
      return;
    }
    runner++;
  } while(--i);
#ifndef REMOVE_STDLIB
  printf("No room!\n");
#endif
}

/*
 * True/false for the direction to be unblocked.
 */
int efu_circles_dir_ok(int idx)
{
  if((idx >= 0) && (idx < CIRCLES_SECTORS * CIRCLES_RADIALS))
    if(var_circles[idx] == 0)
      return 1;
  return 0;
}

/*
 * Calculate the index in the circle array of a given coordinate.
 */
int efu_circles_idx(int radial, int sector)
{
  if(radial < 0)
    return -1;
  else if(radial >= CIRCLES_RADIALS)
    return -1;
  return radial * CIRCLES_SECTORS + modulo(sector, CIRCLES_SECTORS);
}

/*
 * Try to expire a runner after it has stopped drawing.
 */
void efu_circles_expire_runner(CirclesRunner *op, int left, int freezetime)
{
  // If we're not in the end, let's quit
  if(left > CIRCLES_SPEED)
    return;
  // Otherwise try to expire
  efu_light_pillar(shapedb_get_vertex(&shape_circles,
	(op->radial * CIRCLES_SECTORS + op->sector) * 2), op->stime,
      op->etime, freezetime, &shape_pillar_draw, &pillar_draw);
}

/*
 * Draw a runner. Make some polygons while at it.
 */
void efu_circles_runner(CirclesRunner* op, int left, int elapsed,
    int freezetime)
{
  // Jos ei mitään niin ei tehdä
  if(op->dir == 0)
    return;
  // Jos enää pelkkä pilari
  else if(op->dir == -1)
  {
    efu_circles_expire_runner(op, left, freezetime);
    return;    
  }

  // Muutoin voidaan jatkaa
  int dir[4] =
  {
    dir[0] = efu_circles_idx(op->radial, op->sector - 1),
    dir[1] = efu_circles_idx(op->radial, op->sector + 1),
    dir[2] = efu_circles_idx(op->radial - 1, op->sector),
    dir[3] = efu_circles_idx(op->radial + 1, op->sector)
  };

  // Jos kohde ei ole ok, vaihda suuntaa
  if(efu_circles_dir_ok(dir[op->dir - 1]) == 0)
  {
    // Aseta päätös ja ajankohta
    op->dir = -1;
    op->etime = elapsed;
    // Anna uusi mahdollisuus vaihtaa suuntaa 5 kertaa
    int i = 6;
    do
    {
      int j = rand_int(4);
      if(efu_circles_dir_ok(dir[j]))
      {
	op->dir = j + 1;
	break;
      }
    } while(--i);
  }
  // Jos kaikki vieläkin pielessä
  if(op->dir <= 0)
  {
    efu_circles_expire_runner(op, left, freezetime);
    return;
  }

  // Valitaan ja tehdään neliö
  int curr = (op->radial * CIRCLES_SECTORS + op->sector) * 2,
      next = dir[op->dir - 1];
  // Kohde on käytetty
  var_circles[next] = 1;
  next *= 2;

  // Insertoi
  shapedb_add_index(&shape_circles, curr);
  shapedb_add_index(&shape_circles, curr + 1);
  // Haku
  if(left > CIRCLES_SPEED)
  {
    shapedb_add_index(&shape_circles, next + 1);
    shapedb_add_index(&shape_circles, next);
  }
  // Interpoloi
  else
  {
    float len = (float)left / (float)CIRCLES_SPEED;
    shapedb_add_index(&shape_circles, shape_circles.ins_vertex + 1);
    shapedb_add_index(&shape_circles, shape_circles.ins_vertex);
    shapedb_interpolate_v(&shape_circles, curr, next, len, 6);
    shapedb_interpolate_v(&shape_circles, curr + 1, next + 1, len, 7);
    efu_light_pillar(shapedb_get_vertex(&shape_circles,
	  shape_circles.ins_vertex - 2), op->stime, MAX_EFFECT_TIME,
	freezetime, &shape_pillar_draw, &pillar_draw);
  }

  // Jakaudu
  if(!rand_int(CIRCLESRUNNER_DIVIDE))
    efu_circles_create_runner(op->radial, op->sector, rand_int(4) + 1,
	elapsed);

  // Jatkamme matkaa
  switch(op->dir)
  {
    case CIRCLES_CCW:
      op->sector -= 2;
    case CIRCLES_CW:
      op->sector++;
      op->sector = modulo(op->sector, CIRCLES_SECTORS);
      break;
    case CIRCLES_IN:
      op->radial -= 2;
    case CIRCLES_OUT:
      op->radial++;
      break;
  }

  // Vaihda suuntaa
  if(!rand_int(CIRCLESRUNNER_NCHANGE) ||
      ((op->dir >= CIRCLES_IN) && rand_int(2)))
    op->dir = rand_int(4) + 1;
}

/*
 * Ympyräefun ylihommeli.
 */
void efu_circles(int freezetime)
{
  int i, j, k;

  // Tyhjennys ja random
  clearGL(0);
  rand_seed(0x01);

  // Clear the shape
  shapedb_clear(&shape_circles);

  // Generate the vertices
  for(i = 0; i < CIRCLES_RADIALS; i++)
    for(j = 0; j < CIRCLES_SECTORS; j++)
    {
      float ang = (float)j / (float)CIRCLES_SECTORS * M_PI * 2;
      float rad = (i + 1) * CIRCLES_RADIUS;
      // Korkeus on monimutk.
      float height = interpolate_sine(CIRCLES_HEIGHT_BASE / 2.0f,
	CIRCLES_HEIGHT_VAR / 2.0f, (float)j + (float)freezetime / 100.0f,
	CIRCLES_HEIGHT_SPEED) + interpolate_sine(CIRCLES_HEIGHT_BASE / 2.0f,
	  CIRCLES_HEIGHT_VAR / 2.0f,
	  (float)j * 1.5f + (float)freezetime / 15.0f, -CIRCLES_HEIGHT_SPEED);
      // Add
      shapedb_add_cv(&shape_circles, cos(ang) * rad, 0.0f, sin(ang) * rad, 6);
      shapedb_add_cv(&shape_circles, cos(ang) * rad, height, sin(ang) * rad,
	  7);
    }

  // Piirretään
  memclr(var_circles, CIRCLES_RADIALS * CIRCLES_SECTORS);
  var_circles[0] = 1;
  memclr(var_runner, CIRCLES_RUNNER_MAX * sizeof(CirclesRunner));
  efu_circles_create_runner(CIRCLES_RADIALS - 1, 0, CIRCLES_CCW, CIRCLES_START);
  for(i = freezetime, j = CIRCLES_START; i > 0;
      i -= CIRCLES_SPEED, j += CIRCLES_SPEED)
    for(k = 0; k < CIRCLES_RUNNER_MAX; k++)
      efu_circles_runner(&var_runner[k], i, j, freezetime);

  shapedb_draw_elements(&shape_circles, -1, GL_QUADS);
}

/*
 * Ympyräefun pilarihommeli.
 */
void efu_pillar(int freezetime)
{
  efu_light_pillar(origo, 0, PILLAR_DUR, freezetime,
      &shape_pillar_wings, &pillar_wings);
}

/*
 * Ympyräefun teksti.
 */
void efu_oritext(int freezetime)
{
  char string[4] = { 1, 2, 3, 0 };
  camera_ortho(1.0f);
  glLoadIdentity();
  text_draw_string(string, -0.9f, 0.3f, 0.3f, 0.3f, 2, freezetime * 0.01f);
}

//############################################################################
// Flare #####################################################################
//############################################################################

#define FLARE_ROTSPEED	2.0f

// Muuttujat
ShapeDB shape_flare;

/*
 * Draw a GL flare.
 */
void efu_flare(float *pos, int freezetime, float scale)
{
  rand_seed(0xdeadbeef);
  glPushMatrix();
  glTranslatef(pos[0], pos[1], pos[2]);

  float flare_rot = FLARE_ROTSPEED * (float)freezetime;
  int i = 8;
  do
  {
    glPushMatrix();
    glRotate3f(rand_float(360.0f) + flare_rot * rand_float_neg(1.0f),
	rand_float(360.0f) + flare_rot * rand_float_neg(1.0f),
	rand_float(360.0f) + flare_rot * rand_float_neg(1.0f));
    glScalef(scale, scale, scale);
    shapedb_draw_elements(&shape_flare, -1, GL_TRIANGLES);
    glPopMatrix();
  } while(--i);
  glPopMatrix();
}

//############################################################################
// LineTextures ##############################################################
//############################################################################

#define LINETEXTURE_H		512
#define LINETEXTURE_W		32
#define LINETEXTURE_LINES	8
#define LINETEXTURE_CHANGE	20
#define LINECIRCLE_COMPLEXITY	32
#define LINECIRCLE_THICKNESS	0.025f

/*
 * Linetexture variables.
 */
Uint32 linetexture1[LINETEXTURE_W * LINETEXTURE_H],
       linetexture2[LINETEXTURE_W * LINETEXTURE_H],
       linetexture3[LINETEXTURE_W * LINETEXTURE_H];
GLuint linetexture1_bind, linetexture2_bind, linetexture3_bind;
ShapeDB shape_line_circle;

/*
 * Generate the texture.
 */
void generate_line_texture(Uint32 *op, GLuint *bind, int col_op)
{
  Uint32 col = colordb_get_uint(col_op);
  int i = LINETEXTURE_LINES;
  do
  {
    int cw = rand_int(LINETEXTURE_W - 2 + 1),
	h = LINETEXTURE_H;
    Uint32 *bmp = op;
    do
    {
      // Vaihdetaanko kolumnia?
      if(!rand_int(LINETEXTURE_CHANGE))
      {
	int ncw = rand_int(LINETEXTURE_W - 2) + 1;
	int j = ((ncw - cw) > 0) ? 1 : -1;
	for(; cw != ncw; cw += j)
	  bmp[cw] = col;
	cw = ncw;
      }
      // Kolumniin väri
      bmp[cw] = col;
      // Eteenpäin
      bmp += LINETEXTURE_W;
    } while(--h);
  } while(--i);

  glEnable(GL_TEXTURE_2D);
  glGenTextures(1, bind);
  glBindTexture(GL_TEXTURE_2D, *bind);
  gluBuild2DMipmaps(GL_TEXTURE_2D, 4, LINETEXTURE_W, LINETEXTURE_H, GL_RGBA,
      GL_UNSIGNED_BYTE, op);
}

/*
 * Generate line with a line texture from point a to point b.
 */
float efu_line_line(ShapeDB *shp, float start, float end, float other1,
    float other2, float radius, float texpos, float texwrap, int axis)
{
  float pos[3],
	ang = rand_float(M_PI * 2.0f),
	ca = cos(ang) * radius,
	sa = sin(ang) * radius,
	texend = fabs(end - start) / texwrap + texpos;
  int ax1 = (axis + 1) % 3,
      ax2 = (axis + 2) % 3,
      idx = shp->ins_vertex;

  // Indeksit
  shapedb_add_index(shp, idx);
  shapedb_add_index(shp, idx + 1);
  shapedb_add_index(shp, idx + 2);
  shapedb_add_index(shp, idx + 3);
  // Data
  pos[axis] = start;
  pos[ax1] = other1 + ca;
  pos[ax2] = other2 + sa;
  shapedb_add_tv(shp, pos[0], pos[1], pos[2], 0.0f, texpos);
  pos[ax1] = other1 - ca;
  pos[ax2] = other2 - sa;
  shapedb_add_tv(shp, pos[0], pos[1], pos[2], 1.0f, texpos);
  pos[axis] = end;
  shapedb_add_tv(shp, pos[0], pos[1], pos[2], 1.0f, texend);
  pos[ax1] = other1 + ca;
  pos[ax2] = other2 + sa;
  shapedb_add_tv(shp, pos[0], pos[1], pos[2], 0.0f, texend);

  return fmod(texend, 1.0);
}

/*
 * Draw line circle
 */
void efu_line_circle(float *pos, int texture, float radius, float xr,
    float yr, float zr)
{
  glPushMatrix();
  glTranslatef(pos[0], pos[1], pos[2]);
  glRotate3f(xr, yr, zr);
  glScalef(radius, radius * 10.0f, radius);
  shapedb_draw_elements(&shape_line_circle, texture, GL_TRIANGLE_STRIP);
  glPopMatrix();
}

//############################################################################
// GitS -efu #################################################################
//############################################################################

#define GITS_BOUND_X	15000.0f
#define GITS_BOUND_YZ	1500.0f
#define GITS_GRIDSPACE	130
#define GITS_TEXWRAP	1000.0f
#define GITS_GRIDCNT_X	((int)(GITS_TEXWRAP / (float)GITS_GRIDSPACE))
#define GITS_GRIDCNT_YZ	((int)(GITS_BOUND_YZ / (float)GITS_GRIDSPACE))
#define GITS_GRIDLINE_W	5.0f
#define GITS_FOGDENSITY	0.0005f
#define GITS_SPHERES	40
#define GITS_PATH_RAND	500.0f
#define GITS_PATH_BASE	100.0f
#define GITS_PATH_MAX	(GITS_PATH_RAND + GITS_PATH_BASE)
#define GITS_SPHERERAD	125.0f

// Taulukko sphereille
GitsSphere var_gits_spheres[GITS_SPHERES];
GitsPath var_gits_paths[GITS_SPHERES * GITS_LINKSPER];
ShapeDB shape_gits_paths, shape_gits_grid;
int var_gits_path_start[MAX_EFFECT_TIME];
int var_gits_pathcount;

/*
 * Varmistaa ettei jossakin spheressä ole yhteyttä johonkin toiseen.
 */
int sphere_has_no_conn(GitsSphere *op, int conn)
{
  int i = GITS_LINKSPER - 1;
  do
  {
    if(op->connections[i] == conn)
      return 0;
  } while(i--);
  return 1;
}

/*
 * Generoi polku kahden pisteen välille.
 */
void generate_gits_path(GitsSphere *src, GitsSphere *dst, GitsPath *pth)
{
  float *pos = pth->pos, texpos = 0.0f;
  pth->len = 1;
  pth->flen = 0.0f;
  pos[0] = src->pos[0] + rand_float(GITS_SPHERERAD / 2.0f);
  pos[1] = src->pos[1] + rand_float(GITS_SPHERERAD / 2.0f);
  pos[2] = src->pos[2] + rand_float(GITS_SPHERERAD / 2.0f);
  do
  {
    int crd = rand_int(3);
    float diff = dst->pos[crd] - pos[crd];

    if(diff != 0.0f)
    {
      // Jos ollaan liian kaukana niin mennään randomilla
      if(dist_root(pos, dst->pos) > sqrt(GITS_PATH_MAX * GITS_PATH_MAX * 3))
	diff = (rand_float(GITS_PATH_RAND) + GITS_PATH_BASE) *
	  ((diff >= 0) ? 1.0f : -1.0f);
      // Mennään eteenpäin
      int ax1 = (crd + 1) % 3,
	  ax2 = (crd + 2) % 3;
      pos[ax1 + 3] = pos[ax1];
      pos[ax2 + 3] = pos[ax2];
      pos[crd + 3] = pos[crd] + diff;
      // Lisää useampi linja, saadaan paksuutta!
      int i = 7;
      do
      {
	efu_line_line(&shape_gits_paths, pos[crd], pos[crd + 3],
	    pos[ax1] + rand_float_neg(GITS_GRIDLINE_W * 2.0f),
	    pos[ax2] + rand_float_neg(GITS_GRIDLINE_W * 2.0f),
	    GITS_GRIDLINE_W, texpos, GITS_TEXWRAP, crd);
      } while(--i);
      // Liike ok
      pos += 3;
      pth->flen += fabs(diff);
      pth->len++;
      // Jos saavuttu niin end!
      if(dist_root(dst->pos, pos) < GITS_SPHERERAD / 2.0f)
	break;
    }
  } while(1);
}

/*
 * Generoi yhteydet renkaitten välille.
 */
void generate_gits_connections()
{
  int i;

  // Arvo paikat
  i = GITS_SPHERES - 1;
  do
  {
    GitsSphere *sph = var_gits_spheres + i;
    sph->pos[0] = rand_float(GITS_BOUND_X);
    sph->pos[1] = rand_float(GITS_BOUND_YZ);
    sph->pos[2] = rand_float(GITS_BOUND_YZ);
    sph->conncnt = rand_int(GITS_LINKSPER) + 1;
    // Nollaa linkit
    int j = GITS_LINKSPER - 1;
    do
    {
      sph->connections[j] = -1;
    } while(j--);
  } while(i--);


  // Grid
  shapedb_clear(&shape_gits_grid);
  i = GITS_GRIDCNT_X;
  do
  {
    float xpos = (float)i / (float)GITS_GRIDCNT_X * GITS_TEXWRAP;
    int j = GITS_GRIDCNT_YZ;
    do
    {
      float yzpos = (float)j / (float)GITS_GRIDCNT_YZ * GITS_BOUND_YZ,
	    boundmin = -1000.0f,
	    boundmax = GITS_BOUND_YZ + 1000.0f;
      // Vaihdetaan randomilla
      if(rand_int(2))
	xchgf(&boundmin, &boundmax);
      efu_line_line(&shape_gits_grid, boundmin, boundmax, yzpos, xpos,
	  GITS_GRIDLINE_W, rand_float(1.0f), GITS_TEXWRAP, 1);
      if(rand_int(2))
	xchgf(&boundmin, &boundmax);
      efu_line_line(&shape_gits_grid, boundmin, boundmax, xpos, yzpos,
	  GITS_GRIDLINE_W, rand_float(1.0f), GITS_TEXWRAP, 2);
      // Viimeisellä kierroksella piirretään myös loput viivat
      if(i == GITS_GRIDCNT_X)
      {
	int k = GITS_GRIDCNT_YZ;
	do
	{
	  float yzpos2 = (float)k / (float)GITS_GRIDCNT_YZ * GITS_BOUND_YZ,
		boundmin = 0.0f,
		boundmax = GITS_TEXWRAP;
	  if(rand_int(2))
	    xchgf(&boundmin, &boundmax);
	  efu_line_line(&shape_gits_grid, boundmin, boundmax, yzpos, yzpos2,
	      GITS_GRIDLINE_W, rand_float(1.0f), GITS_TEXWRAP, 0);
	} while(k--);
      }
    } while(j--);
  } while(--i);
  
  // Merkkaa yhteydet
  for(i = 0; i < GITS_SPHERES; i++)
  {
    GitsSphere *sph = var_gits_spheres + i;
    //printf("%i has connections: %i\n", i, sph->conncnt);
    // Hae sovittu määrä lähimpiä
    while(sph->conncnt > 0)
    {
      float closest = 10000000.0f;
      int j, closestidx = -1;
      for(j = i + 1; j < GITS_SPHERES; j++)
      {
	GitsSphere *tgt = var_gits_spheres + j;
	float newdist = dist_sqr(tgt->pos, sph->pos);
	if((newdist < closest) && (sphere_has_no_conn(sph, j)))
	{
	  closestidx = j;
	  closest = newdist;
	}
      }
      GitsSphere *tgt = var_gits_spheres + closestidx;
      // Jos ei löydy niin quit
      if(closestidx < 0)
	break;
      // Jos kohteessa ei tilaa niin quit
      if(tgt->conncnt <= 0)
	break;
      // Muutoin kaikki ok
      sph->connections[--(sph->conncnt)] = closestidx;
      tgt->conncnt--;
      //printf("Found connection: %i\n", closestidx);
    }
  }

  // Generoi linkit, erikseen koska tämä voisi olla cool jos se tehtäisiin
  // joka frame!
  shapedb_clear(&shape_gits_paths);
  var_gits_pathcount = 0;
  i = GITS_SPHERES - 1;
  do
  {
    GitsSphere *sph = var_gits_spheres + i;
    int j = GITS_LINKSPER - 1;
    do
    {
      int k = sph->connections[j];
      if(k >= 0)
      {
	GitsSphere *tgt = var_gits_spheres + k;
	generate_gits_path(sph, tgt, var_gits_paths + var_gits_pathcount);
	var_gits_pathcount++;
      }
    } while(j--);
  } while(i--);
}

/*
 * Liiku yksi väli eteenpäin polulla.
 */
int efu_gits_haerpake_advance(float *curr, float **pth_op, int *lenleft,
    float *timeleft, float speed)
{
  // Jos polku loppu, mennään
  if(*lenleft <= 0)
    return -1;

  // Ota polku
  float *pth = *pth_op;

  // Calculate the distance to the next cell
  float dist[3] =
  {
    pth[0] - curr[0],
    pth[1] - curr[1],
    pth[2] - curr[2]
  };
  
  // Kuinka paljon aikaa menee
  float use = fabs(dist[0] + dist[1] + dist[2]) / speed;

  // Jos aika loppuu kesken (saavumme)
  if(use >= *timeleft)
  {
    use = *timeleft / use;
    *timeleft = 0.0f;
    curr[0] += use * dist[0];
    curr[1] += use * dist[1];
    curr[2] += use * dist[2];
    return 1;
  }

  // Ei saavutettu, joten mennään seuraavaan pisteeseen
  *lenleft -= 1;
  *timeleft -= use;
  curr[0] = pth[0];
  curr[1] = pth[1];
  curr[2] = pth[2];
  *pth_op += 3;
  
  return 0;
}

#define GITS_HAERPAKE_COUNT	12
#define GITS_HAERPAKE_DELAY	1
#define GITS_HAERPAKE_TOT	(GITS_HAERPAKE_COUNT * GITS_HAERPAKE_DELAY)
#define GITS_HAERPAKE_SCALEINC	(1.0f / (float)GITS_HAERPAKE_COUNT)

/*
 * Piirrä liikkuva härpäke polulle.
 */
void efu_gits_path_haerpake(GitsPath *pth, int starttime, int currtime,
    float speed)
{

  // Jos ei riitä aika kummiskaan
  if(pth->flen < (currtime - starttime) * speed)
    return;

  // Polun tiedot
  float *pos = pth->pos,
	curr[3] = { pos[0], pos[1], pos[2] },
	scale = GITS_HAERPAKE_SCALEINC,
	timeleft;
  int lenleft = pth->len,
      maxinc;
  
  maxinc = currtime % GITS_HAERPAKE_DELAY;
  timeleft = currtime - maxinc - starttime - GITS_HAERPAKE_TOT;
  maxinc += GITS_HAERPAKE_TOT;

  // Eteenpäin
  pos += 3;

  while(timeleft >= 0.0f)
  {
    // Jos ennen 0:aa
    if(currtime - (int)timeleft < starttime)
      timeleft -= (float)GITS_HAERPAKE_DELAY;
    // Muutoin edetään
    else
      switch(efu_gits_haerpake_advance(curr, &pos, &lenleft, &timeleft, speed))
      {
	// Loppu
	case -1:
	  return;
	  // Saavutettu
	case 1:
	  rand_seed(0xdeadbeef);
	  efu_flare(curr, currtime, scale);
	  // Lisää?
	  if(maxinc <= 0)
	    return;
	  if(maxinc > GITS_HAERPAKE_DELAY)
	  {
	    timeleft += GITS_HAERPAKE_DELAY;
	    maxinc -= GITS_HAERPAKE_DELAY;
	  }
	  else
	  {
	    timeleft += maxinc;
	    maxinc = 0;
	  }
	  scale += GITS_HAERPAKE_SCALEINC;
	  break;
      default:
	break;
    }
  }
}

#define GITSTEXT_APPEAR		50
#define GITSTEXT_DURATION	400
#define GITSTEXT_SPEED		0.55f
#define GITSTEXT_SCALE		0.1f
#define GITSTEXT_HEIGHT		0.14f
#define GITSTEXT_RANGE		0.65f

/*
 * Näyttää textoutit / krediitit.
 */
void efu_gits_text(char *str, int stime, float size, int freezetime)
{
  float y1 = rand_float(GITSTEXT_RANGE * 2.0f - GITSTEXT_HEIGHT) -
    GITSTEXT_RANGE;
  // Vaihto jos tarpeen
  if(rand_int(2))
    size = -size;
  // Paikka
  textout_complex(str, stime, stime + GITSTEXT_DURATION, freezetime,
      GITSTEXT_APPEAR, GITSTEXT_SPEED, size, y1, y1 + GITSTEXT_HEIGHT,
      GITSTEXT_SCALE, 3);
}

/*
 * Piirrä itse gits-efu.
 */
void efu_gits(int freezetime)
{
  // Tyhjennys ja random
  clearGL(1);
  rand_seed(0xdeadbeef);

  // Hyper draiv
  if((freezetime >= 50) && (freezetime <= 180))
    camera_frustum((float)(130 - (freezetime - 50)) / 130.0f * 80.0f + 90.0f);

  // This effect has fog
  glFogfv(GL_FOG_COLOR, colordb_get_float(0));
  glFogi(GL_FOG_MODE, GL_EXP2);
  glFogf(GL_FOG_DENSITY, GITS_FOGDENSITY);
  glEnable(GL_FOG);

  // Randomize a few circles
  glMatrixMode(GL_TEXTURE);
  glPushMatrix();
  glTranslatef(0.0f, 0.001f * freezetime, 0.0f);
  glMatrixMode(GL_MODELVIEW);

  int i = GITS_SPHERES - 1;
  do
  {
    GitsSphere *sph = var_gits_spheres + i;
    int j = 10;
    do
    {
      efu_line_circle(sph->pos, linetexture1_bind,
	  GITS_SPHERERAD + rand_float_neg(25.0f), rand_float(360.0f),
	  rand_float(360.0f), rand_float(360.0f));
    } while(--j);
  } while(i--);

  // Texture matrix again
  glMatrixMode(GL_TEXTURE);
  glTranslatef(0.0f, - 0.0005f * freezetime, 0.0f);

  // Grid and paths
  glMatrixMode(GL_MODELVIEW);
  float gridcrd = -GITS_BOUND_X;
  do
  {
    if(fabs(gridcrd - camera_position[0]) < CAMERA_BACKCLIP)
    {
      glPushMatrix();
      glTranslatef(gridcrd, 0.0f, 0.0f);
      shapedb_draw_elements(&shape_gits_grid, linetexture2_bind, GL_QUADS);
      glPopMatrix();
    }
    gridcrd += GITS_TEXWRAP;
  } while(gridcrd < 2.0f * GITS_BOUND_X);

  // Paths
  shapedb_draw_elements(&shape_gits_paths, linetexture3_bind, GL_QUADS);

  // Return to normal matrix mode
  glMatrixMode(GL_TEXTURE);
  glPopMatrix();
  glMatrixMode(GL_MODELVIEW);

  // Piirrä kaikki härpäkkeet
  for(i = 0; i < freezetime; i+= 10)
  {
    efu_gits_path_haerpake(var_gits_paths + var_gits_path_start[i], i,
	freezetime, 5.0f);
  }

  // Kaikki teksti
  rand_seed(2);
  efu_gits_text(string_gfxcode, 300, 1.3f, freezetime);
  rand_seed(3);
  efu_gits_text(string_trilkk, 380, 0.7f, freezetime);
  rand_seed(4);
  efu_gits_text(string_muscode, 600, 1.1f, freezetime);
  rand_seed(5);
  efu_gits_text(string_juippi, 680, 0.7f, freezetime);
  rand_seed(6);
  efu_gits_text(string_other, 900, 1.2f, freezetime);
  rand_seed(7);
  efu_gits_text(string_warma, 980, 0.6f, freezetime);
  rand_seed(8);
  efu_gits_text(string_greets, 1200, 0.6f, freezetime);
  rand_seed(9);
  efu_gits_text(string_vv, 1280, 0.4f, freezetime);
}

//############################################################################
// Kaupunki ##################################################################
//############################################################################

#define CITY_SIZE		512
#define CITY_SQUARE		64
#define CITY_CONTROL_DENSITY	8
#define CITY_CONTROL_SIZE	(CITY_SIZE / CITY_CONTROL_DENSITY)
#define CITY_CRDSTART		0.4f
#define CITY_CRDEND		0.6f
#define CITY_TERRAIN_MAXINT	255
#define CITY_HEIGHT		48
#define CITY_TERRAIN_MAXHEIGHT	120.0f
#define CITY_SQUARE_SIZE	500.0f
#define CITY_DETAIL		128

// Maastodata
int var_city_terraincontrol[CITY_CONTROL_SIZE * CITY_CONTROL_SIZE];
float var_city_terrain[CITY_SIZE * CITY_SIZE];
int var_city_cityarea[CITY_DETAIL * CITY_DETAIL];

/*
 * Piirrä suorakulmio jotakin korkeutta karttaan.
 */
void city_map_drawrect(int x1, int y1, int x2, int y2, int ht)
{
  int i;
  for(i = y1; i <= y2; i++)
  {
    int j;
    for(j = x1; j <= x2; j++)
      var_city_terraincontrol[i * CITY_CONTROL_SIZE + j] = ht;
  }
}

/*
 * Hae kontrollipiste.
 */
float get_control_point_height(int x, int y, int cx, int cy, int rx, int ry)
{
  int mx = x % CITY_CONTROL_DENSITY,
      my = y % CITY_CONTROL_DENSITY;
  float dist = (float)CITY_CONTROL_DENSITY - dist_2d(mx, my, rx, ry);
  // Vain riittävän lähellä olevat nurkkapisteet hyväksytään
  if(dist <= 0.0f)
    return 0.0f;
  else
    return dist * (CITY_TERRAIN_MAXHEIGHT / (float)CITY_TERRAIN_MAXINT) *
      (float)var_city_terraincontrol[cy * CITY_CONTROL_SIZE + cx];
}

/*
 * Hae jonkin pisteen korkeus (suhteessa kontrollipisteisiin).
 */
float get_point_height(int x, int y)
{
  int minx = x % CITY_CONTROL_DENSITY,
      miny = y % CITY_CONTROL_DENSITY,
      maxx = (minx + 1) % CITY_CONTROL_SIZE,
      maxy = (miny + 1) % CITY_CONTROL_SIZE;
  return get_control_point_height(x, y, minx, miny, 0, 0) +
    get_control_point_height(x, y, maxx, miny, CITY_CONTROL_DENSITY, 0) +
    get_control_point_height(x, y, minx, maxy, 0, CITY_CONTROL_DENSITY) +
    get_control_point_height(x, y, maxx, maxy, CITY_CONTROL_DENSITY,
	CITY_CONTROL_DENSITY);
}

/*
 * Esilaske kaupunki.
 */
void precalculate_city_terrain()
{
  // Ensin maaston kontrollipisteet
  int i = CITY_CONTROL_SIZE * CITY_CONTROL_SIZE - 1;
  do
  {
    var_city_terraincontrol[i] = rand_int(CITY_TERRAIN_MAXINT + 1);
  } while(i--);

  // Tehdään liffa alue kaupunkia varten
  float citystart = ((float)CITY_CONTROL_SIZE * CITY_CRDSTART) - 1,
	cityend = ((float)CITY_CONTROL_SIZE * CITY_CRDEND) + 1;
  city_map_drawrect(citystart, citystart, cityend, cityend, CITY_HEIGHT);

  // Maasto interpoloimalla pisteitten väliä
  for(i = 0; i < CITY_SIZE; i++)
  {
    int j = CITY_SIZE - 1;
    do
    {
      var_city_terrain[i * CITY_SIZE + j] = get_point_height(i, j);
    } while(j--);
  }

  // Generoidaan kaupunki
  for(i = 0; i < CITY_DETAIL; i++)
  {
    int j;
    for(j = 0; j < CITY_DETAIL; j++)
    {

    }
  }
}

/*
 * Piirrä kaupunki.
 */
void efu_city(int op)
{
  return;
}

//############################################################################
// Efektit ###################################################################
//############################################################################

#define EFU_COUNT	7

/*
 * Alustetaan kaikkien efektien ajat.
 */
EffectData var_efu[EFU_COUNT] =
{
  { CIRCLES_START, CIRCLES_END, efu_circles },
  { PILLAR_START, CIRCLES_END, efu_pillar },
  { WINGS_START, WINGS_END, efu_wings },
  { ORITEXT_START, ORITEXT_END, efu_oritext },
  { CITY_START, CITY_END, efu_city },
  { GITS_START, GITS_END, efu_gits }
};

//############################################################################
// Main ######################################################################
//############################################################################

int main(void)
{
  // Lokaalit muuttujat
  int loop = 1;
  SDL_Event event;
  SDL_Surface *screen;

  /*
   * Init.
   */
  
  // SDL
  SDL_Init(SDL_INIT_TIMER | SDL_INIT_VIDEO | SDL_INIT_AUDIO);
  screen = SDL_SetVideoMode(SCREEN_W, SCREEN_H, SCREEN_B, SDL_GFX_FLAGS);
  SDL_SetTimer(TIMER_MS, timer_func);
  // OpenGL
  glViewport(0, 0, SCREEN_W, SCREEN_H);
  glClearDepth(1.0);
  //glDepthFunc(GL_LEQUAL);
  glEnable(GL_BLEND);
  glDisable(GL_DEPTH_TEST);
  //glShadeModel(GL_SMOOTH);

  /*
   * Precalc.
   */
  
  // Värit
  int i = COLORDB_SIZE - 1;
  do
  {
    colordb_convert_u2f(i);
  } while(i--);

  printf("%x\n", var_rand_seed);

  // Tekstuurit
  generate_line_texture(linetexture1, &linetexture1_bind, 9);
  generate_line_texture(linetexture2, &linetexture2_bind, 10);
  generate_line_texture(linetexture3, &linetexture3_bind, 11);

  printf("%x\n", var_rand_seed);

  // Linjaympyrä
  i = LINECIRCLE_COMPLEXITY;
  do
  {
    float percent = (float)i / (float)LINECIRCLE_COMPLEXITY,
	  ang = percent * M_PI * 2,
	  ca = cos(ang),
	  sa = sin(ang); 
    shapedb_add_tv(&shape_line_circle, ca, -LINECIRCLE_THICKNESS, sa,
	0.0f, percent);
    shapedb_add_tv(&shape_line_circle, ca, LINECIRCLE_THICKNESS, sa,
	1.0f, percent);
    shapedb_add_index(&shape_line_circle, i * 2);
    shapedb_add_index(&shape_line_circle, i * 2 + 1);
  } while(i--);

  printf("%x\n", var_rand_seed);

  // GitS-efun yhteydet
  generate_gits_connections();

  printf("%x\n", var_rand_seed);

  // Asiat joita tarvii joka framelle
  i = MAX_EFFECT_TIME * TIMER_MS - 1;
  do
  {
    var_pillar_shard[i] = rand_float(M_PI * 2.0f);
    var_gits_path_start[i] = rand_int(var_gits_pathcount);
  } while(i--);

  // Flare
  shapedb_clear(&shape_flare);
  shapedb_add_cv(&shape_flare, 0.0f, 0.0f, 0.0f, 8);
  shapedb_add_cv(&shape_flare, -15.0f, -15.0f, 0.0f, 0);
  shapedb_add_cv(&shape_flare, 15.0f, -15.0f, 0.0f, 0);
  shapedb_add_cv(&shape_flare, 0.0f, 20.0f, 0.0f, 0);
  shapedb_add_index(&shape_flare, 0);
  shapedb_add_index(&shape_flare, 1);
  shapedb_add_index(&shape_flare, 2);
  shapedb_add_index(&shape_flare, 0);
  shapedb_add_index(&shape_flare, 2);
  shapedb_add_index(&shape_flare, 3);
  shapedb_add_index(&shape_flare, 0);
  shapedb_add_index(&shape_flare, 3);
  shapedb_add_index(&shape_flare, 1);

  // Pilarit
  generate_light_pillar_shard(&shape_pillar_draw, &pillar_draw);
  generate_light_pillar_shard(&shape_pillar_wings, &pillar_wings);

  // Skip
  var_timer = GITS_START;

  // Loop
  while(loop)
  {
    blend_mode(0);
    camera_frustum(90.0f);
    // Olemme modelviewissä
    glLoadIdentity();
    // Katsellaan
    look_at();
    // Kelataan efut läpi
    for(i = 0; i < EFU_COUNT; i++)
    {
      EffectData *efu = var_efu + i;
      if((var_timer >= efu->start) && (var_timer < efu->end))
	efu->func(var_timer - efu->start);
    }
    // Ruudulle
    SDL_GL_SwapBuffers();
    // Pois jos nappia painettu
    if((SDL_PollEvent(&event)) && (event.type == SDL_KEYDOWN))
      switch(event.key.keysym.sym)
      {
	case SDLK_ESCAPE:
	  loop = 0;
	  break;
	default:
	  var_timer += 300;
	  break;
      }
    // Pois jos aika loppu
    if(var_timer >= TIME_END)
      loop = 0;
  }

  // Pois
  SDL_Quit();
  return 0;
}

//############################################################################
// Loppu #####################################################################
//############################################################################
