#!/usr/bin/perl

use warnings;
use strict;

use lib '../include';

use nms qw(switch_connect switch_exec switch_disconnect);

use Net::Telnet::Cisco;
use Net::Ping;

use Term::ANSIColor;

my $patchlist = "../patchlist.txt";
my $switches = "../zyxels.txt";
#my $switches = "../switches.txt";
#my $patchlist = "/home/eirikn/patchlist.txt.eirik";
#my $switches = "/home/eirikn/switches.txt.eirik";
#my $patchlist = "/root/patchlist.txt.d05";
#my $switches = "/root/switches.txt.d05";

open LOG, ">/tmp/zyxel.could.not.connect" or die "Could not open log";

$| = 1;

BEGIN {
	require "../include/config.pm";
	eval {
		require "../include/config.local.pm";
	};
}

sub ios_getroute {
	my ($t, $net) = @_;

	if (join("\n", $t->cmd("show ip route $net")) !~ /directly connected, via Vlan/)
	{
		return 0;
	}
	
	return 1;
}

sub ios_killroute {
	my ($t, $net) = @_;

	my $output = join("\n", $t->cmd("show ip route $net"));
	if ($output =~ /directly connected, via Vlan(\d+)/) {
		return stop_vlan($t, $1);
	} else {
		return 0;
	}
}

sub start_vlan {
	my ($ios, $vlan) = @_;

	$ios->cmd("conf t");
	$ios->cmd("int vlan $vlan");
	$ios->cmd("ip add 192.168.1.254 255.255.255.0 secondary");
	$ios->cmd("exit");
	$ios->cmd("exit");
	# FIXME! --hack to check if this logic is broken, just assume it is done
	return ios_getroute($ios, "192.168.1.0");
}

sub stop_vlan {
	my ($ios, $vlan) = @_;

	$ios->cmd("conf t");
	$ios->cmd("int vlan $vlan");
	$ios->cmd("no ip add 192.168.1.254 255.255.255.0 secondary");
	$ios->cmd("exit");
	$ios->cmd("exit");
	return (ios_getroute($ios, "192.168.1.0") == 0);
}


sub set_gateway {
	my %switch = @_;
	print colored('INFO', 'bold blue'), ": Switch: $switch{name}, distro: $switch{distro_name}/$switch{distro_ip}, vlan: $switch{vlan}\n";
	print colored('INFO', 'bold blue'), ": IP net: $switch{network} [IP: $switch{ip}]\n";

	my ($first, $second, $third, $fourth) = split(/\./, $switch{network});
	my $gw = "$first.$second.$third.".(int($fourth)+1);

	my $ios = Net::Telnet::Cisco->new(Host => $switch{distro_ip},
			Errmode => 'return',
			Prompt => '/(\S+[#>])|(es-3024>)/');
	if (!defined($ios)) {
		print colored('ERROR', 'red'), "/", colored($switch{name}, 'bold'), ": Could not connect to $switch{distro_ip}";
		return;
	}

#	XXX: Hvis du får
#	Use of uninitialized value in pattern match (m//) at /usr/share/perl5/Net/Telnet/Cisco.pm line 214,
#	Så kan du fjerne kommentaren på denne. ;-)
	$ios->autopage(0);

	$ios->login($nms::config::ios_user, $nms::config::ios_pass);
	$ios->enable;

	$ios->cmd("terminal length 0");


	$ios->print("telnet $switch{ip}");
	if (!$ios->waitfor(Match => "/Password:/", Timeout => 5))
	{
		print colored('ERROR', 'red'), "/", colored($switch{name}, 'bold'), ": Could not telnet into!\n\n";
	} else {
		$ios->cmd($nms::config::zyxel_password);

		$ios->cmd("ip route drop 192.168.1.0/24");
		$ios->cmd("ip route add default $gw");
		$ios->cmd("exit");

		print STDERR colored('SUCCESS', 'green'), "/", colored($switch{name}, 'bold'), ": Gateway configured!\n";
	}

	$ios->close;

	return 1;
}

sub set_ip {
	my %switch = @_;

	print colored('INFO', 'bold blue'), ": Switch: $switch{name}, distro: $switch{distro_name}/$switch{distro_ip}, vlan: $switch{vlan}\n";
	print colored('INFO', 'bold blue'), ": IP net: $switch{network} [IP: $switch{ip}]\n";

	my $p = Net::Ping->new();

	if ($p->ping($switch{ip}, 2))
	{
		print colored('INFO', 'bold blue'), ": Skipping $switch{distro_ip}:$switch{vlan}, ip '$switch{ip}' is already responding to ping! \\o/\n";
		return 0;
	}

	my $ios = Net::Telnet::Cisco->new(Host => $switch{distro_ip},
			Errmode => 'return',
			Prompt => '/\S+[#>]/');
	if (!defined($ios)) {
		print STDERR colored('ERROR', 'red'), "/", colored($switch{name}, 'bold'), ": Could not connect to $switch{distro_ip}\n";
		return 0;
	}

#	XXX: Hvis du får
#	Use of uninitialized value in pattern match (m//) at /usr/share/perl5/Net/Telnet/Cisco.pm line 214,
#	Så kan du fjerne kommentaren på denne. ;-)
	$ios->autopage(0);

	$ios->login($nms::config::ios_user, $nms::config::ios_pass);
	$ios->enable;

	# Disable paging
	$ios->cmd("terminal length 0");

	if (ios_getroute($ios, "192.168.1.0") == 1) {
		print colored('INFO', 'bold blue'), ": Route already in place for 192.168.1.0/24, trying to kill.\n";
		if (ios_killroute($ios, "192.168.1.0") == 0) {
			print STDERR colored('ERROR', 'red'), "/", colored($switch{name}, 'bold'), ": Already routed up 192.168.1.0/24, could not kill!\n";
			return 0;
		}
	}

   print STDERR colored('INFO', 'bold blue'), "/", colored($switch{name}, 'bold'), ": Trying to get a route.\n";
	my $zyxeloldip = "192.168.1.1";
	if ( not start_vlan($ios, $switch{vlan}) ) 
	{
		print STDERR colored('ERROR', 'red'), "/", colored($switch{name}, 'bold'), ": Unable to route 192.168.1.0/24\n";
		return 0;
	}	
   print STDERR colored('INFO', 'bold blue'), "/", colored($switch{name}, 'bold'), ": OMG! Wirks! Trying ping..\n";

	my $reply = 0;
	# If you experience that it fails on switches that're up, try using a higher value
	# than 1 here (0 .. 5, e.g.) to increase the timeout. The cisco _could_ spend longer
	# time making the route work :)
	for my $count (0 .. 5)
	{
		if ($p->ping($zyxeloldip, 3))
		{
			$reply = 1;
			last;
		}
	}

	if (!$reply)
	{
		print STDERR colored('ERROR', 'red'), "/", colored($switch{name}, 'bold'), ": Waiting for zyxel on $switch{distro_ip}:$switch{vlan} TIMED OUT, wanted to set ip: $switch{ip}. Skipping!\n";

		if ( not stop_vlan($ios, $switch{vlan}) ) 
		{
			print STDERR colored('ERROR', 'red'), "/", colored($switch{vlan}, 'bold'), ": Unable to drop route to 192.168.1.0/24\n";
		} else {
		}
   		print STDERR colored('INFO', 'bold blue'), "/", colored($switch{name}, 'bold'), ": Dropped route to 192.168.1.0/24..\n";
		print LOG "Could not ping: $switch{name} [IP: $switch{ip}]\n";
		return 0;
	}

	$p->close();

	my $switch_handle = switch_connect('192.168.1.1');
	if (!$switch_handle)
	{
		print STDERR colored('ERROR', 'red'), "/", colored($switch{name}, 'bold'), ": Could not telnet to $switch{ip}\n";
		print LOG "Could not telnet to $switch{name} [IP: $switch{ip}]\n";
		return 0;
	}

	for my $cmd ("sys hostname es-3024", 'ip igmpsnoop enable')
	{
		switch_exec($cmd, $switch_handle);
	}

	my $pid = fork();
	if ($pid == 0) {
		switch_exec("ip ifconfig swif0 $switch{ip}/30", $switch_handle);
		$switch_handle->close();
		exit 0;
	}
	sleep 1;

	stop_vlan($ios, $switch{vlan});

	$ios->close();

	print STDERR colored('SUCCESS', 'green'), "/", colored($switch{name}, 'bold'), ": IP configured!\n\n";

	return 1;
}

## Collect switch ips

my %switchips;

open(SWITCHES, $switches) or die "Unable to open switches";
while(<SWITCHES>) {
	my ($ip, $net, $name) = split;

	if ($name =~ /e\d+-\d/) {
		die "We only support /26 nets for now. You wanted $net" if ($net ne "26");
		$switchips{$name} = $ip;
	}
}
close(SWITCHES);

sub switch_info {
	my ($switch, $distro, $port) = @_;

	$switch =~ /e(\d+)-(\d)/;
	my ($row, $place) = ($1, $2);
	my $ipnet = $switchips{$switch};
	my $vlan = $row . $place;
	my ($first, $second, $third, $fourth) = split(/\./, $ipnet);
	my $ip = "$first.$second.$third.".(int($fourth)+2);
	my $dip = $distro;

	return (name => $switch, row => $row, place => $place, network => $ipnet, vlan => $vlan, ip => $ip, distro_name => $distro, distro_ip => $dip, port => $port);
}

sub ip_run {
	my $only = shift;
	print colored('INFO', 'bold blue'), ": IP run\n";
	open(PATCHLIST, $patchlist) or die "Unable to open patchlist";
	while (<PATCHLIST>) {
		my ($switch_name, $distro, $port) = split;

		next if $switch_name !~ /e\d+-\d/;
		my %switch = switch_info($switch_name, $distro, $port);

		if (defined($only))
		{
			next if (lc($only) ne lc($switch_name) and $only ne $switch{ip});
		}

		set_ip(%switch);
	}
	close(PATCHLIST);
}

sub gateway_run {
	my $only = shift;
	print colored('INFO', 'bold blue'), ": Gateway run\n";
	open(PATCHLIST, $patchlist) or die "Unable to open patchlist";
	while (<PATCHLIST>) {
		my ($switch_name, $distro, $port) = split;

		next if $switch_name !~ /e\d+-\d/;
		my %switch = switch_info($switch_name, $distro, $port);

		if (defined($only))
		{
			next if (lc($only) ne lc($switch_name) and $only ne $switch{ip});
		}

		set_gateway(%switch);
	}
	close(PATCHLIST);
}

sub crontab_run {
	my $only = shift;
	print colored('INFO', 'bold blue'), ": Crontab run\n";
	open(PATCHLIST, $patchlist) or die "Unable to open patchlist";
	while (<PATCHLIST>) {
		my ($switch_name, $distro, $port) = split;

		next if $switch_name !~ /e\d+-\d/;
		my %switch = switch_info($switch_name, $distro, $port);

		if (defined($only))
		{
			next if (lc($only) ne lc($switch_name) and $only ne $switch{ip});
		}

		if (set_ip(%switch))
		{
			print colored('INFO', 'bold blue'), ": IP was set, now forking to set_gateway.\n";
			my $pid = fork();
			if ($pid == 0) {
				sleep(100);
				set_gateway(%switch);
				exit 0;
			}
		}
	}
	close(PATCHLIST);
}


if (@ARGV == 0 || ($ARGV[0] ne '-i' && $ARGV[0] ne '-g' && $ARGV[0] ne '-c')) {
	print "Usage: $0 <-i|-g|-c> [ip or sw#]\n";
	print " -i: Set IP address\n";
	print " -g: Set GW address\n";
	print " -c: Crontab mode (See which ones need setting IP, and if you set ip, set GW. Spawns # switches forks if run when no switches are configured ;)\n";
	print "\n";
	print " (does it for all zyxels, unless the optional second param is used. sw# like 'e12-3')\n";
	exit 1;
}

if ($ARGV[0] eq '-g') {
	gateway_run($ARGV[1]);
} elsif ($ARGV[0] eq '-i') {
	ip_run($ARGV[1]);
} else {
	crontab_run($ARGV[1]);
	while (wait() != -1) {}
}

close(LOG);
