#include <stdio.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <malloc.h>
#include <vector>
#include "flowpusher.h"

FlowPusher::FlowPusher(std::vector<flow_element> &flow) : flow(flow)
{
}

void FlowPusher::find_diff(double x, double &prev_x, unsigned long long y1, unsigned long long prev_y1, unsigned long long y2, unsigned long long prev_y2,
	unsigned long long &yf1, unsigned long long &yf2)
{
	// Heuristics: if last reading was more than ten minutes away, and both
	//             readings went down from last reading, we assume the counters
	//             were zeroed. If not, we assume wraparound.
	bool over_10mins = (x - prev_x >= 600);
	bool y1_wrapped = ((prev_y1 == 0 && y1 == 0) || y1 < prev_y1);
	bool y2_wrapped = ((prev_y2 == 0 && y2 == 0) || y2 < prev_y2);
	bool both_zero = (y1 == 0 && y2 == 0);	
	bool any_wide = (prev_y1 > 4294967296ULL || prev_y2 > 4294967296ULL);
	
	if ((over_10mins && (y1_wrapped || y2_wrapped)) || both_zero || (any_wide && (y1_wrapped || y2_wrapped))) {
		prev_y1 = 0;
		prev_y2 = 0;

		// our best estimate for when it was zeroed :-)
		prev_x = (prev_x + x) / 2;
	}

	if (y1 < prev_y1) {
		yf1 = (unsigned long long)(((unsigned long long)y1 + 4294967296ULL) - prev_y1);
	} else {
		yf1 = (unsigned long long)(y1 - prev_y1);
	}
	
	if (y2 < prev_y2) {
		yf2 = (unsigned long long)(((unsigned long long)y2 + 4294967296ULL) - prev_y2) / (x - prev_x);
	} else {
		yf2 = (unsigned long long)(y2 - prev_y2);
	}
}

void FlowPusher::push(double x, unsigned long long y1, unsigned long long y2)
{
	unsigned long long yf1, yf2;

	find_diff(x, prev_x, y1, prev_y1, y2, prev_y2, yf1, yf2);
	yf1 /= (x - prev_x);
	yf2 /= (x - prev_x);
		
	flow_element fe;
	fe.x = unsigned((prev_x + x)*0.5);
	fe.y1 = yf1;
	fe.y2 = yf2;

	min_x = std::min(min_x, fe.x);
	max_x = std::max(max_x, fe.x);
	
	min_y = std::min(min_y, fe.y1);
	max_y = std::max(max_y, fe.y1);
	
	min_y = std::min(min_y, fe.y2);
	max_y = std::max(max_y, fe.y2);

	flow.push_back(fe);

	prev_x = x;
	prev_y1 = y1;
	prev_y2 = y2;
}

void FlowPusher::reset(double x, unsigned long long y1, unsigned long long y2)
{
	flow.resize(0);
	
	min_y = 0;
	max_y = 10000000;
	
	max_x = time(NULL);
	min_x = max_x - 86400;
	
	prev_x = x;
	prev_y1 = y1;
	prev_y2 = y2;
}
