# convert 3D model from OBJ to custom LINES format
# Mike Erwin for @party 2018

import sys
import os
import struct
from math import sqrt, radians, cos


filename = sys.argv[1]
ifile = open(filename)

verts = []
triangles = []

for line in ifile:
	data = line.split()
	if data[0] == 'v':
		verts.append([float(x) for x in data[1:]])
	elif data[0] == 'f':
		v = [int(x) - 1 for x in data[1:]]
		assert(len(v) == 3) # triangles!
		triangles.append(v)

ifile.close()

print(len(verts), 'verts')
print(len(triangles), 'normals (from triangles)')
# print('verts =', verts)
# print('triangles =', triangles)

# == TODO ==
# [√] compute normal for each face
# [ ] deduplicate normals (probably the slowest part)
# [√] generate lines from triangles edges
# [√] handle non-manifold meshes
# [√] write output file:
#	- v count
#	- e count
#	- f count (100% triangles)
# [√] all in big-endian format
# file format is designed for mmap-ing, precompute as much as possible

def diff(a, b):
	return [a[0] - b[0], a[1] - b[1], a[2] - b[2]]

def dot(a, b):
	return a[0] * b[0] + a[1] * b[1] + a[2] * b[2]

def cross(a, b):
	c = [a[1]*b[2] - a[2]*b[1],
	     a[2]*b[0] - a[0]*b[2],
	     a[0]*b[1] - a[1]*b[0]]
	return c

def scale(s, v):
	return [s * x for x in v]

def mag(v):
	return sqrt(dot(v,v))

def normalize(v):
	return scale(1 / mag(v), v)

def trinorm(t):
	p0 = verts[t[0]]
	p1 = verts[t[1]]
	p2 = verts[t[2]]
	u = diff(p1, p0)
	v = diff(p2, p0)
	return normalize(cross(u, v))

normals = [trinorm(t) for t in triangles]

# generate edge data from triangles
edges = dict()
for idx, t in enumerate(triangles):
	for i in range(3):
		a = t[i]
		b = t[(i + 1) % 3]
		if a < b:
			edges[(a,b)] = [idx]
for idx, t in enumerate(triangles):
	for i in range(3):
		a = t[i]
		b = t[(i + 1) % 3]
		if a > b:
			key = (b,a)
			if key in edges:
				edges[key].append(idx)
			else:
				# edges[key] = [idx,idx] # rendering program can interpret this
				edges[key] = [idx]

inner_edges = []
outer_edges = []

for vi, ti in edges.items():
	if len(ti) == 1:
		outer_edges.append(vi)
	else:
		assert(len(ti) == 2)
		inner_edges.append((*vi, *ti))

assert(len(inner_edges) + len(outer_edges) == len(edges))

print(len(inner_edges), '+', len(outer_edges), '=', len(edges), 'edges')
# print(len(edges), 'edges')
# if rim_ct > 0:
# 	print(rim_ct, 'outer edges')

# TODO: store outer rim edges separately - no tri/normal needed!

ofile = open(os.path.splitext(filename)[0] + '.lines', 'wb')

def align(o):
	A = 32 # size of PowerPC G3 cacheline
	offset = o.tell() % A
	if offset > 0:
		o.write(b'~' * (A - offset))

# element counts
ofile.write(struct.pack('>IIII', len(verts), len(normals), len(inner_edges), len(outer_edges)))

# vertex data
align(ofile)
for v in verts:
	ofile.write(struct.pack('>fff', *v))

# normal data
align(ofile)
for n in normals:
	ofile.write(struct.pack('>fff', *n))

# edge data
align(ofile)
eformat = '>HH' if len(verts) <= 65535 else '>II'
eformat += 'HH' if len(triangles) <= 65535 else 'II'
for values in inner_edges:
	ofile.write(struct.pack(eformat, *values))

# border edge data
align(ofile)
eformat = '>HH' if len(verts) <= 65535 else '>II'
for values in outer_edges:
	ofile.write(struct.pack(eformat, *values))

ofile.close()
