"use strict";

class MainController {
    _camera;
    _cube;
    _renderer;
    _scene;
    _sceneManager;
    _startTime;
    _time;
    _spriteManager;
    _musicManager;
    _textManager;
    _textureManager;
    _materialManager;
    _loadingPromises = [];

    constructor(canvas) {

        this.initializeDebugInfo();
        this.initializeCanvas(canvas);
        this.initializeTextManager();
        this.initializeSprites();
        this.initializeTextures();
        this.initializeMaterials();
        this.initializeScenes();
        this.initializeMusic();
    }

    run() {
        console.log("Waiting for all resources to load");
        Promise.all(this._loadingPromises).then(() => {
            var debugOffset = 0;

            console.log("Resources loaded... action time!");
            this._musicManager.play(debugOffset);
            this._startTime = performance.now() - debugOffset;

            this.render();
        });
    }


    // private

    render() {
        requestAnimationFrame(this.render.bind(this));

        if (this._paused)
            this._startTime = performance.now() - this._time;
        else
            this._time = performance.now() - this._startTime;

        this._sceneManager.updateScenes(this._time);
        this._renderer.render(this._scene, this._camera);
        this._textManager.render(this._scene);
        // this.showDebugInfo();
    };

    initializeCanvas(canvas) {
        this._scene = new THREE.Scene();
        this._camera = new THREE.PerspectiveCamera(75, window.innerWidth / window.innerHeight, 0.1, 1000);

        this._renderer = new THREE.WebGLRenderer({
            canvas: canvas,
            alpha: true
        });
        this._renderer.setSize(window.innerWidth, window.innerHeight);

        this._camera.position.z = 5;
    }

    initializeTextManager() {
        this._textManager = new TextManager(this._scene); // TODO, PJ: FInish textmanager.
    }

    initializeScenes() {
        var me = this;
        var context = new SceneContext();
        context.spriteManager = this._spriteManager;
        context.textManager = this._textManager;
        context.textureManager = this._textureManager;
        context.materialManager = this._materialManager;
        var sceneManager = new SceneManager(this._scene, this._camera, context);
        sceneManager.registerScene(Scene1Controller, 0, 2800);     // I had a little t-rex
        sceneManager.registerScene(Scene2Controller, 2800, 4700);  // His name was Angry Tim
        sceneManager.registerScene(Scene3Controller, 4700, 6810);  // I put him in the lava
        sceneManager.registerScene(Scene4Controller, 6810, 8748);  // to see if he could swim
        sceneManager.registerScene(Scene5Controller, 8748, 11641);  // He drink up all the lava
        sceneManager.registerScene(Scene6Controller, 11641, 13491);  // his face turned blue and red
        sceneManager.registerScene(Scene7Controller, 13491, 16184);  // And now he's in his grave
        sceneManager.registerScene(Scene8Controller, 16184, 18170);  // with flowers on his bed
        sceneManager.registerScene(Scene9Controller, 18170, 22250);  // Flower flower flower, flower flowerbed. 
        sceneManager.registerScene(Scene10Controller, 22250, 27000); // FLower flower flower, cause Angry Tim was dead. 
        sceneManager.registerScene(Scene11Controller, 27000, 40000); // The end

        this._sceneManager = sceneManager;
    }

    initializeSprites() {
        var sprites = {
            "tim": "tim.png",
            "face": "face.png",
            "dead": "dead.png",
            "grave": "grave.png",
            "computer": "computer.png",
            "bottle": "bottle.png",
            "flower": "flower.png",
            "flower2": "flower2.png",
            "glass": "glass.png",
            "grrr1": "grrr1.png",
            "grrr2": "grrr2.png",
        };

        var spriteManager = new SpriteManager();
        for (const spriteName in sprites) {
            var promise = spriteManager.load(spriteName, `./assets/sprites/${sprites[spriteName]}`);
            this._loadingPromises.push(promise);
        }
        this._spriteManager = spriteManager;
    }

    initializeTextures() {
        var textures = {
            "noise": "noise.jpg",
            "lava": "lava.png",
            "grass": "grass.jpg",
        };

        var textureManager = new TextureManager();
        for (const textureName in textures) {
            var promise = textureManager.load(textureName, `./assets/textures/${textures[textureName]}`);
            this._loadingPromises.push(promise);
        }
        this._textureManager = textureManager;
    }

    initializeMaterials() {
        var materialManager = new MaterialManager();
        var promise = materialManager.loadShaders("lava", "./assets/shaders/lava.vert", "./assets/shaders/lava.frag");
        this._loadingPromises.push(promise);
        this._materialManager = materialManager;
    }

    initializeMusic() {
        var musicManager = new MusicManager(this._camera);
        var promise = musicManager.load("./assets/music/angry tim.mp3");
        this._loadingPromises.push(promise);
        this._musicManager = musicManager;
    }

    initializeDebugInfo() {
        this._debugEl = window.document.getElementById("debugInfo");
        var me = this;
        document.addEventListener('keydown', function (event) {
            if (event.keyCode == 32) {
                me._paused = !me._paused;
                if (me._paused)
                    me._musicManager.pause();
                else
                    me._musicManager.play();
            }
        });
    }

    showDebugInfo() {
        if (!this._debugEl)
            return;
        this._debugEl.innerHTML = Math.round(this._time);
    }

}