#include "material.h"
#include "SDL2/SDL_log.h"
#include <assert.h>

static void log_texture(cgltf_texture *tex, char *indent, cgltf_data *data) {
    SDL_Log("%s\tname %s\n", indent, tex->name);
    SDL_Log("%s\timage (%p):\n", indent, (void *)tex->image);
    if (tex->image) {
        SDL_Log("%s\t\tname %s\n", indent, tex->image->name);
        SDL_Log("%s\t\turi %s\n", indent, tex->image->uri);
        if (tex->image->buffer_view) {
            SDL_Log("%s\t\tbuffer_view %zu\n", indent,
                    tex->image->buffer_view - data->buffer_views);
        }
        SDL_Log("%s\t\tmime_type %s\n", indent, tex->image->mime_type);
        SDL_Log("%s\t\textensions_count %zu\n", indent,
                tex->image->extensions_count);
    }

    SDL_Log("%s\tsampler (%p):\n", indent, (void *)tex->sampler);
    if (tex->sampler) {
        SDL_Log("%s\t\tname %s\n", indent, tex->sampler->name);
        SDL_Log("%s\t\tmag_filter %d\n", indent, tex->sampler->mag_filter);
        SDL_Log("%s\t\tmin_filter %d\n", indent, tex->sampler->min_filter);
        SDL_Log("%s\t\twrap_s %d\n", indent, tex->sampler->wrap_s);
        SDL_Log("%s\t\twrap_t %d\n", indent, tex->sampler->wrap_t);
        SDL_Log("%s\t\textensions_count %zu\n", indent,
                tex->sampler->extensions_count);
    }

    SDL_Log("%s\thas_basisu %d\n", indent, tex->has_basisu);
    if (tex->has_basisu && tex->basisu_image) {
        SDL_Log("%s\tbasisu_image (%p)\n", indent, (void *)tex->basisu_image);
        SDL_Log("%s\t\tname %s\n", indent, tex->basisu_image->name);
        SDL_Log("%s\t\turi %s\n", indent, tex->basisu_image->uri);
        if (tex->basisu_image->buffer_view) {
            SDL_Log("%s\t\tbuffer_view %zu\n", indent,
                    tex->basisu_image->buffer_view - data->buffer_views);
        }
        SDL_Log("%s\t\tmime_type %s\n", indent, tex->basisu_image->mime_type);
        SDL_Log("%s\t\textensions_count %zu\n", indent,
                tex->basisu_image->extensions_count);
    }

    SDL_Log("%s\textensions_count %zu\n", indent, tex->extensions_count);
}

static void log_texture_view(cgltf_texture_view *tv, char *indent,
                             cgltf_data *data) {
    SDL_Log("%stexture %p:\n", indent, (void *)tv->texture);
    if (tv->texture) {
        log_texture(tv->texture, indent, data);
    }
    SDL_Log("%stexcoord %d\n", indent, tv->texcoord);
    SDL_Log("%sscale %f\n", indent, tv->scale);
    SDL_Log("%shas_transform %d\n", indent, tv->has_transform);
    if (tv->has_transform) {
        SDL_Log("%stransform.offset: %f %f\n", indent, tv->transform.offset[0],
                tv->transform.offset[1]);
        SDL_Log("%stransform.rotation: %f\n", indent, tv->transform.rotation);
        SDL_Log("%stransform.scale: %f %f\n", indent, tv->transform.scale[0],
                tv->transform.scale[1]);
        SDL_Log("%stransform.has_texcoord: %d\n", indent,
                tv->transform.has_texcoord);
        if (tv->transform.has_texcoord) {
            SDL_Log("%stransform.texcoord: %d\n", indent,
                    tv->transform.texcoord);
        }
    }
    SDL_Log("%sextensions_count %zu\n", indent, tv->extensions_count);
    SDL_Log("%sextensions %p\n", indent, (void *)tv->extensions);
}

static const char *alpha_mode_to_string(cgltf_alpha_mode am) {
    switch (am) {
    case cgltf_alpha_mode_opaque:
        return "opaque";
    case cgltf_alpha_mode_mask:
        return "mask";
    case cgltf_alpha_mode_blend:
        return "blend";
    case cgltf_alpha_mode_max_enum:;
    }
    return "unknown";
}

void material_log(cgltf_material *material, cgltf_data *data) {
    if (material == NULL) {
        return;
    }

    SDL_Log("\t\tmaterial: %p (%s)\n", (void *)material, material->name);

    SDL_Log("\t\t\thas_pbr_metallic_roughness %d\n",
            material->has_pbr_metallic_roughness);
    SDL_Log("\t\t\thas_pbr_specular_glossiness %d\n",
            material->has_pbr_specular_glossiness);
    SDL_Log("\t\t\thas_clearcoat %d\n", material->has_clearcoat);
    SDL_Log("\t\t\thas_transmission %d\n", material->has_transmission);
    SDL_Log("\t\t\thas_volume %d\n", material->has_volume);
    SDL_Log("\t\t\thas_ior %d\n", material->has_ior);
    SDL_Log("\t\t\thas_specular %d\n", material->has_specular);
    SDL_Log("\t\t\thas_sheen %d\n", material->has_sheen);
    SDL_Log("\t\t\thas_emissive_strength %d\n",
            material->has_emissive_strength);
    SDL_Log("\t\t\thas_iridescence %d\n", material->has_iridescence);

    if (material->has_pbr_metallic_roughness) {
        SDL_Log("\t\t\tpbr_metallic_roughness:\n");
        SDL_Log("\t\t\t\tbase_color_texture:\n");
        log_texture_view(&material->pbr_metallic_roughness.base_color_texture,
                         "\t\t\t\t\t", data);
        SDL_Log("\t\t\t\tmetallic_roughness_texture:\n");
        log_texture_view(
            &material->pbr_metallic_roughness.metallic_roughness_texture,
            "\t\t\t\t\t", data);

        SDL_Log("\t\t\t\tbase_color_factor %f %f %f %f\n",
                material->pbr_metallic_roughness.base_color_factor[0],
                material->pbr_metallic_roughness.base_color_factor[1],
                material->pbr_metallic_roughness.base_color_factor[2],
                material->pbr_metallic_roughness.base_color_factor[3]);

        SDL_Log("\t\t\t\tmetallic_factor %f\n",
                material->pbr_metallic_roughness.metallic_factor);
        SDL_Log("\t\t\t\troughness_factor %f\n",
                material->pbr_metallic_roughness.roughness_factor);
    }

    if (material->has_pbr_specular_glossiness) {
        SDL_Log("\t\t\tpbr_specular_glossiness:\n");
        SDL_Log("\t\t\t\tdiffuse_texture:\n");
        log_texture_view(&material->pbr_specular_glossiness.diffuse_texture,
                         "\t\t\t\t\t", data);
        SDL_Log("\t\t\t\tspecular_glossiness_texture:\n");
        log_texture_view(
            &material->pbr_specular_glossiness.specular_glossiness_texture,
            "\t\t\t\t\t", data);
        SDL_Log("\t\t\t\tdiffuse_factor %f %f %f %f\n",
                material->pbr_specular_glossiness.diffuse_factor[0],
                material->pbr_specular_glossiness.diffuse_factor[1],
                material->pbr_specular_glossiness.diffuse_factor[2],
                material->pbr_specular_glossiness.diffuse_factor[3]);
        SDL_Log("\t\t\t\tspecular_factor %f %f %f\n",
                material->pbr_specular_glossiness.specular_factor[0],
                material->pbr_specular_glossiness.specular_factor[1],
                material->pbr_specular_glossiness.specular_factor[2]);
        SDL_Log("\t\t\t\tglossiness_factor %f\n",
                material->pbr_specular_glossiness.glossiness_factor);
    }

    if (material->has_clearcoat) {
        SDL_Log("\t\t\tclearcoat:\n");

        SDL_Log("\t\t\t\tclearcoat_texture:\n");
        log_texture_view(&material->clearcoat.clearcoat_texture, "\t\t\t\t\t",
                         data);
        SDL_Log("\t\t\t\tclearcoat_roughness_texture:\n");
        log_texture_view(&material->clearcoat.clearcoat_roughness_texture,
                         "\t\t\t\t\t", data);
        SDL_Log("\t\t\t\tclearcoat_normal_texture:\n");
        log_texture_view(&material->clearcoat.clearcoat_normal_texture,
                         "\t\t\t\t\t", data);

        SDL_Log("\t\t\t\tclearcoat_factor %f\n",
                material->clearcoat.clearcoat_factor);
        SDL_Log("\t\t\t\tclearcoat_roughness_factor %f\n",
                material->clearcoat.clearcoat_roughness_factor);
    }

    if (material->has_ior) {
        SDL_Log("\t\t\tior: %f\n", material->ior.ior);
    }

    if (material->has_specular) {
        SDL_Log("\t\t\tspecular:\n");
        SDL_Log("\t\t\t\tspecular_texture:\n");
        log_texture_view(&material->specular.specular_texture, "\t\t\t\t\t",
                         data);
        SDL_Log("\t\t\t\tspecular_color_texture:\n");
        log_texture_view(&material->specular.specular_color_texture,
                         "\t\t\t\t\t", data);
        SDL_Log("\t\t\t\tspecular_color_factor %f %f %f\n",
                material->specular.specular_color_factor[0],
                material->specular.specular_color_factor[1],
                material->specular.specular_color_factor[2]);
        SDL_Log("\t\t\t\tspecular_factor %f\n",
                material->specular.specular_factor);
    }

    if (material->has_sheen) {
        SDL_Log("\t\t\tsheen:\n");
        SDL_Log("\t\t\t\tsheen_color_texture:\n");
        log_texture_view(&material->sheen.sheen_color_texture, "\t\t\t\t\t",
                         data);
        SDL_Log("\t\t\t\tsheen_color_factor %f %f %f\n",
                material->sheen.sheen_color_factor[0],
                material->sheen.sheen_color_factor[1],
                material->sheen.sheen_color_factor[2]);
        SDL_Log("\t\t\t\tsheen_roughness_texture:\n");
        log_texture_view(&material->sheen.sheen_roughness_texture, "\t\t\t\t\t",
                         data);
        SDL_Log("\t\t\t\tsheen_roughness_factor %f\n",
                material->sheen.sheen_roughness_factor);
    }

    if (material->has_transmission) {
        SDL_Log("\t\t\ttransmission:\n");
        SDL_Log("\t\t\t\ttransmission_texture:\n");
        log_texture_view(&material->transmission.transmission_texture,
                         "\t\t\t\t\t", data);
        SDL_Log("\t\t\t\ttransmission_factor %f\n",
                material->transmission.transmission_factor);
    }

    if (material->has_volume) {
        SDL_Log("\t\t\tvolume:\n");
        SDL_Log("\t\t\t\tthickness_texture:\n");
        log_texture_view(&material->volume.thickness_texture, "\t\t\t\t\t",
                         data);
        SDL_Log("\t\t\t\tthickness_factor %f\n",
                material->volume.thickness_factor);
        SDL_Log("\t\t\t\tattenuation_color %f %f %f\n",
                material->volume.attenuation_color[0],
                material->volume.attenuation_color[1],
                material->volume.attenuation_color[2]);
        SDL_Log("\t\t\t\tattenuation_distance %f\n",
                material->volume.attenuation_distance);
    }

    if (material->has_emissive_strength) {
        SDL_Log("\t\t\temissive_strength %f",
                material->emissive_strength.emissive_strength);
    }

    if (material->has_iridescence) {
        SDL_Log("\t\t\tiridescence\n");
        SDL_Log("\t\t\t\tiridescence_factor %f",
                material->iridescence.iridescence_factor);

        SDL_Log("\t\t\t\tiridescence_texture:\n");
        log_texture_view(&material->iridescence.iridescence_texture,
                         "\t\t\t\t\t", data);
        SDL_Log("\t\t\t\tiridescence_ior %f",
                material->iridescence.iridescence_ior);
        SDL_Log("\t\t\t\tiridescence_thickness_min %f",
                material->iridescence.iridescence_thickness_min);
        SDL_Log("\t\t\t\tiridescence_thickness_max %f",
                material->iridescence.iridescence_thickness_max);
        SDL_Log("\t\t\t\tiridescence_thickness_texture:\n");
        log_texture_view(&material->iridescence.iridescence_thickness_texture,
                         "\t\t\t\t\t", data);
    }

    SDL_Log("\t\t\tnormal_texture:\n");
    log_texture_view(&material->normal_texture, "\t\t\t\t", data);
    SDL_Log("\t\t\tocclusion_texture:\n");
    log_texture_view(&material->occlusion_texture, "\t\t\t\t", data);
    SDL_Log("\t\t\temissive_texture:\n");
    log_texture_view(&material->emissive_texture, "\t\t\t\t", data);
    SDL_Log("\t\t\temissive_factor %f %f %f\n", material->emissive_factor[0],
            material->emissive_factor[1], material->emissive_factor[2]);
    SDL_Log("\t\t\talpha_mode: %s\n",
            alpha_mode_to_string(material->alpha_mode));
    SDL_Log("\t\t\talpha_cutoff %f\n", material->alpha_cutoff);
    SDL_Log("\t\t\tdouble_sided %d\n", material->double_sided);
    SDL_Log("\t\t\tunlit %d\n", material->unlit);
    SDL_Log("\t\t\textensions_count %zu\n", material->extensions_count);
}

void material_default(material_t *material_out) {
    memset(material_out, 0, sizeof(material_t));
    material_out->base_color_factor[0] = 0.5;
    material_out->base_color_factor[1] = 0.5;
    material_out->base_color_factor[2] = 0.5;
    material_out->base_color_factor[3] = 1.;
    material_out->roughness_factor = 0.5;
}

static size_t texture_index(cgltf_texture_view *tv, cgltf_data *data) {
    if (tv == NULL || tv->texture == NULL || tv->texture->image == NULL) {
        return data->images_count; // Last image is null image
    }

    // TODO account for other texcoords than 0
    assert(tv->texcoord == 0);
    // TODO emulate texture views
    assert(tv->scale == 1.);
    assert(tv->has_transform == 0);
    assert(tv->extensions_count == 0);
    assert(tv->texture->extensions_count == 0);

    // TODO implement samplers
    return tv->texture->image - data->images;
}

void material_init(material_t *material_out, cgltf_material *material,
                   cgltf_data *data) {
    if (material->has_pbr_specular_glossiness) {
        SDL_Log("Warning: unsupported material with specular and glossiness\n");
    }

    if (material->has_pbr_metallic_roughness) {
        cgltf_pbr_metallic_roughness *pbr = &material->pbr_metallic_roughness;

        material_out->base_color_texture =
            texture_index(&pbr->base_color_texture, data);
        material_out->metallic_roughness_texture =
            texture_index(&pbr->metallic_roughness_texture, data);
        memcpy(material_out->base_color_factor, pbr->base_color_factor,
               sizeof(GLfloat) * 4);
        material_out->metallic_factor = pbr->metallic_factor;
        material_out->roughness_factor = pbr->roughness_factor;
    }

    material_out->normal_texture =
        texture_index(&material->normal_texture, data);
    material_out->occlusion_texture =
        texture_index(&material->occlusion_texture, data);
    material_out->emissive_texture =
        texture_index(&material->emissive_texture, data);

    memcpy(material_out->emissive_factor, material->emissive_factor,
           sizeof(GLfloat) * 3);
    material_out->double_sided = material->double_sided;
    material_out->unlit = material->unlit;
}
