#include "texture.h"
#include "SDL2/SDL_log.h"

#define STB_IMAGE_IMPLEMENTATION
#include "stb_image.h"

static GLuint load_texture(unsigned char *image, int width, int height,
                           int bytes_per_pixel) {
    GLuint texture;
    glGenTextures(1, &texture);
    glBindTexture(GL_TEXTURE_2D, texture);
    glTexImage2D(GL_TEXTURE_2D, 0, GL_RGB, width, height, 0, GL_RGB,
                 GL_UNSIGNED_BYTE, image);
    glGenerateMipmap(GL_TEXTURE_2D);
    return texture;
}

GLuint texture_init(void *data, size_t len) {
    int width, height, native_bpp,
        bytes_per_pixel = 3; // Actual bytes per pixel is 3 since we set it here
    unsigned char *image = NULL;

    if (len == 0) {
        image = stbi_load(data, &width, &height, &native_bpp, bytes_per_pixel);
    } else {
        image = stbi_load_from_memory(data, len, &width, &height, &native_bpp,
                                      bytes_per_pixel);
    }

    if (image == NULL) {
        SDL_Log("Failed to load image %s\n", len == 0 ? (char *)data : "");
        SDL_Log("%s\n", stbi_failure_reason());
        return 0;
    }

    if (native_bpp != bytes_per_pixel) {
        SDL_Log("Requested %d bytes per pixel, image has %d\n", bytes_per_pixel,
                native_bpp);
    }

    GLuint texture = load_texture(image, width, height, bytes_per_pixel);

    stbi_image_free(image);

    return texture;
}

GLuint texture_init_file(const char *path) {
    return texture_init((void *)path, 0);
}
