#![allow(unused_variables, dead_code, reason = "Work in progress")]

use crate::{RESOURCE_DIR, SHADER_DIR};
use anyhow::{Context, Result, anyhow};
use image::{GenericImageView, ImageReader};
use sdl3::gpu::{
    CopyPass, Device, ShaderStage, Texture, TextureCreateInfo, TextureFormat, TextureRegion,
    TextureTransferInfo, TextureType, TransferBufferUsage,
};
use std::{collections::HashMap, ffi::OsStr, path::Path};

pub struct ResourceManager {
    gpu: Device,
    textures: HashMap<Box<str>, Texture<'static>>,
}

impl ResourceManager {
    #[must_use]
    pub fn new(gpu: &Device) -> Self {
        Self {
            gpu: Device::clone(gpu),
            textures: HashMap::new(),
        }
    }

    fn load_texture(&self, filename: &str, copy_pass: &CopyPass) -> Result<Texture<'static>> {
        let image = ImageReader::open(Path::new(RESOURCE_DIR).join(filename))
            .with_context(|| format!("Can't open {filename}"))?
            .decode()
            .with_context(|| format!("Can't decode {filename}"))?;
        let (width, height) = image.dimensions();
        let image = image.to_rgba8();
        let bytes = 4 * width * height;

        let texture = self.gpu.create_texture(
            TextureCreateInfo::new()
                .with_format(TextureFormat::R8g8b8a8UnormSrgb)
                .with_type(TextureType::_2D)
                .with_width(width)
                .with_height(height)
                .with_layer_count_or_depth(1)
                .with_num_levels(1)
                .with_usage(sdl3::gpu::TextureUsage::Sampler),
        )?;

        let transfer_buffer = self
            .gpu
            .create_transfer_buffer()
            .with_size(bytes)
            .with_usage(TransferBufferUsage::Upload)
            .build()?;

        let mut buffer_mem = transfer_buffer.map::<u8>(&self.gpu, false);
        buffer_mem.mem_mut().copy_from_slice(image.as_raw());
        buffer_mem.unmap();

        copy_pass.upload_to_gpu_texture(
            TextureTransferInfo::new()
                .with_transfer_buffer(&transfer_buffer)
                .with_offset(0),
            TextureRegion::new()
                .with_texture(&texture)
                .with_layer(0)
                .with_width(width)
                .with_height(height)
                .with_depth(1),
            false,
        );

        Ok(texture)
    }

    fn transfer_texture(
        &self,
        filename: &str,
        copy_pass: Option<&CopyPass>,
    ) -> Result<Texture<'static>> {
        if let Some(copy_pass) = copy_pass {
            self.load_texture(filename, copy_pass)
        } else {
            let mut command_buffer = self.gpu.acquire_command_buffer()?;
            let copy_pass = self.gpu.begin_copy_pass(&command_buffer)?;
            match self.load_texture(filename, &copy_pass) {
                Ok(texture) => {
                    self.gpu.end_copy_pass(copy_pass);
                    command_buffer.submit()?;
                    Ok(texture)
                }
                Err(e) => {
                    command_buffer.cancel();
                    Err(e)
                }
            }
        }
    }

    fn texture(
        &mut self,
        filename: &str,
        copy_pass: Option<&CopyPass>,
    ) -> Result<Texture<'static>> {
        if let Some(texture) = self.textures.get(filename).cloned() {
            return Ok(texture);
        }

        let texture = self.transfer_texture(filename, copy_pass)?;
        self.textures
            .insert(String::from(filename).into_boxed_str(), texture.clone());
        Ok(texture)
    }
}

pub fn load_shader(filename: impl AsRef<Path>) -> Result<(Vec<u8>, ShaderStage)> {
    let filename = filename.as_ref();
    let stage = infer_shader_stage(filename)?;

    // When shaderc feature is enabled, compile from GLSL
    // TODO: Send warning string to text thread(s)
    #[cfg(feature = "shaderc")]
    let (spv, _warn) = compile_shader(Path::new(SHADER_DIR).join(filename), stage)?;

    // When shaderc feature is disabled, load precompiled SPIR-V
    #[cfg(not(feature = "shaderc"))]
    let spv = {
        use std::{fs::File, io::Read, path::PathBuf};

        let mut path = Path::new(RESOURCE_DIR)
            .join(SHADER_DIR)
            .join(filename)
            .into_os_string();
        path.push(".spv"); // TODO: Omit this clumsy workaround when .with_added_extension is stable
        let path = PathBuf::from(path);

        let mut buf = Vec::with_capacity(4096);
        File::open(&path)
            .with_context(|| format!("Can't open {}", path.display()))?
            .read_to_end(&mut buf)
            .with_context(|| format!("Failed to read {}", path.display()))?;
        buf
    };

    Ok((spv, stage))
}

#[cfg(feature = "shaderc")]
pub fn compile_shader(
    path: impl AsRef<Path>,
    stage: ShaderStage,
) -> Result<(Vec<u8>, Option<String>)> {
    use shaderc::{
        CompileOptions, Compiler, EnvVersion, OptimizationLevel, ShaderKind, SourceLanguage,
        SpirvVersion, TargetEnv,
    };

    let path = path.as_ref();

    // TODO: maybe cache this compiler
    let compiler = Compiler::new().context("Can't initialize shader compiler")?;

    // Configuration
    let mut options = CompileOptions::new().context("Can't initialize shader compiler options")?;
    options.set_source_language(SourceLanguage::GLSL);
    options.set_target_env(TargetEnv::Vulkan, EnvVersion::Vulkan1_0 as _);
    options.set_target_spirv(SpirvVersion::V1_0);
    options.set_optimization_level(OptimizationLevel::Performance);
    options.set_include_callback(|name, ty, source, depth| {
        use shaderc::{IncludeType, ResolvedInclude};

        if depth > 1 {
            return Err("Nested #include is not currently supported".into());
        }

        let path = match ty {
            IncludeType::Relative => path.parent().unwrap().join(Path::new(name)),
            IncludeType::Standard => Path::new(SHADER_DIR).join("lib").join(name),
        };

        let content = std::fs::read_to_string(&path)
            .map_err(|e| format!("Can't read {}: {}", path.display(), e))?;

        Ok(ResolvedInclude {
            resolved_name: format!("{}", path.display()),
            content,
        })
    });

    // Read shader source
    let glsl =
        std::fs::read_to_string(path).with_context(|| format!("Can't read {}", path.display()))?;

    // Compile
    let result = compiler
        .compile_into_spirv(
            &glsl,
            match stage {
                ShaderStage::Vertex => ShaderKind::Vertex,
                ShaderStage::Fragment => ShaderKind::Fragment,
            },
            &path.to_string_lossy(),
            "main",
            Some(&options),
        )
        .with_context(|| format!("{}: Shader compiler error", path.display()))?;
    Ok((
        Vec::from(result.as_binary_u8()),
        (result.get_num_warnings() > 0).then(|| result.get_warning_messages()),
    ))
}

pub fn infer_shader_stage(filename: impl AsRef<Path>) -> Result<ShaderStage> {
    let filename = filename.as_ref();
    match filename.extension().map(OsStr::as_encoded_bytes) {
        Some(b"vert") => Ok(ShaderStage::Vertex),
        Some(b"frag") => Ok(ShaderStage::Fragment),
        _ => Err(anyhow!(
            "Can't infer shader stage from {}",
            filename.display()
        )),
    }
}
