; Area 51 - Amstrad CPC version
; Started on 16th November 2006
; Finished on 16th December 2006

; Locations of tables used by various routines.

txtbyt_hb equ &80
redbyt_hb equ txtbyt_hb+1
sprbyt_hb equ redbyt_hb+1
scrcur equ (txtbyt_hb*256)-576

; Definitions of constants.

ROOMNO equ 31              ; room number bit mask.
ALNTYP equ 224             ; alien type bit mask.
NUMSCR equ 7               ; number of screens.
SCRLEN equ NUMSCR*576      ; size of screen data.
SCDATA equ (txtbyt_hb*256)-SCRLEN*(NUMSCR+1)        ; address to which screen data is
                                                    ; expanded.


org (txtbyt_hb*256)-3      ; start address of program.

jp start

; Lookup tables used by various subroutines; all of these need to be
; page-aligned (i.e. start at &xx00)

include 'chrbyt.asm'
include 'redbyt.asm'
include 'sprbyt.asm'

; The Spectrum's font (characters 32 to 127), consisting of 768 (&300) bytes.

include 'specfont.asm'

sprbuf defs 5              ; buffer for holding one line of a sprite; this
                           ; needs to be within a page boundary.

; Table of horizontal collision displacements for each frame (8 bytes); this
; needs to be within a page boundary.

coldis defb 4,11,3,12,4,11,5,10

; Table of sprite pointers for player (16 bytes); this needs to be within a
; page boundary.

playfr defw playl0,playl1,playl0,playl2
       defw playr0,playr1,playr0,playr2

; Table of sprite pointers for patrolling aliens (32 bytes); this needs to be
; within a page boundary.

alnptr defw cart1,cart0,cart2,cart0        ; cart.
       defw frog0,frog1,frog0,frog2        ; frog.
       defw spin0,spin1,spin2,spin3        ; spinning thing.
       defw comp0,comp1,comp2,comp1        ; computer.

; AY register data for playing sound effects and music (14 bytes); this needs
; to be within a page boundary.

snddat defw 0              ; tone registers, channel A.
       defw 0              ; channel B tone registers.
       defw 0              ; as above, channel C.
       defb 0              ; white noise period.
       defb 60             ; tone/noise mixer control.
       defb 16             ; channel A amplitude/envelope generator.
       defb 16             ; channel B amplitude/envelope.
       defb 0              ; channel C amplitude/envelope.
       defw 6000           ; duration of each note.
       defb 0

; Sprites are stored here.

include 'sprites.asm'


; Set the mode, inks and the interrupt routine.

start  di                  ; disable interrupts.
       ld bc,&7f8d         ; use MODE 1 for graphics.
       out (c),c
       ld bc,&7f00         ; use gate array (&7fxx) to set inks and start at
                           ; ink 0.
       ld e,&54            ; black.
start0
       out (c),c           ; select ink.
       out (c),e           ; change value of ink.
       inc c               ; go to next ink.
       ld a,c
       cp 4                ; there are 4 inks to set in MODE 1.
       jr nz,start0        ; repeat for remaining inks.
       ld c,&10            ; select border (effectively ink 16).
       out (c),c
       out (c),e           ; change ink value of border.
       call waitfr         ; wait for frame flyback.

; Change the screen to Spectrum size using the CRTC registers (32 characters
; wide x 24 characters deep)

       ld b,&bc            ; CRTC output addresses are &bcxx and &bdxx
       ld a,6              ; 6 registers to be set.
       ld hl,crtcdt        ; address of register data.

crtc   ld c,(hl)           ; get the register and store it in c.
       out (c),c           ; output it to &bcxx.
       inc b               ; the new register value will be output to &bdxx.
       inc hl              ; move to the register value.
       ld e,(hl)           ; get the value and store it in e.
       out (c),e           ; output it to &bdxx.
       dec b               ; reset b to &bc for output of the next register.
       inc hl              ; move to the next register in the data.
       dec a               ; count number of registers remaining in data.
       jr nz,crtc          ; repeat.

; Reset all game variables to zero and clear the entire screen.

       ld hl,dbyte         ; start address of variable data.
       ld (hl),0           ; reset to zero.
       ld d,h              ; set de to address following hl.
       ld e,l
       inc de
       ld bc,termin-dbyte-1
       ldir                ; clear variables.

       ld hl,&c000         ; start address of screen.
       ld (hl),0
       ld d,h              ; set de to address following hl
       ld e,l
       inc e
       ld bc,&3fff         ; &4000-1 bytes to clear.
       ldir                ; clear screen.

; The screens are compressed and need expanding into RAM at 32768.

       ld hl,lev1sc        ; level data to decompress.
       call decomp         ; decompress screen layouts.
       ld hl,0
       ld (hisc),hl        ; reset the high score.
       xor a
       ld (scno),a         ; screen number.
       call droom          ; display first screen so screen isn't blank when
                           ; game starts for the first time.

; Set up the display at the bottom of the screen.

       call prbar          ; print horizontal bar.

       defb 33,164,12      ; load coordinates of extra lives sprite into hl.
       ld (dispx),hl       ; sprite routine coordinates.
       ld hl,playr2        ; player sprite - facing right.
       call sprite         ; show lives icon.

       defb 33,168,32      ; load coordinates of number of lives into hl.
       ld (dispx),hl       ; set coordinates.
       ld c,2              ; use paper 0, pen 2.
       ld a,'5'            ; player starts with 5 lives.
       call prchar         ; display character.

       defb 33,168,72      ; coordinates to print score string.
       ld (dispx),hl       ; set coordinates.
       ld ix,sctxt         ; score text.
       ld a,2              ; use paper 0, pen 2.
       call prstr          ; print string.
       ld a,168            ; column to print high score string.
       ld (dispy),a        ; set coordinates.
       ld ix,hitxt         ; high score text.
       ld a,2              ; use paper 0, pen 2.
       call prstr          ; print string.

; Set up interrupt routine.

       ld a,&c3            ; &c3 = jp instruction.
       ld (&38),a          ; interrupt is called at location &38 on a CPC.
       ld hl,interr        ; address of interrupt routine.
       ld (&39),hl
       call waitfr         ; wait for frame flyback.
       xor a               ; clear interrupt counter.
       ld (inter0+1),a
       ei                  ; enable interrupt routine.


; Okay, levels and graphics are in position.
; Let's start with the intro screen code.

; Change all pixels on screen to pen 1 using a lookup table.

intro  ld a,1
       ld (musoff),a       ; turn music off (0=on, 1=off).

       call waitfr         ; wait for frame flyback.
       ld hl,inkta0        ; select table with all inks set to black.
       ld (inter1+1),hl    ; set inks for top half of screen.

       ld h,redbyt_hb      ; high byte of start address of lookup table.
       ld de,&c000         ; start address of screen.
       ld a,8              ; screen consists of 8 blocks.

intro0 ex af,af'           ; store number of blocks remaining.
       ld bc,1152          ; each block consists of 1152 bytes (= 64 x 18).
intro1 ld a,(de)           ; get screen byte.
       ld l,a              ; add offset to lookup table address.
       ld a,(hl)           ; get byte from table.
       ld (de),a           ; write byte to screen.
       inc de              ; go to next screen byte.
       dec bc              ; decrement counter.
       ld a,b              ; loop counter high byte.
       or c                ; combine with low byte for zero check.
       jr nz,intro1        ; repeat for remaining bytes in block.

       ld a,h              ; store high byte of lookup table.
       ld hl,&380          ; set screen address for next block by adding
       add hl,de           ; &380.
       ex de,hl            ; screen address now in de.
       ld h,a              ; get high byte of lookup table.

       ex af,af'           ; get number of blocks remaining.
       dec a               ; decrement it.
       jr nz,intro0        ; repeat for remaining blocks.

; Write a message to the screen.
; First, write three lines consisting entirely of spaces, in order to create a
; blank window to write the game title and display an animations of Fizzog.

introx equ 32              ; coordinate for first line of message.

       ld ix,intxt1        ; string of spaces, used to blank out one line of
                           ; screen.
       defb 33,introx,68   ; coordinates to print string.
       ld (dispx),hl       ; set coordinates.
       xor a               ; use pen 0, paper 0.
       call prstr          ; print first blank line.
       ld ix,intxt1
       ld a,introx+8
       ld (dispx),a        ; set coordinates for second blank line.
       xor a
       call prstr          ; print second blank line.
       ld ix,intxt2        ; game title.
       ld a,92
       ld (dispy),a
       ld a,7              ; use paper 1, pen 3 (1*4+3 = 7).
       call prstr          ; print game title within blank window.
       ld ix,intxt1
       defb 33,introx+16,68   ; coordinates to print third blank line.
       ld (dispx),hl
       xor a
       call prstr          ; print third blank line.

; Now print the rest of the message.

       ld ix,intxt3
       defb 33,introx+32,28
       ld (dispx),hl
       ld a,3
       call prstr
       ld ix,intxt4
       ld a,introx+48
       ld (dispx),a
       ld a,3
       call prstr
       ld ix,intxt5
       defb 33,introx+56,40
       ld (dispx),hl
       ld a,3
       call prstr
       ld ix,intxt6
       defb 33,introx+72,44
       ld (dispx),hl
       ld a,2
       call prstr

       call waitfr         ; wait for frame flyback.
       ld hl,inkta1        ; address of ink table for top half of screen.
       ld (inter1+1),hl    ; set inks.

; Wait for the RETURN key to be pressed and display animations of Fizzog
; 'walking' left and right while remaining stationary on the screen.

       defb 33,introx+4,72 ; coordinates to print Fizzog walking left.
       ld (dispx),hl       ; set coordinates.
       ld hl,(playfr)
       ld (intr2a+1),hl    ; set address of sprite data for first frame of
                           ; Fizzog walking left.
       call sprite         ; print sprite.
       defb 33,introx+4,168 ; coordinates to print Fizzog walking left.
       ld (dispx),hl       ; set coordinates.
       ld hl,(playfr+8)
       ld (intr2b+1),hl    ; set address of sprite data for first frame of
                           ; Fizzog walking right.
       call sprite         ; print sprite.
       ld d,0              ; d contains frame number to use (0-3).

intro2 defb 33,introx+4,72 ; coordinates to print Fizzog walking left.
       ld (dispx),hl       ; set coordinates.
       xor a               ; direction facing, 0=left, 8=right.
       ld (direct),a       ; set direction.
       call wait5          ; wait for raster to reach bottom area of screen.
       push de             ; store frame number.
intr2a ld hl,0             ; address of sprite data for previous frame
                           ; (modified).
       call sprite         ; redisplay sprite with exclusive-or to delete it.
       pop de              ; get frame number.

       ld a,d              ; a contains frame number.
       inc a               ; go to next frame.
       and 3               ; if frame number is 3, next frame will be 0.
       ld d,a
       push de             ; store number of next frame.
       rlca                ; multiply by 2.
       call calcpf         ; get address of sprite data for frame.
       ld (intr2a+1),hl    ; set address so that it will be removed in next
                           ; iteration of loop.
       call sprite         ; draw new sprite.
       pop de              ; get frame number.

       defb 33,introx+4,168 ; coordinates to print Fizzog walking right.
       ld (dispx),hl       ; set coordinates.
       ld a,8              ; direction facing, 0=left, 8=right.
       ld (direct),a       ; set direction.
       call wait5          ; wait for raster to reach bottom area of screen.
       push de             ; store frame number.
intr2b ld hl,0             ; address of sprite data for previous frame
                           ; (modified).
       call sprite         ; redisplay sprite with exclusive-or to delete it.
       pop de              ; get frame number.

       ld a,d              ; a contains frame number.
       push de             ; store frame number.
       rlca                ; multiply by 2.
       call calcpf         ; get address of sprite data for frame.
       ld (intr2b+1),hl    ; set address so that it will be removed in next
                           ; iteration of loop.
       call sprite         ; draw new sprite.
       pop de              ; get frame number.

       ld e,2
intro3 call wait5          ; wait for raster to reach bottom area of screen.
       halt                ; another short delay.
       call getkey         ; get keypress.
       ld a,(keytab+2)     ; get info on keyboard matrix line 2.
       and 4               ; bit 2 contains info on key 2*8+2 = 18 (RETURN).
       jp z,initgm         ; if bit is set, it means key wasn't pressed.
       dec e
       jr nz,intro3
       jr intro2

; Strings used in the intro screen.

intxt1 defm '               '
       defb 0
intxt2 defm ' AREA 51 '
       defb 0
intxt3 defb 127
       defm ' 2004 Jonathan Cauldwell'
       defb 0
intxt4 defm 'Amstrad CPC conversion by'
       defb 0
intxt5 defm 'Nicholas Campbell 2006'
       defb 0
intxt6 defm 'Press RETURN to start'
       defb 0


; Player pressed RETURN so initiate the game sequence.

initgm ld hl,inkta0        ; select table with all inks set to black.
       ld (inter1+1),hl    ; set inks for top half of screen.
       ld (inter2+1),hl    ; set inks for bottom half of screen.
       call waitfr         ; wait for frame flyback.

; Clear the top half of the screen but leave the bottom half intact.

       ld hl,&c000         ; start address of screen.
       ld a,8              ; screen consists of 8 blocks.
clrscr ld (hl),l           ; set first byte of block to zero.
       ld d,h              ; set de to address following hl.
       ld e,l
       inc e
       ld bc,&47f          ; &480-1 bytes to fill in each block.
       ldir                ; clear block.
       ld de,&381          ; set screen address for next block.
       add hl,de
       dec a               ; decrement number of blocks remaining.
       jr nz,clrscr        ; repeat for remaining blocks.

       ld (scno),a         ; screen number (a is already zero).
       ld h,a              ; zero in h.
       ld l,a              ; zero in hl pair.
       ld (score),hl       ; score.
       ld a,5              ; number of lives.
       ld (lives),a        ; set lives counter.

; This is the point at which we start each new screen.

rstart xor a               ; zeroise accumulator.
       ld (numobj),a       ; number of objects to collect.
       ld (dflag),a        ; player dead flag.
       ld (toofar),a       ; fallen too far flag.
       ld (fallf),a        ; reset "in mid-air" flag.

       ld a,7              ; set counter used for playing music so that music
       ld (chcntr),a       ; starts at the right moment when game starts.
       ld a,60
       ld (snddat+7),a     ; sound channels A and B on, channel C off.

; Clear alien buffer.

       ld hl,albuff        ; alien buffer.
       ld (hl),128         ; set first alien to off.
       ld de,albuff+1      ; alien buffer.
       ld bc,17            ; size of buffer.
       ldir                ; set all bytes to 128 to switch aliens off.

; Find aliens for present room.

       ld a,(scno)         ; screen number.
       ld c,a              ; put it in c for comparisons.
       ld ix,albuff        ; data for present room.
       ld hl,alndat        ; alien data table.
fndal  ld a,(hl)           ; get alien entry.
       cp 255              ; is this the end of the table?
       jr z,draw           ; yes - we're done.
       and ROOMNO          ; mask off the non-room bits.
       cp c                ; is alien located in this room?
       jr z,fndal0         ; yes, put him in alien buffer.
       ld de,5             ; 5 bytes/alien.
       add hl,de           ; look for next alien.
       jr fndal            ; go round again.
fndal0 ld a,(hl)           ; get alien type.
       and ALNTYP          ; mask off non-type bits.
       ld (ix),a           ; put in alien buffer.
       inc hl              ; next byte.
       ld a,(hl)           ; starting x.
       ld (ix+2),a         ; alien starting x coord.
       ld b,a              ; remember starting x.
       inc hl              ; next byte.
       ld a,(hl)           ; starting y.
       ld d,a              ; store starting y.
       ld (ix+3),a         ; put into alien buffer.
       inc hl              ; next byte.
       ld a,(hl)           ; finishing x.
       cp b                ; is it the same as the starting x coord?
       jr z,fndal1         ; yes, so alien patrols left/right.
       ld (ix+1),3         ; patrolling up/down.
       ld (ix+4),b         ; starting x position.
       inc hl              ; end y coord.
fndal3 ld (ix+5),a         ; end coordinate.
       inc hl              ; first byte of next alien in table.
       ld de,6
       add ix,de           ; next 6 bytes of buffer.
       jr fndal            ; repeat until all aliens searched.
fndal1 ld (ix+1),2         ; patrolling left/right.
       ld (ix+4),d         ; starting y position.
       inc hl              ; next byte of table.
       ld a,(hl)           ; finishing y.
       jr fndal3           ; put into alien buffer.

draw   call droom          ; draw current room.

; Display the name of the screen below the playfield.
; First, find the miscellaneous data for this screen.

       call prbar          ; print horizontal bar before printing screen name.
       ld a,(scno)         ; screen number.
       rlca                ; shift left twice to
       rlca                ; multiply by 4.
       ld e,a
       ld d,0              ; put a*4 in de.
       ld hl,misc          ; miscellaneous data.
       add hl,de           ; point to miscellaneous data for this screen.
       ld e,(hl)           ; x coord of player start.
       inc hl              ; next byte.
       ld d,(hl)           ; y coord of player start.
       inc hl              ; next byte.
       ld (newx),de        ; player starts here.
       ld e,(hl)           ; screen title low byte.
       inc hl              ; next byte.
       ld d,(hl)           ; screen title high byte.

; Screen name now pointed to by de registers.

       ld l,144            ; set coordinates to print name.
       ld a,(de)           ; get coordinate from miscellaneous data.
       ld h,a
       ld (dispx),hl       ; store coordinates.
       inc de              ; go to start of screen name.
       defb &dd
       ld h,d              ; equivalent to ld xh,d.
       defb &dd
       ld l,e              ; equivalent to ld xl,e.
       ld a,7              ; use paper 1, pen 3 (1*4+3 = 7).
       call prstr          ; print the screen name.

; Display lives, current score and high score.

       defb 33,168,32      ; load coordinates of number of lives into hl.
       ld (dispx),hl       ; set coordinates.
       ld c,2              ; use paper 0, pen 2.
       ld a,(lives)        ; number of lives remaining.
       add a,'0'           ; add ASCII code for zero.
       call prchar         ; display character.

       defb 33,168,72      ; coordinates to print score string.
       ld (dispx),hl       ; set coordinates.
       ld ix,sctxt         ; score text.
       ld a,2              ; use paper 0, pen 2.
       call prstr          ; print string.
       call dscore         ; display score.
       ld a,216            ; coordinate to print high score string.
       ld (dispy),a        ; set coordinate.
       ld hl,(hisc)        ; get high score in bc.
       call prnum          ; display high score.

; Initially draw the aliens that will patrol this room.

       ld ix,albuff        ; point to first alien.
       ld b,3              ; number of aliens to draw.
draw0  push bc             ; store loop counter.
       push ix             ; store buffer address.
       call draln          ; draw this alien.
       pop ix              ; get buffer address.
       ld de,6             ; 6 bytes per alien in buffer.
       add ix,de           ; next alien in buffer.
       pop bc              ; restore loop counter.
       djnz draw0          ; repeat for 3 aliens.

; Draw the player for the first time and make the screen visible once
; everything has been drawn.

       ld hl,(newx)        ; get starting player coordinates.
       ld (playx),hl       ; store them for routine to draw player sprite.
       call dplayr         ; display player.

       ld hl,inkta1        ; ink table for top half of screen.
       ld (inter1+1),hl    ; set inks.
       ld hl,inkta2        ; ink table for bottom half of screen.
       ld (inter2+1),hl    ; set inks.

       ld hl,ch1dat        ; start of music data for channel 1.
       ld (chptr1),hl      ; store channel marker.
       call chan2c         ; call music player for first time.
       xor a
       ld (musoff),a       ; turn music on now that it is initialised.


; Okay, initialisation is over so let's enter the main game loop.

mloop  ld a,(fallf)        ; are we in mid-air?
       and a
       jr nz,gravit        ; yes.

; So we're walking on solid ground - allow player to move left and right.

       xor a               ; temporary direction.
       ld (tdir),a         ; set marker to say we haven't moved left/right.

       call getkey         ; get keypress.
       ld a,(keytab+8)     ; get info on keyboard matrix line 8.
       and 8               ; bit 3 contains info on key 8*8+3 = 67 (Q).
       call z,mpl          ; move player left.
       ld a,(keytab+7)     ; get info on keyboard matrix line 7.
       and 8               ; bit 3 contains info on key 7*8+3 = 59 (W).
       call z,mpr          ; move player right.

       ld a,(keytab+5)     ; get info on keyboard matrix line 5.
       and 128             ; bit 7 contains info on key 5*8+7 = 47 (SPACE).
       jp z,jump           ; jump if SPACE bar is pressed.

; Now for the gravity processing.

grav   call getxy          ; get x/y collision positions in dispx.
       ld a,(dispx)        ; vertical position.
       add a,16            ; look 16 pixels down at what's underfoot.
       ld (dispx),a        ; we'll use these coords then.
       cp 144              ; bottom of screen playing area?
       jp nc,grav0         ; yes, so don't check block below player.
       call fpblk          ; find floor, test for objects and spikes etc.
       and a               ; is it an empty space?
       jp nz,grav0         ; not falling.
       call endxy          ; right edge collision coordinates.
       call fpblk          ; what's under the player.
       and a               ; space colours?
       jp nz,grav0         ; not falling.

; We are moving vertically at this point.

gravit ld a,(fallf)        ; were we falling already?
       and a
       jr nz,gravd0        ; yes.

; We weren't falling before - need to set up jump so we're falling.

       ld (jdir),a         ; set jump direction so no left/right.
       ld hl,jtabd         ; beginning of fall sequence.
       ld (jptr),hl        ; set jump pointer.
       inc a
       ld (fallf),a        ; set falling flag.

; In mid-air, we could be ascending or descending.

gravd0 ld a,(jdir)         ; left/right movement.
       rra                 ; moving left?
       push af             ; store accumulator and flags.
       call c,mpl          ; yes - go left.
       pop af              ; restore accumulator.
       rra                 ; moving right?
       call c,mpr          ; yes - go right then.
       ld hl,(jptr)        ; pointer to jump table.
       ld a,(hl)           ; x displacement.
       cp 128              ; 128=we've fallen too far.
       call z,gravd1       ; call routine to set "death from falling" flag.
       inc hl              ; next vertical displacement in jump table.
       ld (jptr),hl        ; set new pointer.
       and a               ; no movement.
       jr z,grav2
       cp 128              ; in which direction are we heading?
       jr nc,gravd2        ; going down.

; Ascending.

       ld b,a              ; going up - set counter.
gravu0 push bc             ; put loop count on stack.
       call getxy          ; put x+y displacement coordinates in dispx.
       ld hl,dispx         ; displacement.
       dec (hl)            ; look up.
       ld a,(hl)           ; check x coordinate.
       inc a               ; are we going off the top of the screen?
       jr z,gravu1         ; yes, so can go no further.
       call fnpblk         ; get block and test for deadly spikes etc.
       and a               ; solid?
       jr nz,gravu1        ; yes - we've hit the roof.
       call endxy          ; get right edge x+y displacements in dispx.
       call fnpblk         ; find block at this position.
       and a               ; solid?
       jr nz,gravu1        ; yes - we've hit the roof.
       ld hl,newx          ; move okay, get x coord (vertical).
       dec (hl)            ; move up.
       pop bc              ; fetch loop count from stack.
       djnz gravu0         ; keep going until move complete.
       jr grav2
gravu1 pop bc              ; pop bc from stack and discard.
       ld hl,jtabd         ; start of descent table.
       ld (jptr),hl        ; set jump pointer so player starts to fall.
       jr grav2

; Descending.

gravd2 neg                 ; going down - negate displacement.
       ld b,a
gravd4 push bc
       call getxy          ; x/y collision coords in dispx.
       ld a,(newx)         ; x coordinate of player.
       add a,16            ; look under feet - 16 pixels down.
       ld (dispx),a        ; new x coord.
       cp 144              ; bottom of screen playing area?
       jr nc,gravd5        ; yes, so don't fall any further.
       call fpblk          ; landed yet?
       and a               ; space colours?
       jr nz,gravd5        ; Not falling.
       call endxy          ; right edge collision coordinates.
       call fpblk          ; do test for objects, spikes etc.
       and a               ; space colours?
       jr nz,gravd5        ; Not falling.
       ld hl,newx          ; x coord (vertical).
       inc (hl)            ; move down.
       pop bc
       djnz gravd4         ; repeat until we've fallen correct distance.
       jr grav2

gravd1 ld (toofar),a       ; set flag to say too far.
       dec hl
       ld a,(hl)
       ret

gravd5 pop bc              ; hit a surface so stop falling.

; So we've hit or are standing on the ground then.

grav0  ld a,(fallf)        ; were we falling previously?
       and a
       jr z,grav2          ; no - we're just standing.
       xor a
       ld (fallf),a
       ld a,(toofar)       ; we've landed - have we fallen too far?
       and a
       jr z,grav2          ; no, the fall wasn't fatal.
       call die            ; strawberry jam!

; Okay, we now know the new coordinates of the player.  Delete the old
; sprite and redisplay the new one at the new position.

grav2  call wait5          ; wait for raster to reach bottom area of screen.
       call dplayr         ; redisplay sprite with exclusive-or to delete it.
       ld hl,(newx)        ; new coords.
       ld (playx),hl       ; make these the next coordinates.
       ld a,(ndirec)       ; new direction.
       ld (direct),a       ; set new direction.
       call dplayr         ; display new sprite.

       ld a,(dflag)        ; is player dead or alive?
       and a               ; zero result indicates he's alive.
       jr nz,killim        ; he's dead.
       ld a,(numobj)       ; objects left to collect.
       and a               ; zero means screen is finished.
       jp z,nexlev         ; screen complete.

       call wait5          ; wait for raster to reach bottom area of screen.
       ld ix,albuff        ; address of the first alien table entry.
       call dmraln         ; delete, move and redraw alien.

       call wait5          ; wait for raster to reach bottom area of screen.
       ld ix,albuff+6      ; address of second alien.
       call dmraln         ; delete, move and redraw alien.

       call wait5          ; wait for raster to reach bottom area of screen.
       ld ix,albuff+12     ; address of third alien.
       call dmraln         ; delete, move and redraw alien.
       jp mloop            ; back to beginning of main loop.

; Death sequence.

killim ld a,1
       ld (musoff),a       ; turn music off.
       ld hl,snddat+4
       ld (hl),0           ; coarse tone (high byte) for sound channel C.
       inc l
       ld (hl),5           ; fine tone (low byte) for sound channel C.
       inc l
       inc l
       ld (hl),63          ; turn all sound channels off.
       ld hl,snddat+10
       ld (hl),12          ; volume for channel C.
       call w8912          ; set AY registers.

       ld ix,chardt        ; address of Spectrum font.
       ld b,100            ; loop is repeated 100 times.

kill0  push bc             ; store loop counter.
       ld hl,snddat+7
       ld (hl),27          ; sound channels A and B off, channel C on.
       ld de,&0107         ; AY register 7 is used to change channel settings.
       call w8912a         ; change channel settings.

       ld a,(ix)           ; get byte from character set.
kill1  dec a               ; decrement byte.
       jr nz,kill1         ; repeat delay loop.
       ld hl,snddat+6
       ld (hl),8           ; noise period for channel C.
       ld hl,snddat+7
       ld (hl),59          ; sound channels A and B off, channel C on.
       call w8912          ; change noise and channel settings.
       ld a,(ix)           ; get byte from character set.
kill2  dec a               ; decrement byte.
       jr nz,kill2         ; repeat delay loop.
       inc ix              ; go to next byte of character set.
       pop bc              ; get loop counter.
       djnz kill0          ; repeat 100 times.

       ld hl,snddat+7
       ld (hl),63          ; turn all sound channels off.
       ld de,&0107         ; AY register 7 is used to change channel settings.
       call w8912a         ; change channel settings.
       ld hl,lives         ; lives counter.
       dec (hl)            ; subtract life.
       jp z,lost           ; zero remaining so player has lost.
       call fade           ; fade to black.
       jp rstart           ; not reached zero so restart screen.

; Next level.
; First of all let's animate a bonus of 100.

nexlev ld a,1
       ld (musoff),a       ; turn music off.
       ld hl,snddat+5
       ld (hl),0           ; coarse tone (high byte) for sound channel C.
       inc l
       inc l
       ld (hl),63          ; turn all sound channels off.
       ld hl,snddat+10
       ld (hl),12          ; volume for channel C.
       call w8912          ; set AY registers.

       ld b,100            ; bonus = 100 points.
nexlv0 push bc             ; store counter.
       ld hl,snddat+7
       ld (hl),59          ; sound channels A and B off, channel C on.
       ld de,&0107         ; AY register 7 is used to change channel settings.
       call w8912a         ; change channel settings.
       pop af              ; get counter (now in a).
       push af             ; store it again.

       ld hl,snddat+4
       rra                 ; 2 right shifts effectively divide counter by 4.
       rra
       and 31              ; discard anything shifted into high bits.
       add a,8
       ld (hl),a           ; fine tone (low byte) for sound channel C.
       ld de,&0104         ; AY register 4 is used to set fine tone of channel
                           ; C.
       ex af,af'           ; store counter.
       call w8912a         ; set fine tone of channel C.
       ex af,af'           ; get counter.

       add a,48            ; insert a short delay; the longer the tone period,
nexlv1 dec a               ; decrement byte.
       jr nz,nexlv1        ; repeat delay loop.
       ld hl,snddat+7
       ld (hl),63          ; turn all sound channels off.
       ld de,&0107         ; AY register 7 is used to change channel settings.
       call w8912a         ; change channel settings.

       ld bc,0             ; insert another (much longer) delay.
nexlv2 nop
       nop
       dec bc              ; decrement counter.
       ld a,b              ; loop counter high byte.
       or c                ; combine with low byte for zero check.
       djnz nexlv2         ; repeat delay loop.
       ld hl,(score)       ; read new score.
       inc hl              ; add 1.
       ld (score),hl       ; write new score.
       call dscore         ; display score.
       pop bc              ; restore loop control.
       djnz nexlv0         ; repeat.
       ld hl,0             ; set both fine and coarse tone of sound channel C.
       ld (snddat+4),hl
       ld hl,snddat+7      ; sound channels A and B on, channel C off.
       ld (hl),63
       call w8912          ; set AY registers.

; Now fade to black and move onto the next level.

       call fade           ; fade to black.
       ld hl,scno          ; address of screen number.
       inc (hl)            ; next one please.
       ld a,(hl)           ; which room is next?
       cp NUMSCR           ; gone past last one?
       jp c,rstart         ; no, carry on.
       ld (hl),0           ; restart on first level.
       ld hl,lives         ; lives counter.
       inc (hl)            ; give player one more life.
       jp rstart

; Player has lost all lives.

; Firstly, store the area underneath where the game over message will be
; written.

lost   ld hl,&c215         ; screen address where game over message is printed.
       ld de,tmpspr        ; storage address.
       ld b,8              ; there are 8 lines to store.
lost0  push bc             ; store number of lines remaining.
       push hl             ; store screen address.
       ld bc,22            ; each line consists of 22 bytes.
       ldir                ; copy all bytes in line.
       pop hl              ; get screen address.
       ld a,h              ; go down to next line of screen by adding
       add a,8             ; &800 to screen address.
       ld h,a              ; set address of next screen byte.
       jr nc,lost1         ; check if screen address needs to be corrected.
       push de             ; store storage address.
       ld hl,&c040         ; correct screen address if line below is on a
       add hl,de           ; new row of 8 lines.
       pop de              ; get storage address.
lost1  pop bc              ; get number of lines remaining.
       djnz lost0          ; repeat for remaining lines.

; Print the game over message to the screen.

       ld ix,gotxt         ; game over text.
       ld a,7              ; use paper 1, pen 3 (1*4+3 = 7).
       defb 33,64,84       ; coordinates to print game over string.
       ld (dispx),hl       ; set coordinates.
       call prstr          ; print string.

; Three second delay.

       ld bc,900           ; 900 iterations required.
delay  halt                ; wait for 1/300th of a second.
       dec bc              ; decrement counter.
       ld a,b              ; loop counter high byte.
       or c                ; combine with low byte for zero check.
       jr nz,delay         ; repeat until zero.

       ld hl,(hisc)        ; high score.
       ld de,(score)       ; player's final score.
       and a               ; reset carry flag.
       sbc hl,de           ; is player's score the higher of the two?
       jr nc,lost2         ; no, remove game over message from screen.
       ld hl,(score)       ; final score.
       ld (hisc),hl        ; make this the new high score.

; Remove the game over message from the screen by restoring what was
; underneath it previously.

lost2  ld hl,tmpspr        ; storage address.
       ld de,&c215         ; screen address where game over message is printed.
       ld b,8              ; there are 8 lines to store.
lost3  push bc             ; store number of lines remaining.
       push de             ; store screen address.
       ld bc,22            ; each line consists of 22 bytes.
       ldir                ; copy all bytes in line.
       pop de              ; get screen address.
       ld a,d              ; go down to next line of screen by adding
       add a,8             ; &800 to screen address.
       ld d,a              ; set address of next screen byte.
       jr nc,lost4         ; check if screen address needs to be corrected.
       push de             ; store screen address.
       ld hl,&c040         ; correct screen address if line below is on a
       add hl,de           ; new row of 8 lines.
       ex de,hl            ; screen address now in de.
       pop de              ; get screen address.
lost4  pop bc              ; get number of lines remaining.
       djnz lost3          ; repeat for remaining lines.

       jp intro            ; restart game.

gotxt  defb ' GAME OVER '
       defb 0

; Move player left.

mpl    ld a,1              ; direction 1 = left.
       ld (tdir),a         ; set temp direction.
       call getxy          ; get coords in dispx.
       ld hl,dispy         ; horizontal position of player.
       dec (hl)            ; look one pixel left.
       call fnpblk         ; find block here.
       and a               ; is there a wall in the way?
       ret nz              ; yes, so can't move left.
       ld de,32            ; next block = 32 bytes down.
       add hl,de           ; one row down.
       call npblk          ; do test for objects, spikes etc.
       and a               ; is there a wall in the way?
       ret nz              ; yes, so can't move left.
       ld a,(newx)         ; player's x coord.
       and 7               ; are we on a character boundary?
       jr z,mpl0           ; yes - no need to check third block then.
       add hl,de           ; one row down.
       call npblk          ; do test for objects, spikes etc.
       and a               ; is there a wall in the way?
       ret nz              ; yes, so can't move left.
mpl0   ld hl,newy          ; y coord of player.
       ld a,(hl)           ; test it.
       cp 2                ; are we already at left edge?
       ret c               ; 'fraid so, can't go any further.
       dec (hl)
       dec (hl)            ; 2 pixels left.
       xor a               ; zeroise accumulator.
       ld (ndirec),a       ; facing direction = left.
       ret

; Move player right.

mpr    ld a,2              ; direction 2 = right.
       ld (tdir),a         ; set temp direction.
       call endxy          ; get right side coords in dispx.
       ld hl,dispy         ; horizontal position of player.
       inc (hl)            ; look one pixel right.
       call fnpblk         ; find block here.
       and a               ; is there a wall in the way?
       ret nz              ; yes, so can't move left.
       ld de,32            ; next block = 32 bytes down.
       add hl,de           ; one row down.
       call npblk          ; do test for objects, spikes etc.
       and a               ; is there a wall in the way?
       ret nz              ; yes, so can't move left.
       ld a,(newx)         ; player's x coord.
       and 7               ; are we on a character boundary?
       jr z,mpr0           ; yes - no need to check third block then.
       add hl,de           ; one row down.
       call npblk          ; do test for objects, spikes etc.
       and a               ; is there a wall in the way?
       ret nz              ; yes, so can't move left.
mpr0   ld hl,newy          ; y coord of ship.
       ld a,(hl)
       cp 244              ; at right edge?
       ret nc              ; yes - can't move right.
       inc (hl)
       inc (hl)            ; 2 pixels right.
       ld a,8              ; 8 = right.
       ld (ndirec),a       ; facing direction = right.
       ret


; Display score.

dscore ld hl,(dispx)       ; get current coordinates in (dispx, dispy).
       push hl             ; store coordinates.
       defb 33,168,128     ; coordinates to print score.
       ld (dispx),hl       ; set coordinates.
       ld hl,(score)       ; get current score.
       call prnum          ; print score.
       pop hl              ; get coordinates.
       ld (dispx),hl       ; change them back to their original settings.
       ret

sctxt  defb 'Score: 0   '
       defb 0
hitxt  defb 'High: 0   '
       defb 0

; Display room we're in.
; Each room is made up of 18 rows and 32 columns of character blocks,
; which means that each room takes up 576 bytes.  To find the address
; of the data for a room we multiply the room number by 576.  Normally
; we would achieve this with a few shifts and a couple of additions
; but for the sake of saving memory we'll have a simple addition loop.

droom  ld hl,SCDATA        ; address of screens data.
       ld a,(scno)         ; current room.
       and a               ; is it zero?
       jr z,droom6         ; yes so we already have the correct address.
       ld de,576           ; each room = 32x18 chars = 576 bytes.
droom5 add hl,de           ; next room
       dec a               ; one less room to skip.
       jr nz,droom5        ; repeat

; Right then, hl should now be pointing to our screen block data.
; The data for the current screen needs to be copied to another location.

droom6 push hl             ; store screen data address.
       ld de,scrcur        ; start address of data for current screen.
       ld bc,576           ; each room = 32x18 chars = 576 bytes.
       ldir                ; copy screen data.
       pop hl              ; get screen data address.

droom3 ld de,&c000         ; start address of screen.
       ld b,18             ; screen is 18 rows high.
droom2 push bc             ; push row counter.
       ld b,32             ; screen is 32 columns wide.
droom1 push bc             ; push column counter.
       push hl             ; store pointer to data block.
       ld a,(hl)           ; get block number.
       push af             ; store block number.
       rlca                ; multiply by 16.
       rlca                ; rotate left 4 times
       rlca                ; to achieve this.
       rlca
       ld c,a              ; remember result - we need it in a minute.
       and 15              ; 4 leftmost bits shifted into high byte.
       ld b,a              ; want these in b register (high).
       ld a,c              ; restore result of shifts.
       and 240             ; take the bits for the low byte this time.
       ld c,a              ; put these in c, bc=a*16.
       ld hl,blkgfx        ; address of block graphics.
       add hl,bc           ; add bc to find address of block's graphic.

       ld bc,&0808         ; 8 lines per block; c is used to calculate screen
                           ; address for next line down.
droom0 ld a,(hl)           ; get first byte of line.
       ld (de),a           ; write byte to screen.
       inc hl              ; next byte to transfer.
       inc de              ; go to next screen byte.
       ld a,(hl)           ; get second byte of line.
       ld (de),a           ; write byte to screen.
       inc hl              ; go to first byte of next line of block.
       dec de              ; go to previous screen byte.
       ld a,d              ; go down to next line of screen by adding
       add a,c             ; &800 to screen address.
       ld d,a              ; set address of next screen byte.
       djnz droom0         ; repeat for rest of character block.

       ld hl,&c002         ; set screen address for next block.
       add hl,de           ; corrected screen address stored in hl.
       ex de,hl            ; screen address now stored in de.

       pop af              ; get block number.
       cp 1                ; is it an object for the player to collect?
       jr nz,droom4        ; no, skip object count.
       ld hl,numobj        ; number of objects on screen.
       inc (hl)            ; add 1.

droom4 pop hl              ; restore address of room data.
       inc hl              ; next byte of room data.
       pop bc              ; pop column counter.
       djnz droom1         ; next column.

       pop bc              ; pop row counter.
       djnz droom2         ; next row.
       ret


; Patrolling alien handling routines.

; Draw the alien to which ix points.
; Each sprite type has a list of four two-byte pointers which
; give the address of the sprite to be used for each frame.
; The sprites position on the screen determines which of these
; four frames are to be used.
; On exit, a, bc, de and hl are corrupted.

draln  ld a,(ix)           ; does alien exist?
       rla
       ret c               ; no, he's been switched off.
       rra
       and 127
       ld l,(ix+2)         ; get vertical position.
       ld h,(ix+3)         ; get horizontal position.
       ld (dispx),hl       ; sprite coords.
       ld a,(ix)           ; get contents of sprite type byte.
       rra                 ; type is in bits d5-d7, so rotate
       rra                 ; into bits d3-d5 for multiples of 8.
       and 56              ; discard the bits we don't want.
       ld hl,alnptr        ; pointer to alien sprite pointers.
       add a,l
       ld l,a
       ld a,(dispx)        ; get vertical position.
       ld b,a
       ld a,(dispy)        ; horizontal position.
       or b                ; combine horizontal and vertical.
       and 6               ; position on screen tells us frame.
       add a,l             ; this is the frame displacement.
       ld l,a              ; hl contains address of one of the four frames.
       ld e,(hl)           ; low byte of sprite address.
       inc hl              ; next byte of pointer.
       ld d,(hl)           ; high byte of sprite address.
       ex de,hl            ; put the sprite address in hl.
       jp sprite           ; draw the sprite.

; Move the alien to which the ix registers are pointing.

movaln ld a,(ix+1)         ; alien movement type (0=vertical, 1=horizontal).
       rra
       jr c,movav          ; move vertical.
       rra
       jr c,movar          ; move alien right.

; Move alien left.

moval  ld a,(ix+3)         ; get y coordinate.
       sub 2               ; move left.
       ld (ix+3),a
       cp (ix+4)           ; reached mimimum yet?
       jr z,movax          ; yes - change direction.
       jr c,movax
       ret

; Move alien right.

movar  ld a,(ix+3)         ; get y coordinate.
       add a,2             ; move right.
       ld (ix+3),a
       cp (ix+5)           ; reached maximum yet?
       jr nc,movax         ; yes - change direction.
       ret

; Move alien vertically.

movav  rra
       jr c,movad

; Move alien up.

movau  ld a,(ix+2)         ; get x coordinate.
       sub 2               ; move up.
       ld (ix+2),a
       cp (ix+4)           ; reached mimimum yet?
       jr z,movax          ; yes - change direction.
       ret

; Move alien down.

movad  ld a,(ix+2)         ; get x coordinate.
       add a,2             ; move down.
       ld (ix+2),a
       cp (ix+5)           ; reached maximum yet?
       jr nc,movax         ; yes - change direction.
       ret

; Change alien direction.

movax  ld a,(ix+1)         ; direction flag.
       xor 2               ; switch direction.
       ld (ix+1),a
       ret

dmraln push ix             ; store address of alien buffer.
       call draln          ; delete alien from screen.
       pop ix              ; get address of alien buffer.
       call movaln         ; move alien.
       push ix             ; store address again.
       call draln          ; re-draw it.
       pop ix              ; get address.

; Now drop through into alcol to check for collision with player sprite.

; Alien collision detection.

alcol  ld a,(ix)           ; does this alien exist?
       rla
       ret c               ; no, he's been switched off.
       call getxy          ; get x/y collision positions in dispx.
       ld a,(dispx)        ; player's x coordinate.
       sub (ix+2)          ; subtract alien's x coordinate.
       jr nc,alcol0        ; result is positive.
       neg                 ; result is negative so make positive.
alcol0 cp 16               ; are they less than 16 pixels apart vertically?
       ret nc              ; no, so they cannot have collided.
       ld a,(dispy)        ; player y coord.
       sub (ix+3)          ; subtract y coordinate of alien.
       cp 16               ; is result less than 16?
       jr c,die            ; yes, so sprites have collided.
       call endxy          ; player's right edge coordinates.
       ld a,(dispy)        ; put player right edge coord in accumulator.
       sub (ix+3)          ; subtract alien left edge coordinate.
       cp 16               ; is result less than 16?
       ret nc              ; no, so there's no collision.

; Set a flag to say player is dead.

die    ld hl,dflag         ; flag indicates when player is dead.
       ld (hl),h           ; non-zero value means it's curtains.
       ret                 ; job done!

jump   ld a,(fallf)        ; get falling flag.
       and a               ; already in mid-air?
       ret nz              ; yes, so we can't jump.
       ld hl,jtabu         ; start of leap table.
       ld (jptr),hl        ; set leap pointer.
       inc a               ; accumulator now set to 1.
       ld (fallf),a        ; set flag to say we're moving vertically.
       ld a,(tdir)         ; direction in which we are moving.
       ld (jdir),a         ; set jump direction.
       ld a,(playy)        ; player's y position.
       ld (newy),a         ; restore it.
       jp gravd0

newx   defb 20
newy   defb 20
playx  defb 20
playy  defb 20

; Jump table.
; Values 1-6 are going up, 250-255 are going down.
; When we hit value 128 the player has fallen too far.

jptr   defw 0
jtabu  defb 6,5,4,3,2,2
       defb 1,1,1,0,1,0
jtabd  defb 255,0,255
       defb 255,255,254
       defb 254,253,252
       defb 251,250,250
       defb 250,250,128


; fnpblk just calls fblk then does a relative jump to npblk.
; It's basically just there to save a few bytes.

fnpblk call fblk           ; find floor.
       jr npblk            ; non-passable block test.

; fpblk just calls fblk then drops through to pblk.

fpblk  call fblk           ; find floor.

; Test block.  Does checking for items, spikes and platforms.
; We set b=1 if platforms are solid, zero for walk-through.

pblk   ld b,1              ; platforms are solid.
       jr tblk
npblk  ld b,0              ; platforms are passable.

;

tblk   ld a,(hl)           ; what object number is the current block?
       and a               ; empty space?
       ret z               ; okay to move there then.
       cp 5                ; downward spikes?
       jr z,tblk0          ; yep, set flag to say player is dead.
       cp 8                ; upward spikes?
       jr z,tblk0          ; yep, set flag to say player is dead.
       cp 1                ; is it a collectable object?
       jr z,getit          ; it is, let's pick it up then.
       rr b                ; are platforms solid?
       jr nc,tblk2         ; no - allow player through.

; So platforms are solid.  We must be checking attributes underfoot,
; therefore we don't need to check when the player is straddling
; character rows, otherwise he'll walk "inside" platforms.

       ld a,(newx)         ; player's x position.
       and 7               ; is he straddling blocks?
       jr nz,tblk1         ; yes, he's missed this platform.

; Okay, let's see if player is walking on a wall or platform.

       ld a,(hl)           ; get object number again.
       ret                 ; return - anything other than SPACE is solid.

; Okay, let's see if player is going through a platform or wall.

tblk2  cp 3                ; red wall?
       ret z               ; yes, so block progress.
       cp 6                ; blue wall?
       ret z               ; yes, so block progress.
       jr tblk1            ; it's a platform so can go through.
tblk0  call die            ; set death flag.
tblk1  xor a               ; allow player to progress through block.
       ret

; Picked up an item.

getit  ld (hl),0           ; set object number to zero.
       push hl             ; store object number.
       ld hl,numobj        ; number of objects left to collect.
       dec (hl)            ; subtract 1.
       ld hl,(score)       ; get score.
       ld de,10            ; 10 points per item.
       add hl,de           ; increment score.
       ld (score),hl       ; increment it.

       pop hl              ; get object number.
       push hl             ; store it again.
       ld de,scrcur        ; start address of data for current screen.
       sbc hl,de           ; hl now contains offset from start of screen data.
       add hl,hl           ; multiply by 2 as each block is 2 bytes wide on
                           ; screen.
       ld de,&c000         ; add offset to start address of screen.
       add hl,de           ; hl now contains screen address of object.
       ex de,hl            ; screen address now in de.

       ld hl,blkgfx+16     ; start address of object sprite data.
       ld bc,&0808         ; 8 lines per block; c is used to calculate screen
                           ; address for next line down.
getit0 ld a,(de)           ; get byte already on screen.
       xor (hl)            ; merge sprite data with it.
       ld (de),a           ; write byte to screen.
       inc hl              ; next byte to transfer.
       inc de              ; go to next screen byte.
       ld a,(de)           ; get byte already on screen.
       xor (hl)            ; merge sprite data with it.
       ld (de),a           ; write byte to screen.
       inc hl              ; go to first byte of next line of block.
       dec de              ; go to previous screen byte.
       ld a,d              ; go down to next line of screen by adding
       add a,c             ; &800 to screen address.
       ld d,a              ; set address of next screen byte.
       djnz getit0         ; repeat for rest of character block.

       call dscore         ; display score.
       pop hl              ; get object number.
       jr tblk1            ; return with zero to say we can pass.

; Find character block at (dispx, dispy) for collision detection.

fblk   ld hl,scrcur        ; start address of data for current screen.
       ld a,(dispx)        ; displacement from top of screen.
       rlca                ; for each 8 pixels down we move 32
       rlca                ; bytes down screen data.
       ld c,a              ; store in c for later.
       and 224             ; rows are multiples of 32.
       ld e,a              ; part of low byte.
       ld a,c              ; get displacement.
       and 3
       ld d,a              ; de contains row number multiplied by 32.
       ld a,(dispy)        ; displacement from left edge.
       rra                 ; 3 right shifts
       rra                 ; effectively divide
       rra                 ; coordinate by 8.
       and 31              ; discard anything shifted into high bits.
       add a,e
       ld e,a
       add hl,de           ; add offset to screen data address.
       ld a,(hl)           ; return with object number in accumulator.
       ret

; Finds horizontal displacement from left edge of sprite.
; On exit hl points to first of 2 bytes, 1st=start, 2nd=end.

getxy  ld a,(newx)         ; player's vertical position from top of screen.
       ld (dispx),a        ; put in displacement coords.
getxy0 ld a,(newy)         ; player's new horizontal position.
       and 6               ; find frame position.
       ld hl,coldis        ; collision displacement table.
       add a,l             ; add offset to table address.
       ld l,a              ; hl points to offset from left edge of sprite.
getxy1 ld a,(newy)         ; sprite y position.
       add a,(hl)          ; add offset.
       ld (dispy),a        ; y displacement.
       ret

endxy  call getxy0         ; get x+y displacement.
       inc hl              ; get right edge.
       jr getxy1           ; return with this displacement instead.


; Display player sprite.

dplayr ld hl,(playx)       ; player's coordinates.
       ld (dispx),hl       ; coords used by sprite routine.
       ld a,h              ; y coordinate determines which frame to use.
       call calcpf         ; calculate address of sprite data for frame.
       jp sprite           ; draw the sprite.

; Write a sprite to the screen, given coordinates in dispx and dispy. This
; routine requires the sprites to be stored in a 'logical' format instead of
; the CPC's screen format, as this makes it much easier to rotate the sprites.
; This rotation allows sprites to be written to anywhere on the screen with
; pixel precision.
; On entry, hl contains sprite address.
; On exit, a, bc, de, hl and ix are corrupted.
;
; The table for converting rotated sprite data to corresponding screen bytes
; (sprbyt) and the buffer for storing a line of a sprite (sprbuf) must both be
; page-aligned (i.e. be located at &xx00)

sprite push hl             ; store sprite address.
       call scadd          ; screen address of top left of sprite in hl.
       ld (sprit4+1),hl    ; set screen address in line display routine.
       pop hl              ; get sprite address.

       ld a,(dispy)
       and 3               ; get number of pixels to rotate sprite by (0-3).
       ld (sprit0+2),a     ; store it for use later on.
       ld (sprit2+2),a
       ld a,16             ; sprite consists of 16 lines; routine is too long
                           ; to allow b to be used as counter.

; Copy the current line of the sprite to the buffer.
; Two different routines are used, depending on how many pixels the sprite
; needs to be rotated by.

sprit0 ex af,af'           ; store number of lines remaining.
       ld a,0              ; get number of bytes to rotate sprite by
                           ; (modified).
       ld de,sprbuf        ; buffer address in de.
       cp 3                ; does sprite need to be rotated by 3 pixels?
       jr z,sprit1         ; yes, so write line to buffer differently.

       push de             ; store buffer address.
       ldi                 ; write line to buffer.
       ldi                 ; each line consists of 4 bytes.
       ldi
       ldi
       ex de,hl            ; address of last byte of buffer now in hl.
                           ; de now contains first byte of next line of sprite.
       ld (hl),0           ; last byte is blank.
       pop hl              ; hl now contains address of first byte of buffer.
       jr sprit2

sprit1 xor a
       ld (de),a           ; first byte of buffer is blank.
       inc e               ; go to next byte of buffer.
       ldi                 ; write line to buffer.
       ldi                 ; each line consists of 4 bytes.
       ldi
       ldi
       dec e               ; go back to last byte of buffer.
       ex de,hl            ; address of last byte of buffer now in hl.
                           ; de now contains first byte of next line of sprite.

; Rotate the line by the required number of pixels.
; Again, two different routines are used.

sprit2 push de             ; store address of next line of sprite.
       ld a,0              ; get amount to rotate sprite by (modified).
       and a               ; does sprite need to be rotated?
       jr z,sprit4         ; no, so jump to line display routine.
       cp 3                ; does sprite need to be rotated by 3 pixels?
       jr nz,sprit3        ; no, so jump to routine to rotate by 1 or 2
                           ; pixels.

       push hl             ; store address of last byte of buffer.
       rl (hl)             ; rotate last byte in buffer left.
       dec l               ; go to previous byte.
       rl (hl)             ; repeat for all bytes in buffer.
       dec l
       rl (hl)
       dec l
       rl (hl)
       dec l
       rl (hl)
       pop hl              ; get address of last byte of buffer.
       and a               ; clear carry flag.
       rl (hl)             ; repeat the procedure.
       dec l
       rl (hl)
       dec l
       rl (hl)
       dec l
       rl (hl)
       dec l
       rl (hl)
       jr sprit4

sprit3
       push hl             ; store address of first byte of buffer.
       and a               ; clear carry flag.
       rr (hl)             ; rotate first byte in buffer right.
       inc l               ; go to next byte.
       rr (hl)             ; repeat for all bytes in buffer.
       inc l
       rr (hl)
       inc l
       rr (hl)
       inc l
       rr (hl)
       pop hl              ; get address of first byte of buffer.
       rr (hl)             ; repeat the procedure.
       inc l
       rr (hl)
       inc l
       rr (hl)
       inc l
       rr (hl)
       inc l
       rr (hl)
       cp 1                ; is sprite to be rotated by 1 pixel?
       jr z,sprit4         ; yes, we've already rotated the line by 1 pixel.

       dec l               ; move back to first byte of buffer.
       dec l
       dec l
       dec l
       push hl             ; store address of first byte of buffer.
       rr (hl)             ; rotate first byte in buffer right.
       inc l               ; go to next byte.
       rr (hl)             ; repeat for all bytes in buffer.
       inc l
       rr (hl)
       inc l
       rr (hl)
       inc l
       rr (hl)
       pop hl              ; get address of first byte of buffer.
       rr (hl)             ; repeat the procedure.
       inc l
       rr (hl)
       inc l
       rr (hl)
       inc l
       rr (hl)
       inc l
       rr (hl)             ; line has now been rotated by 2 pixels.

; Write the line to the screen, using a lookup table to convert the 'logical'
; byte format into the format normally used in MODE 1.
; The logical format used is (bit,pixel): 1,0 0,0 1,1 0,1 1,2 0,2 1,3 0,3
; The format used in MODE 1 is:           0,0 0,1 0,2 0,3 1,0 1,1 1,2 1,3

sprit4 ld de,0             ; screen address of line (modified).
       ld b,5              ; 5 bytes in line.
       ld ix,sprbuf        ; start address of sprite buffer.
       ld h,sprbyt_hb      ; start address of lookup table.

sprit5 ld a,(ix)           ; get byte from buffer.
       ld l,a              ; add offset to lookup table address.
       ld a,(de)           ; get byte already on screen.
       xor (hl)            ; merge sprite data with it.
       ld (de),a           ; write byte to screen.
       defb &dd            ; go to next byte in buffer.
       inc l               ; equivalent to inc xl.
       inc de              ; go to next screen byte.
       djnz sprit5         ; repeat for remaining bytes.

       ld hl,&7fb          ; go down to next line of screen by adding &800-5
       add hl,de           ; to screen address.
       ex de,hl            ; screen address of next line now in de.
       call c,nextln       ; correct screen address if necessary.
       ld (sprit4+1),de    ; set screen address of next line of sprite.

       pop hl              ; get address of next line of sprite.
       ex af,af'           ; get number of lines remaining.
       dec a               ; decrement it.
       jp nz,sprit0        ; repeat for remaining lines.
       ret

; Calculate the address of the sprite data for a particular frame of Fizzog,
; given a frame number and the direction that Fizzog is facing (stored in
; direct).
; On entry, a contains frame number multiplied by 2 (i.e. 0, 2, 4 or 6).
; On exit, hl contains address of sprite data, a and de are corrupted.

calcpf and 6               ; ensure that frame number is 0, 2, 4 or 6.
       ld e,a
       ld a,(direct)       ; direction facing, 0=left, 8=right.
       add a,e             ; add to frame.
       ld hl,playfr        ; player frames.
       add a,l
       ld l,a              ; point to frame pointer.
       ld e,(hl)           ; low byte of frame address.
       inc hl              ; next byte of pointer.
       ld d,(hl)           ; high byte of frame address.
       ex de,hl            ; hl points to frame.
       ret

; Calculate a screen address, given co-ordinates in dispx and dispy.
; On exit, hl contains screen address, a and de are corrupted.
;
; The formula used for calculating the screen address is
; hl = &c000 + (dispx >> 3)*64 + (dispx and 7)*2048 + (dispy >> 2)

scadd  ld de,&c000         ; start address of screen.
       ld a,(dispx)
       and 248             ; reset bits 0-2, so that dispx is effectively
       ld h,0              ; multiplied by 8, then store result in hl.
       ld l,a
       add hl,hl           ; shift hl left 3 times, so that dispx is now
       add hl,hl           ; multiplied by 8^3 = 64.
       add hl,hl
       add hl,de           ; add offset to screen address.

       ex de,hl            ; screen address now in de.
       ld a,(dispx)
       and 7               ; reset bits 3-7, so that dispx = 0-7.
       rlca                ; rotate hl left 3 times, so that dispx is now
       rlca                ; multiplied by 8.
       rlca
       ld l,0              ; storing dispx in h means that dispx is now
       ld h,a              ; multiplied by 256*8 = 2048.
       add hl,de           ; add offset to screen address.

       ex de,hl            ; screen address now in de.
       ld a,(dispy)
       srl a               ; rotate twice, so that dispy is now divided
       srl a               ; by 4.
       ld h,0              ; store dispy in hl.
       ld l,a
       add hl,de           ; add offset to screen address.
       ret

; Screen fade routine.

fade   ld hl,fadtab        ; address of pointers to ink data.
       ld b,7              ; the table contains 7 entries.
fade0  ld e,(hl)           ; get low byte of address of ink data.
       inc hl              ; next byte.
       ld d,(hl)           ; get high byte.
       inc hl              ; next byte.
       ld (inter1+1),de    ; change address of ink data for top half of
                           ; screen.
       push bc             ; store number of entries remaining.
       ld d,8              ; wait for 8 frames.
fade1  call waitfr         ; wait for frame flyback.
       halt                ; another short delay.
       dec d               ; decrement loop counter.
       jr nz,fade1         ; repeat.
       pop bc              ; get number of entries remaining.
       djnz fade0          ; repeat for remaining entries.
       ret

; Table of pointers to ink data.

fadtab defw inkta1
       defw inkta3
       defw inkta4
       defw inkta5
       defw inkta6
       defw inkta7
       defw inkta0


; Print a character to the screen, given co-ordinates in dispx and dispy.
; On entry, a contains character to print, c contains pen/paper combination.
; On exit, a, bc, de, hl and ix are corrupted.
;
; The table for converting character data to corresponding screen bytes
; (txtbyt) must be page-aligned (i.e. be located at &xx00)

prchar push af             ; store character to print.
       call scadd          ; screen address of top left of char in hl.
       pop af              ; get character to print.

       ld b,a              ; store character to print.
       ld a,c              ; get pen/paper combination.
       and 15              ; reset bits 4-7, so that a = 0-15.
       rlca                ; rotate a left 4 times, so that it is now
       rlca                ; multiplied by 16.
       rlca
       rlca
       ld (prchr1+1),a     ; set start address of screen byte data for
                           ; selected pen/paper combination.
       ld a,b              ; get character to print.

; If the screen address to print the character to is already known, and the
; pen/paper combination has already been set at prstr below, you can call
; prchr0 instead.
; On entry, hl contains screen address.

prchr0 ld de,chardt-256    ; start address of character data where character 0
                           ; should be.
       defb &dd
       ld h,0              ; equivalent to ld hx,0
       defb &dd
       ld l,a              ; equivalent to ld lx,a
       add ix,ix           ; shift ix 3 times, so that it is now multiplied by
       add ix,ix           ; 8 (each character is 8 bytes long).
       add ix,ix
       add ix,de           ; start address of character data in ix.
       ex de,hl            ; screen address now in de.

       ld b,8              ; 8 lines in each character.
prchr1 ld hl,txtbyt_hb*256 ; start address of screen byte table (modified).
       push hl             ; store address.
       ld a,(ix)           ; get byte from character data.
       ld c,a              ; store character data in c.
       rlca                ; move bits 4-7 to 0-3.
       rlca
       rlca
       rlca
       and 15              ; reset bits 4-7.

       add a,l             ; add offset to screen byte table address.
       ld l,a
       ld a,(hl)           ; get byte to write to screen.
       ld (de),a           ; write byte to screen.
       inc de              ; go to next screen byte.

       pop hl              ; get address of screen byte table.
       ld a,c              ; get character data.
       and 15              ; reset bits 4-7.
       add a,l             ; add offset to screen byte table address.
       ld l,a
       ld a,(hl)           ; get byte to write to screen.
       ld (de),a           ; write byte to screen.

       dec de              ; go to previous screen byte.
       ld a,d              ; go down to next line of screen by adding
       add a,8             ; &800 to screen address.
       ld d,a              ; set address of next screen byte.
       call c,nextln       ; correct screen address if necessary.

       inc ix              ; go to next byte in character data.
       djnz prchr1         ; repeat for remaining lines.
       ret

nextln ld hl,&c040         ; correct screen address if line below is on a
       add hl,de           ; new row of 8 lines.
       ex de,hl            ; screen address now in de.
       ret

; Print a string to the screen, given co-ordinates in dispx and dispy.
; On entry, ix contains string address, a contains pen/paper combination.
; On exit, a, bc, de, hl and ix are corrupted.

prstr  and 15              ; reset bits 4-7, so that a = 0-15.
       rlca                ; rotate a left 4 times, so that it is now
       rlca                ; multiplied by 16.
       rlca
       rlca
       ld (prchr1+1),a     ; set start address of screen byte data for
                           ; selected pen/paper combination.
       call scadd          ; screen address of start of string in hl.

prstr0 ld a,(ix)           ; get character from string.
       and a               ; character 0 is used for end of string.
       ret z

       push hl             ; store screen address.
       push ix             ; store address of current character in string.
       call prchr0         ; print character.
       pop ix              ; get address of current character in string.
       pop hl              ; get screen address.

       inc ix              ; go to next character.
       inc hl              ; set screen address of next character by moving 2
       inc hl              ; bytes to right.
       jr prstr0           ; repeat for remaining characters.

; Print a 5-digit number to the screen, given co-ordinates in dispx and dispy.
; On entry, hl contains number to print.
; On exit, a, bc, de, hl and ix are corrupted.

prnum  ld de,numtxt        ; address of string to store number.
       ld bc,-10000        ; count the ten thousands.
       call prnum1
       ld bc,-1000         ; count the thousands.
       call prnum1
       ld bc,-100          ; count the hundreds.
       call prnum1
       ld bc,-10           ; count the tens.
       call prnum1
       ld bc,-1            ; count the units.
       call prnum1

       ld b,4              ; check the first four characters of the number
                           ; string (the fifth character is always printed).
       ld ix,numtxt        ; address of number string.
prnum3 ld a,(ix)           ; get character from string.
       cp '0'              ; is it 0?
       jr nz,prnum4        ; no, so start printing from this character.
       inc ix              ; skip this 0 and go to next character.
       djnz prnum3

prnum4 ld a,2              ; use paper 0, pen 2.
       jr prstr            ; print the number with leading zeroes left out.

prnum1 xor a               ; reset the counter.
prnum2 add hl,bc           ; subtract 10000, 1000, 100, 10 or 1 from number.
       inc a               ; increment counter.
       jr c,prnum2         ; repeat until result is less than zero.

       sbc hl,bc           ; undo the last subtraction.
       add a,47            ; decrement counter, add 48 to get corresponding
       ld (de),a           ; ASCII character, then store it in string.
       inc de              ; go to next byte in string.
       ret

numtxt defb '00000'        ; string to store number.
       defb 0

; Print a horizontal bar along the bottom of the screen using spaces.

prbar  defb 33,144,0       ; set coordinates to print bar containing name of
       ld (dispx),hl       ; screen.
       ld b,32             ; bar is 32 characters long.
       ld c,5              ; use paper 1, pen 1 (1*4+1 = 5).
prbar0 push bc             ; store spaces remaining and pen/paper combination.
       ld a,32             ; space corresponds to ASCII character 32.
       call prchar         ; print it.
       ld a,(dispy)
       add a,8             ; go to next column.
       ld (dispy),a
       pop bc              ; get spaces remaining and pen/paper combination.
       djnz prbar0         ; repeat for remaining spaces.
       ret

;Read the keyboard and store the results in a 10-byte table.
;Byte 1 of the table contains info on keys 0-7, byte 2 contains info on keys
;8-15, and so on up to keys 72-79.
;On exit, a, bc and hl are corrupted.

getkey ld hl,keytab        ; keyboard table address in hl.

       ld bc,&f40e         ; select register 14 of PSG using port A of PPI
       out (c),c           ; (&f4).
       ld bc,&f6c0         ; set PPI to 'select PSG register' mode (bit 7 = 1,
       out (c),c           ; bit 6 = 1) using port C (&f6), and select
                           ; register.
       xor a               ; set PPI to inactive mode (bit 7 = 0, bit 6 = 0).
       out (c),a

       ld bc,&f792         ; use control port of PPI (&f7).
       out (c),c           ; set port A to input mode.
       ld c,&40            ; start at keyboard matrix line 0, and set PPI to
                           ; 'read PSG register' mode (bit 7 = 0, bit 6 = 1).

getke1 ld b,&f6            ; use port C.
       out (c),c           ; send matrix line to read, and mode, to port C.
       ld b,&f4            ; use port A.
       in a,(c)            ; read line in keyboard matrix.
       ld (hl),a           ; store it in table.
       inc hl              ; go to next entry in table.

       inc c               ; go to next line in keyboard matrix.
       ld a,c
       cp &4a              ; there are 10 keyboard matrix lines to scan.
       jr nz,getke1        ; repeat for remaining lines.

       ld bc,&f600         ; set PPI to inactive mode using port C.
       out (c),c
       ld bc,&f782         ; use control port.
       out (c),c           ; set port A to output mode (the default).
       ret

keytab defs 10             ; the keyboard consists of an 8 x 10 matrix, so 80
                           ; bits (10 bytes) are required to store it

; Wait for frame flyback.
; On exit, a and b are corrupted.

waitfr ld b,&f5            ; use port B of PPI to read frame flyback status.
waitf0 in a,(c)            ; get info from port B.
       rra                 ; frame flyback status is in bit 0.
       jr nc,waitf0        ; repeat until bit 0 is set, which indicates frame
                           ; flyback.
       ret

; Wait until the raster has reached the bottom half of the screen, using the
; interrupt counter.

wait5  ld a,(inter0+1)     ; get value of interrupt counter.
       cp 5                ; bottom half of screen?
       jr nz,wait5         ; no, continue waiting.
       ret

; Now decompress level data into upper RAM.

decomp ld bc,SCRLEN        ; size of target area.
       ld de,SCDATA        ; address of screen data.
decom2 ld a,(hl)           ; source byte.
       cp 255              ; is it a control code code?
       jr z,decom0         ; it is, process it.
       ld (de),a           ; write target byte.
       inc de              ; next target address.
       inc hl              ; next source address.
       dec bc              ; decrement counter.
       ld a,b              ; loop counter high byte.
       or c                ; combine with low byte for zero check.
       jr nz,decom2        ; not done, go round again.
       ret
decom0 inc hl              ; next source byte.
       ld a,(hl)           ; fill byte we're expanding.
       ld (dbyte),a        ; store it here.
       inc hl
       ld a,(hl)           ; loop counter.
       inc hl
decom1 ex af,af'           ; store counter.
       ld a,(dbyte)
       ld (de),a           ; fill byte in.
       inc de              ; next byte of screen.
       dec bc              ; decrement overall counter.
       ld a,b              ; loop counter high.
       or c                ; hit zero?
       ret z               ; yes, decompression finished.
       ex af,af'           ; restore mini loop counter.
       dec a               ; decrement it.
       jr nz,decom1        ; not finished expanding yet.
       jr decom2

lev1sc defb 1,255,0,4,255,5,6,255,0,6,5,1,255,0,10,5,0,1,255,0,66
       defb 255,6,11,0,0,255,7,6,6,1,255,0,11,6,6,255,0,17,6,255,0,6,6,0,0
       defb 4,4,255,0,20,255,6,4,255,0,3,6,255,0,31,6,255,0,12,255,7,4
       defb 255,0,14,4,6,4,255,0,5,255,7,4,255,0,8,255,6,7,255,0,32,255,7,3
       defb 255,0,7,4,4,255,0,25,255,6,5,255,0,31,1,255,0,3,255,7,4,0,0
       defb 255,7,4,0,0,255,7,4,255,0,23,1,255,0,5,1,255,0,38,255,7,4
       defb 255,0,36,255,6,8,0,0,255,6,4,0,0,255,6,4,255,0,11,255,6,9
       defb 0,0,255,6,4,0,0,255,6,4,255,0,10,255,6,3

lev2sc defb 0,1,255,0,10,255,5,6,3,255,0,5,5,5,255,0,4,1,255,0,18,1,3
       defb 255,0,31,3,255,0,13,255,4,5,255,0,4,255,3,10,4,4,255,0,12,1
       defb 255,0,9,255,5,4,0,0,1,3,255,0,10,255,4,3,255,0,18,3,255,0,3
       defb 255,3,4,255,0,4,1,255,0,5,255,4,5,255,0,9,3,255,0,4,255,5,3
       defb 255,0,24,3,255,0,9,4,255,0,3,4,4,255,0,14,4,4,3,255,0,30,1,3,0
       defb 1,3,255,0,19,255,3,5,255,0,4,3,0,0,3,255,0,5,255,4,5,255,0,4
       defb 4,4,255,0,4,5,0,1,3,255,0,4,3,0,0,255,3,4,255,0,20,255,3,6
       defb 255,0,26,3,1,255,0,3,3,255,0,18,255,3,9,255,0,4,255,3,12,0,0
       defb 4,4,255,0,5,5,0,0,5,1,0,3,2,2,0,0,5,0,1,0,5,0,1,0,5,255,0,69

lev3sc defb 5,5,5,255,0,10,1,255,0,12,5,0,1,255,0,22,1,255,0,68,255,3,4
       defb 255,0,16,2,2,2,255,0,20,3,255,0,31,3,255,0,16,255,2,4,255,0,7,1
       defb 255,0,3,3,255,0,30,1,3,255,4,3,255,0,8,255,3,5,255,0,15,3
       defb 255,0,31,3,255,0,26,4,4,255,3,4,255,0,4,255,3,4,255,0,20,5,5
       defb 1,3,255,0,10,255,2,4,255,0,17,3,255,0,25,255,3,3,255,0,3,3
       defb 255,2,4,255,0,21,5,0,1,255,0,14,255,3,12,255,0,29,1,3,3
       defb 255,0,30,3,3,255,0,6

lev4sc defb 0,1,255,0,104,255,6,11,0,0,6,255,0,18,6,1,255,0,11,6,255,4,4
       defb 255,0,4,255,4,8,0,0,6,255,0,12,6,255,0,9,1,255,0,8,6,255,0,29
       defb 255,6,6,0,0,255,6,5,255,0,5,255,4,6,255,0,8,1,0,6,255,0,5
       defb 6,0,1,0,5,255,0,9,1,0,255,4,3,255,0,7,6,255,0,5,6,255,0,25
       defb 255,6,7,255,0,3,255,4,7,6,255,0,10,4,4,255,0,8,6,255,0,10,6
       defb 255,0,20,6,255,0,10,6,0,0,8,8,255,0,16,6,255,4,5,255,0,5,6
       defb 255,4,17,255,0,14,6,1,255,0,4,1,255,0,22,255,4,3,6,255,0,50,8
       defb 255,0,17

lev5sc defb 0,1,255,0,4,1,255,0,4,5,255,0,14,1,0,0,5,255,0,71,8,255,0,12
       defb 8,255,0,13,255,9,24,255,0,9,1,0,5,255,0,3,1,5,255,0,4,5,0,1
       defb 0,5,255,0,11,255,9,3,255,0,88,255,9,5,255,0,3,255,9,6,255,0,10
       defb 9,9,255,0,15,1,255,0,54,8,255,0,10,255,4,11,255,0,5,255,9,7
       defb 255,0,9,1,255,0,17,1,255,0,70,255,9,6,255,0,41,8,255,0,3,8
       defb 255,0,5,8,255,0,10

lev6sc defb 255,0,5,5,0,0,1,0,0,5,255,0,7,5,255,0,9,1,255,0,126,6,255,0,5
       defb 255,6,22,255,0,4,6,0,0,8,255,0,3,1,6,255,0,9,5,255,0,12,255,6,5
       defb 255,0,4,6,255,0,23,5,0,1,0,255,6,5,0,0,255,2,4,255,0,22,1,5
       defb 255,0,12,2,2,255,0,4,255,2,5,255,0,36,2,2,255,0,7,255,6,4
       defb 255,0,28,255,6,4,255,0,5,255,6,8,255,0,8,255,6,11,255,0,5
       defb 255,6,8,255,0,5,255,2,3,255,6,11,255,0,9,5,255,0,16,1,0,5
       defb 255,0,50,6,6,255,0,30,6,6,8,255,0,3

lev7sc defb 0,1,255,0,60,1,255,0,23,1,255,0,46,7,7,255,0,54,255,9,3,255,7,3
       defb 255,0,10,8,255,0,16,1,255,0,11,255,7,7,255,0,4,255,7,7,255,0,19
       defb 1,255,0,6,1,255,0,45,255,9,3,255,0,19,255,9,6,255,0,40,8,0,0,8
       defb 255,0,4,8,255,0,15,9,9,255,0,4,6,255,7,10,255,0,6,255,9,3
       defb 255,0,12,6,1,255,0,14,1,255,0,15,6,255,0,22,255,7,3,255,0,4,9,9
       defb 6,255,0,23,1,255,0,7,6,255,0,7,8,0,0,8,255,0,14


; Alien data table.
; byte 0: bits d0-d4 = screen number
;              d5-d7 = alien type.
; byte 1: x start
; byte 2: y start
; byte 3: x end
; byte 4: y end

alndat defb 64,32,56,128,56       ; screen 0 spinning thing.
       defb 64,0,104,128,104      ; screen 0 spinning thing.
       defb 01,128,0,128,240      ; screen 1 cart.
       defb 34,128,0,128,168      ; screen 2 frog.
       defb 98,0,32,48,32         ; screen 2 computer.
       defb 98,80,208,128,208     ; screen 2 computer.
       defb 03,128,0,128,96       ; screen 3 cart.
       defb 03,128,120,128,240    ; screen 3 cart.
       defb 36,80,24,80,96        ; screen 4 frog.
       defb 05,128,0,128,192      ; screen 5 cart.
       defb 37,24,16,24,176       ; screen 5 frog.
       defb 101,56,208,112,208    ; screen 5 computer.
       defb 70,0,144,128,144      ; screen 6 spinning thing.
       defb 38,40,168,40,208      ; screen 6 frog.
       defb 255

; Miscellaneous data for each screen.
; First two bytes give the player's starting coordinates,
; the next two point to address of each screen's title.

misc   defb 112,0
       defw miscm1         ; Out of memory.
       defb 8,152
       defw miscm2         ; Invalid argument.
       defb 88,240
       defw miscm3         ; Statement lost.
       defb 64,88
       defw miscm4         ; Tape loading error.
       defb 128,240
       defw miscm5         ; Invalid colour.
       defb 128,128
       defw miscm6         ; Out of screen.
       defb 128,0
       defw miscm7         ; End of file.

miscm1 defb 76
       defm 'Out of memory'
       defb 0
miscm2 defb 64
       defm 'Invalid argument'
       defb 0
miscm3 defb 70
       defm 'Statement lost'
       defb 0
miscm4 defb 56
       defm 'Tape loading error'
       defb 0
miscm5 defb 70
       defm 'Invalid colour'
       defb 0
miscm6 defb 76
       defm 'Out of screen'
       defb 0
miscm7 defb 84
       defm 'End of file'
       defb 0

; Music routines.

chan2c ld hl,ch2dat        ; start of tune.
       jr chan2d           ; process first note.
chan2b ld a,(chrep2)       ; number of repeats.
       dec a               ; decrement counter.
       jr z,chan2d         ; no more repeats, skip to next note.
       ld (chrep2),a       ; store iterations remaining.
       ld hl,(c2loop)      ; start again at beginning of loop.
       jr chan2d
chan2a ld a,(hl)           ; number of repeats.
       ld (chrep2),a       ; store it for later.
       inc hl              ; next byte.
       ld (c2loop),hl      ; remember beginning of loop.
chan2d ld (chptr2),hl      ; store channel marker.
chan2  ld hl,0             ; zero in hl.
       ld (snddat+8),hl    ; default channels a and b to zero volume.
       ld hl,(chptr2)      ; get pointer to tune data.
       ld a,(hl)           ; what's the next note?
       inc hl              ; next byte.
       ld (chptr2),hl      ; store channel marker.
       cp 98               ; beginning of loop marker.
       jr z,chan2a         ; store position.
       cp 99               ; end loop marker.
       jr z,chan2b         ; restore position.
       cp 255              ; end of tune.
       jr z,chan2c         ; restore position.
       call gfreq          ; get frequency in de registers.
       ld (snddat+2),de    ; put into channel 2.
       ld (snddat+9),a     ; set data for amplitude register.
chan1  ld hl,(chptr1)      ; get pointer to tune data.
chan1d ld a,(hl)           ; what's the next note?
       inc hl              ; next byte.
       ld (chptr1),hl      ; store channel marker.
       cp 98               ; beginning of loop marker.
       jr z,chan1a         ; store position.
       cp 99               ; end loop marker.
       jr z,chan1b         ; restore position.
       cp 100              ; is it a rest?
       jr z,w8912          ; yes, ignore note then.
       cp 255              ; end of tune.
       jr z,chan1c         ; restore position.
       call gfreq          ; get frequency in de registers.
       ld (snddat),de      ; put into channel 2.
       ld (snddat+8),a     ; set data for amplitude register.

; Write the contents of our AY buffer to the AY registers.
; On exit, a, bc, de and hl are corrupted.

w8912  ld hl,snddat        ; start of AY-3-8912 register data.
       ld e,0              ; register to start with.
       ld d,14             ; number of registers to write.

; If you only need to write one AY register, you can call w8912a instead.
; On entry, e contains register to write, hl contains address pointing to
; value of register, and d = 1.

w8912a ld b,&f4            ; select PSG register using port A of PPI (&f4).
       out (c),e
       ld bc,&f6c0         ; set PPI to 'select PSG register' mode (bit 7 = 1,
       out (c),c           ; bit 6 = 1) using port C (&f6), and select
                           ; register.
       xor a               ; set PPI to inactive mode (bit 7 = 0, bit 6 = 0).
       out (c),a

       ld b,&f4            ; use port A.
       ld a,(hl)           ; get value to write to register.
       out (c),a           ; write value to register.
       ld bc,&f680         ; set PPI to 'write PSG register' mode (bit 7 = 1,
       out (c),c           ; bit 6 = 0), and write register.
       xor a               ; set PPI to inactive mode.
       out (c),a

       inc e               ; next sound chip register.
       inc l               ; next byte to write.
       dec d               ; decrement loop counter.
       jr nz,w8912a        ; repeat until done.
       ret

chan1c ld hl,ch1dat        ; start of tune.
       jr chan1d           ; process first note.
chan1b ld a,(chrep1)       ; number of repeats.
       dec a               ; decrement counter.
       jr z,chan1d         ; no more repeats, skip to next note.
       ld (chrep1),a       ; store iterations remaining.
       ld hl,(c1loop)      ; start again at beginning of loop.
       jr chan1d
chan1a ld a,(hl)           ; number of repeats.
       ld (chrep1),a       ; store it for later.
       inc hl              ; next byte.
       ld (c1loop),hl      ; remember beginning of loop.
       jr chan1d

; Get frequency of note a and return in de registers.

gfreq  rlca                ; multiply note number by 2.
       ld e,a              ; put into de.
       ld d,0              ; zeroise high byte of de.
       ld hl,freqs         ; frequencies.
       add hl,de           ; find address of frequency.
       ld e,(hl)           ; low byte of note frequency.
       inc hl              ; next byte.
       ld d,(hl)           ; high byte of note frequency.
       ld a,16             ; use envelope generator for amplitude.
       ret

; This is our spooky music data.
; We shall have 4 special flag bytes:
;  98 = beginning of a loop, following byte gives iterations.
;  99 = end of loop.
; 100 = rest.
; 255 = end of tune.

; Channel 1 is the main tune.

ch1dat defb 98,48,100,99
       defb 7,98,5,100,99,7,100,100
       defb 7,9,10,9,98,5,100,99
       defb 8,100,100,7,10,11,12
       defb 98,5,100,99,10,98,4,100,99
       defb 9,8,98,8,100,99,8,9,10,7
       defb 98,5,100,99,7,100,100,7,9
       defb 10,9,98,5,100,99,8,100,100
       defb 7,10,11,12,98,5,100,99,10
       defb 98,4,100,99,9,8,98,5,100
       defb 99,8,9,10,11,9,8

       defb 98,2,11,100,100,100,100,100
       defb 11,10,11,12,11,10,11,100
       defb 100,100,100,100,11,10,11,12
       defb 11,10,12,100,100,100,100
       defb 100,10,11,12,13,12,10,11
       defb 100,100,100,100,100,11,10
       defb 9,11,10,9,99

       defb 98,2,7,8,9,10,9,8,7,100
       defb 100,100,100,100,7,8,9,10,9
       defb 8,7,100,100,100,100,100,8
       defb 9,10,10,9,8,8,100,100,100
       defb 100,100,6,7,8,8,7,6
       defb 7,100,100,100,100,100,99
       defb 255

; Channel 2 is the background melody.

ch2dat defb 98,8,0,4,2,99
       defb 98,4,1,5,3,99
       defb 98,3,4,5,3,99
       defb 4,2,1,255

; Frequencies used for the in-game music.
; A mere 14 notes are used so those are all we need to store.

; The tone periods used in the CPC version are different from those in the
; Spectrum version because the input clock is set at 1MHz instead of
; 1.7734MHz. To obtain a tone period for the CPC, multiply the Spectrum tone
; tone period by 1000000/1773400 = 0.563889.

freqs  defw 478,426,401,358,318,301
       defw 126,120,107,100,90,80
       defw 75,67

; CRTC register and value data for setting the screen to Spectrum size

crtcdt defb 1,32,2,42,6,24,7,30,12,&30,13,0

; The interrupt routine, which is called every 1/300th of a second (6 times
; per frame).

interr push af             ; store all the register pairs.
       push bc
       push de
       push hl
       push ix

inter0 ld a,0              ; the interrupt counter (modified).
       inc a               ; increment it.
       ld (inter0+1),a     ; store the incremented counter.

       cp 1                ; top of screen?
       jr z,inter1         ; set colours for top half of screen.
       cp 5
       jr z,inter2         ; set colours for bottom half of screen.
       cp 6                ; bottom of screen?
       jr c,intret         ; no, so return from interrupt.

       xor a               ; reset interrupt counter to zero.
       ld (inter0+1),a

intret pop ix              ; restore all the register pairs.
       pop hl
       pop de
       pop bc
       pop af
       ei
       ret

; Select inks for top half of screen.

inter1 ld hl,inkta0        ; address of ink table.
       call inter3         ; set inks.
       jr intret           ; return from interrupt.

; Select inks for bottom half of screen.

inter2 ld hl,inkta2        ; address of ink table.
       ld b,74             ; insert a delay before setting inks to ensure that
inte2a djnz inte2a         ; change of inks occurs on exactly the right line.
       call inter3         ; set inks.

       ld a,(chcntr)       ; get value of counter used for playing music.
       inc a               ; increment it.
       and 15              ; range of values is 0-15.
       ld (chcntr),a       ; store the new value.
       ld hl,musoff        ; check if music is turned off.
       or (hl)
       call z,chan2        ; no, so play music.
       jr intret           ; return from interrupt.

inter3 ld bc,&7f01         ; use gate array (&7fxx) to select first ink to
                           ; set; c contains ink number.
inter4 ld a,(hl)           ; get value to change ink to from table.
       out (c),c           ; select ink.
       out (c),a           ; change value of ink.
       inc c               ; go to next ink.
       inc hl              ; go to next entry in ink table.
       ld a,c
       cp 4                ; all 3 inks changed?
       jr nz,inter4        ; no, so repeat for remaining inks.
       ret

inter5 xor a
       ld (chcntr),a
       ret

; Palette tables for use by the interrupt routine.

inkta0 defb &54,&54,&54        ; all inks black.
inkta1 defb &4c,&57,&4b        ; red, light blue, white.
inkta2 defb &55,&53,&4a        ; blue, cyan, yellow.

inkta3 defb &4c,&57,&40        ; red, light blue, grey.
inkta4 defb &4c,&44,&40        ; red, dark blue, grey.
inkta5 defb &5c,&44,&40        ; brown, dark blue, grey.
inkta6 defb &5c,&44,&54        ; brown, dark blue, black.
inkta7 defb &5c,&54,&54        ; brown, black, black.


dbyte  equ $
scno   equ dbyte+1         ; screen number.
albuff equ scno+1          ; alien buffer.
dflag  equ albuff+18       ; indicates player is dead when non-zero.
lives  equ dflag+1         ; lives counter.
numobj equ lives+1         ; number of objects on current screen.
score  equ numobj+1        ; player's score.
hisc   equ score+2         ; highest score.
tdir   equ hisc+2          ; temporary direction.
jdir   equ tdir+1          ; jump direction.
dispx  equ jdir+1
dispy  equ dispx+1
fallf  equ dispy+1         ; falling flag, indicatesd player is in mid-air.
musoff equ fallf+1         ; marker for playing music, 0=on, 1=off.
chcntr equ musoff+1        ; counter used for playing music.
chrep1 equ chcntr+1        ; channel 1 repeat.
chrep2 equ chrep1+1        ; channel 2 repeat.
chptr1 equ chrep2+1        ; channel pointer.
chptr2 equ chptr1+2        ; channel pointer.
c1loop equ chptr2+2        ; channel loop.
c2loop equ c1loop+2        ; channel loop.
tmp1   equ c2loop+2        ; temporary store.
toofar equ tmp1+1          ; fallen too far flag.
direct equ toofar+1        ; old direction player was facing, 0=left.
ndirec equ direct+1        ; new direction player is facing, 8=right.
tmpspr equ ndirec+1        ; start address to store temporary sprite (used to
                           ; store screen data below game over text).
termin equ tmpspr+1        ; end.
