;  bonz.asm - A tiny game for Amstrad CPC computers.
;
;  Copyright (C) 2003, Xavier Glattard, <xavier.glattard@netcourrier.com>.
;
;  This program is free software; you can redistribute it and/or modify
;  it under the terms of the GNU General Public License as published by
;  the Free Software Foundation; either version 1, or (at your option)
;  any later version.
;
;  This program is distributed in the hope that it will be useful,
;  but WITHOUT ANY WARRANTY; without even the implied warranty of
;  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;  GNU General Public License for more details.
;
;  You should have received a copy of the GNU General Public License
;  along with this program; if not, write to the Free Software
;  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

org #4000
run #4000

; BIOS CALLs
.ams_mode   equ #BC0E
.ams_ink    equ #BC32
.ams_border equ #BC38
.ams_gaddr  equ #BC1D

.ams_bgrnd equ #BB9F

.ams_frame equ #BD19
.ams_timer equ #BD0D

.ams_locate equ #BB75
.ams_print equ #BB5A
.ams_inkey equ #BB1B
.ams_waitkey equ #BB18

; Constants to be defined according to the video mode
.scr_width equ 320
.scr_height equ 200
.scr_orgX equ 160 ; needs to be the first pixel of a screen byte (multiple of 4)
.scr_orgY equ 8

.begin
	ld ix,data2  ; start of uninitialized datas

	; uncompress the sprites
	ld iy,ball
	ld de,ball_data
	call sprite_expand
	ld iy,slab
	ld de,slab_data
	call sprite_expand
	ld iy,slabtop
	ld de,slabtop_data
	call sprite_expand

; Start a new game
.new_game
	ld (ix+energyi),100 ; you start with 100%

	ld hl,level
	ld b,level_nfig ; level number figures
.new_game01
	ld (hl),'0'
	inc hl
	djnz new_game01

; Start a new level
.init_level

.init_data
	; energy gauge have to be refilled
	ld a,(energy)
	ld (energyD),a
	ld de,energy_empty
	ld (energy_adr),de

	; initialize variables
	ld hl,data_end
	ld de,data_end+1
	ld bc,data_init_end-data_end-1
	ldir ; (data_end)==0, then we fill with 0

.init_scr
	; initialize the screen and print some text
	ld hl,init_msg
	call print

.init_board
	; initialize the board
	ld a,#03 ; code for an initialized and non empty slab
	ld (board),a    ; start
	ld (board+24),a ; finish

	ld d,5 ; 5 slabs
.draw_board00
	ld e,5 ; 5 slabs
.draw_board01
	push de
	dec e ; slab coordinates from 0 to 4
	dec d
	call draw_slab
	pop de
	dec e
	jr nz,draw_board01
	dec d
	jr nz,draw_board00

	; de is already 0
	;ld de,0 ; first ball pos
	ld (ballPos),de
	ld (ix+ballDZi),#80
	ld a,(ballDDZ0)
	ld (ballDDZ),a

	jr run

.run_loop
	call move_ball
	call ams_frame
	call clear_ball
	bit 0,(ix+tilti) ; is a slab highlighted ?
	call nz,untilt_slab
.run
	; discover the slabs (from warfog)
	ld a,(ix+slabNumi) ; current slab
	or a
	jr z,run03 ; the last one ? (a=0)
	dec (ix+slabNumi)
	ld de,(slabPos)
	push de
	ld hl,(ballPos)
	call slab_show ; if the slab is near the ball then display it
	pop de
	; compute next slab coordinates
	inc e
	ld a,5
	cp e
	jr nz,run04
	ld e,0
	inc d
.run04
	ld (slabPos),de

.run03
	ld a,(ballZ)
	or a ; Z=0 ?
	jr nz,run01 ; No, the ball is in the air

	; Last slab ?
	ld de,(ballPos)
	ld a,4
	cp e
	jr c,run_fall ; too far !
	cp d
	jr c,run_fall ; too far !
	jr nz,run_not_last
	cp e
	jp z,level_up ; Finish !
.run_not_last

	call get_slab0 ; get addr of the current slab code
	ld a,(hl)
	rrca ; Init ok ? (always true)
	rrca ; is a slab here ?
	jr c, run02 ; yes, then the ball bounces !
.run_fall
	; else it falls !
	set 0,(ix+ball_fallingi)
	jr run01

.run02
	; energy decrease at each bound
	dec (ix+energyDi)
	rrca ; simple slab ?
	jr nc,run_tilt ; yes

.run_bonus
	; Bonus slab !
	and #F ; deletes the rrca-rotated bits
	inc a ; will never be zero
	add a,a
	add a,a ; times 4
	add a,(ix+energyDi) ; have to be added to energy
	ld (ix+energyDi),a
	ld (hl),#03 ; the slab is now a simple one

.run_tilt
	; the ball hits the slab that is highlighted
	set 0,(ix+tilti)
	ld de,(ballPos) ; slab position
	call draw_slab

	call bounce ; activates some channges
	; initialize the board uncovering process (warfog)
	ld de,0 ; first slab
	ld (slabPos),de
	ld (ix+slabNumi),5*5
.run01
	call draw_ball
	jp nc,game_over ; the ball is falling out of the screen

	ld de,(energy_adr) ; energy gauge on the screen
	ld a,(energyD) ; energy delta
	or a ; zero ?
	jr z,run_energy99 ; do nothing
	jp m,run_energy00

	; increases energy !
	ld h,a ; save delta
	ld a,(ix+energyi)
	cp 100 ; Maximum ?
	jr nz,run_energy01
	ld (ix+energyDi),0 ; can't be increase more
	jr run_energy99
.run_energy01
	ld a,h ; reload delta
	dec a
	inc (ix+energyi)
	ld (energyD),a

	; fills the gauge
	call prev_line
	ld (energy_adr),de
	ld a,255
	ld (de),a
	inc de
	ld (de),a
	jr run_energy99

.run_energy00
	; decreases energy !
	dec (ix+energyi)
	jp z,game_over2
	inc a
	ld (energyD),a

	; drains the gauge
	xor a
	ld (energy_adr),de
	ld (de),a
	inc de
	ld (de),a
	dec de
	call next_line
	ld (energy_adr),de

.run_energy99

.run_inkey
	call ams_inkey
	jp nc,run_loop

	call run_testkey
	
	jp run_loop

.run_keys equ $
	db 'j','i','k','u'
.run_numkeys equ $-run_keys

.run_testkey
	ld hl,run_keys
	ld bc,run_numkeys
	cpir
	ret nz ; not found
	dec (ix+energyDi) ; each key hit costs 1%
	ld b,c
	inc b ; b is the index of the typed char

	djnz not_u
.run_forward
	inc (ix+ballDY2i)
.not_u	djnz not_k
.run_backward
	dec (ix+ballDY2i)
.not_k	djnz not_i
.run_right
	inc (ix+ballDX2i)
.not_i	djnz not_j
.run_left
	dec (ix+ballDX2i)
.not_j	
	ret

.level_up ; To the next level
	; changes the level number
	ld b,5
	ld hl,level+4 ; last figure
.level_up01
	inc (hl)
	ld a,'9'+1
	cp (hl)
	jr nz,level_up02 ; that all
	ld (hl),'0'
	dec hl ; next figure
	djnz level_up01
.level_up02
	jp init_level ; start the new level

.game_over2
	ld hl,exhausted_msg
	jr game_over0
.game_over
	ld hl,lost_msg
.game_over0
	call print
	ld hl,game_over_msg
	call print
.game_over01
	call ams_waitkey
	cp ' '
	jr nz,game_over01

	jp new_game ; start a new game

; Print a string until char=255
.print

.print00
	ld a,(hl)
	cp 255
	ret z
	push hl
	call ams_print
	pop hl
	inc hl
	jr print00

; Display a slab according to the position of the ball	
.slab_show ; de=slabYX, hl=ballYX
	; compute the distance
	ld a,l
	sub e
	jr nc, slab_show01
	neg
.slab_show01
	ld l,a ; l = abs(X2-X1)
	ld a,h
	sub d
	jr nc, slab_show02
	neg
.slab_show02
	ld h,a ; h = abs(Y2-Y1)
	; loads the greatest into h
	cp l
	jr nc,slab_show03
	ld h,l
	ld l,a
.slab_show03
	ld a,h
	cp 3 ; dist<=2 ?
	ret nc ; too far, do nothing
	add a,l
	cp 4 ; dist<=3
	ret nc ; too far, do nothing

	; this area is now uncovered !
	push de
	call get_slab ; get slab code
	pop de
	ld a,(hl)
	rrca
	ret c ; already initialized
	
.slab_show04
	ld a,r ; (almost) random value
	ld c,a
	rl c ; bit 7 is always zero
	xor a
	; bonus slab ? 1 into 4
	rl c ; one bit into carry
	jr nc,slab_show06 ; no
	rl c ; one bit into carry
	jr nc,slab_show06 ; no
	; load the bonus value into carry
	rl c ; one bit into carry
	rla ; load carry into a
	rl c ; one bit into carry
	rla ; load carry into a
	scf ; activate bonus
.slab_show06
	rla ; load carry into a
	; is a slab here ?
	rl c ; one bit into carry
	rla ; load carry into a
	scf
	adc a,a ; position initialized
	ld (hl),a
	jr draw_slab00

; Return addres of the code byte of a slab
.get_slab0
	ld de,(BallPos)	; the slab under the ball
.get_slab ; de=YX on a 5x5 board
	ld hl,board
	ld a,d
	add a,a ; 5 times
	add a,a
	add a,d
	add a,e
	ld d,0
	ld e,a
	add hl,de
	ret	

; draw a slab
.draw_slab ; de=YX
	bit 0,(ix+tilti) ; is it highlighted ?
	jr z,draw_slab03
	ld c,#FF ;  pen 3 = yellow (highlighted slab)
	jr draw_slab02

.draw_slab03
	push de
	call get_slab
	ld a,(hl)
	pop de
.draw_slab00
	rrca
	jr nc,draw_slab01 ; not yet initialized
	rrca
	jr nc,no_slab ; no slab here !
	rrca
	jr c,draw_slab04
	ld c,#F0 ; pen 1 = grey (simple slab)
	jr draw_slab02
.draw_slab04
	ld c,#0F ; pen 2 = orange (bonus slab)
	jr draw_slab02
.draw_slab01
	ld c,#00 ; pen 0 = black (unitialized slab)

.draw_slab02 ; Draw slab frame
	call slab_S ; screen pos of the slab
	push bc
	push hl
	push de
	ld iy,slab
	call sprite_adr
	ld (slab_adr),de
	ld (slab_data_ptr),hl
	call draw_sprite
	pop de
	pop hl
	pop bc

.draw_slabtop ; Draw slab top
	ld iy,slabtop
	ld (iy+slabtop_colori),c
	call sprite_adr
	ld (slabtop_adr),de
	ld (slabtop_data_ptr),hl
	jp draw_sprite

; Compute screen pos of the slab (depends on DDZ)
.slab_S
	ld h,0
	ld l,d
	ld d,h
	call mult30
	ex de,hl
	call mult30
	ex de,hl
	ret

; clear slab frame (no slab here)
.no_slab
	push bc
	ld bc,clear_poke ; method that clear slab area
	ld iy,slab
	ld (iy+sprite_pokei),c
	ld (iy+sprite_pokei+1),b
	call slab_S
	call sprite_adr
	call draw_sprite
	ld bc,sprite_poke_def ; restore slab poke method
	ld (iy+sprite_pokei),c
	ld (iy+sprite_pokei+1),b
	pop bc
	ret

; hl = hl * 30
.mult30
	push de
	ld b,3
	ld d,h
	ld e,l
	; compute (((val*3)*2+val)*2+val)*2 = 30*val
.mult30_01
	add hl,hl
	add hl,de
	djnz mult30_01
	add hl,hl
	pop de
	ret

; Un-hightlight slab
.untilt_slab
	res 0,(ix+tilti) ; slab is not highlighted
	ld iy,slabtop
	ld (iy+slabtop_colori),#F0 ; pen 1 = grey (simple slab)
	ld de,(slabtop_adr)
	ld hl,(slabtop_data_ptr)
	jp draw_sprite
	
.move_ball
	; first, is the ball falling ?
	bit 0,(ix+ball_fallingi)
	jr nz,ball_fall ; yes, the simple way
.move_ball00
	; compute new vertical speed
	ld a,(ballDZ)
	sub (ix+ballDDZi)
	ld (ballDZ),a
	; scale the speed value for display
	sra a
	sra a
	sra a
	sra a ; signed division by 16 (why not ?!...)
	cp #F8 ; compare ballDZ to #8x
	jr z,move_ball00 ; and skip these values (-8) for symetry reason (-7 to +7)

	; compute the coordinates
	ld b,a
.move_ball03
	; Z = Z + DZ (8 bits)
	ld a,(ballZ)
	add a,b
	jp po,move_ball04
	; overflow -> set a flag
	set 0,(ix+ballZi+1)
.move_ball04
	ld (ballZ),a
	; X = X + DX (16 bits)
	ld hl,(ballX)
	ld c,(ix+ballDXi)
	call extend_bc
	add hl,bc
	ld (ballX),hl
	; Y = Y + DY (16 bits)
	ld hl,(ballY)
	ld c,(ix+ballDYi)
	call extend_bc
	add hl,bc
	ld (ballY),hl

	ret

.ball_fall
	ld b,#F0 ; fast fall
	jr move_ball03 ; downward in right line !

; The ball changes its direction when it hits a slab
.bounce
	; Active changes
	ld h,(ix+ballDDZ0i)
	ld (ix+ballDDZi),h
	ld hl,(ballDS2)
	ld (ballDS),hl
	ld hl,0	; annule
	ld (ballDS2),hl

	ld a,(ballI)
	add a,(ix+ballDXi)
	ld (ballI),a
	ld a,(ballJ)
	add a,(ix+ballDYi)
	ld (ballJ),a

	ret

; Draw the ball	
.draw_ball
	bit 0,(ix+ballZi+1) ; overflow ?
	jr z,draw_ball01 ; no, continue
	or a ; carry=0 means the ball is out of the screen
	ret

.draw_ball01
	ld de,(ballX)
	ld hl,(ballY)

	ld a,(ballZ)
	call translateZ ; move origin (sprites are always drawn at Z=0)

	ld iy,ball
	call sprite_adr
	ret nc ; is the ball gone out of the screen (downward) ?
	ld (ball_adr),de
	call draw_sprite

	ld a,(ballZ)
	neg
	call translateZ ; restore origin

	scf ; all is right, continue
	ret

; Undraw the ball (restore background)
.clear_ball
	ld de,(ball_adr)
	ld iy,ball
	jp clear_sprite

; Compute address of the sprite in the screen memory
.sprite_adr	; de=X hl=Y iy=sprite
	call compute ; 3D to 2D coordinates
	ld a,h
	rla ; test sign bit
	ccf ; complement
	ret nc ; y<0

	; shift coordinates according to sprite center
	ld c,(iy+sprite_DXi) ; X offset
	call extend_bc
	ex de,hl	; de=Y hl=X
	add hl,bc
	ex de,hl	; de=X+dX hl=Y
	ld c,(iy+sprite_DYi) ; Y offset
	call extend_bc
	add hl,bc

	call ams_gaddr
	ex de,hl	; de=screen memory addr
	ld a,c ; 'pixel mask'

	; compute sprite bloc according to pixel mask
	ld l,(iy+sprite_datai)
	ld h,(iy+sprite_datai+1) ; address of 1st bloc
	ld c,(iy+sprite_sizei)
	ld b,(iy+sprite_sizei+1) ; bloc size
.sprite_adr01
	rlca
	ret c ; this bloc !
	add hl,bc ; next bloc
	jr sprite_adr01

; Draw a sprite (generic method)
.draw_sprite	; iy=sprite hl=sprite_data de=ecran
	ld a,(iy+sprite_Hi) ; height
	ld (ix+linei),a ; first line
	ld c,(iy+sprite_savei) ; bc = bg save area
	ld b,(iy+sprite_savei+1)
.draw_sprite01
	ld a,(iy+sprite_Wi) ; width
	ld (ix+rowi),a ; first row
	push de

.draw_sprite03
	call sprite_poke
	dec (ix+rowi)
	jr nz,draw_sprite03 ; next row

	pop de
	call next_line
	dec (ix+linei)
	jr nz,draw_sprite01 ; next line
	ret

; Poke a value into screen memory (virtual method)
.sprite_poke ; iy=sprite hl=datas de=screen bc=save a=(de)
	push hl
	; get poke method addr
	ld h,(iy+sprite_pokei+1)
	ld l,(iy+sprite_pokei)
	ex (sp),hl
	ret ; jump to the real procedure

; Start of simple sprite poke method : never save background
.sprite_poke_def ; iy=sprite hl=donnees de=ecran
	ld a,(hl)
.sprite_poke_def2
	ld (de),a
.sprite_poke_def3
	inc hl
	inc de
	ret

; 'Extended' sprite poke method (data+mask)
.sprite_poke_ext ; iy=sprite hl=datas de=screen bc=save
	ld a,(de)
	ld (bc),a ; save background
	inc bc

.sprite_poke_ext0
	and (hl) ; keep unused bits
	inc hl
.sprite_poke_ext1
	or (hl)  ; set used bits
	jr sprite_poke_def2

; Start of clear poke method -> always poke zero in the whole sprite box
.clear_poke
	xor a ; poke 0
	jr sprite_poke_def2

; Start of slab top poke method : change color according to a member value
.slabtop_poke
	ld a,(de)
	and (hl)
	inc hl
	ld b,a ; sauve
	ld a,(hl)
	and (iy+slabtop_colori)
	or b
	jr sprite_poke_def2

; Compute addr of the prev line of the screen
.prev_line ; de=adr
	push bc
	ex de,hl
	ld bc,#F800
	add hl,bc
	ld a,h
	cp #C0
	jr nc,prev_line01 ; out of the screen ?
	ld bc,#3FB0
	add hl,bc
	scf ; set carry, used as an overflow flag
.prev_line01
	ex de,hl
	pop bc
	ret

; Compute addr of the next line of the screen
.next_line ; de=adr
	push bc
	ex de,hl
	ld bc,#800
	add hl,bc
	jr nc,next_line01 ; out of the screen ?
	ld bc,#C050
	add hl,bc
	scf ; set carry, used as an overflow flag
.next_line01
	ex de,hl
	pop bc
	ret

; Un-draw sprite -> restore saved background
.clear_sprite	; iy=sprite de=screen
	ld a,(iy+sprite_Hi) ; line number
	ld b,0
	ld l,(iy+sprite_savei)
	ld h,(iy+sprite_savei+1)
.clear_sprite01
	push de
	ld c,(iy+sprite_Wi) ; size of a line
	ldir
	pop de
	call next_line
	dec a
	jr nz,clear_sprite01
	ret

; Copy a row of a sprite
.sprite_row_copy ; iy=sprite hl=data de=dest a=0
	push hl
	; get copy method addr
	ld h,(iy+sprite_row_copyi+1)
	ld l,(iy+sprite_row_copyi)
	ex (sp),hl
	ret ; saut vers la procedure virtuelle

; Default row copy method
.sprite_rcopy_def ; iy=sprite hl=data de=dest a=0
	ld b,a ; b=0, so bc=width
	ldir ; copy a line
	; one 'empty' byte reserved for the shifted sprite
	ld (de),a ; data=00 -> draw nothing
	inc de
	ret

; 'Extended' sprite row copy
.sprite_rcopy_ext ; iy=sprite hl=data de=dest a=0
	ld b,a ; b=0, so bc=width
	ldir ; copy a line
	; two 'empty' bytes reserved for the shifted sprite
	dec a ; a=#FF
	ld (de),a ; mask=FF -> keep background
	inc de
	xor a ; a=0
	ld (de),a ; data=00 -> draw nothing
	inc de
	ret

; Copy a bloc of a sprite
.sprite_copy ; iy=sprite de=dest
	ld h,(iy+sprite_datai+1)
	ld l,(iy+sprite_datai)
	ld b,(iy+sprite_Hi)
	ld c,(iy+sprite_row_sizei)
	xor a ; a=0

	push de
.sprite_copy00
	push bc
	call sprite_row_copy
	pop bc
	djnz sprite_copy00 ; next line

.sprite_copy01
	pop hl
	inc (iy+sprite_Wi) ; for the 'empty' bytes
	; new address of the sprite
	ld (iy+sprite_datai+1),h 
	ld (iy+sprite_datai),l

	ret

; Expand sprite data for fastest use
.sprite_expand	;	iy=sprite de=dest
	call sprite_copy ; raw copy (no shift)
	call sprite_shift ; 1st shift
	call sprite_shift ; 2nd shift
	; and a 3rd one...

.sprite_shift  ; hl=src de=dest
	push de ; save destination
	ld b,(iy+sprite_Wi)
	ld c,(iy+sprite_Hi)

.sprite_shift01
	ld (ix+shifted_bitsi),#00 ; begining of a data line
	ld (ix+shifted_mbitsi),#88 ; begining of a mask line
	push bc
.sprite_shift02
	call sprite_cell_shift
	djnz sprite_shift02 ; next cell
	pop bc
	dec c
	jr nz,sprite_shift01 ; next row

	; de = start of the next bloc = next destination
	pop hl ; the previous destination becomes the next source
	ret

; Shift one cell of sprite data (virtual method)
.sprite_cell_shift ; hl=src ix=dest
	push hl
	; get copy method addr
	ld h,(iy+sprite_cshifti+1)
	ld l,(iy+sprite_cshifti)
	ex (sp),hl
	ret ; saut vers la procedure virtuelle

; Shift one 'extended' sprite cell (2 bytes => mask and data)
.sprite_cshift_ext ;  hl=src de=dest
	ld a,(shifted_mbits)
	call sprite_byte_shift
	ld (shifted_mbits),a
	; then data byte

; Default cell shift method (1 byte => data only)
.sprite_cshift_def
	ld a,(shifted_bits)
	call sprite_byte_shift
	ld (shifted_bits),a
	ret

; Shift one sprite byte (data or mask)
.sprite_byte_shift ;  hl=src de=dest a=previous shifted bits
	ld c,(hl) ; byte to be computed
	inc hl
	rrc c ; rotation
	xor c
	and #88
	xor c ; get shifted bits

	ld (de),a ; write the result
	inc de ; ready for the next byte

	ld a,#88 ; only the shifted bits
	and c ; to be used with the next byte
	rrca ; but they need to be swapped
	rrca
	rrca
	rrca

	ret

; Extend sign of c thru b (from signed 8bit value to signed 16bit value)
.extend_bc
	ld b,0
	bit 7,c ; test sign
	ret z
	ld b,#FF
	ret

; Change orign of the screen coordinates
.translate
	ld bc,(orgY)
	add hl,bc
	ld (orgY),hl
	ld hl,(orgX)
	add hl,de
	ld (orgX),hl
.translateZ ; change Z coordinates only
	add a,(ix+orgZi)
	ld (orgZ),a
	ret

; Compute 2D screen coordinates from 3D coordinates according to 3D origin
.compute
	push hl
	ex de,hl
	ld de,(orgX)
	add hl,de
	ex (sp),hl
	ld de,(orgY)
	add hl,de
	pop de
	call ortho

	ld c,(ix+orgZi)
	call extend_bc
	add hl,bc

	ret

; Compute 2D screen coordinates from 3D(iso) coordinates
; Screen origin = top left corner
.ortho
	push hl ; save Y
	push de ; save X

	; Compute X = Y - X + X0
	or a ; carry = 0
	ex de,hl
	sbc hl,de
	ex de,hl
	ld hl,scr_orgX
	add hl,de

	pop de ; restore old X
	ex (sp),hl ; save new X, restore Y

	; Compute Y = (Y + X) / 2 + Y0
	add hl,de
	sra h
	rr l ; divide hl by 2
	ld de,scr_orgY
	add hl,de

	pop de ; restore new X

	ret

.code_end

; ********************************************

.lost_msg
	db 15,2 ; pen
	db 31,13,6 ; locate
	db 'Lost in space...',255
.exhausted_msg
	db 15,2 ; pen
	db 31,14,6 ; locate
	db 'Out of Energy',255
.game_over_msg
	db 15,3 ; pen
	db 31,16,8 ; locate
	db 'GAME OVER'
	db 31,28,25 ; locate
	db 'PRESS ',15,2,'<',15,3,'SPACE',15,2,'>',255

.init_msg
	db 4,1 ; mode 1
	db 29,0,0 ; border 0
	db 28,0,0,0 ; ink 0,0
	db 28,1,13,13 ; ink 1,13
	db 28,2,15,15 ; ink 2,15
	db 28,3,25,25 ; ink 3,25
.energy_msg
	db 15,2 ; pen
	db 31,1,25 ; locate
	db 'ENERGY'
	db 31,1,11 ; locate
	db '%'
.CPCo_msg
	db 31,24,1 ; locate
	db 15,1,'www.',15,3,'cpc',15,2,'oxygen',15,1,'.net'
.BONZ_msg
	db 31,35,25 ; locate
	db 15,2,175,15,3,'BONZ',15,2,'!'
.level_msg
	db 15,2 ; pen
	db 31,1,1 ; locate
	db 'LEVEL '
	db 15,3 ; pen
.level
.level_nfig equ 5
	ds level_nfig,'0'

	db 255 ; fin de chaine

; *******************************************

.data

.ball
.ball_width equ 3
.ball_height equ 12

.sprite_DXi equ $-ball
	db -6
.sprite_DYi equ $-ball
	db 6
.sprite_Wi equ $-ball
	db ball_width
.sprite_Hi equ $-ball
	db ball_height
.sprite_sizei equ $-ball
	dw ball_width+1 *ball_height *2
.sprite_row_sizei equ $-ball
	db ball_width *2
.sprite_pokei equ $-data
	dw sprite_poke_ext
.sprite_cshifti equ $-data
	dw sprite_cshift_ext
.sprite_row_copyi equ $-data
	dw sprite_rcopy_ext

.sprite_datai equ $-ball
	dw ball_data_base
.sprite_savei equ $-ball
	dw ball_save

.ball_data_base
	db #FF,#00, #00,#5F, #FF,#00
	db #CC,#30, #00,#2F, #33,#CC
	db #88,#61, #00,#97, #11,#CE
	db #88,#52, #00,#3F, #11,#EE
	db #00,#E1, #00,#97, #00,#EF
	db #00,#70, #00,#2F, #00,#BF
	db #00,#A1, #00,#97, #00,#4F
	db #00,#70, #00,#4B, #00,#0F
	db #88,#10, #00,#A5, #11,#A4
	db #88,#20, #00,#F0, #11,#68
	db #CC,#00, #00,#B0, #33,#C0
	db #FF,#00, #00,#50, #FF,#00

.slab
.slab_width equ 6
.slab_height equ 10
	db -12 ; DX
	db -4 ; DY
	db slab_width
	db slab_height
	dw slab_width+1 *slab_height
	db slab_width
	dw sprite_poke_def
	dw sprite_byte_shift
	dw sprite_rcopy_def

	dw slab_data_base
	dw 0 ; slab_save (not used)

.slab_data_base
	db #80,#00,#00,#00,#00,#10
	db #A0,#00,#00,#00,#00,#70
	db #80,#80,#00,#00,#10,#F0
	db #80,#20,#00,#00,#70,#F0

	db #40,#00,#80,#10,#F0,#E0
	db #10,#00,#20,#70,#F0,#80
	db #00,#40,#00,#F0,#E0,#00
	db #00,#10,#00,#F0,#80,#00
	db #00,#00,#40,#E0,#00,#00
	db #00,#00,#10,#80,#00,#00

.slabtop
.slabtop_width equ 6
.slabtop_height equ 12
	db -12 ; DX
	db 2 ; DY
	db slabtop_width
	db slabtop_height
	dw slabtop_width+1 *slabtop_height *2
	db slabtop_width *2
	dw slabtop_poke
	dw sprite_cshift_ext
	dw sprite_rcopy_ext

	dw slabtop_data_base
	dw 0 ; slab_save (not used)
.slabtop_color
.slabtop_colori equ $-slabtop
	db 0 ; mask

.slabtop_data_base
	db #FF,#00, #FF,#00, #EE,#11, #77,#88, #FF,#00, #FF,#00
	db #FF,#00, #FF,#00, #88,#77, #11,#EE, #FF,#00, #FF,#00
	db #FF,#00, #EE,#11, #00,#FF, #00,#FF, #77,#88, #FF,#00
	db #FF,#00, #88,#77, #00,#FF, #00,#FF, #11,#EE, #FF,#00
	db #EE,#11, #00,#FF, #00,#FF, #00,#FF, #00,#FF, #77,#88
	db #88,#77, #00,#FF, #00,#FF, #00,#FF, #00,#FF, #11,#EE

	db #88,#77, #00,#FF, #00,#FF, #00,#FF, #00,#FF, #11,#EE
	db #EE,#11, #00,#FF, #00,#FF, #00,#FF, #00,#FF, #77,#88
	db #FF,#00, #88,#77, #00,#FF, #00,#FF, #11,#EE, #FF,#00
	db #FF,#00, #EE,#11, #00,#FF, #00,#FF, #77,#88, #FF,#00
	db #FF,#00, #FF,#00, #88,#77, #11,#EE, #FF,#00, #FF,#00
	db #FF,#00, #FF,#00, #EE,#11, #77,#88, #FF,#00, #FF,#00

.data2

.ballDDZ0i equ $-data2
.ballDDZ0 db #08 ; needs to be a power of 2 => 1,2,4,8,16,32,...

.data_end ; end of initialized datas
	db 0

;*****************************************
; these data are set to zero when a new game starts

.energyi equ $-data2
.energy ds 1

.rowi equ $-data2
.row ds 1
.linei equ $-data2
.line ds 1

.slabPosi equ $-data2
.slabPos ds 2
.slabNumi equ $-data2
.slabNum ds 1

.tilti equ $-data2
.tilt ds 1 ; Is the ball highlighted ?

.orgXi equ $-data2
.orgX ds 2
.orgYi equ $-data2
.orgY ds 2
.orgZi equ $-data2
.orgZ ds 1

.ball_fallingi equ $-data2
.ball_falling ds 1

.ballS
.ballXi equ $-data2
.ballX ds 2 ; 16bit value
.ballYi equ $-data2
.ballY ds 2 ; 16 bit value
.ballZi equ $-data2
.ballZ ds 1 ; 8 bit value
       ds 1 ; overflow flag
.ballDZi equ $-data2
.ballDZ ds 1
.ballDDZi equ $-data2
.ballDDZ ds 1

.ballPos
.ballIi equ $-data2
.ballI ds 1
.ballJi equ $-data2
.ballJ ds 1

.ballDS
.ballDXi equ $-data2
.ballDX ds 1
.ballDYi equ $-data2
.ballDY ds 1

.ballDS2
.ballDX2i equ $-data2
.ballDX2 ds 1
.ballDY2i equ $-data2
.ballDY2 ds 1

.board_width equ 5
.board_height equ 5
.boardi equ $-data2
.board
	ds board_width*board_height

.data_init_end ; end of zero initialized datas

;**************************************
; these datas are not initialized to zero

.energyDi equ $-data2
.energyD ds 1 ; initialied to 100
.energy_empty equ #EF30
.energy_adr ds 2 ; initialied to energy_empty

; local variables (expand sprite methods)
.shifted_bitsi equ $-data2
.shifted_bits ds 1
.shifted_mbitsi equ $-data2
.shifted_mbits ds 1

.ball_adr ds 2
.ball_save ds ball_width+1 *ball_height
.ball_data ds ball_width+1 *ball_height *2*4

.slab_adr ds 2
.slab_data_ptr ds 2
; .slab_save ds slab_width+1 *slab_height ; non used
.slab_data ds slab_width+1 *slab_height *4

.slabtop_adr ds 2
.slabtop_data_ptr ds 2
; .slabtop_save ds slabtop_width+1 *slabtop_height ; not used
.slabtop_data ds slabtop_width+1 *slabtop_height *2*4

.data2_end
