// Copyright (c) 2013 "Pollos Hermanos"
// "Larcenas Legacy", a game for Amastrad CPC 464
//
// This file is part of "Larcenas Legacy".
//
// "Larcenas Legacy" is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.

#include "cpcrslib.h"
#include "cpcwyzlib.h"

#include "graphics.h"
#include "sound.h"

#define GAME_STATE_MENU    2
#define GAME_STATE_HELP    3
#define GAME_STATE_GAME    4
#define GAME_STATE_EXIT    5
#define GAME_STATE_WIN     6
#define GAME_STATE_LOSE    7
#define INITIAL_GAME_STATE GAME_STATE_MENU

#define KEY_1     0
#define KEY_2     1
#define KEY_3     2
#define KEY_4     3
#define KEY_LEFT  4
#define KEY_RIGHT 5
#define KEY_ESC   6
#define KEY_FIRE  7

#define SOUND_PICK_1  0
#define SOUND_FAIL    1
#define SOUND_PICK_2  2
#define SOUND_TICK    3

#define ANIM_PLAYER_STILL     0
#define ANIM_PLAYER_LEFT      1
#define ANIM_PLAYER_RIGHT     2
#define ANIM_PLAYER_ACTION    3
#define ANIM_JUDGE_STILL      4
#define ANIM_JUDGE_LEFT       5
#define ANIM_JUDGE_RIGHT      6
#define ANIM_PROMOTER_STILL   7
#define ANIM_PROMOTER_LEFT    8
#define ANIM_PROMOTER_RIGHT   9
#define ANIM_BANKER_STILL    10
#define ANIM_BANKER_LEFT     11
#define ANIM_BANKER_RIGHT    12
#define ANIM_BRICK_ROTATE    13
#define ANIM_ENVELOPE_ROTATE 14
#define MAX_ANIMATIONS       15

#define TYPE_PLAYER    1
#define TYPE_JUDGE     2
#define TYPE_PROMOTER  3
#define TYPE_BANKER    4
#define TYPE_BRICK     5
#define TYPE_ENVELOPE  6

#define TIME_TO_ACTION_SHORT  35
#define TIME_TO_ACTION_MEDIUM 65
#define TIME_TO_ACTION_LONG   100

#define ACTION_STILL  0
#define ACTION_LEFT   1
#define ACTION_RIGHT  2
#define ACTION_UP     3
#define ACTION_DOWN   4

#define MAX_FRAMES 34
#define MAX_FRAMES_PER_ANIMATION 5

#define MIN_X 0
#define MAX_X 50
#define MAX_Y 200

#define ANIMATION_LOOP 0

#define WALK_DELTA     3
#define BRICK_DELTA    7
#define ENVELOPE_DELTA 7

#define FRAME_DELAY 250

#define NUM_COLORS 16

#define MONEY_TO_WIN 10
#define MAX_HEALTH    4

#define BORDER        1

char *frames[MAX_FRAMES];
char animations[MAX_ANIMATIONS*MAX_FRAMES_PER_ANIMATION];
unsigned char health;
unsigned char money;
char state;
char hasEnvelope;

struct Sprite {
  unsigned char x,y;
  int address;
  char w,h;
  char currentAnimation;
  char currentFrame;
  char type;
  char action;
  char timeToAction;
  char *backBuffer;
};
struct Sprite player, judge, promoter, banker, brick, envelope;

void DrawHealthBar() {
  int red = health*100/MAX_HEALTH;
  int empty = 100-red;
  int y = 145;
  int i;

  for(i=0; i<red; ++i)
    cpc_PutSp(REDBAR, 1,3, cpc_GetScrAddress(68, y--));

  for(i=0; i<empty; ++i)
    cpc_PutSp(EMPTYBAR, 1,3, cpc_GetScrAddress(68, y--));
}

void DrawMoneyBar() {
  int yellow = money*100/MONEY_TO_WIN;
  int empty = 100-yellow;
  int y = 145;
  int i;

  for(i=0; i<yellow; ++i)
    cpc_PutSp(YELLOWBAR, 1,3, cpc_GetScrAddress(75, y--));

  for(i=0; i<empty; ++i)
    cpc_PutSp(EMPTYBAR, 1,3, cpc_GetScrAddress(75, y--));  
}

void DrawHud() {
  cpc_PrintGphStrXY("LARCENAS",  63, 0*16);
  cpc_PrintGphStrXY(";LEGACY",   63, 1*16);

  cpc_PutSp(HEART, 7,4, cpc_GetScrAddress(67,150));
  DrawHealthBar();

  cpc_PutSp(COIN, 7,4, cpc_GetScrAddress(74,150));
  DrawMoneyBar();
}

void DecHealth() {
  if (health>1) {
    --health;
    DrawHealthBar();
  }
  else {
    state = GAME_STATE_LOSE;
  }
}

void DecMoney() {
  if (money>1) {
    --money;
  }
  DrawMoneyBar();
}

void IncMoney() {
  if (money<MONEY_TO_WIN) {
    ++money;
    DrawMoneyBar(); 
  }
  else
    state = GAME_STATE_WIN;
}

void SetPalette(char *pPalette) {
  char nColor;
  for(nColor = 0; nColor < NUM_COLORS; ++nColor) {
    cpc_SetInk(nColor, pPalette[nColor]);
  }
}

char getTimeToAction(char period) {
  return period + (cpc_Random()%(period>>2)) - (period>>3);
  //return (cpc_Random() % period) + (period>>3);
}

unsigned char collide(struct Sprite *a, struct Sprite *b) {
  return !(
    a->x > b->x+b->w
    || b->x > a->x+a->w
    || a->y > b->y+b->h
    || b->y > a->y+a->h);
}

void ResetData() {
  health = MAX_HEALTH;
  money = 0;
  hasEnvelope = 0;

  // Player settings
  player.type = TYPE_PLAYER;
  player.x = cpc_Random() % MAX_X;
  player.y = 168; 
  player.address = cpc_GetScrAddress(player.x,player.y);
  player.w = 10; // width in BYTES, not pixels
  player.h = 31; 
  player.currentAnimation = ANIM_PLAYER_STILL;
  player.currentFrame = 0;

  // Judge settings
  judge.type = TYPE_JUDGE;
  judge.x = cpc_Random() % MAX_X;
  judge.y = 95; 
  judge.address = cpc_GetScrAddress(judge.x,judge.y);
  judge.w = 10; // width in BYTES, not pixels
  judge.h = 31; 
  judge.currentAnimation = ANIM_JUDGE_STILL;
  judge.currentFrame = 0;
  judge.action = ACTION_LEFT;
  judge.timeToAction = getTimeToAction(TIME_TO_ACTION_SHORT);

  // Promoter settings
  promoter.type = TYPE_PROMOTER;
  promoter.x = cpc_Random() % MAX_X;
  promoter.y = 50; 
  promoter.address = cpc_GetScrAddress(promoter.x,promoter.y);
  promoter.w = 10; // width in BYTES, not pixels
  promoter.h = 31; 
  promoter.currentAnimation = ANIM_PROMOTER_STILL;
  promoter.currentFrame = 0;
  promoter.action = ACTION_RIGHT;
  promoter.timeToAction = getTimeToAction(TIME_TO_ACTION_MEDIUM);

  // Promoter settings
  banker.type = TYPE_BANKER;
  banker.x = cpc_Random() % MAX_X;
  banker.y = 5; 
  banker.address = cpc_GetScrAddress(banker.x,banker.y);
  banker.w = 10; // width in BYTES, not pixels
  banker.h = 31; 
  banker.currentAnimation = ANIM_BANKER_STILL;
  banker.currentFrame = 0;
  banker.action = ACTION_RIGHT;
  banker.timeToAction = getTimeToAction(TIME_TO_ACTION_MEDIUM);

  // Brick settings
  brick.type = TYPE_BRICK;
  brick.x = 40;
  brick.y = 150; 
  brick.address = cpc_GetScrAddress(brick.x,brick.y);
  brick.w = 5; // width in BYTES, not pixels
  brick.h = 9; 
  brick.currentAnimation = ANIM_BRICK_ROTATE;
  brick.currentFrame = 0;
  brick.action = ACTION_STILL;
  brick.backBuffer = S05_BACK_BUFFER;

  // Brick settings
  envelope.type = TYPE_ENVELOPE;
  envelope.x = 40;
  envelope.y = 150; 
  envelope.address = cpc_GetScrAddress(envelope.x,envelope.y);
  envelope.w = 5; // width in BYTES, not pixels
  envelope.h = 9; 
  envelope.currentAnimation = ANIM_ENVELOPE_ROTATE;
  envelope.currentFrame = 0;
  envelope.action = ACTION_STILL;
  envelope.backBuffer = S06_BACK_BUFFER;
}

void SetUp() {
  cpc_WyzInitPlayer(SOUND_TABLE_0, RULE_TABLE_0, EFFECT_TABLE, SONG_TABLE_0);
  cpc_WyzLoadSong(0);
  cpc_WyzSetPlayerOn();

  state = INITIAL_GAME_STATE;

  frames[0]  = 0; // index not used

  // Player frames
  frames[1]  = S01_L_F1_data;
  frames[2]  = S01_L_F2_data;
  frames[3]  = S01_R_F1_data;
  frames[4]  = S01_R_F2_data;
  frames[5]  = S01_F_F1_data;

  // Judge frames
  frames[6]  = S02_L_F1_data;
  frames[7]  = S02_L_F2_data;
  frames[8]  = S02_R_F1_data;
  frames[9]  = S02_R_F2_data;  
  frames[10] = S02_F_F1_data;

  // Banker frames
  frames[11] = S03_L_F1_data;
  frames[12] = S03_L_F2_data;
  frames[13] = S03_R_F1_data;
  frames[14] = S03_R_F2_data;  
  frames[15] = S03_F_F1_data;  

  // Promoter frames
  frames[16] = S04_L_F1_data;
  frames[17] = S04_L_F2_data;
  frames[18] = S04_R_F1_data;
  frames[19] = S04_R_F2_data;  
  frames[20] = S04_F_F1_data;  

  // brick frames
  frames[21] = S05_R_F1_data;
  frames[22] = S05_R_F2_data;
  frames[23] = S05_R_F3_data;
  frames[24] = S05_R_F4_data;  

  // envelope frames
  frames[25] = S06_R_F1_data;
  frames[26] = S06_R_F2_data;
  frames[27] = S06_R_F3_data;
  frames[28] = S06_R_F4_data;  

  // hud frames
  frames[29] = HEART;
  frames[30] = COIN;
  frames[31] = REDBAR;
  frames[32] = YELLOWBAR;
  frames[33] = EMPTYBAR;

  // Player animations
  animations[(ANIM_PLAYER_STILL*MAX_FRAMES_PER_ANIMATION)+0] = 5;
  animations[(ANIM_PLAYER_STILL*MAX_FRAMES_PER_ANIMATION)+1] = ANIMATION_LOOP;
  animations[(ANIM_PLAYER_LEFT*MAX_FRAMES_PER_ANIMATION)+0] = 1;
  animations[(ANIM_PLAYER_LEFT*MAX_FRAMES_PER_ANIMATION)+1] = 2;
  animations[(ANIM_PLAYER_LEFT*MAX_FRAMES_PER_ANIMATION)+2] = ANIMATION_LOOP;
  animations[(ANIM_PLAYER_RIGHT*MAX_FRAMES_PER_ANIMATION)+0] = 3;
  animations[(ANIM_PLAYER_RIGHT*MAX_FRAMES_PER_ANIMATION)+1] = 4;
  animations[(ANIM_PLAYER_RIGHT*MAX_FRAMES_PER_ANIMATION)+2] = ANIMATION_LOOP;

  // Judge animations
  animations[(ANIM_JUDGE_STILL*MAX_FRAMES_PER_ANIMATION)+0] = 10;
  animations[(ANIM_JUDGE_STILL*MAX_FRAMES_PER_ANIMATION)+1] = ANIMATION_LOOP;
  animations[(ANIM_JUDGE_LEFT*MAX_FRAMES_PER_ANIMATION)+0] = 6;
  animations[(ANIM_JUDGE_LEFT*MAX_FRAMES_PER_ANIMATION)+1] = 7;
  animations[(ANIM_JUDGE_LEFT*MAX_FRAMES_PER_ANIMATION)+2] = ANIMATION_LOOP;
  animations[(ANIM_JUDGE_RIGHT*MAX_FRAMES_PER_ANIMATION)+0] = 8;
  animations[(ANIM_JUDGE_RIGHT*MAX_FRAMES_PER_ANIMATION)+1] = 9;
  animations[(ANIM_JUDGE_RIGHT*MAX_FRAMES_PER_ANIMATION)+2] = ANIMATION_LOOP;

  // Banker animations
  animations[(ANIM_BANKER_STILL*MAX_FRAMES_PER_ANIMATION)+0] = 15;
  animations[(ANIM_BANKER_STILL*MAX_FRAMES_PER_ANIMATION)+1] = ANIMATION_LOOP;
  animations[(ANIM_BANKER_LEFT*MAX_FRAMES_PER_ANIMATION)+0] = 11;
  animations[(ANIM_BANKER_LEFT*MAX_FRAMES_PER_ANIMATION)+1] = 12;
  animations[(ANIM_BANKER_LEFT*MAX_FRAMES_PER_ANIMATION)+2] = ANIMATION_LOOP;
  animations[(ANIM_BANKER_RIGHT*MAX_FRAMES_PER_ANIMATION)+0] = 13;
  animations[(ANIM_BANKER_RIGHT*MAX_FRAMES_PER_ANIMATION)+1] = 14;
  animations[(ANIM_BANKER_RIGHT*MAX_FRAMES_PER_ANIMATION)+2] = ANIMATION_LOOP;

  // Promoter animations
  animations[(ANIM_PROMOTER_STILL*MAX_FRAMES_PER_ANIMATION)+0] = 20;
  animations[(ANIM_PROMOTER_STILL*MAX_FRAMES_PER_ANIMATION)+1] = ANIMATION_LOOP;
  animations[(ANIM_PROMOTER_LEFT*MAX_FRAMES_PER_ANIMATION)+0] = 16;
  animations[(ANIM_PROMOTER_LEFT*MAX_FRAMES_PER_ANIMATION)+1] = 17;
  animations[(ANIM_PROMOTER_LEFT*MAX_FRAMES_PER_ANIMATION)+2] = ANIMATION_LOOP;
  animations[(ANIM_PROMOTER_RIGHT*MAX_FRAMES_PER_ANIMATION)+0] = 18;
  animations[(ANIM_PROMOTER_RIGHT*MAX_FRAMES_PER_ANIMATION)+1] = 19;
  animations[(ANIM_PROMOTER_RIGHT*MAX_FRAMES_PER_ANIMATION)+2] = ANIMATION_LOOP;  

  // brick animations
  animations[(ANIM_BRICK_ROTATE*MAX_FRAMES_PER_ANIMATION)+0] = 21;
  animations[(ANIM_BRICK_ROTATE*MAX_FRAMES_PER_ANIMATION)+1] = 22;
  animations[(ANIM_BRICK_ROTATE*MAX_FRAMES_PER_ANIMATION)+2] = 23;
  animations[(ANIM_BRICK_ROTATE*MAX_FRAMES_PER_ANIMATION)+3] = 24;
  animations[(ANIM_BRICK_ROTATE*MAX_FRAMES_PER_ANIMATION)+4] = ANIMATION_LOOP;

  // envelope animations
  animations[(ANIM_ENVELOPE_ROTATE*MAX_FRAMES_PER_ANIMATION)+0] = 25;
  animations[(ANIM_ENVELOPE_ROTATE*MAX_FRAMES_PER_ANIMATION)+1] = 26;
  animations[(ANIM_ENVELOPE_ROTATE*MAX_FRAMES_PER_ANIMATION)+2] = 27;
  animations[(ANIM_ENVELOPE_ROTATE*MAX_FRAMES_PER_ANIMATION)+3] = 28;
  animations[(ANIM_ENVELOPE_ROTATE*MAX_FRAMES_PER_ANIMATION)+4] = ANIMATION_LOOP;


  cpc_AssignKey(KEY_1,0x4801);
  cpc_AssignKey(KEY_2,0x4802);
  cpc_AssignKey(KEY_3,0x4702);
  cpc_AssignKey(KEY_4,0x4701);
  cpc_AssignKey(KEY_LEFT,0x4101);
  cpc_AssignKey(KEY_RIGHT,0x4002);
  cpc_AssignKey(KEY_ESC,0x4804);
  cpc_AssignKey(KEY_FIRE,0x4580);

  ResetData();

  cpc_SetModo(0);
}

void SetAnimation(struct Sprite *s, char animation) {
  if (s->currentAnimation!=animation) {
    s->currentAnimation=animation;
    s->currentFrame=0;
  }
}

void Animate(struct Sprite *s) {
  ++s->currentFrame;
  if (animations[(s->currentAnimation*MAX_FRAMES_PER_ANIMATION) + s->currentFrame] == 0)
    s->currentFrame = 0;
}

void RunPlayerLogic()  {
  if (cpc_TestKey(KEY_FIRE)==1) {
    if (envelope.action==ACTION_STILL && brick.action==ACTION_STILL) {
      if (hasEnvelope) {
        hasEnvelope = 0;
       envelope.x = player.x+3;
        envelope.y = player.y;
        envelope.action = ACTION_UP;
        cpc_WyzStartEffect(2, SOUND_TICK);
      } else {
        brick.x = player.x+3;
        brick.y = player.y;
        brick.action = ACTION_UP;
        cpc_WyzStartEffect(2, SOUND_TICK);
      }
    }
  } else if (cpc_TestKey(KEY_LEFT)==1 && player.x>MIN_X) {
    if (player.x-WALK_DELTA>MIN_X)
      player.x -= WALK_DELTA;
    else
      player.x = MIN_X;
    SetAnimation(&player, ANIM_PLAYER_LEFT);
  } else if (cpc_TestKey(KEY_RIGHT)==1 && player.x<MAX_X) {
    if (player.x+WALK_DELTA<MAX_X)
      player.x += WALK_DELTA;
    else
      player.x = MAX_X;
    SetAnimation(&player, ANIM_PLAYER_RIGHT);
  } else {
    SetAnimation(&player, ANIM_PLAYER_STILL);
  }
}

void RunWondererLogic(struct Sprite *s, char animStill, char animLeft, char animRight) {
  switch(s->action) {
    case ACTION_STILL:
      SetAnimation(s, animStill);
      if ((--s->timeToAction)==0) {
        s->action = cpc_Random()%2==0? ACTION_RIGHT : ACTION_LEFT;
        s->timeToAction = getTimeToAction(TIME_TO_ACTION_SHORT);
      }
      break;
    case ACTION_LEFT:
      SetAnimation(s, animLeft);
      if (s->x-WALK_DELTA>MIN_X) {
        s->x -= WALK_DELTA;
      } else {
        s->x = MIN_X;
        s->action = ACTION_RIGHT;
      }
      break;
    case ACTION_RIGHT:
      SetAnimation(s, animRight);
      if (s->x+WALK_DELTA<MAX_X) {
        s->x += WALK_DELTA;
      } else {
        s->x = MAX_X;
        s->action = ACTION_LEFT;
      }
      break;
  }

  if (s->timeToAction==0 || --s->timeToAction==0) {
    s->action = ACTION_STILL;
    s->timeToAction = getTimeToAction(TIME_TO_ACTION_SHORT);
  }
}

void RunJudgeLogic() {
  RunWondererLogic(&judge, ANIM_JUDGE_STILL, ANIM_JUDGE_LEFT, ANIM_JUDGE_RIGHT);
}

void RunBankerLogic() {
  RunWondererLogic(&banker, ANIM_BANKER_STILL, ANIM_BANKER_LEFT, ANIM_BANKER_RIGHT);
}

void RunPromoterLogic() {
  RunWondererLogic(&promoter, ANIM_PROMOTER_STILL, ANIM_PROMOTER_LEFT, ANIM_PROMOTER_RIGHT);
}

void RunBrickLogic() {
  switch (brick.action) {
    case ACTION_UP:
      if (brick.y > BRICK_DELTA) {
        brick.y -= BRICK_DELTA;

        if (collide(&brick,&judge)) {
          brick.action = ACTION_STILL;
          cpc_WyzStartEffect(2, SOUND_FAIL);
          DecHealth();
        } else if (collide(&brick,&promoter)) {
          brick.action = ACTION_STILL;

          // activate envelope
          if (envelope.action==ACTION_STILL) {
            envelope.x = promoter.x+3;
            envelope.y = promoter.y+promoter.h;
            envelope.action = ACTION_DOWN;
          }
          cpc_WyzStartEffect(2, SOUND_TICK);

        } else if (collide(&brick,&banker)) {
          brick.action = ACTION_STILL;
          DecMoney();
          cpc_WyzStartEffect(2, SOUND_FAIL);
        }
      } else {
        brick.action = ACTION_STILL;
        brick.y = 0;
        //brick.address = 0; ////
      }
      break;
  }
}

void RunEnvelopeLogic() {
  switch (envelope.action) {
    case ACTION_DOWN:
      if (envelope.y < (MAX_Y-envelope.h-ENVELOPE_DELTA)) {
        envelope.y += ENVELOPE_DELTA;

        if (collide(&envelope,&player)) {
          envelope.action = ACTION_STILL;
          hasEnvelope=1;
          cpc_WyzStartEffect(2, SOUND_PICK_1);
        }
      } else {
        envelope.action = ACTION_STILL;
        envelope.y = 0;
        cpc_WyzStartEffect(2, SOUND_FAIL);
        //envelope.address = 0; ////
      }
      break;

    case ACTION_UP:
      if (envelope.y > ENVELOPE_DELTA) {
        envelope.y -= ENVELOPE_DELTA;

        if (collide(&envelope,&banker)) {
          envelope.action = ACTION_STILL;
          cpc_WyzStartEffect(2, SOUND_PICK_2);
          IncMoney();
        } else if (collide(&envelope,&judge)) {
          envelope.action = ACTION_STILL;
          cpc_WyzStartEffect(2, SOUND_FAIL);
        }
      } else {
        envelope.action = ACTION_STILL;
        envelope.y = 0;
        cpc_WyzStartEffect(2, SOUND_FAIL);
        // envelope.address = 0; ////
      }
      break;
  }
}

void UpdateSprite(struct Sprite *s) {
  int oldAddress = s->address;
  char oldFrame = animations[(s->currentAnimation*MAX_FRAMES_PER_ANIMATION) + s->currentFrame];

  switch(s->type) {
    case TYPE_PLAYER:
      RunPlayerLogic();
      break;
    case TYPE_JUDGE:
      RunJudgeLogic();
      break;
    case TYPE_PROMOTER:
      RunPromoterLogic();
      break;
    case TYPE_BANKER:
      RunBankerLogic();
      break;
  }

  Animate(s);

  s->address = cpc_GetScrAddress(s->x, s->y);
  if (s->address!=oldAddress || animations[(s->currentAnimation*MAX_FRAMES_PER_ANIMATION) + s->currentFrame]!=oldFrame) {
    if (oldAddress!=0)
      cpc_PutSpXOR(frames[oldFrame], s->h, s->w, oldAddress);
    cpc_PutSp(frames[animations[(s->currentAnimation*MAX_FRAMES_PER_ANIMATION) + s->currentFrame]], s->h, s->w, s->address);
  }
}

void UpdateProjectile(struct Sprite *s) {

  switch(s->type) {
    case TYPE_BRICK:
      RunBrickLogic();
      break;
    case TYPE_ENVELOPE:
      RunEnvelopeLogic();
      break;
  }

  if (s->action!=ACTION_STILL) {
    Animate(s);

    s->address = cpc_GetScrAddress(s->x, s->y);
    if (s->backBuffer!=0)
      cpc_GetSp(s->backBuffer, s->h, s->w, s->address);
    cpc_PutSp(frames[animations[(s->currentAnimation*MAX_FRAMES_PER_ANIMATION) + s->currentFrame]], s->h, s->w, s->address);
  }
}

////////////////////////////////////////////////////////////////////////////////
// GAME STATES
////////////////////////////////////////////////////////////////////////////////

char Game() {
  static int elapsedFrames = 0;
  int i;

  ResetData();

  cpc_ClrScr();
  cpc_SetBorder(BORDER);
  SetPalette(gamePalette);

  DrawHud();

  // Draw floor
  for (i=0;i<12;++i) {
    cpc_PutSp(FLOOR, 3,5, cpc_GetScrAddress(2+i*5,37));
    cpc_PutSp(FLOOR, 3,5, cpc_GetScrAddress(2+i*5,81));
    cpc_PutSp(FLOOR, 3,5, cpc_GetScrAddress(2+i*5,126));
  }

  // Forze paint on enter
  player.address = 0;
  judge.address = 0;
  promoter.address = 0;
  banker.address = 0;

  while (state==GAME_STATE_GAME) {
    if (cpc_TestKey(KEY_ESC)==1) {
      state = GAME_STATE_MENU;
    }
    else {
      if (++elapsedFrames >= FRAME_DELAY) {
        elapsedFrames = 0;

        // Remove projectiles from screen
        if (brick.address!=0) {
          cpc_PutSp(brick.backBuffer, brick.h, brick.w, brick.address);
          brick.address=0;
        }
        if (envelope.address!=0) {
          cpc_PutSp(envelope.backBuffer, envelope.h, envelope.w, envelope.address);
          envelope.address = 0;
        }

        UpdateSprite(&player);
        UpdateSprite(&judge);
        UpdateSprite(&promoter);
        UpdateSprite(&banker);

        UpdateProjectile(&brick);
        UpdateProjectile(&envelope);
      }
    }
  }  

  while (cpc_AnyKeyPressed());

  return state; 
}

char Menu() {
  char choice=-1;

  cpc_ClrScr();
  cpc_SetBorder(BORDER);
  SetPalette(menuPalette);
  SetPalette(menuPalette);
  SetPalette(menuPalette); // A veces falla... 

  cpc_PrintGphStrXY("LARCENAS",    30, 1*16);
  cpc_PrintGphStrXY(";LEGACY",     30, 2*16);

  cpc_PrintGphStrXY("1;JUGAR",    30, 5*16);
  cpc_PrintGphStrXY("2;AYUDA",    30, 6*16);
  cpc_PrintGphStrXY("ESC;SALIR",  29, 7*16);

  cpc_PrintGphStrXY("C;2013;LOS;POLLOS;AMIGOS", 12, 10*16);
  cpc_PrintGphStrXY("DAVID;DIAZ;Y;ALFREDO;SORO", 12, 11*16);

  while (choice==-1) {
    cpc_ScanKeyboard();

    if (cpc_TestKeyF(KEY_1)==1)   
      choice=GAME_STATE_GAME;
    if (cpc_TestKeyF(KEY_2)==1)   
      choice=GAME_STATE_HELP;
    if (cpc_TestKeyF(KEY_4)==1)   
      choice=GAME_STATE_GAME;
    if (cpc_TestKeyF(KEY_ESC)==1)   
      choice=GAME_STATE_EXIT;
  }

  while (cpc_AnyKeyPressed());

  return choice; 
}

char Win() {
  cpc_ClrScr();
  cpc_SetBorder(BORDER);
  SetPalette(menuPalette);

  cpc_PrintGphStrXY(";;;;;;;HAS;GANADO", 12, 4*16);
  cpc_PrintGphStrXY("ERES;UN;AUTENTICO;LADRON",  12, 6*16);
  cpc_PrintGphStrXY("TU;MADRE;ESTARA;ORGULLOSA", 12, 7*16);

  while (!cpc_AnyKeyPressed());
  while (cpc_AnyKeyPressed());

  return GAME_STATE_MENU; 
}

char Lose() {
  cpc_ClrScr();
  cpc_SetBorder(BORDER);
  SetPalette(menuPalette);

  cpc_PrintGphStrXY(";;;;;;;HAS;PERDIDO", 12, 4*16);
  cpc_PrintGphStrXY("EL;CRIMEN;SE;ACABA;PAGANDO",  12, 6*16);
  cpc_PrintGphStrXY("FELIZ;ESTANCIA;EN;CHIRONA", 12, 7*16);

  while (!cpc_AnyKeyPressed());
  while (cpc_AnyKeyPressed());

  return GAME_STATE_MENU; 
}

char Help() {
  cpc_ClrScr();

  cpc_SetBorder(BORDER);
  SetPalette(helpPalette);

  cpc_PutSp(S03_F_F1_data,31,10,cpc_GetScrAddress(0,0));
  cpc_PutSp(S04_F_F1_data,31,10,cpc_GetScrAddress(0,40));
  cpc_PutSp(S02_F_F1_data,31,10,cpc_GetScrAddress(0,80));
  cpc_PutSp(S01_F_F1_data,31,10,cpc_GetScrAddress(0,120));

  cpc_PrintGphStrXY("BANQUERO:PONDRA;TU;DINERO",20,0*11);
  cpc_PrintGphStrXY(";;;;;;;;;FUERA;DEL;ALCANCE",20,1*11);
  cpc_PrintGphStrXY(";;;;;;;;;DEL;FISCO",20,2*11);

  cpc_PrintGphStrXY("PROMOTOR:TE;DARA;SOBRES;A",20,4*11);
  cpc_PrintGphStrXY(";;;;;;;;;CAMBIO;DE",20,5*11);
  cpc_PrintGphStrXY(";;;;;;;;;ADJUDICACIONES",20,6*11);

  cpc_PrintGphStrXY("JUEZ:SU;MAZA;HARA;CAER;SOBRE",20,8*11);
  cpc_PrintGphStrXY(";;;;;;;;;TI;EL;PESO;DE;LA;LEY",20,9*11);

  cpc_PrintGphStrXY("LARCENAS:SIMPATICO;LADRONZUELO",20,11*11);

  cpc_PrintGphStrXY("TECLAS:;CURSORES;Y;BARRA;DE;ESPACIO",3,16*11);


  while (!cpc_AnyKeyPressed());
  while (cpc_AnyKeyPressed());

  return GAME_STATE_MENU; 
}

////////////////////////////////////////////////////////////////////////////////
// MAIN FUNCTION (FINITE STATE MACHINE)
////////////////////////////////////////////////////////////////////////////////

int main() {
  SetUp();

  while (state != GAME_STATE_EXIT) {
    switch(state) {
      case GAME_STATE_MENU:
        state = Menu();
        break;

      case GAME_STATE_HELP:
        state = Help();
        break;

      case GAME_STATE_GAME:
        state = Game();
        break;

      case GAME_STATE_WIN:
        state = Win();
        break;

      case GAME_STATE_LOSE:
        state = Lose();
        break;

      default:
        state = GAME_STATE_EXIT;
        break;
    }
  }

  cpc_WyzSetPlayerOff();

  return 0;  
}