; ---------------------------------------------------------------------------
; Initializing code for the Pac-Man emulator
; ---------------------------------------------------------------------------

    include 'constantes.i'
    include 'hardware.i'
    include 'macros.i'

; ---------------------------------------------------------------------------
; Macro for waiting scanlines and simplify the syntax
; ---------------------------------------------------------------------------
 MACRO WAIT_SCANLINES
    LD   IX,\1 * 4 - 1                                          ; (4) 4
    CALL wait_scanlines_ix                                      ; (5) 3
 ENDM                                                           ; (9) 7 bytes

    ORG  $C000

; ---------------------------------------------------------------------------
initialize_system
    ; Copy Pac-Man ROM to the RAM page 0
    LD   BC,GATE_ARRAY + $C4
    OUT  (C),C
    LD   DE,$0000
    LD   HL,$4000
    LD   BC,$4000
    LDIR

    ; And patch it
    CALL patch_pacman_rom

    ; Initialize AY and create sound table in the uppper half of the original Pac-Man ROM
    CALL initialize_sound

    ; Copy the CPC code to the RAM page 2
    LD   BC,GATE_ARRAY + $C6
    OUT  (C),C
    LD   HL,$4000
    LD   DE,$8000
    LD   BC,$4000
    LDIR

    ; Copy the CPC menu to the RAM page 6 (2*)
    LD   HL,$E000
    LD   DE,$4000
    LD   BC,$2000
    LDIR

    ; Fill both tilemaps with spaces ($40)
    LD   HL,CPC_TILEMAP1        ; $xx40
    LD   (HL),$40               ; $40 = Space
    LD   DE,CPC_TILEMAP1 + 1
    LD   BC,$800 - $40 - 1
    LDIR

    ; Enable bank config for running the emulator
    SET_PAGE RUN_CODE

    ; Clear Pac-Man I/O area ($5000 - $5100)
    LD   HL,DIP_5000
    XOR  A
.loop_clear_io
    LD   (HL),A
    INC  L
    JR   NZ,.loop_clear_io

    ; Set the DIPs values
    LD   A,DIP_5000_VALUE
    LD   (DIP_5000),A
    LD   A,DIP_5040_VALUE
    LD   (DIP_5040),A
    LD   A,DIP_5080_VALUE
    LD   (DIP_5080),A

    ; Make a mini fade out
    WAIT_VBL
    CALL set_fade_palette

    LD  D,5
.loop_mini_fade_out
    WAIT_SCANLINES 52
    ; Wait Vertical Blank
    WAIT_VBL
    DEC  D
    JR   NZ,.loop_mini_fade_out

    ; Set black palette
    CALL set_black_palette

    ; Copy the launcher to the top of the stack
    LD   HL,pacman_launcher
    LD   DE,CPC_STACK - 80
    LD   BC,end_pacman_launcher - pacman_launcher
    LDIR
    
    ; Execute launcher
    JP   CPC_STACK + $17 - 80

; ---------------------------------------------------------------------------
; Wait x scanlines
; Use:
;    WAIT_SCANLINES x                       ; (9)
; ---------------------------------------------------------------------------
wait_scanlines_ix
    DEFS 5,0                                ; (5) 

.loop_wait_scanlines_ix                      
    DEFS 6                                  ; (6)
    DEC  IX                                 ; (3)
    LD   A,IXH                              ; (2)
    OR   IXL                                ; (2)
    JR   NZ,.loop_wait_scanlines_ix         ; (2/3)
                                            ; Total loop --> 16 * (IX - 1) + 15
    RET                                     ; (3)
                                            ; Total Routine --> 64 * SCANLINES

; ---------------------------------------------------------------------------
; Patch the Pac-Man ROM attaching our interrupt handler, fixing the code that
; overwrite the DIPs memory address and patch the call get_random routine.
; ---------------------------------------------------------------------------
patch_pacman_rom
    ; 1.- Patch the ROM interrupt handler, the interrupt Z80 mode used 
    ;     from "IM 2" to "IM 1", insert in $0038 a null interrupt handler
    ;     and disable interrupts during game.

    ; The original speed hack by 40 crisis made:
    ; Patch ROM main loop by disabling interrupts, waiting for vblank,
    ; calling original interrupt, disabling interrupts and then waiting
    ; for task list to complete.
    ; Pac-Man code:
    ; 238d  2a824c    ld      hl,(#4c82)    ; (17,08%)
    ; 2390  7e        ld      a,(hl)        ; (6,82%)
    ; 2391  a7        and     a             ; (3,41%)
    ; 2392  fa8d23    jp      m,#238d       ; (10,23%)
    ;
    ; New code:
    ; launch_cpc_emu_code
	;   CALL cpc_emu_code
    ;   LD   HL,($4C82)
    ;   LD   A,(HL)         		
    ;   AND  A               	        	
    ;   JP   M,launch_cpc_emu_code  ; wait for end of list	
    ;   JP   $2395			        ; back to pacman rom
    
    ; Patch the Pacman main loop to jump to our new main loop
    LD	 A,$C3                      ; JP   $xxxx
    LD   ($238D),A
    LD	 HL,CPC_EMU_CODE            ; cpc_emulation_code
    LD   ($238E),HL
    
    ; Change from "IM 2" to "IM 1"
    LD   A,$56
    LD   ($233C),A

    ; Change the "OUT ($00),A" to "LD I,A"
    LD   HL,$47ED
    LD   ($233F),HL
;    LD   ($3183),HL

    ; The arcade use two interrupt handlers ($30B0 and $008D -changed to $008E
    ; for skip PUSH AF-), the first is used only the first time, the other is 
    ; used all the time. Because that we make a jump during the first to set 
    ; the second and simplify our interrupt handler code. 
    LD   A,$C3                      ; JP   xxxx
    LD   ($3181),A
    LD   HL,CPC_STACK - 80
    LD   ($3182),HL

   ; Restore "SUB $4" in case the ROM used was a Pac-Man bootleg
;   LD   HL,$04D6
;   LD   ($3181),HL

    ; Put a null interrupt handler, only used by first arcade interrupt
    LD   HL,$C9FB                   ; EI + RET
    LD   ($0038),HL

    ; Disable EI in $01D9 and $238C for disabling all the interrupts during game
    LD   A,$C9                      ; RET and saving the POP AF too
    LD   ($01D9),A
    XOR  A                          ; NOP
    LD   ($238C),A

    ; Save the pushing/popping of registers during the arcade main interrupt routine
    LD   HL,$0618                   ; JR $06
    LD   ($0095),HL
    DEC  H                          ; JR $05
    LD   ($01BF),HL
 
    ; 2.- Patch the routines that overwrite the DIPs memory address
    
    ; Change "LD ($5000),A" to "LD ($5001),A" ($5001 not used by Pac-Man)
    LD   A,$01
    LD   ($0093),A
    LD   ($01D7),A
    LD   ($2347),A
    LD   ($238A),A
    LD   ($3194),A
    LD   ($3248),A

    ; Change "LD HL,$5000" to "LD HL,$5001"
    LD   ($230C),A
    LD   ($2353),A

    ; Related to the previous patches, we need to decrease (B--) the size
    ; of the blocks to clean
    LD   A,7
    LD   ($230F),A
    LD   ($2357),A

    ; Disabled the rom checksum test using "XOR A" instead of "AND A"
;    LD   A,$AF
;    LD   ($3019),A

    ; Change "LD HL,$5040" to "LD HL,$5041"
    LD   A,$41
    LD   ($2363),A

    ; And decrease (B--) the size of the block to clean
    LD   A,$3F
    LD   ($2366),A

    ; Change "DJNZ" to "LD B,n" for not cleaning all the RAM
    LD   A,6
    LD   ($30CF),A

    ; 3.- Patch the CALL to the get_random routine in $2A23 to our get_random
    LD   HL,CPC_GET_RANDOM
    LD   ($2927),HL

    ; 4.- Additional fixes taken from the ZX code
    
    ; Skip Pac-Man memory test, instead of jump to $3000, go to $30B0
    LD   A,$B0
    LD   ($3FFA),A

    ; Reset the most significative bit of the display address in horizontal
    ; text screen writes. Change "LD E,(HL)|INC HL|LD D,(HL)" to "CALL nn"
    LD   A,$CD
    LD   ($2C62),A
    LD   HL,CPC_TEXT_FIX
    LD   ($2C63),HL

    ; Disable the 1UP/2UP flashing to save cycles, change "CALL NZ,nn" to "CALL C,nn"
;*; LD   A,$DC
;*; LD   ($0353),A        
;*; LD   ($035E),A

    ; 5.- Extra patches for the CPC version

    ; Patch the routine that sends the sprite data to the hardware at our own buffer,
    ; saving cpu time inserting sprite hardware data in the stack.
    LD   HL,SPRITE1_XPOS
    LD   ($0185),HL
    LD   HL,SPRITE1_ATRB
    LD   ($017A),HL

    ; Patch menu text (NICKNAME << 1)
;    LD   HL,$380B
;    LD   DE,$380A
;    LD   BC,8
;    LDIR
;    LD   A,$40          ; Add Space
;    LD   (DE),A

    ; Patch Namco logo to be centered
    LD   HL,$3931
    LD   ($37F4),HL
    LD   HL,$3038
    LD   ($37F6),HL
    LD   A,$2F
    LD   ($37F8),A

    ; Patch intermission (Cut Scene)
;    LD   HL,$2108                       ; Cut Scene 1
;    LD   HL,$219E                       ; Cut Scene 2
;    LD   HL,$2297                       ; Cut Scene 3
;    LD   ($0A47),HL
 
    ; Patch Red Ghost limit in the Intermission/Cut Scene 2
    LD   A,$7F                          ; SUB $80 -> SUB $7F
    LD   ($2261),A

    ; Patching the messages for print CREDITS xx
    LD   HL,$033F                       ; $803B -> $033F
    LD   ($3723),HL                     ; $3723 Credits Message
    LD   HL,$425F                       ; $4034 -> $425E
    LD   ($2BC0),HL                     ; $2BC0 Coin X-
    LD   HL,$423F                       ; $4033 -> $423E
    LD   ($2BCA),HL                     ; $2BCA Coin -X

    ; Patching the CREDITS printing during the Attract sequence
    LD   HL,$0401                       ; $03FE -> $0401
    LD   ($03CE),HL
    LD   A,$CC                          ; $2BA1 -> $2BCC
    LD   ($23E2),A
    LD   HL,$0118                       ; Skip CALL $2BA1
    LD   ($0952),HL

    RET

; ---------------------------------------------------------------------------
; Set black palette
; ---------------------------------------------------------------------------
set_black_palette
    LD   BC,GATE_ARRAY + INK + NEGRO
    XOR  A
.loop_set_black_palette
    OUT  (C),A
    OUT  (C),C
    INC  A
    CP   $11
    JR   NZ,.loop_set_black_palette
    RET

; ---------------------------------------------------------------------------
; Set game palette
; ---------------------------------------------------------------------------
set_fade_palette
    LD   HL,end_fade_palette - 1
    LD   A,end_fade_palette - fade_palette - 1
    LD   B,>GATE_ARRAY
.loop_set_fade_palette
    OUT  (C),A
    INC  B
    OUTD
    DEC  A
    JP   P,.loop_set_fade_palette
    RET

fade_palette
    DEFB INK + NEGRO,INK + AMARILLO, INK + ROJO,INK + BLANCO
end_fade_palette

; ---------------------------------------------------------------------------
; Initialize PSG and create sound table in the uppper half of the Pac-Man ROM
; used for get_random
; ---------------------------------------------------------------------------
initialize_sound
    ; 1.- Generate sound table
    LD   BC,CPC_SOUND_TABLE
.loop_generate_sound_table
    LD   A,B                ; Map entry address to freq
    AND  $1F
    RRA
    LD   D,A
    LD   A,C
    RRA
    LD   E,A                ; Freq (divisor) now in DE

    LD   HL,0
    EXX

    ; ZX AY frequency
;    LD   DE,$DA7A           ; Dividend in DEHL
;    LD   HL,$8000           ; 111861 << 15 = 0xda7a8000 
;    LD   B,16

    ; CPC AY frequency
    LD  DE,$F424            ; Dividend in DEHL
    LD  HL,$0000            ; (1000000/16) << 16 = 0xf4240000
    LD  B,15

    AND  A
.loop_division
    ADC  HL,HL              ; Shift up for next division
    RL   E
    RL   D
    EXX
    ADC  HL,HL              ; Include new bit
    SBC  HL,DE              ; Does it divide?
    JR   NC,.division_ok
    ADD  HL,DE              ; Add back if not, setting carry
.division_ok
    EXX
    CCF                     ; Set carry if it divided
    DJNZ .loop_division

    ADC  HL,HL              ; Include final bit

    LD   A,H
    EX   AF,AF'
    LD   A,L
    EXX
    LD   (BC),A             ; Note LSB
    INC  C
    EX   AF,AF'
    LD   (BC),A             ; Note MSB
    INC  BC                 ; Freq++
    EX   AF,AF'

    BIT  7,B
    JR   Z,.loop_generate_sound_table
    
    ; Initialize PSG
    LD   HL,psg_init_data
    LD   DE,PPI_PSG_WRITE * 256 + PPI_PSG_SELECT
    XOR  A
.loop_init_psg
    ; Select a PSG Register (PPI_C)
    LD   BC,PPI_C + PPI_PSG_SELECT
    OUT  (C),E

    ; Send PSG register (PPI_A)
    LD   B,>PPI_A
    OUT  (C),A

    ; PSG Inactive (PPI_C)
    LD   B,>PPI_C
    DEFB $ED,$71            ; OUT (C),0 

    ; Send byte to the PSG Register selected (PPI_A)
    DEC  B
    OUTI
    
    ; Write byte in the PSG register (PPI_C)
    LD   B,>PPI_C
    OUT  (C),D

    ; PSG Inactive (PPI_C)
    DEFB $ED,$71        ; OUT (C),0 

    INC  A
    CP   PSG_REG_0B
    JR   NZ,.loop_init_psg
    RET

    ; Set volumes to zero and enable tones A + B + C
psg_init_data
    DEFB 0,0,0,0,0,0,0  ; Frequency for channels A, B, C and Noise
    DEFB %00111000      ; Enable tones in A, B, C | Disable Noise | I/O Ports for input
    DEFB 0,0,0          ; Amplitude for channels A, B and C

; ---------------------------------------------------------------------------
pacman_launcher
    incbin 'pacman_launcher.bin'
end_pacman_launcher

; ---------------------------------------------------------------------------
    END
