; ---------------------------------------------------------------------------
; Constants used by Pac-Man emulator
; ---------------------------------------------------------------------------

;    Memory Map
; ---------------
; (0)  $00000 - $03FFF Pac-Man Patched ROM
; (1)  $04000 - $07FFF VRAM 1
; (2)  $08000 - $0BFFF CPC Code / Stack
; (3)  $0C000 - $0FFFF VRAM 2
; (0*) $10000 - $11FFF Original Pac-Man ROM for get_random (only use the first 8 KBs)
; (0*) $12000 - $13FFF Sound table (8KBs)
; (1*) $14000 - $17FFF Pac-Man RAM
; (2*) $18000 - $1BFFF CPC Code (Copy) *** FREE ***
; (3*) $1C000 - $1FFFF CPC Graphics

; CPC RAM Banking modes
; ---------------------
; 0  1  2  3  
; 0  1  2  3* Update VRAM 1
; 0* 1* 2* 3* 
; 0  3  2  3* Update VRAM 2
; 0  0* 2  3  Sound player
; 0  1* 2  3  Run Game Code
; 0  2* 2  3
; 0  3* 2  3

UPDATE_VRAM1            EQU $C1         ; 0   1   2   3*
UPDATE_VRAM2            EQU $C3         ; 0   3   2   3*
RUN_CODE                EQU $C5         ; 0   1*  2   3
SOUND_TABLE             EQU $C4         ; 0   0*  2   3
RANDOM_TABLE            EQU $C4         ; 0   0*  2   3
MENU_CODE               EQU $C6         ; 0   2*  2   3

LAUNCH_MENU             EQU $4000

; Pac-Man DIPs
DIP_5000                EQU $5000
DIP_5040                EQU $5040
DIP_5080                EQU $5080

; Pac-Man DIPs config (0 Enable / 1 Disable)
DIP_5000_VALUE          EQU %11111111   ; Rack test off, nothing else signalled (%c21tdrlu)
                                        ; c=credit
                                        ; 2=coin2
                                        ; 1=coin1
                                        ; t=rack advance (0 = Go to next level)
                                        ; down, right, left, up (player 1)

DIP_5040_VALUE          EQU %11111111   ; Upright mode, service mode off, no controls pressed (%c--sdrlu)
                                        ; c=cocktail/upright 
                                        ; s=service mode on/off 
                                        ; down, right, left, up (player 2)
                            
DIP_5080_VALUE          EQU %11001001   ; Normal, bonus life at 10K, 3 lives, 1 coin 1 credit (%gdbbllcc)
                                        ; g=alternate/normal ghost names
                                        ; d=hard/normal 
                                        ; bb=bonus life at 10K/15K/20K/none 
                                        ; ll=1/2/3/5 lives 
                                        ; cc=freeplay/1coin1credit/1coin2credits/2coins1credit

; DIP Bit constants
; DIP $5000
P1_UP                   EQU 0
P1_LEFT                 EQU 1
P1_RIGHT                EQU 2
P1_DOWN                 EQU 3
DIP_RACK                EQU 4
DIP_COIN1               EQU 5
DIP_COIN2               EQU 6
DIP_CREDIT              EQU 7

; DIP $5040
P2_UP                   EQU 0
P2_LEFT                 EQU 1
P2_RIGHT                EQU 2
P2_DOWN                 EQU 3
DIP_SERVICE_MODE        EQU 4
DIP_START1              EQU 5
DIP_START2              EQU 6
DIP_COCKTAIL            EQU 7

; DIP $5080
DIP_COIN_MODE0          EQU 0
DIP_COIN_MODE1          EQU 1
DIP_LIVES0              EQU 2
DIP_LIVES1              EQU 3
DIP_BONUS0              EQU 4
DIP_BONUS1              EQU 5
DIP_DIFFICULT           EQU 6

; Variables in the Pac-Man RAM
PAC_FOOTER              EQU $4000       ; Credit and fruit display
PAC_CHARS               EQU $4040       ; Start of main Pac-Man display (skipping the score rows)
PAC_HEADER              EQU $43C0       ; 64 bytes containing the score
RANDOM_SEED             EQU $4DC9
FRUIT_POS               EQU $4DD2       ; 0 = Not fruit bonus visible
PAC_NUM_PLAYER          EQU $4E09       ; 0 = P1 / 1 = P2
PAC_LEVEL_NUMBER        EQU $4E13

; Sound registers 
FREQ_VOL_0              EQU $5051       ; Voice 0 freq and volume
WAVEFORM_0              EQU $5045       ; Voice 0 waveform
FREQ_VOL_1              EQU $5051 + 5   ; Voice 1 freq and volume
WAVEFORM_1              EQU $5045 + 5   ; Voice 1 waveform
FREQ_VOL_2              EQU $5051 + 10  ; Voice 2 freq and volume
WAVEFORM_2              EQU $5045 + 10  ; Voice 2 waveform

; Sprites Variables
; Sprites x,y 
SPRITE1_XPOS            EQU $88D0       ; Sprite 1 coord X
SPRITE1_YPOS            EQU $88D1       ; Sprite 1 coord Y
SPRITE2_XPOS            EQU $88D2       ; Sprite 2 coord X
SPRITE2_YPOS            EQU $88D3       ; Sprite 2 coord Y
SPRITE3_XPOS            EQU $88D4       ; Sprite 3 coord X
SPRITE3_YPOS            EQU $88D5       ; Sprite 3 coord Y
SPRITE4_XPOS            EQU $88D6       ; Sprite 4 coord X
SPRITE4_YPOS            EQU $88D7       ; Sprite 4 coord Y
SPRITE5_XPOS            EQU $88D8       ; Sprite 5 coord X
SPRITE5_YPOS            EQU $88D9       ; Sprite 5 coord Y
SPRITE6_XPOS            EQU $88DA       ; Sprite 6 coord X
SPRITE6_YPOS            EQU $88DB       ; Sprite 6 coord Y
; Sprite coords limits
MIN_SPR_Y               EQU 16+4
MIN_SPR_X               EQU 16+14
MAX_SPR_X               EQU 241

; Sprites Attributes
; First byte:
;    Bits 7 - 2 Sprite Tile Number (0 - 63)
;    Bit      1 Flip X bit
;    Bit      0 Flip Y bit
; Second byte is palette
SPRITE1_ATRB            EQU $88DC
SPRITE2_ATRB            EQU $88DE
SPRITE3_ATRB            EQU $88E0
SPRITE4_ATRB            EQU $88E2
SPRITE5_ATRB            EQU $88E4
SPRITE6_ATRB            EQU $88E6

; Buffers for the background occupied by the sprites (Ghost Height is maximum height -> 14)
SPRITE1_SAVE1           EQU $FFF2 - (5 * 14 + 3) * 12   ; $FC86
SPRITE2_SAVE1           EQU SPRITE1_SAVE1 + 5 * 14 + 3  ; $FCCF
SPRITE3_SAVE1           EQU SPRITE2_SAVE1 + 5 * 14 + 3  ; $FD18
SPRITE4_SAVE1           EQU SPRITE3_SAVE1 + 5 * 14 + 3  ; $FD61
SPRITE5_SAVE1           EQU SPRITE4_SAVE1 + 5 * 14 + 3  ; $FDAA
SPRITE6_SAVE1           EQU SPRITE5_SAVE1 + 5 * 14 + 3  ; $FDF3
SPRITE1_SAVE2           EQU SPRITE6_SAVE1 + 5 * 14 + 3  ; $FE3C
SPRITE2_SAVE2           EQU SPRITE1_SAVE2 + 5 * 14 + 3  ; $FE85
SPRITE3_SAVE2           EQU SPRITE2_SAVE2 + 5 * 14 + 3  ; $FECE
SPRITE4_SAVE2           EQU SPRITE3_SAVE2 + 5 * 14 + 3  ; $FF17
SPRITE5_SAVE2           EQU SPRITE4_SAVE2 + 5 * 14 + 3  ; $FF60
SPRITE6_SAVE2           EQU SPRITE5_SAVE2 + 5 * 14 + 3  ; $FFA9

; Routines and buffers in the CPC code page
CPC_GET_RANDOM          EQU $8A76       ; CPC get_random routine
CPC_TEXT_FIX            EQU $8A96       ; CPC get_text_fix routine
CPC_EMU_CODE            EQU $8A9C       ; Check for VBlank before run our code
CPC_TILEMAP1            EQU $B740       ; Copy of Pac-Man display for Buffer 1 ($xx40)
CPC_TILEMAP2            EQU $BB40       ; Copy of Pac-Man display for Buffer 2 ($xx40)
CPC_STACK               EQU $C000       ; Stack used for the CPC Code (256 bytes)

DEBUG_RASTER            EQU 0
SM_INT_HANDLER          EQU $8B0D + 1 + DEBUG_RASTER * $3F  ; Pointer to the arcade int handler

; CPC Sound
CPC_SOUND_TABLE         EQU $6000       ; Second half of RAM page 4 (0*)

; Arcade VRAM strips used to analyze tile changes
LEN_STRIP0              EQU 6
LEN_STRIP1              EQU 5
LEN_STRIP2              EQU 6
LEN_STRIP3              EQU 5
LEN_STRIP4              EQU 6
LEN_STRIP0_1            EQU LEN_STRIP0 + LEN_STRIP1
LEN_STRIP0_2            EQU LEN_STRIP0 + LEN_STRIP1 + LEN_STRIP2 
LEN_STRIP0_3            EQU LEN_STRIP0 + LEN_STRIP1 + LEN_STRIP2 + LEN_STRIP3 
LEN_STRIP0_4            EQU LEN_STRIP0 + LEN_STRIP1 + LEN_STRIP2 + LEN_STRIP3 + LEN_STRIP4 

; ---------------------------------------------------------------------------
; Constants for offset in graphics tiles
; ---------------------------------------------------------------------------
; Offset in the tiles (Arcade)
BEFORE_FRUIT_TILE       EQU $90
FIRST_GHOST_TILE        EQU $B0
AFTER_GHOST_TILE        EQU FIRST_GHOST_TILE + 6
BEGIN_LABY_TILE         EQU $CE        
RELOC_LABY_TILE         EQU $3E
CPC_BLANK_TILE          EQU $4E

; ---------------------------------------------------------------------------
; CPC graphics *** WIP ***
CPC_TILE_DATA           EQU $8000       ; Background tile graphics data
CPC_SPRITES_WIDTH       EQU 5           ; 5 bytes (20 pixels)
CPC_SPRITES_HEIGHT      EQU 16          ; 16 scanlines (16 pixels)
PRESHIFT                EQU 4           ; Sprites are preshifted 4 times
CPC_SPRITES_NOPRE_WIDTH EQU 8           ; Pac-Man Death and Scores sprites are not preshifted

; CPC sprite heights
GHOST_HEIGHT            EQU 14          ; Normal, Scared and Eaten ghosts   (CY + 1)
IGHOST_BLUE_HEIGHT      EQU 11          ; Blue intermission ghost           (CY + 4)
IGHOST_RED12_HEIGHT     EQU 4           ; Red intermission ghost 12         (CY + 11)
IGHOST_RED13_HEIGHT     EQU 3           ; Red intermission ghost 13         (CY + 12)
GHOST_EYES_HEIGHT       EQU 5           ; Ghost Eyes                        (CY + 4/5/4/2)
BIGPAC_HEIGHT           EQU 16          ; Big Pac-Man (special case)        (CY + 0)
SCORE_HEIGHT            EQU 7           ; Scores                            (CY + 4)
PACMAN44_HEIGHT         EQU 13          ; Pacman 44                         (CY + 1/1)
PACMAN45_HEIGHT         EQU 9           ; Pacman 45                         (CY + 1/6)
PACMAN46_HEIGHT         EQU 13          ; Pacman 46                         (CY + 1/1)
PACMAN47_HEIGHT         EQU 12          ; Pacman 47                         (CY + 1/3)
PAC_BALL_HEIGHT         EQU 13          ; Pac-Ball :P                       (CY + 1)
PAC_DEATH52_HEIGHT      EQU 9 + 1 ; FIX ; Pac-Death 52                      (CY + 3)
PAC_DEATH53_HEIGHT      EQU 8           ; Pac-Death 53                      (CY + 4)
PAC_DEATH54_HEIGHT      EQU 6           ; Pac-Death 54                      (CY + 6)5
PAC_DEATH55_HEIGHT      EQU 5           ; Pac-Death 55                      (CY + 7)6
PAC_DEATH56_HEIGHT      EQU 5           ; Pac-Death 56                      (CY + 8)7
PAC_DEATH57_HEIGHT      EQU 6           ; Pac-Death 57                      (CY + 8)7
PAC_DEATH58_HEIGHT      EQU 7           ; Pac-Death 58                      (CY + 8)7
PAC_DEATH59_HEIGHT      EQU 7           ; Pac-Death 59                      (CY + 8)7
PAC_DEATH60_HEIGHT      EQU 7           ; Pac-Death 60                      (CY + 8)
PAC_DEATH61_HEIGHT      EQU 6           ; Pac-Death 61                      (CY + 8)
PAC_DEATH62_HEIGHT      EQU 10          ; Pac-Death 62                      (CY + 6)
FRUIT00_HEIGHT          EQU 12          ; Fruit 00                          (CY + 2)
FRUIT01_HEIGHT          EQU 12          ; Fruit 01                          (CY + 2)
FRUIT02_HEIGHT          EQU 12          ; Fruit 02                          (CY + 2)
FRUIT03_HEIGHT          EQU 13          ; Fruit 03                          (CY + 2)
FRUIT04_HEIGHT          EQU 12          ; Fruit 04                          (CY + 2)
FRUIT05_HEIGHT          EQU 14          ; Fruit 05                          (CY + 1)
FRUIT06_HEIGHT          EQU 11          ; Fruit 06                          (CY + 3)
FRUIT07_HEIGHT          EQU 13          ; Fruit 07                          (CY + 2)

; CPC sprite pointers
; Fruits
SPR_FRUIT_WIDTH         EQU 4
SPR_FRUIT_00            EQU $88EA
SPR_FRUIT_01            EQU SPR_FRUIT_00 + FRUIT00_HEIGHT * SPR_FRUIT_WIDTH
SPR_FRUIT_02            EQU SPR_FRUIT_01 + FRUIT01_HEIGHT * SPR_FRUIT_WIDTH
SPR_FRUIT_03            EQU SPR_FRUIT_02 + FRUIT02_HEIGHT * SPR_FRUIT_WIDTH
SPR_FRUIT_04            EQU SPR_FRUIT_03 + FRUIT03_HEIGHT * SPR_FRUIT_WIDTH
SPR_FRUIT_05            EQU SPR_FRUIT_04 + FRUIT04_HEIGHT * SPR_FRUIT_WIDTH
SPR_FRUIT_06            EQU SPR_FRUIT_05 + FRUIT05_HEIGHT * SPR_FRUIT_WIDTH
SPR_FRUIT_07            EQU SPR_FRUIT_06 + FRUIT06_HEIGHT * SPR_FRUIT_WIDTH

; Masked Sprites
BASE_MASKED_SPR         EQU $C000

; Ghost
SPR_GHOST_WIDTH         EQU 4
SPR_GHOST_RED_32        EQU BASE_MASKED_SPR
SPR_GHOST_RED_33        EQU SPR_GHOST_RED_32 + SPR_GHOST_WIDTH * GHOST_HEIGHT * PRESHIFT
SPR_GHOST_RED_34        EQU SPR_GHOST_RED_33 + SPR_GHOST_WIDTH * GHOST_HEIGHT * PRESHIFT
SPR_GHOST_RED_35        EQU SPR_GHOST_RED_34 + SPR_GHOST_WIDTH * GHOST_HEIGHT * PRESHIFT
SPR_GHOST_RED_36        EQU SPR_GHOST_RED_35 + SPR_GHOST_WIDTH * GHOST_HEIGHT * PRESHIFT
SPR_GHOST_RED_37        EQU SPR_GHOST_RED_36 + SPR_GHOST_WIDTH * GHOST_HEIGHT * PRESHIFT
SPR_GHOST_RED_38        EQU SPR_GHOST_RED_37 + SPR_GHOST_WIDTH * GHOST_HEIGHT * PRESHIFT
SPR_GHOST_RED_39        EQU SPR_GHOST_RED_38 + SPR_GHOST_WIDTH * GHOST_HEIGHT * PRESHIFT

SPR_GHOST_PINK_32       EQU SPR_GHOST_RED_39 + SPR_GHOST_WIDTH * GHOST_HEIGHT * PRESHIFT
SPR_GHOST_PINK_33       EQU SPR_GHOST_PINK_32 + SPR_GHOST_WIDTH * GHOST_HEIGHT * PRESHIFT
SPR_GHOST_PINK_34       EQU SPR_GHOST_PINK_33 + SPR_GHOST_WIDTH * GHOST_HEIGHT * PRESHIFT
SPR_GHOST_PINK_35       EQU SPR_GHOST_PINK_34 + SPR_GHOST_WIDTH * GHOST_HEIGHT * PRESHIFT
SPR_GHOST_PINK_36       EQU SPR_GHOST_PINK_35 + SPR_GHOST_WIDTH * GHOST_HEIGHT * PRESHIFT
SPR_GHOST_PINK_37       EQU SPR_GHOST_PINK_36 + SPR_GHOST_WIDTH * GHOST_HEIGHT * PRESHIFT
SPR_GHOST_PINK_38       EQU SPR_GHOST_PINK_37 + SPR_GHOST_WIDTH * GHOST_HEIGHT * PRESHIFT
SPR_GHOST_PINK_39       EQU SPR_GHOST_PINK_38 + SPR_GHOST_WIDTH * GHOST_HEIGHT * PRESHIFT

SPR_GHOST_CYAN_32       EQU SPR_GHOST_PINK_39 + SPR_GHOST_WIDTH * GHOST_HEIGHT * PRESHIFT
SPR_GHOST_CYAN_33       EQU SPR_GHOST_CYAN_32 + SPR_GHOST_WIDTH * GHOST_HEIGHT * PRESHIFT
SPR_GHOST_CYAN_34       EQU SPR_GHOST_CYAN_33 + SPR_GHOST_WIDTH * GHOST_HEIGHT * PRESHIFT
SPR_GHOST_CYAN_35       EQU SPR_GHOST_CYAN_34 + SPR_GHOST_WIDTH * GHOST_HEIGHT * PRESHIFT
SPR_GHOST_CYAN_36       EQU SPR_GHOST_CYAN_35 + SPR_GHOST_WIDTH * GHOST_HEIGHT * PRESHIFT
SPR_GHOST_CYAN_37       EQU SPR_GHOST_CYAN_36 + SPR_GHOST_WIDTH * GHOST_HEIGHT * PRESHIFT
SPR_GHOST_CYAN_38       EQU SPR_GHOST_CYAN_37 + SPR_GHOST_WIDTH * GHOST_HEIGHT * PRESHIFT
SPR_GHOST_CYAN_39       EQU SPR_GHOST_CYAN_38 + SPR_GHOST_WIDTH * GHOST_HEIGHT * PRESHIFT

SPR_GHOST_ORANGE_32     EQU SPR_GHOST_CYAN_39 + SPR_GHOST_WIDTH * GHOST_HEIGHT * PRESHIFT
SPR_GHOST_ORANGE_33     EQU SPR_GHOST_ORANGE_32 + SPR_GHOST_WIDTH * GHOST_HEIGHT * PRESHIFT
SPR_GHOST_ORANGE_34     EQU SPR_GHOST_ORANGE_33 + SPR_GHOST_WIDTH * GHOST_HEIGHT * PRESHIFT
SPR_GHOST_ORANGE_35     EQU SPR_GHOST_ORANGE_34 + SPR_GHOST_WIDTH * GHOST_HEIGHT * PRESHIFT
SPR_GHOST_ORANGE_36     EQU SPR_GHOST_ORANGE_35 + SPR_GHOST_WIDTH * GHOST_HEIGHT * PRESHIFT
SPR_GHOST_ORANGE_37     EQU SPR_GHOST_ORANGE_36 + SPR_GHOST_WIDTH * GHOST_HEIGHT * PRESHIFT
SPR_GHOST_ORANGE_38     EQU SPR_GHOST_ORANGE_37 + SPR_GHOST_WIDTH * GHOST_HEIGHT * PRESHIFT
SPR_GHOST_ORANGE_39     EQU SPR_GHOST_ORANGE_38 + SPR_GHOST_WIDTH * GHOST_HEIGHT * PRESHIFT

; Scare Ghost
SPR_SGHOST_YELLOW_28    EQU SPR_GHOST_ORANGE_39 + SPR_GHOST_WIDTH * GHOST_HEIGHT * PRESHIFT
SPR_SGHOST_YELLOW_29    EQU SPR_SGHOST_YELLOW_28 + SPR_GHOST_WIDTH * GHOST_HEIGHT * PRESHIFT
SPR_SGHOST_BLUE_28      EQU SPR_SGHOST_YELLOW_29 + SPR_GHOST_WIDTH * GHOST_HEIGHT * PRESHIFT
SPR_SGHOST_BLUE_29      EQU SPR_SGHOST_BLUE_28 + SPR_GHOST_WIDTH * GHOST_HEIGHT * PRESHIFT

; Pac-Man
SPR_PAC_WIDTH           EQU 6
SPR_PACMAN_RIGHT_44     EQU SPR_SGHOST_BLUE_29 + SPR_GHOST_WIDTH * GHOST_HEIGHT * PRESHIFT
SPR_PACMAN_DOWN_45      EQU SPR_PACMAN_RIGHT_44 + SPR_PAC_WIDTH * PACMAN44_HEIGHT * PRESHIFT
SPR_PACMAN_RIGHT_46     EQU SPR_PACMAN_DOWN_45 + SPR_PAC_WIDTH * PACMAN45_HEIGHT * PRESHIFT
SPR_PACMAN_DOWN_47      EQU SPR_PACMAN_RIGHT_46 + SPR_PAC_WIDTH * PACMAN46_HEIGHT * PRESHIFT
SPR_PACMAN_LEFT_44      EQU SPR_PACMAN_DOWN_47 + SPR_PAC_WIDTH * PACMAN47_HEIGHT * PRESHIFT
SPR_PACMAN_UP_45        EQU SPR_PACMAN_LEFT_44 + SPR_PAC_WIDTH * PACMAN44_HEIGHT * PRESHIFT
SPR_PACMAN_LEFT_46      EQU SPR_PACMAN_UP_45 + SPR_PAC_WIDTH * PACMAN45_HEIGHT * PRESHIFT
SPR_PACMAN_UP_47        EQU SPR_PACMAN_LEFT_46 + SPR_PAC_WIDTH * PACMAN46_HEIGHT * PRESHIFT
SPR_PACMAN_48           EQU SPR_PACMAN_UP_47 + SPR_PAC_WIDTH * PACMAN47_HEIGHT * PRESHIFT

; Eyes (not preshifted)
SPR_EYES_RIGHT          EQU SPR_PACMAN_48 + SPR_PAC_WIDTH * PAC_BALL_HEIGHT * PRESHIFT 
SPR_EYES_DOWN           EQU SPR_EYES_RIGHT + CPC_SPRITES_NOPRE_WIDTH * GHOST_EYES_HEIGHT
SPR_EYES_LEFT           EQU SPR_EYES_DOWN + CPC_SPRITES_NOPRE_WIDTH * GHOST_EYES_HEIGHT
SPR_EYES_UP             EQU SPR_EYES_LEFT + CPC_SPRITES_NOPRE_WIDTH * GHOST_EYES_HEIGHT

; Score (not preshifted)
SPR_SCORE_40            EQU SPR_EYES_UP + CPC_SPRITES_NOPRE_WIDTH * GHOST_EYES_HEIGHT
SPR_SCORE_41            EQU SPR_SCORE_40 + CPC_SPRITES_NOPRE_WIDTH * SCORE_HEIGHT
SPR_SCORE_42            EQU SPR_SCORE_41 + CPC_SPRITES_NOPRE_WIDTH * SCORE_HEIGHT
SPR_SCORE_43            EQU SPR_SCORE_42 + CPC_SPRITES_NOPRE_WIDTH * SCORE_HEIGHT

; Pac-Man Death (not preshifted)
SPR_PACDEATH_52         EQU SPR_SCORE_43 + CPC_SPRITES_NOPRE_WIDTH * SCORE_HEIGHT
SPR_PACDEATH_53         EQU SPR_PACDEATH_52 + CPC_SPRITES_NOPRE_WIDTH * PAC_DEATH52_HEIGHT
SPR_PACDEATH_54         EQU SPR_PACDEATH_53 + CPC_SPRITES_NOPRE_WIDTH * PAC_DEATH53_HEIGHT
SPR_PACDEATH_55         EQU SPR_PACDEATH_54 + CPC_SPRITES_NOPRE_WIDTH * PAC_DEATH54_HEIGHT
SPR_PACDEATH_56         EQU SPR_PACDEATH_55 + CPC_SPRITES_NOPRE_WIDTH * PAC_DEATH55_HEIGHT
SPR_PACDEATH_57         EQU SPR_PACDEATH_56 + CPC_SPRITES_NOPRE_WIDTH * PAC_DEATH56_HEIGHT
SPR_PACDEATH_58         EQU SPR_PACDEATH_57 + CPC_SPRITES_NOPRE_WIDTH * PAC_DEATH57_HEIGHT
SPR_PACDEATH_59         EQU SPR_PACDEATH_58 + CPC_SPRITES_NOPRE_WIDTH * PAC_DEATH58_HEIGHT
SPR_PACDEATH_60         EQU SPR_PACDEATH_59 + CPC_SPRITES_NOPRE_WIDTH * PAC_DEATH59_HEIGHT
SPR_PACDEATH_61         EQU SPR_PACDEATH_60 + CPC_SPRITES_NOPRE_WIDTH * PAC_DEATH60_HEIGHT
SPR_PACDEATH_62         EQU SPR_PACDEATH_61 + CPC_SPRITES_NOPRE_WIDTH * PAC_DEATH61_HEIGHT

; Intermission Ghost
SPR_IGHOST_08           EQU SPR_PACDEATH_62 + CPC_SPRITES_NOPRE_WIDTH * PAC_DEATH62_HEIGHT
SPR_IGHOST_09           EQU SPR_IGHOST_08 + CPC_SPRITES_NOPRE_WIDTH * GHOST_HEIGHT
SPR_EGHOST_50           EQU SPR_IGHOST_09 + CPC_SPRITES_NOPRE_WIDTH * GHOST_HEIGHT
SPR_EGHOST_51           EQU SPR_EGHOST_50 + CPC_SPRITES_NOPRE_WIDTH * GHOST_HEIGHT
SPR_IGHOST_10           EQU SPR_EGHOST_51 + CPC_SPRITES_NOPRE_WIDTH * GHOST_HEIGHT
SPR_IGHOST_11           EQU SPR_IGHOST_10 + CPC_SPRITES_NOPRE_WIDTH * IGHOST_BLUE_HEIGHT
SPR_IGHOST_12           EQU SPR_IGHOST_11 + CPC_SPRITES_NOPRE_WIDTH * IGHOST_BLUE_HEIGHT
SPR_IGHOST_13           EQU SPR_IGHOST_12 + CPC_SPRITES_NOPRE_WIDTH * IGHOST_RED12_HEIGHT
    
; Big Pac-Man
SPR_BIGPAC_WIDTH        EQU 9
SPR_BIGPAC_SPR_01       EQU SPR_IGHOST_13 + CPC_SPRITES_NOPRE_WIDTH * IGHOST_RED13_HEIGHT
SPR_BIGPAC_SPR_02       EQU SPR_BIGPAC_SPR_01 + SPR_BIGPAC_WIDTH * BIGPAC_HEIGHT * PRESHIFT
SPR_BIGPAC_SPR_03       EQU SPR_BIGPAC_SPR_02 + SPR_BIGPAC_WIDTH * BIGPAC_HEIGHT * PRESHIFT

; ---------------------------------------------------------------------------
; CRTC values for the Pac-Man screen (256 x 256)
; ---------------------------------------------------------------------------
WIDTH_CRTC              EQU 32
HEIGHT_CRTC             EQU 32
HORIZONTAL_OFFSET       EQU 26 + WIDTH_CRTC / 2    
; PAL
VERTICAL_OFFSET_PAL     EQU 16 + (HEIGHT_CRTC * 8) / 14
VERTICAL_TOTAL_PAL      EQU 38
VERTICAL_ADJUST_PAL     EQU 0
; NTSC
VERTICAL_OFFSET_NTSC    EQU 31
VERTICAL_TOTAL_NTSC     EQU 31
VERTICAL_ADJUST_NTSC    EQU 4 ;6

; ---------------------------------------------------------------------------
; Zone with 0s in the Pac-Man rom
ZERO_ZONE               EQU $3E5C           ; Size 414 bytes

; ---------------------------------------------------------------------------
