; ---------------------------------------------------------------------------
; Menu code for the Pac-Man emulator
; ---------------------------------------------------------------------------

    include 'constantes.i'
    include 'hardware.i'
    include 'macros.i'
    include 'macros_pacman.i'
    
; ---------------------------------------------------------------------------
VRAM_BUFFER             EQU $6000
HEIGHT_MENU             EQU 15
FIRST_ROW               EQU 5

; ---------------------------------------------------------------------------
psg_diff                EQU 45                  ; play_sound - psg_sound_list

; ---------------------------------------------------------------------------
; Tile constants for the menus
; Numbers
T_0                     EQU $10
T_1                     EQU $11
T_2                     EQU $12
T_3                     EQU $13
T_4                     EQU $14
T_5                     EQU $15
T_6                     EQU $16
T_7                     EQU $17
T_8                     EQU $18
T_9                     EQU $19

; Letters
T_A                     EQU $1A
T_B                     EQU $1B
T_C                     EQU $1C
T_D                     EQU $1D
T_E                     EQU $1E
T_F                     EQU $1F
T_G                     EQU $20
T_H                     EQU $21
T_I                     EQU $22
T_J                     EQU $23
T_K                     EQU $24
T_L                     EQU $25
T_M                     EQU $26
T_N                     EQU $27
T_O                     EQU $28
T_P                     EQU $29
T_Q                     EQU $2A
T_R                     EQU $2B
T_S                     EQU $2C
T_T                     EQU $2D
T_U                     EQU $2E
T_V                     EQU $2F
T_W                     EQU $30
T_X                     EQU $31
T_Y                     EQU $32
T_Z                     EQU $33

; Special chars
T_SPC                   EQU $4E         ; Space
T_SEP                   EQU $0F         ; -
T_PAC                   EQU $0B         ; Pac-Live
T_LB_UR                 EQU $6F ;5D         ; -||
T_LB_UL                 EQU $70 ;5E         ; ||-
T_LB_VR                 EQU $5F         ;  ||
T_LB_VL                 EQU $60         ; ||
T_LB_EU                 EQU $67         ; -
T_LB_ED                 EQU $68         ; _
T_LB_CRD                EQU $79         ; |=_
T_LB_CLD                EQU $7A         ; _=|
T_LB_CRU                EQU $79         ; |=-
T_LB_CLU                EQU $7A         ; -=|

; ---------------------------------------------------------------------------
; Macro for simplify the syntax in the menu definition
; ---------------------------------------------------------------------------
 MACRO BLANK_LINE
    DEFS 32,T_SPC
 ENDM

 MACRO MENU_BLANK_LINE
    DEFB T_SPC,T_LB_VR
    DEFS 32 - 4,T_SPC
    DEFB T_LB_VL,T_SPC
 ENDM

; ---------------------------------------------------------------------------
; Macro for waiting scanlines and simplify the syntax
; ---------------------------------------------------------------------------
 MACRO WAIT_SCANLINES
    LD   IY,\1 * 4 - 1                                          ; (4) 4
    CALL wait_scanlines_iy                                      ; (5) 3
 ENDM                                                           ; (9) 7 bytes

    ORG  $4000

; ---------------------------------------------------------------------------
; No tocar IXH, ya que mantiene la página
launch_menu
    ; Save DIP config
    LD   (DIP_TMP),A
    LD   (OLD_DIP),A
 
    ; Save Level number
    LD   A,E
    LD   (LEVEL_TMP),A
    LD   (OLD_LEVEL),A
    EX   AF,AF'
    XOR  A                      ; A' = first char of level string
    LD   (control_menu.pretest_up + 1),A    ; Skip update level
    EX   AF,AF'
 
    ; Shut PSG
    CALL shut_psg

    ; Show VRAM in $4000
    LD   C,VRAM_P1
    CALL show_vram

    ; Set Menu palette
    LD   HL,end_menu_palette - 1
    LD   A,end_menu_palette - menu_palette - 1
    CALL set_palette
    
    ; Save VRAM in $C000 that is going to be covered by the menu (8 KBs maximum)
    CALL save_vram

    ; Patch Interrupt handler
    LD   HL,($0038)
    LD   (.sm_int_handler + 1),HL
    LD   HL,$C9FB                   ; EI ($FB) + RET ($C9)
    LD   ($0038),HL
    EI

    ; Update level number
    CALL update_level_number

    ; Print Main menu
    CALL print_main_menu

    ; Print options
    CALL decode_and_print_options

    ; Initialize cursor position
    LD   A,(control_menu.sm_cursor_position + 1)
    CALL print_cursor

    ; Show VRAM in $C000
    LD   C,VRAM_P3
    CALL show_vram

    ; Wait until not key is pressed
    CALL no_key_press

    XOR  A
    LD   (control_menu.sm_ignore_keys + 1),A

    ; Manage menu
    CALL control_menu

    ; Restore Interrupt handler
    DI
.sm_int_handler
    LD   HL,$0000
    LD   ($0038),HL

    ; Show VRAM in $4000
    LD   C,VRAM_P1
    CALL show_vram

    ; Restore VRAM in $C000 that has been covered by the menu (8 KBs maximum)
    CALL restore_vram

    ; Restore the VRAM visible before enter in the menu
    LD   A,IXH                          ; Get the page is being update
    XOR  $02                            ; Get the page is visible
    RLCA
    RLCA
    RLCA
    RLCA                                ; A << 4
    AND  $30                            ; Mask the bits
    LD   C,A
    CALL show_vram

    ; Set Game palette
    LD   HL,end_game_palette - 1
    LD   A,end_game_palette - game_palette - 1
    CALL set_palette

    ; Wait until not key is pressed
    CALL no_key_press

    ; Test for changes in the config
    LD   HL,DIP_TMP
    LD   A,(HL)
    INC  HL
    CP   (HL)
    JR   NZ,.reset_arcade_and_exit

    ; Changes in the level number
    LD   HL,LEVEL_TMP
    LD   A,(HL)
    INC  HL
    CP   (HL)    
    RET                                 ; Normal exit

.reset_arcade_and_exit
    ; Update DIP config
    LD   (end_pacman_relauncher - 7),A

    ; Fill both tilemaps with spaces ($40)
    LD   HL,CPC_TILEMAP1        ; $xx40
    LD   (HL),$40               ; $40 = Space
    LD   DE,CPC_TILEMAP1 + 1
    LD   BC,$800 - $40 - 1
    LDIR 

    ; Clear VRAM in $C000
    LD   HL,$C000
    LD   DE,$C001
    LD   BC,$3FFF
    LD   (HL),0
    LDIR

    ; The arcade use two interrupt handlers ($30B0 and $008D), the first is used
    ; only the first time, the other is used all the time. Because that we make 
    ; a jump during the first to set the second and simplify our int handler code. 
    LD   A,$C3                  ; JP   xxxx
    LD   ($3181),A
    LD   HL,CPC_STACK - 80
    LD   ($3182),HL 

    ; Set Pac-Man interrupt handler for the first run
    LD   HL,$30B0
    LD   (SM_INT_HANDLER),HL
    
    ; Copy the launcher to the top of the stack
    LD   HL,pacman_relauncher
    LD   DE,CPC_STACK - 80
    LD   BC,end_pacman_relauncher - pacman_relauncher
    LDIR
    
    ; Execute launcher
    JP   CPC_STACK + $17 - 80 

pacman_relauncher
    incbin 'pacman_relauncher.bin'
end_pacman_relauncher
 
; ---------------------------------------------------------------------------
; raster, read_key, print_menu
control_menu
    CALL wait_vblank

    ; Center screen
.sm_screen_hcenter
    LD   A,$00
    ADD  HORIZONTAL_OFFSET
    LD   BC,CRTC_SELECT + REG_02
    OUT  (C),C
    INC  B
    OUT  (C),A

.sm_vertical_total
    LD   A,VERTICAL_TOTAL_PAL
    LD   BC,CRTC_SELECT + REG_04
    OUT  (C),C
    INC  B
    OUT  (C),A

   
.sm_vertical_adjust
    LD   A,VERTICAL_ADJUST_PAL
    DEC  B
    INC  C
    OUT  (C),C
    INC  B
    OUT  (C),A
    
.sm_vertical_offset
    LD   A,VERTICAL_OFFSET_PAL
    LD   BC,CRTC_SELECT + REG_07
    OUT  (C),C
    INC  B
    OUT  (C),A

.sm_wait_upper_sync
    JR  .wait_upper_pal
.wait_upper_ntsc
    WAIT_SCANLINES 26
    JR  .begin_upper_rasters
.wait_upper_pal
    HALT
    HALT
.begin_upper_rasters
    WAIT_SCANLINES 32
    SET_PEN 1,AZUL_BRILLANTE            ; Border
    SET_PEN 2,AMARILLO_BRILLANTE        ; Tittle
    WAIT_SCANLINES 8
    SET_PEN 2,BLANCO_BRILLANTE          ; Letters
    SET_PEN 3,AMARILLO_BRILLANTE        ; Cursor
    HALT

    ; Read keyboard and decode
.sm_ignore_keys
    LD   A,$00
    OR   A
    JR   Z,.check_controls
    
    ; Check for keys pressed
    CALL anykey_press
    JP   C,.end_key_check

    LD   A,(.sm_ignore_keys + 1)
    XOR  $FF
    LD   (.sm_ignore_keys + 1),A
    JP   .end_key_check

.check_controls
    CALL read_controls

    ; TAB/Fire 3
.test_tab
    BIT  JOY_FIRE3,A
    RET  Z

.test_fire_1
    BIT  JOY_FIRE1,A
    JR   NZ,.test_fire_2
    SET  JOY_FIRE1,A
.test_fire_1_left
    BIT  JOY_LEFT,A
    JR   NZ,.test_fire_1_right
    ; Set ignore keyboard
    LD   A,(.sm_ignore_keys + 1)
    XOR  $FF
    LD   (.sm_ignore_keys + 1),A
    ; Shift screen to the left
    LD   A,(.sm_screen_hcenter + 1)
    CP   3
    JP   Z,.end_key_check
    INC  A
    LD   (.sm_screen_hcenter + 1),A
    JP   .end_key_check
    
.test_fire_1_right
    BIT  JOY_RIGHT,A
    JR   NZ,.test_fire_2
    ; Set ignore keyboard
    LD   A,(.sm_ignore_keys + 1)
    XOR  $FF
    LD   (.sm_ignore_keys + 1),A
    ; Shift screen to the right
    LD   A,(.sm_screen_hcenter + 1)
    CP   -3
    JP   Z,.end_key_check
    DEC  A
    LD   (.sm_screen_hcenter + 1),A
    JP   .end_key_check

.test_fire_2
    PUSH AF                         ; Save key state
    BIT  JOY_FIRE2,A
    JP   NZ,.pretest_up
    ; Update the level string position
    LD   IY,LEVEL_CHAR
    EX   AF,AF'
    OR   A
    JR   Z,.end_update_iy
    INC  IY
    CP   1
    JR   Z,.end_update_iy
    INC  IY
.end_update_iy
    EX   AF,AF'

    LD   HL,fkeys_buffer
.test_line_0
    LD   A,(HL)
    LD   (HL),$FF
    INC  HL
    CPL
    AND  %00111000
    JR   Z,.test_line_1
.test_f3
    BIT  KEY_F3,A
    JR   Z,.test_f6
.is_f3
    LD   (IY + 0),3
    JR   .inc_aa
.test_f6
    BIT  KEY_F6,A
    JR   Z,.test_f9
.is_f6
    LD   (IY + 0),6
    JR   .inc_aa
.test_f9
    BIT  KEY_F9,A
    JR   Z,.test_line_1
.is_f9
    LD   (IY + 0),9
    JR   .inc_aa

.test_line_1
    LD   A,(HL)
    LD   (HL),$FF
    INC  HL
    CPL
    AND  %11111100
    JR   Z,.test_line_2

.test_f0
    RLA
    JR   NC,.test_f2
.is_f0
    LD   (IY + 0),0
    JR   .inc_aa
.test_f2
    RLA
    JR   NC,.test_f1
.is_f2
    LD   (IY + 0),2
    JR   .inc_aa
.test_f1
    RLA
    JR   NC,.test_f5
.is_f1
    LD   (IY + 0),1
    JR   .inc_aa
.test_f5
    RLA
    JR   NC,.test_f8
.is_f5
    LD   (IY + 0),5
    JR   .inc_aa
.test_f8
    RLA
    JR   NC,.test_f7
.is_f8
    LD   (IY + 0),8
    JR   .inc_aa
.test_f7
    RLA
    JR   NC,.test_line_2
.is_f7
    LD   (IY + 0),7
    JR   .inc_aa

.test_line_2
    LD   A,(HL)
    LD   (HL),$FF
    BIT  KEY_F4,A
    JR   NZ,.test_up
.is_f4
    LD   (IY + 0),4

.inc_aa
    POP  AF             ; Skip key state
    EX   AF,AF'
    CP   3
    JR   Z,.end_inc_aa
    INC  A
.end_inc_aa
    EX   AF,AF'

    LD   A,2
    LD   (.pretest_up + 1),A
    
    ; Set ignore keyboard
    LD   A,(.sm_ignore_keys + 1)
    XOR  $FF
    LD   (.sm_ignore_keys + 1),A

    JP   .end_key_check

.pretest_up
    JR   .enable_update_level_number
    JR   .test_up
.enable_update_level_number
    PUSH AF
    XOR  A
    LD   (.pretest_up + 1),A
    LD   (.sm_skip_loop + 1),A
    CALL dec2bin                    ; Convert level number to binary
    POP  AF

    ; Cursor Up/Joy Up
.test_up
    POP  AF                         ; Restore key state
    BIT  JOY_UP,A
    JR   NZ,.test_down
    ; Set ignore keyboard
    LD   A,(.sm_ignore_keys + 1)
    XOR  $FF
    LD   (.sm_ignore_keys + 1),A
    LD   A,(.sm_cursor_position + 1)
    LD   (.sm_old_cursor_position + 1),A
    CP   1
    JP   Z,.end_key_check
    DEC  A
    LD   (.sm_cursor_position + 1),A
    JP   .end_key_check

.test_down
    BIT  JOY_DOWN,A
    JR   NZ,.test_left
    ; Set ignore keyboard
    LD   A,(.sm_ignore_keys + 1)
    XOR  $FF
    LD   (.sm_ignore_keys + 1),A
    LD   A,(.sm_cursor_position + 1)
    LD   (.sm_old_cursor_position + 1),A
    CP   5
    JP   Z,.end_key_check
    INC  A
    LD   (.sm_cursor_position + 1),A
    JP   .end_key_check

.test_left
    BIT  JOY_LEFT,A
    JR   NZ,.test_right
    ; Set ignore keyboard
    LD   A,(.sm_ignore_keys + 1)
    XOR  $FF
    LD   (.sm_ignore_keys + 1),A

    LD   A,(DIP_TMP)
    LD   B,A
    LD   A,(.sm_cursor_position + 1)
.dec_coins
    DEC  A
    JR   NZ,.dec_lives
    LD   A,B
    AND  %00000011
    JR   Z,.end_dec_options
    DEC  A
    LD   C,A
    LD   A,B
    AND  %11111100
    OR   C
    LD   (DIP_TMP),A
    JR   .end_dec_options

.dec_lives
    DEC  A
    JR   NZ,.dec_bonus
    LD   A,B
    AND  %00001100
    JR   Z,.end_dec_options
    SUB  4
    LD   C,A
    LD   A,B
    AND  %11110011
    OR   C
    LD   (DIP_TMP),A
    JR   .end_dec_options

.dec_bonus
    DEC  A
    JR   NZ,.dec_difficult
    LD   A,B
    AND  %00110000
    JR   Z,.end_dec_options
    SUB  16
    LD   C,A
    LD   A,B
    AND  %11001111
    OR   C
    LD   (DIP_TMP),A
    JR   .end_dec_options

.dec_difficult
    DEC  A
    JR   NZ,.dec_names
    LD   A,B
    SET  6,A
    LD   (DIP_TMP),A
    JR   .end_dec_options

.dec_names
    LD   A,B
    SET  7,A
    LD   (DIP_TMP),A

.end_dec_options
    JP   .end_key_check

.test_right
    BIT  JOY_RIGHT,A
    JR   NZ,.test_return
    ; Set ignore keyboard
    LD   A,(.sm_ignore_keys + 1)
    XOR  $FF
    LD   (.sm_ignore_keys + 1),A

    LD   A,(DIP_TMP)
    LD   B,A
    LD   A,(.sm_cursor_position + 1)
.inc_coins
    DEC  A
    JR   NZ,.inc_lives
    LD   A,B
    AND  %00000011
    CP   %00000011
    JR   Z,.end_inc_options
    INC  A
    LD   C,A
    LD   A,B
    AND  %11111100
    OR   C
    LD   (DIP_TMP),A
    JR   .end_inc_options

.inc_lives
    DEC  A
    JR   NZ,.inc_bonus
    LD   A,B
    AND  %00001100
    CP   %00001100
    JR   Z,.end_inc_options
    ADD  4
    LD   C,A
    LD   A,B
    AND  %11110011
    OR   C
    LD   (DIP_TMP),A
    JR   .end_inc_options

.inc_bonus
    DEC  A
    JR   NZ,.inc_difficult
    LD   A,B
    AND  %00110000
    CP   %00110000
    JR   Z,.end_inc_options
    ADD  16
    LD   C,A
    LD   A,B
    AND  %11001111
    OR   C
    LD   (DIP_TMP),A
    JR   .end_inc_options

.inc_difficult
    DEC  A
    JR   NZ,.inc_names
    LD   A,B
    RES  6,A
    LD   (DIP_TMP),A
    JR   .end_inc_options

.inc_names
    LD   A,B
    RES  7,A
    LD   (DIP_TMP),A

.end_inc_options
    JP   .end_key_check

.test_return
    BIT  KEY_DEL,A
    JR   NZ,.end_key_check
    ; Set ignore keyboard
    LD   A,(.sm_ignore_keys + 1)
    XOR  $FF
    LD   (.sm_ignore_keys + 1),A

    ; Toggle PAL/NTSC
    LD   A,(.sm_vertical_total + 1)
    CP   VERTICAL_TOTAL_PAL
    JR   NZ,.change_pal
.change_ntsc
    LD   A,VERTICAL_TOTAL_NTSC
    LD   (.sm_vertical_total + 1),A
    LD   A,VERTICAL_ADJUST_NTSC
    LD   (.sm_vertical_adjust + 1),A
    LD   A,VERTICAL_OFFSET_NTSC
    LD   (.sm_vertical_offset + 1),A
    LD   A,$00
    LD   (.sm_wait_upper_sync + 1),A
    LD   (.sm_wait_lower_sync + 1),A
    JP   .end_key_check
.change_pal
    LD   A,VERTICAL_TOTAL_PAL
    LD   (.sm_vertical_total + 1),A
    LD   A,VERTICAL_ADJUST_PAL
    LD   (.sm_vertical_adjust + 1),A
    LD   A,VERTICAL_OFFSET_PAL
    LD   (.sm_vertical_offset + 1),A
    LD   A,$09
    LD   (.sm_wait_upper_sync + 1),A
    LD   (.sm_wait_lower_sync + 1),A
.end_key_check

.sm_wait_lower_sync
    JR  .wait_lower_pal
.wait_lower_ntsc
    WAIT_SCANLINES 25
    JR  .begin_lower_rasters
.wait_lower_pal
    HALT
.begin_lower_rasters
    WAIT_SCANLINES 17
    SET_PEN 2,AZUL_BRILLANTE            ; Foot
    WAIT_SCANLINES 16
    DEFS 6
    SET_PEN 1,AZUL                      ; Fade Screen
    SET_PEN 2,AMARILLO                  ; Fade Screen
    SET_PEN 3,ROJO                      ; Fade Screen

    ; Update Cursor
.update_cursor
.sm_cursor_position
    LD   A,1
.sm_old_cursor_position
    CP   2
    JR   Z,.skip_update_cursor

    CALL print_cursor
    
    LD   A,(.sm_old_cursor_position + 1)
    CALL erase_cursor

.skip_update_cursor
    LD   A,(.sm_cursor_position + 1)
    LD   (.sm_old_cursor_position + 1),A

    ; Update Options
    DEC  A
    ADD  A,A
    LD   HL,decode_options_list
    ADD  A,L
    JR   NC,.no_inc_h
    INC  H
.no_inc_h
    LD   L,A
    LD   A,(HL)
    INC  HL
    LD   H,(HL)
    LD   L,A
    LD   (.sm_decode_call + 1),HL

    ; Print option update
    LD   A,(DIP_TMP)                    ; Read DIP
.sm_decode_call
    CALL decode_coin_game               ; decode_coin_game, decode_lives, decode_bonus_life
                                        ; decode_difficult, decode_ghost_names


.sm_skip_loop
    JR   .make_loop
    CALL reupdate_level_number
    LD   A,3+2+3
    LD   (.sm_skip_loop + 1),A
.make_loop
    JP   control_menu

; ---------------------------------------------------------------------------
; Read keyboard and joystick
; RESULT:
;    A : %s3chdrlu (SPACE, TAB, COPY, SHIFT + Right/Left, Down, Right, Left, Up)
; ---------------------------------------------------------------------------
read_controls
    LD   HL,fkeys_buffer                        ; Storage functions keys

    ; Select the PSG IOPORT_A
    LD   BC,PPI_A + PSG_IOPORT_A
    OUT  (C),C

    ; Operation Select PSG Register and Joystick 0 key Line 
    LD   BC,PPI_C + PPI_PSG_SELECT              ; + JOY0 <--- Puto ERROR!!! xDDD
    OUT  (C),C

    ; PSG Inactive (Needed in CPC+, because the PPI emulation do not automatically
    ; resets the Ports A, B and C to $00, when PPI Control changes (bit 7 set)
    DEFB $ED,$71                                ; OUT (C),0 

    ; Set the PPI_A in input mode
    LD   BC,PPI_CONTROL + PPI_A_INPUT
    OUT  (C),C

    ; Read a byte from the joyport
    LD   BC,PPI_C + PPI_PSG_READ + JOY0
    OUT  (C),C
    LD   B,>PPI_A
    IN   A,(C)                                  ; A = byte received
    SET  JOY_FIRE2,A                            ; Disable FIRE 2 button

.read_cursor_keys
    ; Check Cursors Up, Down and Right
    LD   BC,PPI_C + PPI_PSG_READ + KEY_LINE0
    OUT  (C),C                  
    LD   B,>PPI_A
    IN   B,(C)                                  ; B = byte received
    LD   (HL),B                                 ; Save Key line 0
    INC  HL
.cursor_up
    BIT  CURSOR_UP,B
    JR   NZ,.cursor_right
    RES  JOY_UP,A
.cursor_right
    BIT  CURSOR_RIGHT,B
    JR   NZ,.cursor_down
    RES  JOY_RIGHT,A
.cursor_down
    BIT  CURSOR_DOWN,B
    JR   NZ,.cursor_left
    RES  JOY_DOWN,A
.cursor_left

    ; Check Cursor Left
    LD   BC,PPI_C + PPI_PSG_READ + KEY_LINE1
    OUT  (C),C                  
    LD   B,>PPI_A
    IN   B,(C)                                  ; B = byte received
    LD   (HL),B                                 ; Save Key line 1
    INC  HL

    BIT  CURSOR_LEFT,B
    JR   NZ,.check_copy
    RES  JOY_LEFT,A
.check_copy
    BIT  KEY_COPY,B
    JR   NZ,.check_shift
    RES  JOY_FIRE2,A
.check_shift

    ; Check SHIFT
    LD   BC,PPI_C + PPI_PSG_READ + KEY_LINE2
    OUT  (C),C                  
    LD   B,>PPI_A
    IN   B,(C)                                  ; B = byte received
    LD   (HL),B                                 ; Save Key line 2
    INC  HL

    BIT  KEY_SHIFT,B
    JR   NZ,.check_space
    RES  JOY_FIRE1,A
    RES  JOY_FIRE2,A

.check_space
    LD   BC,PPI_C + PPI_PSG_READ + KEY_LINE5
    OUT  (C),C                  
    LD   B,>PPI_A
    IN   B,(C)                                  ; B = byte received
    BIT  KEY_SPACE,B                            ; NTSC/PAL
    JR   NZ,.check_tab
    RES  KEY_DEL,A
    
.check_tab
    LD   BC,PPI_C + PPI_PSG_READ + KEY_LINE8
    OUT  (C),C                  
    LD   B,>PPI_A
    IN   B,(C)                                  ; B = byte received
    BIT  KEY_TAB,B
    JR   NZ,.check_esc
    RES  JOY_FIRE3,A
.check_esc
    BIT  KEY_ESC,B
    JR   NZ,.end_check_keys
    RES  JOY_FIRE3,A
    LD   HL,OLD_DIP
    LD   DE,DIP_TMP
    LDI
.end_check_keys

    ; Set the PPI_A in output mode
    LD   BC,PPI_CONTROL + PPI_A_OUTPUT
    OUT  (C),C

    ; PSG Inactive (Needed in CPC+)
    DEC  B                                  ; B = PPI_C
    DEFB $ED,$71                            ; OUT (C),0 
    
    RET

fkeys_buffer
    DEFS 3

; ---------------------------------------------------------------------------
; Test if any key is pressed, except SHIFT & COPY. And marking in Carry flag.
; ---------------------------------------------------------------------------
anykey_press
    INIT_PSG_FOR_SCANKEYS
    LD   D,%01000000                        ; Key line 0 - 9
.loop_anykey_press
    LD   B,>PPI_C
    OUT  (C),D                              ; Send Key line
    LD   B,>PPI_A
    IN   C,(C)                              ; Read Key line
    LD   A,D
.test_copy_key
    CP   %01000001                          ; Is the COPY line?
    JR   NZ,.test_shift_key
.skip_copy_key
    SET  KEY_COPY,C
    JR   .no_more_special_keys
.test_shift_key
    CP   %01000010                          ; Is the SHIFT line?
    JR   NZ,.test_joy0_fire1
.skip_shift_key
    SET  KEY_SHIFT,C
    JR   .no_more_special_keys
.test_joy0_fire1
    CP   %01001001                          ; Is the JOY0 line?
    JR   NZ,.no_more_special_keys
.skip_joy0_fire1
    SET  JOY_FIRE1,C
.no_more_special_keys
    LD   A,C
    CPL
    AND  A
    JR   NZ,.key_press
    INC  D
    LD   A,%01001010
    CP   D
    JR   NZ,.loop_anykey_press
    JR   .no_key_press
.key_press
    SCF
    JR   .end_anykey_press
.no_key_press
    AND  A
.end_anykey_press
    RESTORE_PSG_OF_SCANKEYS
    RET

; ---------------------------------------------------------------------------
; Wait until no key is pressed
; ---------------------------------------------------------------------------
no_key_press
    CALL anykey_press
    JR   C,no_key_press
    RET

; ---------------------------------------------------------------------------
; Wait vblank
wait_vblank
    WAIT_VBL
    RET

; ---------------------------------------------------------------------------
; Wait x scanlines
; Use:
;    WAIT_SCANLINES x                       ; (9)
; ---------------------------------------------------------------------------
wait_scanlines_iy
    DEFS 5,0                                ; (5) 

.loop_wait_scanlines_iy                      
    DEFS 6                                  ; (6)
    DEC  IY                                 ; (3)
    LD   A,IYH                              ; (2)
    OR   IYL                                ; (2)
    JR   NZ,.loop_wait_scanlines_iy         ; (2/3)
                                            ; Total loop --> 16 * (IY - 1) + 15
    RET                                     ; (3)
                                            ; Total Routine --> 64 * SCANLINES

; ---------------------------------------------------------------------------
; C = VRAM page
show_vram
    ; Wait vblank
    CALL wait_vblank

    ;Set VRAM page
    LD   A,C
    LD   BC,CRTC_SELECT + REG_0C
    OUT  (C),C
    INC  B
    OUT  (C),A
    RET

; ---------------------------------------------------------------------------
; Set palette
; HL : end_palette - 1
;  A : end_palette - palette - 1
set_palette
    LD   B,>GATE_ARRAY
.loop_set_palette
    OUT  (C),A
    INC  B
    OUTD
    DEC  A
    JP   P,.loop_set_palette
    RET

; ---------------------------------------------------------------------------
; Save VRAM covered by the settings menu
; ---------------------------------------------------------------------------
save_vram
    LD   HL,$C000 + 64 * FIRST_ROW 
    LD   DE,VRAM_BUFFER
    LD   A,8
.loop_copy_vram
    LD   BC,64 * HEIGHT_MENU
    LDIR
    LD   BC,2048 - 64 * HEIGHT_MENU
    ADD  HL,BC
    
    DEC  A
    JR   NZ,.loop_copy_vram

    RET

; ---------------------------------------------------------------------------
; Restore VRAM was covered by the settings menu
; ---------------------------------------------------------------------------
restore_vram
    LD   DE,$C000 + 64 * FIRST_ROW 
    LD   HL,VRAM_BUFFER
    LD   A,8
.loop_restore_vram
    LD   BC,64 * HEIGHT_MENU
    LDIR
    EX   DE,HL
    LD   BC,2048 - 64 * HEIGHT_MENU
    ADD  HL,BC
    EX   DE,HL
    
    DEC  A
    JR   NZ,.loop_restore_vram

    RET

; --------------------------------------------------------------------------- 
; Shut the PSG
;    HL : psg_sound_list
; --------------------------------------------------------------------------- 
shut_psg
    ; Set the PPI_A in output mode
    LD   BC,PPI_CONTROL + PPI_A_OUTPUT
    OUT  (C),C

    ; PSG Inactive (Needed in CPC+)
    DEC  B                                  ; B = PPI_C
    DEFB $ED,$71                            ; OUT (C),0 

    ; Fill with zeros the psg_sound_list
    LD   B,9
.loop_put_zeros
    LD   (HL),0
    INC  HL
    DJNZ .loop_put_zeros

    LD   BC,psg_diff
    ADD  HL,BC

    JP   (HL)                               ; play_sound

; ---------------------------------------------------------------------------
; Print main menu
print_main_menu
    LD   IY,main_menu                       ; IY = Menu tilemap
    CALL print_string
    
    ; Fix down corners
    LD   HL,extra_tile_left
    LD   DE,$C000 + 64 * (FIRST_ROW + HEIGHT_MENU - 2) + 2
    CALL draw_tile

    LD   HL,extra_tile_right
    LD   DE,$C000 + 64 * (FIRST_ROW + HEIGHT_MENU - 2) + 2 * 30
    CALL draw_tile
    
    RET

; ---------------------------------------------------------------------------
;  A: Cursor Position
erase_cursor
    LD   DE,$84E0                           ; Blank tile
    JR   print_cursor_next
print_cursor
    LD   DE,cursor_tile
    
print_cursor_next    
    LD   HL,$C000 + 64 * (FIRST_ROW + 1) + 2 * 3
    LD   BC,64*2
.loop_print_cursor
    ADD  HL,BC
    DEC  A
    JR   NZ,.loop_print_cursor

    EX   DE,HL
    JP   draw_tile

; ---------------------------------------------------------------------------
; Update level number in the menu
; ---------------------------------------------------------------------------
reupdate_level_number
    CALL update_level_number
    LD   A,(level_string)                       ; A  = Tile number
    LD   DE,$C000 + 64 * (FIRST_ROW + 13) + 34  ; DE = Screen address
    CALL draw_tilenumber

    LD   A,(level_string + 1)                   ; A  = Tile number
    LD   DE,$C000 + 64 * (FIRST_ROW + 13) + 36  ; DE = Screen address
    CALL draw_tilenumber

    LD   A,(level_string + 2)                   ; A  = Tile number
    LD   DE,$C000 + 64 * (FIRST_ROW + 13) + 38  ; DE = Screen address
    JP   draw_tilenumber

; ---------------------------------------------------------------------------
; Update level number in the menu
; ---------------------------------------------------------------------------
update_level_number
    LD   A,(LEVEL_TMP)
    LD   H,0
    LD   L,A
    INC  HL                                     ; Fix Nerd level number :P
    LD   DE,level_string
    CALL bin2dec
    LD   A,(level_string)
    CP   T_0
    RET  NZ
    LD   A,T_SPC
    LD   (level_string),A
    RET

; ---------------------------------------------------------------------------
; Convert from binary to ascii decimal string (Baze)
; ENTRY:
;    HL : Value to convert
;    DE : Pointer to the string
; ---------------------------------------------------------------------------
bin2dec
.convierte_centenas
    LD   BC,-100
    CALL .bin2dec1
.convierte_decenas
    LD   C,-10
    CALL .bin2dec1
.convierte_unidades
    LD   C,B
.bin2dec1
    LD   A,T_0 - 1
.bin2dec2
    INC  A
    ADD  HL,BC
    JR   C,.bin2dec2
    SBC  HL,BC
    LD   (DE),A
    INC  DE
    RET

; ---------------------------------------------------------------------------
; Convert a string with the level number to binary
; ---------------------------------------------------------------------------
dec2bin
    LD   IY,LEVEL_CHAR
    LD   DE,0
    EX   AF,AF'
    LD   B,A
    XOR  A
    EX   AF,AF'
    DEC  B
    JR   NZ,.is_2_or_3
.is_1
    LD   C,(IY + 0)
    JR   .end_read_string
.is_2_or_3
    DEC  B
    JR   NZ,.is_3
.is_2
    LD   E,(IY + 0)
    LD   C,(IY + 1)
    JR   .end_read_string
.is_3
    LD   D,(IY + 0)
    LD   E,(IY + 1)
    LD   C,(IY + 2)
.end_read_string
    LD   (IY + 0),0
    LD   (IY + 1),0
    LD   (IY + 2),0

.decode_centenas
    LD   B,0
    LD   A,D
    OR   A
    JR   Z,.decode_unidades
.is_100_or_bigger
    DEC  D
    JR   NZ,.is_200_or_bigger
.is_100
    LD   B,100
    JR   .decode_unidades
.is_200_or_bigger
    DEC  D
    JR   Z,.is_200
.is_bigger
    LD   B,0
    JR   .end_decode
.is_200
    LD   B,200
.decode_unidades
    LD   A,C
    ADD  A,B
    LD   B,A
.decode_decenas
    LD   A,E
    OR   A
    JR   Z,.end_decode
    LD   A,B
.loop_decenas
    ADD  A,10
    JR   C,.is_bigger
    DEC  E
    JR   NZ,.loop_decenas
    JR   .save_level_update
.end_decode
    LD   A,B
.save_level_update
    DEC  A                          ; More fix Nerd level number :P
    LD  (LEVEL_TMP),A

    RET

; ---------------------------------------------------------------------------
; Print a string finished in 255
; IY : String
print_string
    LD   E,(IY + 0)
    INC  IY
    LD   D,(IY + 0)                         ; Screen address
    INC  IY
.loop_print_string
    LD   A,(IY + 0)                         ; Get Tile number
    CP   255
    RET  Z                                  ; Finish???
    
    INC  IY                                 ; Tilemap++

    ; Print tile
    PUSH DE
    CALL draw_tilenumber
    POP  DE
    
    ; Next screen address
    INC  DE
    INC  DE
    
    JR   .loop_print_string

; --------------------------------------------------------------------------- 
;  A : Tile number
; DE : Screen address
draw_tilenumber
    ; Get the tile address ($8000 + tile_number << 4) 
    LD   H,%00001000
    SLA  A
    RL   H
    RLA
    RL   H
    RLA
    RL   H
    RLA
    RL   H
    LD   L,A                ; HL hold tile address

; HL : Tile address
; DE : Screen address
draw_tile
    DI
    LD   (.sm_old_stack + 1),SP
    LD   SP,HL
    EX   DE,HL

    ; 1º Scanline
    POP  DE
    LD   (HL),E
    INC  L
    LD   (HL),D
    SET  3,H
    ; 2º Scanline
    POP  DE
    LD   (HL),D
    DEC  L
    LD   (HL),E
    SET  4,H
    ; 4º Scanline
    POP  DE
    LD   (HL),E
    INC  L
    LD   (HL),D
    RES  3,H
    ; 3º Scanline
    POP  DE
    LD   (HL),D
    DEC  L
    LD   (HL),E
    SET  5,H
    ; 7º Scanline
    POP  DE
    LD   (HL),E
    INC  L
    LD   (HL),D
    RES  4,H
    ; 5º Scanline
    POP  DE
    LD   (HL),D
    DEC  L
    LD   (HL),E
    SET  3,H
    ; 6º Scanline
    POP  DE
    LD   (HL),E
    INC  L
    LD   (HL),D
    SET  4,H
    ; 8º Scanline
    POP  DE
    LD   (HL),D
    DEC  L
    LD   (HL),E

.sm_old_stack
    LD   SP,$0000
    EI
    RET

; ---------------------------------------------------------------------------
; Decode options and print
decode_and_print_options
    LD   A,(DIP_TMP)                    ; Read DIP
    LD   I,A
    CALL decode_coin_game

    LD   A,I
    CALL decode_lives

    LD   A,I
    CALL decode_bonus_life

    LD   A,I
    CALL decode_difficult

    LD   A,I
    CALL decode_ghost_names

    RET

; ---------------------------------------------------------------------------
; Pac-Man DIPs config (0 Enable / 1 Disable)
; (%gdbbllcc)
; g=alternate (0) / normal (1) ghost names
; d=hard (0) / normal (1) 
; bb=bonus life at 10K (0) / 15K (1) / 20K (2) / none (3) 
; ll=1 (0) / 2 (1) / 3 (2) / 5 (3) lives 
; cc=freeplay (0) / 1coin1credit(1) / 1coin2credits(2) / 2coins1credit(3)
; ---------------------------------------------------------------------------
; %------cc ==> cc = freeplay (0) | 1coin/1credit (1) | 1coin/2credits (2) | 2coins/1credit (3)
decode_coin_game
    AND  %00000011
    JR   NZ,.is_1_1
.is_free_play
    LD   IY,coins_free
    JP   print_string
.is_1_1
    CP   1
    JR   NZ,.is_1_2
    LD   IY,coins_1_1
    JP   print_string
.is_1_2
    CP   2
    JR   NZ,.is_2_1
    LD   IY,coins_1_2
    JP   print_string
.is_2_1
    LD   IY,coins_2_1
    JP   print_string

; ---------------------------------------------------------------------------
; %----ll-- ==> ll = 1 life (0) | 2 lives (1) | 3 lives (2) | 5 lives (3)
decode_lives
    AND  %00001100
    JR   NZ,.is_2_lives
.is_1_life
    LD   IY,lives_1
    JP   print_string
.is_2_lives
    CP   1<<2
    JR   NZ,.is_3_lives
    LD   IY,lives_2
    JP   print_string
.is_3_lives
    CP   2<<2
    JR   NZ,.is_5_lives
    LD   IY,lives_3
    JP   print_string
.is_5_lives
    LD   IY,lives_5
    JP   print_string

; ---------------------------------------------------------------------------
; %--bb---- ==> bb = Bonus life at 10K (0) | 15K (1) | 20K (2) | None (3) 
decode_bonus_life
    AND  %00110000
    JR   NZ,.is_15000
.is_10000
    LD   IY,bonus_10000
    JP   print_string
.is_15000
    CP   1<<4
    JR   NZ,.is_20000
    LD   IY,bonus_15000
    JP   print_string
.is_20000
    CP   2<<4
    JR   NZ,.is_00000
    LD   IY,bonus_20000
    JP   print_string
.is_00000
    LD   IY,bonus_00000
    JP   print_string


; ---------------------------------------------------------------------------
; %-d------ ==> d = hard (0) | normal (1) 
decode_difficult
    AND  %01000000
    JR   NZ,.is_normal
.is_difficult
    LD   IY,hard_game
    JP   print_string
.is_normal
    LD   IY,normal_game
    JP   print_string

; ---------------------------------------------------------------------------
; %g------- ==> g = Alternate ghost names (0) | Normal ghost names (1)
decode_ghost_names
    RLCA
    JR   C,.is_ghost
.is_friki
    LD   IY,friki_name
    JP   print_string
.is_ghost
    LD   IY,ghost_name
    JP   print_string

; ---------------------------------------------------------------------------
menu_palette
    DEFB INK + NEGRO,INK + AZUL, INK + AMARILLO,INK + ROJO
end_menu_palette 

game_palette
    DEFB INK + NEGRO,INK + AZUL_BRILLANTE, INK + AMARILLO_BRILLANTE,INK + ROJO_BRILLANTE
end_game_palette 

; ---------------------------------------------------------------------------
; |   COINS-GAME    FREEPLAY   | 1-1|1-2|2-1|FREEPLAY
; |   LIVES         1          | 1/2/3/5
; |   BONUS LIFE    00000      | 10000/15000/20000/00000
; |   DIFFICULTY    NORMAL     | NORMAL/HARD
; |   GHOST NAMES   ALTERNATE  | NORMAL/ALTERNATE
; ---------------------------------------------------------------------------
main_menu
    DEFW $C000 + 64 * FIRST_ROW         ; Screen address
    ; 1º
    BLANK_LINE

    ; 2º
    DEFB T_SPC,T_LB_UL
    DEFS 8,T_LB_ED ;U
    DEFB T_LB_CLU,T_SPC,T_S,T_E,T_T,T_T,T_I,T_N,T_G,T_S,T_SPC,T_LB_CRU
    DEFS 8,T_LB_ED ;U
    DEFB T_LB_UR,T_SPC

    ; 3º
    MENU_BLANK_LINE

    ; 4º COINS-GAMES  1-1|1-2|2-1|FREE PLAY
    DEFB T_SPC,T_LB_VR,T_SPC,T_SPC,T_SPC
    DEFB T_C,T_O,T_I,T_N,T_S,T_SEP,T_G,T_A,T_M,T_E,T_S
    DEFS 14,T_SPC
    DEFB T_LB_VL,T_SPC
    
    ; 5º
    MENU_BLANK_LINE

    ; 6º LIVES       1/2/3/5
    DEFB T_SPC,T_LB_VR,T_SPC,T_SPC,T_SPC
    DEFB T_L,T_I,T_V,T_E,T_S
    DEFS 20,T_SPC
    DEFB T_LB_VL,T_SPC

    ; 7º
    MENU_BLANK_LINE

    ; 8º BONUS LIFE  10000/15000/20000/None
    DEFB T_SPC,T_LB_VR
    DEFS 3,T_SPC
    DEFB T_B,T_O,T_N,T_U,T_S,T_SPC,T_L,T_I,T_F,T_E
    DEFS 15,T_SPC
    DEFB T_LB_VL,T_SPC
   
    ; 9º
    MENU_BLANK_LINE

    ; 10º DIFFICULTY  NORMAL/HARD
    DEFB T_SPC,T_LB_VR
    DEFS 3,T_SPC
    DEFB T_D,T_I,T_F,T_F,T_I,T_C,T_U,T_L,T_T,T_Y
    DEFS 15,T_SPC
    DEFB T_LB_VL,T_SPC

    ; 11º
    MENU_BLANK_LINE

    ; 12º GHOSTS NAMES ARCADE/CUSTOM
    DEFB T_SPC,T_LB_VR
    DEFS 3,T_SPC
    DEFB T_G,T_H,T_O,T_S,T_T,T_S,T_SPC,T_N,T_A,T_M,T_E,T_S 
    DEFS 13,T_SPC
    DEFB T_LB_VL,T_SPC
    
    ; 13º
    MENU_BLANK_LINE
    
    ; 14º
    DEFB T_SPC
    DEFB T_SPC
    DEFS 8,T_LB_ED
    DEFB T_LB_CLD
    DEFB T_SPC,T_L,T_E,T_V,T_E,T_L
level_string
    DEFB T_X,T_X,T_X
    DEFB T_SPC
    DEFB T_LB_CRD
    DEFS 8,T_LB_ED
    DEFB T_SPC
    DEFB T_SPC

    ; 15º
    BLANK_LINE

    DEFB 255

; ---------------------------------------------------------------------------
; Menu options
; COINS-GAMES  1-1|1-2|2-1|FREE PLAY
coins_1_1
    DEFW $C000 + 64 * (FIRST_ROW + 3) + 19 * 2  ; Screen address
    DEFB T_1,T_SEP,T_1
    DEFS 6,T_SPC
    DEFB 255

coins_1_2
    DEFW $C000 + 64 * (FIRST_ROW + 3) + 19 * 2  ; Screen address
    DEFB T_1,T_SEP,T_2
    DEFS 6,T_SPC
    DEFB 255

coins_2_1
    DEFW $C000 + 64 * (FIRST_ROW + 3) + 19 * 2  ; Screen address
    DEFB T_2,T_SEP,T_1
    DEFS 6,T_SPC
    DEFB 255
    
coins_free
    DEFW $C000 + 64 * (FIRST_ROW + 3) + 19 * 2  ; Screen address
    DEFB T_F,T_R,T_E,T_E,T_SPC,T_P,T_L,T_A,T_Y
    DEFB 255

; LIVES       1/2/3/5
lives_1
    DEFW $C000 + 64 * (FIRST_ROW + 5) + 19 * 2  ; Screen address
    DEFB T_1
    DEFB 255

lives_2
    DEFW $C000 + 64 * (FIRST_ROW + 5) + 19 * 2  ; Screen address
    DEFB T_2
    DEFB 255

lives_3
    DEFW $C000 + 64 * (FIRST_ROW + 5) + 19 * 2  ; Screen address
    DEFB T_3
    DEFB 255

lives_5
    DEFW $C000 + 64 * (FIRST_ROW + 5) + 19 * 2  ; Screen address
    DEFB T_5
    DEFB 255

; BONUS LIFE  10000/15000/20000/None
bonus_10000
    DEFW $C000 + 64 * (FIRST_ROW + 7) + 19 * 2  ; Screen address
    DEFB T_1,T_0,T_0,T_0,T_0
    DEFB 255

bonus_15000
    DEFW $C000 + 64 * (FIRST_ROW + 7) + 19 * 2  ; Screen address
    DEFB T_1,T_5,T_0,T_0,T_0
    DEFB 255

bonus_20000
    DEFW $C000 + 64 * (FIRST_ROW + 7) + 19 * 2  ; Screen address
    DEFB T_2,T_0,T_0,T_0,T_0
    DEFB 255

bonus_00000
    DEFW $C000 + 64 * (FIRST_ROW + 7) + 19 * 2  ; Screen address
    DEFB T_N,T_O,T_N,T_E,T_SPC
    DEFB 255

; DIFFICULTY  NORMAL/HARD
normal_game
    DEFW $C000 + 64 * (FIRST_ROW + 9) + 19 * 2  ; Screen address
    DEFB T_N,T_O,T_R,T_M,T_A,T_L
    DEFB 255
    
hard_game
    DEFW $C000 + 64 * (FIRST_ROW + 9) + 19 * 2  ; Screen address
    DEFB T_H,T_A,T_R,T_D,T_SPC,T_SPC
    DEFB 255

; GHOST NAMES ARCADE/CUSTOM
ghost_name
    DEFW $C000 + 64 * (FIRST_ROW + 11) + 19 * 2  ; Screen address
    DEFB T_O,T_R,T_I,T_G,T_I,T_N,T_A,T_L,T_SPC
    DEFB 255

friki_name
    DEFW $C000 + 64 * (FIRST_ROW + 11) + 19 * 2  ; Screen address
    DEFB T_A,T_L,T_T,T_E,T_R,T_N,T_A,T_T,T_E
    DEFB 255


; ---------------------------------------------------------------------------
; Extra tiles needed for the down corners
extra_tile_left
    DEFB $00,$90,$00,$90,$00,$90,$00,$90    ; 1,2,4,3
    DEFB $00,$40,$00,$80,$00,$80,$00,$30    ; 7,5,6,8

extra_tile_right
    DEFB $90,$00,$90,$00,$90,$00,$90,$00    ; 1,2,4,3
    DEFB $20,$00,$10,$00,$10,$00,$C0,$00    ; 7,5,6,8

cursor_tile
    DEFB $33,$CC,$77,$EE,$EE,$00,$FF,$88    ; 1,2,4,3
    DEFB $33,$CC,$FF,$88,$77,$EE,$00,$00    ; 7,5,6,8

; ---------------------------------------------------------------------------
decode_options_list
    DEFW decode_coin_game,decode_lives,decode_bonus_life,decode_difficult,decode_ghost_names

; ---------------------------------------------------------------------------
; Copy of arcade RAM variables (DIP_5080 and PAC_LEVEL_NUMBER)
DIP_TMP
    DEFB 0

OLD_DIP
    DEFB 0

LEVEL_TMP
    DEFB 0

OLD_LEVEL
    DEFB 0

LEVEL_CHAR
    DEFB 0,0,0

; ---------------------------------------------------------------------------
    END
