/*
This file is part of VoidHawk.

VoidHawk is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

VoidHawk is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with VoidHawk.  If not, see <http://www.gnu.org/licenses/>.
*/
typedef struct
{
    int x;
    int y;
} Vector2D;

void Vector2D_copy(Vector2D* b, Vector2D* a)
{
    b->x = a->x;
    b->y = a->y;
}

void Vector2D_clamp(Vector2D* a, int minX, int maxX, int minY, int maxY)
{
	a->x = clamp(a->x, minX, maxX);
	a->y = clamp(a->y, minY, maxY);
}

void Vector2D_add(Vector2D* c, Vector2D* a, Vector2D* b)
{
    c->x = a->x + b->x;
    c->y = a->y + b->y;
}

void Vector2D_sub(Vector2D* c, Vector2D* a, Vector2D* b)
{
    c->x = a->x - b->x;
    c->y = a->y - b->y;
}

void Vector2D_mul_int(Vector2D* c, Vector2D* a, int b)
{
	c->x = a->x * b;
	c->y = a->y * b;
}

void Vector2D_div_int(Vector2D* c, Vector2D* a, int b)
{
	if (b > 0)
	{
		c->x = a->x / b;
		c->y = a->y / b;
	}
	else
	{
		c->x = 0;
		c->y = 0;
	}
}

char Vector2D_near(Vector2D* a, Vector2D* b, int distanceX, int distanceY)
{
	int x = abs(a->x - b->x);
	int y = abs(a->y - b->y);
	return ((x <= distanceX) && (y <= distanceY));
}

int Vector2d_length(Vector2D* a, Vector2D* b)
{
	int x = a->x - b->x;
	int y = a->y - b->y;
	return abs(x) + abs(y);
}

int Vector2D_length_100(Vector2D* a, Vector2D* b)
{
	int x = a->x - b->x;
	int y = a->y - b->y;
	return abs(x / 100) + abs(y / 100);
}

void Vector2D_normalize(Vector2D* a)
{
	int distance = 0;

	Vector2D unit;
	unit.x = 0;
	unit.y = 0;

	distance = Vector2D_length_100(&unit, a);
	Vector2D_div_int(a, a, distance);
}

void Vector2D_mul_percent(Vector2D* a, int max)
{
	Vector2D_mul_int(a, a, max);
	Vector2D_div_int(a, a, 100);
}

void Vector2D_truncate(Vector2D* a, int max)
{
	int distance = 0;

	Vector2D* unit;
	unit->x = 0;
	unit->y = 0;

	distance = Vector2D_length_100(unit, a);
	if (distance > max)
	{
		Vector2D_normalize(a);
		Vector2D_mul_percent(a, max);
	}
}

void seek(Vector2D* seek, Vector2D* location, Vector2D* velocity, Vector2D* target, int maxspeed, int maxforce)
{
    Vector2D desired;

    Vector2D_sub(&desired, target, location);
    Vector2D_normalize(&desired);
    Vector2D_mul_percent(&desired, maxspeed);

    Vector2D_sub(seek, &desired, velocity);
    Vector2D_truncate(seek, maxforce);
}

void pursuit(Vector2D* pursuit, Vector2D* location, Vector2D* velocity, Vector2D* target, Vector2D* target_velocity, int maxspeed, int maxforce, int T)
{
    Vector2D futurePosition;

    Vector2D_mul_int(&futurePosition, target_velocity, T);
    Vector2D_add(&futurePosition, target, &futurePosition);

    seek(pursuit, location, velocity, &futurePosition, maxspeed, maxforce);
}

char Colision(Vector2D* object1, int ancho1, int alto1, Vector2D* object2, int ancho2, int alto2)
{
	short int left1, left2;
	short int right1, right2;
	short int top1, top2;
	short int bottom1, bottom2;

	left1 = object1->x;
	left2 = object2->x;
	right1 = object1->x + ancho1;
	right2 = object2->x + ancho2;
	top1 = object1->y;
	top2 = object2->y;
	bottom1 = object1->y + alto1;
	bottom2 = object2->y + alto2;

	if (bottom1 < top2) return 0;
	if (top1 > bottom2) return 0;

	if (right1 < left2) return 0;
	if (left1 > right2) return 0;

	return 1;
}

char ColisionPunto(Vector2D* punto, Vector2D* object, int ancho, int alto)
{
    int left;
    int right;
    int top;
    int bottom;

    left = object->x;
    right = object->x + ancho;
    top = object->y;
    bottom = object->y + alto;

    if (left <= punto->x && right >= punto->x && top <= punto->y && bottom >= punto->y)
        return 1;

    return 0;
}
