nolist
; MEMORY LAYOUT

; BANK0
; &0000 - &3E00  Code
; &3E00 - &3EFF  Lookup Table for updating cell pairs
; &3F00 - &3FFF  Stack

; BANK1
; &4000 - &5FFF  Active cells list (new or current)
; &6000 - &7FFF  Active cells list (current or new)

; BANK2
; &8000 - &BFFF  Cell data (old states + neighbour counts)

; BANK3
; &C000 - &FFFF  Video RAM (current states + colors)

; BANK4 (mapped to BANK1 when needed)
; Contains both the information panel video RAM and some 
; routines that need access to the lower ROM.

; SCREEN LAYOUT
; We work in mode 0.
; There are 16 (horizontal) mode 0 caracters on each row, 
; each one 4 bytes wide, so 64 bytes per line. 
; There are 256 (vertical) lines grouped in 32 8-line height rows. 
; Screen addresses take this form (2 bytes)

; BB lll rrr - rr cccccc

; where
; BB       - Memory bank, normally takes value 11 (memory bank 3)
;	     can also be 10 (bank 2) when in double buffered mode
; lll      - Line (vertical) position inside the row (0 - 7)
; rrr - rr - Row (vertical) position, splitted along the high and 
;            low bytes of the screen address (0 - 32)
; cccccc   - Byte position (horizontal) inside the row (0 - 63)

; ALGORITHM OVERVIEW

; This algorithm is an adaptation from the one described at Chapter 18 of 
; Michael Abrash's "Graphics Programmer's Black Book" (downloads.gamedev.net/pdf/gpbb/gpbb18.pdf)
; to the particular 8-bit architecture of the Z80 and the screen memory layout 
; of the Amstrad CPC.

; The representation of cell data in memory bank 2 matches closely the screen
; layout of the video ram in bank 3

; In mode 0 each video ram byte encodes 2 pixels, that in our program 
; represent the dead/alive state of 2 adjacent cells using the two LSB bits. 

; In the video bank, the remaining 6 bits are used to store color information, 
; and in the cell data bank the algorithm uses them to track the neighbour 
; counts for the two cells EXLUDING the other cell within the pair out of the
; count, this partial neighbour count fits in just 3 bits (0 - 7). So a byte
; in the cell data bank takes this form.

; (Nl Nl Nl) (Nr Nr Nr) L R

; where
; (Nl Nl Nl) - Partial neigbour count for left  cell (0 - 7)
; (Nr Nr Nr) - Partial neigbour count for right cell (0 - 7)
; L          - State of left cell, 1 for alive and 0 for dead
; R          - State of right cell, 1 for alive and 0 for dead

; As the count's missing neighbour is already present within the byte (it's 
; just the other cell pair) we can still build a 256 bytes lookup table
; that generates a new live/dead state for the two cells based on its current
; state and its neighbour counts following the rules of classic Conway's Game
; of Life or other variants. 
; Aside from the new state of the cell pair, the lookup table also outputs 
; colour information for each cell, depending on whether it has changed from
; live to dead (or viceversa), or it keeps its live/dead state.

; The algorithm keeps a list of the "active cells" (the ones that are
; changing) so that only those are updated. Each generation is updated in two 
; phases. The process is as follows;

; At the beginning the current state of each cell pair (toghether with its colour 
; information) is stored in the video memory bank, the cell data bank holds 
; the previous states and its neighbour counts.

; During Phase 1 the neighbour counts for the active cell pair's neighbours 
; are upated based on its previous and current state values. 
; This is, either one cell of the pair changed its state during the last 
; generation (from alive to dead or vice-versa), or the two of them changed, 
; or maybe none of them. In any possible case an appropiate routine is called 
; to reflect those changes on the neigbour counts of the neighbours of the 
; cell pair in question.
; Also, current state values are copied from the video memory bank to the cell
; data bank after the old values had been used to calculate the transitions
; and are no longer needed.

; During Phase 2 the active cells list (and all its neighbours) is scanned 
; again, this time updating the states of the cell pairs using the 
; lookup table. At the same time it regenerates the active cells list for the
; next generation by adding the cells that actually changed its state.

; The updated states (toghether with its colour information) are written 
; directly to the video memory bank and the current ones are kept unmodified
; in the cell data bank. Prior to the addition of a particular cell pair to 
; the new active cells list, it is checked whether it has been already updated
; to the video memory or not, thus avoiding duplicate entries.

; At the end we have the updated (now current) states in the video memory bank,
; and the current (now previous) ones in the cell data bank. Which leds us to 
; the starting point again.

run entry

WRITEDIRECT	equ 0
LOOKUPTABLE	equ &3e00
ACPSTART	equ &4000
VIDEORAM	equ &c000

read "macros.asm"
read "macros_phase1.asm"
read "macros_phase2.asm"

org &40
if WRITEDIRECT
write direct "a:yagol.bin"
endif

.firm_int
dw 0
.changelist
dw ACPSTART

.ACPcounter
dw 0
.savetime
dw 0
.time_lapse_spd
dw &ffff
.gencounter
dw 0
.framecounter
db 0

; Flags byte
; Bit 0 - Split (panel) on/off
; Bit 1 - Switch off split (panel)
; Bit 2 - Panel update allowed by time
; Bit 3 - Evolution Running/Paused
; Bit 4 - Double buffer on/off
; Bit 5 - Switch off double buffer 
.flags
db 1

.initvideocfg
db &1,32,&2,42
db &4,22,&5,7,&6,32,&7,35

.liveareaoffset
db &0c,&30,&0d,&00
.liveareacfg
db &4,28,&5,0,&7,34
.paneloffset
db &0c,&10,&0d,&00
.panelcfg
db &4,8,&5,7,&7,6

.speedascii
db "000.00",0
.genascii
db "00000",0
.ACPascii
db "000%",0

align 16
.keybmapraw
ds 10,&ff
align 16
.keybmapfiltered
ds 10
.prevkey
db &00

.diskdrivenumber
db 0
.genlimit
db 0
.dircommand
db "DI","R"+128

; Colour mode
; Bit 0 - Recently dead cells should be painted
; Bit 1 - Newborn cells should be distinguished
.colourmode
db 3
.savecolourmode
db 3

; Default colors for the program
.textcolour
db &4a
.bordercolour
db &5c
.backgroundcolour
db &54
.livecolour
db &57
.newborncolour
db &52
.rdeadcolour
db &4c

;; table to convert from hardware colour number to firmware colour number
.colourconversion
defb &54,&44,&55,&5c,&58,&5d,&4c,&45,&4d,&56,&46,&57,&5e,&40,&5f,&4e,&47,&4f
defb &52,&42,&53,&5a,&59,&5b,&4a,&43,&4b,&41,&48,&49,&50,&51

read "interrupts.asm"
read "interrupts_nosplit.asm"
read "routines.asm"
read "neighupdt.asm"
read "neighupdt_edge.asm"

.entry
; Save drive letter
ld hl,(&be7d)
ld a,(hl)
ld (diskdrivenumber),a

; Save firmware "jump restore" function address
ld hl,(&bd38)
ld (restorefirmjumpblock+1),hl

if WRITEDIRECT
ld hl,bank1start
ld de,ROW0BUFFER
ld bc,bank1end-bank1start
ldir
endif

read "inithardware.asm"
read "tablegen.asm"
read "loadpattern.asm"

.begin_evolution
; Wait if evolution is paused
ld hl,flags
bit 3,(hl)
jp z,begin_evolution

; Reset time counter
di
exx
ld de,0
exx
ei

read "initpattern.asm"

jp mainloop_entry

.mainloop

read "phase1.asm"

displayCellData

read "phase2.asm"

displayVideoBank

.mainloop_entry

; Increment generation counter and check if it's a multiple
; multiple of 4
ld hl,(gencounter)
ld a,l
inc hl
ld (gencounter),hl
xor l
; If it is, save time elapsed since 4 generations ago, to
; calculate speed when updating the panel
bit 2,a
jp z,skip_savetime
di
exx
ld (time_lapse_spd),de
ld de,0
exx
ei
.skip_savetime

ld hl,flags
; Check if there is a generation limit countdown active
ld a,(genlimit)
or a
jp z,skip_pause_evolution
; Decrement countdown
dec a
ld (genlimit),a
jp nz,skip_pause_evolution
; And pause evolution if it reaches zero
res 3,(hl)
set 2,(hl)
.skip_pause_evolution

; Update panel if allowed by time
bit 2,(hl)
call nz,updatepanel

; Check if evolution is paused
ld hl,flags
bit 3,(hl)
jp nz,skip_pause
; Save current time
di
exx
ld (savetime),de
exx
ei
.pause_loop
; Disable double buffer mode if required
call dbuff_disable
; Loop until evolution is unpaused
ld hl,flags
bit 3,(hl)
jp z,pause_loop
; Restore time 
di
exx
ld de,(savetime)
exx
ei
.skip_pause

jp mainloop

.still_life
jr still_life

.ACPoverflow
ld hl,flags
call updatepanel
.ACPoverflow_loop
jr ACPoverflow_loop

if WRITEDIRECT
limit &4000
.bank1start
incbin "bank1.bin"
.bank1end
endif

read "bank1.asm"
