.H 2 "A Command's Environment"
All the variables (with their associated values) that are known to
a command at the beginning of execution of that command constitute its 
.I environment .
This environment includes variables that the command inherits
from its parent process and variables specified as
.I "keyword parameters"
on the command line that invokes the command.
.P
The variables that a shell passes to its child processes are those that
have been named as arguments to the 
.B export 
command.
The 
.B export 
command places the named variables in
the environments of both the shell 
.I and
all its future child processes.
.P
Keyword parameters are variable-value pairs that appear in the form of
assignments, normally 
.I before
the procedure name on a command line.
Such variables are placed in the environment of the procedure being
invoked.
For example:
.DS I
:	keycommand
echo $a $b
.DE
is a simple procedure that echoes the values of two variables.
If it is invoked as:
.DS I
a=key1 b=key2 keycommand
.DE
then the resulting output is:
.DS I
key1 key2
.DE
Keyword parameters are 
.I not
counted as arguments to the procedure
and do not affect $#.
.P
A procedure may access the value of any variable in its environment.
However, if changes are made to the value of a variable, these
changes are not reflected in the environment; they are local to
the procedure in question.
In order for these changes to be placed in
the environment that the procedure 
passes to 
.I its
child processes, the variable must be named as an argument to the 
.B export 
command within that procedure. 
To obtain a list of variables that have been made 
exportable from the current shell, type:
.DS I
export
.DE
You will also get a list of variables that have been made
.B readonly .
To get a list of name-value pairs in the
current
environment, type:
.DS I
set
.DE
.H 2 "Invoking the Shell"
The shell is a command and may be invoked in the
same way as any other command:
.VL 25 3
.LI sh~\f2proc\^\fP~[~\f2arg~.\|.\|.\^\fP~]
A new instance of the shell is explicitly invoked to read
.FN proc .
Arguments, if any, can be manipulated. 
.LI sh~\-v\|~\f2proc\^\fP\|~[~\f2arg\|.\|.\|.\^\fP~]
This is equivalent to putting
.Q "set \-v"
at the beginning of
.FN proc .
Similarly, for the 
.B \-x , 
.B \-e , 
.B \-u , 
and 
.B \-n 
flags. 
.LI \f2proc\^\fP~[~\f2arg~.\|.\|.\^\fP~]
If
.I proc
is marked executable,
and is not a compiled, executable program,
the effect is similar to that of:
.DS I
sh proc args 
.DE
An advantage of this form is that variables that have been exported
in the shell will still be exported from
.I proc\^
when this form is used 
(because the shell only forks
to read commands from
.I proc ).
Thus any changes made within 
.FN proc
to the values of
exported variables
will be passed on to subsequent commands invoked from
.FN proc .
.br
.LE
.P
.H 2 "Passing Arguments to the Shell"
When a command line is scanned,
any character sequence of the form 
.RI $ n
is replaced by the
.IR n th
argument to the shell,
counting the name of the shell procedure itself as $0.
This notation permits direct 
reference to the procedure name and to as many as
nine positional parameters. 
Additional arguments can be processed using the
.B shift
command
or by using a
.B for
loop. 
.P
The 
.B shift 
command shifts arguments to the left; i.e., the value of $1 
is thrown away, 
$2 replaces $1, 
$3 replaces $2, and so on.
The highest-numbered positional parameter becomes 
.I unset
($0 is never shifted).
For example, in the shell procedure
.I ripple
below,
.B echo
writes its arguments to the standard output;
.B while
is discussed later (it is a looping command);
and lines that begin with a colon (:) are comments:
.DS I
:	ripple command
while test $# != 0
do
	echo $1 $2 $3 $4 $5 $6 $7 $8 $9
	shift
done
.DE
.P
If the procedure were invoked by
.DS I
ripple a b c
.DE
it would print:
.DS I
a b c
b c
c
.DE
.P
The special shell variable 
.Q "star"
($*) 
causes substitution of all positional parameters except $0.
Thus, the
.B echo
line in the
.I ripple
example above could be
written more compactly as:
.DS I
echo $*
.DE
.ne 2i
.P
These two
.B echo
commands are 
.I not
equivalent:
the first prints at most nine positional parameters;
the second prints 
.I all
of the current positional parameters.
The shell star variable ($*)
is more concise and less error-prone.
One obvious application is in
passing an arbitrary number of arguments to a command:
.DS I
nroff -man $*
.DE
.P
It is important to understand the sequence of
actions used by the shell in
scanning command lines and substituting arguments.
The shell first reads input up to a newline or semicolon, 
and then parses that much of the input.
Variables are replaced by their values and then command substitution
(via back quotation marks)
is attempted.
I/O redirection arguments are detected, 
acted upon, and deleted from the command line.
Next, the shell scans the
resulting command line for
.I "internal field separators" ,
that is, for any characters specified by IFS
to break the command line
into distinct arguments;
.I explicit
null arguments (specified by "" or \'\') are retained,
while implicit
null arguments resulting from evaluation of variables 
that are null or not set are removed.
Then filename generation occurs, 
with all metacharacters being expanded.
The resulting command line is then executed by the shell.
.P
Sometimes, command lines are built inside a shell procedure.
In this case, it is sometimes useful 
to have the shell rescan
the command line after all the initial substitutions and
expansions have been performed.
The special command
.B eval
is available for this purpose.
.B Eval
takes a command line as its argument and simply rescans the line,
performing any variable or command substitutions that are specified.
Consider the following (simplified) situation:
.DS I
command=who
output=\' | wc -l\'
eval $command $output
.DE
This segment of code results in the execution
of the command line 
.DS I
who | wc -l 
.DE
.P
The output of 
.B eval 
cannot be redirected.
However, uses of 
.B eval
can be nested, so that a command line
can be evaluated several times.
.H 2 "Control Commands"
The shell provides several commands that implement
a variety of
control structures useful in creating shell procedures.
Before describing these structures, a few terms need to be defined.
.P
A
.I "simple~command"
is any single irreducible command specified by the
name of an executable file.
I/O redirection arguments can appear in a simple command line 
and are passed to the shell,
.I not
to the command.
.P
A
.I command
is a simple command
or any of the shell control commands described below.
A
.I pipeline
is a sequence of one or more commands separated by 
a vertical bar (\||\|).
In a pipeline, the standard output of each command 
but the last is connected
(by a
.I pipe )
to the standard input of the next command.
Each command
in a pipeline
is run separately; the shell waits for the last command to finish.
The exit status of a pipeline is nonzero if the exit
status of either the first or last process 
in the pipeline is nonzero.
.P
A
.I "command list"
is a sequence of one or more pipelines separated by a
semicolon 
.RB ( ; ), 
an ampersand 
.RB ( & ), 
an 
.Q "and-if"
symbol
.RB ( && ), 
or an 
.Q "or-if"
.RB (\| |\|| \|) 
symbol,
and optionally terminated by 
a semicolon or an ampersand.
A semicolon causes sequential execution 
of the previous pipeline. 
This means that the shell waits for the pipeline 
to finish before reading the next pipeline. 
On the other hand, the ampersand 
.RB ( & ) 
causes asynchronous background execution of the
preceding pipeline.
Thus, both sequential and background execution are allowed.
A background pipeline continues execution until
it terminates voluntarily, or until its processes
are killed.
.P
Other uses of  the ampersand
include off-line printing,
background compilation,
and generation of jobs to be sent to other computers.
For example, if you type
.DS I
nohup cc prog.c&
.DE
.P
you may continue working while the C compiler runs in the background.
A command line ending with an ampersand
is immune to interrupts or quits that you might
generate by typing 
.SM <INTERRUPT>
or 
.SM <QUIT> .
It is also immune to logouts with
.SM <CONTROL-D>.
However, 
.SM <CONTROL-D>
.I will
abort the command if you are operating over a dial-up line.
In this case, it is wise to make the command
immune to hang-ups (i.e., logouts) as well.
The
.B nohup
command is used for this purpose.
In the above example without
.B nohup ,
if you log out from a dial-up line while
.B cc
is still executing,
.B cc
will be killed and your output will disappear.
.P
The ampersand
operator should be used with restraint,
especially on heavily-loaded systems.
Other users will not consider you a good citizen if you start up
a large number of background processes 
without a compelling reason for doing so.
.P
The and-if and or-if (&& and |\||\|) operators
cause conditional execution of pipelines.
Both of these are of equal precedence 
when evaluating command lines (but both are lower than 
the ampersand
.RB ( & )
and the vertical bar
.RB (\| | \|)).
In the command line
.DS I
cmd1 || cmd2
.DE
the first command, 
.I cmd1,
is executed and its exit status examined.
Only if 
.I cmd1 
fails (i.e., has a nonzero exit status) is
.I cmd2
executed.
Thus, this is a more terse notation for:
.DS I
if	cmd1
	test $? != 0
then
	cmd2
fi
.DE
.P
The and-if operator (&&)
operator yields a complementary test.
For example, in the following command line
.DS I
cmd1 && cmd2
.DE
the second command is executed
only if the first 
.I succeeds 
(and has a zero exit status).
In the sequence below, each command is executed in order
until one fails:
.DS I
cmd1 && cmd2 && cmd3 && ... && cmdn
.DE
.P
A simple command in a pipeline may be replaced by
a command list enclosed in
either parentheses or braces.
The output of all the commands so enclosed is combined
into one stream that becomes the 
input to the next command in the pipeline.
The following line formats and prints 
two separate documents: 
.DS I
{ nroff -mm text1; nroff -mm text2; } | lpr
.DE
Note that a space is usually needed after the left
brace and before the right brace.
.H 3 "Structured Conditional:  if"
The shell provides structured conditional capability with the
.B if
command.
The simplest
.B if
command has the following form:
.DS I
if command-list
then command-list
fi
.DE
The command list following the 
.B if
is executed and if the last command 
in the list has a zero exit status, 
then the command list that follows
.B then
is executed.
The word
.B fi
indicates the end of the
.B if
command.
.P
To cause an alternative 
set of commands to be executed when there is a nonzero
exit status, 
an 
.B else 
clause can be given with the following structure:
.DS I
if \fIcommand-list\fR
then \fIcommand-list\fR
else \fIcommand-list\fR
fi
.DE
Multiple tests can be achieved in an
.B if
command by using the
.B elif
clause, although the 
.B case
statement is better for large numbers of tests.
For example:
.DS I
if	test -f "$1" 
:			is $1 a file?
then	pr $1
elif	test -d "$1" 
:			else, is $1 a directory?
then	(cd $1; pr *)
else	echo $1 is neither a file nor a directory
fi
.DE
The above example is executed as follows: if
the value of the first positional parameter is a filename,
then print that file; if not, then check to see if it is
the name of a directory.
If so, change to that
directory and print all the files there.
Otherwise, 
.B echo 
the error message.
.P
The
.B if
command may be nested (but be sure to end each one with a 
.B fi ).
The newlines in the above examples of
.B if
may be replaced by semicolons.
.P
The exit status of the
.B if
command is the exit status of the last command executed in any
.B then
clause
or
.B else
clause.
If no such command was executed,
.B if
returns a zero exit status.
.H 3 "Multiple Way Branch:  case"
.br
A multiple way branch is provided by the
.B case
command.
The basic format of
.B case
is:
.DS I
case string in
	pattern ) command-list ;;
	...
	pattern ) command-list ;;
esac
.DE
The shell tries to match
.I string\^
against each pattern in turn, 
using the same pattern-matching conventions
as in filename generation.
If a match is found, the command list following the
matched pattern is executed; 
the double semicolon (;;)
serves as a break out of the 
.B case 
and is required
after each command list except the last.
Note that only one pattern is ever matched, 
and that matches are
attempted in order, 
so that if a star (*)
is the first pattern in a 
.B case ,
no other patterns are looked at.
.P
More than one pattern may be associated 
with a given command list by specifying
alternate patterns separated by vertical bars (\||\|).
For example:
.DS I
case $i in
	*.c) 	cc $i
		;;
	*.h | *.sh)	
		: do nothing
		;;
	*)	echo "$i of unknown type"
		;;
esac
.DE
In the above example, 
no action is taken for the second set of patterns
because the null, colon 
.RB ( : )
command is specified.
The star (*)
is used as a default pattern,
because it matches any word.
.P
The exit status of
.B case
is the exit status of the last command executed in the
.B case
command.
If no commands were executed, then
.B case
has a zero exit status.
.H 3 "Conditional Looping:  while and until"
.br
A
.B while
command has the general form:
.DS I
while \fIcommand-list\fP
do
	\fIcommand-list\fP
done
.DE
The commands in the first command list are executed,
and if the exit status of the last command
in that list is zero, 
then the commands in the second list are executed.
This sequence is repeated as long as the exit status 
of the first command list is zero.
A loop can be executed as long as the first command list 
returns a nonzero exit
status by replacing 
.B while
with 
.B until .
.P
Any newline
in the above example may be replaced by a semicolon.
The exit status of a
.B while 
(or
.B  until )
command is the exit status of the last command
executed in the 
second
command list.
If no such command is executed,
.B while 
(or
.B  until )
has a zero exit status.
.H 3 "Looping Over a List:  for"
.br
Often, one wishes to perform some set of operations
for each file in a set of files, or execute some command
once for each of several arguments.
The
.B for
command can be used to accomplish this.
The
.B for
command has the format
.DS I
.B
for \fIvariable\fP in \fIword-list\fP
do
	\fIcommand-list\fP
done
.R
.DE
where 
.I word-list
is a list of strings separated by blanks.
The commands in the command-list are executed once
for each word in the word list.
.I Variable
takes on as its value each
word from the word list, in turn.
The word list is fixed after it is evaluated the first time.
For example, the following 
.B for 
loop
causes each of the C source files 
.FN xec.c ,
.FN cmd.c ,
and
.FN word.c
in the current directory to be 
.Q "diffed"
with a file of the same name in the directory
.FN /usr/src/cmd/sh :
.DS I
for CFILE in xec cmd word
do	diff $CFILE.c /usr/src/cmd/sh/$CFILE.c
done
.DE
.P
The first occurrence of CFILE immediately after the
word 
.B for
has no
preceding dollar sign, since the name of the
variable is wanted and not its value.
.P
One can omit the 
.Q "in \fIword-list\fR"
part of a 
.B for 
command; this 
causes the current set of positional parameters 
to be used in place of word-list.
This is useful when writing a command
that performs the same set of commands 
for each of an unknown number of arguments.
For example, assume that the following command is given the
name 
.Q "echo2" :
.DS I
for word
do echo $word$word
done
.DE
Then this command could be given execute status and
executed as follows:
.DS I
chmod +x echo2
echo2 ma pa bo fi yo no so ta
.DE
The output from this command would be:
.DS I
mama
papa
bobo
fifi
yoyo
nono
soso
tata
.DE
.H 3 "Loop Control:  break and continue"
.br
The 
.B break 
command
can be used to terminate execution of a 
.B while 
or
a 
.B for 
loop.
.B Continue
requests the execution of the next iteration of the loop.
These commands are effective only when they appear between
.B do
and
.B done .
.P
The 
.B break 
command
terminates execution of the smallest
(i.e., innermost)
enclosing loop, 
causing execution to resume after the nearest following
unmatched
.B done .
Exit from
.I n\^
levels is obtained by
.B break 
.I n.
.P
The 
.B continue 
command
causes execution to resume at the nearest enclosing
.B for ,
.B while ,
or
.B until
statement,
i.e., the one that begins
the innermost loop containing the
.B continue.
You can also specify an argument 
.I n
to
.B continue
and execution will resume at the
.I n th
enclosing loop:
.DS I
: This procedure is interactive. 
: "Break" and "continue" commands are used 
: to allow the user to control data entry.
while true
do	echo "Please enter data"
	read response
	case "$response" in
	"done")	break	
		: no more data
		;;
	"")	continue
		;;
	*) 	: process the data here
		;;
	esac
done
.DE
.H 3 "End-of-File and exit"
.br
When the shell reaches the end-of-file,
it terminates execution, 
returning to its parent the exit status
of the last command executed prior to the end-of-file.
The top level shell is terminated by typing a
.SM <CONTROL-D>
which is the same as logging out.
.P
The
.B exit
command simply reads to the end-of-file and returns,
setting the exit status to the value of its argument, if any.
Thus, a procedure can be terminated
normally
by placing 
.Q "exit\ 0"
at the end of the file.
.H 3 "Command Grouping: Parentheses and Braces"
.br
There are two methods for grouping commands in the shell:
parentheses and braces.
Parentheses cause the shell to create a subshell
that reads the enclosed commands.
Both the right and left
parentheses
are recognized 
wherever they appear in a command line\*(EMthey can appear as 
literal parentheses 
.I only
by being quoted.
For example,
if you type 
.DS I
garble(stuff) 
.DE
the shell interprets this
as four discrete words: 
.Q "garble" , 
.Q "(" , 
.Q "stuff" , 
and 
.Q ")" .
.P
This subshell capability is useful when performing
operations without affecting
the values of variables in the current shell, or when temporarily
changing directory and executing commands in the
new directory without having to explicitly return to
the current directory.
.P
The current environment is passed to the subshell and
variables that are exported
in the current shell are also exported
in the subshell.
Thus
.DS I
CURRENTDIR=\`pwd\`; cd /usr/docs/otherdir;
nohup nroff doc.n | lpr& ; cd $CURRENTDIR
.DE
and
.DS I
(cd /usr/docs/otherdir; nohup nroff doc.n | lpr&)
.DE
accomplish the same result: a copy of 
.FN /usr/docs/otherdir/doc.n
is sent to the lineprinter.
However, the second example automatically puts you back in your
original working directory.
In the second example above, blanks or newlines surrounding the
parentheses are allowed but not necessary.
The shell will prompt with the value of the shell
variable PS2 
if an end parenthesis is expected.
.P
Braces ({) and (})
may also be used to group commands together.
Both the left and the right brace are recognized 
.I only
if they appear as the first (unquoted) word of a command.
The opening brace 
may be followed by a newline 
(in which case the shell prompts for more input).
Unlike parentheses,
no subshell is created for braces: 
the enclosed commands are simply read by the shell.
The braces are convenient when you wish 
to use the (sequential) output
of several commands as input to one command. 
.P
The exit status of a set of commands 
grouped by either parentheses or braces is
the exit status of the last enclosed executed command.
.H 3 "Input/Output Redirection and Control Commands"
The shell normally does not fork 
when it recognizes the control
commands (other than parentheses) described above.
However, each command in a pipeline
is run as a separate process in order 
to direct input to or output from each command.
Also, when redirection of input/output
is specified explicitly for a control command, 
a separate process is
spawned to execute that command.
Thus, when
.B if ,
.B while ,
.B until ,
.B case ,
and
.B for
are used in a pipeline consisting of
more than one command, the shell forks
and a subshell runs the control command.
This has two implications:
.AL 1
.LI
The most noticeable implication is that
any changes made to variables within the control command
are not effective once that control command finishes
(this is similar to the effect of 
using parentheses to group commands).
.LI
The second implication is that
control commands run slightly slower 
when redirected,
because of the additional overhead of creating a
shell for the control command.
.LE
.H 3 "Transfer to Another File and Back: The Dot (\s+2.\s-2) Command"
A command line of the form
.DS I
\&. proc
.DE
causes the shell to read commands from
.I proc
without spawning a new process.
Changes made to variables in
.I proc
are in effect after the
dot command finishes.
This is a good way to gather
a number of shell variable initializations
into one file.
A common use of this command is to reinitialize the
top level shell by reading the 
.FN .profile 
file with:
.DS I
\&. .profile
.DE
.H 3 "Interrupt Handling:  trap"
A program may choose to
catch an interrupt from the terminal,
to ignore it completely, or to be terminated by it.
Shell procedures can use the
.B trap
command to obtain the same effects.
.DS I
trap \fIarg signal-list\fP
.DE
is the form of the
.B trap
command, where 
.I arg
is a string 
to be interpreted as a command list and
.I "signal-list"
consists of one or more signal numbers 
as described in
.IR signal (2)).
The commands in 
.I arg
are scanned at least once, when the shell first encounters the
.B trap
command.
Because of this, it is usually wise to use single rather
than double quotation marks to surround these commands.
The former inhibit
immediate command and variable substitution.
This becomes important, for instance, 
when one wishes to remove temporary files and
the names of those files have not yet been determined when the
trap command is first read by the shell.
The following procedure will print the name of the current directory
in the file 
.FN errdirect 
when it is interrupted, thus giving
the user information as to how much of the job was done:
.DS I
trap \'echo \`pwd\` >errdirect\' 2 3 15
for i in /bin /usr/bin /usr/gas/bin
do
	cd $i
	   : commands to be executed in directory $i here
done
.DE
.P
Beware that
the same procedure with double rather than single quotation marks
does something different:
.DS I
(trap "echo `pwd` >errdirect" 2 3 15) 
.DE
.P
This prints the name of the directory
from which the procedure was executed.
.P
Signal 11 may never be trapped, 
because the shell itself needs to catch it
to deal with memory allocation.
Zero is interpreted by the
.B trap
command as a signal generated by exiting from a shell.
This occurs either with an 
.B exit 
command, or by 
.Q "falling through"
to the end of a procedure.
If
.I arg
is not specified, then the action taken upon receipt
of any of the signals in
the signal list is reset to the default system action.
If
.I arg
is an explicit null string 
( \'\' or "" ),
then the signals in the signal list
are ignored by the shell.
.P
The 
.B trap
command is most frequently used to make 
sure that temporary files are removed upon
termination of a procedure.
The preceding example would be written more typically as follows:
.DS I
temp=$HOME/temp/$$
trap \'rm $temp; trap 0; exit\' 0 1 2 3 15
ls > $temp
     : commands that use $temp here
.DE
.P
In this example, whenever
signal 1 (hangup), 2 (interrupt), 3 (quit), or 15 (kill) 
is received by the shell procedure, or
whenever the shell procedure is about to exit,
the commands enclosed between the single quotation marks are executed.
The
.B exit
command must be included, 
or else the shell continues reading commands
where it left off when the signal was received.
The 
.Q "trap\00"
in the above procedure
turns off the original trap on exits 
from the shell, so that the
.B exit
command does not reactivate the execution of the trap commands.
.P
Sometimes 
the shell continues reading commands after
executing trap commands.
The following procedure takes each directory 
in the current directory, changes to that directory, prompts
with its name, and executes commands 
typed at the terminal until an end-of-file 
(\s-2<CONTROL-D>\s0) 
or an interrupt is received.
An end-of-file causes the 
.B read 
command to return a nonzero exit status, and
thus the 
.B while 
loop terminates and the next
directory cycle is initiated. 
An interrupt is ignored while executing the
requested commands, but causes termination of the
procedure when it is waiting for input:
.DS I
d=\`pwd\`
for i in *
do	if test \-d $d/$i
	then	cd $d/$i
		while 	echo "$i:"
			trap exit 2
			read x
		do	trap : 2 	
			: ignore interrupts
			eval $x
		done
	fi
done
.DE
.P
Several
.B trap s
may be in effect at the same time: 
if multiple signals are received
simultaneously, they are serviced in numerically ascending order.
To determine which traps are currently set, type:
.DS I
trap
.DE
.P
It is important to understand some things 
about the way in which the shell implements
the
.B trap 
command. 
When a signal (other than 11) is received by the shell, 
it is passed on
to whatever child processes are currently executing.
When these
(synchronous) processes terminate, normally or abnormally, 
the shell 
polls any traps that happen to be set and executes the appropriate
.B trap
commands.
This process is straightforward, except in the case of
traps set at the command (outermost, or login) level.
In this case, it is possible
that no child process is running, so before the
shell polls the traps, it waits
for the termination of the first process spawned
.I after
the signal was received.
.P
For internal commands, 
the shell normally polls traps on completion of the command.
An exception to this rule is made for the 
.B read 
command, for which traps are serviced immediately, so that 
.B read 
can be interrupted while waiting for input.
.H 2 "Special Shell Commands"
There are several special commands 
that are 
.I internal
to the shell (some of which have already been mentioned).
These commands should be used whenever possible,
because they are, in general, faster and more efficient
than other \*(x1 commands.
The shell does not fork to execute these commands,
so no additional processes are spawned.
The trade-off for this efficiency is that redirection
of input/output is not allowed for most of these special commands.
.P
Several of the special commands have already been described 
because they affect the flow of control.
They are dot
.RB (\| . \|),
.B break ,
.B continue ,
.B exit ,
and
.B trap .
The
.B set
command is also a special command.
Descriptions of the remaining special commands are given here:
.VL 22 3
.LI "\fB:\fR"
The
null
command. 
This command does nothing
and is used to insert commments in shell procedures.
Its exit status is zero (true).
Beware:
any arguments to the null command are parsed for syntactic
correctness; when in doubt, quote such arguments.
Parameter substitution takes place, just as in other commands.
.LI  cd~\f2arg\^\fP
Make
.I arg\^
the current directory.
If
.I arg\^
is not a valid directory, or the user is not authorized to
access it, a nonzero exit status is returned.
Specifying
.B cd
with no
.I arg\^
is equivalent to typing
.Q "cd~$HOME" .
.LI  exec~\f2arg\^\fP~.\|.\|.
If
.I arg\^
is a command, then the shell executes it without forking.
No new process is created.
Input/output redirection arguments are allowed
on the command line.
If 
.I only
input/output redirection arguments appear,
then the input/output of the shell itself is modified accordingly.
.LI newgrp~\f2arg\^\fP~.\|.\|.
The
.B newgrp 
command is executed, 
replacing the shell. 
.B Newgrp 
in turn creates a new shell.
Beware: only environment variables
will be known in the
shell created by the
.B newgrp
command.
Any variables that were
exported will no longer be marked as such.
.LI read~\f2var\^\fP~.\|.\|.
One line (up to a newline)
is read from standard input and 
the first word is assigned to the first
variable, 
the second word to the second variable, and so on.
All words left over are
assigned to the
.I last
variable.
The exit status of
.B read
is zero unless an
end-of-file
is read.
.LI readonly~\f2var\^\fP~.\|.\|.
The specified variables are made
.B readonly
so that no subsequent assignments may be made to them.
If no arguments are given, a list of all
.B readonly
and of all exported variables is given.
'\"	.LI test
'\"	A conditional expression is evaluated.
.LI times
The accumulated user and system times 
for processes run from the current shell are printed.
.LI umask~\f2nnn\^\fP
The user file creation mask is set to
.I nnn .
If
.I nnn\^
is omitted, then the current value of the mask is printed.
.LI wait
The shell waits for all currently active child processes to terminate.
The exit status of 
.B wait 
is always zero.
.LE
.H 2 "Creation and Organization of Shell Procedures"
A shell procedure can be created in two simple steps.
The first is building an ordinary text file.
The second is changing the
.I mode
of the file to make it
.I executable ,
thus permitting it to be invoked by
.DS I
proc args
.DE
rather than
.DS I
sh proc args
.DE
.P
The second step may be omitted for a procedure to be used
once or twice and then discarded,
but is recommended for longer-lived ones.
Here is the entire input needed to set up a simple procedure:
.DS I
ed
a
nroff -T450 -man $*
\&.
w draft
q
chmod +x draft
.DE
.P
It may then be invoked as
.DS I
draft file1 file2
.DE
Note that shell procedures must always be at least readable, so that
the shell itself can read commands from the file.
.P
If
.FN draft
were thus created in a directory 
whose name appears in the user's PATH variable,
the user could change working directories and still invoke the
.I draft
command.
.P
Shell procedures may be created dynamically.
A procedure may generate a file of commands,
invoke another instance of the shell to execute that file,
and then remove it.
An alternate approach is 
that of using the \f2dot\^\fP command
(.)
to make the current shell read commands from the new file, 
allowing use of existing shell variables and
avoiding the spawning
of an additional process for another shell.
.P
Many users prefer to write shell procedures instead of C programs.
This is true for several reasons:
.AL 1
.LI
A shell procedure is easy to create and maintain
because it is only a file of ordinary text.
.LI
A shell procedure has no corresponding object program that must
be generated and maintained.
.LI
A shell procedure is easy to create on the fly,
use a few times, and then remove.
.LI
Because shell procedures are usually short in length,
written in a high-level programming language,
and kept only in their source-language form,
they are generally easy to find, understand, and modify.
.LE
.P
By convention,
directories that contain only commands and
shell procedures are named
.FN bin .
This name is derived from the word 
.Q "binary" ,
and is used because compiled and executable programs
are often called 
.Q "binaries"
to distinguish
them from program source files.
Most groups of users sharing common interests
have one or more
.FN bin
directories set up to
hold common procedures.
Some users have their PATH
variable list several such directories.
Although you can have a number of such directories,
it is unwise to go overboard: it may become
difficult to keep track of your environment
and efficiency may suffer.
.H 2 "More About Execution Flags"
There are several execution flags available in the shell that can
be useful in shell procedures:
.VL 9 3
.LI \-e
This flag causes the shell to exit immediately if any command
that it executes exits with a nonzero exit status.
This flag is useful
for shell procedures composed of simple command lines; 
it is not intended for use in conjunction with 
other conditional constructs.
.LI \-u
This flag causes unset variables to be considered
errors when substituting variable values.
This flag can be used to effect a
global check on variables, 
rather than using conditional substitution
to check each variable.
.LI \-t
This flag causes
the shell to exit after reading and executing the
commands on the remainder of the current input line.
.LI \-n
This is a
.Q "don't execute"
flag.
On occasion, one may want to check a procedure for syntax errors, 
but not execute the commands in the procedure.
Using 
.Q "set~\-nv"
at the beginning of a file will accomplish this.
.LI \-k
This flag causes all arguments of the form
\fIvariable\^\fP=\fIvalue\^\fP
to be treated as keyword parameters.
When this flag is 
.I not
set, only such arguments that appear before
the command name are treated as keyword parameters.
.LE
.H 2 "Supporting Commands and Features"
Shell procedures can make use of any
\*(x1
command.
The commands described in this section 
are either used especially frequently
in shell procedures,
or are explicitly designed for such use.
.H 3 "Conditional Evaluation:  test"
The
.B test
command
evaluates the expression specified by its arguments and, if the
expression is true, 
.B test
returns a zero
exit status. 
Otherwise, a nonzero (false) exit status is returned.
.B Test
also returns a nonzero exit status if it has no arguments.
Often it is convenient to use the
.B test
command as the first
command
in the command list following an 
.B if
or a 
.B while .
Shell variables used in
.B test
expressions should be enclosed in double
quotation marks if there is 
any chance of their being null or not set.
.P
On some \*(x1 systems, 
the square brackets 
may be used as an alias to 
.B test , 
for example 
.DS I
[ \fIexpression\fR ] 
.DE
has the same effect as:
.DS I
test \fIexpression\fP
.DE
Note that the spaces before and after the 
.I expression 
in brackets are essential.
.P
The following is a partial list of the primaries that can
be used to construct a conditional expression:
.VL 15 3
.LI "\-r\ \f2file\^\fP"
True if the named file exists and is readable by the user.
.LI "\-w\ \f2file\^\fP"
True if the named file exists and is writable by the user.
.LI "\-x\ \f2file\^\fP"
True if the named file exists and is executable by the user.
.LI "\-s\ \f2file\^\fP"
True if the named file exists and has a size greater than zero.
.LI "\-d\ \f2file\^\fP"
True if the named file is a directory.
.LI "\-f\ \f2file\^\fP"
True if the named file is an ordinary file.
.LI "\-z\ \f2s1\^\fP"
True if the length of string
.I s1\^
is zero.
.LI "\-n\ \f2s1\^\fP"
True if the length of the string
.I s1\^
is nonzero.
.LI "\-t\ \f2fildes\^\fP"
True if the open file 
whose file descriptor number is \f2fildes\^\fP
is associated with a terminal device.
If \f2fildes\^\fP is not specified, 
file descriptor 1 
is used by default.
.LI "\f2s1\^\fP\ =\ \f2s2\^\fP"
True if strings
.I s1\^
and
.I s2\^
are identical.
.LI "\f2s1\^\fP\ !=\ \f2s2\^\fP"
True if strings
.I s1\^
and
.I s2\^
are
.I not\^
identical.
.LI "\f2s1\^\fP"
True if
.I s1\^
is 
.I not
the null string.
.LI "\f2n1\^~-eq~n2\^\fP"
True if the integers
.I n1\^
and
.I n2\^
are algebraically equal;
other algebraic comparisons are indicated by
.I \-ne ,
.I \-gt ,
.I \-ge ,
.I \-lt ,
and
.I \-le .
.LE
.P
These primaries may be combined with the following operators:
.VL 15 3
.LI !
Unary negation operator.
.LI \-a
Binary logical
.I and\^
operator.
.LI \-o
Binary logical
.I or\^
operator;
it has lower precedence than
the binary logical or operator (\-a).
.LI (\f2expr\^\fP)
Parentheses for grouping;
they must be escaped
to remove their significance to the shell.
In the absence of parentheses, 
evaluation proceeds from left to right.
.LE
.P
Note that all primaries, operators, filenames, 
etc. are separate arguments to
.B test .
.H 3 "Simple Output: echo"
.br
The
.B echo
command
is invoked as:
.DS I
.B
echo [ \fIarg\fP ... ]
.R
.DE
.B Echo
copies its arguments to the standard output,
each followed by a single space,
except for the last argument, 
which is normally followed by a newline.
Often, it is used to prompt the user for input,
to issue diagnostics in shell procedures,
or to add a few lines to an output stream in the middle of a pipeline.
Another use is to verify the argument list generation process
before issuing a command that does something drastic.
The command
.DS I
ls
.DE
is often replaced by
.DS I
echo *
.DE
because the latter is faster and prints fewer lines of output.
.P
'\"	The 
'\"	.B echo 
'\"	command recognizes several escape sequences.
'\"	A backslash, followed by entry of the newline
'\"	character for your terminal,
'\"	yields the newline character.
The \-n 
switch to 
.B echo
removes the newline from the end of the echoed line.
The following two commands prompt for input and
then allow typing on the same line as the prompt:
.DS I
echo -n \'enter name:\'
read name
.DE
.H 3 "Expression Evaluation: expr"
.br
The 
.B expr 
command
provides arithmetic
and logical operations on integers and
some pattern-matching facilities on its arguments.
It evaluates a single expression and writes the result on
the standard output;
.B expr
can be used inside grave accents to set a variable.
Typical examples are:
.DS I
:	increment $A
A=\`expr $a + 1\`
:	put third through last characters of
:	$1 into substring
substring=\`expr "$1" : \'..\e(.*\e) \' \`
:	obtain length of $1
c=\`expr "$1" :  \'.* \' \`
.DE
.P
The most common uses of
.B expr
are in counting iterations of a loop and in using
its pattern-matching capability
to pick apart strings.
.H 3 "True and False"
.br
The
.B true
and
.B false
commands perform the obvious functions of exiting
with zero and nonzero exit status, respectively.
The 
.B true
command
is often used to implement an unconditional loop.
.H 3 "In-Line Input Documents"
.br
Upon seeing a command line of the form
.DS I
\fIcommand\fB << \fIeofstring\fR
.DE
where
.I eofstring
is any arbitrary string,
the shell will take the subsequent lines 
as the standard input of
.I command
until a line is read consisting only of
.I eofstring .
'\"	(By appending a minus (\-)
'\"	to the input redirection symbol (<<),
'\"	leading spaces and tabs are deleted 
'\"	from each line of the input document
'\"	before the shell passes 
'\"	the line to 
'\"	.I command .)
.P
The shell creates a temporary file containing the input
document and performs variable and command substitution on
its contents before passing it to the command.
Pattern matching on filenames is performed 
on the arguments of command lines
in command substitutions.
In order to prohibit all substitutions, one may quote any character of
.I eofstring :
.DS I
command << \eeofstring
.DE
.P
The in-line input document feature is especially useful
for small amounts of input data, where it
is more convenient to place the data in the shell procedure than to
keep it in a separate file.
'\"	For instance, one could type:
'\"	.DS I
'\"	cat <<- xx
'\"		This message will be printed on the
'\"		terminal with leading tabs and spaces
'\"		removed.
'\"	xx
'\"	.DE
'\"	This in-line input document feature is most useful 
'\"	in shell procedures.
'\"	Note that in-line input documents may not appear 
'\"	within grave accents.
.H 3 "Input\|/\|Output Redirection Using File Descriptors"
We mentioned above that a command occasionally 
directs output to some
file associated with a file descriptor other than 
1 or 2.
In languages such as C, 
one can associate output with any file descriptor
by using the
.IR write (2)
system call.
The shell provides its own mechanism 
for creating an output file associated
with a particular file descriptor.
By typing
.DS I
.B
\fIfd1\fP>&\fIfd2\fP
.R
.DE
where
.I fd1\^
and
.I fd2\^
are valid file descriptors, 
one can direct output that would normally
be associated with file descriptor
.I fd1\^
onto
the file associated with
.I fd2 .
The default value for
.I fd1\^
and
.I fd2\^
is 1.
If, at run time, no file is associated with
.I fd2 ,
then the redirection is void.
The most common use of this mechanism is that of directing
standard error output to the same file as standard output.
This is accomplished by typing:
.DS I
command 2>&1
.DE
If you wanted to redirect both standard output 
and standard error output
to the same file, you would type:
.DS I
command 1>file 2>&1
.DE
The order here is significant:
first, file descriptor 1 
is associated with 
.I file ;
then file descriptor 2 
is associated with the same file as is 
currently associated with file descriptor 1.
If the order of the redirections were reversed, standard
error output would go to the terminal, 
and standard output would go to
.FN file ,
because at the time of the error output redirection, 
file descriptor 1 
still would have been associated with the terminal.
.P
This mechanism can also be generalized to the redirection 
of standard input.
You could type
.DS I
fda<&fdb
.DE
to cause both file descriptors
.I fda\^
and
.I fdb\^
to be associated with the same input file.
If
.I fda\^
or
.I fdb\^
is not specified, file descriptor 0 is assumed.
Such input redirection is useful for a command that uses two or
more input sources.
.H 3 "Conditional Substitution"
Normally, the shell replaces occurrences of 
.RI $ variable 
by
the string value assigned to 
.I variable , 
if any.
However, there exists a special
notation to allow conditional substitution, dependent upon
whether the variable is set or not null.
By definition, a variable is
set if it has ever been assigned a value.
The value of a variable can be the null
string, which may be assigned to a variable in anyone of the
following ways:
.DS I
A=
bcd=""
efg=\'\'
set \'\' ""
.DE
The first three examples assign null to each of the
corresponding shell variables.
The last example sets the first
and second positional parameters to null.
The following conditional expressions depend upon
whether a variable is set and not null.
Note that the meaning of braces in these expressions differs 
from their meaning when used in grouping shell commands.
.I Parameter
as used below refers to either a digit or a variable name.
.VL 23 3
.LI ${\f2variable\^\fP:-\f2string\^\fP}
If
.I variable\^
is set and is
nonnull,
then substitute the value 
.RI $ variable
in
place of this expression.
Otherwise, replace the expression with \f2string\^\fP.
Note that
the value of
.I variable\^
is \f2not\^\fP changed by the evaluation of this expression.
.LI ${\f2variable\^\fP:=\f2string\^\fP}
If
.I variable\^
is set and is nonnull, 
then substitute the value 
$\f2variable\^\fP in place
of this expression.
Otherwise, set \f2variable\^\fP to \f2string\^\fP, and then
substitute the value 
.RI $ variable 
in place of this expression.
Positional parameters may not be assigned values in this fashion.
.LI ${\f2variable\^\fP:?\f2string\^\fP}
If
.I variable\^
is set and is
nonnull,
then substitute the value of \f2variable\^\fP for the expression.
Otherwise, print a message of the form
.DS I
variable:  string
.DE
and exit from the current shell.
(If the shell is the login shell, it is
not exited.)~
If
.I string\^
is omitted in this form, then the message
.DS I
variable:  parameter null or not set
.DE
is printed instead.
.LI ${\f2variable\^\fP:+\f2string\^\fP}
If
.I variable\^
is set and is nonnull,
then substitute
.I string\^
for this expression. 
Otherwise, substitute the null string.
Note that the value of
the variable is not altered by the evaluation of this expression.
.LE
.P
These expressions may also be used without the colon.
In this variation, the shell does not check whether
the variable is null or not; 
it only checks whether
the variable has ever been set.
.P
The two examples below illustrate the use of this
facility:
.AL 1 6
.LI
This example performs an explicit assignment to 
the PATH variable:
.DS I
.Q "PATH"=${PATH:\-\':/bin:/usr/bin\'}
.DE
This says, if PATH 
has ever been set and is not null,
then keep its current value;
otherwise, set it to the string 
.Q ":/bin:/usr/bin" .
.LI
This example automatically assigns the
HOME variable a value:
.DS I
cd ${HOME:=\'/usr/gas\'}
.DE
If HOME 
is set, and is not null, then change directory to it.
Otherwise set HOME to the given value 
and change directory to it. 
.LE
.H 3 "Invocation Flags"
There are four flags that may be specified on the command line
when invoking the shell.
These flags may not be turned on with the 
.B set 
command:
.VL 9 3
.LI \fB\-i\fR
If this flag is specified,
or if the shell's input and output are both attached
to a terminal, the shell is
.I interactive .
In such a shell, 
INTERRUPT (signal 2) is caught and ignored, and
TERMINATE (signal 15) and 
QUIT (signal 3) are ignored.
.LI \fB\-s\fR
If this flag
is specified or if no input/output redirection 
arguments are given, the
shell reads commands from standard input.
Shell output is written to
file descriptor 2.
The shell you get upon logging into the system has the
.B \-s
flag turned on.
.LI \fB\-c\fR
When this flag is turned on,
the shell reads commands from the first string
following the flag.
Remaining arguments are ignored.
Double quotation marks should
be used to enclose a multiword string, 
in order to allow for variable
substitution.
.LE
.bp
.H 2 "Effective and Efficient Shell Programming"
This section outlines strategies for writing
efficient shell procedures,
ones that do not unreasonably waste resources in
accomplishing their purposes.
The primary reason for choosing a shell procedure
to perform a specific function 
is to achieve a desired result at a minimum
human cost.
Emphasis should always be placed on simplicity, clarity,
and readability,
but efficiency can also be gained 
through awareness of a few design strategies.
In many cases, an effective redesign
of an existing procedure improves its efficiency by reducing its size,
and often increases its comprehensibility.
In any case, you should not worry about optimizing shell procedures 
unless they are intolerably slow or are known 
to consume an inordinate amount of a system's resources.
.P
The same kind of iteration cycle
should be applied to shell procedures as to other programs:
write code, measure it, 
and optimize only the \f2few\^\fP important parts.
The user should become familiar with the
.B time
command,
which can be used to measure both entire procedures and parts thereof.
Its use is strongly recommended;
human intuition is notoriously unreliable when used to estimate
timings of programs,
even when the style of programming is a familiar one.
Each timing test should be run several times,
because the results are easily disturbed by 
variations in system load.
.H 3 "Number of Processes Generated"
When large numbers of short commands are executed,
the actual execution time of the commands may well be dominated by
the overhead of creating processes.
The procedures that incur
significant amounts of
such overhead are those that perform much looping
and those that generate command sequences to be interpreted
by another shell.
.P
If you are worried about
efficiency,
it is important to know which commands
are currently built into the shell, and which are not.
Here is the alphabetical list of those that are built-in:
.DS I
break   case    cd        continue   eval
exec    exit    export    for        if
newgrp  read    readonly  set        shift
test    times   trap      umask      until
wait    while   .         :          {}
.DE
.P
Parentheses, (\|),
are built into the shell, but
commands enclosed within them are executed as a child process, 
i.e., the shell does a
.B fork ,
but no
.B exec .
Any command not in the above list requires both
.B fork
and
.B exec .
.P
The user should always have at least a vague idea of the number of
processes generated by a shell procedure.
In the bulk of observed procedures,
the number of processes 
created (not necessarily simultaneously) can be described by:
.DS I
processes = (k\^*\^n) + c
.DE
where
.I k
and
.I c
are constants,
and
.I n
may be the number of procedure arguments,
the number of lines in some input file,
the number of entries in some directory,
or some other obvious quantity.
Efficiency improvements are most commonly 
gained by reducing the value of
.I k ,
sometimes to zero.
.if t \{
Any procedure whose complexity measure includes
.I n\^~\v'-.4m'\s-32\s+3\v'.4m'
terms or higher powers of
.I n\^
is likely to be
intolerably expensive.\}
.P
As an example, here is an analysis
of a procedure named
.I split ,
whose text is given below:
.DS I
:	split
trap \'rm temp$$; trap 0; exit\' 0 1 2 3 15
start1=0 start2=0
b=\'[A\-Za\-z]\'
cat > temp$$	
		: read stdin into temp file
		: save original lengths of $1, $2
if test \-s "$1"
then start1=\`wc \-l < $1\`
fi
if test \-s "$2"
then start2=\`wc \-l < $2\`
fi
grep "$b" temp$$ >> $1	
		: lines with letters onto $1
grep -v "$b" temp$$ | grep \'[0-9]\' >> $2
		: lines with only numbers onto $2
total="\`wc \-l < temp$$\`"
end1="\`wc \-l < $1\`"
end2="\`wc \-l < $2\`"
lost="\`expr $total \- \e($end1 \- $start1\e) \e
\- \e($end2 \- $start2\e)\`"
echo "$total read, $lost thrown away"
.DE
.P
For each iteration of the loop,
there is one
.B expr
plus either an
.B echo
or another
.B expr .
One additional
.B echo
is executed at the end.
If
.I n
is the number of lines of input,
the number of processes is 2\^\(**\^n\|\^+\|1.
On the other hand,
the number of processes 
in the following (equivalent) procedure is
12, regardless of the number of lines of input:
.P
Some types of procedures should
.I not
be written using the shell.
For example, if
one or more processes are generated for each character in some file,
it is a good indication that the procedure 
should be rewritten in C.
.P
Shell procedures should not be used 
to scan or build files a character at a time.
.H 3 "Number of Data Bytes Accessed"
It is worthwhile considering any action that 
reduces the number of bytes read or written.
This may be important for those
procedures whose time is spent 
passing data around among a few processes,
rather than in creating large numbers of short processes.
Some filters shrink their output,
others usually increase it.
It always pays to
put the
.I shrinkers\^
first when the order is irrelevant.
For instance, the second of the following examples 
is likely to be faster:
.DS I
sort file | grep pattern
grep pattern file | sort
.DE
.H 3 "Directory Searches"
Directory searching can consume a great deal of time,
especially in those applications that
utilize deep directory structures and long pathnames.
Judicious use of
.B cd
can help shorten long pathnames and thus reduce the number of
directory searches needed.
As an exercise, try the following commands 
(on a fairly quiet system):
.DS I
time sh -c \'ls \-l /usr/bin/* >/dev/null\'
time sh -c \'cd /usr/bin; ls \-l * >/dev/null\'
.DE
.H 3 "Directory-Search Order and the  PATH Variable"
The PATH variable is a convenient mechanism
for allowing organization and sharing of procedures.
However, it must be used in a sensible fashion,
or the result may be a great increase in system overhead that
occurs in a subtle, but avoidable, way.
.P
The process of finding a command involves reading every directory
included in every pathname that precedes the needed pathname in the
current PATH 
variable.
As an example, consider the effect of invoking
.B nroff
(i.e., 
.FN /usr/bin/nroff ) 
when the value of  PATH is
.Q ":/bin:/usr/bin" .
The sequence of directories read is: 
.DS I
\&. 
/ 
/bin 
/
/usr 
/usr/bin 
.DE
.P
This is a total of six directories.
A long path list assigned to PATH
can increase this number significantly.
.P
The vast majority of command executions are of commands found in
.FN /bin
and, to a somewhat lesser extent, in
.FN /usr/bin .
Careless PATH setup may lead to
a great deal of unnecessary searching.
The following four examples are ordered from worst to best
(but
.I only\^
with respect to the efficiency of command searches):
.DS I
:/usr/john/bin:/usr/lbin:/bin:/usr/bin
:/bin:/usr/john/bin:/usr/lbin:/usr/bin
:/bin:/usr/bin:/usr/john/bin:/usr/lbin
/bin::/usr/bin:/usr/john/bin:/usr/lbin
.DE
.P
The first one above should be avoided.
The others are acceptable and the choice among them is
dictated by the rate of change in the set of commands
kept in
.FN /bin
and
.FN /usr/bin . 
.P
A procedure that is expensive because it invokes many
short-lived commands may often be speeded up by setting the 
PATH variable inside the procedure so that the
fewest possible directories are searched in an optimum order. 
.H 3 "Good Ways to Set Up Directories"
It is wise to avoid directories that are larger than necessary.
You should be aware of several
special sizes.
A directory that contains entries 
for up to 30 files (plus the required
.B \&.
and
.B \&.. )
fits in a single disk block and can be searched very efficiently.
One that has up to 286 entries is still a small file;
anything larger is usually a 
disaster when used as a working directory.
It is especially important to keep
login directories small, preferably one block at most.
Note that, as a rule, directories never shrink.
