/*
 * i215.h
 *	215/218/220 Driver declarations.
 *
 * Details of the 215/218 hardware may be found in Intel manual order
 * number 121593-002 (215), 121583-001 (218).
 *
 * Derived from iRMX86 version of the same thing.
 *
 * Written by Bob Beck, 1-24-82.
 * Modified 2-12-82 for inclusion of "format" stuff.
 * Modified 2-19-82 for use of different sector-sizes.
 * Modified 2-23-82 for new configuration:
 *	partitions based on sector #
 *	partition table per drtab entry
 *	config-table lists pointers to drtab's per unit.
 * Modified 3-2-82 for more efficient int-level -> board mapping.
 * Modified 3-3-82 for enhanced format ioctl.
 *
 *
 *	Modifications.......
 *
 *	10-5-82		giz		I003
 *		Changed N215 define to support change I003
 *		in i215.c. This change is to support binary 
 *		configuability of 3 215 like devices.
 *
 */


 
#define	SPL		 spl5		/* for driver mutex */
#define	I215RETRY	 10		/* retry count */
#define	NUMSPINDLE	 12		/* # spindles per board */
#define	FIRSTREMOV	 4		/* first removable unit-number */
#define NEXT_REMOVE_UNIT 4		/* increment to next removable unit */
#define	FIXEDMASK	 (FIRSTREMOV-1)	/* mask for fixed-unit given unit */
#define UNIT_MASK	 0x30		/* I004 unit field of cib status byte */

/*
 * Encoding of "minor" device number (8-bits).
 *
 * Units (for 215/218).  Note: 8 different spindles.
 *	0-3	Wini
 *	4-7	Floppy
 *
 * Units (for 220).  Note: fixed/removable share spindle (4 spindles).
 *	0-3	Fixed
 *	4-7	Removable
 *
 * Drtab's are selected per unit.  See i215cfg, below.
 * Partitions are selected per drtab entry.  See i215cdrt, below.
 *
 * Major device number selects which 215/220 controller.  All such
 * controllers must be contiguous in the bdevsw/cdevsw tables.  The
 * boards must live in the SAME indicies in these tables, due to the
 * dev -> board mapping in the driver.  i215fmaj holds this index.
 *
 * Note that <drtab,unit> number selects actual drtab entry.
 */

struct	i215minor {
	unsigned partition: 4;		/* was 2 I003 partition number */
	unsigned drtab:4;		/* was 3 I003 alternate drtab's */
	unsigned unit: 4;		/* was 3 I003 unit number */
	unsigned board:4;		/* was 0 I003 board number */
};

#define	UNIT(dev)	(i215minor[minor(dev)].unit)	/* dev -> unit# map I003 */
#define	DRTAB(dev)	(i215minor[minor(dev)].drtab)	/* dev -> drtab-index map I003 */
#define	PARTITION(dev)	(i215minor[minor(dev)].partition)/* dev -> partition-index map I003 */
#define BOARD(dev)	(i215minor[minor(dev)].board)	/* board number I003 */
#define i215MINOR(bnum,unum,drnum,panum) ((bnum<<12)|(unum<<8)|(drnum<<4)|panum)	/* I003 used in c215.c */


#define LHWORD(secnum)  (LOW(secnum),HIGH(secnum)) /* I004 c order problem fix for user ease in c215.c */
#define	LOW(x)		((x)&0xFF)		/* "low" byte */
#define	HIGH(x)		(((x)>>8)&0xFF)		/* "high" byte */

/*
 * Partition structure.  One per drtab[] entry.
 */

struct	i215part {
	daddr_t	p_fsec;			/* first sector */
	daddr_t	p_nsec;			/* number sectors */
};

/*
 * Per-board configuration.  Two of these per 215/218/217 or 220.
 * c_devcod indicates what kind of device/floppies/tape are there.
 * This should be:
 *	DEV8FLPY	For 8" floppies on a 215
 *	DEV5FLPY	For 5.25" floppies on a 215
 *	DEV220		For a 220
 *	STREAMER	streamer tape (Archive)
 *	STARTSTOP	start/stop tape (Kennedy)
 *
 * The c_drtab field is a pointer to a list of drtab entries per-unit.  A zero
 * value implies non-existant unit (ie, must have a drtab entry for a unit to
 * use the unit).
 */

struct	i215cfg	{
	unsigned	c_wua;			/* Physical Wake-Up Address */
	char		c_devcod[(NUMSPINDLE/FIRSTREMOVE - 1)];		/* what flavor of 215/218 or 220 */
	char		c_level;		/* what interrupt level */
	struct i215cdrt	*c_drtab[NUMSPINDLE];	/* per-spindle pointer to drive-characteristics table */
};

/*
 * Per-board driver "dynamic" data.
 */

struct	i215state {
	char		s_mod;			/* I004 modifier to allow no ram clearing on wini */
	char		s_exists;		/* flag that board exists */
	char		s_state;		/* what just finished (for interrupt) */
	char		t_error;		/* errors from an ioc call */
	int		t_flags;		/* I004 tape flags */
	char		t_state;		/* I004 tape state */
	char		s_opunit;		/* current unit being programmed */
	char		s_level;		/* what interrupt level (for i215io) */
	unsigned	s_wua;			/* copy of i215cfg.c_wua */
	char		s_flags[NUMSPINDLE];	/* flags per spindle; see below */
	char		s_popen[NUMSPINDLE];	/* bit[i] ==> partition[i] open */
	char		s_devcod[NUMSPINDLE];	/* device-code for iopb */
	char		s_unit[NUMSPINDLE];	/* "unit" code for iopb */
	char		s_init[NUMSPINDLE];	/* status from init op */
	struct i215cdrt	**s_cdrtab;		/* -> i215cfg.c_drtab.c_drtab */
	struct buf	*s_bufh;		/* -> buffer header */
	unsigned	s_hcyl;			/* hold cylinder # during restore */
};

/*
 * Per-Unit State Flags.
 */

#define	SF_OPEN		0x01			/* unit is open */
#define	SF_READY	0x02			/* unit is ready; reset by media-change */

/*
 * Per-board tape Flags
 */
#define TF_START        0x01                    /* a call to the start proc is needed */
#define TF_IOC_WAIT	0x02			/* waiting for the device */
#define TF_STATUS_CHK	0x04			/* need to check hardware */
#define TF_LOAD_TAPE	0x08			/* load tape on initialize */
#define TF_UNLOAD_TAPE  0x10			/* un load the tape for close */
#define TF_RESTART	0x20			/* have tape req waiting start*/
#define TF_IOWAITING	0x40			/* io proc waiting on an interrupt */
#define TF_CLOSING	0x80			/* closing a tape device */
#define TF_WIOC		0x100			/* waiting for device to finish in ioc */
#define TF_QUE 		0 			/* add to the queue flag */
#define TF_DEQUE 	1 			/* put the queue back flag */

/*
 * Macros to make things easier to read/code/maintain/etc...
 */

#define	IS220(dd)	((dd)->d_state.s_devcod[0] == DEV220)
#define	IO_OP(bp)	((bp->b_flags&B_READ) ? READ_OP : ((bp->b_flags&B_FORMAT) ? FORMAT_OP : WRITE_OP))
#define ISTAPE(dd,unit)    (((dd)->d_state.s_devcod[unit] == STARTSTOP) || ((dd)->d_state.s_devcod[unit] == STREAMER))    /* I004 */
#define NOT_BTAPE(dev)	(UNIT((dev)) >= FIRSTREMOV + NEXT_REMOVE_UNIT)

/*
 * 215 Wake-Up Block.  Lives at wakeup-address, points at CCB.
 */

struct	i215wub {
	char		w_sysop;	/* Must == 0x01 */
	char		w_rsvd;		/* reserved */
	struct i215ccb	*w_ccb;		/* "offset" of CCB pointer */
	unsigned	w_ccb_b;	/* "base" == Kernel DS == 0 */
};

/*
 * CCB (Channel-Control-Block).  See 215 manual.
 */

struct	i215ccb {
	char		c_ccw1;		/* 1 ==> Use 215 Firmware */
	char		c_busy1;	/* 0x00 ==> Idle, 0xFF ==> busy */
	struct i215cib	*c_cib;		/* "offset" of CIB pointer */
	unsigned	c_cib_b;	/* "base" == Kernel DS == 0 */
	unsigned	c_rsvd0;	/* reserved */
	char		c_ccw2;		/* Must == 0x01 */
	char		c_busy2;	/* Not useful to Host */
	unsigned	*c_cpp;		/* -> i215ccb.c_cp */
	unsigned	c_cpp_b;	/* "base" == Kernel DS == 0 */
	unsigned	c_cp;		/* Control Pointer == 0x04 */
};

/*
 * CIB (Channel Invocation Block).  See 215 manual.
 */

struct	i215cib {
	char		c_cmd;		/* reserved */
	char		c_stat;		/* Operation Status (see below) */
	char		c_cmdsem;	/* Not used by 215 */
	char		c_statsem;	/* 0xFF ==> new status avail */
	unsigned	c_csa[2];	/* 215 Firmware; MUST == 0 */
	struct i215iopb	*c_iopb;	/* IOPB pointer */
	unsigned	c_iopb_b;	/* "base" == Kernel DS == 0 */
	unsigned	c_rsvd1[2];	/* reserved */
};

/*
 * IOPB (I/O Parameter Block).  See 215 manual.
 */

struct	i215iopb {
	unsigned	i_rsvd[2];	/* reserved */
	unsigned	i_actual;	/* actual transfer count */
	unsigned	i_actfill;	/* fill actual to 32-bits; Unused */
	unsigned	i_device;	/* Device Code (see below) */
	char		i_unit;		/* Unit: <4> == fixed/rem, <1,0> == unit # */
	char		i_funct;	/* Function Code (see below) */
	unsigned	i_modifier;	/* Modifier.  0 ==> normal, interrupt */
	unsigned	i_cylinder;	/* starting cylinder # */
	char		i_head;		/* starting head # */
	char		i_sector;	/* starting sector # */
	char		*i_buffp;	/* physical offset of buffer */
	unsigned	i_buffp_b;	/* physical base of buffer */
	unsigned	i_xfrcnt;	/* Requested Xfr Count */
	unsigned	i_cntfill;	/* count fill.  Unused */
	unsigned	i_gaddr_ptr[2];	/* general address ptr (not-used) */
};

/*
 * Drive-Data Table (used to initialize drives).  See 215 manual.
 * Note allignment problem on secsiz.
 * Note: fields thru dr_nalt are programmed into controler for an init.
 *	 Other fields are for more efficient programming.
 * The i215cdrt structure is for configuring the same data (no allingment
 *	worries).
 */

struct	i215drtab {
	unsigned	dr_ncyl;	/* # cylinders */
	char		dr_nfhead;	/* # fixed heads */
	char		dr_nrhead;	/* # removable heads */
	char		dr_nsec;	/* # sectors per track */
	char		dr_lsecsiz;	/* "low" of sector-size */
	char		dr_hsecsiz;	/* "high" of sector-size */
	char		dr_nalt;	/* # alternate cylinders */
					/* if floppy, 0==FM, 1==MFM */
	unsigned	dr_spc;		/* actual sectors/cylinder */
	unsigned	dr_spb;		/* sectors/block */
	unsigned	dr_secsiz;	/* sector-size (bytes) */
	struct i215part	*dr_part;	/* partition table pointer */
};

struct	i215cdrt {
	unsigned	cdr_ncyl;	/* # cylinders */
	char		cdr_nfhead;	/* # fixed heads */
	char		cdr_nrhead;	/* # removable heads */
	char		cdr_nsec;	/* # sectors per track */
	unsigned	cdr_secsiz;	/* sector-size */
	char		cdr_nalt;	/* # alternate cylinders */
	struct i215part	*cdr_part;	/* partition table pointer */
};

/*
 * Error Status-Structure, Returned on status inquiry.  See 215 manual.
 * Only 1st 2 fields used.  Note another allignment problem (ignored).
 */

struct	i215err {
	unsigned	e_hard;		/* Hard Error Status (see below) */
	char		e_soft;		/* soft error status */
	unsigned	e_req_cyl;	/* desired cylinder */
	char		e_req_head;	/* desired head and volume */
	char		e_req_sec;	/* desired sector */
	unsigned	e_act_cyl;	/* actual cylinder & flags */
	char		e_act_head;	/* actual head & volume */
	char		e_act_sec;	/* actual sector */
	char		e_retries;	/* # retries attempted */
};

/*
 * Format Structure.  1 per "board", usage mutexed via use of "raw" buffer-
 * header (see i215.c).
 * i215ftk is the argument structure to the format ioctl.
 */

struct	i215format {
	char	f_trtype;		/* format track-type code */
	char	f_pattern[4];		/* pattern; depends on f_trtype */
	char	f_interleave;		/* interleave-factor */
};

struct	i215ftk	{
	int	f_track;		/* track # */
	int	f_intl;			/* interleave factor */
	int	f_skew;			/* track skew -- ignored by 215 */
	char	f_type;			/* format type-code */
	char	f_pat[4];		/* pattern data */
};

/*
 * 215 Per-Board Device-Data.  One per board (declared in driver).
 */

struct	i215dev {
	struct	i215state	d_state;
	struct	i215ccb		d_ccb;
	struct	i215cib		d_cib;
	struct	i215iopb	d_iopb;
	struct	i215drtab	d_drtab[NUMSPINDLE];
	struct	i215err		d_error;
	struct	i215format	d_format;
};

/*
 * Values of buffer-header b_active, used for mutual-exclusion of
 * opens and other IO requests.
 */

#define	IO_IDLE		0		/* idle -- anything goes */
#define	IO_OPEN_WAIT	1		/* open waiting */
#define	IO_BUSY		2		/* something going on */
#define IOC_WAIT 	3 		/* waiting for the device */
#define	B_FORMAT	040000		/* "new" buf.h flag: must NOT overlap buf.h! */

/*
 * Values of i215state.s_state, internal driver state.
 */

#define	NOTHING			0	/* normal situation, RW */
#define	GET_BAD_STATUS		1	/* retrieveing status; last cmd got error */
#define	RESTORING		2	/* seeking to track 0 for retry */
#define	INITIALIZING		3	/* going thru init-sweep */
#define	READING_LABEL		4	/* reading label for device-characteristics */
#define FORMAT0                 5	/* unused */
#define FORMAT1                 6       /* unused */
#define FORMAT2                 7       /* unused */
#define FORMAT3                 8       /* unused */
#define TAPE_DONE		9	/* I004 aux tape done state */

/*
 * I004 tape state variables for 
 * state.t_state.
 */
#define T_BOTTIME		0	/* tape not ready state */
#define T_INIT			1	/* initialing controller */
#define T_RESET 	        2	/* reseting drive */
#define T_INIT_DONE		3	/* end of tape initializing */
#define T_BUSY			4	/* tape device is doing something */
#define T_LONG_TERM		5	/* long term command in progress */
#define T_READY			6	/* tape device ready for next cmd */
#define T_CLOSE			7	/* tape device is blocked for loading */


/*
 * Tape exclusion variable.
 * This variable will prevent multiple opens
 * on tape units. This is here because the Archive
 * drive will not allow access to a second unit if the
 * first unit is not at the logical load point.
 */
#define T_ACTIVE		1	/* block all tape activity */

#define TAPE_BAD_STATUS         13      /* to print out debug info */

#define	T0CHANGING		0x80	/* Track 0 is changing */

/*
 * IOPB fields/flags definitions.
 */

#define	UNIT_REMOVABLE		0x10	/* ==> removable unit */

/*
 * 215 Wake-up command codes.  These get output to the wakeup-address-port.
 */

#define	WAKEUP_CLEAR_INT	0x0000
#define	WAKEUP_START		0x0001
#define	WAKEUP_RESET		0x0002

/*
 * 215 IOPB Command Codes.
 */

#define INIT_OP                 0
#define	STATUS_OP		1
#define	FORMAT_OP		2
#define	READ_ID_OP		3
#define	READ_OP			4
#define	VERIFY_OP		5
#define	WRITE_OP		6
#define	WRITE_BUFFER_OP		7
#define	SEEK_OP			8
/*
 * I004 iopb commands for tape only. These commands work only
 * on the iSBC 215G series of controller boards which supports
 * the iSBX 217 tape controller.
 *
 * l.t. (long term command)
 * s.t. (short term command)
 */
#define TAPEINIT_OP		0x10	/* s.t. initialize 217 firmware */
#define REW_OP			0x11	/* l.t. tape rewind */
#define SFFM_OP			0x12	/* l.t. forward a file mark */
#define SBFM_OP			0x13	/* l.t. backward a file mark */
#define WRFM_OP			0x14	/* s.t. write filemark */
#define ERASETAPE_OP		0x17	/* l.t. erase tape (format command) */
#define LOADTAPE_OP		0x18	/* l.t. tape to logical load point */
#define UNLOADTAPE_OP		0x19	/* l.t. tape to physical end of tape */
#define SFREC_OP		0x1A	/* s.t. forward a record */
#define SBREC_OP		0x1B	/* s.t. backward a record */
#define TAPERESET_OP		0x1C	/* s.t. reset tape drive */
#define RETTAPE_OP		0x1D	/* l.t. retention tape */
#define TAPE_STATUS_OP		0x1E	/* s.t. get long term status info */


#define FIRST_TAPE_OP		TAPEINIT_OP	/* I004 first tape command code */

/*
 * 215 IOPB Modifier Bits.
 */

#define	MOD_NO_INT		0x0001	/* no interrupt */
#define	MOD_NO_RETRY		0x0002	/* no retry attempts */
#define	MOD_DELETED_DATA	0x0004	/* 218 deleted-data RW */
#define MOD_RESERVED		0x0008	/* reserved don't use */
#define MOD_24_BIT_ADDR		0x0010	/* set 215 to 24 bit address mode */
#define MOD_NO_CLEAR		0x0020	/* 215G no clear the ram on init bit */
#define MOD_LT_STATUS		0x0040	/* 215G tape status for long command */

/*
 * Device Codes (for iopb.i_device).
 */

#define	DEVWINI		0		/* Wini */
#define	DEV8FLPY	1		/* 8" 218 Floppy */
#define	DEV220		2		/* 220 */
#define	DEV5FLPY	3		/* 5.25" 218 Floppy */
#define STREAMER	4		/* streamer tape (Archive)  */
#define STARTSTOP	5		/* start/stop tape (Kennedy) */
#undef INVALID
#define INVALID		0xF		/* invalid device code */


/*
 * Floppy FM/MFM codes for drtab[*].nalt.
 */

#define	FLPY_FM		0		/* FM -- single density */
#define	FLPY_MFM	1		/* MFM -- double density */

/*
 * Operation Status Bits.  Returned by controller in i215cib.c_stat.
 *
 * I004 Note: the 215 controller has two additional bit patterns
 * for tape identification. 0x?f for tape long term command complete
 * and 0x?e for media change. Since we will need to treat long term
 * command completion diffently any way it has been defined as a byte mask
 * instead of a bit mask.
 */		

#define	ST_OP_COMPL		0x01	/* operation complete */
#define	ST_SEEK_COMPL		0x02	/* seek complete */
#define	ST_MEDIA_CHANGE		0x04	/* media changed */
#define	ST_FLOPPY		0x08	/* ==> 218 floppy */
#define	ST_UNIT			0x30	/* unit mask */
#define	ST_HARD_ERR		0x40	/* 0 ==> was soft, recovered error */
#define	ST_ERROR		0x80	/* summary error */
#define ST_TAPE_MEDIA		0x0E	/* I004 tape media change detected */
#define ST_LONG_COMPL		0x0F	/* I004 long term command and complete */
#define TAPE_NOT_READY		0x6457	/* bits in hard status that lock tape */

/*
 * Error Bits.
 *
 * Errors returned to user in b_error (byte).  Error is either soft-status
 * byte, or high-byte of hard-status byte.  b_error needs to be a word,
 * and can be used as:
 *	Bits	Contents
 *	 6-0	EIO
 *	  7	0 ==> Hard, 1 ==> Soft status
 *	15-8	High-order byte of hard status, or soft status byte.
 */

#define	HARD_WRITE_PROT		0x8000	/* write-protected drive */
#define	HARD_NOT_READY		0x4000	/* went not-ready */
#define	HARD_NO_SECTOR		0x1000	/* couldn't find sector */
#define	SOFT_NO_SECTOR		0x0001	/* set in soft status, "reserved" bit */

/*
 * Misc Format definitions, for i215ftk.f_type.
 */

#define	FORMAT_DATA		0x00	/* format data track */
#define	FORMAT_BAD		0x80	/* format bad track */
#define	FORMAT_ALTERNATE	0x40	/* format alternate track */

/*
 * iSBC 215 ioctl mnemonics.
 */

#define	I215_IOC_FMT		(('W'<<8)|0)
/*
 * Tape ioctl mnemonics I004
 */
#define I215_REW		 (('W'<<8)|1)
#define I215_SFFM		 (('W'<<8)|2)
#define I215_SBFM		 (('W'<<8)|3)
#define I215_RETTAPE		 (('W'<<8)|4)
#define I215_SFREC		 (('W'<<8)|5)
#define I215_SBREC		 (('W'<<8)|6)
#define I215_WRFM		 (('W'<<8)|7)
#define I215_ERASE		 (('W'<<8)|8)
