/*
 * mmu.h
 *	Memory Management Definitions
 *
 * used by mmu.c
 * supports scatter-loading of individual processes
 * param.h has basic page-size/etc definitions.
 *
 * Mods for iSBC 308/309 by Bob Beck, 12-15-81.
 * Mods for DS = 0 12-30-81.
 */

extern	short	mm_nfree;		/* number of free pages */
extern	short	mm_free;		/* first free page */
extern	short	mm_pages[];		/* page list */
extern	short	mm_size;		/* # mem pages, initialized to NCOREL */

/*
 * Bit masks for the contents of mm_pages
 */

#define MM_NUMBER	0x3FFF		/* normally, next page in linked list */
#define MM_STMASK	0xC000		/* status bits */

/*
 * status values -- It is important the MM_FREE not be 0,
 * so mm_pages is automatically loaded with all pages busy.
 */

#define MM_LINK		0x0000		/* general in-use */
#define MM_FREE		0x8000		/* on free list */
#define MM_END		0x4000		/* end of list. MM_NUMBER holds index
					   into proc array of proc structures */
#define MM_TEXTEND	0xC000		/* end of text. MM_NUMBER holds index
					   into text array of text structures */

/*
 * If the MM_FREE bit is set, then (mm_pages[i] & MM_NUMBER) yields
 * the page number of the next page on the core free list.
 * The core free list is null-terminated.
 *
 * If the MM_LINK bit is set, then (mm_pages[i] & MM_NUMBER) is
 * used as part of a linked list of the pages allocated to
 * a particular process.  (Pages used by the kernel are marked
 * MM_LINK | 0.)
 *
 * The conventions for the use of the MM_NUMBER field of each page
 * marked with MM_END or MM_TEXTEND may be changed.  The intention
 * here is to be able to trace any page back to the process or
 * text structure which "owns" it.
 */

#define NPNUMB	8		/* number of address spaces (program numbers)	*/
#define NPBANK  4               /* # of 64K banks per addr space */
#define NPSEGM	0		/* no segments */
#define MMPAGE	1		/* Page (Central Data) type mmu */

#define NDPAGE    NPAGEPS       /* pages per D space */
#define NIPAGE    90            /* pages per I space */
#define	MAXPAGE	 0x01FF		/* largest valid physical page number */
#define	MAXKPAGE 0x001F		/* max kernel data page */

/*
 * Segment I/O Ports
 *
 * The MMU (308/309) has 1024 2K pages that map into a 1M address space.
 * The pages are divided into 8 "maps" of 256K each.  Each map contains
 * 4 64K banks, that must be individually selected for programming.
 * A process is allocated a map while it runs.  A user process is only
 * allowed the use of 128K of the map space, the remainder being used
 * for the convenience of the kernel.  Each 2K page is mapped via a one
 * word value, giving the physical 2K page number and the R/W permissions.
 * Map[0] is permanently allocated to the kernel address space.  All user
 * process maps must share the low-order pages (read-only) with the kernel,
 * to allow the use of interrupt vectors and some "cannonical" kernel code
 * that is used during process switching and trap handling.
 *
 * The kernel address space (map[0]) is given below.  Any un-mapped portions
 * of the kernel address space are made invalid.
 *
 *	logical-page	mode	physical-page	usage
 *	    0		 RW	     0		Interrupt Vectors and 957 Data
 *          1            RW          1          buf[0] and buf[1]
 *	  2 - n(<=31)    RW	   2 - n	Kernel Data (DS = 0)
 *	    30		 RW	  Variable	Kernel Utility Page
 *	    31		 RW	   U-area	User Process Data
 *	 32 - 32+m(<=31) RO	 n+1 - n+1+m	Kernel Code (CS = 1000)
 *	 64 - 64+15	 RW	  Variable	Kernel "from" work area
 *	 80 - 80+15	 RW	  Variable	Kernel "to" work area
 *
 *       96 - 96+27      --         None        Invalid (unused) - normal
 *
 *       96 - 96+8       --       Variable      Histogram counts
 *      104 - 104+19     --         None        Invalid (unused)
 *
 *	124 - 127	 RO	 508 - 511	957 Monitor Code/bootstrap
 *
 * The layout of the first few kernel data pages is trickey.  The interrupt
 * vectors cover the first 1K of page 0, the 957 reserves the next 1K.
 * The second 2k page is available, but the first several bytes of the
 * third page are reserved for device wake-up blocks (WUBs).  These addrs
 * are hard-wired into the controllers.  Unfortunately theres no way to
 * tell the current linker to make sure that the WUB structure ends up
 * in the right place so we have wub.s which .bss-es 4K then defines the
 * WUBs.  This is linked first in the /xenix image thus positioning the
 * WUBs correctly.  We keep the 2nd 2K page from being wasted by
 * using it as 2 buffers.
 * There is a special hack in the boot so that it omits loading /xenix
 * into that first 4K (thus preserving the 957s data).  This means that
 * the first 4K of /xenix data are 0.
 *
 * An active user process map is illustrated below.  Any un-mapped portions
 * of the user address space are made invalid.  Note that the entire kernel
 * text and vectors is RO in the user process; this is overkill, but allows
 * mch.a86 to grow without worry about the top address of the "cannonical"
 * procedures.
 *
 *	logical-page	mode	physical-page	usage
 *	    0		 RO	     0		Kernel Vectors
 *          1            --         None        Invalid (unused)
 *          2            RO          2          Kernel mch.s data routines{
 *        3 - 34         RW       Variable      User data (DS = 0180)
 *       35 - 127        RO       Variable      User Ttex (CS = 1180)
 *
 *        Note that pages 96-103 are mapped RW to the histogram area
 *        if histogramming.
 *
 * User pages 96-127 will be identical to 64-95 if the code is not split
 * I/D.  User pages 96-127 map the user data + stack; a hole exists in
 * between, with the stack at the high addresses, and they grow together.
 */

/*      Predefined Map Numbers  */

#define	MM_KMAP		0	/* pre-allocated kernel map */
#    define MMK_WUB     1       /* Wake-Up Block "page" in KMAP */
			/* maps 1 through 7 allocated to user procs */



#define	MM_BSTACK	0x1000	/* top of temp boot stack */

/*
 * Segment Registers
 */
 
#define	SEGKI	0x1000		/* Kernel CS */
#define	SEGKD	0x0000		/* Kernel DS */
#define SEGUI   0x1180          /* User CS */
#define SEGUD   0x0180          /* User DS */
#define SEGCF   0x2000          /* Kernel "from" page-pair */
#define SEGCT   0x2100          /* Kernel "to" page-pair */
#define SEGFU   0x2200          /* fubyte/fuword working page-pair */
#define SEGSU   0x2300          /* subyte/suword working page-pair */
#define SEGSF   0x2400          /* [fs]u[il]word/byte working page-pair */

#define	SEGCOPYMAX	16	/* max 16 pages (32K) for from/to */

/*
 * I/O Ports
 *
 * MM_PROC format:
 *	Bits	Meaning
 *	15-6	ignored
 *	  5	enable mapping
 *	 4-2	map number (0-7)
 *	 1-0	64K bank number
 *
 * Page Port format:
 *	Bits	Meaning
 *	15-11	low-order physical-page number (bits 15-11 of phys addr)
 *	10-8	ignored
 *	 7	SR
 *	 6	TR
 *	 5	read permission
 *	 4	write permission
 *	3-0	high order physical page number (bits 19-16 of phys addr)
 *
 * <SR,TR> valid on input from any page-port; meaning:
 *	00	user mode or after clear status
 *	01	exception occured (MMU violation)
 *	10	maskable interrupt occured
 *	11	system call occured
 *
 * Page-port and MM_PROC port word formats could be more convenient for the
 * software as shown below.  Of the two, the page-port format is more painful
 * in its current form; Xenix maintains the physical page number as an
 * integer (0-511); this requires 4 instructions to fix to the proper form.
 *
 * Page Port format: (i.e., leave the physical page number unshifted)
 *	Bits	Meaning
 *	15-13	ignored
 *	12	SR
 *	11	TR
 *	10	read permission
 *	 9	write permission
 *	8-0	Physical Page Number
 *
 * MM_PROC format: (i.e., leave map # unshifted)
 *	Bits	Meaning
 *	15-6	ignored
 *	  5	enable mapping
 *	 4-3	64K bank number
 *	 2-0	map number (0-7)
 */

#define	PGZERO	0x400		/* Page 0 port */
#define	PGDELTA	0x800		/* delta between ports */
#define PGPORT(i) (PGDELTA*i+PGZERO)

#define	MM_PROC	 0x8000		/* select process/bank */
#  define MMP_ENA       0x0020         /* Bit to enable mapping in MM_PROC */
#define	MM_SCALL 0x8800		/* output ==> system call */
#define	MM_CLRST 0x8800		/* input ==> clear SR,TR flags */
#define	MM_SCSYS 0x8A00		/* input/output ==> set/clear system mode */


#define	MM_RD	0x0020		/* read permission */
#define	MM_WR	0x0010		/* write permission */
#define	MM_SR	0x0080		/* system-call request */
#define	MM_TR	0x0040		/* trap occured */

/*
 * MMU "segment"s for mapping.  This is the value the MMU wants to
 * select a paticular 64K bank.
 */

#define	SEGKDATA 0		/* kernel data */
#define	SEGKTEXT 1		/* kernel text */
#define	SEGKCOPY 2		/* kernel copy */
#define	SEGKMON  3		/* 957 monitor code */

#define	OFFWRKPG 0xF000		/* kernel data offset of work page (logical) */
#define	OFFUSRPG 0xF800		/* kernel data offset of U page (logical) */

#define MMSEG_D  3              /* flags to mmuppg() in mch.s */
#define MMSEG_T  35


/*
 * Important page ports.
 */

#define PGADRSD PGPORT(0)	/* first system data address register */
#define PGADRUT PGPORT(30)	/*	 system work address register */
#define PGADRUU PGPORT(31)	/*	 system u_   address register */
#define PGADRSI PGPORT(0)	/* first system instruction address register */
#define PGADRUD PGPORT(3)       /* first user data address register */
#define PGADRUI PGPORT(3)       /* first user instruction address register */
#define PGADRCF PGPORT(0)	/* first copybuf 'from' register */
#define PGADRCT PGPORT(2)       /* first copybuf 'to'   register */
#define PGADRFU PGPORT(4)       /* first fuword         register */
#define PGADRSU PGPORT(6)       /* first suword         register */
#define PGADRSF PGPORT(8)       /* first fuiword/suiword register */

/*
 * Misc constants.
 */

#define OFFMASK	(MMPGSZ-1)	/* develop offset within page from addr */
#define LLOWWORD 0x0000FFFF	/* low word of a long word */
#define LHIWORD	 0xFFFF0000	/* hi  word of a long word */
#define LHISHIFT 16		/* hi  word of a long word */

#define RO	MM_RD		/* tells mmuset to setup code to make */
#define RW	(MM_RD+MM_WR)	/* kernal access read-only, or read/write. */
