/*
 * 'MSM' disk driver - 67MB and 256MB
 */

#include "../h/param.h"
#include "../h/buf.h"
#include "../h/conf.h"
#include "../h/selch.h"
#include "../h/systm.h"
#include "../h/dskmap.h"

#define DN	3		/* disk number for monitoring */
#define NBPT	32		/* blocks per track */
#define NBPC	(msmsize? 608 : 160)	/* blocks per cylinder */
#define NMSMERR	10		/* number of error retries */

extern int	nmsm;		/* number of disks */
extern int	msmselch;	/* selch address for MSM */
extern int	msmcntl;	/* controller address */
extern char	msmaddr[];	/* drive addresses */
extern int	msmsize;	/* 0 => 67MB,  1 => 256MB */
extern struct dskmap msmmap[];	/* logical device mapping */

int msmseek(), msmscintr();

struct buf	msmtab;
struct buf	rmsmbuf;

struct selchq msmscq {
	&msmseek,
	&msmscintr,
	0
};

int	msmdrive;
int	msmcyl;
int	msmhead;
int	msmsector;
int	msmsad, msmead;


/* disk drive status & commands */
#define	ALTBSY	0x20
#define	UNS	0x10
#define UNREADY	0x08
#define	SINC	0x02
#define	OFFL	0x01

#define	DISABLE	0x80
#define ENABLE	0x40
#define	DISARM	0xc0
#define SETHEAD	0x20
#define	SETCYL	0x10
#define	SEEK	0x02
#define RELEASE	0xb0

/* disk controller status & commands */
#define	CNTL_UNRCV	0xc1
#define	CYL_OV		0x10
#define IDLE		0x02

#define	READ		0x01
#define	WRITE		0x02
#define	RESET		0x08

/*
 * Status in msmtab.b_active
 */
#define WSELCH	1	/* waiting for selch */
#define WALT	2	/* waiting for alternate channel release */
#define WSEEK	3	/* waiting for seek to complete */
#define WIO	4	/* waiting for I/O to complete */

/*
 *	disk strategy routine
 */
msmstrategy(bp)
register struct buf	*bp;
{
	bp->b_resid = bp->b_bcount;

	if ((minor(bp->b_dev)>>3) >= nmsm) {
		bp->b_flags |= B_ERROR;
		iodone(bp);
		return;
	}

	/*
	 * Block no. too high -- looks like EOF for raw read, error otherwise
	 */
	if (bp->b_blkno >= msmmap[minor(bp->b_dev)&07].dm_nblk) {
		if ((bp->b_flags&(B_PHYS|B_READ)) != (B_PHYS|B_READ))
			bp->b_flags |= B_ERROR;
		iodone(bp);
		return;
	}
	dk_numb[DN]++;
	dk_wds[DN] += (bp->b_bcount>>6);
	bp->av_forw = 0;
	spl5();
	if (msmtab.b_actf == 0)
		msmtab.b_actf = bp;
	else
		msmtab.b_actl->av_forw = bp;
	msmtab.b_actl = bp;
	if (msmtab.b_active == 0)
		msmstart();
	spl0();
}

/*
 * start next disk i/o operation
 *	- set up drive address, cylinder/head/sector address
 *	- set up memory start & end addresses
 *	- initiate seek
 */

msmstart()
{
	register struct buf *bp;
	register stat;
	register bn;

	if (!(bp = msmtab.b_actf))
		return;
	msmtab.b_active = WSELCH;
	dk_busy |= 1<<DN;

	trace(0x80<<16, "mstart", bp);

	msmdrive = msmaddr[minor(bp->b_dev)>>3];
	bn = bp->b_blkno + msmmap[minor(bp->b_dev)&07].dm_baddr;
	msmcyl = bn / NBPC;
	bn %= NBPC;
	msmhead = bn / NBPT;
	msmsector = (bn % NBPT)<<1;

	msmsad = bp->b_un.b_addr;
	msmead = bp->b_un.b_addr + bp->b_bcount - 1;

	selchreq(msmselch, &msmscq);
}

msmseek()
{
	register stat;

	trace(0x80<<16, "seek", msmdrive);
	trace(0x80<<16, "cyl", msmcyl);

	while ((ss(msmcntl)&IDLE) == 0)
		;
	trace(0x80<<16, "stat", ss(msmdrive));
	trace(0x80<<16, "stat", ss(msmdrive));
	if ((stat = ss(msmdrive))&ALTBSY) {
		trace(0x80<<16, "alt", stat);
		msmtab.b_active = WALT;
		oc(msmdrive, ENABLE);
		selchfree(msmselch);
		return;
	}
	trace(0x80<<16, "stat", stat);
	msmtab.b_active = WSEEK;
	wh(msmdrive, msmcyl);
	while (((stat=ss(msmcntl))&IDLE) == 0)
		;
	oc(msmdrive, SETCYL|DISARM);
	while (((stat=ss(msmcntl))&IDLE) == 0)
		;
	oc(msmdrive, SEEK|ENABLE);
	while (((stat=ss(msmcntl))&IDLE) == 0)
		;
	trace(0x80<<16, "stat", ss(msmdrive));
}

/*
 * disk interrupt routine
 *	-disk interrupts after a seek, or when Alternate Channel Busy clears
 *	-check status, initiate read/write or seek
 */

msmintr(dev,stat)
{
	register struct buf *bp;
	register scmd, ccmd;

	trace(0x40<<16, "interrupt", msmaddr[dev]);
	trace(0x40<<16, "status", stat);

	if (!(bp = msmtab.b_actf))
		return;
	switch (msmtab.b_active) {
		case WALT:
			msmtab.b_active = WSELCH;
			selchreq(msmselch, &msmscq);
			return;

		case WSEEK:
			msmtab.b_active = WIO;
			break;

		default:
			return;
	}

	if (stat & (UNS|UNREADY|SINC|OFFL|ALTBSY)) {
		msmerror(bp, stat, msmdrive);
		return;
	}

	if (bp->b_flags & B_READ) {
		scmd = READ_GO;
		ccmd = READ|ENABLE;
	} else {
		scmd = GO;
		ccmd = WRITE|ENABLE;
	}

	trace(0x80<<16, "mcmd", ccmd);
	trace(0x80<<16, "head", msmhead);
	trace(0x80<<16, "sector", msmsector);

	oc(msmselch, STOP);
	trace(0x80<<16, "bufstart", msmsad);
	trace(0x80<<16, "bufend", msmead);
	wdh(msmselch, msmsad);
	wdh(msmselch, msmead);
	wh(msmdrive, msmhead);
	oc(msmdrive, SETHEAD|DISARM);
	while ((ss(msmcntl)&IDLE) == 0)
		;
	wd(msmcntl, msmsector);
	wh(msmcntl, (msmhead<<10)+msmcyl);
	wh(msmdrive, msmhead);
	oc(msmdrive, SETHEAD|DISARM);
	while ((ss(msmcntl)&IDLE) == 0)
		;
	oc(msmcntl, ccmd);
	oc(msmselch, scmd);
}

/*
 * selch interrupt routine
 *	-selch interrupt will be followed by a controller interrupt
 *		( nothing to do here? )
 */

msmscintr(dev, stat)
{
	register struct buf *bp;

	oc(msmselch, STOP);
}

/*
 * disk controller interrupt routine
 *	-check ending status, signal i/o done
 */

msmcintr(dev, stat)
{
	register struct buf *bp;
	register struct dskmap *dm;

	trace(0x40<<16, "interrupt", msmcntl);
	trace(0x40<<16, "status", stat);


	if (!(bp = msmtab.b_actf))
		return;
	if (msmtab.b_active != WIO)
		return;

	if (stat & CNTL_UNRCV) {
		oc(msmcntl, RESET);
		msmerror(bp, stat, msmcntl);
		return;
	}
	bp->b_resid = 0;

	/*
	 * Cylinder overflow
	 */
	if (stat & CYL_OV) {
		oc(msmcntl, RESET);
		oc(msmselch, STOP);
		msmsad += (rdh(msmselch)-msmsad+2)&~0377;
		trace(0x40<<16, "cyl ov", msmsad);
		msmsector = msmhead = 0;
		bp->b_resid = msmead - msmsad;
		dm = &msmmap[minor(bp->b_dev)&07];
		if ((++msmcyl*NBPC) < dm->dm_baddr + dm->dm_nblk) {
			msmseek();
			return;
		} else
			if ((bp->b_flags&(B_PHYS|B_READ)) != (B_PHYS|B_READ))
				bp->b_flags |= B_ERROR;
	}

	trace(0x80<<16, "release", msmdrive);
	oc(msmdrive, RELEASE);
	while ((ss(msmcntl)&IDLE) == 0)
		;
	msmtab.b_errcnt = 0;
	msmtab.b_active = 0;
	dk_busy &= ~(1<<DN);
	msmtab.b_actf = bp->av_forw;
	iodone(bp);
	selchfree(msmselch);
	msmstart();
}

msmerror(bp, stat, addr)
register struct buf	 *bp;
{
	deverror(bp, stat, addr);
	if (++msmtab.b_errcnt <= NMSMERR) {
		msmseek();
		return;
	}
	trace(0x80<<16, "release", msmdrive);
	oc(msmdrive, RELEASE);
	while ((ss(msmcntl)&IDLE) == 0)
		;
	msmtab.b_errcnt = 0;
	bp->b_flags |= B_ERROR;
	msmtab.b_active = 0;
	dk_busy &= ~(1<<DN);
	msmtab.b_actf = bp->av_forw;
	iodone(bp);
	selchfree(msmselch);
	msmstart();
}

/*
 * 'Raw' disc interface
 */
msmread(dev)
{
	physio(msmstrategy, &rmsmbuf, dev, B_READ);
}

msmwrite(dev)
{
	physio(msmstrategy, &rmsmbuf, dev, B_WRITE);
}
