/***********************************************************************
*
* simdsk.c - TILINE Disk IO processing for the TI 990 Simulator.
*
* Changes:
*   11/11/03   DGP   Original.
*   12/10/03   DGP   Change TPCS usage.
*   05/11/04   DGP   Added DEV_NULL support and fixed to work with boot ROM.
*   07/07/04   DGP   Added disk overhead/record support.
*   07/29/04   DGP   Added dskreset function.
*
* The disk is formatted as follows:
*    4 bytes (big-endian): # cylinders
*    4 bytes (big-endian): # heads
*    4 bytes (big-endian): # bytes overhead << 8 | # sectors/track
*    4 bytes (big-endian): # bytes/sector
*    N bytes: raw data for each sector - the order is:
*      cylinder 0 head 0 sector 0
*      cylinder 0 head 0 sector 1, ...
*      cylinder 0 head 0 sector N,
*      cylinder 0 head 1 sector 0, ... 
*      cylinder 0 head M sector N,
*      cylinder 1 head 0 sector 0, ...
*
***********************************************************************/

#include <stdio.h>
#include <string.h>
#include <memory.h>
#include <ctype.h>
#include <errno.h>

#if defined (_WIN32)
#define __TTYROUTINES 0
#include <conio.h>
#include <windows.h>
#include <signal.h>
#endif

#if defined(UNIX)
#include <sys/time.h>
#include <sys/ioctl.h>
#include <termios.h>
#include <pthread.h>
#endif

#include "simdef.h"

extern uint16 pcreg;	/* The program PC */
extern uint16 statreg;	/* The program status register */
extern uint16 wpreg;	/* The program Workspace Pointer */
extern uint16 lights;	/* The panel lights */

extern int run;
extern int idle;
extern int fault;
extern int devcnt;
extern int model;
extern int mapenabled;
extern int pancount;
extern uint32 memlen;
extern unsigned long instcount;
extern char view[MAXVIEW][81];

extern uint8 memory[MEMSIZE];
extern Device devices[MAXDEVICES];

#define DSKOVERHEAD 16

/*
** Command words access macros
*/

#define GETW0(d)   GETMEM0((d)->devaddr+TPCSSTART)
#define GETW7(d)   GETMEM0((d)->devaddr+TPCSSTART+14)

#define PUTW0(d,v) PUTMEM0((d)->devaddr+TPCSSTART,v)
#define PUTW7(d,v) PUTMEM0((d)->devaddr+TPCSSTART+14,v)

#define GETCMD(d)  GETMEMB0((d)->devaddr+TPCSSTART+2) & 0x7
#define GETHEAD(d) GETMEMB0((d)->devaddr+TPCSSTART+3) & 0x3F
#define GETSECT(d) GETMEMB0((d)->devaddr+TPCSSTART+5)
#define GETCYL(d)  GETMEM0((d)->devaddr+TPCSSTART+6)
#define GETLEN(d)  GETMEM0((d)->devaddr+TPCSSTART+8)
#define GETADDR(d) \
(GETMEMB0((d)->devaddr+TPCSSTART+13)<<16)|GETMEM0((d)->devaddr+TPCSSTART+10)
#define GETUNIT(d) GETMEMB0((d)->devaddr+TPCSSTART+12) & 0xF

/*
** Command word control bits
*/

#define OLBIT	0x8001		/* Off Line */
#define NRBIT	0x4000		/* Not Ready */
#define WPBIT	0x2000		/* Write Protect */
#define USBIT	0x1000		/* UnSafe */
#define ECBIT	0x0800		/* End of Cylinder */
#define SIBIT	0x0400		/* Seek Incomplete */
#define OSABIT	0x0200		/* Offset Active */
#define PCBIT	0x0100		/* Unsafe reason */

#define STBEBIT	0x20		/* Strobe Early */
#define STBLBIT	0x10		/* Strobe Late */
#define TIHBIT	0x08		/* Transfer Inhibit */

#define OSBIT	0x80		/* Head Offset */
#define OSFBIT	0x40		/* Head Offset Forward */

#define IDLEBIT	0x8000		/* Idle */
#define COMPBIT	0x4000		/* Complete */
#define ERRBIT	0x2000		/* Error */
#define INTBIT	0x1000		/* Interrupt enable */
#define LOCKBIT	0x0800		/* Lockout */
#define RETBIT	0x0400		/* Retry */
#define ECCBIT	0x0200		/* ECC corrected */
#define ABNBIT	0x0100		/* Abnormal Completion */
#define MEBIT	0x0080		/* Memory Error */
#define DEBIT	0x0040		/* Data Error */
#define TTBIT	0x0020		/* Tiline Timeout */
#define IEBIT	0x0010		/* ID Error */
#define REBIT	0x0008		/* Rate Error */
#define CTBIT	0x0004		/* Command Timer */
#define SEBIT	0x0002		/* Search Error */
#define UEBIT	0x0001		/* Unit Error */

/***********************************************************************
* dskreadint - Read an integer.
***********************************************************************/

static int
dskreadint (FILE *fd)
{
   int r;
   int i;

   r = 0;
   for (i = 0; i < 4; i++)
   {
      int c;
      if ((c = fgetc (fd)) < 0)
      {
	 sprintf (view[0], "dskreadint: read failed: %s",
	       strerror (ERRNO));
         return (-1);
      }
      r = (r << 8) | (c & 0xFF);
   }
   return (r);
}

/***********************************************************************
* dskopen - Open disk and read geometry.
***********************************************************************/

int 
dskopen (Device *dev)
{
   int diskdata;

#ifdef DEBUGDSK
   fprintf (stderr, "dskopen: ENTERED\n");
#endif

   if (dev->fd == DEV_NULL)
   {
      PUTW0 (dev, OLBIT | WPBIT);
      PUTW7 (dev, IDLEBIT | COMPBIT); /* Mark idle */
      return (0);
   }

   /*
   ** Seek to beginning of the disk
   */

   if (fseek (dev->fd, 0, SEEK_SET) < 0)
   {
      sprintf (view[0], "dskopen: seek failed: %s",
	    strerror (ERRNO));
      sprintf (view[1], "filename: %s", dev->file);
      return (-1);
   }

   /*
   ** Read the disk geometry
   */

   if ((dev->info.dskinfo.cyls = dskreadint (dev->fd)) < 0) return (-1);
   if ((dev->info.dskinfo.heads = dskreadint (dev->fd)) < 0) return (-1);
   if ((diskdata = dskreadint (dev->fd)) < 0) return (-1);
   dev->info.dskinfo.secttrk = diskdata & 0xFF;
   dev->info.dskinfo.overhead = (diskdata >> 8) & 0xFF;
   if ((dev->info.dskinfo.sectlen = dskreadint (dev->fd)) < 0) return (-1);
   dev->info.dskinfo.tracklen = dev->info.dskinfo.secttrk * 
	       (dev->info.dskinfo.sectlen + dev->info.dskinfo.overhead);
	       

#ifdef DEBUGDSK
   fprintf (stderr, "Disk geometry:\n");
   fprintf (stderr, "   cyls      = %d\n", dev->info.dskinfo.cyls);
   fprintf (stderr, "   heads     = %d\n", dev->info.dskinfo.heads);
   fprintf (stderr, "   secttrk   = %d\n", dev->info.dskinfo.secttrk);
   fprintf (stderr, "   sectlen   = %d\n", dev->info.dskinfo.sectlen);
   fprintf (stderr, "   overhead  = %d\n", dev->info.dskinfo.overhead);
   fprintf (stderr, "   tracklen  = %d\n", dev->info.dskinfo.tracklen);
   fprintf (stderr, "   disk size = %d bytes\n", dev->info.dskinfo.tracklen *
	       dev->info.dskinfo.heads * dev->info.dskinfo.cyls);
#endif

   PUTW7 (dev, IDLEBIT | COMPBIT); /* Mark idle */
   
   return (0);
}

/***********************************************************************
* dskputmem - Put data into memory.
***********************************************************************/

static void
dskputmem (uint32 ma, uint16 v)
{
   if (ma < memlen)
   {
      PUTMEM0 (ma, v);
   }
}

/***********************************************************************
* dskgetmem - Get data from memory.
***********************************************************************/

static uint16
dskgetmem (uint32 ma)
{
   if (ma >= memlen) return (0);

   return (GETMEM0 (ma));
}

/***********************************************************************
* dskreset - Reset device
***********************************************************************/

void
dskreset (Device *dev)
{
#ifdef DEBUGDSK
   fprintf (stderr, "dskreset: ENTERED\n");
#endif

   if (dev->fd == DEV_NULL)
   {
      PUTW0 (dev, OLBIT | WPBIT);
   }
   else
   {
      PUTW0 (dev, (dev->info.dskinfo.writeprotect ? WPBIT : 0));
   }

   PUTW7 (dev, IDLEBIT | ERRBIT | ABNBIT); 

   return;
}

/***********************************************************************
* dskseek - Seek to specified disk location.
***********************************************************************/

static int
dskseek (Device *udev, int cyl, int head, int sector)
{
   int dskloc;
   uint16 W0, W7;

   /*
   ** Ensure cylinder is OK
   */

   if (cyl >= udev->info.dskinfo.cyls)
   {
      W0 |= SIBIT | (udev->info.dskinfo.writeprotect ? WPBIT : 0); 
      W7 |= ERRBIT | IDLEBIT | COMPBIT | UEBIT;
      PUTW0 (udev, W0);
      PUTW7 (udev, W7);
      return (-1);
   }

   /*
   ** Calculate location
   */

   dskloc = (udev->info.dskinfo.tracklen * head) +
      (udev->info.dskinfo.tracklen * udev->info.dskinfo.heads * cyl) +
      ((udev->info.dskinfo.overhead + udev->info.dskinfo.sectlen) * sector) +
      DSKOVERHEAD;

#ifdef DEBUGDSK
   fprintf (stderr, "dskseek: cyl = %d, head = %d, sector = %d\n",
	    cyl, head, sector);
   fprintf (stderr, "   dskloc = %d\n", dskloc);
#endif

   /*
   ** Go to the specified location on disk
   */

   if (fseek (udev->fd, dskloc, SEEK_SET) < 0)
   {
      sprintf (view[0], "dskseek: seek failed: %s",
	    strerror (ERRNO));
      sprintf (view[1], "filename: %s", udev->file);
      return (-1);
   }

   return (0);
}

/***********************************************************************
* dskdocmd - Do disk command
***********************************************************************/

int 
dskdocmd (Device *dev)
{
   Device *udev;
   int cmd;
   int i;
   int len;
   int cyl, head, sector;
   int tinhibit;
   uint32 addr, saddr;
   uint16 W0, W7;
   uint8 unit;

   /*
   ** Get common command word values
   */

   W0 = GETW0 (dev) & 0x00FF;
   W7 = GETW7 (dev);

   cmd = GETCMD (dev);
   unit = GETUNIT (dev);
   saddr = addr = ((GETADDR (dev)) + 1) & 0x1FFFFE;
   len = (GETLEN (dev) + 1) & 0xFFFE;

#ifdef DEBUGDSK
   fprintf (stderr, "%08.8ld:   pc = %04X, wp = %04X, st = %04X\n",
	    instcount, pcreg, wpreg, statreg);
   fprintf (stderr,
	    "dskdocmd: ENTRY: cmd = %d, unit = %d, addr = %06X, len = %d\n",
	    cmd, unit, addr, len);
   for (i = 0; i < 16; i+=2)
      fprintf (stderr, "   ctl[%04X] %04X\n",
	    dev->devaddr+TPCSSTART+i, GETMEM0 (dev->devaddr+TPCSSTART+i));
#endif

   /*
   ** Find selected unit
   */

   for (i = 0; i < devcnt; i++) /* find selected unit */
      if (devices[i].devaddr == dev->devaddr && devices[i].unit == unit)
	 break;
   if (i == devcnt)
   {
#ifdef DEBUGDSK
      fprintf (stderr, "dskdocmd: unit = %d: Unit not selected\n", unit);
#endif
      /* Drive not ready */
      W0 |= OLBIT;
      W7 |= ERRBIT | COMPBIT | IDLEBIT;
      PUTW0 (dev, W0);
      PUTW7 (dev, W7);
      return (W7 & INTBIT ? dev->intlvl : 0);
   }
   udev = &devices[i];

   if (udev->fd == DEV_NULL)
   {
#ifdef DEBUGDSK
      fprintf (stderr, "dskdocmd: unit = %d: Unit is NULL\n", unit);
#endif
      PUTW0 (udev, OLBIT | WPBIT);
      PUTW7 (udev, IDLEBIT | COMPBIT);
      return (W7 & INTBIT ? udev->intlvl : 0);
   }

   /*
   ** If command is not Store Registers or Restore, check disk parameters
   */

   if (cmd != 0 && cmd != 7)
   {
      cyl = GETCYL (udev);
      if (cyl >= udev->info.dskinfo.cyls)
      {
#ifdef DEBUGDSK
	 fprintf (stderr, "dskdocmd: unit = %d: Invalid cylinder: %d\n",
		  unit, cyl);
#endif
	 W0 |= SIBIT | (udev->info.dskinfo.writeprotect ? WPBIT : 0); 
	 W7 |= ERRBIT | IDLEBIT | COMPBIT | UEBIT;
	 PUTW0 (udev, W0);
	 PUTW7 (udev, W7);
	 return (W7 & INTBIT ? udev->intlvl : 0);
      }
      if (cmd == 6) /* Seeks only use cylinder */
      {
         head = 0;
	 sector = 0;
      }
      else
      {
	 head = GETHEAD (udev);
	 if (head >= udev->info.dskinfo.heads)
	 {
#ifdef DEBUGDSK
	    fprintf (stderr, "dskdocmd: unit = %d: Invalid head: %d\n",
		     unit, head);
#endif
	    W0 |= ECBIT | (udev->info.dskinfo.writeprotect ? WPBIT : 0); 
	    W7 |= ERRBIT | COMPBIT | IDLEBIT;
	    PUTW0 (udev, W0);
	    PUTW7 (udev, W7);
	    return (W7 & INTBIT ? udev->intlvl : 0);
	 }
	 sector = GETSECT (udev);
	 if (sector >= udev->info.dskinfo.secttrk)
	 {
#ifdef DEBUGDSK
	    fprintf (stderr, "dskdocmd: unit = %d: Invalid sector: %d\n",
		     unit, sector);
#endif
	    W0 |= TTBIT | (udev->info.dskinfo.writeprotect ? WPBIT : 0); 
	    W7 |= ERRBIT | COMPBIT | IDLEBIT;
	    PUTW0 (udev, W0);
	    PUTW7 (udev, W7);
	    return (W7 & INTBIT ? udev->intlvl : 0);
	 }
      }

      /*
      ** Parms OK, seek
      */

      if (dskseek (udev, cyl, head, sector) < 0)
      {
	 return (W7 & INTBIT ? udev->intlvl : 0);
      }
   }

   /*
   ** Set initial sate
   */

   udev->select = TRUE;
   udev->intenabled = FALSE;

   /*
   ** Process command
   */

   switch (cmd)
   {
   case 0: /* Store registers */
#ifdef DEBUGDSK
      fprintf (stderr, "dskdocmd: STRREG: \n");
#endif
      if (len > 0) 
      {
         dskputmem (addr, udev->info.dskinfo.tracklen >> 1);
#ifdef DEBUGDSK
	 fprintf (stderr, "   W1 = %04X\n", udev->info.dskinfo.tracklen >> 1);
#endif
         len -= 2;
	 addr += 2;
      }
      if (len > 0) 
      {
	 
         dskputmem (addr, (udev->info.dskinfo.secttrk << 8) | 
			  (udev->info.dskinfo.overhead >> 1));
#ifdef DEBUGDSK
	 fprintf (stderr, "   W2 = %04X\n",
			(udev->info.dskinfo.secttrk << 8) | 
			(udev->info.dskinfo.overhead >> 1));
#endif
         len -= 2;
	 addr += 2;
      }
      if (len > 0) 
      {
         dskputmem (addr,
	       (udev->info.dskinfo.heads << 11) | udev->info.dskinfo.cyls);
#ifdef DEBUGDSK
	 fprintf (stderr, "   W3 = %04X\n", 
	       (udev->info.dskinfo.heads << 11) | udev->info.dskinfo.cyls);
#endif
      }
      break;

   case 1: /* Write format */
      len = udev->info.dskinfo.tracklen;
#ifdef DEBUGDSK
      fprintf (stderr, "dskdocmd: FORMAT: len = %d\n", len);
#endif
      if (!udev->info.dskinfo.writeprotect)
      {
         int truelen;

	 truelen = udev->info.dskinfo.sectlen + udev->info.dskinfo.overhead;

	 while (len)
	 {
	    int slen;

	    /*
	    ** Format one sector at a time
	    */

	    slen = len < truelen ?  len : truelen;

	    for (i = 0; i < slen; i += 2)
	    {
	       fputc ('\0', udev->fd);
	       fputc ('\0', udev->fd);
	    }

	    len -= slen;
	    sector++;
	    if (sector == udev->info.dskinfo.secttrk)
	    {
	       sector = 0;
	       head++;
	       if (head == udev->info.dskinfo.heads)
	       {
		  head = 0;
		  cyl++;
	       }
	    }

	    if (len)
	       if (dskseek (udev, cyl, head, sector) < 0) 
		  return (W7 & INTBIT ? udev->intlvl : 0);
	 }
	 udev->info.dskinfo.curcyl = cyl;
	 udev->info.dskinfo.curhead = head;
	 udev->info.dskinfo.cursect = sector;
      }
      break;

   case 2: /* Read data */
#ifdef DEBUGDSK
      fprintf (stderr, "dskdocmd: READ: \n");
#endif
      tinhibit = GETMEMB0(udev->devaddr+TPCSSTART+2) & TIHBIT;
      while (len)
      {
	 int slen;

	 /*
	 ** Read one sector at a time
	 */

	 slen = (len < udev->info.dskinfo.sectlen) ?
	       len : udev->info.dskinfo.sectlen;

	 for (i = 0; i < slen; i += 2)
	 {
	    uint16 d, c;

	    d = fgetc (udev->fd);
	    c = fgetc (udev->fd);
	    if (!tinhibit)
	    {
	       d = (d << 8) | c;
	       dskputmem (addr, d);
	       addr += 2;
	    }
	 }

	 len -= slen;
	 sector++;
	 if (sector == udev->info.dskinfo.secttrk)
	 {
	    sector = 0;
	    head++;
	    if (head == udev->info.dskinfo.heads)
	    {
	       head = 0;
	       cyl++;
	    }
	 }

	 if (len)
	    if (dskseek (udev, cyl, head, sector) < 0)
	       return (W7 & INTBIT ? udev->intlvl : 0);
      }
#ifdef DEBUGDSKDATA
      if (!tinhibit)
      {
	 HEXDUMP (stderr, &memory[saddr], GETLEN(udev), saddr);
      }
      else
      {
         fprintf (stderr, "Data inhibited\n");
      }
#endif
      udev->info.dskinfo.curcyl = cyl;
      udev->info.dskinfo.curhead = head;
      udev->info.dskinfo.cursect = sector;
      break;

   case 3: /* Write data */
#ifdef DEBUGDSK
      fprintf (stderr, "dskdocmd: WRITE: \n");
#endif
      if (!udev->info.dskinfo.writeprotect)
      {
#ifdef DEBUGDSKDATA
	 HEXDUMP (stderr, &memory[saddr], len, saddr);
#endif
	 while (len)
	 {
	    int slen;

	    /*
	    ** Write one sector at a time
	    */

	    slen = (len < udev->info.dskinfo.sectlen) ?
		  len : udev->info.dskinfo.sectlen;

	    for (i = 0; i < slen; i += 2)
	    {
	       uint16 d;

	       d = dskgetmem (addr);
	       fputc ((d >> 8) & 0xFF, udev->fd);
	       fputc (d & 0xFF, udev->fd);
	       addr += 2;
	    }

	    /*
	    ** Zero fill short sectors
	    */

#ifdef DEBUGDSKDATA
	    if (i < udev->info.dskinfo.sectlen)
	       fprintf (stderr, "Pad %d bytes\n",
		     udev->info.dskinfo.sectlen - i);
#endif
	    for (; i < udev->info.dskinfo.sectlen; i++)
	       fputc ('\0', udev->fd);

	    len -= slen;
	    sector++;
	    if (sector == udev->info.dskinfo.secttrk)
	    {
	       sector = 0;
	       head++;
	       if (head == udev->info.dskinfo.heads)
	       {
		  head = 0;
		  cyl++;
	       }
	    }

	    if (len)
	       if (dskseek (udev, cyl, head, sector) < 0)
		  return (W7 & INTBIT ? udev->intlvl : 0);
	 }

	 udev->info.dskinfo.curcyl = cyl;
	 udev->info.dskinfo.curhead = head;
	 udev->info.dskinfo.cursect = sector;
      }
      break;

   case 4: /* Unformatted read */
#ifdef DEBUGDSK
      fprintf (stderr, "dskdocmd: UREAD: \n");
#endif
      if (len > 0) 
      {
         dskputmem (addr,
	       (head << 11) | cyl);
#ifdef DEBUGDSK
	 fprintf (stderr, "   W1 = %04X\n", 
		  (head << 11) | cyl);
#endif
         len -= 2;
	 addr += 2;
      }
      if (len > 0) 
      {
         dskputmem (addr, 0x100 | sector);
#ifdef DEBUGDSK
	 fprintf (stderr, "   W2 = %04X\n", 
		  0x100 | sector);
#endif
         len -= 2;
	 addr += 2;
      }
      if (len > 0) 
      {
         dskputmem (addr,
	       udev->info.dskinfo.sectlen >> 1);
#ifdef DEBUGDSK
	 fprintf (stderr, "   W3 = %04X\n", 
	       udev->info.dskinfo.sectlen >> 1);
#endif
      }
      udev->info.dskinfo.curcyl = cyl;
      udev->info.dskinfo.curhead = head;
      udev->info.dskinfo.cursect = sector;
      break;

   case 5: /* Unformatted write */
#ifdef DEBUGDSK
      fprintf (stderr, "dskdocmd: UWRITE: \n");
#endif
      if (!udev->info.dskinfo.writeprotect)
      {
#ifdef DEBUGDSKDATA
	 HEXDUMP (stderr, &memory[saddr], GETLEN(udev), saddr);
#endif
	 for (i = 0; i < len; i += 2)
	 {
	    uint16 d;

	    d = dskgetmem (addr);
	    fputc ((d >> 8) & 0xFF, udev->fd);
	    fputc (d & 0xFF, udev->fd);
	    addr += 2;
	 }
	 udev->info.dskinfo.curcyl = cyl;
	 udev->info.dskinfo.curhead = head;
	 udev->info.dskinfo.cursect = sector;
      }
      break;

   case 6: /* Seek */
#ifdef DEBUGDSK
      fprintf (stderr, "dskdocmd: SEEK: \n");
#endif
      udev->info.dskinfo.curcyl = cyl;
      udev->info.dskinfo.curhead = head;
      udev->info.dskinfo.cursect = sector;
      W0 |= (udev->unit << 4);
      break;

   case 7: /* Restore */
#ifdef DEBUGDSK
      fprintf (stderr, "dskdocmd: RESTORE: \n");
#endif
      udev->info.dskinfo.curcyl = 0;
      udev->info.dskinfo.curhead = 0;
      udev->info.dskinfo.cursect = 0;
      W0 |= (udev->unit << 4);
      break;

   }

   if (W7 & INTBIT)
   {
      W7 = W7 & ~INTBIT;
      udev->intenabled = TRUE;
   }
   W0 |= (udev->info.dskinfo.writeprotect ? WPBIT : 0); 
   W7 |= COMPBIT | IDLEBIT;
   PUTW0 (udev, W0);
   PUTW7 (udev, W7);
#ifdef DEBUGDSK
   fprintf (stderr, "returns: intenabled = %s\n",
	 udev->intenabled ? "TRUE" : "FALSE");
   for (i = 0; i < 16; i+=2)
      fprintf (stderr, "   ctl[%04X] %04X\n",
	    udev->devaddr+TPCSSTART+i, GETMEM0 (udev->devaddr+TPCSSTART+i));
#endif
   return (udev->intenabled ? udev->intlvl : 0);
}

/***********************************************************************
* dskboot - Boot from device.
***********************************************************************/

int
dskboot (Device *dev)
{
   int i;
   uint16 lp;
   uint16 val;

   run = FALSE;
   lp = LOADADDRESS;
   SET_MASK (0);

   if (model == 4 || model == 9)
   {
      sprintf (view[0], "dskboot: TILINE not supported on 990/%d\n", model);
      return (-1);
   }

#ifdef USEDSKROM
   wpreg = GETMEM0 (TPCSSTART+0xFFFC) & 0xFFFE;
   pcreg = GETMEM0 (TPCSSTART+0xFFFE) & 0xFFFE;

   PUTREG (0, 0xFFFF);
   PUTREG (1, dev->devaddr);
   PUTREG (2, dev->unit << 8);
#else /* !USEDSKROM */
   {
      int len;
      uint16 cyl;
      uint16 head;
      uint16 sect;

      if (dskseek (dev, 0, 0, 0) < 0) return (-1);

      i = dev->info.dskinfo.sectlen;
      if (fread (&memory[lp], 1, i, dev->fd) != i)
      {
	 sprintf (view[0], "dskboot: read failed: %s",
	       strerror (ERRNO));
	 sprintf (view[1], "filename: %s", dev->file);
	 return (-1);
      }

#ifdef DUMPBOOT
      fprintf (stderr, "dskboot: devaddr = %04X, unit = %02X\n", 
	       dev->devaddr, dev->unit);
      fprintf (stderr, "cyl = 0, head = 0, sect = 0, len = %d\n", i);
      HEXDUMP (stderr, &memory[lp], i, 0);
#endif

      val = GETMEM0 (lp + 0xE);
      cyl = val / dev->info.dskinfo.heads;
      head = val % dev->info.dskinfo.heads;
      sect = 0;

      pcreg = GETMEM0 (lp + 0x18);
      wpreg = 0x80;

      len = GETMEM0 (lp + 0x1A);
      while (len)
      {
	 if (dskseek (dev, cyl, head, sect) < 0) return (-1);
	 i = (len < dev->info.dskinfo.sectlen) ?
	      len : dev->info.dskinfo.sectlen;
	 if (fread (&memory[lp], 1, i, dev->fd) != i)
	 {
	    sprintf (view[0], "dskboot: read failed: %s",
		  strerror (ERRNO));
	    sprintf (view[1], "filename: %s", dev->file);
	    return (-1);
	 }
	 lp += i;
         len -= i;
	 sect++;
	 if (sect >= dev->info.dskinfo.secttrk)
	 {
	    sect = 0;
	    head++;
	    if (head >= dev->info.dskinfo.heads)
	    {
	       head = 0;
	       cyl++;
	    }
	 }
      }

#ifdef DUMPBOOT
      fprintf (stderr, "cyl = %d, head = %d, sect = 0, len = %d\n",
	       cyl, head, i);
      HEXDUMP (stderr, &memory[LOADADDRESS], i, LOADADDRESS);
#endif

      PUTREG (0, 0xABC0);
      PUTREG (1, dev->devaddr);
      PUTREG (2, dev->unit << 8);

   }
#endif /* USEDSKROM */

   run = TRUE;

   return (0);
}

