//---------------------------------------------------------------------------
//
// %FILE     pic.c
// %VSS-REV  $Revision: 14 $
// %CREATED  1995.12.17
// %REVISED  $Date: 4/18/97 4:17p $
// %AUTHOR   Noreen Bell
// %PROJECT  NS486SXF evaluation board software
// %PART     NS486SXF
// %SUMMARY  Programmable Interrupt Controller module
//     
// %VSS      $Author: Miked $ $Date: 4/18/97 4:17p $ $Revision: 14 $
//
// DESCRIPTION
//
//   This file contains code for the NS486 Programmable
//   Interrupt Controller (PIC).  The NS486 PIC "looks" like two
//   9259A's in a standard cascaded mode.  Therefore, in this code, the
//   NS486 PIC is refered to as two PICs: a master and a slave.
//
// INPUTS
//
//   Locations of main registers: (#define <word>)
//
//     PIC_MASTER0
//     PIC_MASTER1
//     PIC_SLAVE0
//     PIC_SLAVE1
//
//   Location of NS486 Internal Interrupt Selection Registers: (#d <word>)
//
//     PIC_IInSR          n = 0-15 excluding 0,2,8
//
//   Values to write to IISRs: (#define <byte>)
//
//     PIC_IInS           n = 0-15 excluding 0,2,8
//
//   Interrupt vectors for each controller: (enum IRQmap)
//
//     IRQIV_Controller1
//     IRQIV_Controller2
//
// HISTORY
//
/*
 *
 * $History: pic.c $ 
 * 
 * *****************  Version 14  *****************
 * User: Miked        Date: 4/18/97    Time: 4:17p
 * Updated in $/nsdemo
 * Added writes to misc. interrupt selection registers.  Moved CLI
 * instruction.   New header (comment) changes.
 * 
 * *****************  Version 13  *****************
 * User: Miked        Date: 4/07/97    Time: 2:36p
 * Updated in $/nsdemo
 * Added semicolon to CPU_ macros to accomodate for nsglobal.h change.
 * 
 * *****************  Version 12  *****************
 * User: Miked        Date: 12/04/96   Time: 4:06p
 * Updated in $/nsdemo
 * Moved function prototypes to header file.  Added some clarifying
 * paranthethis.
 * 
 * *****************  Version 11  *****************
 * User: Miked        Date: 8/06/96    Time: 11:59a
 * Updated in $/nsdemo
 * Version 1.4.  Maintainance release.  See README.TXT for info.
 * 
 * *****************  Version 10  *****************
 * User: Miked        Date: 8/05/96    Time: 6:04p
 * Updated in $/nsdemo
 * Added support for RTTarget (does not initialize PICs when using
 * RTTraget).
 * 
 * *****************  Version 9  *****************
 * User: Miked        Date: 7/23/96    Time: 2:25p
 * Updated in $/nsdemo
 * Maintainance release.  README.TXT describes changes.
 * 
 * *****************  Version 8  *****************
 * User: Miked        Date: 7/16/96    Time: 11:54a
 * Updated in $/nsdemo
 * Updated for rev C0 release.
 * 
 * *****************  Version 7  *****************
 * User: Miked        Date: 6/28/96    Time: 1:45p
 * Updated in $/nsdemo
 * Calls assembly functions to do CLI and STI, instead of inline assembly.
 * 
 * *****************  Version 6  *****************
 * User: Noreen       Date: 6/26/96    Time: 3:19p
 * Updated in $/nstest
 * Added inline assembly commands  for masking and unmasking CPU
 * interrupts
 * 
 * *****************  Version 5  *****************
 * User: Miked        Date: 5/03/96    Time: 2:50p
 * Updated in $/nsdemo
 * Maintainence release.
 * 
 * *****************  Version 4  *****************
 * User: Noreen       Date: 4/30/96    Time: 3:28p
 * Updated in $/nstest
 * Added code to PIC_Initialize for  mapping misc internal interrupt
 * selection register for PCMCIA
 * 
 * *****************  Version 3  *****************
 * User: Noreen       Date: 4/18/96    Time: 12:41p
 * Updated in $/nsdemo
 * Updated comments for version  release
 * 
 * *****************  Version 2  *****************
 * User: Miked        Date: 4/12/96    Time: 2:29p
 * Updated in $/nstest
 * Revised headers for VSS.
 *
 */
//
// COPYRIGHT
//
//      (c) 1996, 1997 National Semiconductor Corporation
//
//---------------------------------------------------------------------------

#include "pic.h"

//---------------------------------------------------------------------------
//
// FUNCTION    PIC_Status
//
// INPUT       USHORT IRQLevel - IRQ to get status on
// INPUT/OUTPUT IRQ_STATUS *pStatus - a pointer to the structure
//                                    which holds the status.
// RETURN      USHORT
//               SUCCESS if Status returned
//               INVALID_PARAMETER if NULL pointer is passed to function
//               or if IRQ Level specified is not available
//
// DESCRIPTION
//      This function uses a pointer to a structure to return the
//      status of the IRQ level information was requested on.
//
// NOTES
//      typedef struct {
//              BOOL enable                   -unmasked=0,masked=1
//
//              USHORT vector                 -InterruptVectorAddress
//
//              BOOL IRR                      -NoIRQpending=0,IRQpending=1
//
//              BOOL ISR                      -NotInService=0,InService=1
//      } IRQ_STATUS;
//
//---------------------------------------------------------------------------

USHORT PIC_Status(USHORT IRQLevel, IRQ_STATUS * pStatus)
{
	// Check for NULL pointer
	if(pStatus == NULL)
	   return INVALID_PARAMETER;

	// Check for Valid Interrupt
	if(IRQLevel > MAX_IRQ)
	    return INVALID_PARAMETER;

	// Get value of Interrupt Mask Register
	if(IRQLevel < IRQ_PER_DEVICE)
	   // Read from Master
	   if( ( IOR_BYTE(PIC_M_IMR) & (1 << IRQLevel) ) == 0)
	      pStatus->enable = unmasked;
	   else
	      pStatus->enable = masked;
	else
	   // Read from Slave
	   if( ( IOR_BYTE(PIC_S_IMR) & (1 << IRQLevel) ) == 0)
	      pStatus->enable = unmasked;
	   else
	      pStatus->enable = masked;

	if(IRQLevel < IRQ_PER_DEVICE)
	   pStatus->vector = ICW2_Master;
	else
	   pStatus->vector = ICW2_Slave;

	if(IRQLevel < IRQ_PER_DEVICE)
	{
	   // Get value of Interrupt Request Register from Master
	   IOW_BYTE(PIC_M_OCW3, OCW3_ReadIRequest);
	   if( ( IOR_BYTE(PIC_M_IRR) & (1 << IRQLevel) ) == 0)
	      pStatus->IRR = NoIRQpending;
	   else
	      pStatus->IRR = IRQpending;
	}
	else
	{
	   // Get value of Interrupt Request Register from Slave
	  IOW_BYTE(PIC_S_OCW3, OCW3_ReadIRequest);
	  if( ( IOR_BYTE(PIC_S_IRR) & (1 << IRQLevel) ) == 0)
	      pStatus->IRR = NoIRQpending;
	   else
	      pStatus->IRR = IRQpending;
	}

	if(IRQLevel < IRQ_PER_DEVICE)
	{
	   // Get value of In Service Register from Master
	   IOW_BYTE(PIC_M_OCW3, OCW3_ReadInServiceStatus);
	   if( ( IOR_BYTE(PIC_M_ISR) & (1 << IRQLevel) ) == 0)
	      pStatus->ISR = NotInService;
	   else
	      pStatus->ISR = InService;
	}
	else
	{
	   // Get value of In Service Register from Slave
	   IOW_BYTE(PIC_S_OCW3, OCW3_ReadInServiceStatus);
	   if( ( IOR_BYTE(PIC_S_ISR) & (1 << IRQLevel) ) == 0)
	      pStatus->ISR = NotInService;
	   else
	      pStatus->ISR = InService;
	}

	return SUCCESS;
}

//---------------------------------------------------------------------------
//
// FUNCTION    PIC_EOI
//
// INPUT       USHORT IRQLevel - IRQ to enable
// OUTPUT      none
// RETURN      USHORT
//               SUCCESS if EOI command issued
//               INVALID_PARAMETER if IRQ specified is Channel2
//               or if IRQ Level specified is not available
//
// DESCRIPTION
//
//   This function issues a non specified EOI command to the PIC
//   It should be called at the end of an interrupt service routine
//   to enable processing of additional interrupts
//
//---------------------------------------------------------------------------

USHORT PIC_EOI(USHORT IRQLevel)
{
	// Do not allow EOI to be issued to cascaded channel
	if(IRQLevel == 2)
	   return INVALID_PARAMETER;

	if(IRQLevel > MAX_IRQ)
	  return INVALID_PARAMETER;

	// Issue EOI to Master
	IOW_BYTE(PIC_M_OCW2, OCW2_NonSpecificEOI);
	if(IRQLevel >= IRQ_PER_DEVICE)
	  // Also Issue EOI to Slave
	  IOW_BYTE(PIC_S_OCW2, OCW2_NonSpecificEOI);

	return SUCCESS;
}

//---------------------------------------------------------------------------
//
// FUNCTION    PIC_Disable
//
// INPUT       USHORT IRQLevel  - IRQ level to disable
// OUTPUT      none
// RETURN      USHORT
//               SUCCESS if call successful
//               INVALID_PARAMETER if IRQ Level specified is not available
//
// DESCRIPTION
//   This function disables interrupts for a particular IRQ line by
//   masking it in the PIC.
//
//---------------------------------------------------------------------------

USHORT PIC_Disable(USHORT IRQLevel)
{
  
	//Prevent user from disabling IRQ2, used for cascaded channel
	if(IRQLevel == 2)
	   return INVALID_PARAMETER;

	if(IRQLevel > MAX_IRQ)
	  return INVALID_PARAMETER;

	// Figure out which controller to mask based
	// on the IRQ level
	if (IRQLevel < IRQ_PER_DEVICE)
	    // Read the current setting and OR the Mask to preserve
	    // the bits that are already set, and SET our bit.
	    IOW_BYTE(PIC_M_IMR, (BYTE) ((1 << IRQLevel) | IOR_BYTE(PIC_M_IMR)) );
	else
	  // Read the current setting and OR the Mask to preserve
	  // the bits that are already set, and SET our bit.
	  IOW_BYTE(PIC_S_IMR, 
	  (BYTE) ( ( 1 << (IRQLevel-IRQ_PER_DEVICE) ) | IOR_BYTE(PIC_S_IMR)) );
      
	return SUCCESS;

}

//---------------------------------------------------------------------------
//
// FUNCTION    PIC_Enable
//
// INPUT       USHORT IRQLevel - IRQ level to enable
// OUTPUT      none
// RETURN      USHORT
//               SUCCESS if call successful
//               INVALID_PARAMETER if IRQ Level specified is not available
//
// DESCRIPTION
//   This function enables interrupts for the specified IRQ level by
//   unmasking it in the PIC.
//
//---------------------------------------------------------------------------

USHORT PIC_Enable(USHORT IRQLevel)
{

	//Prevent user from enabling IRQ2, used for cascaded channel
	if(IRQLevel == 2)
	   return INVALID_PARAMETER;

	if(IRQLevel > MAX_IRQ)
	    return INVALID_PARAMETER;

	// Figure out which controller we need to mask based
	// on the IRQ level
	if (IRQLevel < IRQ_PER_DEVICE)
	    // Read the current setting and the Mask to preserve
	    // the bits that are already set, and clear our bit.
	    IOW_BYTE(PIC_M_IMR, (BYTE) (( ~(1 << IRQLevel) ) & IOR_BYTE(PIC_M_IMR)) );
	 else
	   // Read the current setting and the Mask to preserve
	   // the bits that are already set, and clear our bit.
	   IOW_BYTE(PIC_S_IMR, (BYTE) (( ~(1 << (IRQLevel - IRQ_PER_DEVICE))) & IOR_BYTE(PIC_S_IMR)) );

	return SUCCESS;

}

//---------------------------------------------------------------------------
//
// FUNCTION    PIC_Initialize
//
// INPUT       none
// OUTPUT      none
// RETURN      USHORT
//               SUCCESS - if call successful
//               FAIL - if BIU has not been configured to
//                      enable accesses to PIC
// DESCRIPTION
//   This initializes the NS486 PIC. In particular, it calls a function to
//   map the NS486 internal interrupts, writes the ICWs to the master
//   and slave PIC, masks all interrupts (except cascade), leaves the PICs
//   in read IRR mode, and enables the CPU interrupt.
//
//---------------------------------------------------------------------------

USHORT PIC_Initialize(void)
{

	//If PIC is not enabled in the BIU, return FAIL
	if ((IOR_BYTE(BIU_CONTROL1) & 0x10) != 0x10)
	   return FAIL;

	// Disable CPU Interrupts
	CPU_CLI;
	
	// Map NS486 internal interrupts to pins or devices
	// Note: internal interrupts 0, 2, and 8 do not have selectable
	// interrupt sources
	IOW_BYTE(PIC_II1SR, PIC_II1S);
	IOW_BYTE(PIC_II3SR, PIC_II3S);
	IOW_BYTE(PIC_II4SR, PIC_II4S);
	IOW_BYTE(PIC_II5SR, PIC_II5S);
	IOW_BYTE(PIC_II6SR, PIC_II6S);
	IOW_BYTE(PIC_II7SR, PIC_II7S);
	IOW_BYTE(PIC_II9SR, PIC_II9S);
	IOW_BYTE(PIC_II10SR, PIC_II10S);
	IOW_BYTE(PIC_II11SR, PIC_II11S);
	IOW_BYTE(PIC_II12SR, PIC_II12S);
	IOW_BYTE(PIC_II13SR, PIC_II13S);
	IOW_BYTE(PIC_II14SR, PIC_II14S);
	IOW_BYTE(PIC_II15SR, PIC_II15S);

	IOW_BYTE(PIC_MISR1,  PIC_MIS1);
	IOW_BYTE(PIC_MISR2,  PIC_MIS2);
	IOW_BYTE(PIC_MISR3,  PIC_MIS3);

	// Before operation, the PIC must be initialized by a
	// sequence of four initialization command words (ICW1-ICW4)

      #ifndef ONTIME // on RTTarget, leave their PIC setup intact
	
	// Initialize Master PIC
	IOW_BYTE(PIC_M_ICW1, ICW1_Master);
	IOW_BYTE(PIC_M_ICW2, ICW2_Master);
	IOW_BYTE(PIC_M_ICW3, ICW3_Master);
	IOW_BYTE(PIC_M_ICW4, ICW4_Master);

	// Initialize Slave PIC
	IOW_BYTE(PIC_S_ICW1, ICW1_Slave);
	IOW_BYTE(PIC_S_ICW2, ICW2_Slave);
	IOW_BYTE(PIC_S_ICW3, ICW3_Slave);
	IOW_BYTE(PIC_S_ICW4, ICW4_Slave);

	// Disable all IRQ levels except level 2 which is always enabled
	IOW_BYTE(PIC_S_IMR, IMR_MaskAllIntSlave);
	IOW_BYTE(PIC_M_IMR, IMR_MaskAllIntMaster);
       
      #endif

	// Enable CPU Interrupts
	CPU_STI;

	// all set
	return SUCCESS;
}

//---------------------------------------------------------------------------
// END       pic.c
//---------------------------------------------------------------------------
