//---------------------------------------------------------------------------
//
// %FILE     rtctest.c
// %VSS-REV  $Revision: 28 $
// %CREATED  1996.01.11
// %REVISED  $Date: 4/22/97 11:33a $
// %AUTHOR   Noreen Bell
// %PROJECT  NS486SXF evaluation board software
// %PART     NS486SXF (B0+), NS486SXL
// %SUMMARY  Real Time Clock test module
//     
// %VSS      $Author: Miked $ $Date: 4/22/97 11:33a $ $Revision: 28 $
//
// DESCRIPTION
//
//   This module contains simple functions for either testing the Real Time
//   clock or using the RTC to test other peripherals.
//
// HISTORY
//
/*
 *
 * $History: rtctest.c $
 * 
 * *****************  Version 28  *****************
 * User: Miked        Date: 4/22/97    Time: 11:33a
 * Updated in $/nsdemo
 * Fixed bug: TimeValidate was updating Month field if DayOfWeek was out
 * of range.
 * 
 * *****************  Version 27  *****************
 * User: Miked        Date: 4/18/97    Time: 4:24p
 * Updated in $/nsdemo
 * Massive rewrite of most of the test functions.  Will now work with
 * current RTC time, date, and mode if valid.  If VRT is valid will
 * perform many sanity checks.   New header (comment) changes.
 * 
 * *****************  Version 26  *****************
 * User: Miked        Date: 4/07/97    Time: 2:16p
 * Updated in $/nsdemo
 * Updated ID/REV code for SXL.
 * 
 * *****************  Version 25  *****************
 * User: Miked        Date: 12/05/96   Time: 2:31p
 * Updated in $/nsdemo
 * Changed period interrupt rate in RTCTEST - was too fast and was
 * crashing
 * Borland build (CPU couldn't get control long enough to run code).
 * 
 * *****************  Version 24  *****************
 * User: Miked        Date: 12/04/96   Time: 4:09p
 * Updated in $/nsdemo
 * Got rid of unused variables.  Modified retval slightly to get rid of
 * compiler warnings.
 * 
 * *****************  Version 23  *****************
 * User: Miked        Date: 12/04/96   Time: 1:46p
 * Updated in $/nsdemo
 * Updated version to 1.6.
 * 
 * *****************  Version 22  *****************
 * User: Miked        Date: 10/03/96   Time: 4:58p
 * Updated in $/nsdemo
 * Got rid of some function calls that directly accessed RTC registers and
 * replaced with calls to new RTC commands that do the equivalent.  Moved
 * some code around a little for clarity.  Fixed several non critical bugs
 * in how if/else statements were written.
 * 
 * *****************  Version 21  *****************
 * User: Miked        Date: 8/06/96    Time: 11:59a
 * Updated in $/nsdemo
 * Version 1.4.  Maintainance release.  See README.TXT for info.
 * 
 * *****************  Version 20  *****************
 * User: Miked        Date: 8/05/96    Time: 6:05p
 * Updated in $/nsdemo
 * Added RTTarget support (hooking of IRQ, #ifdef ONTIME).  Also
 * eliminated Watcom differences by using __cdecl for ISR functions.
 * 
 * *****************  Version 19  *****************
 * User: Miked        Date: 7/23/96    Time: 2:26p
 * Updated in $/nsdemo
 * Maintainance release.  README.TXT describes changes.
 * 
 * *****************  Version 18  *****************
 * User: Miked        Date: 7/23/96    Time: 11:53a
 * Updated in $/nsdemo
 * Modified to work with ISR THUNK routines in CPU.ASM.  Interrupts now
 * vector to ISR Thunk routines, which in turn call the C ISRs.  This is a
 * more stable and consistent way to handle interrupts (vs. trying to
 * force C functions to properly
 * save and restore registers and do an IRETD).
 * 
 * *****************  Version 17  *****************
 * User: Miked        Date: 7/22/96    Time: 11:01a
 * Updated in $/nsdemo
 * Changed to check for REV != B instead of REV=C for Int tests.  This
 * way,
 * newer chips will do this test without changing this code.
 * 
 * *****************  Version 16  *****************
 * User: Miked        Date: 7/16/96    Time: 11:54a
 * Updated in $/nsdemo
 * Updated for rev C0 release.
 * 
 * *****************  Version 15  *****************
 * User: Miked        Date: 7/12/96    Time: 3:47p
 * Updated in $/nsdemo
 * Changed printed output slightly to fit with rest of NSDEMO.
 * 
 * *****************  Version 14  *****************
 * User: Miked        Date: 7/01/96    Time: 11:36a
 * Updated in $/nsdemo
 * Removed stuff about Dates Of Month Alarm - which is not useful.  An
 * update to the datasheet will explain how this register works.
 * 
 * *****************  Version 13  *****************
 * User: Miked        Date: 6/28/96    Time: 1:45p
 * Updated in $/nsdemo
 * Modified to call external assembly function for IRETD.  Also SSI and
 * Pharlap
 * now use same code.  Also modified printed messages.
 * 
 * *****************  Version 12  *****************
 * User: Miked        Date: 6/26/96    Time: 3:13p
 * Updated in $/nsdemo
 * Fixed bug in CMOS read.  Was passing a pointer to a pointer instead of
 * a pointer to a USHORT.
 * 
 * *****************  Version 11  *****************
 * User: Noreen       Date: 6/19/96    Time: 11:13a
 * Updated in $/nsdemo
 * Added support for RTC functions. This module is basically the same as
 * rtctest.c in nstest archive, the only difference being the
 * revision/history
 * 
 * *****************  Version 10  *****************
 * User: Miked        Date: 5/03/96    Time: 2:50p
 * Updated in $/nsdemo
 * Maintainence release.
 * 
 * *****************  Version 9  *****************
 * User: Noreen       Date: 4/18/96    Time: 12:35p
 * Updated in $/nsdemo
 * Updated comments for version release
 * 
 * *****************  Version 8  *****************
 * User: Noreen       Date: 4/17/96    Time: 3:26p
 * Updated in $/nsdemo
 * Changed time
 * 
 * *****************  Version 7  *****************
 * User: Miked        Date: 4/17/96    Time: 11:33a
 * Updated in $/nsdemo
 * Modified display messages for nsdemo format.
 * 
 * *****************  Version 6  *****************
 * User: Noreen       Date: 4/17/96    Time: 10:37a
 * Updated in $/nsdemo
 * Removed PeriodInt, AlarmInt and TestCMOS functions.
 * 
 * *****************  Version 5  *****************
 * User: Noreen       Date: 4/17/96    Time: 10:23a
 * Updated in $/nsdemo
 * Redesigned print messages to dprintf
 * 
 * *****************  Version 4  *****************
 * User: Noreen       Date: 4/16/96    Time: 2:47p
 * Updated in $/nsdemo
 * Converted print messages for SSI and Pharlap.
 * Removed calls to AlarmInt and PeriodInt functions.
 * 
 * *****************  Version 3  *****************
 * User: Miked        Date: 4/12/96    Time: 3:07p
 * Branched in $/nsdemo
 * NS Demo
 * 
 * *****************  Version 2  *****************
 * User: Noreen       Date: 4/12/96    Time: 3:01p
 * Updated in $/nstest
 * Updated headers for VSS
 *
 */
//
// COPYRIGHT
//
//      (c) 1996, 1997 National Semiconductor Corporation
//
//---------------------------------------------------------------------------

#include "rtctest.h"

//---------------------------------------------------------------------------
// Globals

PRIVATE volatile BOOL rtc_flag;

//---------------------------------------------------------------------------
// Function Prototypes

PRIVATE USHORT AlarmInt(void);
PRIVATE USHORT TimeCheck(void);
PRIVATE USHORT TestCMOS(void);
PRIVATE USHORT PeriodInt(void);
PRIVATE USHORT TimeValidate( USHORT InitStatus );

//---------------------------------------------------------------------------

void __cdecl ISR_RTC_THUNK(void);  // assembly ISR hooked to interrupt

//---------------------------------------------------------------------------
//
// FUNCTION    RTC_Test
//
// STATUS      genesis
//
// INPUT       none
// OUTPUT      none
// RETURN      USHORT
//               SUCCESS - if call successful
//               FAIL - if any of RTC functions fail
//
// DESCRIPTION
//
//  This function calls the functions implemented to test the real
//  time clock module.
//
//---------------------------------------------------------------------------

USHORT RTC_Test(void)
{
  
  #ifdef PHARLAP
    USHORT irqvec;
    FARPTR old_handler;
    FARPTR new_handler;
  #endif

  USHORT InitStatus;
  USHORT retval = SUCCESS;

  // Start of test
  dprintf("Real Time Clock Test:\r\n");

  // Initialize Real Time Clock
  InitStatus = RTC_Initialize();

  // SUCCESS means VRT bit was high (valid).  FAIL means VRT was low (invalid).
  // any other return is an error 
  if ( ( InitStatus != SUCCESS ) && ( InitStatus != FAIL ) )
  {
    dprintf("  Initialization - **FALIED**\r\n\n");
	return FAIL;
  }

  // This call will set the time and date if invalid, or perform a sanity
  // check on the time and date if invalid.
  if ( TimeValidate( InitStatus ) == SUCCESS )
  {
    
    // If RTC is working, call TimeCheck to make sure RTC is updating
	if ( TimeCheck() != SUCCESS )
	  retval = FAIL;
  
  }
  else
    retval = FAIL;

  // Don't perform interrupt test for NS486SXF rev B
  if (!(  (gDeviceInfo.id == ID_NS486SXF)
	   && (gDeviceInfo.rev == SXF_REV_B) ))
  {
  
    // Hook the Interrupt Handler
    #ifdef PHARLAP
      if ( IRQ_RTC < 8 )
        irqvec = IRQIV_Controller1 + IRQ_RTC;
      else
        irqvec = IRQIV_Controller2 + IRQ_RTC - 8;
      FP_SET(new_handler, ISR_RTC_THUNK, 0x18); // set up the "far pointer"
      _dx_pmiv_get(irqvec, &old_handler);       // get the old handler
      _dx_pmiv_set(irqvec, new_handler);        // set the new handler
    #endif
    
    #ifdef ONTIME
      RTSetVector(RTIRQ0Vector+IRQ_RTC, (void (*)(void)) ISR_RTC_THUNK);
    #endif
	    
    //Use RTC to generate an interrupt
	if ( AlarmInt() != SUCCESS )
	  retval = FAIL;

	//Generate an interrupt based on the periodic interrupt rate
	if ( PeriodInt() != SUCCESS )
	  retval = FAIL;
  
    // Restore the old handler
    #ifdef PHARLAP
      _dx_pmiv_set(irqvec, old_handler);
    #endif
	
  }

  //Simple test to validate CMOS RAM
  if ( TestCMOS() != SUCCESS )
    retval = FAIL;

  // All done
  return retval;

}

//---------------------------------------------------------------------------

// INPUT        InitStatus - SUCCESS or FAIL as shown below
// OUTPUT       none
// RETURN       USHORT - FAIL if anything goes wrong
//                       SUCCESS otherwise
//
// DESCRIPTION  This function will set the date and time (and RTC mode)
//              if InitStatus == FAIL (indicating the VRT bit was
//              invalid).  If InitStatus == SUCCESS (the VRT bit was
//              set indicating RTC registers are valid) this code will
//              perform limit checks to make sure each time register
//              is within bounds.  If any are out of bounds, they will
//              be programmed to a default.  In both cases the date and
//              time is printed at the end of this code.
//
// NOTES        This function is long, but simple.  The length is mostly
//              because both Binary and BCD modes as well as 12 and 24 hour
//              timekeeping modes are supported so that this code can work
//              with whatever mode the RTC was in before it is called.

PRIVATE USHORT TimeValidate( USHORT InitStatus )
{

  BYTE tByte;
  RTC_SET RTCInfo;
  RTC_SET * pRTCInfo = &RTCInfo;
  BOOL writeTimeDate = FALSE;
  const char * DayMap[] = { "Invalid", "Sunday", "Monday", "Tuesday", 
                            "Wednesday", "Thursday", "Friday", "Saturday" };

  // Start of Time Validation
  dprintf("  Time Validation:\r\n");

  // If VRT was invalid, we need to program the time, date, and mode
  if ( InitStatus == FAIL )
  {
    
    // Indicate status
    dprintf("    VRT bit indicates Time and Date are invalid; will be set\r\n");

    // Set RTC mode.  Use Binary, 24 hour, enable daylight savings
	pRTCInfo->Command = SET_MODE;
    pRTCInfo->ModeData = 
        ( RTC_MODE_BINARY | RTC_MODE_24 | RTC_MODE_DAYLIGHT_SAVING_ENABLE );
	if ( RTC_Command( pRTCInfo ) != SUCCESS )
	{   
	  dprintf("    Mode Set - **FALIED**\r\n");
	  return FAIL;
	}

	// Set Default Time of 9:30AM (arbitrary)
	pRTCInfo->Command = SET_TIME;
	pRTCInfo->Seconds = 0;
	pRTCInfo->Minutes = 30;
	pRTCInfo->Hours   = 9;
    if ( RTC_Command( pRTCInfo ) != SUCCESS )
    {
      dprintf("    Time Set - **FAILED**\r\n");
      return FAIL;
    }

    // Set Default Date of Thursday January 2, 1997 (arbitrary)
	pRTCInfo->Command     = SET_DATE;
	pRTCInfo->Year        = 97;
	pRTCInfo->Month       = 1;
	pRTCInfo->DateOfMonth = 2;
	pRTCInfo->DayOfWeek   = 5;
    if ( RTC_Command( pRTCInfo ) != SUCCESS )
    {
      dprintf("    Date Set - **FAILED**\r\n");
      return FAIL;
    }

  }

  // If VRT was valid, we want to perform a sanity check on the
  // time and date registers and adjust if needed.

  else if ( InitStatus == SUCCESS )
  {
    
    // Indicate status
    dprintf("    VRT bit indicates Time and Date are valid already\r\n");

    // Get date, time, and mode
    pRTCInfo->Command = GET_TIME_DATE;
    if ( RTC_Command( pRTCInfo ) != SUCCESS )
    {
      dprintf("    Read or Time and Date - **FAILED**\r\n");
      return FAIL;
    }

    // Sanity Check on Date and Time
    if ( ( pRTCInfo->ModeData & RTC_MODE_BINARY ) == RTC_MODE_BINARY )
    {
      if ( pRTCInfo->Year > 99 )
      {
        dprintf("    Year invalid - will be set to 97\r\n");
        pRTCInfo->Year = 97;
        writeTimeDate = TRUE;
      }
      if ( (pRTCInfo->Month > 12) || (pRTCInfo->Month < 1) )
      {
        dprintf("    Month invalid - will be set to 1\r\n");
        pRTCInfo->Month = 1;
        writeTimeDate = TRUE;
      }
      if ( (pRTCInfo->DateOfMonth > 31) || (pRTCInfo->DateOfMonth < 1) )
      {
        dprintf("    Date of Month invalid - will be set to 2\r\n");
        pRTCInfo->DateOfMonth = 2;
        writeTimeDate = TRUE;
      }
      if ( (pRTCInfo->DayOfWeek > 7) || (pRTCInfo->DayOfWeek < 1) )
      {
        dprintf("    Day of Week invalid - will be set to Thursday\r\n");
        pRTCInfo->DayOfWeek = 5;
        writeTimeDate = TRUE;
      }

      if ( ( pRTCInfo->ModeData & RTC_MODE_24 ) == RTC_MODE_24 )
      {
        
        if ( pRTCInfo->Hours > 23 )
        {
          dprintf("    Hours invalid - will be set to 9 AM\r\n");
          pRTCInfo->Hours = 9;
          writeTimeDate = TRUE;
        }

      }
      else
      {

        // want to ignore the AM/PM bit
        tByte = ( ( (BYTE) pRTCInfo->Hours ) & 0x7F );

        if ( (tByte < 1) || (tByte > 12) )
        {
          dprintf("    Hours invalid - will be set to 9 AM\r\n");
          pRTCInfo->Hours = 9;
          writeTimeDate = TRUE;
        }

      }

      if ( pRTCInfo->Minutes > 59 )
      {
        dprintf("    Minutes invalid - will be set to 30\r\n");
        pRTCInfo->Minutes = 30;
        writeTimeDate = TRUE;
      }
      if ( pRTCInfo->Seconds > 59 )
      {
        dprintf("    Seconds invalid - will be set to 0\r\n");
        pRTCInfo->Seconds = 0;
        writeTimeDate = TRUE;
      }

    } // if RTC_MODE_BINARY
    else
    {
      if ( ( ( ((BYTE)pRTCInfo->Year) & 0x0F ) > 0x09 ) ||
           ( ( ((BYTE)pRTCInfo->Year) & 0xF0 ) > 0x90 ) )
      {
        dprintf("    Year invalid - will be set to 97\r\n");
        pRTCInfo->Year = 0x97;
        writeTimeDate = TRUE;
      }
      if ( ( ( ((BYTE)pRTCInfo->Month) & 0x0F ) > 0x09 ) ||
           ( pRTCInfo->Month > 0x12 ) ||
           ( pRTCInfo->Month < 0x01 ) )
      {
        dprintf("    Month invalid - will be set to 1\r\n");
        pRTCInfo->Month = 0x01;
        writeTimeDate = TRUE;
      }
      if ( ( ( ((BYTE)pRTCInfo->DateOfMonth) & 0x0F ) > 0x09 ) ||
           ( pRTCInfo->DateOfMonth > 0x31 ) ||
           ( pRTCInfo->DateOfMonth < 0x01 ) )
      {
        dprintf("    Date of Month invalid - will be set to 2\r\n");
        pRTCInfo->DateOfMonth = 0x02;
        writeTimeDate = TRUE;
      }
      if ( (pRTCInfo->DayOfWeek > 0x07) || (pRTCInfo->DayOfWeek < 0x01) )
      {
        dprintf("    Day of Week invalid - will be set to Thursday\r\n");
        pRTCInfo->DayOfWeek = 0x05;
        writeTimeDate = TRUE;
      }

      if ( ( pRTCInfo->ModeData & RTC_MODE_24 ) == RTC_MODE_24 )
      {
        
        if ( ( ( ((BYTE)pRTCInfo->Hours) & 0x0F ) > 0x09 ) ||
             ( pRTCInfo->Hours > 0x23 ) )
        {
          dprintf("    Hours invalid - will be set to 9 AM\r\n");
          pRTCInfo->Hours = 0x09;
          writeTimeDate = TRUE;
        }

      }
      else
      {

        // want to ignore the AM/PM bit
        tByte = ( ( (BYTE) pRTCInfo->Hours ) & 0x7F );

        if ( ( ( tByte & 0x0F ) > 0x09 ) ||
             ( tByte > 0x12 ) ||
             ( tByte < 0x01 ) )
        {
          dprintf("    Hours invalid - will be set to 9 AM\r\n");
          pRTCInfo->Hours = 0x09;
          writeTimeDate = TRUE;
        }

      }

      if ( ( ( ((BYTE)pRTCInfo->Minutes) & 0x0F ) > 0x09 ) ||
           ( pRTCInfo->Minutes > 0x59 ) )
      {
        dprintf("    Minutes invalid - will be set to 30\r\n");
        pRTCInfo->Minutes = 0x30;
        writeTimeDate = TRUE;
      }
      if ( ( ( ((BYTE)pRTCInfo->Seconds) & 0x0F ) > 0x09 ) ||
           ( pRTCInfo->Seconds > 0x59 ) )
      {
        dprintf("    Seconds invalid - will be set to 0\r\n");
        pRTCInfo->Seconds = 0x00;
        writeTimeDate = TRUE;
      }

    } // RTC_MODE_BCD

    // Program the Date and Time if needed
    // Only the invalid stuff will be changed since the other parameters
    // in the pRTCInfo structure were unchanged from the earlier command
	if ( writeTimeDate == TRUE )
	{
      pRTCInfo->Command = SET_TIME;
      if ( RTC_Command( pRTCInfo ) != SUCCESS )
      {
        dprintf("    Time Set - **FAILED**\r\n");
        return FAIL;
      }
	  pRTCInfo->Command     = SET_DATE;
      if ( RTC_Command( pRTCInfo ) != SUCCESS )
      {
        dprintf("    Date Set - **FAILED**\r\n");
        return FAIL;
      }
    }  // writeTimeDate == TRUE

  } // InitStatus == SUCCESS

  // Now lets print the current time and date!
  pRTCInfo->Command = GET_TIME_DATE;
  if ( RTC_Command( pRTCInfo ) != SUCCESS )
  {
    dprintf("    Read of Time and Date - **FAILED**\r\n");
    return FAIL;
  }
  
  if ( ( pRTCInfo->ModeData & RTC_MODE_BINARY ) == RTC_MODE_BINARY )
  {
    dprintf("    Current date is %s %d.%d.%d \r\n",
            DayMap[pRTCInfo->DayOfWeek],
            pRTCInfo->Year,
            pRTCInfo->Month,
            pRTCInfo->DateOfMonth);
    if ( ( pRTCInfo->ModeData & RTC_MODE_24 ) == RTC_MODE_24 )
    {
      dprintf("    Current time is %d:%d:%d \r\n",
              pRTCInfo->Hours,
              pRTCInfo->Minutes,
              pRTCInfo->Seconds);
    }
    else
    {
      tByte = ( (BYTE) pRTCInfo->Hours );
      dprintf("    Current time is %d:%d:%d %s \r\n",
              (tByte&0x7F),
              pRTCInfo->Minutes,
              pRTCInfo->Seconds,
              ( (tByte&0x80) ? "pm":"am") 
              );
    }
  }
  else
  {
    dprintf("    Current date is %s %X.%X.%X \r\n",
            DayMap[pRTCInfo->DayOfWeek],
            pRTCInfo->Year,
            pRTCInfo->Month,
            pRTCInfo->DateOfMonth);
    if ( ( pRTCInfo->ModeData & RTC_MODE_24 ) == RTC_MODE_24 )
    {
      dprintf("    Current time is %X:%X:%X \r\n",
              pRTCInfo->Hours,
              pRTCInfo->Minutes,
              pRTCInfo->Seconds);
    }
    else
    {
      tByte = ( (BYTE) pRTCInfo->Hours );
      dprintf("    Current time is %X:%X:%X %s \r\n",
              (tByte&0x7F),
              pRTCInfo->Minutes,
              pRTCInfo->Seconds,
              ( (tByte&0x80) ? "pm":"am") 
              );
    }
  }

  // If we made it this far, all is OK
  return SUCCESS;

}

//---------------------------------------------------------------------------
//
// FUNCTION    TimeCheck
//
// INPUT       none
// OUTPUT      none
// RETURN      USHORT
//               SUCCESS - if call successful
//               FAIL - if any of RTC functions fail
//
// DESCRIPTION
//
//   This function verifies that the RTC seconds register is updating.
//
//---------------------------------------------------------------------------

PRIVATE USHORT TimeCheck(void)
{

  int i;
  BYTE tByte;
  RTC_SET InitialRTCInfo;
  RTC_SET FinalRTCInfo;
  RTC_SET * pInitialRTCInfo = &InitialRTCInfo;
  RTC_SET * pFinalRTCInfo = &FinalRTCInfo;

  // Start of Test
  dprintf("  Making sure the RTC is updating:\r\n");

  // Read initial time
  pInitialRTCInfo->Command = GET_TIME_DATE;
  if ( RTC_Command( pInitialRTCInfo ) != SUCCESS )
  {
    dprintf("    Read of Time and Date - **FAILED**\r\n");
    return FAIL;
  }
  
  // Display Initial Time
  if ( ( pInitialRTCInfo->ModeData & RTC_MODE_BINARY ) == RTC_MODE_BINARY )
  {
    if ( ( pInitialRTCInfo->ModeData & RTC_MODE_24 ) == RTC_MODE_24 )
    {
      dprintf("    Initial time is %d:%d:%d \r\n",
              pInitialRTCInfo->Hours,
              pInitialRTCInfo->Minutes,
              pInitialRTCInfo->Seconds);
    }
    else
    {
      tByte = ( (BYTE) pInitialRTCInfo->Hours );
      dprintf("    Initial time is %d:%d:%d %s \r\n",
              (tByte&0x7F),
              pInitialRTCInfo->Minutes,
              pInitialRTCInfo->Seconds,
              ( (tByte&0x80) ? "pm":"am") 
              );
    }
  }
  else
  {
    if ( ( pInitialRTCInfo->ModeData & RTC_MODE_24 ) == RTC_MODE_24 )
    {
      dprintf("    Initial time is %X:%X:%X \r\n",
              pInitialRTCInfo->Hours,
              pInitialRTCInfo->Minutes,
              pInitialRTCInfo->Seconds);
    }
    else
    {
      tByte = ( (BYTE) pInitialRTCInfo->Hours );
      dprintf("    Initial time is %X:%X:%X %s \r\n",
              (tByte&0x7F),
              pInitialRTCInfo->Minutes,
              pInitialRTCInfo->Seconds,
              ( (tByte&0x80) ? "pm":"am") 
              );
    }
  }

  // Delay for 2 seconds
  dprintf("    Two second delay\r\n");
  for(i=0; i<40; i++) PIT_Delay(1,50);

  // Read final time
  pFinalRTCInfo->Command = GET_TIME_DATE;
  if ( RTC_Command( pFinalRTCInfo ) != SUCCESS )
  {
    dprintf("    Read of Time and Date - **FAILED**\r\n");
    return FAIL;
  }

  // Display Final Time
  if ( ( pFinalRTCInfo->ModeData & RTC_MODE_BINARY ) == RTC_MODE_BINARY )
  {
    if ( ( pFinalRTCInfo->ModeData & RTC_MODE_24 ) == RTC_MODE_24 )
    {
      dprintf("    Final time is %d:%d:%d \r\n",
              pFinalRTCInfo->Hours,
              pFinalRTCInfo->Minutes,
              pFinalRTCInfo->Seconds);
    }
    else
    {
      tByte = ( (BYTE) pFinalRTCInfo->Hours );
      dprintf("    Final time is %d:%d:%d %s \r\n",
              (tByte&0x7F),
              pFinalRTCInfo->Minutes,
              pFinalRTCInfo->Seconds,
              ( (tByte&0x80) ? "pm":"am") 
              );
    }
  }
  else
  {
    if ( ( pFinalRTCInfo->ModeData & RTC_MODE_24 ) == RTC_MODE_24 )
    {
      dprintf("    Final time is %X:%X:%X \r\n",
              pFinalRTCInfo->Hours,
              pFinalRTCInfo->Minutes,
              pFinalRTCInfo->Seconds);
    }
    else
    {
      tByte = ( (BYTE) pFinalRTCInfo->Hours );
      dprintf("    Final time is %X:%X:%X %s \r\n",
              (tByte&0x7F),
              pFinalRTCInfo->Minutes,
              pFinalRTCInfo->Seconds,
              ( (tByte&0x80) ? "pm":"am") 
              );
    }
  }

  // See if the clock is ticking
  if ( pInitialRTCInfo->Seconds == pFinalRTCInfo->Seconds )
  {
    dprintf("    RTC seconds are not updating! - **FAILED**\r\n");
    return FAIL;
  }
  else
  {
    dprintf("    RTC seconds are updating! - PASSED\r\n");
  }

  // If we made it this far, all is OK
  return SUCCESS;

}

//---------------------------------------------------------------------------
//
// FUNCTION    PeriodInt
//
// STATUS      genesis
//
// INPUT       none
// OUTPUT      none
// RETURN      USHORT
//               SUCCESS - if call successful
//               FAIL - if any of RTC functions fail
//
// DESCRIPTION
//  This function generates an interrupt condition based on the periodic
//  interrupt rate.
//
//---------------------------------------------------------------------------

PRIVATE USHORT PeriodInt(void)
{
	RTC_SET TimeInfo;
	USHORT retval = SUCCESS;

	//Initialize pointer to RTC_SET structure
	RTC_SET * pTimeInfo = &TimeInfo;

	// Set interrupt flag to false
	rtc_flag = FALSE;
	
	// Enable Interrupt Level
	PIC_Enable(IRQ_RTC);

	// Enable Periodic Interrupts
	pTimeInfo->Command = SET_PERIODIC_ALARM;
	pTimeInfo->PeriodicRate = RTC_PERIODIC_RATE_9;
	if(RTC_Command(pTimeInfo) == SUCCESS)
	{
	   // Delay
	   PIT_Delay(1,70);

	   // Check for success
	   if(rtc_flag == TRUE)
	   {
	      dprintf("  RTC Period Alarm Interrupt Test - PASSED \r\n");
	   }
	   else
	   {
	     dprintf("  RTC Period Alarm Interrupt Test - FAILED \r\n");
	     retval = FAIL;
	   }
	  
	  // Disable further interrupts
	  pTimeInfo->Command = DISABLE_PERIODIC_ALARM;
	  RTC_Command(pTimeInfo);
	
	}
	else
	{
	  dprintf("  RTC Setting of Periodic Alarm - FAILED \r\n\n");
	  retval = FAIL;
	}
	    
  PIC_Disable(IRQ_RTC);

  return retval;

}

//---------------------------------------------------------------------------
//
// FUNCTION    AlarmInt
//
// STATUS      genesis
//
// INPUT       none
// OUTPUT      none
// RETURN      USHORT
//               SUCCESS - if call successful
//               FAIL - if any of RTC functions fail
//
// DESCRIPTION
//  This function generates an Alarm interrupt condition.
//
// NOTES
//  After each update of the timing registers, the seconds, minutes and hours
//  are compared with the seconds alarm, minutes alarm and hours alarm. If
//  equal and the Alarm interrupt enable is set in control register B an
//  interrupt is generated.
//
//---------------------------------------------------------------------------

PRIVATE USHORT AlarmInt(void)
{

	BYTE i;
	RTC_SET TimeInfo;
	USHORT retval = SUCCESS;
	
	//Initialize pointer to RTC_SET structure
	RTC_SET * pTimeInfo = &TimeInfo;
	
	// Set interrupt flag to false
	rtc_flag = FALSE;

    // Get current time
    pTimeInfo->Command = GET_TIME_DATE;
    if ( RTC_Command( pTimeInfo ) != SUCCESS )
    {
      dprintf("  Read of Time and Date during Alarm check - **FAILED**\r\n");
      return FAIL;
    }
    
    // Set alarm to go off in two seconds.  Set hours and minutes alarms
    // to don't care so we don't have to compute if they will roll past
    // their current value.  This we are setting an alarm to go off
    // once a minute.  But we will shut it off after two seconds.

	pTimeInfo->MinutesAlarm = ALARM_DONT_CARE;
	pTimeInfo->HoursAlarm   = ALARM_DONT_CARE;

    if ( ( pTimeInfo->ModeData & RTC_MODE_BINARY ) == RTC_MODE_BINARY )
    {
      if ( ( pTimeInfo->Seconds ) > 57 )
        pTimeInfo->SecondsAlarm = ( pTimeInfo->Seconds + 2 - 60 );
      else
        pTimeInfo->SecondsAlarm = ( pTimeInfo->Seconds + 2 );
    }
    else
    {
      if ( ( pTimeInfo->Seconds ) > 0x57 )
        pTimeInfo->SecondsAlarm = ( pTimeInfo->Seconds + 2 - 0x5A );
      else
        pTimeInfo->SecondsAlarm = ( pTimeInfo->Seconds + 2 );
    }

    // Enable Interrupt
	PIC_Enable(IRQ_RTC);

    // Set Alarm
	pTimeInfo->Command = SET_ALARM;
	RTC_Command(pTimeInfo);

	// Implement 4 sec delay
	for(i=0; i<80; i++) PIT_Delay(1,50);

    // See if the interrupt happened
	if(rtc_flag == TRUE)
	{
	  dprintf("  RTC Alarm Interrupt Test - PASSED \r\n");
	}
	else
	{  
	  dprintf("  RTC Alarm Interrupt Test - FAILED \r\n");
	  retval = FAIL;
	}

	// Disable Interrupt    
    PIC_Disable(IRQ_RTC);

    // Disable Alarm
	pTimeInfo->Command = DISABLE_ALARM;
	RTC_Command(pTimeInfo);

    // Thats it!
    return retval;

}

//---------------------------------------------------------------------------
//
// FUNCTION    TestCMOS
//
// STATUS      genesis
//
// INPUT       none
// OUTPUT      none
// RETURN      USHORT
//               SUCCESS - if read/writes to CMOS locations PASS
//               FAIL - if any of read/writes to CMOS locations FAIL
//
// DESCRIPTION
//  This function performs a series of reads and writes to each
//  of the locations in the CMOS RAM. It maintains a count of the
//  read/writes that passed and failed and displays this information
//  to the user. If all of the locations in CMOS RAM can be written to
//  and read from successfully the function returns SUCCESS otherwise
//  FAIL is returned.
//
//---------------------------------------------------------------------------

PRIVATE USHORT TestCMOS(void)
{
    USHORT location;        //Location of RAM 1-50
    USHORT  value=2;       //Value to write to RAM
    USHORT rvalue;         // value read
    USHORT failcount=0;

    for(location=0; location<=50; location++)
    {
      if(RTC_RAM_Write(location, value) == SUCCESS)
	 if(RTC_RAM_Read(location, &rvalue) == SUCCESS)
	 {
	    if(rvalue != value)
	       failcount++;
	 }
    }

    if(failcount >= 1)
    {
       dprintf("  CMOS Test - FAILED\r\n\n");
       return FAIL;
    }
    else
    {
      dprintf("  CMOS Test - PASSED\r\n\n");
      return SUCCESS;
    }
}

//---------------------------------------------------------------------------
//
// FUNCTION    ISR_RTC
//
// STATUS      genesis
//
// INPUT       none
// OUTPUT      none
// RETURN      none
//
// DESCRIPTION
//  This routine services interrupts generated by the Real Time Clock.
//
// NOTES
//---------------------------------------------------------------------------

// This is the ISR for the RTC interrupt.  The assembly function 
// ISR_RTC_THUNK is the actual function pointed to by the IDT.  That
// function calls this function.

void __cdecl ISR_RTC(void)
{
  
  RTC_SET TimeInfo;

  // Set RTC ISR flag
  rtc_flag = TRUE;
	
  // Clear interrupt in RTC
  TimeInfo.Command = CLEAR_INTERRUPT;
  RTC_Command(&TimeInfo);

  // Issue End of Interrupt to PIC
  PIC_EOI(IRQ_RTC);

}

//---------------------------------------------------------------------------
// END       rtctest.c
//---------------------------------------------------------------------------
