/**********************************************************/
/* IBM 3278 keyboard to PS/2 Interface                    */
/*                                                        */
/* Tested with ATmega168 & ATmega88                       */
/*                                                        */
/* Copyright (c) 2013, Henk Stegeman                      */
/*               2001, Jamie Honan                        */
/*               ????, Chaitanya Thummar                  */
/* All rights reserved.                                   */
/*                                                        */
/* This program is free software; you can redistribute it */
/* and/or modify it under the terms of the GNU General    */
/* Public License as published by the Free Software       */
/* Foundation; either version 2 of the License, or        */
/* (at your option) any later version.                    */
/*                                                        */
/* This program is distributed in the hope that it will   */
/* be useful, but WITHOUT ANY WARRANTY; without even the  */
/* implied warranty of MERCHANTABILITY or FITNESS FOR A   */
/* PARTICULAR PURPOSE.  See the GNU General Public        */
/* License for more details.                              */
/*                                                        */
/* You should have received a copy of the GNU General     */
/* Public License along with this program; if not, write  */
/* to the Free Software Foundation, Inc.,                 */
/* 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA */
/*                                                        */
/* Licence can be viewed at                               */
/* http://www.fsf.org/licenses/gpl.txt                    */
/*                                                        */
/* History:                                               */
/* 07/08/2013 More or less initial version.               */
/* 16/08/2013 Correction [] and added ALT+PFxx codes.     */
/* 25/08/2013 Completed all CNTL/x control codes.         */
/* 24/11/2013 Correction RESET cmd logic + adj clk speed  */
/* 24/11/2013 Completed HOST command logic                */
/*                                                        */
/**********************************************************/

// Some includes.
#include <avr/io.h>
#include <avr/pgmspace.h>
#include <avr/interrupt.h>

#include "global.h"
#include "IBM_keyboard.h"
#include "keysta.h"
#include "uart.h"

// #define DEBUG                          // Mark it as comment if no debug.

extern volatile unsigned char kbd_flags;

uint8_t buf[16+1]= { 0x00 };           // Scan codes buffer.
int8_t  buf_idx = -1;                  // Buffer index pointer.  
uint8_t shift_state = SHIFT_OFF;
uint8_t alt_state = ALT_OFF;
uint16_t delay_count = F_CPU / 4000;   // Needed for msdelay routine.
uint8_t kb_ident;                      // Keyboard jumper settings.


// *** Hex to ascii conversion routine.
char hex2asc(uint8_t hex) {
   
   uint8_t asc;
   
   hex = hex & 0x0F;
   if (hex > 0x09) {
      asc = 'A' + (hex - 0x0A);
   } else {
      asc = '0' + hex;
   }
   return (asc);
}


// *** Acknowledge 3278 kbd.
void kbd_ack(void) {
   cbi(PORTD, KB_ACK);
   ms_delay(2);
   sbi(PORTD, KB_ACK);
}


// *** Push byte to buffer. 
void push_buf(uint8_t byte) {
   cli();

   // Transfer PS2 scan code to buffer.
   if (buf_idx < 16)       // Push pointer.
      buf_idx = buf_idx + 1;
   if (buf_idx == 16) {    // Buffer full ?
      buf[buf_idx] = BUF_OVERFLOW;
   } else {
      buf[buf_idx] = byte;
   } 
 
   sei();
}


// *** Pull byte from buffer.
uint8_t pull_buf(void) {
   uint8_t byte, i;

   cli();
   byte = buf[0];
   // Pull buffer content
   for (i = 0; i < 17; i++) 
      buf[i] = buf[i+1];
   buf_idx = buf_idx - 1;  // Pull pointer
   sei();
  
   return(byte);
}
   

// ************************************
// ***   Main routine starts here.  ***
// ************************************

int main () {
   uint8_t byte, last_sent = 0;
	uint8_t i; 

   // Initialize all I/O pins.
   init_io();  

   buf_idx = -1;            // Clear buffer    

   ms_delay(500);           // POR reset keyboard
   
   // Read and save keyboard ident jumper settings.
   kb_ident = PINB & (1 << KB_IDENT_0 | 1 << KB_IDENT_1);
   if (kb_ident == 0x00) {       // Start UART debugger ?
      
      // *** ENTERING DEBUG MODE ***
      init_uart();               // Init UART   

      EIMSK |= (1 << INT1);      // Enable INT1
      EICRA |= (1 << ISC11);     // Trigger on falling edge of INT1
      sei();                     // Enable global interrupts 

      i = uart_putchar('\n');    // Xmit prompt 
      i = uart_putchar('>');
      
      while(1) {
         if (buf_idx != -1) {    // Scan code in buffer ?
            byte = pull_buf();   // Yes, get it. 
            // Transmit it in hex.
            i = uart_putchar(' ');
            i = uart_putchar(hex2asc(byte>>4));
            i = uart_putchar(hex2asc(byte));
         }
      }     
               
   } else {                      // Start normal mode 

      // *** ENTERING NORMAL MODE ***
      init_kbd();                // Init PS/2 interface + timer. 

#ifdef DEBUG
      init_uart();               // Init UART
#endif		  
      ms_delay(500);             // Simulate BAT time.
      
      EIMSK |= (1 << INT1);      // Enable INT1
      EICRA |= (1 << ISC11);     // Trigger on falling edge of INT1
      sei();                     // Enable global interrupts 
 
      // Acknowledge it.
      kbd_ack();

      alt_state = BREAK;
      shift_state = BREAK;

      buf_idx = -1;              // Clear buffer.
      push_buf(SELF_TEST_OK);    // Signal HOST: Self Test Passed.  
 
      // *** Main loop... 
      while(1) {
         // If command received from HOST: process it.
         if (kbd_flags & RX_BUF_FULL) {  // Any char received from HOST ?
            // Get and save received byte
            byte = kbd_get_rx_char();
#ifdef DEBUG
            // Transmit it via RS232 port in hex
            i = uart_putchar(' ');
            i = uart_putchar('R');
            i = uart_putchar(hex2asc(byte>>4));
            i = uart_putchar(hex2asc(byte));
#endif
            switch(byte) {
               case 0xEE:        // Echo...
                  push_buf(0xEE);
                  continue;
               case 0xF2:        // Read id */
                  push_buf(PS2_ACK);
                  ms_delay(1);
                  push_buf(0xAB);
                  ms_delay(1);
                  push_buf(0x83);
                  continue;
               case 0xFF:        // Reset
					   ms_delay(1);
                  push_buf(PS2_ACK);
						ms_delay(250);
						push_buf(SELF_TEST_OK);
                  continue;
               case 0xFE:        // Resend 
                  kbd_set_tx(last_sent); // Re-transmit it to host.
                  continue;
/*             case 0xF0:        // scan code set 
                  push_buf(0xFA);
                  continue;                                  */
               case 0xED:        // LED indicators
                  push_buf(PS2_ACK);
						// default: will ACK 2nd byte. 
                  continue;
               case 0xF3:        // Set typematic rate
                  push_buf(PS2_ACK);
						// default: will ACK 2nd byte. 
                  continue; 
               case 0xF4:        // Enable
					   ms_delay(1);
                  buf_idx = -1;  // Clear buffer 
                  push_buf(PS2_ACK);
                  continue;
/*             case 0xF5:        // Disable
                  push_buf(PS2_ACK);
                  continue;
               case 0xF6:        // Set Default
                  buf_idex = -1;      */
               default:
                  push_buf(PS2_ACK);
                  break;
            }
         }

         // If buffer filled: transmit it's content to HOST. 
         if (kbd_flags & TX_BUF_EMPTY) {   // Ready to transmit to HOST ?
            if (buf_idx != -1) { // Scan code in buffer ?
               byte = pull_buf();
#ifdef DEBUG 
               // Transmit it via RS232 port in hex
               i = uart_putchar(' ');
               i = uart_putchar('T');
               i = uart_putchar(hex2asc(byte>>4));
               i = uart_putchar(hex2asc(byte));
#endif
               ms_delay(1);
               kbd_set_tx(byte); // Transmit it to host.
					last_sent = byte; // save last byte sent. 
            }           
         } 
      } // End main loop
   }           
}


// ***************************************
// ***   Interrupt 1 handler routine.  ***
// ***************************************

ISR(SIG_INTERRUPT1) {
   // Data Available signal from IBM keyboard. 
   uint8_t kbd_sc;
      
   // Read key scan code and make/break bit.   
   kbd_sc = (PINC & 0x3F);
   kbd_sc = kbd_sc | (PIND & (1 << KB_SCAN_6)); 
   kbd_sc = kbd_sc | (PINB & (1 << KB_MAK_BRK));

   if (kbd_sc == ALT_MAKE) {
      alt_state = MAKE;    // ALT key is pressed
      
      // Acknowledge it.
      kbd_ack();
      return;
   } 
   
   if (kbd_sc == ALT_BREAK) {  
      alt_state = BREAK;   // ALT key is released
      
      // Acknowledge it.
      kbd_ack();
      return;
   }

   if (kbd_sc == CAPS_MAKE) {
      shift_state = MAKE;  // CAPS key is pressed
      
      // Acknowledge it.
      kbd_ack();
      return;
   } 
   
   if (kbd_sc == CAPS_BREAK) {  
                           // CAPS key is released
      // Acknowledge it.
      kbd_ack();
      return;
   } 

   if ((kbd_sc == LSHIFT_MAKE) || (kbd_sc == RSHIFT_MAKE))  {
      shift_state = MAKE;  // A SHIFT key is pressed
      
      // Acknowledge it.
      kbd_ack();
      return;
   } 
   
   if ((kbd_sc == LSHIFT_BREAK) || (kbd_sc == RSHIFT_BREAK)) {  
      shift_state = BREAK; // SHIFT key is released
      
      // Acknowledge it.
      kbd_ack();
      return;
   } 
   
   // Translate kbd scan code to PS2 scancode(s)
   // and transfer it to Tx buffer.
   kbd_to_ps2(kbd_sc, shift_state, alt_state); 

   ms_delay(8);

   // Acknowledge it.
   kbd_ack();
}

 
