/**********************************************************/
/* IBM 3278 keyboard to PS/2                              */
/*                                                        */
/* Copyright (c) 2013, Henk Stegeman                      */
/*                                                        */
/* All rights reserved.                                   */
/*                                                        */
/* This program is free software; you can redistribute it */
/* and/or modify it under the terms of the GNU General    */
/* Public License as published by the Free Software       */
/* Foundation; either version 2 of the License, or        */
/* (at your option) any later version.                    */
/*                                                        */
/* This program is distributed in the hope that it will   */
/* be useful, but WITHOUT ANY WARRANTY; without even the  */
/* implied warranty of MERCHANTABILITY or FITNESS FOR A   */
/* PARTICULAR PURPOSE.  See the GNU General Public        */
/* License for more details.                              */
/*                                                        */
/* You should have received a copy of the GNU General     */
/* Public License along with this program; if not, write  */
/* to the Free Software Foundation, Inc.,                 */
/* 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA */
/*                                                        */
/* Licence can be viewed at                               */
/* http://www.fsf.org/licenses/gpl.txt                    */
/*                                                        */
/**********************************************************/

// Translate 3278 kbd scan code to a PS/2 scan code
// Note: currently only 1 table implemented. 

#include <inttypes.h>
#include <avr/io.h>
#include <avr/pgmspace.h>
#include <avr/interrupt.h>
#include <string.h>

#include "global.h"

#define NO_MAKE        0
#define ALT_MAKE       1
#define SHIFT_MAKE     2
#define SHIFT_ALT_MAKE 3

void push_buf(uint8_t byte);

extern uint8_t kb_ident;               // Id jumpers
uint8_t reset_flag = BREAK;            // Reset make/break flag                      


// ****************************************************
// *** Table #1 (Jumpers: 0 = removed, 1 = present) ***
// ****************************************************

static uint8_t table1_sb_ab[] PROGMEM = {  // Shift break, alt break.
//  SC   LGTH  --- PS2 SC -->
   0x03, 0x03, 0x1C, 0xF0, 0x1C,  // a
   0x43, 0x03, 0x32, 0xF0, 0x32,  // b         
   0x23, 0x03, 0x21, 0xF0, 0x21,  // c           
   0x63, 0x03, 0x23, 0xF0, 0x23,  // d           
   0x13, 0x03, 0x24, 0xF0, 0x24,  // e          
   0x53, 0x03, 0x2B, 0xF0, 0x2B,  // f          
   0x33, 0x03, 0x34, 0xF0, 0x34,  // g            
   0x73, 0x03, 0x33, 0xF0, 0x33,  // h 
   0x0B, 0x03, 0x43, 0xF0, 0x43,  // i           
   0x4B, 0x03, 0x3B, 0xF0, 0x3B,  // j           
   0x2B, 0x03, 0x42, 0xF0, 0x42,  // k    
   0x6B, 0x03, 0x4B, 0xF0, 0x4B,  // l 
   0x1B, 0x03, 0x3A, 0xF0, 0x3A,  // m 
   0x5B, 0x03, 0x31, 0xF0, 0x31,  // n 
   0x3B, 0x03, 0x44, 0xF0, 0x44,  // o 
   0x7B, 0x03, 0x4D, 0xF0, 0x4D,  // p 
   0x07, 0x03, 0x15, 0xF0, 0x15,  // q 
   0x47, 0x03, 0x2D, 0xF0, 0x2D,  // r 
   0x27, 0x03, 0x1B, 0xF0, 0x1B,  // s 
   0x67, 0x03, 0x2C, 0xF0, 0x2C,  // t 
   0x17, 0x03, 0x3C, 0xF0, 0x3C,  // u 
   0x57, 0x03, 0x2A, 0xF0, 0x2A,  // v 
   0x37, 0x03, 0x1D, 0xF0, 0x1D,  // w 
   0x77, 0x03, 0x22, 0xF0, 0x22,  // x 
   0x0F, 0x03, 0x35, 0xF0, 0x35,  // y 
   0x4F, 0x03, 0x1A, 0xF0, 0x1A,  // z 

   0x5E, 0x03, 0x0E, 0xF0, 0x0E,  // `
   0x42, 0x03, 0x16, 0xF0, 0x16,  // 1 
   0x22, 0x03, 0x1E, 0xF0, 0x1E,  // 2     
   0x62, 0x03, 0x26, 0xF0, 0x26,  // 3    
   0x12, 0x03, 0x25, 0xF0, 0x25,  // 4    
   0x52, 0x03, 0x2E, 0xF0, 0x2E,  // 5    
   0x32, 0x03, 0x36, 0xF0, 0x36,  // 6    
   0x72, 0x03, 0x3D, 0xF0, 0x3D,  // 7    
   0x0A, 0x03, 0x3E, 0xF0, 0x3E,  // 8    
   0x4A, 0x03, 0x46, 0xF0, 0x46,  // 9    
   0x02, 0x03, 0x45, 0xF0, 0x45,  // 0 

   0x3F, 0x03, 0x4C, 0xF0, 0x4C,  // ;
   0x06, 0x03, 0x4E, 0xF0, 0x4E,  // -
   0x44, 0x03, 0x55, 0xF0, 0x55,  // =
   0x54, 0x03, 0x5D, 0xF0, 0x5D,  // '\'
   0x24, 0x03, 0x52, 0xF0, 0x52,  // '
   0x78, 0x06, 0x12, 0x54, 0xF0, 0x54, 0xF0, 0x12,  // {
   0x48, 0x06, 0x12, 0x41, 0xF0, 0x41, 0xF0, 0x12,  // <
   0x66, 0x03, 0x41, 0xF0, 0x41,  // , 
   0x26, 0x03, 0x49, 0xF0, 0x49,  // .
   0x14, 0x03, 0x4A, 0xF0, 0x4A,  // '/'
   
   0x18, 0x05, 0xE0, 0x70, 0xE0, 0xF0, 0x70,  // Insert
   0x58, 0x05, 0xE0, 0x71, 0xE0, 0xF0, 0x71,  // Delete  

   0x05, 0x03, 0x76, 0xF0, 0x76,  // Esc
         
   0x04, 0x03, 0x29, 0xF0, 0x29,  // Space
   0x08, 0x03, 0x5A, 0xF0, 0x5A,  // Enter [<-+]
   0x0C, 0x03, 0x5A, 0xF0, 0x5A,  // Enter 
   0x09, 0x03, 0x5A, 0xF0, 0x5A,  // Enter [PF21] 
   0x69, 0x03, 0x5A, 0xF0, 0x5A,  // Enter [PF24]
  
   0x38, 0x06, 0xE0, 0x75, 0xF0, 0xE0, 0xF0, 0x75,  // Arrow up
   0x64, 0x06, 0xE0, 0x72, 0xF0, 0xE0, 0xF0, 0x72,  // Arrow down
   0x34, 0x06, 0xE0, 0x6B, 0xF0, 0xE0, 0xF0, 0x6B,  // Arrow left
   0x2C, 0x06, 0xE0, 0x74, 0xF0, 0xE0, 0xF0, 0x74,  // Arrow right 

   0x36, 0x03, 0x0D, 0xF0, 0x0D,                    // Tab ->
   0x56, 0x06, 0x12, 0x0D, 0xF0, 0x0D, 0xF0, 0x12,  // Tab <- 
   0x46, 0x03, 0x66, 0xF0, 0x66,  // Back space
     
   0xFF 
   } ;

static uint8_t table1_sm_ab[] PROGMEM = {  // Shift make, alt break.
//  SC   LGTH  --- PS2 SC -->
   0x03, 0x06, 0x12, 0x1C, 0xF0, 0x1C, 0xF0, 0x12,  // A
   0x43, 0x06, 0x12, 0x32, 0xF0, 0x32, 0xF0, 0x12,  // B         
   0x23, 0x06, 0x12, 0x21, 0xF0, 0x21, 0xF0, 0x12,  // C           
   0x63, 0x06, 0x12, 0x23, 0xF0, 0x23, 0xF0, 0x12,  // D           
   0x13, 0x06, 0x12, 0x24, 0xF0, 0x24, 0xF0, 0x12,  // E          
   0x53, 0x06, 0x12, 0x2B, 0xF0, 0x2B, 0xF0, 0x12,  // F          
   0x33, 0x06, 0x12, 0x34, 0xF0, 0x34, 0xF0, 0x12,  // G            
   0x73, 0x06, 0x12, 0x33, 0xF0, 0x33, 0xF0, 0x12,  // H 
   0x0B, 0x06, 0x12, 0x43, 0xF0, 0x43, 0xF0, 0x12,  // I           
   0x4B, 0x06, 0x12, 0x3B, 0xF0, 0x3B, 0xF0, 0x12,  // J           
   0x2B, 0x06, 0x12, 0x42, 0xF0, 0x42, 0xF0, 0x12,  // K    
   0x6B, 0x06, 0x12, 0x4B, 0xF0, 0x4B, 0xF0, 0x12,  // L 
   0x1B, 0x06, 0x12, 0x3A, 0xF0, 0x3A, 0xF0, 0x12,  // M 
   0x5B, 0x06, 0x12, 0x31, 0xF0, 0x31, 0xF0, 0x12,  // N 
   0x3B, 0x06, 0x12, 0x44, 0xF0, 0x44, 0xF0, 0x12,  // O 
   0x7B, 0x06, 0x12, 0x4D, 0xF0, 0x4D, 0xF0, 0x12,  // P 
   0x07, 0x06, 0x12, 0x15, 0xF0, 0x15, 0xF0, 0x12,  // Q 
   0x47, 0x06, 0x12, 0x2D, 0xF0, 0x2D, 0xF0, 0x12,  // R 
   0x27, 0x06, 0x12, 0x1B, 0xF0, 0x1B, 0xF0, 0x12,  // S 
   0x67, 0x06, 0x12, 0x2C, 0xF0, 0x2C, 0xF0, 0x12,  // T 
   0x17, 0x06, 0x12, 0x3C, 0xF0, 0x3C, 0xF0, 0x12,  // U 
   0x57, 0x06, 0x12, 0x2A, 0xF0, 0x2A, 0xF0, 0x12,  // V 
   0x37, 0x06, 0x12, 0x1D, 0xF0, 0x1D, 0xF0, 0x12,  // W 
   0x77, 0x06, 0x12, 0x22, 0xF0, 0x22, 0xF0, 0x12,  // X 
   0x0F, 0x06, 0x12, 0x35, 0xF0, 0x35, 0xF0, 0x12,  // Y 
   0x4F, 0x06, 0x12, 0x1A, 0xF0, 0x1A, 0xF0, 0x12,  // Z 
   0x42, 0x06, 0x12, 0x16, 0xF0, 0x16, 0xF0, 0x12,  // ! 
   0x22, 0x06, 0x12, 0x1E, 0xF0, 0x1E, 0xF0, 0x12,  // @     
   0x62, 0x06, 0x12, 0x26, 0xF0, 0x26, 0xF0, 0x12,  // #    
   0x12, 0x06, 0x12, 0x25, 0xF0, 0x25, 0xF0, 0x12,  // $    
   0x52, 0x06, 0x12, 0x2E, 0xF0, 0x2E, 0xF0, 0x12,  // %    
   0x32, 0x06, 0x12, 0x36, 0xF0, 0x36, 0xF0, 0x12,  // ^    
   0x72, 0x06, 0x12, 0x3D, 0xF0, 0x3D, 0xF0, 0x12,  // &    
   0x0A, 0x06, 0x12, 0x3E, 0xF0, 0x3E, 0xF0, 0x12,  // *    
   0x4A, 0x06, 0x12, 0x46, 0xF0, 0x46, 0xF0, 0x12,  // (    
   0x02, 0x06, 0x12, 0x45, 0xF0, 0x45, 0xF0, 0x12,  // )
   
   0x5E, 0x06, 0x12, 0x0E, 0xF0, 0x0E, 0xF0, 0x12,  // ~
   0x06, 0x06, 0x12, 0x4E, 0xF0, 0x4E, 0xF0, 0x12,  // _
   0x44, 0x06, 0x12, 0x55, 0xF0, 0x55, 0xF0, 0x12,  // +
   0x6C, 0x06, 0x12, 0x16, 0xF0, 0x16, 0xF0, 0x12,  // !
   0x54, 0x06, 0x12, 0x5D, 0xF0, 0x5D, 0xF0, 0x12,  // |
   0x3F, 0x06, 0x12, 0x4C, 0xF0, 0x4C, 0xF0, 0x12,  // :
   0x24, 0x06, 0x12, 0x52, 0xF0, 0x52, 0xF0, 0x12,  // "
   0x78, 0x06, 0x12, 0x5B, 0xF0, 0x5B, 0xF0, 0x12,  // }
   0x66, 0x03, 0x41, 0xF0, 0x41,  // , 
   0x26, 0x03, 0x49, 0xF0, 0x49,  // .
   0x14, 0x06, 0x12, 0x4A, 0xF0, 0x4A, 0xF0, 0x12,  // ?

   0x04, 0x03, 0x29, 0xF0, 0x29,  // Space
   0x08, 0x03, 0x5A, 0xF0, 0x5A,  // Enter [<-+]
   0x0C, 0x03, 0x5A, 0xF0, 0x5A,  // Enter
   0x46, 0x03, 0x66, 0xF0, 0x66,  // Back space
    
   0xFF 
   } ;

static uint8_t table1_sb_am[] PROGMEM = {   // Shift break, alt make.
//  SC   LGTH  --- PS2 SC -->
   0x03, 0x06, 0x14, 0x1C, 0xF0, 0x1C, 0xF0, 0x14,  // Cntl/a
   0x43, 0x06, 0x14, 0x32, 0xF0, 0x32, 0xF0, 0x14,  // Cntl/b
   0x23, 0x06, 0x14, 0x21, 0xF0, 0x21, 0xF0, 0x14,  // Cntl/c
   0x63, 0x06, 0x14, 0x23, 0xF0, 0x23, 0xF0, 0x14,  // Cntl/d           
   0x13, 0x06, 0x14, 0x24, 0xF0, 0x24, 0xF0, 0x14,  // Cntl/e          
   0x53, 0x06, 0x14, 0x2B, 0xF0, 0x2B, 0xF0, 0x14,  // Cntl/f          
   0x33, 0x06, 0x14, 0x34, 0xF0, 0x34, 0xF0, 0x14,  // Cntl/g            
   0x73, 0x06, 0x14, 0x33, 0xF0, 0x33, 0xF0, 0x14,  // Cntl/h 
   0x0B, 0x06, 0x14, 0x43, 0xF0, 0x43, 0xF0, 0x14,  // Cntl/i           
   0x4B, 0x06, 0x14, 0x3B, 0xF0, 0x3B, 0xF0, 0x14,  // Cntl/j           
   0x2B, 0x06, 0x14, 0x42, 0xF0, 0x42, 0xF0, 0x14,  // Cntl/k    
   0x6B, 0x06, 0x14, 0x4B, 0xF0, 0x4B, 0xF0, 0x14,  // Cntl/l 
   0x1B, 0x06, 0x14, 0x3A, 0xF0, 0x3A, 0xF0, 0x14,  // Cntl/m 
   0x5B, 0x06, 0x14, 0x31, 0xF0, 0x31, 0xF0, 0x14,  // Cntl/n 
   0x3B, 0x06, 0x14, 0x44, 0xF0, 0x44, 0xF0, 0x14,  // Cntl/o 
   0x7B, 0x06, 0x14, 0x4D, 0xF0, 0x4D, 0xF0, 0x14,  // Cntl/p 
   0x07, 0x06, 0x14, 0x15, 0xF0, 0x15, 0xF0, 0x14,  // Cntl/q
   0x47, 0x06, 0x14, 0x2D, 0xF0, 0x2D, 0xF0, 0x14,  // Cntl/r
   0x27, 0x06, 0x14, 0x1B, 0xF0, 0x1B, 0xF0, 0x14,  // Cntl/s 
   0x67, 0x06, 0x14, 0x2C, 0xF0, 0x2C, 0xF0, 0x14,  // Cntl/t 
   0x17, 0x06, 0x14, 0x3C, 0xF0, 0x3C, 0xF0, 0x14,  // Cntl/u 
   0x57, 0x06, 0x14, 0x2A, 0xF0, 0x2A, 0xF0, 0x14,  // Cntl/v   
   0x37, 0x06, 0x14, 0x1D, 0xF0, 0x1D, 0xF0, 0x14,  // Cntl/w 
   0x77, 0x06, 0x14, 0x22, 0xF0, 0x22, 0xF0, 0x14,  // Cntl/x   
   0x0F, 0x06, 0x14, 0x35, 0xF0, 0x35, 0xF0, 0x14,  // Cntl/y 
   0x4F, 0x06, 0x14, 0x1A, 0xF0, 0x1A, 0xF0, 0x14,  // Cntl/z 

   0x78, 0x03, 0x54, 0xF0, 0x54,                    // [

   0x38, 0x05, 0xE0, 0x7D, 0xE0, 0xF0, 0x7D,        // Page down
   0x64, 0x05, 0xE0, 0x7A, 0xE0, 0xF0, 0x7A,        // Page down

   0x42, 0x09, 0x14, 0x11, 0x05, 0xF0, 0x05,        // Cntl/Alt/F1         
               0xF0, 0x11, 0xF0, 0x14,
   0x22, 0x09, 0x14, 0x11, 0x06, 0xF0, 0x06,        // Cntl/Alt/F2         
               0xF0, 0x11, 0xF0, 0x14,
   0x62, 0x09, 0x14, 0x11, 0x04, 0xF0, 0x04,        // Cntl/Alt/F3         
               0xF0, 0x11, 0xF0, 0x14,
   0x12, 0x09, 0x14, 0x11, 0x0C, 0xF0, 0x0C,        // Cntl/Alt/F4         
               0xF0, 0x11, 0xF0, 0x14,
   0x52, 0x09, 0x14, 0x11, 0x03, 0xF0, 0x03,        // Cntl/Alt/F5         
               0xF0, 0x11, 0xF0, 0x14,
   0x32, 0x09, 0x14, 0x11, 0x0B, 0xF0, 0x0B,        // Cntl/Alt/F6         
               0xF0, 0x11, 0xF0, 0x14,
   0x72, 0x09, 0x14, 0x11, 0x83, 0xF0, 0x83,        // Cntl/Alt/F7         
               0xF0, 0x11, 0xF0, 0x14,
   0x0A, 0x09, 0x14, 0x11, 0x0A, 0xF0, 0x0A,        // Cntl/Alt/F8         
               0xF0, 0x11, 0xF0, 0x14,
   0x4A, 0x09, 0x14, 0x11, 0x01, 0xF0, 0x01,        // Cntl/Alt/F9         
               0xF0, 0x11, 0xF0, 0x14,
   0x02, 0x09, 0x14, 0x11, 0x09, 0xF0, 0x09,        // Cntl/Alt/F10         
               0xF0, 0x11, 0xF0, 0x14,
   0x06, 0x09, 0x14, 0x11, 0x78, 0xF0, 0x78,        // Cntl/Alt/F11         
               0xF0, 0x11, 0xF0, 0x14,
   0x44, 0x09, 0x14, 0x11, 0x07, 0xF0, 0x07,        // Cntl/Alt/F12         
               0xF0, 0x11, 0xF0, 0x14,                                                                                        

   0xFF 
   } ;

static uint8_t table1_sm_am[] PROGMEM = {   // Shift make, alt make.
//  SC   LGTH  --- PS2 SC -->
   0x58, 0x0B, 0x14, 0x11, 0xE0, 0x71,              // Cntl/Alt/Del
               0xE0, 0xF0, 0x71, 0xF0, 0x11, 0xF0, 0x14, 
   0x78, 0x03, 0x5B, 0xF0, 0x5B,                    // ]                                            

   0xFF 
   } ;


// ******************************************************************
// ***   Translate a kbd scan code to a PS2 scan codes sequence   ***
// ******************************************************************
void kbd_to_ps2(uint8_t kbd_sc, uint8_t shift_state, uint8_t alt_state) {

   uint8_t *tbl_ptr = 0;  // Pointer to PS2 scan codes in prog memory
   uint8_t byte, i;

   // Select the translate set.
   switch(kb_ident) {
      case 0x01:             // Translate set 1.
      default:               // TEMP 
         // Select the right table within the set.
         switch((shift_state << 1) | alt_state) {
            case NO_MAKE:
               tbl_ptr = table1_sb_ab;
               break;
            case ALT_MAKE:
               tbl_ptr = table1_sb_am;
               break;
            case SHIFT_MAKE: 
               tbl_ptr = table1_sm_ab;
               break;
            case SHIFT_ALT_MAKE:
               tbl_ptr = table1_sm_am;
         }
   }

   if (kbd_sc == 0x16) {
      reset_flag = MAKE;        // Reset pressed
      return;
   }     

   if (reset_flag == MAKE) { 
      tbl_ptr = table1_sb_am;   // It's dirty, I know.
      reset_flag = BREAK;       // but it works. 
   } 
   
   // Lookup the kbd scan code in the selected table. 
   while (pgm_read_byte(tbl_ptr) != 0xFF) {   // Till end of table
      if (pgm_read_byte(tbl_ptr) == kbd_sc) {    
         
         // Found !
         tbl_ptr++;             // Point to length scan code. 
         // Push scan codes to buffer. 
         for (i = pgm_read_byte(tbl_ptr++); i > 0; i--) {
            byte = pgm_read_byte(tbl_ptr++);
            push_buf(byte);     
         }
         return;      
      
      } else {                  // Try next entry...
         tbl_ptr = tbl_ptr + (pgm_read_byte(tbl_ptr + 1)) + 2; 
      }
   }
       
   // Kbd scan code not found. No transfer to buffer.
   return;  
}



