PROGRAM HP1650;


(*
  Program designed to make setup of HP1650 much easier to do. User is allowed
  to enter data on PC, and save/load config files to PC disk. The user can
  then load the config files to the HP1650. It is assumed that the proper mode
  command has been issued before running this code.

     MODE COM1:9600,n,8,1 (Assuming that the HP1650 is running at these)

  RAL 2/2/89

*)

{$V-}

USES
  TPW60, CRT;

CONST
   Black     = 0;     Blue      = 1;    Green     = 2;     Cyan      = 3;
   Red       = 4;     Magenta   = 5;    Brown     = 6;     L_Gray    = 7;
   Gray      = 8;     L_Blue    = 9;    L_Green   = 10;    L_Cyan    = 11;
   L_Red     = 12;    L_Magenta = 13;   Yellow    = 14;    White     = 15;


   WBColor    = Brown;   (* Window border color *)
   LabelColor = Yellow;  (* Window label color *)

   DelayTime   = 400; (* Delay between commands sent to HP1650 *)

   UpperCenter = 2;   (* TitleWindow for TPW60 *)

   ZoomControl   = 0; (* 0 - No zoom, 1 = Zoom *)


   HP_KeyPressDelay = 100;

   HP_IOKey         = 36;   (* Key codes for SYSTEM:KEY command. Page 5-22 *)
   HP_Select        = 3;
   HP_KnobRight     = 25;


TYPE
  LongLineType  = STRING[ 255 ];

  LabelNameType = STRING[ 6 ];

  FormatType = RECORD
                  LabelName : LabelNameType;
                  Polarity  : CHAR;
                  Pod       : ARRAY[ 1..5 ] OF LONGINT;
               END;

VAR
  FormatTab     : ARRAY[ 1..100 ] OF FormatType;
  FormatTabCnt  : INTEGER;       (* Number of records currently in use *)
  PodCount      : INTEGER;       (* Number of pods in use 1-5 *)
  MachineNumber : CHAR;          (* HP1650 MachineNumber = 1 or 2 *)
  MachineName   : STRING[ 10 ];  (* Machine Name...Defaults to VPIPHYSICS *)

  Com1          : TEXT;          (* Used to access COM1 port of PC *)
  Done          : BOOLEAN;
  Letter        : CHAR;


{$V-}
PROCEDURE UpCaseString( VAR Line : LongLineType );
VAR
  X : INTEGER;
BEGIN
  FOR X := 1 TO Length( Line ) DO
    Line[ X ] := UpCase( Line[ X ] );
END;  (* UpCaseString *)


FUNCTION raisetopower( Num, power : INTEGER ) : LONGINT;
BEGIN
  IF Power = 0 THEN
     raisetoPower := 1
  ELSE
     raisetopower := Num * raisetopower( Num, power - 1 );
END;  (* raisetopower *)



PROCEDURE DisplayPod( PodValue : LongInt );
VAR
  X         : INTEGER;
  Factor    : LongInt;
  SaveColor : INTEGER;

BEGIN
  SaveColor := TextAttr;

  TextBackGround( Cyan );
  TextColor( Yellow );

  Factor := 32768;
  FOR X := 0 TO 15 DO
   BEGIN
    IF PodValue >= Factor THEN
      BEGIN
        Write( '*' );
        PodValue := PodValue - Factor;
      END
    ELSE
      Write( '.' );
    Factor := Factor DIV 2;
   END;

  TextAttr := SaveColor;

  Write( ' ' );
END;  (* DisplayPod *)



PROCEDURE DisplayTableLine( Num : INTEGER );
VAR
  X : INTEGER;
BEGIN
  WITH FormatTab[ Num ] DO
   BEGIN
     TextColor( Cyan );
     Write( Num:2 );

     TextColor( Yellow );
     Write( ' ', LabelName );
     Write( ' ':7 - Length( LabelName ) );
     IF Polarity = '+' THEN
       TextColor( Red )
     ELSE
       TextColor( L_Green );
     Write( Polarity );

     FOR X := PodCount DOWNTO 1 DO
       DisplayPod( Pod[ X ] );
     WriteLn;
   END;
END;  (* DisplayTableLine *)



PROCEDURE DisplayTable;
VAR
 X : INTEGER;
BEGIN
  TextColor( Yellow );
  WriteLn( '  Machine Name = ', MachineName );
  WriteLn( '     Pod Count = ', PodCount );
  WriteLn( 'Machine Number = MACHINE', MachineNumber );
  FOR X := 1 TO FormatTabCnt DO
    DisplayTableLine( X );
END;  (* DisplayTable *)




PROCEDURE RefreshMainScreen;
BEGIN
    ClrScr;
    GotoXY( 1, 23 );
    DisplayTable;
END;  (* RefreshMainScreen *)




PROCEDURE GetPodContents( PodNumber : INTEGER; VAR Answer : LongInt );
(*
   Create a window, and interactively get the 16-bit pod contents for a
   HP1650 pod
*)
CONST
   Null          = ^@;
   UpArrowKey    = 72;   (* Values returned by ReadKey if       *)
   DownArrowKey  = 80;   (*  the first value returned is a Null *)
   LeftArrowKey  = 75;
   RightArrowKey = 77;


VAR
  Key       : CHAR;
  PodString : ARRAY[ 0..15 ] OF CHAR;
  X         : INTEGER;
  Position  : INTEGER;
  Title     : String[ 20 ];
  Temp      : LongInt;
  Factor    : LongInt;

PROCEDURE Show( Position : INTEGER );
CONST
   YPos  = 2;
   XBias = 16;
BEGIN
  GotoXY( XBias - Position, YPos );
  Write( PodString[ Position ] );
  GotoXY( XBias - Position, YPos );
END;  (* Show *)


BEGIN
  MakeWindow( 5 * ( PodNumber - 1) + 1, 5 + PodNumber,
              4, 19, Green, WBColor, 4, Gray, 6, ZoomControl );

  Str( PodNumber, Title );
  Title := 'Pod ' + Title;
  TitleWindow( UpperCenter, Attr( LabelColor, WBColor ), Title );

  TextColor( Green );
  WriteLn( '5432109876543210' );

  TextBackGround( Cyan );
  TextColor( Yellow );

  Factor := 32768;
  Temp   := Answer;
  FOR X := 15 DOWNTO 0 DO
   BEGIN
    IF Temp >= Factor THEN   (* Convert from a integer format to a string *)
     BEGIN                   (*  format, and show to user as we go        *)
      PodString[ X ] := '*';
      Temp := Temp - Factor;
     END
    ELSE
      PodString[ X ] := '.';

    Factor := Factor DIV 2;
    Show( X );
   END;

  Position := 15;
  Show( Position );

  REPEAT
     Key := ReadKey;
     IF KEY = Null THEN                   (* Check for a cursor key *)
       BEGIN
         Key := ReadKey;
         CASE ORD( KEY ) OF

           RightArrowKey : BEGIN           (* RIGHT *)
                  IF Position > 0 THEN
                   BEGIN
                    DEC( Position );  Show( Position );
                   END
                  ELSE
                   Write( ^G );
                END;

           LeftArrowKey : BEGIN            (* Left *)
                  IF Position < 15 THEN
                   BEGIN
                    INC( Position );  Show( Position );
                   END
                  ELSE
                   Write( ^G );
                END;

           UpArrowKey : BEGIN              (* Up *)
                  PodString[ Position ] := '*';
                  Show( Position );
                  IF Position > 0 THEN
                   BEGIN
                    DEC( Position );
                    Show( Position );
                   END;
                END;

           DownArrowKey : BEGIN            (* Down *)
                  PodString[ Position ] := '.';
                  Show( Position );
                  IF Position > 0 THEN
                   BEGIN
                    DEC( Position );
                    Show( Position );
                   END;
                END;

         END;
       END;

  UNTIL Key = ^M;

  Answer := 0;

  FOR X := 15 DOWNTO 0 DO
    IF PodString[ X ] = '*' THEN
      Answer := Answer + RaiseToPower( 2, X );
END;   (* GetPodContents *)



PROCEDURE MakeSpecLine( Number : INTEGER );
VAR
  Letter     : CHAR;
  X          : INTEGER;
  TempString : LabelNameType;

BEGIN
  WITH FormatTab[ Number ] DO
   BEGIN
      MakeWindow( 2, 40, 3, 20, Green, WBColor, 4, Gray, 6, ZoomControl );
      TitleWindow( UpperCenter, Attr( LabelColor, WBColor ), 'Label?' );
      Write( '[', LabelName, '] :' );

      ReadLn( TempString );
      IF TempString <> '' THEN
        LabelName := TempString;
      UpCaseString( LabelName );
      WriteLn;
      Write( LabelName );

      MakeWindow( 6, 40, 3, 10, Green, WBColor, 4, Gray, 6, ZoomControl );
      TitleWindow( UpperCenter, Attr( LabelColor, WBColor ), '+/-?' );
      Write( '[', Polarity, '] :' );
      REPEAT
        Letter := ReadKey;
      UNTIL Letter IN [ '+', '-', ^M ];

      IF Letter <> ^M THEN
        Polarity := Letter;
      WriteLn;
      Write( Polarity );


      FOR X := PodCount DOWNTO 1 DO
       GetPodContents( X, Pod[ X ] );
   END;

   FOR X := PodCount DOWNTO 1 DO
     RemoveWindow;

   RemoveWindow;      (* Polarity *)

   RemoveWindow;      (* Label Window *)
END;  (* MakeSpecLine *)



PROCEDURE AddSpecLine;
VAR
  X : INTEGER;
BEGIN
  INC( FormatTabCnt );
  WITH FormatTab[ FormatTabCnt ] DO
   BEGIN
     Str( FormatTabCnt, LabelName );
     LabelName := 'X' + LabelName;
     Polarity := '+';
     FOR X := 1 TO 5 DO           (* Clear all of the pods, even we are not *)
        Pod[ X ] := 0;            (*  currently using them.                 *)
   END;
  MakeSpecLine( FormatTabCnt );
END;  (* AddSpecLine *)




PROCEDURE DeleteSpecLine;
VAR
 DeleteNumber : INTEGER;
 X            : INTEGER;
BEGIN
  MakeWindow( 2, 10, 3, 50, Green, WBColor, 4, Gray, 6, ZoomControl );
  TitleWindow( UpperCenter, Attr( LabelColor, WBColor ), 'Delete' );
  REPEAT
    Write( 'Enter Line Number to Delete ? ' );
    ReadLn( DeleteNumber );
  UNTIL ( DeleteNumber > 0 ) AND ( DeleteNumber <= FormatTabCnt );

  FOR X := ( DeleteNumber + 1 ) TO FormatTabCnt DO
    BEGIN
      FormatTab[ X - 1 ].LabelName := FormatTab[ X ].LabelName;
      FormatTab[ X - 1 ].Polarity  := FormatTab[ X ].Polarity;
      FormatTab[ X - 1 ].Pod       := FormatTab[ X ].Pod
    END;

  DEC( FormatTabCnt );

  RemoveWindow;
END;  (* DeleteSpecLine *)




PROCEDURE InsertSpecLine;
VAR
 Number : INTEGER;
 X      : INTEGER;
BEGIN
  MakeWindow( 2, 10, 3, 50, Green, WBColor, 4, Gray, 6, ZoomControl );
  TitleWindow( UpperCenter, Attr( LabelColor, WBColor ), 'Insert' );
  REPEAT
    Write( 'Enter Line Number to Insert BEFORE ? ' );
    ReadLn( Number );
  UNTIL ( Number > 0 ) AND ( Number <= ( FormatTabCnt + 1 ));

  RemoveWindow;

  FOR X := FormatTabCnt DOWNTO Number DO
    BEGIN
      FormatTab[ X + 1 ].LabelName := FormatTab[ X ].LabelName;
      FormatTab[ X + 1 ].Polarity  := FormatTab[ X ].Polarity;
      FormatTab[ X + 1 ].Pod       := FormatTab[ X ].Pod
    END;

  INC( FormatTabCnt );

  MakeSpecLine( Number );   (* Now add the new line *)

END;  (* InsertSpecLine *)




PROCEDURE SwapSpecLine;
VAR
  Line1 : INTEGER;
  Line2 : INTEGER;
BEGIN
  MakeWindow( 2, 10, 3, 50, Green, WBColor, 4, Gray, 6, ZoomControl );
  TitleWindow( UpperCenter, Attr( LabelColor, WBColor ), 'Swap One' );
  REPEAT
    Write( 'Enter first line number ? ' );
    ReadLn( Line1 );
  UNTIL ( Line1 > 0 ) AND ( Line1 <= ( FormatTabCnt ));
  Write( 'Swap line one = ', Line1 );

  MakeWindow( 7, 10, 3, 50, Green, WBColor, 4, Gray, 6, ZoomControl );
  TitleWindow( UpperCenter, Attr( LabelColor, WBColor ), 'Swap Two' );
  REPEAT
    Write( 'Enter second line number ? ' );
    ReadLn( Line2 );
  UNTIL ( Line2 > 0 ) AND ( Line2 <= ( FormatTabCnt ));
  Write( 'Swap line two = ', Line2 );

  FormatTab[ FormatTabCnt + 1 ].LabelName := FormatTab[ Line1 ].LabelName;
  FormatTab[ FormatTabCnt + 1 ].Polarity  := FormatTab[ Line1 ].Polarity;
  FormatTab[ FormatTabCnt + 1 ].Pod       := FormatTab[ Line1 ].Pod;

  FormatTab[ Line1 ].LabelName := FormatTab[ Line2 ].LabelName;
  FormatTab[ Line1 ].Polarity  := FormatTab[ Line2 ].Polarity;
  FormatTab[ Line1 ].Pod       := FormatTab[ Line2 ].Pod;

  FormatTab[ Line2 ].LabelName := FormatTab[ FormatTabCnt + 1 ].LabelName;
  FormatTab[ Line2 ].Polarity  := FormatTab[ FormatTabCnt + 1 ].Polarity;
  FormatTab[ Line2 ].Pod       := FormatTab[ FormatTabCnt + 1 ].Pod;

  RemoveWindow;
  RemoveWindow;
END;  (* SwapSpecLine *)




PROCEDURE LoadHP1650;
VAR
  X,Y  : INTEGER;
BEGIN
  Write( Com1, 'SYSTEM:MENU FORMAT,', MachineNumber, ^J );
  DELAY( DelayTime );

                  (* Clear machine *)
  Write( Com1, 'MACHINE', MachineNumber, ':SFORMAT:REMOVE ALL', ^J );
  DELAY( DelayTime );

  Write( Com1, 'MACHINE', MachineNumber, ':NAME "', MachineName, '"', ^J );
  Delay( DelayTime );

  Write( Com1, 'MACHINE', MachineNumber, ':ASSIGN ' );
  FOR X := PodCount DOWNTO 1 DO
   BEGIN
     Write( Com1, X );
     IF X <> 1 THEN
       Write( Com1, ',' );
   END;
  Write( Com1, ^J );

  DELAY( DelayTime );

  FOR X := 1 TO FormatTabCnt DO
    WITH FormatTab[ X ] DO
      BEGIN
        Write( COM1, 'MACHINE', MachineNumber, ':SFORMAT:LABEL ' );
        Write( COM1, '"', LabelName, '",' );
        CASE Polarity OF
          '+' : Write( Com1, 'POS,' );
          '-' : Write( Com1, 'NEG,' );
        END;

        FOR Y := PodCount DOWNTO 1 DO
         BEGIN
          Write( Com1, Pod[ Y ] );
          IF Y <> 1 THEN
            Write( Com1, ',' );
         END;

        Write( Com1, ^J );

        DELAY( DelayTime );
      END;
END;  (* LoadHP1650 *)



PROCEDURE DoPCStuff;
VAR
 X, Y     : INTEGER;
 FileName : String[ 50 ];
PROCEDURE DoSavePCFile;
VAR
 O_F    : TEXT;
 X, Y    : INTEGER;
BEGIN
  MakeWindow( 6, 15, 3, 50, Green, WBColor, 4, Gray, 6, ZoomControl );
  TitleWindow( UpperCenter, Attr( LabelColor, WBColor ), 'Save - FileName?' );
  ReadLn( FileName );

  FileName := FileName + '.LGA';

  Assign( O_F, FileName );
  Rewrite( O_F );

  WriteLn( O_F, PodCount );
  WriteLn( O_F, FormatTabCnt );
  WriteLn( O_F, MachineNumber );
  WriteLn( O_F, MachineName );

  FOR X := 1 TO FormatTabCnt DO
    WITH FormatTab[ X ] DO
     BEGIN
       WriteLn( O_F, LabelName );
       WriteLn( O_F, Polarity );
       FOR Y := PodCount DOWNTO 1 DO
         WriteLn( O_F, Pod[ Y ] );
     END;

  Close( O_F );
  RemoveWindow;
END;  (* DoSavePCFile *)

PROCEDURE DoLoadPCFile;
VAR
 I_F     : TEXT;
 Letter  : CHAR;
 X, Y    : INTEGER;

BEGIN
  MakeWindow( 6, 15, 3, 50, Green, WBColor, 4, Gray, 6, ZoomControl );
  TitleWindow( UpperCenter, Attr( LabelColor, WBColor ), 'Load - FileName?' );
  ReadLn( FileName );

  FileName := FileName + '.LGA';

  Assign( I_F, FileName );
  Reset( I_F );

  ReadLn( I_F, PodCount );
  ReadLn( I_F, FormatTabCnt );
  ReadLn( I_F, MachineNumber );
  ReadLn( I_F, MachineName );

  FOR X := 1 TO FormatTabCnt DO
    WITH FormatTab[ X ] DO
     BEGIN
       ReadLn( I_F, LabelName );
       ReadLn( I_F, Polarity );

       FOR Y := PodCount DOWNTO 1 DO
         ReadLn( I_F, Pod[ Y ] );
     END;

  Close( I_F );

  RemoveWindow;
END;  (* DoLoadPCFile *)


BEGIN
  REPEAT
    MakeWindow( 2, 5, 6, 50, Green, WBColor, 4, Gray, 6, ZoomControl );
    TitleWindow( UpperCenter, Attr( LabelColor, WBColor ), 'PC Stuff' );

    REPEAT
      WriteLn( 'L)oad File, S)ave File' );
      WriteLn( 'Q)uit' );
      Write( 'Enter command letter -> ' );
      ReadLn( Letter );
      Letter := UpCase( Letter );
    UNTIL Letter IN [ 'L', 'S', 'Q' ];

    CASE Letter OF
      'L' : DoLoadPCFile;
      'S' : DoSavePCFile;
    END;

    RemoveWindow;

    IF Letter <> 'Q' THEN
       RefreshMainScreen;

  UNTIL Letter ='Q';

END;  (* DoPCStuff *)



PROCEDURE DoHPStuff;
VAR
  FileName : STRING[ 10 ];
PROCEDURE DoLoadHPFile;
BEGIN
  MakeWindow( 6, 15, 3, 20, Green, WBColor, 4, Gray, 6, ZoomControl );
  TitleWindow( UpperCenter, Attr( LabelColor, WBColor ), 'Load - FileName?' );
  ReadLn( FileName );
  UpCaseString( FileName );
  Write( FileName );
  Write( Com1, ':MMEMORY:LOAD "', FileName, '"', ^J );
  RemoveWindow;
END;  (* DoLoadHPFile *)

PROCEDURE DoPurgeHPFile;
VAR
 Letter : CHAR;
BEGIN
  MakeWindow( 6, 15, 3, 21, Green, WBColor, 4, Gray, 6, ZoomControl );
  TitleWindow( UpperCenter, Attr( LabelColor, WBColor ), 'Purge - FileName?' );
  ReadLn( FileName );
  UpCaseString( FileName );
  Write( FileName );

  MakeWindow( 12, 15, 3, 12, Green, WBColor, 4, Gray, 6, ZoomControl );
  TitleWindow( UpperCenter, Attr( LabelColor, WBColor ), 'Sure Y/x?' );
  ReadLn( Letter );

  IF Letter = 'Y' THEN
    Write( Com1, ':MMEMORY:PURGE "', FileName, '"', ^J );

  RemoveWindow;
  RemoveWindow;
END;  (* DoPurgeHPFile *)

PROCEDURE DoSaveHPFile;
VAR
  Description : STRING[ 32 ];
BEGIN
  MakeWindow( 6, 15, 3, 20, Green, WBColor, 4, Gray, 6, ZoomControl );
  TitleWindow( UpperCenter, Attr( LabelColor, WBColor ), 'Store - FileName?' );
  ReadLn( FileName );
  UpCaseString( FileName );
  Write( FileName );

  MakeWindow( 12, 15, 3, 50, Green, WBColor, 4, Gray, 6, ZoomControl );
  TitleWindow( UpperCenter, Attr( LabelColor, WBColor ), 'Description?' );
  ReadLn( Description );
  Write( Description );

  Write( Com1, ':MMEMORY:STORE "', FileName, '","', Description, '"', ^J );
  RemoveWindow;
END;  (* DoSaveHPFile *)

BEGIN
    (* Make sure we are at the system menu when we start *)
  Write( Com1, 'SYSTEM:MENU SCONFIG,', MachineNumber, ^J );
  DELAY( DelayTime );

    (* By simulating key presses, get us into I/O disk menu *)
  Write( Com1, 'SYSTEM:KEY ', HP_IOKey, ^J );
  Delay( HP_KeyPressDelay );
  Write( Com1, 'SYSTEM:KEY ', HP_KnobRight, ^J );
  Delay( HP_KeyPressDelay );
  Write( Com1, 'SYSTEM:KEY ', HP_KnobRight, ^J );
  Delay( HP_KeyPressDelay );
  Write( Com1, 'SYSTEM:KEY ', HP_KnobRight, ^J );
  Delay( HP_KeyPressDelay );
  Write( Com1, 'SYSTEM:KEY ', HP_Select, ^J );
  Delay( HP_KeyPressDelay );

  REPEAT
    MakeWindow( 2, 5, 6, 50, Green, WBColor, 4, Gray, 6, ZoomControl );
    TitleWindow( UpperCenter, Attr( LabelColor, WBColor ), 'HP Stuff' );

    REPEAT
      WriteLn( 'L)oad File,  S)tore File' );
      WriteLn( 'P)urge File, Z) Pack Disk' );
      WriteLn( 'Q)uit' );
      Write( 'Enter command letter -> ' );
      ReadLn( Letter );
      Letter := UpCase( Letter );
    UNTIL Letter IN [ 'L', 'S', 'Q', 'P', 'Z' ];

    CASE Letter OF
      'L' : DoLoadHPFile;
      'S' : DoSaveHPFile;
      'Z' : Write( Com1, ':MMEMORY:PACK', ^J );
      'P' : DoPurgeHPFile;
    END;

    RemoveWindow;

    IF Letter <> 'Q' THEN
       RefreshMainScreen;

  UNTIL Letter ='Q';

    (* Put us back at system menu when we are done *)
  Write( Com1, 'SYSTEM:MENU SCONFIG,', MachineNumber, ^J );
  DELAY( DelayTime );
END;  (* DoHPStuff *)



PROCEDURE EditLine;
VAR
  X : INTEGER;
BEGIN
  MakeWindow( 6, 15, 3, 21, Green, WBColor, 4, Gray, 6, ZoomControl );
  TitleWindow( UpperCenter, Attr( LabelColor, WBColor ), 'Edit Line Number?' );
  REPEAT
    ReadLn( X );
  UNTIL ( X > 0 ) AND ( X <= FormatTabCnt );
  MakeSpecLine( X );

  RemoveWindow;
END;  (* EditLine *)



PROCEDURE DoConfigure;
PROCEDURE GetPodCount;
BEGIN
  MakeWindow( 6, 15, 3, 20, Green, WBColor, 4, Gray, 6, ZoomControl );
  TitleWindow( UpperCenter, Attr( LabelColor, WBColor ), 'Pod Count 1-5?' );
  REPEAT
    ReadLn( PodCount );
  UNTIL ( PodCount <= 5 ) AND ( PodCount > 0 );
  RemoveWindow;
END;  (* GetPodCount *)

PROCEDURE GetMachineNumber;
VAR
  X : INTEGER;
BEGIN
  MakeWindow( 6, 15, 3, 20, Green, WBColor, 4, Gray, 6, ZoomControl );
  TitleWindow( UpperCenter, Attr( LabelColor, WBColor ), 'Machine # 1/2?' );
  REPEAT
    ReadLn( MachineNumber );
  UNTIL MachineNumber in [ '1', '2' ];

  RemoveWindow;
END;  (* GetMachineNumber *)

PROCEDURE GetMachineName;
VAR
  X : INTEGER;
BEGIN
  MakeWindow( 6, 15, 3, 20, Green, WBColor, 4, Gray, 6, ZoomControl );
  TitleWindow( UpperCenter, Attr( LabelColor, WBColor ), 'Machine Name?' );
  ReadLn( MachineName );
  UpCaseString( MachineName );
  RemoveWindow;
END;  (* GetMachineName *)

BEGIN
  REPEAT
    MakeWindow( 2, 5, 6, 50, Green, WBColor, 4, Gray, 6, ZoomControl );
    TitleWindow( UpperCenter, Attr( LabelColor, WBColor ), 'Configure' );
    REPEAT
      WriteLn( 'P)od Count, M)achine Number' );
      WriteLn( 'N)ame of Machine' );
      WriteLn( 'Q)uit' );
      Write( 'Enter command letter -> ' );
      ReadLn( Letter );
      Letter := UpCase( Letter );
    UNTIL Letter IN [ 'P', 'M', 'N', 'Q' ];

    CASE Letter OF
      'P' : GetPodCount;
      'M' : GetMachineNumber;
      'N' : GetMachineName;
    END;

    RemoveWindow;

    IF Letter <> 'Q' THEN
       RefreshMainScreen;
  UNTIL Letter = 'Q';
END;  (* DoConfigure *)




PROCEDURE ZapLines;
BEGIN
  MakeWindow( 6, 15, 3, 20, Green, WBColor, 4, Gray, 6, ZoomControl );
  TitleWindow( UpperCenter, Attr( LabelColor, WBColor ), 'Clear? Y/x' );
  ReadLn( Letter );

  IF Letter = 'Y' THEN
   FormatTabCnt := 0;   (* Throw away all of the data!!!! *)

  RemoveWindow;
END;



PROCEDURE Menu( VAR Letter : CHAR );
VAR
  Ready : BOOLEAN;

BEGIN
  MakeWindow( 2, 10, 7, 40, Green, WBColor, 4, Gray, 6, ZoomControl );
  TitleWindow( UpperCenter, Attr( LabelColor, WBColor ), 'Main Menu' );

  Ready := FALSE;

  REPEAT
    WriteLn( 'A)ppend, I)nsert, S)wap, D)elete' );
    WriteLn( 'E)dit,   C)onfig, Z)ap' );
    WriteLn( 'L)oad HP1650,     P)C stuff' );
    WriteLn( 'Q)uit             H)P stuff' );
    Write( 'Enter Command Letter --> ' );
    ReadLn( Letter );
    Letter := UpCase( Letter );

    IF Letter IN [ 'Z', 'Q', 'A', 'D', 'I', 'S', 'L', 'P', 'C', 'E', 'H' ] THEN
      Ready := TRUE
    ELSE
      WriteLn( '** Bad Command **' );

  UNTIL Ready;

  RemoveWindow;
END;  (* Menu *)




BEGIN
  Assign( Com1, 'AUX' );          (* Set up the port for us *)
  Rewrite( Com1 );

  PodCount      := 2;              (* Default to two pods *)
  MachineNumber := '1';            (* Default to MACHINE1 *)
  MachineName   := 'VPIPHYSICS';   (* Default MachineName *)
  FormatTabCnt  := 0;              (* Number of lines in FormatTab *)

  RefreshMainScreen;
  MakeWindow( 10, 20, 7, 35, Green, WBColor, 4, Gray, 6, ZoomControl );
  TitleWindow( UpperCenter, Attr( LabelColor, WBColor ), 'Version 1.0' );

  TextColor( L_Green );
  WriteLn( '  VPI Physics Electronics Shop  ' );
  WriteLn( 'HP1650 Logic Analyzer Controller' );
  WriteLn( '          Version 1.1           ' );
  WriteLn( '           RAL 2/3/89           ' );

  TextColor( Gray );
  Write( 'Press any Key to continue' );
  REPEAT
  UNTIL KeyPressed;
  Letter := ReadKey;
  RemoveWindow;


  Done := FALSE;

  REPEAT
    RefreshMainScreen;

    Menu( Letter );

    CASE Letter OF
      'Q'  : Done := TRUE;
      'A'  : AddSpecLine;
      'D'  : IF FormatTabCnt > 0 THEN
              DeleteSpecLine
             ELSE
              Write( ^G );

      'I'  : InsertSpecLine;
      'S'  : SwapSpecLine;
      'L'  : LoadHp1650;
      'P'  : DoPCStuff;
      'C'  : DoConfigure;
      'Z'  : ZapLines;
      'E'  : EditLine;
      'H'  : DoHPStuff;
    END;

  UNTIL Done = TRUE;

  Close( Com1 );

  GotoXY( 1, 23 );
END.
