/*
 * Copyright (c) 1996-1997 Distributed Processing Technology Corporation
 * All rights reserved.
 *
 * Redistribution and use in source form, with or without modification, are
 * permitted provided that redistributions of source code must retain the
 * above copyright notice, this list of conditions and the following disclaimer.
 *
 * This software is provided `as is' by Distributed Processing Technology and
 * any express or implied warranties, including, but not limited to, the
 * implied warranties of merchantability and fitness for a particular purpose,
 * are disclaimed. In no event shall Distributed Processing Technology be
 * liable for any direct, indirect, incidental, special, exemplary or
 * consequential damages (including, but not limited to, procurement of
 * substitute goods or services; loss of use, data, or profits; or business
 * interruptions) however caused and on any theory of liability, whether in
 * contract, strict liability, or tort (including negligence or otherwise)
 * arising in any way out of the use of this driver software, even if advised
 * of the possibility of such damage.
 *
 * DPT SCSI host adapter driver
 *
 * TODO:
 *      Add support for Software RAID-0 over multiple Adapters.
 *
 *      dpt V3.05 1997/03/19 salyzyn@dpt.com
 *              Multichannel support appears to have broken, fixed it.
 *      dpt V3.04 1996/12/09 pjd@bsdi.com & salyzyn@dpt.com
 *              Move eisa primary check to under BUS_EISA bus port type case.
 *      dpt V3.03 1996/11/08 salyzyn@dpt.com
 *              Timeing out on the card is bad, removed timeouts.
 *      dpt V3.02 1996/11/4 salyzyn@dpt.com
 *              Added IRQSHARE to the interrupt specification on boards that
 *              have level triggered interrupts.
 *      dpt V3.01 1996/09/1 salyzyn@dpt.com
 *              Fixed problems with the pass through interface associated
 *              with system information (dptioctl.h)
 *      dpt V3.00 1996/08/14 salyzyn@dpt.com
 *              Prepared the driver for the 3.0 release and the new SCSI
 *              subsystem.
 *
 */

#define DPT_VERSION     3
#define DPT_REVISION    '0'
#define DPT_SUBREVISION '5'
#define DPT_MONTH       3
#define DPT_DAY         19
#define DPT_YEAR        17      /* 1997 */

/*
 *      DPT_DEBUG_FLAGS
 *              - any non-zero value enables the ha_debug flag and is
 *                equivalent to a DPT_DEBUG_BASE being set.
 */
# define DPT_DEBUG_BASE    0x0001       /* Minor Overhead for validity checks */
# define DPT_DEBUG_CHECK   0x0002       /* Additional more detailed checks */
# define DPT_DEBUG_CALL    0x0004       /* Print subroutine entry */
# define DPT_DEBUG_VERBOSE 0x0004       /* Detailed packet entry */
# define DPT_DEBUG_SET     0x8000       /* Set ha_debug default */
#define DPT_DEBUG_FLAGS 0

#include        <i386/isa/dptsig.h>

static dpt_sig_S dpt_sig = {
        'd', 'P', 't', 'S', 'i', 'G', SIG_VERSION, PROC_INTEL,
        PROC_386 | PROC_486 | PROC_PENTIUM | PROC_SEXIUM, FT_HBADRVR, 0,
        OEM_DPT, OS_BSDI_UNIX, CAP_ABOVE16MB, DEV_ALL, ADF_ALL_MASTER,
        0, 0, DPT_VERSION, DPT_REVISION, DPT_SUBREVISION,
        DPT_MONTH, DPT_DAY, DPT_YEAR,
        "DPT BSDi BSD/OS 3.0 Unix SCSI HBA Driver"
        /*               ^^^   dptattach alters these to match OS */
};
extern char     osrelease[];

#include <sys/param.h>
#include <sys/kernel.h>
#include <sys/systm.h>
#include <sys/buf.h>
#include <sys/device.h>
#include <sys/malloc.h>
#include <sys/proc.h>
#include <sys/conf.h>

#include <vm/vm.h>

#include <dev/scsi/scsi.h>
#include <dev/scsi/scsivar.h>
#include <dev/scsi/scsi_spivar.h>
#include <dev/scsi/disk.h>
#include <dev/scsi/tape.h>

#include <machine/cpu.h>

#include <i386/isa/isa.h>
#include <i386/isa/isavar.h>
#include <i386/eisa/eisa.h>
#include <i386/pci/pci.h>
#include <i386/isa/dma.h>
#include <i386/isa/icu.h>
#include <i386/i386/sg_map.h>

extern int maxmem;      /* Gives us the number of pages of available memory */

#define STATIC static
#if (defined(DEBUG_FLAGS) && (DEBUG_FLAGS > 0))
# undef STATIC
# define STATIC
#endif

/* Per-request Flag values      */
#define FL_NOSLEEP      0x01    /* Not a user... don't sleep            */
#define FL_NOMASK       0x02    /* dont allow interrupts.. booting      */
#define FL_QUIET        0x04    /* Don't print error messages           */


/* Configuration Definitions    */

#define NCCBS           64      /* Max CCBs, Also Max Queue Size        */
#define SG_SIZE         64      /* Scatter Gather list Size             */
/* Maximum TARGET Supported (Maximum target determined by HBA)  */
#define MAX_TARGET_ID   16      /* Maximum Target ID supported by OS    */
#define MAX_LUN         8       /* Maximum LUN Supported        */
/* Maximum Channel # Supported by driver */
#define MAX_CHANNEL     (int)(sizeof(((DptConfig_t *)NULL)->HBA) \
                               / sizeof(((DptConfig_t *)NULL)->HBA[0]))

/************************** board definitions *******************************/

/*
 * I/O Port Interface
 */

typedef struct {
        char    dpt_data;
        char    dpt_error;
        char    dpt_addr[4];
#define immed_mod dpt_addr[3]
        char    immed_func;
        union   {
                char    dpt_command;
                char    dpt_status;
        }       dpt_reg;
        char    dpt_aux_status;
} DptPort_t;

#define HA_DATA         ((int)(&(((DptPort_t *)0L)->dpt_data)))
#define HA_ERROR        ((int)(&(((DptPort_t *)0L)->dpt_error)))
#define HA_DMA_BASE     ((int)(((DptPort_t *)0L)->dpt_addr))
#define HA_COMMAND      ((int)(&(((DptPort_t *)0L)->dpt_reg.dpt_command)))
/* EATA PIO commands */
#define CP_READ_CFG_PIO 0xF0    /* Read Configuration Data, PIO   */
#define CP_PIO_CMD              0xF2    /* Execute Command, PIO           */
#define CP_TRUNCATE_CMD 0xF4    /* Truncate Transfer Command, PIO */
#define CP_EATA_RESET           0xF9    /* Reset Controller And SCSI Bus  */
/* EATA DMA commands */
#define CP_READ_CFG_DMA 0xFD    /* Read Configuration Data, DMA   */
#define CP_DMA_CMD              0xFF    /* Execute Command, DMA           */
/* Other commands */
#define CP_IMMEDIATE            0xFA    /* EATA Immediate command         */

#define HA_STATUS       ((int)(&(((DptPort_t *)0L)->dpt_reg.dpt_status)))
#define HA_ST_ERROR             0x01
#define HA_ST_INDEX             0x02
#define HA_ST_CORRCTD           0x04
#define HA_ST_DRQ               0x08
#define HA_ST_SEEK_COMPLETE     0x10
#define HA_ST_WRT_FLT           0x20
#define HA_ST_READY             0x40
#define HA_ST_BUSY              0x80
#define HA_ST_DATA_RDY (HA_ST_SEEK_COMPLETE | HA_ST_READY | HA_ST_DRQ)

#define HA_AUX_STATUS   ((int)(&(((DptPort_t *)0L)->dpt_aux_status)))
#define HA_AUX_BUSY             0x01    /* Aux Register Busy bit.         */
#define HA_AUX_INTR             0x02    /* Aux Register Interrupt Pending */

#define DPT_NPORT       sizeof(DptPort_t)

/*************************************************************************
**                      EATA Command Packet Definitions                 **
**************************************************************************/

#define EATA_SIG (((long)'E')|(((long)'A')<<8L)|(((long)'T')<<16L)|(((long)'A')<<24L))

/*******************************************************************************
** EATA Command Packet Control Bits Structure Definition - This Controlls The **
**      Data Direction, CDB Interpret, And Other Misc Controll Bits.          **
*******************************************************************************/

typedef vm_offset_t     vm_addr_t;      /* Byte reversed addresses      */
typedef vm_size_t       vm_len_t;       /* Byte reversed counts         */

typedef struct {
        union { struct {
                u_char  reserved[5];
                u_char  x_FWNest:1;     /* Firmware Nested Bit For RAID */
                u_char  reserved1:7;
                u_char  x_PhsUnit:1;    /* Physical Bit For RAID        */
                u_char  x_IAT:1;        /* Inhibit Address Translation  */
                u_char  x_Disable_Cache:1; /* Inhibit caching           */
                }       x;
                struct {
                u_char  y_SCSI_Reset:1; /* Issue A SCSI Bus Reset       */
                u_char  y_HBA_Init:1;   /* ReInitialize The Controller  */
                u_char  y_Auto_Req_Sen:1;/* Do Auto Request Sense       */
                u_char  y_Scatter_Gather:1;/* Data Pointer To S.G. List*/
                u_char  y_Quick:1;      /* Perform Quick PIOin, DMA in/out*/
                u_char  y_Interpret:1;  /* CDB Interpreted By Controller*/
                u_char  y_DataOut:1;    /* Data Out Phase With Command  */
                u_char  y_DataIn:1;     /* Data In Phase With Command   */
                u_char  y_Req_Len;      /* AutoRequestSense Data length.*/
                u_char  reserved2[3];
                u_char  y_FWNest;       /* Firmware Nested Byte for RAID*/
                u_char  y_PhsUnit;      /* Physical Byte for RAID       */
                u_char  y_id:5;         /* SCSI Target ID               */
                u_char  y_Channel:3;    /* HBA Channel Number           */
                }       y;
        }       cp_u;
#define cp_SCSI_Reset           cp_u.y.y_SCSI_Reset
#define cp_HBA_Init             cp_u.y.y_HBA_Init
#define cp_Auto_Req_Sen         cp_u.y.y_Auto_Req_Sen
#define cp_Scatter_Gather       cp_u.y.y_Scatter_Gather
#define cp_Quick                cp_u.y.y_Quick
#define cp_Interpret            cp_u.y.y_Interpret
#define cp_DataOut              cp_u.y.y_DataOut
#define cp_DataIn               cp_u.y.y_DataIn
#define cp_Req_Len              cp_u.y.y_Req_Len
#define cp_FWNest               cp_u.y.y_FWNest
#define cp_PhsUnit              cp_u.y.y_PhsUnit
#define cp_IAT                  cp_u.x.x_IAT
#define cp_Disable_Cache        cp_u.x.x_Disable_Cache
#define cp_id                   cp_u.y.y_id
#define cp_Channel              cp_u.y.y_Channel
        u_char  cp_msg_lun:3;           /* Message Lun          */
        u_char  reserved3:2;
        u_char  cp_msg_luntar:1;        /* Target Routine       */
        u_char  cp_msg_disco_rico:1;    /* Disconnect Reconnect */
        u_char  cp_msg_identify:1;      /* Indentify            */
        u_char  cp_msg[3];      /* Identify and Disconnect Message.     */
        union { struct {
                u_char  x_scsi_cmd;     /* Partial SCSI CDB def */
                u_char  x_extent:1;
                u_char  x_bytchk:1;
                u_char  x_reladr:1;
                u_char  x_cmplst:1;
                u_char  x_fmtdata:1;
                u_char  x_lun:3;
                u_char  x_page;
                u_char  reserved4;
                u_char  x_len;
                u_char  x_link:1;
                u_char  x_flag:1;
                u_char  reserved5:4;
                u_char  x_vendor:2;
                }       x;
                u_char  z[12];          /* Total SCSI CDB               */
        }       cp_w;
#define cp_cdb  cp_w.z
#define cp_scsi_cmd     cp_w.x.x_scsi_cmd
#define cp_extent       cp_w.x.x_extent
#define cp_lun          cp_w.x.x_lun
#define cp_page         cp_w.x.x_page
#define cp_len          cp_w.x.x_len
        vm_len_t  cp_datalen;   /* Byte Swapped Data Length In Bytes    */
        caddr_t   cp_vp;        /* Command Packet Virtual Pointer.      */
        vm_addr_t cp_dataDMA;   /* Byte Swapped Data Physical Address   */
        vm_addr_t cp_statDMA;   /* Byte Swapped Status Packet Physical  */
        vm_addr_t cp_reqDMA;    /* Byte Swapped AutoRequestSense Phys   */
} EataCp_t;

/*************************************************************************
** Scatter Gather List Element Structure.                               **
**************************************************************************/

typedef struct {        /* Remember these are byte reversed */
        vm_addr_t       data_addr;      /* Physical Data Address        */
        vm_len_t        data_len;       /* Count In Bytes               */
} SgElement_t;

typedef struct DptBounce {      /* DMA Bounce support */
        struct DptBounce *      bo_next;/* Next bounce page in list     */
        caddr_t         bo_dst; /* virtual address of destination       */
        vm_offset_t     bo_p;   /* physical address of bounce page      */
        caddr_t         bo_v;   /* Virtual address of bounce page       */
        vm_size_t       bo_len; /* amount of data in the page           */
} DptBounce_t;

/******************************************************************************
** Controller Command Block - CCB - This Structure Contains The EATA Command **
**   Packet, OsRequest_t Pointer And Other Command Control Information. The  **
**   EATA Command Packet Should Remain At The Begining Of The Structure For  **
**   Ease Of Processing The Command To The Controller.                       **
*******************************************************************************/

typedef struct DptCcb {
/************** First part for convenience should be an EataCp **********/
        EataCp_t        EataCp;

/***************************** Response flags ***************************/

        u_char          ccb_isa;        /* CCB ok for use in ISA system */
        vm_addr_t       ccb_addr;       /* CCB Physical Address.        */
        struct DptCcb * ccb_next;       /* Pointer To Next active CCB   */
        DptBounce_t *   ccb_Bounce;
        void *          ccb_Device;     /* Device CCB is attached to    */

/********************** Additional flags for BSDi ***********************/

        scp_t *         ccb_scp;        /* pointer back to initiating scb */

/********************** Scatter Gather Portion **************************/

        vm_addr_t       ccb_SG_addr;    /* Byte Swapped Physical S.G. List */
        /* Must be Last as it could be dynamically allocated ? */
        SgElement_t     ccb_SG[SG_SIZE];/* Scatter Gather List          */
} DptCcb_t;

/******************************************************************************
** Status Packet Data Structure - This Structure Is Returned By Controller   **
**   Upon Command Completion.   It Contains Status, Message Info And Pointer **
**   To The Initiating Command CCB (virtual).                                **
*******************************************************************************/

typedef struct {
        u_char          sp_HBA_stat:7;  /* Host Adapter Status          */
        u_char          sp_EOC:1;       /* End of Command (Not Safe)    */
        u_char          sp_SCSI_stat;   /* SCSI Bus Status              */
        u_char          reserved[2];    /* Reserved                     */
        vm_len_t        sp_inv_residue; /* Bytes Not Transfered         */
        DptCcb_t *      sp_vp;          /* Command Packet Virtual Pointer. */
        u_char          sp_ID_Message;
        u_char          sp_Que_Message;
        u_char          sp_Tag_Message;
        u_char          sp_Messages[9];
} DptStat_t;

/*
 *      sp_EOC is not `safe', so I will check sp_Messages[0] instead!
 */
#define DptStat_BUSY(x) ((x)->sp_Messages[0])
#define DptStat_Reset_BUSY(x)   ((x)->sp_Messages[0] = 0xA5, (x)->sp_EOC = 0, (x)->sp_vp = (DptCcb_t *)NULL)

/*************************************************************************
**              HBA Status Packet Host Status Definitions               **
**************************************************************************/

#define HA_NO_ERROR             0x00    /* No Error on command          */
#define HA_ERROR_SEL_TO         0x01    /* Device Selection Time Out    */
#define HA_ERROR_CMD_TO         0x02    /* Device Command Time Out      */
#define HA_ERROR_RESET          0x03    /* SCSI Bus was RESET !         */
#define HA_INIT_POWERUP         0x04    /* Initial Controller Power Up  */
#define HA_UNX_BUSPHASE         0x05    /* Unexpected BUS Phase         */
#define HA_UNX_BUS_FREE         0x06    /* Unexpected BUS Free          */
#define HA_BUS_PARITY           0x07    /* SCSI Bus Parity Error        */
#define HA_SCSI_HUNG            0x08    /* SCSI Bus Hung                */
#define HA_UNX_MSGRJCT          0x09    /* Unexpected Message Reject    */
#define HA_RESET_STUCK          0x0A    /* SCSI Bus Reset Stuck         */
#define HA_RSENSE_FAIL          0x0B    /* Auto-Request Sense Failed    */
#define HA_PARITY               0x0C    /* HBA Memory Parity error      */
#define HA_ABORT_NA             0x0D    /* CP aborted - NOT on Bus      */
#define HA_ABORTED              0x0E    /* CP aborted - WAS on Bus      */
#define HA_RESET_NA             0x0F    /* CP was reset - NOT on Bus    */
#define HA_RESET                0x10    /* CP was reset - WAS on Bus    */
#define HA_ECC                  0x11    /* HBA Memory ECC Error         */
#define HA_PCI_PARITY           0x12    /* PCI Parity Error             */
#define HA_PCI_MASTER           0x13    /* PCI Master Abort             */
#define HA_PCI_TARGET           0x14    /* PCI Target Abort             */
#define HA_PCI_SIGNAL_TARGET    0x15    /* PCI Signalled Target Abort   */
#define HA_ABORT                0x20    /* Software Abort (too many retries) */

/*******************************************************************************
** ReadConfig Data Structure - This Structure Contains The EATA Configuration **
*******************************************************************************/

typedef struct {
        u_char  DevType;
        u_char  PageCode;
        u_char  Reserved0;
        u_char  ConfigLength;           /* Length In Bytes After This Field   */
        u_char  EATAsignature[4];       /* EATA Signature Bytes               */
        u_char  EATAversion;            /* EATA Version Number                */
        struct {
                u_char x_OverLapCmds:1; /* Overlapped Cmds Supported          */
                u_char x_TargetMode:1;  /* Target Mode Supported              */
                u_char x_TrunNotNec:1;  /* Truncate Command Not Supported     */
                u_char x_MoreSupported:1;/* More Command Supported            */
                u_char x_DMAsupported:1; /* DMA Mode Supported                */
                u_char x_DMAChannelValid:1;/* DMA Channel Field Is Valid.     */
                u_char x_ATAdevice:1;   /* This Is An ATA Device              */
                u_char x_HBAvalid:1;    /* HBA field Is Valid                 */
        }       flag0;
#define OverLapCmds     flag0.x_OverLapCmds
#define TargetMode      flag0.x_TargetMode
#define TrunNotNec      flag0.x_TrunNotNec
#define MoreSupported   flag0.x_MoreSupported
#define DMAsupported    flag0.x_DMAsupported
#define DMAChannelValid flag0.x_DMAChannelValid
#define ATAdevice       flag0.x_ATAdevice
#define HBAvalid        flag0.x_HBAvalid
        u_char  PadLength[2];           /* Pad Bytes For PIO Commands         */
        u_char  HBA[4];                 /* Host Adapter SCSI ID               */
        u_char  CPlength[4];            /* Command Packet Length              */
        u_char  SPlength[4];            /* Status Packet Length               */
        union {
                u_char  QueueSize[2];   /* Controller Queue Depth             */
                u_short Queue_Len;      /*      Corrected Byte Order          */
        } u1;
#define QueueSize       u1.QueueSize
#define Queue_Len       u1.Queue_Len    /* Accessed from ha_Config            */
        u_char  Reserved1[2];
        union {
                u_char  SG_Size[2];     /* Maximum Scatter Gather List Size   */
                u_short SG_Len;         /*      Corrected Byte Order          */
        } u2;
#define SG_Size         u2.SG_Size
#define SG_Len          u2.SG_Len       /* Accessed from ha_Config            */
        u_char  IRQ_Number:4;           /* IRQ Number                         */
        u_char  IRQ_Trigger:1;          /* IRQ Trigger: 0 = Edge, 1 = Level   */
        u_char  Secondary:1;            /* Controller Not At Address 0x170    */
        u_char  DMA_Channel:2;          /* DMA Channel Index For ISA          */

        u_char  IRQ;                    /* IRQ address                        */
        struct {
                u_char x_Disable:1;     /* ISA I/O Address Disabled           */
                u_char x_ForceAddr:1;   /* PCI Forced To An EISA/ISA Addr     */
                u_char x_SG64K:1;       /* 64K of SG space                    */
                u_char x_SGUnaligned:1; /* Supports unaligned SG, otherwise 4 */
                u_char x_Reserved2:4;   /* Reserved Field                     */
        }       flag1;
#define Disable flag1.x_Disable
#define ForceAddr       flag1.x_ForceAddr
#define SG64K   flag1.x_SG64K
#define SgUnaligned     flag1.x_SgUnaligned
        u_char  MaxScsiID:5;            /* Maximun SCSI Target ID Supported   */
        u_char  MaxChannel:3;           /* Maximun Channel Number Supported   */
        u_char  MaxLUN;                 /* Maximum LUN Supported              */
        struct {
                u_char x_Reserved3:3;   /* Reserved Field                     */
                u_char x_AutoTermination:1;/* Support auto term (low byte)    */
                u_char x_PCIM1:1;       /* PCI M1 chipset                     */
                u_char x_RaidIDQuestionable:1;/* Raid ID may be questionable  */
                u_char x_PCIbus:1;      /* PCI Adapter Flag                   */
                u_char x_EISAbus:1;     /* EISA Adapter Flag                  */
        }       flag2;
#define AutoTermination flag2.x_AutoTermination
#define PCIM1   flag2.x_PCIM1
#define RaidIDQuestionable      flag2.x_RaidIDQuestionable
#define PCIbus  flag2.x_PCIbus
#define EISAbus flag2.x_EISAbus

        u_char  RaidNum;                /* Raid Host Adapter Number           */
} DptConfig_t;

/**************************************************************************
** DPT Host Adapter structure - One Structure For Each Host Adapter That **
**  Is Configured Into The System.  The Structure Supplies Configuration **
**  Information, Status Info, Queue Info And An Active CCB List Pointer. **
***************************************************************************/

typedef struct DptHa {
        struct spi_hba          ha_hba; /* must be first              */
        struct isadev           ha_isa;
        struct intrhand         ha_ih;
        u_short                 ha_IO_Pending;
        u_short                 ha_Total_IO_Pending;    /* Parent Only        */

        u_short                 ha_Base;        /* base port for each board   */
        DptCcb_t *              ha_CCB;         /* CCBs in use                */
        u_char                  ha_Channel:2;   /* The Channel *we* represent */
        u_char                  ha_debug:1;     /* Debug Flag                 */

        DptConfig_t             ha_Config;
#define is_pci(sc)      ((sc)->ha_Config.PCIbus)
#define is_eisa(sc)     ((sc)->ha_Config.EISAbus)
#define is_isa(sc)      (!is_pci(sc) && !is_eisa(sc))
        /* Status Packet For This Adapter (not for the BUS)  */
        DptStat_t *             ha_Status_Packet;       /* Common for HBA */
        vm_addr_t               ha_Status_Packet_addr;  /* Byte swapped */
        struct DptHa *          ha_next;        /* Virtual HBA list     */
        struct DptHa *          ha_parent;      /* Virtual HBA parent   */
        struct DptHa *          ha_next_parent;/* Next parent           */
} DptHa_t;

DptHa_t *       DptHa;

/*
 *      Describe the Inquiry Data returned on Page 0 from the Adapter. The
 *      Page C1 Inquiry Data is described in the DptConfig_t structure above.
 */
typedef struct {
        u_char  deviceType;
        u_char  rm_dtq;
        u_char  otherData[6];
        u_char  vendor[8];      /* 'DPT     '   */
        u_char  modelNum[7];    /* 'PM????A'    */
        u_char  modelSuf[9];    /* '?? -RDLW '  */
        u_char  firmware[3];    /* v'07G'       */
        u_char  revision;       /*      .'0'    */
} DptInq_t;

/*
 *      Prototypes of the routines we have in this object.
 */
STATIC int      dptprobe __P((struct device *parent, struct cfdata *cf,
                    void *aux));
STATIC void     dptattach __P((struct device *parent, struct device *dev,
                    void *args));
STATIC int      dptstart __P((scp_t *scp));
STATIC int      dptstart_s __P((scp_t *scp));
STATIC void     dpthbareset __P((struct spi_hba *hba));
STATIC void     dptqueue __P((DptHa_t *sc, DptCcb_t *ccb));
STATIC int      dptioctl __P((dev_t dev, u_long cmd, caddr_t data, int flag,
                    struct proc *p));
STATIC int      dptopen __P((dev_t dev, int flags, int ifmt, struct proc *p));
STATIC int      dptintr __P((DptHa_t *sc));

/* Bounce and CCB management routines */
STATIC void     dpt_cleanup __P((DptHa_t *sc, DptCcb_t *ccb));

STATIC void     dpt_init_sccb __P((scp_t *scp, DptCcb_t *ccb));

/* CCB allocation routines */
STATIC void     dpt_init_ccb __P((DptHa_t *sc));
STATIC DptCcb_t *dpt_get_ccb __P((DptHa_t *sc));
STATIC void     _dpt_free_ccb __P((int flags, DptCcb_t *ccb));
STATIC inline void dpt_free_ccb __P((DptHa_t *sc, int flags, DptCcb_t *ccb));

/* Bounce buffer allocation routines */
STATIC void     dpt_init_bounce __P((DptHa_t *sc));
STATIC DptBounce_t *dpt_get_bounce __P((DptHa_t *sc, int flags));
STATIC void     _dpt_free_bounce __P((int flags, DptBounce_t *bounce));
STATIC inline void dpt_free_bounce __P((DptHa_t *sc, int flags,
                                DptBounce_t *bounce));

/* Utility routines */
static inline void              dpt_prstring __P((u_char *s, int n));
static inline vm_offset_t       KVTOPHYS __P((caddr_t v));
static inline vm_addr_t         VMTOSWAP __P((vm_offset_t v));
static inline vm_offset_t       SWAPTOVM __P((vm_addr_t v));
static inline vm_len_t          LMTOSWAP __P((vm_size_t l));
static inline vm_size_t         SWAPTOLM __P((vm_len_t l));
static inline DptInq_t *        dpt_inquiry __P((DptHa_t *sc, int target,
                                        int channel));
static inline vm_offset_t       dpt_bounce __P((DptHa_t *sc, DptCcb_t *ccb,
                                        caddr_t v, vm_size_t len, int rw));
static inline void              dpt_cmd __P((int io_addr, vm_offset_t up));
static inline int               dpt_wait __P((int io_addr,int bits,int state));
static inline void              dpt_add_to_do __P((DptHa_t *sc, int Channel));
static inline void              dpt_remove_to_do __P((void));
#if (defined(DPT_DEBUG_FLAGS) && (DPT_DEBUG_FLAGS))
static inline void              dpt_error __P((DptHa_t *sc, DptCcb_t *ccb,
                                        int targetStatus, int hostStatus,
                                        int residue));
#endif
/*
 *      Conversion macros.
 *      The GNU compiler knows about inline, a far better way of dealing with
 *      macros as we can ensure strong type checking.
 *
 * pjd - Unfortunately inline functions tend to screw up the compiler's
 * register allocation mechanism. Its probably better to use macros than
 * inline functions.
 */
#define ISA_ADDR        ((vm_offset_t)ISA_MAXADDR)

static inline vm_offset_t
KVTOPHYS(caddr_t v)
{
        return ((vm_offset_t)pmap_extract (kernel_pmap, (vm_addr_t)v));
}

static inline vm_addr_t
VMTOSWAP(vm_offset_t v)
{
        return ((vm_addr_t)ntohl (v));
}

static inline vm_offset_t
SWAPTOVM(vm_addr_t v)
{
        return ((vm_offset_t)htonl (v));
}

static inline vm_len_t
LMTOSWAP(vm_size_t l)
{
        return ((vm_len_t)ntohl (l));
}

static inline vm_size_t
SWAPTOLM(vm_len_t l)
{
        return ((vm_size_t)htonl (l));
}

/*
 *      Provided to give a consistent interface should we decide to do more
 *      cleanup later.
 */
static inline void
dpt_free_ccb(DptHa_t *sc, int flags, DptCcb_t *ccb)
{
        _dpt_free_ccb (flags, ccb);
}

static inline void
dpt_free_bounce(DptHa_t *sc, int flags, DptBounce_t *bounce)
{
        _dpt_free_bounce (flags, bounce);
}

/*
 *      List of all known EATA or DPT EISA ids. This is handled better by
 *      doing a generalized test like the PCI probe, but the BSDi OS chose
 *      an explicit list for their implementation. In any case, it forces us
 *      to document all of the EISA products we work with. This following
 *      list does not imply that there is lack of support for ISA and PCI
 *      versions of our cards, far from it ...
 */
static char *dpt_ids[] = {
        "DPT2402",      /* DPT PM2012A/9X               */
        "DPTA401",      /* DPT PM2012B/9X               */
        "DPTA402",      /* DPT PM2012B2/9X              */
        "DPTA410",      /* DPT PM2x22A/9X               */
        "DPTA411",      /* DPT Spectre                  */
        "DPTA412",      /* DPT PM2021A/9X               */
        "DPTA420",      /* DPT Smart Cache IV (PM2042)  */
        "DPTA501",      /* DPT PM2012B1/9X              */
        "DPTA502",      /* DPT PM2012Bx/9X              */
        "DPTA701",      /* DPT PM2011B1/9X              */
        "DPTBC01",      /* DPT PM3011/7X ESDI           */
        "NEC8200",      /* NEC EATA SCSI                */
        "ATT2408",      /* ATT EATA SCSI                */
        NULL
};

/*
 *      Here is the auto-probe structure used to nest our tests appropriately
 *      during the startup phase of the operating system.
 */
struct cfdriver dptcd =
        { 0, "dpt", dptprobe, dptattach, DV_DULL, sizeof(DptHa_t), dpt_ids, 0 };
/*
 *      Here is the driver entry points supported by the SCSI handler.
 */
static struct spi_hbadriver dpthbadriver =
        { dptstart, dptstart_s, dpthbareset };

/*
 * devsw for dpt hba driver
 *
 * only ioctl is used. the sd driver provides all other access.
 */
struct devsw dptsw = {
        &dptcd,
        dptopen,
        nullclose,
        noread,
        nowrite,
        dptioctl,
        noselect,
        nommap,
        nostrat,
        nodump,
        nopsize,
        0,
        nostop
};

/*
 *      dpt_cmd (io_port, up);
 *      Perform a CP_DMA_CMD. This routine might best be optimized into
 *      assembler code, but I chose to support portability instead.
 */
static inline void
dpt_cmd (io_addr, up)
        int             io_addr;
        vm_offset_t     up;
{       /*
         *      We could be called by an Interrupt Service, so lets not
         * sleep or wait unduely long (50ms). We must escape quickly from
         * any tests here so do not do any spinloops or sleep delays.
         *
         *      A Timeout would be very bad, so lets not ...
#define dpt_cmd_TIMEOUT 50000L
         */
#       if (defined(dpt_cmd_TIMEOUT))
                if (inb(io_addr + HA_AUX_STATUS) & HA_AUX_BUSY) {
                        struct timeval  stamp = time;
#                       if (dpt_cmd_TIMEOUT > 1000000L)
                                stamp.tv_sec += dpt_cmd_TIMEOUT / 1000000L;
#                       endif
                        if ((stamp.tv_usec += (dpt_cmd_TIMEOUT%1000000L))
                            >= 1000000L) {
                                ++stamp.tv_sec;
                                stamp.tv_usec -= 1000000L;
                        }
                        while ((inb(io_addr + HA_AUX_STATUS) & HA_AUX_BUSY)
                         && ((time.tv_sec == stamp.tv_sec)
                           ? (time.tv_usec < stamp.tv_usec)
                           : (time.tv_sec < stamp.tv_sec)));
                }
#       else
                while (inb(io_addr + HA_AUX_STATUS) & HA_AUX_BUSY);
#       endif
#if (BYTE_ORDER == LITTLE_ENDIAN)
                outb (io_addr + HA_DMA_BASE + 0, ((u_char)(u_long)up));
                outb (io_addr + HA_DMA_BASE + 1, ((u_short)(u_long)up) >> 8);
                outb (io_addr + HA_DMA_BASE + 2, ((u_long)up) >> 16);
                outb (io_addr + HA_DMA_BASE + 3, ((u_long)up) >> 24);
#elif (BYTE_ORDER == BIG_ENDIAN)
                outb (io_addr + HA_DMA_BASE + 0, ((u_long)up) >> 24);
                outb (io_addr + HA_DMA_BASE + 1, ((u_long)up) >> 16);
                outb (io_addr + HA_DMA_BASE + 2, ((u_short)(u_long)up) >> 8);
                outb (io_addr + HA_DMA_BASE + 3, ((u_char)(u_long)up));
#elif (BYTE_ORDER == PDP_ENDIAN)
                outb (io_addr + HA_DMA_BASE + 0, ((u_long)up) >> 16);
                outb (io_addr + HA_DMA_BASE + 1, ((u_long)up) >> 24);
                outb (io_addr + HA_DMA_BASE + 2, ((u_char)(u_long)up));
                outb (io_addr + HA_DMA_BASE + 3, ((u_short)(u_long)up) >> 8);
#else
                UNKNOWN BYTE_ORDER BYTE ORDER
#endif
        outb (io_addr + HA_COMMAND, CP_DMA_CMD);
}

/*
 *      Wait for the DPT card to accept the next command when manually polling.
 *      This routine need not be efficient, as it is only to be used at
 *      probe time.
 */
static inline int
dpt_wait(io_addr, bits, state)
        u_short io_addr;
        u_char  bits;
        u_char  state;
{       u_short i;
        u_char  c;

        for (i = 400;; ) {      /* wait 20ms for not busy */
                c = inb (io_addr + HA_STATUS) & bits;
                if (c == state)
                        return (0);
                DELAY (50);     /* 50 us */
                if (--i <= 0)
                        return (-1);
        }
}

#if (defined(DPT_DEBUG_FLAGS) && (DPT_DEBUG_FLAGS))
static void
Print_EataCp (ccb)
        DptCcb_t *      ccb;
{
        printf ("%x[EataCp]=%s%s%s%s%s%s%s%s%s%s ReqLen=%d\n (%d,%d,%d)%s%s%s %02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x\n Len=%d vp=%x dma=%x stat=%x req=%x\n Bounce=%x Device=%x\n",
                ccb,
                ccb->EataCp.cp_DataIn ? " In" : "",
                ccb->EataCp.cp_DataOut ? " Out" : "",
                ccb->EataCp.cp_Interpret ? " Interpret" : "",
                ccb->EataCp.cp_Quick ? " Quick" : "",
                ccb->EataCp.cp_Scatter_Gather ? " ScatterGather" : "",
                ccb->EataCp.cp_Auto_Req_Sen ? " AutoRequestSense" : "",
                ccb->EataCp.cp_HBA_Init ? " Init" : "",
                ccb->EataCp.cp_SCSI_Reset ? " Reset" : "",
                ccb->EataCp.cp_FWNest ? " FWNest" : "",
                ccb->EataCp.cp_PhsUnit ? " PhysUnit" : "",
                ccb->EataCp.cp_Req_Len,
                ccb->EataCp.cp_Channel, ccb->EataCp.cp_id,
                ccb->EataCp.cp_msg_lun,
                ccb->EataCp.cp_msg_luntar ? " Target" : "",
                ccb->EataCp.cp_msg_disco_rico ? " Disco/Reco" : "",
                ccb->EataCp.cp_msg_identify ? " Identify" : "",
                ccb->EataCp.cp_cdb[0], ccb->EataCp.cp_cdb[1],
                ccb->EataCp.cp_cdb[2], ccb->EataCp.cp_cdb[3],
                ccb->EataCp.cp_cdb[4], ccb->EataCp.cp_cdb[5],
                ccb->EataCp.cp_cdb[6], ccb->EataCp.cp_cdb[7],
                ccb->EataCp.cp_cdb[8], ccb->EataCp.cp_cdb[9],
                ccb->EataCp.cp_cdb[10], ccb->EataCp.cp_cdb[11],
                SWAPTOLM(ccb->EataCp.cp_datalen), ccb->EataCp.cp_vp,
                SWAPTOVM(ccb->EataCp.cp_dataDMA),
                SWAPTOVM(ccb->EataCp.cp_statDMA),
                SWAPTOVM(ccb->EataCp.cp_reqDMA),
                ccb->ccb_Bounce, ccb->ccb_Device);
        DELAY(10000000);
}
#endif
/*
 *      Read EATA Config and test for supported HBA - returns pointer to
 *      Configuration structure if supported
 */
STATIC DptConfig_t *
dpt_EATA_ReadConfig (io_addr, drq)
        u_short         io_addr;
        u_short         drq;
{       int             i, j;
        u_char          stat;
        u_short *       ip;
        DptCcb_t *      ccb;
        DptStat_t *     status;
        DptConfig_t *   config;
        vm_offset_t     addr;

        while ((((stat = inb(io_addr + HA_STATUS))
         != (HA_ST_READY|HA_ST_SEEK_COMPLETE))
          && (stat != (HA_ST_READY|HA_ST_SEEK_COMPLETE|HA_ST_ERROR))
          /* This results from the `wd' probe at our addresses */
          && (stat != (HA_ST_READY|HA_ST_SEEK_COMPLETE|HA_ST_ERROR|HA_ST_DRQ)))
         || (dpt_wait(io_addr, HA_ST_BUSY, 0))) {
                /*
                 *      RAID Drives still Spinning up? (This should only occur
                 *      if the DPT controller is in a NON PC (PCI?) platform).
                 */
                if ((inb(io_addr + HA_ERROR) != 'D')
                 || (inb(io_addr + HA_ERROR + 1) != 'P')
                 || (inb(io_addr + HA_ERROR + 2) != 'T'))
                        return ((DptConfig_t *)NULL);
        }
        if ((config = (DptConfig_t *)malloc (sizeof(DptConfig_t), M_TEMP,
            M_WAITOK)) == (DptConfig_t *)NULL)
                return ((DptConfig_t *)NULL);
        /*
         *      If any of our buffers allocate above the 16M boundary for isa
         * looking cards, then resort to a Read EATA Config command instead.
         */
        if (io_addr >= 0x400) do {
                addr = KVTOPHYS ((caddr_t)config);
                if ((ccb = (DptCcb_t *)malloc (
                    sizeof(DptCcb_t), M_TEMP, M_WAITOK)) == (DptCcb_t *)NULL)
                        break;
                if ((status = (DptStat_t *)malloc (sizeof(DptStat_t), M_TEMP,
                    M_WAITOK)) == (DptStat_t *)NULL) {
                        free ((caddr_t)ccb, M_TEMP);
                        break;
                }
                /*
                 * Do a Read Vital Data Page SCSI command instead of using the
                 * CP_READ_CFG_PIO/CP_READ_CFG_DMA comand!
                 */
                (void)bzero ((char *)ccb, sizeof(DptCcb_t));
                ccb->EataCp.cp_dataDMA = VMTOSWAP (addr);
                addr = KVTOPHYS ((caddr_t)status);
                ccb->EataCp.cp_statDMA = VMTOSWAP (addr);
                ccb->EataCp.cp_Interpret = 1;
                ccb->EataCp.cp_DataIn = 1;
                ccb->EataCp.cp_Auto_Req_Sen = 1;/* Clear sense data on error */
                ccb->EataCp.cp_id = 7;/* Doesn't matter since interpret mode */
                ccb->EataCp.cp_scsi_cmd = CMD_INQUIRY;
                ccb->EataCp.cp_extent = 1;
                ccb->EataCp.cp_page = 0xC1;
                ccb->EataCp.cp_len = sizeof(DptConfig_t);
                ccb->EataCp.cp_datalen=LMTOSWAP((vm_size_t)sizeof(DptConfig_t));
                ccb->EataCp.cp_vp = (caddr_t)ccb;
                ccb->EataCp.cp_msg_identify = 1;

                (void)bzero ((char *)config, sizeof(DptConfig_t));
                (void)bzero ((char *)status, sizeof(DptStat_t));
                DptStat_Reset_BUSY (status);
                dpt_cmd (io_addr, KVTOPHYS ((caddr_t)ccb));
                /*
                 * Wait for the board to report a finished instruction.
                 */
                i = 20000;      /* One second to respond */
                while (DptStat_BUSY (status)) {
                        DELAY(50);
                        if (--i <= 0)
                                break;
                }

                i = (!(status->sp_HBA_stat)) && (!(status->sp_SCSI_stat))
                 && (!(status->sp_inv_residue));

                /* Clear interrupt by reading status register */

                if (((inb(io_addr+HA_STATUS) & HA_ST_ERROR) == 0) && (i)
                 && (*((u_long *)(config->EATAsignature)) == EATA_SIG)
                 && config->HBAvalid && config->DMAsupported) {
                        free ((caddr_t)ccb, M_TEMP);
                        free ((caddr_t)status, M_TEMP);

                        /* 4G.Z did this to me */
                        if ((i = config->ConfigLength
                            + (int)(&(((DptConfig_t *)0L)->ConfigLength))
                            + sizeof(config->ConfigLength))
                            < (sizeof(DptConfig_t))) {
                                (void)bzero ((char *)config + i,
                                    sizeof(DptConfig_t) - i);
                                /* Defaults for older Firmware */
                                        /* 2001? */
                                ip = (u_short *)((char *)config
                                    + config->ConfigLength);
                                if (ip <= (u_short *)(&config->HBA[
                                    MAX_CHANNEL-1])) {
                                        config->HBA[MAX_CHANNEL - 1] = 7;
                                }
                                        /* 2012 */
                                if ((ip <= (u_short *)(&(config->flag2)))
                                 && ((io_addr & 0xFFF) == 0xC88))
                                        config->EISAbus = 1;
                        }
                        return (config);
                }
                outb (io_addr + HA_COMMAND, CP_EATA_RESET);
                DELAY (750000L);        /* Wait 1/2 second and try following */
                free ((caddr_t)ccb, M_TEMP);
                free ((caddr_t)status, M_TEMP);
        } while (0);    /* break marker */
        /*
         * I *could* do a Read Config INQUIRY using a polled instruction, but
         * have been asked to not call these instructions if at all possible.
         * So, we do a CP_READ_CFG_PIO for the older (2011 + 2012) boards,
         * or for ISA boards when we mistakenly allocated buffers above the
         * 16MB mark.
         */
        if (dpt_wait(io_addr, HA_ST_BUSY, 0)) {
                free ((caddr_t)config, M_TEMP);
                return ((DptConfig_t *)NULL);
        }
        outb (io_addr + HA_COMMAND, CP_READ_CFG_PIO);
        (void)bzero ((char *)(ip = (u_short *)config), sizeof(DptConfig_t));
        i = ((int)(&(((DptConfig_t *)0L)->ConfigLength))
            + sizeof(config->ConfigLength)) / sizeof(u_short);
/* ? - mgs */
        if (dpt_wait(io_addr, 0xFF, HA_ST_DATA_RDY)) {
                free ((caddr_t)config, M_TEMP);
                return ((DptConfig_t *)NULL);
        }
        do {
                *(ip++) = inw (io_addr + HA_DATA);
        } while (--i > 0);

        if ((i = config->ConfigLength) > (sizeof(DptConfig_t)
            - (int)(&(((DptConfig_t *)0L)->ConfigLength))
            - sizeof(config->ConfigLength)))
                i = sizeof(DptConfig_t)
                  - (int)(&(((DptConfig_t *)0L)->ConfigLength))
                  - sizeof(config->ConfigLength);
        j = i + (int)(&(((DptConfig_t *)0L)->ConfigLength))
            + sizeof(config->ConfigLength);
        i /= sizeof(u_short);
        do {
                *(ip++) = inw (io_addr + HA_DATA);
        } while (--i > 0);
        /*
         *      Flush until we have read 512 bytes.
         */
        i += (512 - j) / sizeof(u_short);
        do {
                (void)inw (io_addr + HA_DATA);
        } while (--i > 0);
        /* Defaults for older Firmware */
        if (ip <= (u_short *)(&config->HBA[MAX_CHANNEL - 1])) {
                config->HBA[MAX_CHANNEL - 1] = 7;       /* 2001? */
        }
        if ((ip <= (u_short *)(&(config->flag2)))
         && ((io_addr & 0xFFF) == 0xC88))
                config->EISAbus = 1;                    /* 2012 */
        if ((config->OverLapCmds == 0)
         && (config->ConfigLength == 0x1C)
         && (config->QueueSize[1] == (1 & 0xFF))
         && (config->QueueSize[0] == (1 >> 8))
         && (config->CPlength[3] == 0x20)) {    /* 2011 & 2012 3G.0 */
                /*
                 *      There are some versions of Firmware that report
                 *      no overlap comands when you do a PIO retrieval of
                 *      the parameters. Lets Lie and say they can handle it.
                 */
                /* config->OverLapCmds = 1; */
                config->QueueSize[1] = NCCBS & 0xFF;
                config->QueueSize[0] = NCCBS >> 8;
                config->SG_Size[1] = (SG_SIZE & 0xFF);
                config->SG_Size[0] = (SG_SIZE >> 8);
                config->CPlength[3] = 0x2C;
        }
        if (((inb(io_addr + HA_STATUS) & HA_ST_ERROR) == 0)
         && (*((u_long *)(config->EATAsignature)) == EATA_SIG)
         && config->HBAvalid && config->DMAsupported)
                return (config);

        free ((caddr_t)config, M_TEMP);
        return ((DptConfig_t *)NULL);
}

/*
 *      Return true if the Adapter is a DPT SCSI HBA. The entire line of
 *      DPT PCI products are detected here. This includes:
 *              PM2024          68000 based Traditional Product, Narrow
 *              PM2044W         68000 based Smart Cache IV Narrow/Wide
 *              PM2144W         68020 based Smart Cache IV Narrow/Wide
 *              PM2144UW        68020 based Smart Cache IV Ultra
 *              PM3224W         68030 based SmartRaid II, MultiChannel capable
 *              PM3334W         68040 based SmartRaid III, MultiChannel capable
 *              PM3334UW        68040 based SmartRaid III, Ultra MultiChannel
 */
STATIC int
dpt_pci_match(pci)
        pci_devaddr_t * pci;
{
        return ((pci_inw (pci, PCI_VENDOR_ID) == 0x1044)
         && (pci_inw (pci, PCI_DEVICE_ID) == 0xA400)
         && (pci_inb (pci, PCI_CLASSCODE_L) == 0x00)
         && (pci_inw (pci, PCI_CLASSCODE_H) == 0x0100));
}

/*
 *      The following sections of code help virtualize the individual channels.
 *      It is a dynamically allocated list of virtual adapters to create on
 *      subsequent non-forced device probes. First in First out placement and
 *      removal.
 *
 *      This is required because the BSDi operating system SCSI subsystem does
 *      not have MultiChannel support capability.
 */
typedef struct {
        u_char          todo_Channel;   /* Actual channel (0 based)        */
        DptHa_t *       todo_parent;    /* *real* Host Bus Adapter         */
} Dpt_ToDo_t;

STATIC Dpt_ToDo_t *     dpt_todo;
STATIC int              dpt_todo_len;

static inline void
dpt_add_to_do (sc, Channel)
        DptHa_t *       sc;
        int             Channel;
{       Dpt_ToDo_t *    p;

        if (!(p=malloc(sizeof(Dpt_ToDo_t)*(++dpt_todo_len), M_TEMP, M_WAITOK)))
                return;
        if (dpt_todo) {
                if (dpt_todo_len > 1)
                        (void)bcopy ((caddr_t)dpt_todo, (caddr_t)p,
                            sizeof(Dpt_ToDo_t) * (dpt_todo_len - 1));
                free (dpt_todo, M_TEMP);
        }
        dpt_todo = p;
        p += dpt_todo_len - 1;
        p->todo_parent = sc;
        p->todo_Channel = Channel;
}

static inline void
dpt_remove_to_do ()
{
        if (!dpt_todo)
                return;
        if (--dpt_todo_len <= 0) {
                dpt_todo_len = 0;
                free (dpt_todo, M_TEMP);
                dpt_todo = (Dpt_ToDo_t *)NULL;
                return;
        }
        (void)bcopy ((caddr_t)(dpt_todo + 1), (caddr_t)dpt_todo,
            sizeof(Dpt_ToDo_t) * dpt_todo_len);
}

/*
 * Probe for DPT controller.  If we find it we will use it. We may return
 * virtual adapters and allow non-forced probing in Software RAID order.
 */
STATIC int
dptprobe(parent, cf, aux)
        struct device *         parent;
        struct cfdata *         cf;
        void *                  aux;
{
        struct isa_attach_args *ia = (struct isa_attach_args *) aux;
        DptConfig_t *           config;
        u_short *               ip;
        int                     i;
        pci_devaddr_t *         pci;
        u_short                 board_drq;
        int                     checkmaster = cf->cf_flags & 1;

        extern u_short          eisa_slot_map;
        extern int              dpt_todo_len;
        extern Dpt_ToDo_t *     dpt_todo;

        static DptConfig_t      Config;
        static u_short          dpt_eisa_slot_map;
        static int              primary_done;
        static u_short          isa_regs[] = {
                0x1F0, 0x170, 0x330, 0x230
        };
#       define  isa_regs_1F0    0
#       define  isa_regs_170    1

        if (!ia)
                return (0);
        /*
         * Base I/O address not specified in config file? Search all
         * possible controller locations.
         */
        if (ia->ia_iobase == 0) {
                /*
                 * Check todo list for virtual controllers
                 * return any found
                 */
                if ((dpt_todo_len > 0) && dpt_todo) {
                        ia->ia_aux = (void *)NULL;
                        ia->ia_iosize = 0;
                        ia->ia_irq = 0;
                        ia->ia_drq = DRQNONE;
                        ia->ia_maddr = (caddr_t)NULL;
                        ia->ia_msize = 0;
                        return (1);
                }

                /*
                 *      Lets probe all possible controllers. First do Primary,
                 * PCI, EISA and then ISA cards. This is done in this order to
                 * allow us to support software RAID (later).
                 */
                dpt_eisa_slot_map |= eisa_slot_map;
                config = (DptConfig_t *)NULL;

                switch(ia->ia_bustype) {
                case BUS_PCI: /* PCI */
                    while (pci = pci_scan(dpt_pci_match)) {
                    ia->ia_iobase = pci_inw (pci, PCI_PORT_BA) & 0xFFFFFFFE;
                    /*
                     *      The following may be unnecessary since
                     * we check for PCI Forced Address in the
                     * configuration. I do the following to reduce
                     * calls to read config that may bear false
                     * information, or superfluous calls to read
                     * the configuration when obviously forced.
                     */
                    /*
                     * Check to make sure it is not a forced
                     * address EISA that could be probed later.
                     */
                    switch (inb (i = ia->ia_iobase)) {
                        case 0x12:  /* DPT */
                            if (inb (++i) == 0x14)
                                    continue;
                            break;
                        case 0x38:  /* NEC */
                            if ((inb (++i) == 0xA3) && (inb (++i) == 0x82))
                                    continue;
                            break;
                        case 0x06:
                            if ((inb (++i) == 0x94) && (inb (++i) == 0x24))
                                    continue;
                            break;
                    }
                    /*
                     * Check to make sure it is not a forced
                     * address ISA already in use, or overlaps
                     * another device.
                     */
                    if (isa_portcheck (ia->ia_iobase+=0x10, DPT_NPORT) == 0)
                            continue;
                    /*
                     * Check to make sure it is not a forced
                     * address ISA that could be probed later.
                     */
                    for (ip = isa_regs; ip < &isa_regs[sizeof(isa_regs)
                        / sizeof(isa_regs[0])]; ++ip)
                            if (*ip == ia->ia_iobase)
                                    break;

                    if (ip < &isa_regs[sizeof(isa_regs)
                        / sizeof(isa_regs[0])])
                            continue;
                    /*
                     *      Now, check if it is a *valid* DPT
                     * controller. It is too late to add this back
                     * to the PCI slots :-(
                     */
                    if (config = dpt_EATA_ReadConfig (ia->ia_iobase,
                        ia->ia_drq)) {
                            /*
                             *      Well, lets now check if the
                             * card tells me it is forced to an
                             * ISA/EISA address.
                             */
                            if (config->PCIbus && config->ForceAddr) {
                                    free (config, M_TEMP);
                                    config = NULL;
                                    continue;
                            }
                            break;
                        }
                    }
                    break;

                case BUS_EISA: /* EISA */
                    /*
                     * Check for "Primary" controller which must be EISA by
                     * definition.
                     */
                    if (primary_done == 0) {
                        ++primary_done;
                        i = 1;
                        do {ia->ia_iobase = EISA_PROD_ID_BASE(i) + 8;
                            /*
                             * Remove any EISA cards that have the
                             * incorrect PAL.
                             */
                            if (eisa_match(cf, ia) == 0) {
                                dpt_eisa_slot_map |= 1 << i;
                                continue;
                            }
                            /*
                             * If we've already found one EISA card and its
                             * a "Primary", don't read the config from the
                             * rest.
                             * Do all EISA now for PAL test ???
                             */
                            if (config)
                                continue;

                            if (config = dpt_EATA_ReadConfig (ia->ia_iobase,
                              ia->ia_drq)) {
                                /* Save Primary Address */
                                primary_done = ia->ia_iobase;
                                /*
                                 * if EISA and not disabled, clear address
                                 * of secondary so we don't find it later.
                                 */
                                if (config->EISAbus && !(config->Disable))
                                  isa_regs[config->Secondary
                                    ? isa_regs_170 : isa_regs_1F0] = 0;

                                /*
                                 * Its NOT EISA or we found a secondary,
                                 * skip it.
                                 */
                                if (!config->EISAbus || config->Secondary) {
                                    free (config, M_TEMP);
                                    config = NULL;
                                }
                            }
                        } while (++i < EISA_NUM_PHYS_SLOT);

                        /*
                         * If we didn't already find a "Primary", check
                         * its well known port address, the first one
                         * possible. If we find it, clear the first
                         * possible address slot so we don't find it
                         * again later.
                         */
                        if (isa_regs[isa_regs_1F0]
                         && isa_portcheck(isa_regs[isa_regs_1F0], DPT_NPORT)
                         && !config && checkmaster
                         && (config = dpt_EATA_ReadConfig (
                            isa_regs[isa_regs_1F0], ia->ia_drq))) {
                                /* Save Primary Address */
                                primary_done = isa_regs[isa_regs_1F0];
                                isa_regs[isa_regs_1F0] = 0;
                        }
                        /* 1 or Primary Address */
                        ia->ia_iobase = primary_done;
                    }
                    /*
                     *  Now scan for next available slot
                     */
                    if (!config)
                    for (i = 1; i < EISA_NUM_PHYS_SLOT;
                      dpt_eisa_slot_map |= 1 << (i++)) {
                        if ((dpt_eisa_slot_map & (1 << i))
                         || (dpt_eisa_slot_map |= 1 << i,
                          ((config = dpt_EATA_ReadConfig (ia->ia_iobase
                          = EISA_PROD_ID_BASE(i) + 8, ia->ia_drq)) == NULL)))
                                continue;
                        /* Primary has already been dealt with */
                        if (config->EISAbus && !(config->Disable) &&
                         config->Secondary)
                                isa_regs[isa_regs_170] = 0;
                        break;
                    }
                    break;

                default: /* ISA */
                    for (ip = isa_regs;
                      ip < &isa_regs[sizeof(isa_regs)/sizeof(isa_regs[0])];
                      ++ip) {
                        /*
                         * Check to make sure it is not overlapping another.
                         */
                        if (((i = *ip) == 0) ||
                         (*ip = 0, (isa_portcheck (i, DPT_NPORT) == 0)) ||
                         (!(config=dpt_EATA_ReadConfig (ia->ia_iobase = i,
                         ia->ia_drq))))
                                continue;
                        break;
                    }
                    break;
                }
                if (!config) {
                    return (0);
                }
        } else {
            /*
             * a non-zero io_base pretty much means ISA bus. - pjd
             */

            /*
             * It is decided that any forced addressing should not
             * restrict our configuration to handle any possible future
             * problems. This includes *even* re-using an address that may
             * have already been eliminated by an auto probe ...
             */
            if ((isa_portcheck (ia->ia_iobase, DPT_NPORT) == 0) ||
             (!(config = dpt_EATA_ReadConfig (ia->ia_iobase, ia->ia_drq))))
                return (0);
             /*
              * Make sure that a subsequent auto probe will not
              * re-use an EISA address.
              */
             if ((ia->ia_iobase & 0xFFF) == 0xC88)
                 dpt_eisa_slot_map |= 1 << ((ia->ia_iobase >> 12) & 0xF);
             /*
              * Make sure that a subsequent auto probe will not
              * re-use the ISA addresses.
              */
              if (config->EISAbus && !(config->Disable))
                  isa_regs[config->Secondary ? isa_regs_170 : isa_regs_1F0] = 0;
              for (ip = isa_regs;
                ip < &isa_regs[sizeof(isa_regs)/sizeof(isa_regs[0])];
                ++ip) {
                  if (*ip == ia->ia_iobase)
                      *ip = 0;
              }
        }
#if (defined(BETA_DPT))
        {       static u_char   not_first_time_through;

                if (!not_first_time_through) {
                        ++not_first_time_through;
                        printf ("Copyright 1996 Distributed Processing Technology. Evaluation copy not for sale!\n");
                }
        }
#endif
        /*
         *      Note, it is not safe to use ia_aux to point to dynamically
         *      allocated memory as there are no guarantees it will be used
         *      by the attach function.
         */

        Config = *config;

        free (config, M_TEMP);
        /*
         *      Fill in the isa_attach arguments
         */
        ia->ia_aux = &Config;
        if (((i = Config.IRQ_Number) || (i = Config.IRQ)) && (i == 2))
                i = 9;

        /* verify correctness of irq here */
        if (i) {
            if (ia->ia_irq == IRQUNK)
                ia->ia_irq = irq_indextomask(i);
            else if (ia->ia_irq != irq_indextomask(i)) {
                printf("dpt: probe failed - hardcoded IRQ doesn't match config data (irq%d)\n", i);
                return(0);
            }
        }
        /*
         *      Setting the size to something other than DPT_NPORT will make no
         * difference since the size is only translated to an isa address range.
         */
        ia->ia_iosize = DPT_NPORT;
        if (Config.EISAbus) {
                eisa_slotalloc ((ia->ia_iobase >> 12) & 0xF);
                /*
                 *      The EISA card is also visible at this ISA address.
                 */
                if (!(config->Disable))
                        isa_portalloc (config->Secondary ? 0x170 : 0x1F0,
                            DPT_NPORT);
        }
        /*
         *      PCI does this as part of it's probe, but lets just do this here
         * to cover all cases.
         */
        if (Config.IRQ_Trigger) {
                outb (IO_ELCR1, inb (IO_ELCR1) | ia->ia_irq);
                outb (IO_ELCR2, inb (IO_ELCR2) | ia->ia_irq >> 8);
                ia->ia_irq |= IRQSHARE;
        }
        if (Config.DMAChannelValid) {
            /* verify correctness of drq here */
            board_drq = "\0\7\6\5"[Config.DMA_Channel];
            if ((ia->ia_drq == DRQNONE) || (ia->ia_drq == (u_short)-1)) {
  printf("dpt: warning - no DRQ is configured, however board has hard-coded DRQ(drq%d)\n", board_drq);
            } else if (ia->ia_drq != board_drq) {
                printf("dpt: probe failed - hardcoded DRQ (drq%d) doesn't match config data (drq%d)\n", ia->ia_drq, board_drq);
                return(0);
            }
        }
        ia->ia_maddr = (caddr_t)NULL;
        ia->ia_msize = 0;

        return (1);
}

/*
 * Allocate an outbound bounce page, copying data if necessary.
 */
static inline vm_offset_t
dpt_bounce(sc, ccb, v, len, rw)
        DptHa_t *       sc;
        DptCcb_t *      ccb;
        caddr_t         v;
        vm_size_t       len;
        int             rw;
{       DptBounce_t *   bo;
        vm_size_t       o;

        bo = dpt_get_bounce (sc, 0 /*FL_NOMASK*/);
        bo->bo_next = ccb->ccb_Bounce;
        ccb->ccb_Bounce = bo;
        /*
         *      This bit of offset magic has no known purpose. I am simply
         * copying what was done in other HBA drivers assuming that it had a
         * performance advantage for the bcopy routine.
         */
/* check this! - pjd */
        o = (int)v & PGOFSET;
        if (rw == B_READ) {
                bo->bo_dst = (caddr_t)v;
                bo->bo_len = len;
        } else  (void)bcopy((void *)v, (void *)(bo->bo_v + o), len);
        return (bo->bo_p + o);
}

/*
 *      Check if a device is available at the specified target and channel.
 * This test is used to determine if a deeper scan (to virtualize * channels
 * and luns > MAX_LUN, or to even do a scan for all LUNs) is necessary and thus
 * is hard coded for LUN 0 only. I can't use the spivar/scp based routines
 * since the key routines are still labelled to be `private' and no doubt
 * can change without notice.
 *
 *      To do an inquiry to the HBA, the target is set to -1. We did this
 * *manually* during probe time to pick up page C1, but since we have the
 * Interrupt/DMA connections on now, lets use the tools here to pick up the
 * page 0 values.
 */
static inline DptInq_t *
dpt_inquiry (sc, target, channel)
        DptHa_t *               sc;
        int                     target, channel;
{       DptInq_t *              inquire;
        scp_t *                 scp;
        extern scp_t *          spi_create_unattached_scp (scp_t *      scp,
                                                           spi_hba_t *  hba,
                                                           targ_t *     targ,
                                                           int          tid,
                                                           int          lun,
                                                           size_t       hpsize,
                                                           int          flags);
        extern void             spi_destroy_scp (scp_t * scp);
        extern int              scsi_enquiry(scp_t *scp, void *buf, int len);

        /* Ohh nooo, not the HBA! (unless target is -1) */
        if ((target == sc->ha_Config.HBA[MAX_CHANNEL - 1 - (channel & 0x3)])
         || ((inquire = (DptInq_t *)malloc (sizeof(DptInq_t), M_TEMP, M_WAITOK))
            == (DptInq_t *)NULL))
                return ((DptInq_t *)NULL);

        scp = spi_create_unattached_scp((scp_t *)NULL, &(sc->ha_hba),
            (targ_t *)NULL, target + (channel << 5), 0, 0, SCP_C_NOINT);

        scsi_enquiry(scp, inquire, sizeof(DptInq_t));

        /*
         * Since our driver is set up for Auto Request Sense, no need to
         * perform a sense request should the target inquiry be so inclined.
         */
        if (scp->scp_status & HA_STATUS) {
                free ((caddr_t)inquire, M_TEMP);
                inquire = (DptInq_t *)NULL;
        }
        spi_destroy_scp (scp);
        return (inquire);
}

static inline void
dpt_prstring (s, n)
        u_char *        s;
        int             n;
{       while ((--n >= 0) && (*s) && (*s != ' ') && (*s != '-'))
                printf ("%c", *(s++));
}

/*
 *      Attach the devices, and virtual devices to the driver list.
 */
STATIC void
dptattach (parent, self, aux)
        struct device *         parent;
        struct device *         self;
        void *                  aux;
{       struct cfdata *         cf;
        int                     target;
#define sc              ((DptHa_t *)self)
        DptInq_t *              iq;
        static DptHa_t *        last_parent;
        scsi_attach_args_t      sa;
        DptBounce_t *           bounce;

        if (!aux)
                return;

        if (!parent || !(cf = parent->dv_cfdata)) {
                if (!(((struct isa_attach_args *)aux)->ia_aux))
                        dpt_remove_to_do ();
                return;
        }
        if (!DptHa)
                DptHa = sc;
        /*
         * Most of the work is in common code, but we need to do some
         * DPT-specific setup in between, so the job is split.
         */
        sc->ha_ih.ih_fun = dptintr;

        /*
         *      Lets see if we are virtualizing other channels and targets.
         */
        if (!(((struct isa_attach_args *)aux)->ia_aux)) {
                if ((!dpt_todo) || (dpt_todo_len <= 0))
                        return;
                /* Link to the parent software structure */
                sc->ha_parent = dpt_todo->todo_parent;
                sc->ha_next = dpt_todo->todo_parent->ha_next;
                dpt_todo->todo_parent->ha_next = sc;
                /* Fill in the important instance variables */
                sc->ha_Channel = dpt_todo->todo_Channel;
                /* Use common data */
                sc->ha_Base = dpt_todo->todo_parent->ha_Base;
                sc->ha_Config = dpt_todo->todo_parent->ha_Config;
                sc->ha_Status_Packet = dpt_todo->todo_parent->ha_Status_Packet;
                sc->ha_Status_Packet_addr
                    = dpt_todo->todo_parent->ha_Status_Packet_addr;
                dpt_remove_to_do ();
        } else {
                int             size;
                vm_offset_t     addr;

                /*
                 *      This is the real McCoy!
                 */
                sc->ha_parent = sc;
                sc->ha_Base = ((struct isa_attach_args *)aux)->ia_iobase;
                sc->ha_Config = *((DptConfig_t *)(
                    ((struct isa_attach_args *)aux)->ia_aux));
                if ((size = sc->ha_Config.SPlength[3]
                  + (sc->ha_Config.SPlength[2] << 8)
                  + (sc->ha_Config.SPlength[1] << 16)
                  + (sc->ha_Config.SPlength[0] << 24)) < sizeof(DptStat_t))
                        size = sizeof(DptStat_t);
                if (addr = (vm_offset_t)malloc (size, M_DEVBUF, M_WAITOK)) {
                        (void)bzero ((char *)addr, sizeof(DptStat_t));
                        sc->ha_Status_Packet = (DptStat_t *)addr;

                        if (((addr = KVTOPHYS ((caddr_t)addr)) > ISA_ADDR)
                         && is_isa (sc)) {
                                printf (
                                    "%s: status struct cannot malloc ISA!\n",
                                    sc->ha_hba.hba_dev.dv_xname);
                                return;
                        }
                        sc->ha_Status_Packet_addr = VMTOSWAP (addr);
                } else {
                        printf ("%s: status struct cannot malloc!\n",
                           sc->ha_hba.hba_dev.dv_xname);
                        return;
                }
                /*
                 *      Construct the Bounce Buffers.
                 */
                if (sc->ha_Config.SG64K) {      /* Ignore value in SG_Size */
                        sc->ha_Config.SG_Size[1] = ((65536/sizeof(SgElement_t))
                            & 0xFF);
                        sc->ha_Config.SG_Size[0] = ((65536/sizeof(SgElement_t))
                            >> 8);
                }
                if (((size = sc->ha_Config.SG_Size[1]
                  + (sc->ha_Config.SG_Size[0] << 8)) > SG_SIZE)
                 || (size <= 0))
                        size = SG_SIZE;
                sc->ha_Config.SG_Len = size;
                if (is_isa (sc) && ((vm_offset_t)(maxmem * NBPG) > ISA_ADDR))
                        dpt_init_bounce (sc);
                /*
                 * link the ccb's into a free-list
                 */
                if (((size = sc->ha_Config.QueueSize[1]
                    + (sc->ha_Config.QueueSize[0] << 8)) > SG_SIZE)
                 || (size <= 0))
                        size = NCCBS;
                sc->ha_Config.Queue_Len = size;
                dpt_init_ccb (sc);
                /*
                 *      Link parents
                 */
                if (last_parent)
                        last_parent->ha_next_parent = sc;
                last_parent = sc;
        }
#       if (defined(DPT_DEBUG_FLAGS) && (DPT_DEBUG_FLAGS & DPT_DEBUG_SET))
                sc->ha_debug = DPT_DEBUG_FLAGS & ~DPT_DEBUG_SET;
#       endif

        /* We only allocate storage for each MAX_TARGET_IDs */
        if (sc->ha_Config.MaxScsiID <= 0)
                sc->ha_Config.MaxScsiID = MAX_TARGET_ID - 1;    /* default */

        /*
         *      Link into isa and set interrupt handler (our caller has
         *      already set ih_fun). negotiate bus master DMA.
         */
        isa_establish (&(sc->ha_isa), &(sc->ha_hba.hba_dev));
        sc->ha_ih.ih_arg = sc;
        if ((((struct isa_attach_args *)aux)->ia_aux)) {
                int     drq;

                intr_establish (((struct isa_attach_args *)aux)->ia_irq,
                    &sc->ha_ih, DV_DISK);
                if ((drq = ((struct isa_attach_args *)aux)->ia_drq) != DRQNONE)
                        at_dma_cascade (drq);
        }
        /*
         * Check installed targets?
         */
        sc->ha_hba.hba_driver = &dpthbadriver;
        /*
         *      Only do a bus/HBA reset on the first time through.
         */
        if ((((struct isa_attach_args *)aux)->ia_aux))
                dpthbareset (&(sc->ha_hba));
        printf (":");

        /*
         *      Print the HBA model number as inquired from the card.
         */
        if (iq = dpt_inquiry (sc,-1,0)) {
                if (iq->vendor[0] && (iq->vendor[0] != ' ')) {
                        printf (" ");
                        dpt_prstring (iq->vendor, sizeof(iq->vendor));
                }
                if (iq->modelNum[0] && (iq->modelNum[0] != ' ')) {
                        printf (" ");
                        dpt_prstring (iq->modelNum, sizeof(iq->modelNum));
                        /* Differential? */
                        if (iq->modelSuf[5] && (iq->modelSuf[5] != ' '))
                                printf ("%c", iq->modelSuf[5]);
                        /* Wide Board? */
                        if (iq->modelSuf[7] && (iq->modelSuf[7] != ' '))
                                printf ("%c", iq->modelSuf[7]);
                        dpt_prstring (iq->modelSuf, sizeof(iq->modelSuf));
                        printf (" SCSI HBA");
                }
                if (iq->firmware[0] && (iq->firmware[0] != ' ')) {
                        printf (" v");
                        if (iq->firmware[0] == '0')
                                dpt_prstring (iq->firmware + 1,
                                    sizeof(iq->firmware) - 1);
                        else    dpt_prstring (iq->firmware,
                                    sizeof(iq->firmware));
                        printf (".%c", iq->revision);
                }
                free ((caddr_t)iq, M_TEMP);
        }
        /*
         *      Add in additional probe responses for more channels. We
         * are reusing the variable `target' for a channel loop counter.
         */
        if (sc->ha_Config.MaxChannel > 0) {
                if (((struct isa_attach_args *)aux)->ia_aux)
                 for (target=1; target <= sc->ha_Config.MaxChannel; ++target) {
                        dpt_add_to_do (sc, target);
                }
                printf (" bus %d", sc->ha_Channel);
        }
        printf ("\n");
        /*
         * Split the concurrency between pseudo adapters.
         */
        SPI_HBACONCURRENCY(&(sc->ha_hba), sc->ha_Config.Queue_Len
                / (sc->ha_Config.MaxChannel + 1));
        bzero (&sa, sizeof(sa));
        /*
         *      Max the target and unit limits so that the only real
         *      limit is the number per HBA.
         */
        sa.sa_maxuops = sa.sa_maxtops = (sc->ha_Config.Queue_Len)
                / (sc->ha_Config.MaxChannel + 1);
        for (target = 0; target <= sc->ha_Config.MaxScsiID; ++target) {
                if (!(iq = dpt_inquiry (sc, target, sc->ha_Channel)))
                        continue;
                free ((caddr_t)iq, M_TEMP);
                /*
                 * Register the target and it's LUNs.
                 */
                sa.sa_tid = target;
                SPI_FOUNDTARGET (&(sc->ha_hba), &sa);
        }
        /*
         *      To allow pass through, we stage at least one bounce buffer for
         * use as a ccb hold register (to restore the state of the ccb after
         * the transaction).
         */
        if (bounce = dpt_get_bounce (sc, FL_NOSLEEP))
                dpt_free_bounce (sc, 0, bounce);
        else    dpt_init_bounce ((DptHa_t *)NULL);
        /*
         *      Fixup the OS revision as saved in the dptsig for the
         *      engine (dptioctl.h) to pick up.
         */
        dpt_sig.dsDescription[16] = osrelease[0];
        dpt_sig.dsDescription[17] = osrelease[1];
        dpt_sig.dsDescription[18] = osrelease[2];
#undef sc
}

/*
 *      Reset the HBA, targets and BUS.
 *              Currently this resets *all* the SCSI busses, which could
 *              affect other pseudo adapters. Except for Tape drives on
 *              multichannel adapters, this will not adversely affect the
 *              system. Per SCSI bus reset is possible through the EATA
 *              immediate command, but it was not implemented in all
 *              firmware revisions prompting me to currently just use the
 *              All Bus reset.
 */
STATIC void
dpthbareset(hba)
        struct spi_hba *        hba;
{
        outb (((DptHa_t *)hba)->ha_Base + HA_COMMAND, CP_EATA_RESET);
        /* Old firmware has a headache if you talk to it too soon! */
        DELAY (2000000);
}

/*
 * Finish up bouncing in on a read. Must be called at splbio. This routine
 * removes the CCB from any active or pending lists for an HBA as well.
 */
STATIC void
dpt_cleanup(sc, ccb)
        DptHa_t *       sc;
        DptCcb_t *      ccb;
{       vm_offset_t     dst;
        DptBounce_t *   bo;
        DptCcb_t *      ccb_p;
        DptCcb_t *      ccb_l;

        /*
         *      Clear up any `read' bounce pages.
         */
        while (bo = ccb->ccb_Bounce) {
                if (dst = (vm_offset_t)bo->bo_dst) {
                        (void)bcopy((void *)(bo->bo_v + (dst & PGOFSET)),
                            (void *)dst, bo->bo_len);
                        bo->bo_dst = (caddr_t)NULL;
                }
                ccb->ccb_Bounce = bo->bo_next;
                dpt_free_bounce (sc, 0 /*FL_NOMASK*/, bo);
        }
/* seems like common queue and dequeue routines would be useful here - pjd */
        /* Remove from list of active ccbs for this HBA */
        for (ccb_l = (DptCcb_t *)NULL, ccb_p = sc->ha_CCB;
            ccb_p;
            ccb_l = ccb_p, ccb_p = ccb_p->ccb_next) {
                if (ccb_p == ccb) {
                        if (ccb_l == (DptCcb_t *)NULL)
                                sc->ha_CCB = ccb_p->ccb_next;
                        else    ccb_l->ccb_next = ccb_p->ccb_next;
                        break;
                }
        }
        dpt_free_ccb (sc, 0 /*FL_NOMASK*/, ccb);
}

/*
 * Initialize (most of the changeable section) a SCCB.
 * We assume that the CDB has already been set up, so all we do here is
 * generate the Scatter Gather list.
 */
STATIC void
dpt_init_sccb(scp, ccb)
        scp_t *         scp;
        DptCcb_t *      ccb;
{       int             i, n, o, isa, rw, sgpages, pages;
        DptHa_t *       sc = (DptHa_t *)(scp->scp_hba);
        SgElement_t *   sg;
        u_int           aphys;
        vm_offset_t     offset, phys;
        caddr_t         v;
        vm_size_t       size, len;
#       if (defined(DPT_DEBUG_FLAGS) && (DPT_DEBUG_FLAGS))
                static char     SG_size[] = "SG size";
#       endif
        struct buf      buf;
        struct buf *    bp;

        ccb->ccb_scp = scp;
        if (scp->scp_tid == -1) {
                ccb->EataCp.cp_Interpret = 1;
                scp->scp_tid = ccb->EataCp.cp_id
                    = sc->ha_Config.HBA[MAX_CHANNEL - 1 - (sc->ha_Channel & 3)];
        }
        /*
         *      EATA Pass Through initializations (for drivers without
         *      Pass Through, this block can be deleted).
         */
        else if (scp->scp_tid == -2) {
                DptBounce_t *   bo;
                /*
                 *      EATA Pass through initialization
                 */
                /* Save EataCp, returned through bounce buffer handling */
                bo = dpt_get_bounce (sc, 0);
                bo->bo_next = ccb->ccb_Bounce;
                ccb->ccb_Bounce = bo;
                bo->bo_dst = (caddr_t)&(ccb->EataCp);
                bo->bo_len = sizeof(EataCp_t);
                (void)bcopy((void *)&(ccb->EataCp), (void *)(bo->bo_v
                    + (vm_size_t)((int)(&(ccb->EataCp))&PGOFSET)),
                    sizeof(EataCp_t));
                /* Copy pass through EataCp */
                ccb->EataCp = *((EataCp_t *)(scp->scp_data));
                /* correct any EataCp errors and override */
                ccb->EataCp.cp_vp = (caddr_t)ccb;
                ccb->EataCp.cp_Auto_Req_Sen = 1;
                ccb->EataCp.cp_Req_Len = sizeof(struct scsi_sense);
                /* Hope optimizer sees this is the same as the exit code */
                ccb->EataCp.cp_statDMA = sc->ha_Status_Packet_addr;
                ccb->EataCp.cp_reqDMA = VMTOSWAP(KVTOPHYS((caddr_t)&(
                    scp->scp_sn)));
                return;
        } else  ccb->EataCp.cp_id = scp->scp_tid;
        ccb->EataCp.cp_Scatter_Gather = 0;
        ccb->EataCp.cp_DataIn = 0;
        ccb->EataCp.cp_DataOut = 0;
        bcopy (&scp->scp_cdb, ccb->EataCp.cp_cdb, sizeof(ccb->EataCp.cp_cdb));
        ccb->EataCp.cp_Channel = sc->ha_Channel;
        ccb->EataCp.cp_msg_lun = scp->scp_lun;

        /*
         *      Create a fake bp if not part of the scb
         */

        if (scp->scp_flags & SCP_NOBP) {
                bzero (bp = &buf, sizeof(buf));
                bp->b_bcount = bp->b_iocount = scp->scp_len;
                bp->b_un.b_addr = scp->scp_data;
                /* assume SCP_READ == B_READ */
                bp->b_flags = scp->scp_flags & B_READ;
        } else  bp = scp->scp_bp;

        /*
         * Given a buffer describing a transfer, set up a scatter/gather map
         * in a ccb to map that SCSI transfer. Too bad their generic routine
         * has the sg32 structure backwards with respect to the DPT variant,
         * we had to handle the ISA bounce in any case ... Initialize the CCB
         * opcode for the correct transfer type.
         */

        rw = bp->b_flags & B_READ;
        isa = is_isa (sc);
        /*
         * Given a transfer described by a `struct buf', figure out what kind
         * of transfer is needed (direct or scatter/gather).  If scatter/gather
         * is required, set up the given s/g map and return the number of
         * entries; otherwise store the physical address for the transfer's
         * page(s) and return 0.
         *
         * If the transfer uses a chain (bp->b_chain != NULL), we assume that
         * no <addr,len> pair ever crosses a page boundary.  In any case, at
         * most two pages can be partial (one at the start and one at the end).
         *
         * (Lots of ifdef'd panics here, from sheer raging paranoia.)
         */
        sg = &ccb->ccb_SG[0];

        /*
         * ###: We used to xfer 0 bytes at addr 0 for len < 0, but I
         * think this is an error.  If not, our caller will have to
         * check len < 0 and change that to len = 0 anyway.
         */
        len = bp->b_iocount;
#       if (defined(DPT_DEBUG_FLAGS) && (DPT_DEBUG_FLAGS))
                if ((sc->ha_debug) && (len < 0))
                        panic("count < 0");
#       endif
        if (len <= 0)
                aphys = 0;              /* physadr = 0 */
        else if ((sgpages = i386_btop (i386_round_page (len + (o = (int)(v
            = bp->b_un.b_addr) & PGOFSET)))) == 1) {
                /* Transfer lies entirely within a page. */
#               if (defined(DPT_DEBUG_FLAGS) && (DPT_DEBUG_FLAGS))
                        if ((sc->ha_debug) && (bp->b_chain))
                                panic("SG chain 1page");
#               endif
                aphys = KVTOPHYS(v);
#               if (defined(DPT_DEBUG_FLAGS) && (DPT_DEBUG_FLAGS))
                        if ((sc->ha_debug)
                         && ((aphys + len - 1) != KVTOPHYS(v + len - 1)))
                                panic("SG split page");
#               endif
                /*
                 * Point the CCB directly at the transfer, or at a
                 * bounce buffer. The i386_btop program will ensure that
                 * we never reach here if we have an operation that
                 * excedes the page size (the fundamental size of the
                 * bounce buffer pool).
                 */
                if (isa && (aphys >= ISA_ADDR))
                        aphys = dpt_bounce (sc, ccb, v, len, rw);
                aphys = VMTOSWAP (aphys);
        } else {/*
                 * Transfer too big (at least 2 pages) -- we'll need a
                 * scatter/gather map.  (Actually, if the underlying pages
                 * are contiguous, we could be OK, but this gets complicated,
                 * especially when using bounce buffers, so just use the
                 * map initially and then see if the resulting buffer is
                 * contiguous).
                 *
                 * Transfer may also be chained, in which case we must step
                 * to the next buffer each time bp->b_bcount runs out (note
                 * that len is the head buffer's bp->b_iocount).
                 */
                if ((pages = sgpages) > (sc->ha_Config.SG_Len))
                        panic("SG pages");

                /* First page may start at some offset, and hence be short. */
                n = bp->b_bcount;
                size = NBPG - o;
                phys = KVTOPHYS(v);
                if (isa && (phys >= ISA_ADDR))
                        phys = dpt_bounce (sc, ccb, v, (vm_size_t)size, rw);
                sg->data_addr = VMTOSWAP (offset = phys);
                (sg++)->data_len = LMTOSWAP (size);
                v += size;
                n -= size;

                /* Pages 1 through (pages-1), if pages > 2, are full size. */
                for (i = 2; i < sgpages; i++) {
                        if (n <= 0) {
#                               if (defined(DPT_DEBUG_FLAGS) && (DPT_DEBUG_FLAGS))
                                        if ((sc->ha_debug)
                                         && (n < 0 || (bp->b_chain == NULL)))
                                                panic("ab_sgmap mid chain");
#                               endif
                                n = (bp = bp->b_chain)->b_bcount;
                                v = bp->b_un.b_addr;
#                               if (defined(DPT_DEBUG_FLAGS) && (DPT_DEBUG_FLAGS))
                                        if (sc->ha_debug && ((int)v & PGOFSET))
                                                panic("SG mid addr");
#                               endif
                        }
                        phys = KVTOPHYS(v);
                        if (isa && (phys >= ISA_ADDR))
                                phys = dpt_bounce (sc, ccb, v, (vm_size_t)NBPG,
                                    rw);
                        /*
                         *      Coalesce physically adjacent buffers. This
                         * works out to a great optimization for ISA, and a
                         * barely perceptable optimization for EISA or PCI.
                         */
                        if ((offset + size) == phys) {
                                size += NBPG;
                                --sg;
                                --pages;
                        } else {
#                               if (defined(DPT_DEBUG_FLAGS) && (DPT_DEBUG_FLAGS))
                                     if (sc->ha_debug) {
                                        if (sg > &ccb->ccb_SG[SG_SIZE - 1])
                                                panic (SG_size);
                                        if (sg > &ccb->ccb_SG[
                                            sc->ha_Config.SG_Len - 1])
                                                printf (SG_size);
                                    }
#                               endif
                                sg->data_addr = VMTOSWAP (offset = phys);
                                size = NBPG;
                        }
                        (sg++)->data_len = LMTOSWAP (size);
                        v += NBPG;
                        n -= NBPG;
                }

                /* Last page is n remaining bytes. */
                if (n <= 0) {
#                       if (defined(DPT_DEBUG_FLAGS) && (DPT_DEBUG_FLAGS))
                                if ((sc->ha_debug)
                                 && (n < 0 || bp->b_chain == NULL))
                                        panic("SG last chain");
#                       endif
                        n = (bp = bp->b_chain)->b_bcount;
                        v = bp->b_un.b_addr;
#                       if (defined(DPT_DEBUG_FLAGS) && (DPT_DEBUG_FLAGS))
                                if ((sc->ha_debug) && ((int)v & PGOFSET))
                                        panic("SG last addr");
#                       endif
                }
#               if (defined(DPT_DEBUG_FLAGS) && (DPT_DEBUG_FLAGS))
                        if ((sc->ha_debug) && (n > NBPG))
                                panic("SG lastpg %d", n);
#               endif
                /*
                 *      Coalesce physically adjacent buffers. See note above.
                 */
                phys = KVTOPHYS(v);
                if (isa && (phys >= ISA_ADDR))
                        phys = dpt_bounce (sc, ccb, v, (vm_size_t)n, rw);
                if ((offset + size) == phys) {
                        n += size;
                        --sg;
                        --pages;
                } else {
#                       if (defined(DPT_DEBUG_FLAGS) && (DPT_DEBUG_FLAGS))
                            if (sc->ha_debug) {
                                if (sg > &ccb->ccb_SG[SG_SIZE - 1])
                                        panic (SG_size);
                                if (sg > &ccb->ccb_SG[
                                     sc->ha_Config.SG_Len - 1])
                                        printf (SG_size);
                            }
#                       endif
                        sg->data_addr = VMTOSWAP (phys);
                }
                sg->data_len = LMTOSWAP (n);
                /*
                 * Point the CCB at the scatter/gather map if there
                 * remains more than one scatter gather entry.
                 */
                if (pages <= 1) {
                        aphys = ccb->ccb_SG[0].data_addr;
                        /*
                         *      I wish I could use
                         *              len = bp->b_iocount;
                         *      but we have long lost the bp counter.
                         */
                        len = SWAPTOLM (ccb->ccb_SG[0].data_len);
                } else {aphys = ccb->ccb_SG_addr;
                        len = (vm_size_t)(pages * sizeof (ccb->ccb_SG[0]));
                        ccb->EataCp.cp_Scatter_Gather = 1;
                }
        }
        if (len) {
                if (rw) ccb->EataCp.cp_DataIn = 1;
                else    ccb->EataCp.cp_DataOut = 1;
        }
        ccb->EataCp.cp_dataDMA = aphys;
        ccb->EataCp.cp_datalen = LMTOSWAP (len);
        ccb->EataCp.cp_statDMA = sc->ha_Status_Packet_addr;
        ccb->EataCp.cp_reqDMA = VMTOSWAP(KVTOPHYS((caddr_t)&(scp->scp_sn)));
}

/*
 * Fire off a CCB.
 *      dptqueue - Queue one into the adapter.
 */
STATIC void
dptqueue(sc, ccb)
        DptHa_t *       sc;
        DptCcb_t *      ccb;
{       int             i;

        /*
         *      Limit the number of CCBs are sent to this HBA. Better to sleep,
         *      than to hardware loop like a nut! By limiting the number of
         *      CCBs to an individual HBA here, we manage to perform all the
         *      processing of the CCB ready to drop the next one into the
         *      controller. We could limit the CCBs we are allowed to take,
         *      but that may have a performance hit.
         */
        ++(sc->ha_IO_Pending);
        ++(sc->ha_parent->ha_Total_IO_Pending);
        while (sc->ha_parent->ha_Total_IO_Pending > (sc->ha_Config.Queue_Len))
                tsleep ((caddr_t)&(sc->ha_parent->ha_Total_IO_Pending), PRIBIO,
                    "dptqueue", 0);

        dpt_cmd (sc->ha_Base, ccb->ccb_addr);
}

#if (defined(DPT_DEBUG_FLAGS) && (DPT_DEBUG_FLAGS))
STATIC void
dpt_error (sc, ccb, Status, residue)
        DptHa_t *       sc;
        DptCcb_t *      ccb;
        int             Status, residue;
{       int             errno, size;
        char **         table;
        char *          msg;
        static char *   HBA_Status[] = {
                NULL /* "No Error on command" */,
                "Device Selection Time Out",
                "Device Command Time Out",
                "SCSI Bus was RESET !",
                "Initial Controller Power Up",
                "Unexpected BUS Phase",
                "Unexpected BUS Free",
                "SCSI Bus Parity Error",
                "SCSI Bus Hung",
                "Unexpected Message Reject",
                "SCSI Bus Reset Stuck",
                "Auto-Request Sense Failed",
                "HBA Memory Parity error",
                "CP aborted - NOT on Bus",
                "CP aborted - WAS on Bus",
                "CP was reset - NOT on Bus",
                "CP was reset - WAS on Bus",
                "HBA Memory ECC Error",
                "PCI Parity Error",
                "PCI Master Abort",
                "PCI Target Abort",
                "PCI Signalled Target Abort",
                NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,
                "Software Abort"
        };
        static char *   Target_Status[] = {
                NULL /* "Success, command done" */,
                NULL,
                "Check condition",
                NULL,
                "Condition met",
                NULL, NULL, NULL,
                "Busy",
                NULL, NULL, NULL, NULL, NULL, NULL, NULL,
                "succeeded, doing linked command",
                NULL, NULL, NULL,
                "Condition met, doing linked command",
                NULL, NULL, NULL,
                "Reservation conflict",
                NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,
                "Command terminated",
                NULL, NULL, NULL, NULL, NULL,
                "Queue full",                           /* SCSI 2 */
                NULL, NULL, NULL, NULL, NULL, NULL, NULL,
                "Auto Contingent Allegiance condition"  /* SCSI 3 */
        };
        static struct {
                u_char  command;
                char *  name;
        } scsi_commands[] = {
                { 0x00, "Test Unit Ready" },
                { 0x01, "Rezero Unit" },
                { 0x03, "Request Sense" },
                { 0x04, "Format Unit" },
                { 0x07, "Reassign Blocks" },
                { 0x08, "Read" },
                { 0x0A, "Write" },
                { 0x0B, "Seek" },
                { 0x12, "Inquiry" },
                { 0x15, "Mode Select" },
                { 0x16, "Reserve" },
                { 0x17, "Release" },
                { 0x18, "Copy" },
                { 0x1A, "Mode Sense" },
                { 0x1B, "Start/Stop/Eject/Load Unit" },
                { 0x1C, "Receive Diagnostic Results" },
                { 0x1D, "Send Diagnostic" },
                { 0x1E, "Prevent/Allow Media Removal" },
                { 0x25, "Read Capacity" },
                { 0x26, "Last Session" },
                { 0x28, "Extended Read" },
                { 0x2A, "Extended Write" },
                { 0x2B, "Extended Seek" },
                { 0x2E, "Write and Verify" },
                { 0x2F, "Verify" },
                { 0x30, "Search Data High" },
                { 0x31, "Search Data Equal" },
                { 0x32, "Search Data Low" },
                { 0x33, "Set Limits" },
                { 0x34, "Pre-Fetch" },
                { 0x35, "Synchronize Cache" },
                { 0x36, "Lock/Unlock Cache" },
                { 0x37, "Read Defect Data" },
                { 0x39, "Compare" },
                { 0x3A, "Copy and Verify" },
                { 0x3B, "Write Buffer" },
                { 0x3C, "Read Buffer" },
                { 0x3E, "Read Long" },
                { 0x3F, "Write Long" },
                { 0x40, "Change Definition" },
                { 0x41, "Write Same" },
                { 0x42, "Read Sub-Channel" },
                { 0x43, "Read Table of Contents" },
                { 0x44, "Read Header" },
                { 0x45, "Play Audio" },
                { 0x47, "Play Audio MSF" },
                { 0x48, "Play Audio TRK" },
                { 0x49, "Play Audio REL" },
                { 0x4B, "Pause/Resume" },
                { 0x4C, "Log Select" },
                { 0x4D, "Log Sense" },
                { 0x55, "Extended Mode Select" },
                { 0x5A, "Extended Mode Sense" },
                { 0xA5, "Play Audio 12" },
                { 0xA8, "Read 12" },
                { 0xA9, "Play Audio REL 12" },
                { 0xC0, "Set Address Format" },
                { 0xC1, "Read Table of Contents (SONY)" },
                { 0xC2, "Read Sub-Q" },
                { 0xC3, "Set Stop Time" },
                { 0xC4, "Playback Status" },
                { 0xC5, "Pause/Resume VU" },
                { 0xC6, "Play Audio TRK (SONY)" },
                { 0xC7, "Play Audio MSF (SONY)" },
                { 0xC8, "Play Audio VU" },
                { 0xC9, "Playback Control" },
                { 0xCA, "Pause/Resume (PIONEER)" },
                { 0xCB, "Audio Stop" },
                { 0xCC, "Audio Status" },
                { 0xD4, "Read D/A (NEC)" },
                { 0xD8, "Read D/A" },
                { 0xD9, "Audio Play" },
                { 0xDA, "Set Speed" },
                { 0xDC, "Eject/Load" },
                { 0xDD, "Set Speed (CHINON)" },
                { 0xDE, "Read Table of Contents (NEC)" },
                { 0xE0, "Drive Status" },
                { 0xE3, "Write CDP" },
                { 0xE4, "Read CDP" }
        };
        int     i;

        if (((errno = Status) & HBA_STATUS) == 0) {
                if (errno == STS_GOOD)
                        return;
                msg = "Target";
                table = Target_Status;
                size = sizeof(Target_Status) / sizeof(Target_Status[0]);
        } else {msg = "Host";
                table = HBA_Status;
                size = sizeof(HBA_Status) / sizeof(HBA_Status[0]);
                errno &= ~HBA_STATUS;
        }

        printf ("%s ", sc->ha_hba.hba_dev.dv_xname);
        if (ccb->EataCp.cp_Interpret)
                printf ("HBA ");
        if (ccb->EataCp.cp_PhsUnit)
                printf ("Physical ");
        if (ccb->EataCp.cp_FWNest)
                printf ("SW ");
        if (sc->ha_Config.MaxChannel)
                printf ("bus%d ", sc->ha_Channel);
        printf ("target%d lun%d: ",
            ccb->EataCp.cp_id, ccb->EataCp.cp_lun);
        for (i=0; i < (sizeof(scsi_commands)/sizeof(scsi_commands[0])); ++i)
                if (ccb->EataCp.cp_scsi_cmd == scsi_commands[i].command)
                        break;
        if (i < (sizeof(scsi_commands)/sizeof(scsi_commands[0])))
                printf ("%s", scsi_commands[i].name);
        else    printf ("%x", ccb->EataCp.cp_scsi_cmd);
        printf (" %s ", msg);
        if ((errno < size) && (msg = table[errno]))
            printf ("%s", msg);
        else
            printf ("error 0x%x", errno);
        if (residue)
            printf (" residue=%d", residue);
        printf ("\n");
        Print_EataCp (ccb);
}
#endif

STATIC int
dptstart_s(scp)
        scp_t *         scp;
{       DptHa_t *       sc = (DptHa_t *)scp->scp_hba;
        int             i, status, s;
        DptCcb_t *      ccb_p;
        DptCcb_t *      ccb_l;
        DptCcb_t *      ccb;

        SCP_PUSH (scp, scsi_nullcall);
        dpt_init_sccb (scp, ccb = dpt_get_ccb (sc));
        s = splbio ();
        dptqueue (sc, ccb);

        /*
         * Wait for the board to report a finished instruction. We should
         * time out on this but we could be doing a format command (NBL) ...
         */
        i = 40000;      /* Wait 2 seconds on selected commands */
        for (;;) {
                (void)dptintr (sc);
                for (ccb_p = sc->ha_CCB; ccb_p; ccb_p = ccb_p->ccb_next)
                        if (ccb_p == ccb)
                                break;
                if (ccb_p == (DptCcb_t *)NULL)
                        break;
                switch (ccb->EataCp.cp_scsi_cmd) {
                case CMD_TEST_UNIT_READY:
                case CMD_INQUIRY:
                case CMD_REQUEST_SENSE:
                        if (--i <= 0) {
                                /*
                                 *      We could have simply lost the interrupt
                                 *      (SCSI Bus Reset on old hardware)
                                 */
                                if (sc->ha_Status_Packet->sp_vp == ccb) {
                                        if ((scp->scp_status =
                                          sc->ha_Status_Packet->sp_HBA_stat)
                                            == HA_NO_ERROR)
                                            scp->scp_status =
                                              sc->ha_Status_Packet->sp_SCSI_stat;
                                        scp->scp_resid = SWAPTOLM (
                                          sc->ha_Status_Packet->sp_inv_residue);
                                } else {
                                        scp->scp_status = HA_ABORT;
                                        dpthbareset (&(sc->ha_hba));
                                }
                                dpt_cleanup (sc, ccb);
                                break;
                        }
                        DELAY (50);     /* Gives meaning to the timer */
                default:continue;
                }
                break;
        }
#       if (defined(DPT_DEBUG_FLAGS) && (DPT_DEBUG_FLAGS & DPT_DEBUG_VERBOSE))
                dpt_error (sc, ccb,
                    (sc->ha_Status_Packet->sp_HBA_stat == HA_NO_ERROR)
                        ? sc->ha_Status_Packet->sp_SCSI_stat
                        : (HBA_STATUS | sc->ha_Status_Packet->sp_HBA_stat),
                    SWAPTOLM(sc->ha_Status_Packet->sp_inv_residue));
#       endif
        splx(s);
        return (1);
}

/*
 * Start a transfer. Allocate DMA & bus
 */
STATIC int
dptstart(scp)
        scp_t *         scp;
{       DptHa_t *       sc = (DptHa_t *)(scp->scp_hba);
        DptCcb_t *      ccb = dpt_get_ccb (sc);

        dpt_init_sccb (scp, ccb);
        dptqueue (sc, ccb);
        return (0);
}

/*
 * Handle processing of current CCB as pointed to by the Status.
 */
STATIC int
dptintr (sc)
        DptHa_t *       sc;
{       u_char          Host_Status, Target_Status;
        vm_len_t        Residue;
        DptCcb_t *      ccb;
        DptCcb_t *      ccb_p;
        int             target, io_addr;
        scp_t *         scp;

        /*
         * First *check* for the interrupt, Then if it's not telling
         * about a completed operation just proceed to the next
         * adapter. We don't check the real status register because
         * this would cause us to lose the Status Packet!
         */
        if (!(inb ((io_addr = sc->ha_Base) + HA_AUX_STATUS) & HA_AUX_INTR))
                return (0);
        /*
         * If it is not busy then process the completed operation as
         * pointed to by the status packet.
         */
        ccb = sc->ha_Status_Packet->sp_vp;
        if (!ccb || !(ccb->ccb_Device)) {
                (void)inb (io_addr + HA_STATUS);
                return (0);
        }
        Host_Status = sc->ha_Status_Packet->sp_HBA_stat;
        Target_Status = sc->ha_Status_Packet->sp_SCSI_stat;
        Residue = sc->ha_Status_Packet->sp_inv_residue;
        /*
         *      Just to be safe, I found the 2012 not filling in the
         *      Status Packet in some revisions of it's firmware.
         */
        sc->ha_Status_Packet->sp_vp = (DptCcb_t *)NULL;

        /*
         *      Now, clear the interrupt so the next Status could be
         * transfered to the structure now that we have cleared out the
         * information we need.
         */
        (void)inb (io_addr + HA_STATUS);

        /*
         *      Do a validity check of the CCB and the virtual HA (incoming
         * sc is not necessarily the sc of the virtual target/bus).
         */
        if ((sc = ((DptHa_t *)(ccb->ccb_Device)))->ha_IO_Pending > 0)
                sc->ha_IO_Pending--;
        if ((sc->ha_parent->ha_Total_IO_Pending > 0)
         && ((--(sc->ha_parent->ha_Total_IO_Pending))
            <= (sc->ha_Config.Queue_Len)))
                wakeup ((caddr_t)&(sc->ha_parent->ha_Total_IO_Pending));

        if (scp = ccb->ccb_scp) {
                if (Target_Status == STS_CHECKCOND)
                        scp->scp_sn_avail = 1;
                scp->scp_status = (Host_Status == HA_NO_ERROR)
                        ? Target_Status : (Host_Status | HBA_STATUS);
                scp->scp_resid = SWAPTOLM (Residue);
                SCP_CALL(scp);
        }
        dpt_cleanup (sc, ccb);

        return (1);
}

/*
 *      The following is all the buffer pool allocation and management
 *      routines. This includes bounce buffers and Software CCBs.
 */
/*
 *      Called to allocate additional CCB resources
 *      By allocating resources in page size chunks, we can save memory and
 *      physical to virtual mapping resources.
 */
STATIC DptCcb_t *       dpt_ccb_free;           /* EISA/PCI based CCBs     */
STATIC DptCcb_t *       dpt_ccb_free_isa;       /* ISA/EISA/PCI based CCBs */

STATIC void
dpt_init_ccb (sc)
        DptHa_t *       sc;
{       DptCcb_t *      ccb;
        int             i, n, residual;
#define         MAX_CCB_PG      (NBPG / sizeof(DptCcb_t))
        static u_char   dpt_ccb_isa_error;
        static u_short  dpt_ccb_num_isa;        /* Actual number of ISA */
        static u_short  dpt_ccb_total_num_isa;  /* Required number of ISA */

        /*
         *      Always allocate one extra CCB so that we can start the
         *      initialization of a CCB while the HBA is busy processing the
         *      Maximal CCBs it is capable of dealing with.
         */
        n = (sc->ha_Config.Queue_Len) + 1;
        for (residual = i = 0; i < n; i++) {
                if (--residual > 0) {
                        ++ccb;
                } else {if ((residual = n - i) > MAX_CCB_PG)
                                residual = MAX_CCB_PG;
                        if (!(ccb = malloc (residual * sizeof(DptCcb_t),
                            M_DEVBUF, M_WAITOK))) {
                                printf ("%s: ccb struct cannot malloc!\n",
                                    sc->ha_hba.hba_dev.dv_xname);
                                break;
                        }
                        (void)bzero ((char *)ccb, residual * sizeof(DptCcb_t));
                }
                if (is_isa (sc))
                        ++dpt_ccb_total_num_isa;
                ccb->EataCp.cp_vp = (caddr_t)ccb;
                ccb->EataCp.cp_Req_Len = sizeof(struct scsi_sense);
                ccb->EataCp.cp_Auto_Req_Sen = 1;
                ccb->EataCp.cp_msg_identify = 1;
                ccb->EataCp.cp_msg_disco_rico = 1;
                ccb->ccb_SG_addr = VMTOSWAP(KVTOPHYS((caddr_t)(ccb->ccb_SG)));
                ccb->ccb_addr = KVTOPHYS((caddr_t)ccb);
                if (ccb->ccb_addr < ISA_ADDR) {
                        ccb->ccb_isa = 1;
                        if ((++dpt_ccb_num_isa >= dpt_ccb_total_num_isa)
                         && (dpt_ccb_isa_error)) {
                                printf (
                                  "%s: ccb struct has managed to malloc ISA!\n",
                                  sc->ha_hba.hba_dev.dv_xname);
                                dpt_ccb_isa_error = 0;
                        }
                } else {if ((dpt_ccb_num_isa < dpt_ccb_total_num_isa)
                         && (dpt_ccb_isa_error == 0)) {
                                printf ("%s: ccb struct cannot malloc ISA!\n",
                                  sc->ha_hba.hba_dev.dv_xname);
                                ++dpt_ccb_isa_error;
                        }
                }
                dpt_free_ccb (sc, 0, ccb);
        }
}

/*
 *      Get a free ccb, and do some initializations.
 */
STATIC DptCcb_t *
dpt_get_ccb (sc)
        DptHa_t *               sc;
{       unsigned                s;
        DptCcb_t *              ccb;
        DptCcb_t *              ccb_p;
        int                     isa;

        isa = is_isa (sc);
        s = splbio();
        /*
         *      If we can and have to, sleep waiting for one to come free.
         */
        while ((isa || (!(ccb = dpt_ccb_free)))
         && (!(ccb = dpt_ccb_free_isa)))
                tsleep (isa
                    ? (caddr_t)&dpt_ccb_free_isa
                    : (caddr_t)&dpt_ccb_free, PRIBIO, "dpt_get_ccb", 0);
        if (ccb == (DptCcb_t *)NULL) {
                splx(s);
                return (ccb);
        }
        if (ccb->ccb_isa)
                dpt_ccb_free_isa = ccb->ccb_next;
        else    dpt_ccb_free = ccb->ccb_next;
        splx(s);
        ccb->ccb_Device = (void *)sc;
        ccb->EataCp.cp_Interpret = 0;
        ccb->ccb_scp = (scp_t *)NULL;
        /*
         *      Add to HBA resource list of CCBs (in order of arrival)
         */
        if (ccb_p = sc->ha_CCB) {
                while (ccb_p->ccb_next)
                        ccb_p = ccb_p->ccb_next;
                ccb_p->ccb_next = ccb;
        } else  sc->ha_CCB = ccb;
        ccb->ccb_next = (DptCcb_t *)NULL;
        return (ccb);
}

STATIC void
_dpt_free_ccb (flags, ccb)
        int             flags;
        DptCcb_t *      ccb;
{       unsigned        s;

        if (!(flags & FL_NOMASK))
                s = splbio ();
        ccb->ccb_Device = (void *)NULL;
        if (ccb->ccb_isa) {
                ccb->ccb_next = dpt_ccb_free_isa;
                dpt_ccb_free_isa = ccb;
        } else {
                ccb->ccb_next = dpt_ccb_free;
                dpt_ccb_free = ccb;
        }
        /*
         *      If there were none, wake anybody waiting for one to come free,
         * starting with queued entries.
         */
        if (ccb->ccb_next == (DptCcb_t *)NULL) {
                if (ccb->ccb_isa)
                        wakeup((caddr_t)&dpt_ccb_free_isa);
                /* Wake up regardless if ISA or not */
                wakeup((caddr_t)&dpt_ccb_free);
        }
        if (!(flags & FL_NOMASK))
                splx(s);
}

/*
 *      Called to allocate additional Bounce resources
 *      By allocating resources in NBPG chunks, we can save some memory and
 *      physical to virtual memory mapping overhead.
 */
STATIC DptBounce_t *    dpt_bounce_free;        /* ISA based bounce buffers */

STATIC void
dpt_init_bounce (sc)
        DptHa_t *       sc;
{       DptBounce_t *   bounce;
        int             i, n, residual, pass;
#define         MAX_BOUNCE_PG   (NBPG / sizeof(DptBounce_t))

        /* For ISA cards, we could starve for resources in extreme cases */
        n = (sc) ? sc->ha_Config.SG_Len : 1;
        for (residual = i = 0; i < n; i++) {
                if (--residual > 0) {
                        ++bounce;
                        ++pass;
                } else {if ((residual = n - i) > MAX_BOUNCE_PG)
                                residual = MAX_BOUNCE_PG;
                        if (!(bounce = malloc (residual * sizeof(DptBounce_t),
                            M_DEVBUF, M_WAITOK))) {
                                printf ("%s: bounce struct cannot malloc!\n",
                                    sc->ha_hba.hba_dev.dv_xname);
                                break;
                        }
                        bzero ((caddr_t)bounce, residual * sizeof(DptBounce_t));
                        pass = 0;
                }
                if (!(bounce->bo_v = malloc (NBPG, M_DEVBUF, M_WAITOK))) {
                        if (pass == 0)
                                free (bounce, M_DEVBUF);
                        printf ("%s: bounce buffer cannot malloc!\n",
                            sc->ha_hba.hba_dev.dv_xname);
                        break;
                }
                if ((bounce->bo_p = KVTOPHYS(bounce->bo_v)) > ISA_ADDR) {
                        free (bounce->bo_v, M_DEVBUF);
                        if (pass == 0)
                                free (bounce, M_DEVBUF);
                        printf ("%s: bounce buffer cannot malloc ISA!\n",
                            sc->ha_hba.hba_dev.dv_xname);
                        break;
                }
                dpt_free_bounce (sc, 0, bounce);
        }
#undef MAX_BOUNCE_PG
}

/*
 *      Get a free bounce, and do some initializations.
 */
STATIC DptBounce_t *
dpt_get_bounce (sc, flags)
        DptHa_t *       sc;
        int             flags;
{       unsigned        s;
        DptBounce_t *   bounce;

        if (!(flags & FL_NOMASK))
                s = splbio();
        /*
         *      If we can and have to, sleep waiting for one to come free.
         */
        while ((!(bounce = dpt_bounce_free)) && (!(flags & FL_NOSLEEP)))
                tsleep((caddr_t)&dpt_bounce_free, PRIBIO, "dptbounce", 0);
        if (bounce)
                dpt_bounce_free = bounce->bo_next;
        if (!(flags & FL_NOMASK))
                splx(s);
        return (bounce);
}

/*
 *      Name: dpt_free_bounce
 *      Description: Free up the bounce buffer, but to make the chances of this
 *      buffer being received sequentially, we place them back in physical
 *      address order.
 */
void
_dpt_free_bounce (flags, bounce)
        int             flags;
        DptBounce_t *   bounce;
{       unsigned        s;
        DptBounce_t *   this;
        DptBounce_t *   last = (DptBounce_t *)NULL;

        if (!(flags & FL_NOMASK))
                s = splbio ();
        /*
         *      If there were none, wake anybody waiting for one to come free,
         * starting with queued entries, or search to sort the entry into the
         * list of bounce buffers.
         */
        if ((this = dpt_bounce_free) == (DptBounce_t *)NULL)
                wakeup((caddr_t)&dpt_bounce_free);
        else for (;this && (bounce->bo_p < this->bo_p);
                last = this, this = this->bo_next);
        bounce->bo_next = this;
        if (last)
                last->bo_next = bounce;
        else    dpt_bounce_free = bounce;
        /*
         *      If there were none, wake anybody waiting for one to come free,
         * starting with queued entries.
         */
        if (bounce->bo_next == (DptBounce_t *)NULL)
                wakeup((caddr_t)&dpt_bounce_free);
        if (!(flags & FL_NOMASK))
                splx(s);
}

#undef QueueSize        /* Grrrr */
#undef SG_Size          /* Grrrr */

#include        <i386/isa/dptioctl.h>
