#!/bin/sh
#
# File:         PLUSconfig-TMA-utils.sh
#
# Product:      Tivoli Module
# Description:  Shell Script
# Author:               Module Generator
#
# Generated:    
#
# (C) COPYRIGHT Tivoli Systems, Inc., 1999
# All Rights Reserved
# Licensed Material - Property of Tivoli Systems, Inc.
#
#  This skeleton implements utility functions for the handling 
#  of TMA dependency sets for modules generated by the Tivoli 
#  Module Builder, and is used by the Builder to generate a 
#  shell script that is intended to be "." sourced by a module's 
#  install "after script."  Utility functions herein that
#  are prefixed with "TMA_...", especially those with the
#  prefix "TMA_config_...", are intended for external use.
#  In the general case, a module install after script that is
#  running on the TMR server will source the generated script, 
#  normally named <MODULE_TYPE>config-TMA-utils.sh.  If needed,
#  (i.e., if the defaults are not sufficient), the script
#  would then set variables that indicate the desired dependency 
#  files.  These may be set using settings as described below, 
#  or overriden using the following shell variables in the after 
#  script: TMA_UNIX_TOOLS, TMA_WIN_TOOLS, TMA_BINS, TMA_LIBS, 
#  TMA_PERL_SCRIPTS, and TMA_PROD_FILES.  Default values include
#  perl for TMA_UNIX_TOOLS and most (if not all) relevant
#  content of the lcf_bundle's tools directory for 
#  TMA_WIN_TOOLS (i.e., bash, perl, ls, echo, etc.).  
#  After sourcing the generated script and setting/overriding
#  the variables as noted above, the simplest use of this
#  script by the module install "after script" is to invoke 
#  TMA_config_full (no arguments), followed by invocation of 
#  TMA_sync_gateways:
#
#      if is_TMR_server; then
#        . $PRODDIR/${MODULE_TYPE}config-TMA-utils.sh
#        TMA_config_full
#        TMA_sync_gateways
#      fi
#
#  These utilites manage two dependency set hierarchies: 1) one
#  applies to all installed modules and 2) one applies only to
#  a given module type (e.g., PLUS), or module.  
#
#  1) For the first (non-module-specific) case, these utilites 
#  manage a single "root" dependency set per applicable class. 
#  These dependency sets are named TMB_ROOT_task, TMB_ROOT_mon, 
#  and TMB_ROOT_fp.  TMB_ROOT_task is intended to become the 
#  root dependency that is added to class methods of the 
#  framework's TaskEndpoint class; TMB_ROOT_mon is available 
#  to be added as the root dependency for class methods of 
#  the TBD Distributed Monitoring class; and TMB_ROOT_fp is 
#  intended to be the root dependency that is added to class 
#  methods of the TMF_FP class.  Other dependency sets having
#  file dependencies are nested under the TMB_ROOT_* 
#  dependendencies as needed; no file dependencies should be 
#  added directly to the TMB_ROOT_* dependency sets.
#  
#  Files that are to be dependencies of a non-module-specific 
#  method (such as the run_task method of the TaskEndpoint class)
#  are assumed to be commonly shared by all modules, and may be added 
#  to one of the following secondary dependency sets: TMB_tools, 
#  which refers to "tools" that are commonly shared by all applications, 
#  such as bash and perl, TMB_etc for files to be placed under 
#  /etc/Tivoli on the endpoint, and TMB_bin and TMB_lib for binaries 
#  and libraries respectively.  As noted above, these secondary 
#  dependency sets are  added to the appropriate TMB_ROOT_* 
#  dependency set, which may also hold other (displaced) 
#  non-TMB dependency sets.
#
#  Normally the default non-module-specific dependency files are
#  sufficient for a module.  Otherwise, these may be specified
#  by setting values for the following environment variables:
#  TMA_UNIX_TOOLS, TMA_WIN_TOOLS, TMA_BINS, TMA_LIBS, and 
#  TMA_PERL_SCRIPTS.  Dependency files that are specified using
#  these variables are added to an appropriate secondary 
#  dependency set that is nested under TMB_ROOT_*.  TMB_ROOT_* is 
#  then added as the dependency set on the method(s) of interest 
#  (since for any method only one dependency set is allowed).
#  If perchance a dependency set that is not TMB_ROOT_* is
#  already attached to the method, these utilities will detach
#  that dependency set from the method without deleting the, 
#  dependency set, re-attach it as a subdependency of TMB_ROOT_*, 
#  and then attach TMB_ROOT_* to the method.  Any failure in this 
#  swap will result in an attempt to reattach the original 
#  dependency set to the method.
#
#  2) Dependency files that are only needed by a given module, 
#  such as a .fmt file for logfile adapter configuration, would 
#  become dependencies of all modules if they were to be made part 
#  of the TMB_* hierarchy of dependency sets.  For this reason,
#  it is recommended that an alternative module-specific class
#  and methods be used for module-specific dependencies.  These
#  utility functions support creating and customizing such a 
#  secondary hierarchy for module-specific purposes.  
#
#  By default, for modules of type "PLUS", a set of three
#  "root" dependency sets are created to which other dependency 
#   sets may be attached:
#      PLUS_ROOT_fp
#      PLUS_ROOT_mon
#      PLUS_ROOT_task
#  These root dependency sets are holders for the following nested 
#  dependency sets:
#      PLUS_<MODULE_NAME>_fp     - nested under PLUS_ROOT_fp
#      PLUS_<MODULE_NAME>_mon    - nested under PLUS_ROOT_mon
#      PLUS_<MODULE_NAME>_task   - nested under PLUS_ROOT_task
# 
#  Then, for the PLUS-specific class "SolutionsLink", the following 
#  class methods are assigned dependencies as follows:
#      faux_job_launch     - dependency set PLUS_ROOT_task
#      faux_task_launch    - dependency set PLUS_ROOT_task
#      faux_distribute     - dependency set PLUS_ROOT_mon
#      faux_fp_distribute  - dependency set PLUS_ROOT_fp
#      faux_ofp_distribute - dependency set PLUS_ROOT_fp
#
#  Assumptions:
#    Dependencies on the content of default lcf_bundle content (i.e., 
#    files in the lcf_bundle directory on the gateway that are
#    installed with the platform) lcf_bundle content is additive, 
#    out-of-cache, and made part of the TMB_tools, TMB_bin, or TMB_lib 
#    dependency sets.  Dependencies on the content of the module binaries 
#    is out-of-cache and made part of the module-specific dependency set.
#
#    With regard to terminology, non module-specific classes and methods
#    are referred to as "basic".  Dependency sets on these are referred
#    to as "common".
#
#    If the module type is GEM, this script assumes that the 
#    endpoint version of the wpostemsg binary is to be made a
#    dependency (which means that the Adapter Configuration
#    Facility should be installed on the gateway).  Hence, if
#    the module type is GEM, these utilities assume that
#    wpostemsg is to be included in the TMA_BINS list by default.
#
#  Settings:
#    MODULE_TYPE (no default)
#    TMA_MODULE_PRODPATH (default: ../generic_unix/TME/PLUS/IM2/)
#    TMA_ALL_TOOLS (default: "true") - if false, configuration of tools dep
#      set is skipped.
#    TMA_WIN_TOOLS (no default) - additions to the list of windows tools, 
#      which currently include the following: basename, bash, chmod, cut, 
#      expr, find, mkdir, mv, perl, sed, sh, tar, awk, cat, cp, cpp, date,
#      diff, dirname, grep, ls, make, rm, sleep, sort, touch, tr, wc, 
#      whoami, wrunuiep, uname, win32gnu.dll
#    TMA_UNIX_TOOLS (no default) - additions to the list of unix tools, 
#      which currently include perl.
#    TMA_PERL_SCRIPTS (no default) - perl scripts (non module-specific)
#    TMA_LIBS (no default) - libraries (non module-specific)
#    TMA_BINS (no default) - additions to the default bin files, which 
#      currently include wpostemsg if MODULE_TYPE is GEM.
#    TMA_PROD_FILES (no default) - additions to the default prod files, 
#      which currently include <MODULE_TYPE>product-info.sh
#    TMA_PROD_FILES_TASK (no default) - additions to the default 
#      prod files for task methods, which currently include 
#      <MODULE_TYPE>product-info.sh, *.rls, *.fmt, and *.baroc,
#      if MODULE_TYPE is PLUS.
#    TMA_PROD_FILES_MON (no default) - module binaries for monitor 
#      methods.  If this setting has a value, then 
#      <MODULE_TYPE>product-info.sh is also made part of the list.
#    TMA_PROD_FILES_FP (no default) module binaries for file package
#      methods.  If this setting has a value, then 
#      <MODULE_TYPE>product-info.sh is also made part of the list.

###
#
#	errecho
#
errecho() {
  echo "$*" >&2
}

###
#
# is_OS_NT
#
is_OS_NT() {
  if [ "`uname`" = "Windows_NT" ]; then
    return 0
  else
    return 1
  fi
}

###
#
# is_oserv_alive
#
is_oserv_alive() {
  if $OSERV_CHECK > /dev/null 2>&1; then
    return 0
  else
    return 1
  fi
}

###
#
# check_oserv
#
check_oserv() {
  if $OSERV_CHECK 1> /dev/null 2>& 1; then
    return 0
  else
    errecho "oserv has died, trying to restart ..."
    if is_OS_NT; then
      o_dispatch=${o_dispatch:-94}
      restart_cmd="net start oserv $o_dispatch"
    else
      restart_cmd="/etc/Tivoli/oserv.rc start"
    fi
    $restart_cmd 1>&2

    if [ $? -ne 0 ]; then
      errecho "Could not restart oserv using $restart_cmd ..."
      return 1
    fi
  fi
}

###
#
# is_TMA_supported
#
is_TMA_supported() {
  [ "$DEBUG" = true ] && set -x
  if wlsinst -p | grep Framework > /dev/null; then
    # "TME 10 Framework<EOL>" is THE monikker for Framework 3.2 ...
    if wlsinst -p | grep "TME 10 Framework$" > /dev/null; then
      errecho "Framework 3.6 is required for TMA configuration"
      return 1
    elif wlookup -ar Gateway > /dev/null; then
      return 0  ## we got Gateway AND not 3.2, must be 3.6 or higher!!!
    else
      return 1
    fi
  else
    return 1
  fi
}

###
#
# is_TMR_server
#
is_TMR_server() {
  ODNUM=`odadmin | grep Dispatcher | awk '{print $3}'`
  if [ $ODNUM -ne 1 ]; then
    return 1 
  fi
  return 0
}

###
#
# is_existing_dep_object
#
is_existing_dep_object() {
  dep_object=$1
  if wdepset -v $dep_object > /dev/null 2>&1; then
    return 0
  elif wdepset -v @DependencyMgr:${dep_object}#${IRONAME} > /dev/null 2>&1; then
    return 0
  else
    return 1
  fi
}

###
#
# get_root_depsets_common
#
get_root_depsets_common() {
  [ "$DEBUG" = true ] && set -x

  all_depsets=""
  for c in $BASIC_ALL_CLASS_NAMES; do
    all_depsets="$all_depsets `map_class_to_root $c`"
  done
  echo $all_depsets
}

###
#
# get_root_depsets_module
#
get_root_depsets_module() {
  [ "$DEBUG" = true ] && set -x

  all_depsets=""
  for c in $MODULE_ALL_CLASS_NAMES; do
    for t in TASK; do
      all_depsets="$all_depsets `map_class_to_root $c $t`"
    done
  done
  echo $all_depsets
}

###
#
# map_class_to_root
#
map_class_to_root() {
  [ "$DEBUG" = true ] && set -x

  class=$1
  type=$2  # optional: one of TASK, MON, FP, or ALL

  for i in $BASIC_TASK_CLASS_NAMES; do
    if [ "$i" = "$class" ]; then
      echo $TMB_ROOT_DEPSET_TASK
      return
    fi
  done
   
  # Must be mapping a module depset, do it by type
  eval names='"$'MODULE_${type}_CLASS_NAMES'"'
  for i in $names; do
    if [ "$i" = "$class" ]; then
      eval dset='"$'MODULE_ROOT_DEPSET_${type}'"'
      echo $dset
      return
    fi
  done
}

###
#
#      add_id_to_list
#
add_id_to_list() {
  [ "$DEBUG" = true ] && set -x
  match_id="$1"
  list_name="$2"
  eval list='"$'${list_name}'"'
  for i in $list; do
    if [ "$i" = "$match_id" ]; then
      return 
    fi
  done
  eval $list_name='"'${list} $match_id'"'
}

###
#
#      lookup_id_in_list 
#
lookup_id_in_list() {
  [ "$DEBUG" = true ] && set -x
  match_name="$1"
  list="$2"
  for i in $list; do
    eval name='"$'${i}'"'
    if [ "$name" = "$match_name" ]; then
      echo "$i"
      break
    fi
  done
}

###
#
# wchdep_get (a more robust version of wchdep -g)
#
wchdep_get() {
  [ "$DEBUG" = true ] && set -x

  class=$1
  method=$2

  if wchdep_check $class $method; then
    wchdep -g @Classes:${class} $method
  fi
}

###
#
# wchdep_check  (support for wchdep_get)
#
wchdep_check() {
  [ "$DEBUG" = true ] && set -x

  class=$1
  method=$2

  # We may have a platform bug that causes the oserv to die if we do a
  # wchdep and there is no dep on a method!  We'll try to handle this
  # robustly ... until we know it's fixed!
  check_oserv || return 1

  # see if the class exists, return nothing if not
  wlookup -r Classes $class > /dev/null 2>&1 || return 1


  if wchdep -g @Classes:$class $method 1> /dev/null 2>& 1; then
    # figure out the existing dependency, if there is one
    dep="`wchdep -g @Classes:$class $method 2> /dev/null`"
    if [ -n "$dep" ]; then
      if is_existing_dep_object $dep; then
        return 0
      fi
    fi
  fi

  # restart it now in case we killed it
  check_oserv || return 1

  # Now that we KNOW there wasn't an existing dependency, do a 
  # wchdep that will inoculate us against this problem next time!
  wchdep @Classes:$class OBJECT_NIL $method > /dev/null
  return 1
}

###
#
# wchdep_rm  (remove our depset from a method ... don't delete it)
#
wchdep_rm() {
  [ "$DEBUG" = true ] && set -x

  class=$1
  method=$2

  d_oid=`wchdep_get $class $method`
  if is_depset_ours $d_oid; then
    cleanup_depset_children $d_oid
    if has_depset_children $d_oid; then
      errecho "Unable to remove $d_oid, still has children!"
    else
      wchdep @Classes:$class OBJECT_NIL $method
    fi
  fi
}

###
#
# wchdep_delete  (delete and remove our root depset from a method)
#
wchdep_delete() {
  [ "$DEBUG" = true ] && set -x

  class=$1
  method=$2

  d_name="`TMA_get_method_depset_name $class $method`"
  wdepset_delete $d_name
  if is_existing_dep_object $d_name; then
    errecho "Failed to delete and remove $d_name from $class $method!"
  else
    wchdep_rm $class $method
  fi
}

###
#
# wdepset_cmd
#
wdepset_cmd() {
  [ "$DEBUG" = true ] && set -x

  depset_name="$1"
  depset_args="$2"
  if [ ! "$depset_args" ]; then
    return 0
  fi
  if wlookup -r DependencyMgr ${depset_name}#${IRONAME} 1> /dev/null 2>& 1; then
    opt="-e"
    depset_name="@DependencyMgr:${depset_name}#${IRONAME}"
  else
    opt="-c"
  fi

  wdepset $opt $depset_name $depset_args 1> /dev/null
}
 
###
#
# wdepset_view
#
wdepset_view() {
  depset=$1
  if is_existing_dep_object "@DependencyMgr:${depset}#${IRONAME}"; then
    wdepset -v "@DependencyMgr:${depset}#${IRONAME}"
  fi
}

###
#
# wdepset_delete
#
wdepset_delete() {
  depset=$1
  if wlookup -r DependencyMgr "${depset}#${IRONAME}" > /dev/null 2>&1; then
    d_oid=`wlookup -r DependencyMgr "${depset}#${IRONAME}"`
    if is_depset_ours $d_oid; then
      cleanup_depset_children $d_oid
      if has_depset_children $d_oid; then
        errecho "Unable to remove $d_oid, still has children!"
      else
        wdepset -d @DependencyMgr:"${depset}#${IRONAME}"
      fi
    fi
  fi
}

###
#
# config_module_files
#
config_module_files() {
  [ "$DEBUG" = true ] && set -x
  is_TMA_supported || return 0
  type=$1  # one of TASK, MON, FP, or ALL
  paths="$2"
  args=""
  for p in $paths; do
    path_to_add="${TMA_MODULE_PRODPATH}/${p}"
    if is_subpath_in_lcf_bundle $path_to_add; then
      args="$args \
          -a generic ${path_to_add} +p %BIN%"
    fi
  done

  eval depset='"$'MODULE_X_DEPSET_${type}'"'
  eval classes='"$'MODULE_${type}_CLASS_NAMES'"'

  wdepset_cmd $depset "$args"
  for c in $classes; do
    class_root="`map_class_to_root $c $type`"
    methods="`TMA_get_module_class_methods $c $type`"
    TMA_attach_depset_to_roots $depset $class_root
    TMA_attach_root_to_class_methods $class_root $c "$methods"
  done
}

###
#
# eval_symlist
#
eval_symlist() {
  symlist="$*"
  result=""
  for i in $symlist; do
    eval sym='"$'${i}'"'
    result="$result $sym"
  done
  echo $result
}

###
#
# eval_symlist_basenames
#
eval_symlist_basenames() {
  symlist="$*"
  result=""
  for i in $symlist; do
    eval sym='"$'${i}'"'
    result="$result `basename $sym`"
  done
  echo $result
}

###
#
# get_interps_with_subpath
#
get_interps_with_subpath() {
  [ "$DEBUG" = true ] && set -x
  if [ ! "$1" ]; then
    return 0
  fi
  subpath_type=$1
  subpath_under_interp="$2"
  list=""
  ls -d ${INST_DIR}/lcf_bundle/${subpath_type}/* 1> /dev/null 2>& 1 || return 0

  dirs=`ls -d ${INST_DIR}/lcf_bundle/${subpath_type}/* 2> /dev/null`
  for i in $dirs; do
    if [ ! -d "$i" ]; then
      continue
    fi
    interp=`echo $i | awk -F/ '{print $NF}'`
    fullpath="${i}/${subpath_under_interp}"
    if ls $fullpath 1> /dev/null 2>&1; then
      if [ -n "`ls $fullpath 2> /dev/null`" ]; then
        list="$list $interp"
      fi
    fi
  done
  echo $list
}

###
#
# is_subpath_in_lcf_bundle
#
is_subpath_in_lcf_bundle() {
  [ "$DEBUG" = true ] && set -x
  subpath=$1
  fullpath="${INST_DIR}/lcf_bundle/${subpath}"
  if [ -f "${fullpath}" \
    -o -f "${fullpath}.exe" \
    -o -f "${fullpath}.nlm" \
    -o -d "${fullpath}" ]; then
    return 0
  else
    return 1
  fi
}

###
#
# is_tools_binary
#
is_tools_binary() {
  [ "$DEBUG" = true ] && set -x
  subpath=$1
  top_dir=`dirname $subpath`
  tool_name=`basename $subpath`
  if [ "$top_dir" != "." \
	-a "$top_dir" != tools \
	-a "$top_dir" != tools/bin ]; then
    return 1
  fi
  win_tool_path_1="${INST_DIR}/lcf_bundle/bin/w32-ix86/tools/${tool_name}"
  win_tool_path_2="${INST_DIR}/lcf_bundle/bin/w32-ix86/tools/${tool_name}.exe"
  unix_tool_path="${INST_DIR}/lcf_bundle/bin/solaris2/tools/bin/${tool_name}"

  for i in $win_tool_path_1 $win_tool_path_2 $unix_tool_path; do
    if [ -f "${i}" ]; then
      return 0
    fi
  done
  return 1
}

###
#
# is_unix_tools_interp
#
is_unix_tools_interp() {
  [ "$DEBUG" = true ] && set -x
  interp=$1
  for i in $UNIX_TOOLS_INTERPS; do
    if [ "$i" = "$interp" ]; then
      return 0
    fi
  done
  return 1
}

###
#
# is_win_tools_interp
#
is_win_tools_interp() {
  [ "$DEBUG" = true ] && set -x
  interp=$1
  for i in $WIN_TOOLS_INTERPS; do
    if [ "$i" = "$interp" ]; then
      return 0
    fi
  done
  return 1
}

###
#
# is_win_interp
#
is_win_interp() {
  [ "$DEBUG" = true ] && set -x
  interp=$1
  for i in $WIN_INTERPS; do
    if [ "$i" = "$interp" ]; then
      return 0
    fi
  done
  return 1
}

###
#
# is_nw_interp
#
is_nw_interp() {
  [ "$DEBUG" = true ] && set -x
  interp=$1
  for i in $NW_INTERPS; do
    if [ "$i" = "$interp" ]; then
      return 0
    fi
  done
  return 1
}

###
#
# is_unix_interp
#
is_unix_interp() {
  [ "$DEBUG" = true ] && set -x
  interp=$1
  for i in $UNIX_INTERPS; do
    if [ "$i" = "$interp" ]; then
      return 0
    fi
  done
  return 1
}

###
#
#      is_depset_ours
#
is_depset_ours() {
  [ "$DEBUG" = true ] && set -x
  depset_name="`TMA_get_depset_name $1`"
  for d in $ALL_DEPSETS; do
    if [ "$depset_name" = $d ]; then
      return 0
    fi
  done
  return 1
}

###
#
# has_depset_children
#
has_depset_children() {
  [ "$DEBUG" = true ] && set -x
  if wdepset -v $1 | grep "\#Depends::Mgr\#$" > /dev/null 2>&1; then
    return 0
  else
    return 1
  fi
}

###
#
# has_living_depset_children
#
has_living_depset_children() {
  [ "$DEBUG" = true ] && set -x
  d_oid=$1
  if has_depset_children $d_oid; then
    kids=`get_depset_children $d_oid`
    for k in $kids; do
      if is_existing_dep_object $k; then
        return 0
      fi
    done
  fi
  return 1
}

###
#
# get_depset_children
#
get_depset_children() {
  [ "$DEBUG" = true ] && set -x
  if has_depset_children $1; then
    wdepset -v $1 | grep "\#Depends::Mgr\#$"
  fi
}

###
#
# cleanup_depset_children
#
cleanup_depset_children() {
  [ "$DEBUG" = true ] && set -x
  d_oid=$1
  if has_depset_children $d_oid; then
    kids=`get_depset_children $d_oid`
    for k in $kids; do
      if is_existing_dep_object $k; then
        :
      else
        errecho "... deleting reference to invalid depset $k"
        wdepset -e $d_oid -r depset $k
      fi
    done
  fi
}

###
#
# has_non_depset_child
#
has_non_depset_child() {
  [ "$DEBUG" = true ] && set -x
  top="$1"
  non_depsets=""
  if wdepset -v @DependencyMgr:${top}#${IRONAME} 2> /dev/null \
       | grep -v "\#Depends::Mgr\#$" | grep -v "^depset:" | grep ":"; then
    non_depsets=`wdepset -v @DependencyMgr:${top}#${IRONAME} \
       | grep -v "\#Depends::Mgr\#$" | grep -v "^depset:" | grep ":"`
  fi
  if [ "$non_depsets" ]; then
    wdepset -v @DependencyMgr:${top}#${IRONAME} \
       | grep -v "\#Depends::Mgr\#$" | grep -v "^depset:"
    return 0
  else
    return 1
  fi
}

###
#
# delete_depsets
#
delete_depsets() {
  [ "$DEBUG" = true ] && set -x
  for i in $*; do
    if wlookup -r DependencyMgr ${i}#${IRONAME} > /dev/null 2>&1; then
      d_name="$i"
      d_oid=`wlookup -r DependencyMgr ${i}#${IRONAME}`
      cleanup_depset_children $d_oid
      remaining="`get_depset_children $d_oid`"
      if [ "`echo $remaining`" ]; then
        errecho "Could not delete $d_name; still has children $remaining!"
      else
        wdepset_delete $d_name
      fi
    fi
  done
}

###
#
# TMA_delete_all_depsets
#
TMA_delete_all_depsets() {
  [ "$DEBUG" = true ] && set -x
  # first the module-specific hierarchy ...
  delete_depsets $MODULE_X_DEPSETS 
  delete_depsets $MODULE_ROOT_DEPSETS

  # then the non-module hierarchy (common tools, etc).
  delete_depsets $TMB_COMMON_DEPSETS 
  delete_depsets $TMB_ROOT_DEPSETS
}

###
#
# TMA_get_method_depset_name
#
TMA_get_method_depset_name() {
  [ "$DEBUG" = true ] && set -x
  class=$1
  method=$2

  depset_name=""
  depset_oid=`wchdep_get ${class} ${method}`
  if [ "$depset_oid" ]; then
    depset_name=`TMA_get_depset_name $depset_oid`
  fi
  echo $depset_name
}

###
#
# TMA_get_depset_name
#
TMA_get_depset_name() {
  [ "$DEBUG" = true ] && set -x
  d_oid=$1
  d_name=""
  if is_existing_dep_object $d_oid; then
    d_name=`wlookup -ar DependencyMgr | grep "$d_oid$" | awk '{print $1}'`
  fi
  echo $d_name
}

###
#
# TMA_get_classes_basic_all
#
TMA_get_classes_basic_all() {
  [ "$DEBUG" = true ] && set -x
  eval_symlist "$BASIC_ALL_CLASSES"
}

###
#
# TMA_get_classes_basic_fp
#
TMA_get_classes_basic_fp() {
  [ "$DEBUG" = true ] && set -x
  eval_symlist "$BASIC_FP_CLASSES"
}

###
#
# TMA_get_classes_basic_mon
#
TMA_get_classes_basic_mon() {
  [ "$DEBUG" = true ] && set -x
  eval_symlist "$BASIC_MON_CLASSES"
}

###
#
# TMA_get_classes_basic_task
#
TMA_get_classes_basic_task() {
  [ "$DEBUG" = true ] && set -x
  eval_symlist "$BASIC_TASK_CLASSES"
}

###
#
# TMA_get_classes_module_all
#
TMA_get_classes_module_all() {
  [ "$DEBUG" = true ] && set -x
  eval_symlist "$MODULE_ALL_CLASSES"
}

###
#
# TMA_get_methods_for_class
#
TMA_get_methods_for_class() {
  [ "$DEBUG" = true ] && set -x
  match_class=$1
  method_list=""
  for i in $BASIC_ALL_CLASSES; do
    eval this_class='"$'${i}'"'
    if [ "$this_class" = "$match_class" ]; then
      eval method_list='"$'${i}_METHODS'"'
      break
    fi
  done
  if [ ! "$method_list" ]; then
    for i in TASK; do
      method_list="$method_list `TMA_get_module_class_methods $match_class $i`"
    done
  fi
  echo $method_list
}

###
#
# TMA_attach_depset_to_class_method
#
TMA_attach_depset_to_class_method() {

  [ "$DEBUG" = true ] && set -x
  is_TMA_supported || return 0
  depset=$1
  class=$2
  method=$3

  # if we got here with no depset, just return
  wlookup -r DependencyMgr ${depset}#${IRONAME} > /dev/null 2>&1 || return 0

  if wlookup -r Classes ${class} > /dev/null 2>&1; then
    :
  else
    errecho "could not attach depset $depset to $method method (class $class does not exist)"
    return 0
  fi

  existing_dep_name="`TMA_get_method_depset_name $class $method`"
  if [ "$existing_dep_name" = "$depset" ]; then
    # we're already attached ... don't want to attach to ourself!
    return 0
  fi
  if [ "$existing_dep_name" ]; then
    errecho "### Replacing $existing_dep_name with $depset on class $class, method $method ..."
  else
    errecho "### Creating $depset on class $class, method $method ..."
  fi
  if wlookup -r Classes ${class} > /dev/null 2>&1; then
    wchdep @Classes:${class} @DependencyMgr:${depset}#${IRONAME} $method 1> /dev/null
  fi
  errecho "### Done setting $depset on class $class, method $method"
}

###
#
# TMA_attach_root_depsets_to_methods_basic
#
TMA_attach_root_depsets_to_methods_basic() {
  [ "$DEBUG" = true ] && set -x

  if [ "`TMA_get_root_depsets_for_classes_basic`" ]; then
    :
  else
    # no dependencies were created! just skip it
    return 0
  fi

  classes="`TMA_get_classes_basic_all`"
  for class in $classes; do
    root_depset="`map_class_to_root $class`"
    methods=`TMA_get_methods_for_class $class`
    for i in $methods; do
      previous_depset="`TMA_get_method_depset_name $class $i`"
      if [ ! "$previous_depset" ]; then
           ### no previous depset ... just attach root straight-on to method!
        TMA_attach_depset_to_class_method $root_depset $class $i || return 1
      elif [ "$previous_depset" = "$root_depset" ]; then
        :  ### our root is already there, don't need to do anything else!
      else

        #
        # reparent the previous depset (not ours) to be a sub-dep of our root
        # 

          #    first attach previous onto our root ...
          errecho
          errecho "### Attaching $previous_depset to $root_depset ..."
          TMA_attach_depset_to_roots $previous_depset $root_depset || return 1
          errecho "### ... DONE attaching $previous_depset to $root_depset!"
          errecho

          #    NOW put our root on the method, replacing theirs (WITH CARE)!
          if TMA_attach_depset_to_class_method $root_depset $class $i; then
             : ### swap was okay!
          else
               ### try to recover ... put the previous one back on if possible
            TMA_attach_depset_to_class_method $previous_depset $class $i
            return 1
        fi
      fi
    done
  done

  # let's see what we got ...
  TMA_view_root_depsets
}

###
#
# TMA_attach_root_to_class_methods
#
TMA_attach_root_to_class_methods() {
  [ "$DEBUG" = true ] && set -x

  root_depset=$1
  class=$2
  methods="$3"

  if is_existing_dep_object $root_depset; then
    :
  else
    # no dependencies were created! just skip it
    return 0
  fi

  for i in $methods; do
      previous_depset="`TMA_get_method_depset_name $class $i`"
      if [ ! "$previous_depset" ]; then
           ### no previous depset ... just attach root straight-on to method!
        TMA_attach_depset_to_class_method $root_depset $class $i || return 1
      elif [ "$previous_depset" = "$root_depset" ]; then
        :  ### our root is already there, don't need to do anything else!
      else

        #
        # reparent the previous depset (not ours) to be a sub-dep of our root
        #

          #    first attach previous onto our root ...
          errecho
          errecho "### Attaching $previous_depset to $root_depset ..."
          TMA_attach_depset_to_roots $previous_depset $root_depset || return 1
          errecho "### ... DONE attaching $previous_depset to $root_depset!"
          errecho

          #    NOW put our root on the method, replacing theirs (WITH CARE)!
          if TMA_attach_depset_to_class_method $root_depset $class $i; then
             : ### swap was okay!
          else
               ### try to recover ... put the previous one back on if possible
            TMA_attach_depset_to_class_method $previous_depset $class $i
            return 1
        fi
      fi
  done

  # let's see what we got ...
  TMA_view_root_depsets $root_depset $previous_depset
}

###
#
# TMA_attach_depset_to_roots
#
TMA_attach_depset_to_roots() {
  [ "$DEBUG" = true ] && set -x

  depset=$1
  parent_depsets="$2"
  if [ ! "$parent_depsets" ]; then
    parent_depsets="`get_root_depsets_common`"
  fi

  # if we got here with no real depset, just return
  wlookup -r DependencyMgr ${depset}#${IRONAME} 2> /dev/null || return 0

  for root_parent in $parent_depsets; do
    if [ "$depset" = "$root_parent" ]; then
      # we ARE root ... don't want to attach to ourself!
      continue
    fi

    # HERE, check to see if $depset is already in the $root_parent
    #    hierarchy, and if so skip the next wdepset ...

    # now create or modify the root_parent depset with the depset to attach
    args="-a depset @DependencyMgr:${depset}#${IRONAME}"
    wdepset_cmd $root_parent "$args" 1> /dev/null

    # and just in case we haven't attached already ...
    #### whoa!!! an endless loop could happen here
    #### TMA_attach_root_depsets_to_methods_basic || return 1
 
    # take a look at the result
    #wdepset -v @DependencyMgr:$root_parent#${IRONAME}
  done
}

###
#
# TMA_get_root_depsets_for_classes_basic
#
TMA_get_root_depsets_for_classes_basic() {
  [ "$DEBUG" = true ] && set -x
  classes="$*"
  depsets=
  if [ ! "$classes" ]; then
    classes="$BASIC_ALL_CLASS_NAMES"
  fi    
  for class in $classes; do
    depset_name=`map_class_to_root $class`
    if wlookup -r DependencyMgr ${depset_name}#${IRONAME} 1> /dev/null 2>&1; then
      depsets="$depsets `wlookup -r DependencyMgr ${depset_name}#${IRONAME}`"
    fi
  done
  echo "$depsets"
}

###
#
# TMA_get_tools_depset
#
TMA_get_tools_depset() {
  wlookup -r DependencyMgr "${TMB_TOOLS_DEPSET}#${IRONAME}" 2> /dev/null
}

###
#
# TMA_get_module_depset_general
#
TMA_get_module_depset_general() {
  wlookup -r DependencyMgr "${MODULE_X_DEPSET_ALL}#${IRONAME}" 2> /dev/null
}

###
#
# TMA_get_module_depset_msgcat
#
TMA_get_module_depset_msgcat() {
  wlookup -r DependencyMgr "${MODULE_X_DEPSET_MSGCAT}#${IRONAME}" 2> /dev/null
}

###
#
# TMA_get_module_depset_fp
#
TMA_get_module_depset_fp() {
  wlookup -r DependencyMgr "${MODULE_X_DEPSET_FP}#${IRONAME}" 2> /dev/null
}

###
#
# TMA_get_module_depset_mon
#
TMA_get_module_depset_mon() {
  wlookup -r DependencyMgr "${MODULE_X_DEPSET_MON}#${IRONAME}" 2> /dev/null
}

###
#
# TMA_get_module_depset_task
#
TMA_get_module_depset_task() {
  wlookup -r DependencyMgr "${MODULE_X_DEPSET_TASK}#${IRONAME}" 2> /dev/null
}

###
#
# TMA_config_module_depset_msgcat
#
TMA_config_module_depset_msgcat() {
  [ "$DEBUG" = true ] && set -x
  is_TMA_supported || return 0
  depset="$MODULE_X_DEPSET_MSGCAT"
  paths="$*"
  args=""
  msg_cat_prefix="msg_cat"
  msg_cat_extension="cat"
  for p in $paths; do
    # massage a prepending msg_cat off there if they gave us one
    p=`echo $p | sed -e 's,^$msg_cat_prefix/,,'`

    # see if it's really a cat file ...
    if echo $p | grep "\.$msg_cat_extension$" 1> /dev/null; then
      num_fields=`echo $p | awk -F/ '{print NF}'`
      if [ $num_fields -eq 1 ]; then
        # we only got a file name, so locate it under msg_cat/C
        path_to_add=${TMA_MSGCAT_PATH}/C/${p}
      else
        # we got what is likely to be <lang>/<cat-file>, 
        # so just prepend the msg_cat to it
        path_to_add="${TMA_MSGCAT_PATH}/${p}"
      fi
      if is_subpath_in_lcf_bundle $path_to_add; then
        args="$args \
            -a generic ${path_to_add} +p %MSGCAT%"
      fi
    fi
  done
  wdepset_cmd $depset "$args"

  for c in $MODULE_ALL_CLASSES; do
    for type in TASK; do
      class_root="`map_class_to_root $c $type`"
      methods="`TMA_get_module_class_methods $c $type`"
      TMA_attach_depset_to_roots $depset $class_root
      TMA_attach_root_to_class_methods $class_root $c "$methods"
    done
  done
}

###
#
# TMA_config_module_depset_fp
#
TMA_config_module_depset_fp() {
  config_module_files FP "$*"
}

###
#
# TMA_config_module_depset_mon
#
TMA_config_module_depset_mon() {
  config_module_files MON "$*"
}

###
#
# TMA_config_module_depset_task
#
TMA_config_module_depset_task() {
  config_module_files TASK "$*"
}

###
#
# TMA_config_module_depset_general
#
TMA_config_module_depset_general() {
  config_module_files ALL "$*"
}

###
#
# TMA_config_common_depset_bin
#	Append to the lcf_bundle dependency and that is associated with 
#	all modules and all methods
#
TMA_config_common_depset_bin() {
  [ "$DEBUG" = true ] && set -x
  is_TMA_supported || return 0
  paths=$*
  args=""
  tool_list=""
  for p in $paths; do
    # we'll sort out the tools ... and do them later
    if is_tools_binary $p; then
      tool_name=`basename $p`
      tool_list="$tool_list $tool_name"
    else
      #  this is apparently not working ... 
      #  the file name is not resolved at run time!
      #args="$args \
      #        -a bin ${p} +a +p %BIN%"

      for interp in $ALL_INTERPS; do
        exe=""
        is_win_interp $interp && exe=".exe"
        is_nw_interp $interp && exe=".nlm"
        path_to_add="bin/${interp}/${p}${exe}"
        if is_subpath_in_lcf_bundle bin/${interp}; then   ##can't check whole path_to_add ...

            ## args="$args \
            ##    -a $interp ${path_to_add} +a +p %BIN%"

            ## put it in %TOOLS% instead of %BIN%, somehow %BIN% is not in the endpoint path!
            args="$args \
                -a $interp ${path_to_add} +a +p %TOOLS%"
        fi
      done
    fi
  done
  wdepset_cmd $TMB_BIN_DEPSET "$args"
  TMA_attach_depset_to_roots $TMB_BIN_DEPSET "`get_root_depsets_common`"

  # now do the tools!
  TMA_config_common_depset_tools $tool_list
}

###
#
# TMA_config_common_depset_lib
#	Append to the lcf_bundle dependency that is associated with 
#	all modules and all methods
#
TMA_config_common_depset_lib() {
  [ "$DEBUG" = true ] && set -x
  is_TMA_supported || return 0
  paths=$*
  args=""
  for p in $paths; do
      args="$args \
              -a lib ${p} +a +p %LIB%"

    #for interp in $ALL_INTERPS; do
    #  path_to_add="lib/${interp}/${p}"
    #  if is_subpath_in_lcf_bundle $path_to_add; then
    #    args="$args \
    #        -a $interp ${path_to_add} +a +p %LIB%"
    #  fi
    #done
  done
  wdepset_cmd $TMB_LIB_DEPSET "$args"
  TMA_attach_depset_to_roots $TMB_LIB_DEPSET "`get_root_depsets_common`"
  TMA_attach_root_depsets_to_methods_basic
}

###
#
# TMA_config_common_depset_tools
#	Append to the common dependency that is associated with all methods
#
TMA_config_common_depset_tools() {
  [ "$DEBUG" = true ] && set -x
  is_TMA_supported || return 0
  files=$*
  args=""
  for f in $files; do
    tool_name=`basename $f | sed s/.exe\$//`
    for interp in $ALL_TOOLS_INTERPS; do
      if is_win_tools_interp $interp; then
        # take .dll and .cmd files with their given extension, otherwise construct the .exe
        if echo "$tool_name" | grep "\.dll$" > /dev/null; then
          path_to_add="bin/${interp}/tools/${tool_name}"
        elif echo "$tool_name" | grep "\.cmd$" > /dev/null; then
          path_to_add="bin/${interp}/tools/${tool_name}"
        else
          path_to_add="bin/${interp}/tools/${tool_name}.exe"
        fi
      else
        path_to_add="bin/${interp}/tools/bin/${tool_name}"
      fi
      if is_subpath_in_lcf_bundle $path_to_add; then
        args="$args \
	    -a $interp ${path_to_add} +a +p %TOOLS%"
      fi
    done
  done

  ## This is a band-aid that will create an os2 version of bash.exe
  ## on the TMR lcf_bundle by copying sh.exe
  sh_path="bin/os2-ix86/tools/sh.exe"
  bash_path="bin/os2-ix86/tools/bash.exe"
  if is_subpath_in_lcf_bundle $bash_path; then
    :  # it's already in the list!
  else
    if cp "${INST_DIR}/lcf_bundle/${sh_path}" "${INST_DIR}/lcf_bundle/${bash_path}"; then
      args="$args \
          -a os2-ix86 ${bash_path} +a +p %TOOLS%"
    fi
  fi

  wdepset_cmd $TMB_TOOLS_DEPSET "$args"
  TMA_attach_depset_to_roots $TMB_TOOLS_DEPSET "`get_root_depsets_common`"
  TMA_attach_root_depsets_to_methods_basic
}

###
#
# TMA_config_common_depset_perlscripts
#
TMA_config_common_depset_perlscripts() {
  [ "$DEBUG" = true ] && set -x
  is_TMA_supported || return 0
  scripts=$*
  path_list=""
  for i in $scripts; do
    path_list="$path_list tools/lib/perl/${i}"
  done
  TMA_config_common_depset_bin $path_list
}

###
#
# TMA_config_perl
#
TMA_config_perl() {
  [ "$DEBUG" = true ] && set -x
  is_TMA_supported || return 0
  tool_name=perl
  for interp in $ALL_INTERPS; do
    if is_unix_tools_interp $interp; then
      path_to_add="bin/${interp}/tools/bin/${tool_name}"
      args="-a $interp ${path_to_add} +p /etc/Tivoli/bin +a"
      wdepset_cmd $TMB_ETC_DEPSET "$args"
      TMA_attach_depset_to_roots $TMB_ETC_DEPSET "`get_root_depsets_common`"
      TMA_attach_root_depsets_to_methods_basic
    fi
  done
}

###
#
# TMA_view_root_depsets
#
TMA_view_root_depsets() {
  view_depsets="$*"
  if [ ! "$view_depsets" ]; then
    view_depsets="`get_root_depsets_common` `get_root_depsets_module`"
  fi
  for i in $view_depsets; do
    wdepset_view "$i"
  done
}

###
#
# TMA_view_tools_depset
#
TMA_view_tools_depset() {
  wdepset_view "$TMB_TOOLS_DEPSET"
}

###
#
# TMA_view_module_depset_general
#
TMA_view_module_depset_general() {
  wdepset_view "$MODULE_X_DEPSET_ALL"
}

###
#
# TMA_view_module_depset_msgcat
#
TMA_view_module_depset_msgcat() {
  wdepset_view "$MODULE_X_DEPSET_MSGCAT"
}

###
#
# TMA_view_module_depset_fp
#
TMA_view_module_depset_fp() {
  wdepset_view "$MODULE_X_DEPSET_FP"
}

###
#
# TMA_view_module_depset_mon
#
TMA_view_module_depset_mon() {
  wdepset_view "$MODULE_X_DEPSET_MON"
}

###
#
# TMA_view_module_depset_task
#
TMA_view_module_depset_task() {
  wdepset_view "$MODULE_X_DEPSET_TASK"
}

###
#
# TMA_view_depset_hierarchy
#
TMA_view_depset_hierarchy() {
  [ "$DEBUG" = true ] && set -x
  classes="`TMA_get_classes_basic_all` `TMA_get_classes_module_all`"
  nested_list=""
  echo "==========================================================="
  echo "================= Depset Hierarchy ========================"
  for class in $classes; do

    methods=`TMA_get_methods_for_class $class`
    for i in $methods; do
      echo "==========================================================="
      echo "CLASS: $class"
      echo "METHOD: $i"
      existing_depset="`TMA_get_method_depset_name $class $i`"
      if [ "$existing_depset" ]; then
        echo "DEPENDENCY SET:  $existing_depset"
        echo "  ---"
        echo "  wdepset -v @DependencyMgr:${existing_depset}#${IRONAME}"
        echo "  ---"
        wdepset -v @DependencyMgr:${existing_depset}#${IRONAME}
        echo
        if is_existing_dep_object @DependencyMgr:${existing_depset}#${IRONAME}; then
          raw_nesteds=`get_depset_children @DependencyMgr:${existing_depset}#${IRONAME}`
        fi
        for j in $raw_nesteds; do
          addit=true
          for k in $nested_list; do
            if [ "$j" = "$k" ]; then
              addit=false
            fi
          done
          if $addit; then
            nested_list="$nested_list $j"
          fi
        done
      else
        echo "DEPENDENCY SET:  (none)"
      fi
    done
  done
  echo "==========================================================="
  echo
  for i in $nested_list; do
    echo "==========================================================="
    name=`TMA_get_depset_name $i`
    echo "DEPENDENCY SET: $name"
    echo "DEPSET_OID: $i"
    echo "  ---"
    echo "  wdepset -v @DependencyMgr:${name}#${IRONAME}"
    echo "  ---"
    wdepset -v @DependencyMgr:${name}#${IRONAME}
    echo "==========================================================="
    echo
  done
}

###
#
# TMA_delete_root_depsets_module
#
TMA_delete_root_depsets_module() {
  [ "$DEBUG" = true ] && set -x
  delete_depsets $MODULE_X_DEPSETS 
  delete_depsets $MODULE_ROOT_DEPSETS
}

###
#
# TMA_delete_root_depsets_common
#
TMA_delete_root_depsets_common() {
  delete_depsets $TMB_COMMON_DEPSETS 
  delete_depsets $TMB_ROOT_DEPSETS
}

###
#
# TMA_delete_tools_depset
#
TMA_delete_tools_depset() {
  wdepset_delete "$TMB_TOOLS_DEPSET"
}

###
#
# TMA_delete_module_depset_general
#
TMA_delete_module_depset_general() {
  wdepset_delete "$MODULE_X_DEPSET_ALL"
}

###
#
# TMA_delete_module_depset_msgcat
#
TMA_delete_module_depset_msgcat() {
  wdepset_delete "$MODULE_X_DEPSET_MSGCAT"
}

###
#
# TMA_delete_module_depset_fp
#
TMA_delete_module_depset_fp() {
  wdepset_delete "$MODULE_X_DEPSET_FP"
}

###
#
# TMA_delete_module_depset_mon
#
TMA_delete_module_depset_mon() {
  wdepset_delete "$MODULE_X_DEPSET_MON"
}

###
#
# TMA_delete_module_depset_task
#
TMA_delete_module_depset_task() {
  wdepset_delete "$MODULE_X_DEPSET_TASK"
}

###
#
# TMA_config_full
#
TMA_config_full() {
  [ "$DEBUG" = true ] && set -x
  is_TMA_supported || return 0
  TMA_config_common_depset_tools $TMA_UNIX_TOOLS $TMA_WIN_TOOLS
  TMA_config_common_depset_lib $TMA_LIBS
  TMA_config_common_depset_bin $TMA_BINS
  TMA_config_common_depset_perlscripts $TMA_PERL_SCRIPTS
  TMA_config_module_depset_task $TMA_PROD_FILES_TASK
  # TMA_config_module_depset_mon $TMA_PROD_FILES_MON
  # TMA_config_module_depset_fp $TMA_PROD_FILES_FP
  TMA_config_module_depset_general $TMA_PROD_FILES
  ##TMA_config_module_depset_msgcat $MSGCATS
  TMA_config_perl
}

###
#
# TMA_sync_gateways
#
TMA_sync_gateways() {
  [ "$DEBUG" = true ] && set -x
  is_TMA_supported || return 0
  for gw in `wlookup -L -r Gateway -a`; do
    wgateway $gw dbcheck
  done
}

###
#
# TMA_set_module_classes
#
TMA_set_module_classes() {
  [ "$DEBUG" = true ] && set -x
  MODULE_CLASSES=""
  N=0
  for class in $*; do
    N=`expr $N + 1`
    eval module_class_id='"MODULE_CLASS_$N"'
    export $module_class_id
    eval $module_class_id='"$class"'
    MODULE_CLASSES="$MODULE_CLASSES $module_class_id"
  done
  export MODULE_CLASSES
}

###
#
# TMA_set_module_class_methods_task
#
TMA_set_module_class_methods_task() {
  TMA_set_module_class_methods TASK $1 "$2"
}

###
#
# TMA_set_module_class_methods_mon
#
TMA_set_module_class_methods_mon() {
  TMA_set_module_class_methods MON $1 "$2"
}

###
#
# TMA_set_module_class_methods_fp
#
TMA_set_module_class_methods_fp() {
  TMA_set_module_class_methods FP $1 "$2"
}

###
#
# TMA_get_module_class_methods
#
TMA_get_module_class_methods() {
  [ "$DEBUG" = true ] && set -x
  class=$1
  type=$2
  class_id=`lookup_id_in_list $class "$MODULE_CLASSES"`
  eval method_id_list='"$'${type}_${class_id}_METHODS'"'
  eval method_ids='"${method_id_list}"'
  eval_symlist "$method_ids"
}

###
#
# TMA_set_module_class_methods
#
TMA_set_module_class_methods() {
  [ "$DEBUG" = true ] && set -x
  type=$1
  class=$2
  methods="$3"
  M=0
  class_id=`lookup_id_in_list $class "$MODULE_CLASSES"`
  for meth in $methods; do
    M=`expr $M + 1`
    eval method_id='"${type}_${class_id}_METHOD_${M}"'
    export $method_id
    eval $method_id='"$meth"'
    eval method_id_list='"${type}_${class_id}_METHODS"'
    add_id_to_list $method_id $method_id_list
  done
  export $method_id_list

  eval class_list='"MODULE_${type}_CLASSES"'
  add_id_to_list $class_id $class_list
  add_id_to_list $class_id MODULE_ALL_CLASSES
}

####################################################################
#
# Initialize variables ...
#
set -e

DEBUG=$DEBUG
if [ "$DEBUG" = true ]; then set -x; else set +x; fi

MODULE_NAME="$SHORT_PRODNAME"
if [ ! "$MODULE_NAME" ]; then
  MODULE_NAME="UnknownModuleName"
fi

MODULE_TYPE="PLUS"

TMB_ROOT_DEPSET="TMB_ROOT"
TMB_ROOT_DEPSET_TASK="TMB_ROOT_task"
TMB_TOOLS_DEPSET="TMB_tools"
TMB_BIN_DEPSET="TMB_bin"
TMB_LIB_DEPSET="TMB_lib"
TMB_ETC_DEPSET="TMB_etc"

BASIC_TASK_CLASSES="BASIC_TASK_CLASS_1"
BASIC_TASK_CLASS_1="TaskEndpoint"
BASIC_TASK_CLASS_1_METHODS="run_task batch_run_task"

BASIC_ALL_CLASSES="$BASIC_TASK_CLASSES"

BASIC_TASK_CLASS_NAMES="`TMA_get_classes_basic_task`"
BASIC_ALL_CLASS_NAMES="`TMA_get_classes_basic_all`"

#
#  For module-specific dependencies, assume there is a single
#  class that has methods we will use for each of task, mon, and fp
#
if [ "$MODULE_TYPE" = PLUS ]; then
  TMA_MODULE_CLASS_NAME=SolutionsLink
  TMA_MODULE_TASK_METHODS="faux_job_launch faux_task_launch"
 else
  TMA_MODULE_CLASS_NAME=
  TMA_MODULE_TASK_METHODS=""
fi

if [ "$TMA_MODULE_CLASS_NAME" ]; then
  TMA_set_module_classes $TMA_MODULE_CLASS_NAME 
  TMA_set_module_class_methods_task $TMA_MODULE_CLASS_NAME "$TMA_MODULE_TASK_METHODS"

  MODULE_TASK_CLASS_NAMES="`eval_symlist $MODULE_TASK_CLASSES`"
  MODULE_ALL_CLASS_NAMES="`eval_symlist $MODULE_ALL_CLASSES`"

  MODULE_ROOT_QUALIFIER=PLUS_ROOT
  ##MODULE_ROOT_QUALIFIER=${MODULE_ROOT_QUALIFIER}_${TMA_MODULE_CLASS_NAME}_ROOT

  MODULE_ROOT_DEPSET_ALL="${MODULE_ROOT_QUALIFIER}"
  MODULE_ROOT_DEPSET_TASK="${MODULE_ROOT_QUALIFIER}_task"

  MODULE_X_DEPSET_ALL="${MODULE_TYPE}_${MODULE_NAME}"
  MODULE_X_DEPSET_TASK="${MODULE_TYPE}_${MODULE_NAME}_task"
  MODULE_X_DEPSET_MSGCAT="${MODULE_TYPE}_${MODULE_NAME}_msgcat"
fi

####### The depset hierarchies (ROOT and MODULE):
# $TMB_ROOT_DEPSET
#
### TMB_ROOT* depsets are attached to basic "raw" app methods
#
TMB_ROOT_DEPSETS="
  $TMB_ROOT_DEPSET
  $TMB_ROOT_DEPSET_TASK
"

### TMB_COMMON* depsets are attached to TMB_ROOT*, as are other 
#   (unknown) depsets that were previously attached to the "raw" 
#   app methods
#
TMB_COMMON_DEPSETS="
  $TMB_TOOLS_DEPSET
  $TMB_BIN_DEPSET
  $TMB_LIB_DEPSET
  $TMB_ETC_DEPSET
"

### MODULE_ROOT* depsets are attached to methods on a class that
#   is specific to a particular module type (e.g., on methods of
#   the SolutionsLink class for plus modules)
#
MODULE_ROOT_DEPSETS="
  $MODULE_ROOT_DEPSET_ALL
  $MODULE_ROOT_DEPSET_TASK
"

### MODULE_X* depsets are attached to the MODULE_ROOT*
#   depsets, as are depsets for other modules of the same type
#
MODULE_X_DEPSETS="
  $MODULE_X_DEPSET_ALL
  $MODULE_X_DEPSET_TASK
  $MODULE_X_DEPSET_MSGCAT
"
### ALL_DEPSETS groups 'em all together for convenience
#
ALL_DEPSETS="
  $TMB_ROOT_DEPSETS
  $TMB_COMMON_DEPSETS
  $MODULE_ROOT_DEPSETS
  $MODULE_X_DEPSETS
"

OSERV=`objcall 0.0.0 get_oserv`
export OSERV
INST_DIR=`objcall $OSERV query install_dir`

#
#       Get the region name to avoid collisions
#
IRO=`wlookup InterRegion`
IRONAME=`idlattr -t -g $IRO name string`
IRONAME=`eval echo $IRONAME`

OSERV_CHECK="wls"

UNIX_TOOLS_INTERPS="`get_interps_with_subpath bin tools/bin/perl`"
WIN_TOOLS_INTERPS="`get_interps_with_subpath bin tools/perl.exe`"
ALL_TOOLS_INTERPS="$UNIX_TOOLS_INTERPS $WIN_TOOLS_INTERPS"
ALL_PERL_INTERPS="`get_interps_with_subpath bin tools/lib/perl`"
WIN_INTERPS="`get_interps_with_subpath bin mrt/lcfd.exe`"
NW_INTERPS="`get_interps_with_subpath bin mrt/lcfd.nlm`"
UNIX_INTERPS="`get_interps_with_subpath bin mrt/lcfd`"
ALL_INTERPS="`get_interps_with_subpath bin`"

DEFAULT_WIN_TOOLS="
  basename
  bash
  chmod
  cut
  expr
  find
  mkdir
  mv
  perl
  sed
  sh
  tar
  awk
  cat
  cp
  cpp
  date
  diff
  dirname
  grep
  ls
  make
  rm
  sleep
  sort
  touch
  tr
  wc
  whoami
  wrunuiep
  uname
  win32gnu.dll
"
if wlsinst -p | grep -i "TME 10 Framework 3.7" > /dev/null; then
    DEFAULT_WIN_TOOLS="DEFAULT_WIN_TOOLS cygwin1.dll"
elif wlsinst -p | grep -i "Tivoli Management Framework 4.1" > /dev/null; then
    DEFAULT_WIN_TOOLS="DEFAULT_WIN_TOOLS cygwin1.dll"
fi


DEFAULT_UNIX_TOOLS="
  perl
"

if [ "$MODULE_TYPE" = GEM ]; then
  DEFAULT_BINS="
    bin/wpostemsg
  "
else
  DEFAULT_BINS="
  "
fi

# The following are candidates to values for DEFAULT_PROD_FILES deps ...
# but we don't assume anything!

#
#       Baroc files
#
TEC_BAROC_FILE_1="$PRODDIR/ins_evt.baroc"
TEC_BAROC_FILES="TEC_BAROC_FILE_1"

#
#       FMT files
#
TEC_FMT_FILES=""

#
#       Rules files
#
TEC_RULE_FILE_1="$PRODDIR/ins_evt.rls"
TEC_RULE_FILE_2="$PRODDIR/ins_start.rls"
TEC_RULE_FILES="TEC_RULE_FILE_1 TEC_RULE_FILE_2"

PRODINFO_FILE="${MODULE_TYPE}product-info.sh"
SCRIPTS="check_tme_server.sh PLUSconfig-TMA-utils.sh plus_about.sh cim_exec_job.sh _INI_File_Entry.sh PLUSconfig.sh launch_sans.sh open_tt.sh PLUSInstallAfter_MN.sh browser_location.sh OS_Version.sh config_snmp.sh PLUSinstall_after.sh launch_im7.sh web_agent.sh launch_iLo.sh config_tec.sh check_tec_server.sh PLUSproduct-info.sh verify_installed.sh run_collect.sh "
OTHER_FILES="ins_evt.oid user_predicates ins_start.rls ins_correl ins_evt.rls StandardPlus.xpm cim_admin_collection.d ins_evt.cds ins_evt.baroc tecad_snmp.baroc ins_correl.wic Standard.xpm PlusIconStdXE.xpm PlusIconStd.xpm PlusIconStdSans.xpm collect.exe"
JAR_FILES=""
MCSL_FILES=""
TLL_FILES="HPQ_Insight_Hidden_Tasks__V4_1.tll HPQ_Insight_Administrative_Tasks__V4_1.tll HPQ_Insight_TEC_Configuration__V4_1.tll HPQ_Insight_SNMP_Adapter_Configuration__V4_1.tll HPQ_Insight_Plus_Configuration__V4_1.tll "
DSL_FILES="web_agent_sh.d HP_Storage_Management_Appliance_Agents_icon.d Plus_Configuration_icon.d config_tec_sh.d About_Insight_Tivoli_Module_icon.d HP_Systems_Insight_Manager_icon.d launch_im7_sh.d launch_sans_sh.d config_snmp_sh.d Setup_TEC_Event_Server_icon.d Configure_SNMP_Adapter_icon.d IM2_icon.d browser_location_sh.d HP_System_Management_Homepage_icon.d Set_Internet_Browser_Location_icon.d run_collect_sh.d Initiate_Inventory_Collection_icon.d launch_iLo_sh.d HP_Remote_Server_Management_icon.d"
MSGCATS="IM2_config_snmp_sh_msg.cat IM2_Plus_Configuration_icon_msg.cat IM2_HP_Storage_Management_Appliance_Agents_icon_msg.cat IM2_HP_System_Management_Homepage_icon_msg.cat IM2_launch_im7_sh_msg.cat IM2_browser_location_sh_msg.cat IM2_Configure_SNMP_Adapter_icon_msg.cat IM2_About_Insight_Tivoli_Module_icon_msg.cat IM2_Set_Internet_Browser_Location_icon_msg.cat IM2Cat__0_.cat IM2_web_agent_sh_msg.cat IM2_launch_sans_sh_msg.cat IM2_Setup_TEC_Event_Server_icon_msg.cat IM2_config_tec_sh_msg.cat IM2_HP_Systems_Insight_Manager_icon_msg.cat IM2_run_collect_sh_msg.cat IM2_Initiate_Inventory_Collection_icon_msg.cat IM2_launch_iLo_sh_msg.cat IM2_HP_Remote_Server_Management_icon_msg.cat"
CDF_FILES=" "
FP_DEFS=""
PIX_MAPS=""
BAROC_FILES=""
RULE_FILES=""
FMT_FILES=""

ALL_BAROC_FILES="$BAROC_FILES `eval_symlist_basenames $TEC_BAROC_FILES`"
ALL_RULE_FILES="$RULE_FILES `eval_symlist_basenames $TEC_RULE_FILES`"
ALL_FMT_FILES="$FMT_FILES `eval_symlist_basenames $TEC_FMT_FILES`"

TASK_PROD_FILES="`echo $ALL_BAROC_FILES $ALL_RULE_FILES $ALL_FMT_FILES`"

TMA_MODULE_PRODPATH=../generic_unix/TME/PLUS/IM2/
TMA_MSGCAT_PATH=../../msg_cat

# Let tk user skip configuring tools if they know they don't need to ...
TMA_ALL_TOOLS="true"
if [ "$TMA_ALL_TOOLS" != true ]; then
  DEFAULT_WIN_TOOLS=""
  DEFAULT_UNIX_TOOLS=""
fi

TMA_WIN_TOOLS="$DEFAULT_WIN_TOOLS "
TMA_UNIX_TOOLS="$DEFAULT_UNIX_TOOLS "
TMA_PERL_SCRIPTS="$DEFAULT_PERL_SCRIPTS "
TMA_LIBS="$DEFAULT_LIBS "
TMA_BINS="$DEFAULT_BINS "
TMA_PROD_FILES="$DEFAULT_PROD_FILES "
TMA_PROD_FILES_TASK="$TASK_PROD_FILES "

if [ "`echo $TMA_PROD_FILES`" ]; then
  TMA_PROD_FILES="$TMA_PROD_FILES $PRODINFO_FILE"
fi
if [ "`echo $TMA_PROD_FILES_TASK`" ]; then
  TMA_PROD_FILES_TASK="$TMA_PROD_FILES_TASK $PRODINFO_FILE"
fi
