#!/bin/sh
# $Id: $
#
# Component:	TivoliPlus (PLUS instruct utilities)
#
# $Source: $
# $Revision: $
#
# Description:  Common support functions for instructing TivoliPlus modules.
#		This script should be "sourced" at the top of the TivoliPlus
#		module initialization scripts (for ALI/server and clients).
#
# External Variable References / Initialization:
#
# Author:	Annelle Harrison
#		Tivoli Systems, Inc.
# Modifications:
#
# (C) COPYRIGHT Tivoli Systems, Inc. 1996-2000
# Unpublished Work
# All Rights Reserved
# Licensed Material - Property of Tivoli Systems, Inc.
#

######################################################################
#
# In general, these global variables pertain to the current module
#
init_module_globals() 
{
        [ "$DEBUG" = true ] && set -x

	NULL="__NULL__"; export NULL

	#
	# The following are managed by some functions defined herein
	#
	TIV_LIBRARY_OBJ=`GetTivLibrary`; export TIV_LIBRARY_OBJ
	COLLECTION_GUI_OBJ=`GetCollectionGUIObj`; export COLLECTION_GUI_OBJ
	if LinkClassExists; then
		LINK_CLASS_OBJ=`GetLinkClassObj`; export LINK_CLASS_OBJ
	fi
	if TopCollExists; then
		TOP_COLL_OBJ=`GetTopCollObj`; export TOP_COLL_OBJ
		if ModuleCollExists; then
			MODULE_COLL_OBJ=`GetModuleCollObj`; export MODULE_COLL_OBJ
		fi
	fi
}

######################################################################
#
#	IsValidPolicyRegion policy-region-name
#
IsValidPolicyRegion()
{
if IsValidProfile PolicyRegion "$1"; then
	return 0
else
	return 1
fi
}

######################################################################
#
#	IsValidTask task-name tasklib-name
#
IsValidTask()
{
task_name="$1"
task_lib="$2"
if wlstlib "$task_lib#$IRONAME" 2> /dev/null | grep "^(task)	${task_name}$" > /dev/null 2>& 1; then
	return 0
else
	return 1
fi
}

######################################################################
#
#	IsValidGenericCollection collection-name
#
IsValidGenericCollection()
{
coll_name="$1"
COLLECTIONGUI=`GetCollectionGUIObj`
if idlcall "$COLLECTIONGUI" lookup_object '"'$coll_name'" {0}' > /dev/null; then
  return 0
else
  return 1
fi
}

######################################################################
#
#	IsValidLink link-name
#
IsValidLink()
{
link_name="$1"
if LinkClassExists; then
  LINK_CO=`GetLinkClassObj`
  set +e
  LINKOBJ=`idlcall $LINK_CO lookup_object '"'$link_name'" {0}'` > /dev/null 2>& 1
  status=$?
  set -e
  if [ $status -eq 0 -a "$LINKOBJ" ]; then
	if IsModuleMemberObj "$link_name" > /dev/null; then
		return 0
	fi
  fi
fi
return 1
}

######################################################################
#
# IsModuleCollectionEmpty
#
IsModuleCollectionEmpty() {

module_obj=
if ModuleCollExists; then
	module_obj=`GetModuleCollObj`
else
	return 0
fi
if [ "$module_obj" ]; then
	if [ `objcall "$module_obj" _get_members | wc -l` -eq 0 ]; then
		return 0
	fi
fi
return 1	
}
		
######################################################################
#
#	GetTopCollObj
#
GetTopCollObj()
{
tco=$TOP_COLL_OBJ
if [ ! "$tco" ]; then
  COLLECTIONGUI=`GetCollectionGUIObj`
  if idlcall "$COLLECTIONGUI" lookup_object '"'$TOP_PR_TO_USE'" {0}' > /dev/null; then
	tco=`idlcall "$COLLECTIONGUI" lookup_object '"'$TOP_PR_TO_USE'" {0}'`
  else
	return 1
  fi
fi
echo "$tco"
return 0
}

######################################################################
#
#	TopCollExists
#
TopCollExists()
{
tco=$TOP_COLL_OBJ
if [ ! "$tco" ]; then
  COLLECTIONGUI=`GetCollectionGUIObj`
  if idlcall "$COLLECTIONGUI" lookup_object '"'$TOP_PR_TO_USE'" {0}' > /dev/null; then
	return 0
  else
	return 1
  fi
fi
return 0
}

######################################################################
#
#	GetModuleCollObj
#
GetModuleCollObj()
{
mco=$MODULE_COLL_OBJ
if [ ! "$mco" ]; then
  parent_obj=`GetTopCollObj`
  if [ $? -eq 0 -a "$parent_obj" ]; then
	if idlcall "$parent_obj" lookup_object '"'$MEDIUM_PRODNAME'" {0}' > /dev/null; then
		mco=`idlcall "$parent_obj" lookup_object '"'$MEDIUM_PRODNAME'" {0}'`
	else
		return 1
	fi
  else
	return 1
  fi
fi
echo "$mco"
return 0
}


######################################################################
#
#	ModuleCollExists
#
ModuleCollExists()
{
mco=$MODULE_COLL_OBJ
if [ ! "$mco" ]; then
  parent_obj=`GetTopCollObj`
  if [ $? -eq 0 -a "$parent_obj" ]; then
	if idlcall "$parent_obj" lookup_object '"'$MEDIUM_PRODNAME'" {0}' > /dev/null; then
		return 0
	else
		return 1
	fi
  else
	return 1
  fi
fi
return 0
}


######################################################################
#
#	IsModuleMemberObj
#
IsModuleMemberObj()
{
member_name="$1"
if ModuleCollExists; then
	parent_obj=`GetModuleCollObj`
fi

if [ $? -eq 0 -a "$parent_obj" ]; then
	if idlcall "$parent_obj" lookup_object '"'$member_name'" {0}' > /dev/null; then
		return 0
	fi
fi
return 1
}

######################################################################
#
#	LinkClassExists
#
LinkClassExists()
{
if wlookup -r Classes SolutionsLink > /dev/null 2>&1; then
	return 0
else
	return 1
fi
}

######################################################################
#
#	GetLinkClassObj
#
GetLinkClassObj()
{
lco=$LINK_CLASS_OBJ
if [ ! "$lco" ]; then
	if wlookup -r Classes SolutionsLink > /dev/null 2>&1; then
		lco=`wlookup -r Classes SolutionsLink`
	else
		return 1
	fi
fi
echo "$lco"
return 0
}

######################################################################
#
#	GetTivLibrary
#
GetTivLibrary()
{
tlo=$TIV_LIBRARY_OBJ
if [ ! "$tlo" ]; then
	if wlookup Library > /dev/null 2>&1; then
		tlo=`wlookup Library`
	else
		return 1
	fi
fi
echo "$tlo"
return 0
}

######################################################################
#
#	GetCollectionGUIObj
#
GetCollectionGUIObj()
{
cgo=$COLLECTION_GUI_OBJ
if [ ! "$cgo" ]; then
	LIBRARY=`GetTivLibrary`
	if idlcall "$LIBRARY" lookup_object '"imp_TMF_TGC::CollectionGUI" {0}' > /dev/null; then
		cgo=`idlcall "$LIBRARY" lookup_object '"imp_TMF_TGC::CollectionGUI" {0}'`
	else
		return 1
	fi
fi
echo "$cgo"
return 0
}

######################################################################
#
#	set_path_and_dir
#
set_path_and_dir() {
	[ "$DEBUG" = true ] && set -x
	PATH=.:$PRODDIR:$LINKDIR:/bin:/usr/bin:$BINDIR/AEF:$PATH:/usr/ucb
	export PATH
	cd $PRODDIR || return 1
}

######################################################################
#
#	set_region_vars - Set parallel "R_*" variables with 
#                       region name appended to avoid collisions,
#			set some other vars also
#
set_region_vars() {
	[ "$DEBUG" = true ] && set -x
	DFLT_TASKLIB=`outmsg LinkInstallCat 200 "%1\\$s Utility Task Library" "$MEDIUM_PRODNAME_XLATED"`
	DEFAULT_TASKLIB="$MEDIUM_PRODNAME Utility Task Library"
	for var in $ARM_PMS $SENTRY_PMS $INDCOLLS $SUBLIST_PMS $TASKLIBS $SENTRY_PM_INDCOLLS $FILEPACKS; do
	  eval R_$var='"$'$var'#$IRONAME"'
	done
}

######################################################################
#
#	lookup_filepack_id
#
lookup_filepack_id() {
  [ "$DEBUG" = true ] && set -x
  match_name=$1
  eval match_name="$match_name"
  for i in $FILEPACKS; do
    eval name='"$'${i}'"'
    eval uiname='"$'${i}'_UINAME"'
    if [ "$name" = "$match_name" ]; then
      echo "$i"
      return 0
    elif [ "$uiname" = "$match_name" ]; then
      echo "$i"
      return 0
    fi
  done
  return 1
}

######################################################################
#
#	replace_nested_fp_name
#
replace_nested_fp_name() {
  [ "$DEBUG" = true ] && set -x
  fpid="$1"
  fpdesc="$2"
  tmpdesc=${fpdesc}.tmp
  rm -f $tmpdesc

  eval fpname='"$'$fpid'"'
  eval fpnestfps='"$'$fpid'_FPNESTFPS"'

  if [ -f "$fpnestfps" ]; then
    # Copy the nested file package list, appending the inter-region name
    while read line; do
      if [ "$line" ]; then
        nestref=${line}
	if IsValidProfile FilePackage "${nestref}#${IRONAME}"; then
          continue
        elif lookup_filepack_id "$nestref"; then
          nestid=`lookup_filepack_id "$nestref"`
          eval nestname='"$'$nestid'"'
          nestname=${nestname}#${IRONAME}
          sed -e "s~^${nestref}~\"${nestname}\"~" $fpdesc > $tmpdesc
        else
          errmsg LinkInstallCat 8 "Unable to locate nested file package!"
          return 1
        fi
      fi
    done <$fpnestfps
  fi
  if [ -f "$tmpdesc" ]; then
    mv $tmpdesc $fpdesc
  fi
}

######################################################################
#
log_start_instruct()
{
DATE=`date`
log_echo `outmsg LinkInstallCat 49 "***** Start %1\\$s: %2\\$s *****" "$SCRIPTNAME" "$MEDIUM_PRODNAME_XLATED"`
if $UNDO ; then
  log_echo `outmsg LinkInstallCat 50 "*****   (performing "UNDO") *****"`
fi
log_echo `outmsg LinkInstallCat 51 "***** $DATE *****"`
}

######################################################################
#
log_end_instruct()
{
DATE=`date`
log_echo `outmsg LinkInstallCat 51 "***** $DATE *****"`
log_echo `outmsg LinkInstallCat 52 "***** End %1\\$s: %2\\$s *****" "$SCRIPTNAME" "$MEDIUM_PRODNAME_XLATED"`
if $UNDO ; then
  log_echo `outmsg LinkInstallCat 50 "*****   (performing "UNDO") *****"`
fi
}

#	
#
######################################################################
#
copy_module_files()
{
  [ "$DEBUG" = true ] && set -x

  info_script=$1
  export_loc=$2

  info_file_name=PLUSproduct-info.sh
  module_dir=$SHORT_PRODNAME

  tiv_export_bin=${export_loc}/bin
  tiv_install_bin=$INST_DIR
  plus_export_path=${tiv_export_bin}/generic_unix/TME/PLUS
  plus_install_path=${PLUSDIR}
  msg_export_path=${tiv_export_bin}/../msg_cat
  msg_install_path=${tiv_install_bin}/../msg_cat

  PATH=${plus_export_path}/LINK:$PATH
  export PATH

  if [ ! -d "${plus_export_path}/${module_dir}" ]; then
    errmsg LinkInstallCat 9 "%1\$s/%2\$s is not a directory in the export tree" "${plus_export_path}" "${module_dir}"
    return 1
  elif [ ! -d "${tiv_install_bin}" ]; then
    errmsg LinkInstallCat 10 "%1\$s is not a directory in the install tree" "${tiv_install_bin}"
    return 1
  fi

  # copy our winstructed info script into place
  info_script_export=${plus_export_path}/${module_dir}/${info_file_name}
  if diff $info_script $info_script_export >/dev/null 2>&1; then
    :
  else
    cp $info_script $info_script_export || return 1
  fi

  ## AMP extraction may cause execute permission to be taken away; add it back here
  chmod -R +x  ${plus_export_path}/LINK/* >/dev/null 2>&1 || return 1
  chmod -R +x  ${plus_export_path}/${module_dir}/* >/dev/null 2>&1 || return 1
    # chmod others?

  #
  # Install LINK binaries and message catalogs if we've got a new version incoming
  #
  if [ -d  "${plus_export_path}/LINK" ]; then
    mkdir.sh "${plus_install_path}/LINK" || return 1    #just in case it's not there
    if needs_replacement  Link.*_*_* \
                     "$plus_install_path/LINK" \
                     "$plus_export_path/LINK" ; then
      cp -r ${plus_export_path}/LINK/* ${plus_install_path}/LINK || return 1
    fi
  fi
  for langdir in `ls $msg_export_path` ; do
    if [ -d  "${msg_export_path}/${langdir}" ]; then
      mkdir.sh "${msg_install_path}/${langdir}" || return 1   #just in case it's not there
      if needs_replacement  LinkCat.*_*_*.cat \
                     "${msg_install_path}/${langdir}" \
                     "${msg_export_path}/${langdir}" ; then
        cp ${msg_export_path}/${langdir}/*.cat ${msg_install_path}/${langdir} || return 1
      fi
    fi
  done


  #
  # Now install module binaries and message catalogs if we've got a new version incoming
  #
  if [ -d  "${plus_export_path}/${module_dir}" ]; then
    mkdir.sh "${plus_install_path}/${module_dir}" || return 1  #just in case it's not there
    if needs_replacement  $SHORT_PRODNAME \
                     "$plus_install_path/${module_dir}" \
                     "$plus_export_path/${module_dir}" ; then
      cp -r ${plus_export_path}/${module_dir}/* ${plus_install_path}/${module_dir} || return 1
    fi
  fi
  for langdir in `ls $msg_export_path` ; do
    if [ -d  "${msg_export_path}/${langdir}" ]; then
      mkdir.sh "${msg_install_path}/${langdir}" || return 1   #just in case it's not there
      if needs_replacement  $SHORT_PRODNAME \
                     "${msg_install_path}/${langdir}" \
                     "${msg_export_path}/${langdir}" ; then
	if ls ${msg_export_path}/${langdir}/${REAL_PRODNAME}*.cat > /dev/null 2>&1; then
          content=`ls ${msg_export_path}/${langdir}/${REAL_PRODNAME}*.cat 2> /dev/null`
          if [ "$content" ]; then
            cp ${msg_export_path}/${langdir}/${REAL_PRODNAME}*.cat ${msg_install_path}/${langdir} || return 1
          fi
	fi
	if ls ${msg_export_path}/${langdir}/${SHORT_PRODNAME}*.cat > /dev/null 2>&1; then
          content=`ls ${msg_export_path}/${langdir}/${SHORT_PRODNAME}*.cat 2> /dev/null`
          if [ "$content" ]; then
            cp ${msg_export_path}/${langdir}/${SHORT_PRODNAME}*.cat ${msg_install_path}/${langdir} || return 1
          fi
	fi
	if ls ${msg_export_path}/${langdir}/${SHORT_Prodname}*.cat > /dev/null 2>&1; then
          content=`ls ${msg_export_path}/${langdir}/${SHORT_Prodname}*.cat 2> /dev/null`
          if [ "$content" ]; then
            cp ${msg_export_path}/${langdir}/${SHORT_Prodname}*.cat ${msg_install_path}/${langdir} || return 1
          fi
	fi
	if ls ${msg_export_path}/${langdir}/${SHORT_prodname}*.cat > /dev/null 2>&1; then
          content=`ls ${msg_export_path}/${langdir}/${SHORT_prodname}*.cat 2> /dev/null`
          if [ "$content" ]; then
            cp ${msg_export_path}/${langdir}/${SHORT_prodname}*.cat ${msg_install_path}/${langdir} || return 1
          fi
	fi
      fi
    fi
  done

}

######################################################################
#
# create_task_job
#
# Sets up task, job, dialog, etc.
# See below for defaults.
# External dependencies:	R_TASKLIB, JOB_PM, MNNAME, PRNAME
#
create_task_job()	# -t <task_nm> -m <method> [-d <dialog>] [-r <role>] [-u <user>]
		#[-M <job mode>] [-T <job timeout>] [-p <pm>] [-h <mannode>]
		#[-i <icon state>] [ [-taskonly] | [-jobonly] ]
{

[ "$DEBUG" = true ] && set -x

#
# Variable initializations / defaults:
#
role=admin
user="\$root_user"
mode=parallel
timeout=0
taskonly=false
tasklib=$DEFAULT_TASKLIB

set +e
unset	task method dialog dest notarget # collection
unset	iconstate		# Choices:  SLINK CLINK NCLINK SULINK JLINK
set -e

#
# Parse arguments:
#
#		-c*) collection=$2;		shift	;;	# collection name

while [ $# -gt 0 ]; do
	case $1 in
		-taskonly) taskonly=true		;;	# Skip remaining setup
		-jobonly) jobonly=true			;;	# Skip task setup
		-t*) task="$2";		shift	;;	# Task name
		-m*) method=$2;		shift	;;	# Method name
		-d*) dialog=$2;		shift	;;	# Dialog descriptor
		-r*) role="$2";		shift	;;	# Execution role
		-u*) user=$2;		shift	;;	# Execution user
		-M*) mode=$2;		shift	;;	# Job mode
		-T*) timeout=$2;	shift	;;	# Job timeout
		-i*) iconstate=$2;	shift	;;	# Icon state
		-j*) jobname=$2;	shift	;;	# Job name
		-l*) tasklib=$2;	shift	;;	# Task library name
		-h*) dest="$dest -h \"$2#$IRONAME\"";	shift	;; # Job target mannode
		-p*) dest="$dest -p \"$2#$IRONAME\"";	shift	;; # Job target profmgr
		-n*) notarget=1				;;	# No Job target
		*)
			errmsg LinkInstallCat 11 "create_task_job:  '%1\$s' invalid." "$1"
			return 1
			;;
	esac
	if [ $# -lt 1 ]; then
		errmsg LinkInstallCat 12 "create_task_job:  syntax error:  %1\$s" "$*"
		return 1
	fi
	shift
done

log_info `outmsg LinkInstallCat 53 "task:        %1\\$s" "$task"`
log_info `outmsg LinkInstallCat 54 "method:      %1\\$s" "$method"`
log_info `outmsg LinkInstallCat 55 "dialog:      %1\\$s" "$dialog"`
log_info `outmsg LinkInstallCat 56 "role:        %1\\$s" "$role"`
log_info `outmsg LinkInstallCat 57 "user:        %1\\$s" "$user"`
log_info `outmsg LinkInstallCat 58 "mode:        %1\\$s" "$timeout"`
log_info `outmsg LinkInstallCat 59 "iconstate:   %1\\$s" "$iconstate"`
log_info `outmsg LinkInstallCat 60 "tasklib:     %1\\$s" "$tasklib"`
log_info `outmsg LinkInstallCat 61 "jobname:     %1\\$s" "$jobname"`
log_info `outmsg LinkInstallCat 62 "dest:        %1\\$s" "$dest"`
log_info `outmsg LinkInstallCat 63 "notarget:    %1\\$s" "$notarget"`

if [ -z "$task" -o -z "$method" ]; then
	errmsg LinkInstallCat 13 "create_task_job:  must specify task and method names."
	return 1
fi

if is_predefined_dep_task "$task"; then
	log_info `outmsg LinkInstallCat 64 "%1\\$s is a predefined dependency task ... skipping" "$task"`
        return 0
fi

[ -z "$dest$notarget" -a -n "$JOB_PM" ] && dest="-p \"$JOB_PM#$IRONAME\""

if [ "$user" = "$NULL" ]; then
	user=\$root_user
fi

r_tasklib="${tasklib}#${IRONAME}"

#
# Set up task
#
if [ ! "$jobonly" = true ]; then
  if wlstlib "$r_tasklib" | grep "^(task)	$task\$" >/dev/null 2>&1; then
	cmd=wsettask
  else
	cmd=wcrttask
  fi
  $cmd -t "$task" -l "$r_tasklib" -r "$role" -u "$user" -i default "$MNNAME" "$method"
  $taskonly && return 0
fi

#
# Skip creating a plus icon for the predefined Product_Install task, it wouldn't make sense
#
if [ "$task" = "Product_Install" ]; then
  return 0
fi

#
# Set up job
#

# Make sure job name is unique-ified
task_job="`echo $task | sed s/\" ($SHORT_PRODNAME)$\"//` ($SHORT_PRODNAME)"

if IsValidTask "$task" "$tasklib"; then
	if wlstlib "$r_tasklib" | grep "^(job)	$task_job\$" >/dev/null 2>&1; then
		cmd=wsetjob
		#-#-#
		log_info `outmsg LinkInstallCat 65 "%1\\$s job exists, overriding settings" "$task_job"`
	else
		cmd=wcrtjob
		#-#-#
		log_info `outmsg LinkInstallCat 66 "%1\\$s job does NOT exist, creating" "$task_job"`
	fi
	eval $cmd -j '"$task_job"' -l '"$r_tasklib"' -t '"$task"' -M '"$mode"' -m $timeout -D -o 15 $dest
else
	errmsg LinkInstallCat 14 "create_task_job:  Cannot create job %1\$s, its task does not exist." "$task"
	return 1
fi
#
# Finish
#
if [ "$dialog" = "$NULL" ]; then
	dialog=
fi

if [ -n "$dialog" -a ! -f "$dialog" ]; then
	log_info `outmsg LinkInstallCat 67 "Could not find dialog %1\\$s ... using default" "$dialog"`
fi
if [ -f "$dialog" ]; then
	#-#-#
	log_info `outmsg LinkInstallCat 69 "Adding dialog to %1\\$s" "$task"`
#	wputdialog -l @TaskLibrary:"$DEFAULT_TASKLIB" "$task" >/dev/null 2>&1 < $dialog
	wputdialog -l @TaskLibrary:"$r_tasklib" "$task" >/dev/null 2>&1 < $dialog
	wputdialog -l @TaskLibrary:"$r_tasklib" "$task_job" >/dev/null 2>&1 < $dialog
fi

wcrtjlink "$MEDIUM_PRODNAME" "$tasklib" "$task" "$jobname" "$task_job" $iconstate
#wcrtjlink "$collection" "$DEFAULT_TASKLIB" "$task" "$task" $iconstate
#-#-#
log_success `outmsg LinkInstallCat 70 "%1\\$s installed" "$task"`
}

######################################################################
#
# create_administrative_jobs
#
create_administrative_jobs()
{
[ "$DEBUG" = true ] && set -x
TASK_LIST="$TASKS"
TASK_TYPE=`outmsg LinkInstallCat 201 "Administrative"`
TASK_ICON=JLink_icon.xpm
JOBONLY=true

	#-#-#
	log_start_action `outmsg LinkInstallCat 74 "Create Administrative Jobs: %1\\$s" "$TASK_LIST"`

create_tasks_and_jobs

	#-#-#
	log_end_action
	
}

######################################################################
#
# create_plus_config_jobs
#
create_plus_config_jobs()
{
[ "$DEBUG" = true ] && set -x
TASK_LIST="$CONFIGTASKS $ABOUTTASKS"
TASK_TYPE=`outmsg LinkInstallCat 202 "Plus Configuration"`
TASK_ICON=SULink_icon.xpm
JOBONLY=true

	#-#-#
	log_start_action `outmsg LinkInstallCat 75 "Create Plus Config Jobs: %1\\$s" "$TASK_LIST"`

create_tasks_and_jobs

	#-#-#
	log_end_action
	
}

######################################################################
#
# create_tec_config_jobs
#
create_tec_config_jobs()
{
[ "$DEBUG" = true ] && set -x
TASK_LIST="$TECTASKS"
TASK_TYPE=`outmsg LinkInstallCat 203 "TEC Configuration"`
TASK_ICON=TEC_icon.xpm
JOBONLY=true

	#-#-#
	log_start_action `outmsg LinkInstallCat 76 "Create TEC Config Jobs: %1\\$s" "$TASK_LIST"`

create_tasks_and_jobs

	#-#-#
	log_end_action
	
}

######################################################################
#
# create_log_config_jobs
#
create_log_config_jobs()
{
[ "$DEBUG" = true ] && set -x
TASK_LIST="$LOGTASKS"
TASK_TYPE=`outmsg LinkInstallCat 204 "Logfile Adapter Configuration"`
TASK_ICON=LOG_icon.xpm
JOBONLY=true

	#-#-#
	log_start_action `outmsg LinkInstallCat 77 "Create Logfile Adapter Config Jobs: %1\\$s" "$TASK_LIST"`

create_tasks_and_jobs

	#-#-#
	log_end_action
	
}

######################################################################
#
# create_adapter_config_jobs
#
create_adapter_config_jobs()
{
[ "$DEBUG" = true ] && set -x
TASK_LIST="$ADAPTERTASKS"
TASK_TYPE=`outmsg LinkInstallCat 204 "Custom Adapter Configuration"`
TASK_ICON=LOG_icon.xpm
JOBONLY=true

	#-#-#
        log_start_action `outmsg LinkInstallCat 77 "Create Custom Adapter Config Jobs: %1\\$s" "$TA
SK_LIST"`

create_tasks_and_jobs

	#-#-#
	log_end_action

}


######################################################################
#
# create_tasks_and_jobs
#
create_tasks_and_jobs()
{
[ "$DEBUG" = true ] && set -x
DEFAULT_TIMEOUT=0
DEFAULT_MODE=parallel
JOBFLAG=""
if [ "$JOBONLY" = true ]; then
  JOBFLAG="-jobonly"
fi
  

for task in $TASK_LIST; do
	eval TASKNAME='"$'$task'"'
	eval TASKJOB='"$'$task'_JOB"'
	eval TASKSCRIPT='"$'$task'_SCRIPT"'
	eval TASKUSER='"$'$task'_USER"'
	eval TASKROLE='"$'$task'_ROLE"'
	eval TASKDIALOG='"$'$task'_DIALOG"'
	eval TASKLIBR='"$'$task'_LIBR"'
	eval TASKDEPENDS='"$'$task'_DEPS"'
	eval TASKICONDSL='"$'$task'_ICONDSL"'
	eval TASKICONSTATE='"$'$task'_ICONSTATE"'
	eval TASKTARGETS='"$'$task'_TARGETS"'
	eval TASKMODE='"$'$task'_MODE"'
	eval TASKTIMEOUT='"$'$task'_TIMEOUT"'

	if [ -z "$TASKLIBR" ] ; then
		TASKLIBR=$DEFAULT_TASKLIB
	fi

	if [ -z "$TASKUSER" ] ; then
		TASKUSER=$NULL
	fi

	if [ -n "$TASKDIALOG" -a ! -f $PRODDIR/$TASKDIALOG ]; then
		log_info `outmsg LinkInstallCat 68 "Could not find %1\\$s ... using default" "$TASKDIALOG"`
	fi
	if [ -z "$TASKDIALOG" -o ! -f $PRODDIR/$TASKDIALOG ] ; then
		TASKDIALOG=$NULL
	else
		TASKDIALOG=$PRODDIR/$TASKDIALOG
	fi

	if [ -z "$TASKMODE" ]; then
		TASKMODE=$DEFAULT_MODE
        fi

	if [ -z "$TASKTIMEOUT" ]; then
		TASKTIMEOUT=$DEFAULT_TIMEOUT
        fi

	if [ -n "$TASKROLE" ]; then
		TASKROLE=" -r `echo $TASKROLE | sed -e 's/,/:/g' -e 's/ //g'`"
	else
		TASKROLE=" -r admin"
	fi

	#-#-#
	log_start_action `outmsg LinkInstallCat 78 "Create View of %1\\$s Job %2\\$s" "$TASK_TYPE" "$TASKNAME"`

	if IsValidTaskLibrary "$TASKLIBR" && IsValidTask "$TASKNAME" "$TASKLIBR"; then

		if [ -n "$TASKICONDSL" -a ! -f $PRODDIR/$TASKICONDSL ]; then
			log_info `outmsg LinkInstallCat 68 "Could not find %1\\$s ... using default" "$TASKICONDSL"`
		fi
		if [ -n "$TASKICONDSL" -a -f $PRODDIR/$TASKICONDSL ]; then
			LINK_CO=`GetLinkClassObj`
			initpres $LINK_CO -class -put \
				state "$TASKICONSTATE" $PRODDIR/$TASKICONDSL \
					xpm $LINKDIR/$TASK_ICON
		else
			TASKICONSTATE=JLINK
		fi
	
		if check_dependencies $TASKDEPENDS; then
			if [ -z "$TASKTARGETS" ]; then
				TARGETS="-h \"$MNNAME\""
			else
				TARGETS=""
				for target in $TASKTARGETS; do
					eval TARG='"$'$target'"'
					TARGETS="$TARGETS -p \"@ProfileManager:$TARG\""
				done

			fi
			eval create_task_job \
				-t \"$TASKNAME\" \
				-m $PRODDIR/$TASKSCRIPT \
				-d $TASKDIALOG \
				$TARGETS \
				-i \"$TASKICONSTATE\" \
				-j \"$TASKJOB\" \
				-l \"$TASKLIBR\" \
				"$TASKROLE" \
				-u $TASKUSER \
				-M $TASKMODE \
				-T $TASKTIMEOUT \
				$JOBFLAG
		fi

	fi

	#-#-#
	log_end_action
done
refresh_plus_collection

}

######################################################################
#
# create_file_package_links
#
create_file_package_links()
{
#
#	Create the OmniFilePackage profile, and also create a hidden file
#	package for each one defined.
#

[ "$DEBUG" = true ] && set -x

for fp in $FILEPACKS; do
	eval FILEPACKNAME='"$'$fp'"'
	eval FPNAME='"$'$fp'_SHORTNAME"'
	eval FPPMNAME='"$'$fp'_PARENT"'
	eval FPUINAME='"$'$fp'_UINAME"'
	eval FPDEF='"$'$fp'_FPDEF"'
	eval FPCFG='"$'$fp'_SCRIPT"'
	eval FPDIALOG='"$'$fp'_DIALOG"'
	eval FPCONFIG='"$'$fp'_CONFIG_TASKS"'
	eval FPICONDSL='"$'$fp'_ICONDSL"'
	eval FPICONSTATE='"$'$fp'_ICONSTATE"'
	eval FPTARGETS='"$'$fp'_TARGETS"'
	eval FPDEPS='"$'$fp'_DEPS"'

	if [ -z "$FPTARGETS" ]; then
		TARGETS="@ManagedNode:$MNNAME"
	else
		TARGETS=""
		for target in $FPTARGETS; do
			eval TARG='"$'$target'"'
			TARGETS="$TARGETS \"@ProfileManager:$TARG\""
		done
	fi

	#-#-#
	log_info `outmsg LinkInstallCat 79 "Checking filepack dependencies"`

	if check_dependencies $FPDEPS; then
		#-#-#
		log_start_action `outmsg LinkInstallCat 80 "Create File Package %1\\$s" "$FILEPACKNAME"`

		if IsValidProfileManager "$FPPMNAME" && IsValidFilePackageProfile "$FILEPACKNAME"; then

			#-#-#
			log_info  `outmsg LinkInstallCat 81 "Updating subscription list for %1\\$s" "$FPPMNAME"`
			eval wpmsub \"$FPPMNAME\" $TARGETS

			#-#-#
			log_info `outmsg LinkInstallCat 82 "Creating state %1\\$s" "$FPICONSTATE"`

			if [ -n "$FPICONDSL" -a ! -f $PRODDIR/$FPICONDSL ]; then
				log_info `outmsg LinkInstallCat 68 "Could not find %1\\$s ... using default" "$FPICONDSL"`
			fi
			if [ -n "$FPICONDSL" -a -f $PRODDIR/$FPICONDSL ]; then
				LINK_CO=`GetLinkClassObj`
				initpres $LINK_CO -class -put \
					state "$FPICONSTATE" $PRODDIR/$FPICONDSL \
					xpm $LINKDIR/NCLink_icon.xpm
			else
				FPICONSTATE=NCLINK
			fi
	

			if [ "$FPCONFIG" != "TRUE" ]; then
				#-#-#
				log_info `outmsg LinkInstallCat 83 "No fp config tasks specified for \"%1\\$s\"!" "$FILEPACKNAME"`
			else
				#-#-#
				log_start_action `outmsg LinkInstallCat 84 "Creating OmniFilePackage \"%1\\$s\"" "$FILEPACKNAME"`

				eval wcrtsprfl -s $FPICONSTATE \"$MEDIUM_PRODNAME\" OmniFilePackage \
					\"$FPUINAME\" \"$FPPMNAME\" \"$FPUINAME\"

				#-#-#
				log_success `outmsg LinkInstallCat 85 "OmniFilePackage \"%1\\$s\" creation successful" "$FILEPACKNAME"`
				#-#-#
				log_end_action
			fi

		fi
	fi

done
refresh_plus_collection


###??? only do the following for lowest level nested filepacks ??? i.e., ones
###      that have no NESTED FPS?
###    AND only do the following for filepacks that do not have config tasks???

###    ALSO, if there is more than one level of nesting, the nested
###      filepack names will need to be computed and appended to the
###      definition file ... users will not be able to predetermine
###      the names of nested filepacks!!

###    OR I guess we could import all filepacks, even configurable ones,
###    provided we perform a step of making sure the config files refer
###    correctly to existing (created by winstruct_file) nested FilePacks.
###    This could be done by the toolkit not generating refs to nested
###    FilePacks in the *.fp files, and instead letting us append that
###    info to the FPDEF file HERE!

###    Another problem is that executing the "configure filepack" job fails 
###    because it is referring ??? to the "LINK" names for the filepack
###    and the nested filepack!  See a new script, config_fp_new.sh, for
###    some needed mods ...

###   UNDER TEST: use of replace_nested_fp_names below!

#
#	Import each file package description.  This is done as a separate step
#	so we don't have to determine the order to create filepackages because
#	of nesting.
#
for fp in $FILEPACKS; do
	eval FILEPACKNAME='"$'$fp'"'
	eval FPNAME='"$'$fp'_SHORTNAME"'
	eval FPPMNAME='"$'$fp'_PARENT"'
	eval FPDEF='"$'$fp'_FPDEF"'
	eval FPCFG='"$'$fp'_SCRIPT"'
	eval FPDIALOG='"$'$fp'_DIALOG"'
	eval FPDEPS='"$'$fp'_DEPS"'

	#-#-#
	log_info `outmsg LinkInstallCat 79 "Checking filepack dependencies"`

	if check_dependencies $FPDEPS; then

		if IsValidProfileManager "$FPPMNAME" && IsValidFilePackageProfile "$FILEPACKNAME"; then

			#-#-#
			log_start_action `outmsg LinkInstallCat 86 "Importing definition into File Package %1\\$s" "$FILEPACKNAME"`

			# Import the file package definition
			#-#-#
			log_info `outmsg LinkInstallCat 87 "Importing file package definition from %1\\$s" "$FPDEF"`

			TMPFPDESC=$TEMP/$FPNAME.fp
			rm -f $TMPFPDESC
			cat $PRODDIR/$FPDEF >$TMPFPDESC
			replace_nested_fp_name $fp $TMPFPDESC

			wimprtfp -h @ManagedNode:"$MNNAME#$IRONAME" \
				@FilePackage:"$FILEPACKNAME#$IRONAME" < $TMPFPDESC

			wsetfpopts -T gen -S $PRODDIR @FilePackage:"$FILEPACKNAME#$IRONAME"

			#-#-#
			log_success `outmsg LinkInstallCat 88 "File Package %1\\$s definition import successful" "$FILEPACKNAME"`

			#-#-#
			log_end_action
		fi
	fi
done
}

######################################################################
#
#	Create filepack configuration jobs
#
create_filepack_config_jobs() {

[ "$DEBUG" = true ] && set -x
for fp in $FILEPACKTASKS; do
	eval FPTASKDEPS='"$'$fp'_DEPS"'

	if check_dependencies $FPTASKDEPS; then
		eval FPNAME='"$'$fp'"'
		eval FPLIBR='"$'$fp'_LIBR"'
		eval FPJOB='"$'$fp'_JOB"'
		eval FPSCRIPT='"$'$fp'_CONFIGSCRIPT"'
		eval FPUSER='"$'$fp'_USER"'
		eval FPROLE='"$'$fp'_ROLE"'
		eval FPDIALOG='"$'$fp'_DIALOG"'
		eval FPTASKICONDSL='"$'$fp'_ICONDSL"'
		eval FPTASKICONSTATE='"$'$fp'_ICONSTATE"'

		if [ -z "$FPUSER" ] ; then
			FPUSER=$NULL
		fi

		#-#-#
		log_start_action `outmsg LinkInstallCat 89 "Create Configurable File Package %1\\$s" "$FPNAME"`

		if IsValidTaskLibrary "$FPLIBR" && IsValidTask "$FPNAME" "$FPLIBR"; then
 
			if [ -n "$FPDIALOG" -a ! -f $PRODDIR/$FPDIALOG ]; then
				log_info `outmsg LinkInstallCat 68 "Could not find %1\\$s ... using default" "$FPDIALOG"`
			fi
			if [ -z "$FPDIALOG" -o ! -f $PRODDIR/$FPDIALOG ] ; then
				FPDIALOG=$NULL
			else
				FPDIALOG=$PRODDIR/$FPDIALOG
			fi

			if [ -n "$FPROLE" ]; then
				FPROLE=" -r `echo $FPROLE | sed -e 's/,/:/g' -e 's/ //g'`"
			else
				FPROLE=" -r senior"
			fi

			if [ -n "$FPTASKICONDSL" -a ! -f $PRODDIR/$FPTASKICONDSL ]; then
				log_info `outmsg LinkInstallCat 68 "Could not find %1\\$s ... using default" "$FPTASKICONDSL"`
			fi
			if [ -n "$FPTASKICONDSL" -a -f $PRODDIR/$FPTASKICONDSL ]; then
				LINK_CO=`GetLinkClassObj`
				initpres $LINK_CO -class -put \
					state "$FPTASKICONSTATE" $PRODDIR/$FPTASKICONDSL \
						xpm $LINKDIR/SULink_icon.xpm
			else
				FPTASKICONSTATE=SULINK
			fi
	
			create_task_job \
				-t "$FPNAME" \
				-m $PRODDIR/$FPSCRIPT \
				-d $FPDIALOG \
				-h "$MNNAME" \
				-l "$FPLIBR" \
				"$FPROLE" \
				-u $FPUSER \
				-M parallel \
				-T 60 \
				-j "$FPJOB" \
				-i $FPTASKICONSTATE \
				-jobonly

		fi

		#-#-#
		log_end_action
	fi
done
}

######################################################################
#
# create_sentry_profile_links
#
create_sentry_profile_links()
{

[ "$DEBUG" = true ] && set -x

COLLOBJ=`GetModuleCollObj`

#
# Create Indicator Collections
#
for indcoll in $INDCOLLS; do
	eval INDCOLL='"$'$indcoll'"'
	eval R_INDCOLL='"$R_'$indcoll'"'
	eval COLLDEPENDS='"$'$indcoll'_DEPS"'
	eval COLL_ICONDSL='"$'$indcoll'_ICONDSL"'
	eval COLL_ICONSTATE='"$'$indcoll'_ICONSTATE"'

	#-#-#
	log_start_action `outmsg LinkInstallCat 90 "Create Indicator collection %1\\$s" "$INDCOLL"`

	if check_dependencies $COLLDEPENDS; then
		if IsValidIndicatorCollection "$INDCOLL"; then

			#make sure it gets added:
			SNTCOLL=`wlookup -r IndicatorCollection "$R_INDCOLL"`
			gcoadd "$COLLOBJ" "$SNTCOLL"

			#-#-#
			log_success `outmsg LinkInstallCat 71 "%1\\$s exists" "$INDCOLL"`
		else
			wsetpr IndicatorCollection @PolicyRegion:"$APP_PR_NAME#$IRONAME"
			SNTCOLL=`wcrtsntcoll "$APP_PR_NAME#$IRONAME" "$INDCOLL"`
			if [ -n "$COLL_ICONDSL" -a ! -f $PRODDIR/$COLL_ICONDSL ]; then
			  log_info `outmsg LinkInstallCat 68 "Could not find %1\\$s ... using default" "$COLL_ICONDSL"`
			fi
			if [ -n "$COLL_ICONDSL" -a -f $PRODDIR/$COLL_ICONDSL ]; then
				wputicon -r IndicatorCollection $COLL_ICONSTATE \
					$PRODDIR/$COLL_ICONDSL \
					$INST_DIR/generic_unix/TME/PLUS/LINK/PM_icon.xpm
				idlcall "$SNTCOLL"  _set_state '"'$COLL_ICONSTATE'"'
			fi
			gcoadd "$COLLOBJ" "$SNTCOLL"
			log_success `outmsg LinkInstallCat 72 "%1\\$s created" "$INDCOLL"`
		fi
	fi

	#-#-#
	log_end_action
done
refresh_plus_collection
        
#
# Create View of Sentry Profile Managers
#
for sentrypm in $SENTRY_PMS; do
	eval PROFNAME='"$'$sentrypm'"'
	eval R_PROFNAME='"$R_'$sentrypm'"'
	eval PMNAME='"$'$sentrypm'_PARENT"'
	eval R_PMNAME='"$R_'$sentrypm'_PARENT"'
	eval UINAME='"$'$sentrypm'_UINAME"'
	eval SPM_INDCOLLS='"$'$sentrypm'_INDCOLLS"'
	eval SPM_DEPENDS='"$'$sentrypm'_DEPS"'
	eval ICONDSL='"$'$sentrypm'_ICONDSL"'
	eval ICONSTATE='"$'$sentrypm'_ICONSTATE"'
	eval PM_TARGETS='"$'$sentrypm'_TARGETS"'

	#-#-#
	log_start_action `outmsg LinkInstallCat 91 "Create view of Sentry Profile Manager %1\\$s" "$PMNAME"`

	if IsValidProfileManager "$PMNAME" && IsValidSentryProfile "$PROFNAME"; then

		if check_dependencies $SPM_DEPENDS; then
			# Set the subscrition list
			if [ -z "$PM_TARGETS" ]; then
				TARGETS=@ManagedNode:$MNNAME
			else
				TARGETS=""
				for target in $PM_TARGETS; do
					eval TARG='"$'$target'"'
					TARGETS="$TARGETS \"@ProfileManager:$TARG\""
				done
			fi

			log_info `outmsg LinkInstallCat 92 "%1\\$s exists, updating subscription list" "$R_PROFNAME"`
			eval wpmsub \"$PMNAME\" $TARGETS

			if [ -n "$ICONDSL" -a ! -f $PRODDIR/$ICONDSL ]; then
				log_info `outmsg LinkInstallCat 68 "Could not find %1\\$s ... using default" "$ICONDSL"`
			fi
			if [ -n "$ICONDSL" -a -f $PRODDIR/$ICONDSL ]; then
				LINK_CO=`GetLinkClassObj`
				initpres $LINK_CO -class -put \
					state "$ICONSTATE" $PRODDIR/$ICONDSL \
						xpm $LINKDIR/SLink_icon.xpm
			else
				ICONSTATE=SLINK
			fi

			eval wcrtsprfl -s $ICONSTATE \"$MEDIUM_PRODNAME\" SentryProfile \
				\"$PROFNAME\" \
				\"$PMNAME\" \
				\"$UINAME\"

			# Profile must run as $root_user
			wsetsntid 0 0 "$R_PROFNAME"

			for indcoll in $SPM_INDCOLLS; do
				eval INDCOLL='"$'$indcoll'"'
				eval R_INDCOLL='"$'$indcoll'#$IRONAME"'

				if wsetcoll "$R_INDCOLL" "$R_PROFNAME"; then
					log_success `outmsg LinkInstallCat 93 "done"`
				else
					#-#-#
					log_success `outmsg LinkInstallCat 94 "%1\\$s in %2\\$s" "$R_PROFNAME" "$INDCOLL"`
				fi
			done

		        add_cond_levels "$PROFNAME" "$TIV_COND_LEVELS"

    		fi

	fi

	#-#-#
	log_end_action
done
refresh_plus_collection
}

######################################################################
#
# delete_link_obj
#
delete_link_obj()
{

[ "$DEBUG" = true ] && set -x

LINKNAME="$1"

if IsValidLink "$LINKNAME"; then
  parent_obj=`GetModuleCollObj`
  if [ $? -eq 0 -a "$parent_obj" ]; then
	if idlcall "$parent_obj" lookup_object '"'$LINKNAME'" {0}' > /dev/null; then
		LINKOBJ=`idlcall "$parent_obj" lookup_object '"'$LINKNAME'" {0}'`
		if [ $? -eq 0 -a "$LINKOBJ" ]; then
		  objcall "$LINKOBJ" _delete
		fi
	fi
  fi
fi
}

######################################################################
#
# delete_log_config_jobs
#
delete_log_config_jobs()
{
[ "$DEBUG" = true ] && set -x
TASK_LIST="$LOGTASKS"
TASK_TYPE=`outmsg LinkInstallCat 204 "Logfile Adapter Configuration"`

	#-#-#
	log_start_action `outmsg LinkInstallCat 95 "Delete Logfile Adapter Config Jobs: %1\\$s" "$TASK_LIST"`

	delete_jobs

	#-#-#
	log_end_action
	
}

######################################################################
#
# delete_adapter_config_jobs
#
delete_adapter_config_jobs()
{
[ "$DEBUG" = true ] && set -x
TASK_LIST="$ADAPTERTASKS"
TASK_TYPE=`outmsg LinkInstallCat 204 "Custom Adapter Configuration"`

	#-#-#
	log_start_action `outmsg LinkInstallCat 95 "Delete Custom Adapter Config Jobs: %1\\$s" "$TASK_LIST"`
        delete_jobs

	#-#-#
	log_end_action

}

######################################################################
#
# delete_tec_config_jobs
#
delete_tec_config_jobs()
{
[ "$DEBUG" = true ] && set -x
TASK_LIST="$TECTASKS"
TASK_TYPE=`outmsg LinkInstallCat 203 "TEC Configuration"`

	#-#-#
	log_start_action `outmsg LinkInstallCat 96 "Delete TEC Config Jobs: %1\\$s" "$TASK_LIST"`

	delete_jobs

	#-#-#
	log_end_action
	
}

######################################################################
#
# delete_plus_config_jobs
#
delete_plus_config_jobs()
{
[ "$DEBUG" = true ] && set -x
TASK_LIST="$CONFIGTASKS $ABOUTTASKS"
TASK_TYPE=`outmsg LinkInstallCat 202 "Plus Configuration"`

	#-#-#
	log_start_action `outmsg LinkInstallCat 97 "Delete Plus Config Jobs: %1\\$s" "$TASK_LIST"`

	delete_jobs

	#-#-#
	log_end_action
	
}

######################################################################
#
# delete_administrative_jobs
#
delete_administrative_jobs()
{
[ "$DEBUG" = true ] && set -x
TASK_LIST="$TASKS"
TASK_TYPE=`outmsg LinkInstallCat 201 "Administrative"`

	#-#-#
	log_start_action `outmsg LinkInstallCat 98 "Delete Administrative Jobs: %1\\$s" "$TASK_LIST"`

	delete_jobs

	#-#-#
	log_end_action
	
}

######################################################################
#
# delete_filepack_config_jobs
#
delete_filepack_config_jobs()
{
[ "$DEBUG" = true ] && set -x
TASK_LIST="$FILEPACKTASKS"
TASK_TYPE=`outmsg LinkInstallCat 205 "File Package Configuration"`

	#-#-#
	log_start_action `outmsg LinkInstallCat 99 "Delete Filepack Config Jobs: %1\\$s" "$TASK_LIST"`

	delete_jobs

	#-#-#
	log_end_action
}

######################################################################
#
# delete_sublist_task_link
#
delete_sublist_task_link()
{

	[ "$DEBUG" = true ] && set -x

        SUBLIST_TASK_NAME="`sublist_task_name`"

	#-#-#
	log_start_action `outmsg LinkInstallCat 100 "Delete Sublist Creation Task: %1\\$s" "$SUBLIST_TASK_NAME"`

	delete_link_obj "$SUBLIST_TASK_NAME"

	#-#-#
	log_end_action
	
}

######################################################################
#
# delete_jobs
#
delete_jobs()
{
[ "$DEBUG" = true ] && set -x
for task in $TASK_LIST; do
	eval TASKNAME='"$'$task'"'
	eval TASKJOB='"$'$task'_JOB"'
	eval TASKLIBR='"$'$task'_LIBR"'
	R_TASKLIBR="${TASKLIBR}#${IRONAME}"

        if [ "$USE_TLL" = true ]; then
                TASKNAME=`normalize_name "$TASKNAME"`
        fi

        # Make sure job name is unique-ified
        TASKNAME_TO_USE="`echo $TASKNAME | sed s/\" ($SHORT_PRODNAME)$\"//` ($SHORT_PRODNAME)"

	#-#-#
	log_start_action `outmsg LinkInstallCat 101 "Delete %1\\$s Job %2\\$s" "$TASK_TYPE" "$TASKNAME_TO_USE"`

	if wlstlib "$R_TASKLIBR" 2> /dev/null | grep "^(job)	$TASKNAME_TO_USE\$" >/dev/null 2>&1; then
		#-#-#
		log_info `outmsg LinkInstallCat 103 "%1\\$s exists, deleting" "$task"`
		wdeljob "$TASKNAME_TO_USE" "$R_TASKLIBR"
	fi

	if [ "$TASKJOB" ]; then
		delete_link_obj "$TASKJOB"
	fi

	#-#-#
	log_end_action

done
refresh_plus_collection
}

######################################################################
#
# remove_plus_collection
#
remove_plus_collection()
{

[ "$DEBUG" = true ] && set -x

#-#-#
log_start_action `outmsg LinkInstallCat 104 "Removing plus collection for module"`

if IsModuleCollectionEmpty; then

	#
	# Retrieve the OID of the plus collection 
	#
	if ModuleCollExists; then
		coll_obj=`GetModuleCollObj`
		idlcall $coll_obj remove
	fi

else
  #-#-#
  log_failure `outmsg LinkInstallCat 105 "Error removing module collection: collection is not empty"`

fi

# don't remove the top collection, just in case there's another plus module in it ...

refresh_top_collection

#-#-#
log_end_action
}

######################################################################
#
# delete_app_launch
#
delete_app_launch()
{

  [ "$DEBUG" = true ] && set -x

  #   undo the cfgappicon (and associated dialogs, bitmaps?),
  #   including removing the Presentation object  
  #       first, delete $PRESNAME presentation object, i.e.,
  #             objcall `wlookup -r Presentation "$PRESNAME"` _delete
  #       then, delete APPNAME's instance of TMF_SysAdmin::InstanceManager
  #       last, delete the object from the plus collection and refresh

CLO=`GetTivLibrary`
CO=`idlcall $CLO TMF_SysAdmin::Library::lookup_object \
		'"TMF_imp_TSysAdmin::PolicyDrivenBase" {0}'`
MNOID=`objcall 0.0.0 get_host_location`
eval HOSTNAME=`idlattr -t -g $MNOID label string`
SOL_NAME="$MEDIUM_PRODNAME"

for icon in $LAUNCH_ICONS; do
	eval LAUNCHNAME='"$'$icon'"'

	APPNAME="$LAUNCHNAME"
	PRESNAME="$APPNAME"_"$HOSTNAME"_"$SOL_NAME"
	OBJ_NAME="$APPNAME"@"$HOSTNAME"

	#-#-#
	log_start_action `outmsg LinkInstallCat 102 "Delete AppLaunch icon %1\\$s" "$LAUNCHNAME"`

	if IsValidPresentationObject "$PRESNAME"; then
		ICON=`wlookup -r Presentation "$PRESNAME"`
		if [ $? -eq 0 -a "$ICON" ]; then
			objcall "$ICON" _delete
		fi
	fi

	if ModuleCollExists; then
		coll_obj=`GetModuleCollObj`
		if idlcall "$coll_obj" lookup_object '"'$OBJ_NAME'" {0}' > /dev/null; then
			YOUR_OBJECT=`idlcall "$coll_obj" lookup_object '"'$OBJ_NAME'" {0}'`
			if [ "$YOUR_OBJECT" ]; then
				objcall "$YOUR_OBJECT" _delete		
			fi
		fi
	fi

	#-#-#
	log_success `outmsg LinkInstallCat 73 "%1\\$s deleted" "$LAUNCHNAME"`

	#-#-#
	log_end_action
done
refresh_plus_collection

}

######################################################################
#
# refresh_plus_collection
#
refresh_plus_collection() {
	[ "$DEBUG" = true ] && set -x
	if ModuleCollExists; then
	  coll_obj=`GetModuleCollObj`
	  idlcall "$coll_obj" refresh_collection
	fi
}

######################################################################
#
# refresh_top_collection
#
refresh_top_collection() {
	[ "$DEBUG" = true ] && set -x
	if IsValidGenericCollection "$TOP_PR_TO_USE"; then
	  TOPCOLL=`GetTopCollObj`
	  idlcall "$TOPCOLL" refresh_collection
	fi
}

######################################################################
#
# refresh_desktop
#
refresh_desktop() 
{
[ "$DEBUG" = true ] && set -x
ALI=`wlookup TME_server`
for var in `objcall $ALI get_identity`; do ADMIN=$var; done
if [ "$ADMIN" ]; then
  #Refresh the desktop
  idlcall $ADMIN refresh_collection
fi 
}

######################################################################
#
# remove_host_from_subscription_lists
#
remove_host_from_subscription_lists()
{ 
[ "$DEBUG" = true ] && set -x
for sublist in $SUBLIST_PMS; do
	eval SUBLIST='"$'$sublist'"'
	eval R_SUBLIST='"$R_'$sublist'"'
	eval SUBLIST_DEPS='"$'$sublist'_DEPS"'

	#-#-#
	log_start_action `outmsg LinkInstallCat 106 "Remove host %1\\$s from Subscription list %2\\$s" "$MNNAME" "$R_SUBLIST"`

	if IsValidProfileManager "$SUBLIST"; then
	        wunsub -l @ProfileManager:"$R_SUBLIST" @ManagedNode:$MNNAME#$IRONAME
	fi

	#-#-#
	log_success `outmsg LinkInstallCat 107 "Unsubscribing %1\\$s from %2\\$s" "$MNNAME" "$R_SUBLIST"`

	#-#-#
	log_end_action
done
}

######################################################################
#
# delete_subscription_lists
#
delete_subscription_lists()
{
  #   need to remove the profile manager for each sub list ...
  #   NOTE: deleting these will remove them from the PR_ policy region also

[ "$DEBUG" = true ] && set -x
for sublist in $SUBLIST_PMS; do
	eval SUBLIST='"$'$sublist'"'
	eval R_SUBLIST='"$R_'$sublist'"'

	#-#-#
	log_start_action `outmsg LinkInstallCat 108 "Delete Subscription List %1\\$s" "$SUBLIST"`

	if IsValidProfileManager "$SUBLIST"; then

	  log_info `outmsg LinkInstallCat 109 "%1\\$s exists, clearing subscribers" "$SUBLIST"`

	  reregister_pm "$SUBLIST"
	  if wlookup -r ProfileManager "$R_SUBLIST" > /dev/null 2>&1; then
	    wunsub -a "@ProfileManager:$R_SUBLIST"
	    wdel "@ProfileManager:$R_SUBLIST"
	  fi

	fi

	#-#-#
	log_end_action

done
refresh_plus_collection

}

######################################################################
#
# remove_install_options
#
remove_install_options()
{
  #   we need to delete the presentation object that is created
  #   for each install option (after removing removing the pres object's 
  #   attribute?) 
  #       (see wls /Library/TMF_imp_TSysAdmin::PolicyDrivenBase)

[ "$DEBUG" = true ] && set -x

#-#-#
log_start_action `outmsg LinkInstallCat 111 "Removing install options"`

CLO=`GetTivLibrary`
CO=`idlcall $CLO TMF_SysAdmin::Library::lookup_object \
		'"TMF_imp_TSysAdmin::PolicyDrivenBase" {0}'`

for installopt in $INSTALL_OPTS; do
	eval OPTNAME='"$'$installopt'_NAME"'

	# locate hidden presentation object holding the install options
	PRESNAME="$OPTNAME"_"$MNNAME"

	#-#-#
	log_info `outmsg LinkInstallCat 112 "Deleting Presentation object %1\\$s" "$PRESNAME"`

	if wlookup -r Presentation "$PRESNAME" > /dev/null 2>&1; then
		PRES=`wlookup -r Presentation "$PRESNAME"`
		if [ $? -eq 0 -a "$PRES" ]; then
			objcall "$PRES" _delete
		fi
	fi

done
log_end_action
}

######################################################################
#
# delete_sentry_profiles
#
delete_sentry_profiles()
{
#
# Delete View of Sentry Profile Managers
#
[ "$DEBUG" = true ] && set -x
for sentrypm in $SENTRY_PMS; do
	eval PROFNAME='"$'$sentrypm'"'
	eval R_PROFNAME='"$R_'$sentrypm'"'
	eval PMNAME='"$'$sentrypm'_PARENT"'
	eval R_PMNAME='"$R_'$sentrypm'_PARENT"'
	eval UINAME='"$'$sentrypm'_UINAME"'
	eval SPM_INDCOLLS='"$'$sentrypm'_INDCOLLS"'

	# Some defaults here will let us remove link objects even if this module 
	# was not instructed!
	if [ ! "$PMNAME" ]; then
		eval PMNAME='"$'$sentrypm'"'
		eval R_PMNAME='"$R_'$sentrypm'"'
	fi
	if [ ! "$UINAME" ]; then
		eval UINAME='"$'$sentrypm'"'
	fi

	# remove imbedded blanks from collection name
	PROFNAME="`echo $PROFNAME`"
	R_PROFNAME="`echo $R_PROFNAME`"
	PMNAME="`echo $PMNAME`"
	R_PMNAME="`echo $R_PMNAME`"
	UINAME="`echo $UINAME`"


	#-#-#
	log_start_action `outmsg LinkInstallCat 168 "Delete view of Sentry Profile Manager %1\\$s" "$PMNAME"`

	if IsValidProfileManager "$PMNAME" && IsValidSentryProfile "$PROFNAME"; then

			log_info `outmsg LinkInstallCat 110 "%1\\$s exists, clearing subscription list" "$R_PROFNAME"`

			PM_CLASS=`wlookup -r Classes ProfileManager`
			PRFMGR=`idlcall $PM_CLASS lookup_object '"'$PMNAME'" {0}'`
			reregister_pm "$PMNAME"
			wunsub -a @ProfileManager:"$PMNAME#$IRONAME"
			wregister -u -r ProfileManager "$PMNAME#$IRONAME"

			#should be able to unset the indicator affiliation, but can't!
			#  wsetcoll "" "$R_PMNAME"

	fi

	# Delete the Link object
	delete_link_obj "$UINAME"

	#-#-#
	log_end_action
done
refresh_plus_collection


#
# Delete Indicator Collections
#
for indcoll in $INDCOLLS; do
	eval INDCOLL='"$'$indcoll'"'
	eval R_INDCOLL='"$R_'$indcoll'"'

	#-#-#
	log_start_action `outmsg LinkInstallCat 114 "Delete Indicator collection %1\\$s" "$INDCOLL"`

	if IsValidIndicatorCollection "$INDCOLL"; then
        SNTCOLL=`wlookup -r IndicatorCollection "$R_INDCOLL"`
	  if [ $? -eq 0 -a "$SNTCOLL" ]; then
	    objcall "$SNTCOLL" _delete
	  fi
      fi

	log_success `outmsg LinkInstallCat 73 "%1\\$s deleted" "$INDCOLL"`

	#-#-#
	log_end_action

done
refresh_plus_collection

}

######################################################################
#
# delete_inventory_profile
#
delete_inventory_profile()
{
#
# Delete View of Inventory Profile Manager
#
[ "$DEBUG" = true ] && set -x

PMNAME="$INV_PM"
R_PMNAME="${PMNAME}#${IRONAME}"

#-#-#
log_start_action `outmsg LinkInstallCat 217 "Delete view of Inventory Profile Manager %1\\$s" "$PMNAME"`

if IsValidProfileManager "$PMNAME" && IsValidProfile InventoryProfile "$PMNAME"; then

	log_info `outmsg LinkInstallCat 110 "%1\\$s exists, clearing subscription list" "$R_PMNAME"`

	PM_CLASS=`wlookup -r Classes ProfileManager`
	PRFMGR=`idlcall $PM_CLASS lookup_object '"'$PMNAME'" {0}'`
	reregister_pm "$PMNAME"
	wunsub -a @ProfileManager:"$PMNAME#$IRONAME"
	wregister -u -r ProfileManager "$PMNAME#$IRONAME"
fi

# Delete the Link object
delete_link_obj "$PMNAME"

#-#-#
log_end_action

refresh_plus_collection

}

######################################################################
#
# delete_mar_profiles
#
delete_mar_profiles()
{
#
# Delete View of Mar Profile Managers
#
[ "$DEBUG" = true ] && set -x
for marpm in $ARM_PMS; do
	eval PMNAME='"$'$marpm'"'
	eval R_PMNAME='"$R_'$marpm'"'
	eval UINAME='"$'$marpm'_UINAME"'

	# Some defaults here will let us remove link objects even if this module
	# was not instructed!
	if [ ! "$PMNAME" ]; then
		eval PMNAME='"$'$marpm'"'
		eval R_PMNAME='"$R_'$marpm'"'
	fi
	if [ ! "$UINAME" ]; then
		eval UINAME='"$'$marpm'"'
	fi
	
	#-#-#
	log_start_action `outmsg LinkInstallCat 207 "Delete view of Mar Profile Manager %1\\$s" "$PMNAME"`

	if IsValidProfileManager "$PMNAME" && IsValidProfile MarProfile "$PROFNAME"; then

		log_info `outmsg LinkInstallCat 110 "%1\\$s exists, clearing subscription list" "$R_PROFNAME"`

		PM_CLASS=`wlookup -r Classes ProfileManager`
		PRFMGR=`idlcall $PM_CLASS lookup_object '"'$PMNAME'" {0}'`
		reregister_pm "$PMNAME"
		wunsub -a @ProfileManager:"$PMNAME#$IRONAME"
		wregister -u -r ProfileManager "$PMNAME#$IRONAME"
	fi

	# Delete the Link object
	delete_link_obj "$UINAME"

	#-#-#
	log_end_action
done
refresh_plus_collection

}

######################################################################
#
# delete_file_packages
#
delete_file_packages()
{
#
#	Delete the OmniFilePackage profile, and also delete the hidden file
#	package for each one defined.
#
[ "$DEBUG" = true ] && set -x
for fp in $FILEPACKS; do
	eval FILEPACKNAME='"$'$fp'"'
	eval FPNAME='"$'$fp'_SHORTNAME"'
	eval FPPMNAME='"$'$fp'_PARENT"'
	eval FPUINAME='"$'$fp'_UINAME"'
	eval FPDEF='"$'$fp'_FPDEF"'
	eval FPDIALOG='"$'$fp'_DIALOG"'
	eval FPCONFIG='"$'$fp'_CONFIG_TASKS"'
	eval FPICONDSL='"$'$fp'_ICONDSL"'
	eval FPICONSTATE='"$'$fp'_ICONSTATE"'
	eval FPTARGETS='"$'$fp'_TARGETS"'

	# Setting some defaults here will let us remove link objects even if this module 
	# was not instructed!
	if [ ! "$FPPMNAME" ]; then
		eval FPPMNAME='"$'$fp'"'
	fi
	if [ ! "$FPUINAME" ]; then
		eval FPUINAME='"$'$fp'"'
	fi

	if [ -z "$FPTARGETS" ]; then
		TARGETS="@ManagedNode:$MNNAME"
	else
		TARGETS=""
		for target in $FPTARGETS; do
			eval TARG='"$'$target'"'
			TARGETS="$TARGETS \"@ProfileManager:$TARG\""
		done
	fi

	#-#-#
	log_start_action `outmsg LinkInstallCat 113 "Delete view of File Package %1\\$s" "$FILEPACKNAME"`

	if [ -n "$FPCONFIG" ]; then
		#-#-#
		log_start_action `outmsg LinkInstallCat 115 "Removing OmniFilePackage \"%1\\$s\"" "$FILEPACKNAME"`

		# Delete the hidden collection (but not its parent)
		PARENT_NAME="TS Hidden Collections"
		if IsValidGenericCollection "$PARENT_NAME"; then
			PARENT_OBJ=`idlcall $COLLECTIONGUI lookup_object '"'$PARENT_NAME'" {0}'`
			if idlcall "$PARENT_OBJ" lookup_object '"'$FPUINAME'" {0}' > /dev/null; then
				HCOLL=`idlcall "$PARENT_OBJ" lookup_object '"'$FPUINAME'" {0}'`
				if [ "$HCOLL" ]; then
					objcall "$HCOLL" _delete
				fi
			fi
		fi

		# Delete the Link object
		delete_link_obj "$FPUINAME"

		#-#-#
		log_success `outmsg LinkInstallCat 116 "OmniFilePackage \"%1\\$s\" removal successful" "$FILEPACKNAME"`
	fi

	#-#-#
	log_end_action
done
refresh_plus_collection


}

######################################################################
#
# delete_infra_inventory
#
delete_infra_inventory() {
[ "$DEBUG" = true ] && set -x

PMNAME="$INV_PM"
R_PMNAME="${PMNAME}#${IRONAME}"

if wlookup -r InventoryProfile "$R_PMNAME" > /dev/null 2>&1; then
	wdel "@InventoryProfile:${R_PMNAME}"
fi
reregister_pm "$PMNAME"
if wlookup -r ProfileManager "$R_PMNAME" > /dev/null 2>&1; then
	wunsub -a "@ProfileManager:${R_PMNAME}"
	wdel "@ProfileManager:${R_PMNAME}"
fi

}

######################################################################
#
# delete_infra_sentries
#
delete_infra_sentries() {
  [ "$DEBUG" = true ] && set -x
  for i in $SENTRY_PMS; do
    comp_name="`eval echo $i`"
    eval prfname='"$'${comp_name}'"'
    eval pmgrname='"$'${comp_name}'_PARENT"'
    eval r_prfname='"$R_'${comp_name}'"'
    eval r_pmgrname='"$R_'${comp_name}'_PARENT"'

    if [ ! "$pmgrname" ]; then
      eval pmgrname='"$'${comp_name}'"'
      eval r_pmgrname='"$R_'${comp_name}'"'
    fi

    # make sure there are no imbedded spaces based on collection name
    prfname="`echo $prfname`"
    pmgrname="`echo $pmgrname`"
    r_prfname="`echo $r_prfname`"
    r_pmgrname="`echo $r_pmgrname`"

    if wlookup -r SentryProfile "$r_prfname" > /dev/null 2>&1; then
      wdel "@SentryProfile:${r_prfname}"
    fi
    reregister_pm "$pmgrname"
    if wlookup -r ProfileManager "$r_pmgrname" > /dev/null 2>&1; then
      wunsub -a "@ProfileManager:${r_pmgrname}"
      wdel "@ProfileManager:${r_pmgrname}"
    fi
  done

}

######################################################################
#
# delete_infra_mars
#
delete_infra_mars() {
  [ "$DEBUG" = true ] && set -x
  for i in $ARM_PMS; do
    comp_name="`eval echo $i`"
    eval prfname='"$'${comp_name}'"'
    eval r_prfname='"$R_'${comp_name}'"'
    eval pmgrname='"$'${comp_name}'_PARENT"'
    eval r_pmgrname='"$R_'${comp_name}'_PARENT"'

    if [ ! "$pmgrname" ]; then
      eval pmgrname='"$'${comp_name}'"'
      eval r_pmgrname='"$R_'${comp_name}'"'
    fi
    if wlookup -r MarProfile "$r_prfname" > /dev/null 2>&1; then
      wdel "@MarProfile:${r_prfname}"
    fi
    reregister_pm "$pmgrname"
    if wlookup -r ProfileManager "$r_pmgrname" > /dev/null 2>&1; then
      wunsub -a "@ProfileManager:${r_pmgrname}"
      if wdel "@ProfileManager:${r_pmgrname}" > /dev/null 2>&1; then
	:
      fi
      #wdel "@ProfileManager:${r_pmgrname}"
    fi
  done

}

######################################################################
#
# delete_infra_filepacks
#
delete_infra_filepacks() {
  [ "$DEBUG" = true ] && set -x
  for i in $FILEPACKS; do
    comp_name="`eval echo $i`"
    eval prfname='"$'${comp_name}'"'
    eval pmgrname='"$'${comp_name}'_PARENT"'
    eval r_prfname='"$R_'${comp_name}'"'
    eval r_pmgrname='"$R_'${comp_name}'_PARENT"'

    if [ ! "$pmgrname" ]; then
      eval pmgrname='"$'${comp_name}'"'
      eval r_pmgrname='"$R_'${comp_name}'"'
    fi

    if wlookup -r FilePackage "$r_prfname" > /dev/null 2>&1; then
      wdel "@FilePackage:${r_prfname}"
    fi
    reregister_pm "$pmgrname"
    if wlookup -r ProfileManager "$r_pmgrname" > /dev/null 2>&1; then
      wunsub -a "@ProfileManager:${r_pmgrname}"
      wdel "@ProfileManager:${r_pmgrname}"
    fi
  done
}

######################################################################
#
# delete_tasklib_content
#
delete_tasklib_content() {
  [ "$DEBUG" = true ] && set -x
  lib="$1"
  num_items=$2
  if [ ! "$num_items" -o $num_items -eq 0 ]; then
    return
  fi
  cmd=""
  while [ "true" ]; do
    read type name
    if [ "$type" = "(task)" ]; then
      cmd=wdeltask
    elif [ "$type" = "(job)" ]; then
      cmd=wdeljob
    fi
    if [ "$cmd" ]; then
      $cmd "$name" "$lib#$IRONAME"
    fi
    if [ $num_items -eq 1 ]; then
      return  #don't let expr make it go to 0; triggers a prob on NT
    fi
    num_items=`expr $num_items - 1`
  done
}

######################################################################
#
# delete_infra_tasks
#
delete_infra_tasks() {
  [ "$DEBUG" = true ] && set -x
  tasks_to_delete="$*"

  for t in $tasks_to_delete; do
    eval task_name="`echo '$'\`echo $t\``"
    if [ "$USE_TLL" = true ]; then
      task_name=`normalize_name "$task_name"`
    fi
    eval libname="`echo '$'${t}_LIBR`"
    r_libname="${libname}#${IRONAME}"

    if wlstlib "$r_libname" 2> /dev/null | grep "^(task)	${task_name}$" > /dev/null 2>& 1; then
      wdeltask "$task_name" "$r_libname"
    fi
  done
}

#######################################################################
#
# delete_default_tasklib
#
delete_default_tasklib() {

  # Handle the utility library that may exist in older toolkit plus modules;
  # we do this blindly because we don't know what's in it!

  if IsValidTaskLibrary "$DEFAULT_TASKLIB" ; then
    num_entries=`wlstlib "$DEFAULT_TASKLIB#$IRONAME" | wc -l`
    wlstlib "$DEFAULT_TASKLIB#$IRONAME" | delete_tasklib_content "$DEFAULT_TASKLIB" $num_entries
  fi
}

######################################################################
#
# delete_infra_tasklibs
#
delete_infra_tasklibs() {
  [ "$DEBUG" = true ] && set -x
  TASKLIB_DEFAULT=DEFAULT_TASKLIB  #some modules may use this variable ...
  for i in $TASKLIBS $TASKLIB_DEFAULT; do
    comp_name="`eval echo $i`"
    eval libname='"$'${comp_name}'"'
    r_libname="${libname}#${IRONAME}"

    if IsValidTaskLibrary "$r_libname"; then
      tasklib_contents=`wlstlib "$r_libname" 2> /dev/null`
      if [ "`echo $tasklib_contents`" ]; then
        errmsg LinkInstallCat 15 "  Error: \"%1\$s\" still has contents:" "$libname"
        wlstlib "$r_libname"
      else
        wdel @TaskLibrary:"${r_libname}"
      fi
    fi
  done
}

######################################################################
#
# delete_infra_app_pr
#
delete_infra_app_pr() {
  [ "$DEBUG" = true ] && set -x
  APP_LABEL="@PolicyRegion:${APP_PR_NAME}#${IRONAME}"
  if [ `wls "$APP_LABEL" 2> /dev/null | wc -l` -eq 0 ]; then
    if wls "$APP_LABEL" > /dev/null 2>&1; then
      wdel "$APP_LABEL"
    fi
  else
    errmsg LinkInstallCat 16 "Could not complete undo: \"%1\$s\" still has contents" "$APP_LABEL"
  fi
}

######################################################################
#
# reregister_pm
#
reregister_pm() {
  [ "$DEBUG" = true ] && set -x
  pm=$1
  if wlookup -r ProfileManager "${pm}#${IRONAME}" > /dev/null 2>&1; then
    :
  else
    PM_CLASS=`wlookup -r Classes ProfileManager`
    if idlcall $PM_CLASS lookup_object '"'$pm'" {0}' > /dev/null ; then
      pmobj=`idlcall $PM_CLASS lookup_object '"'$pm'" {0}'`
      if [ "$pmobj" ]; then
        wregister -r ProfileManager "${pm}" $pmobj > /dev/null 2>&1
      fi
    fi
  fi
}

######################################################################
#
# needs_replacement
#
needs_replacement() {
  [ "$DEBUG" = true ] && set -x
  tag="$1"
  installed=$2
  new=$3
  new_vers=""
  installed_vers=""
  
  set +e
  content=`ls ${installed}/${tag} 2> /dev/null`
  if [ $? -eq 0 -a "$content" ]; then
    installed_vers=`basename \`ls ${installed}/${tag} 2> /dev/null | sort -rf | (read c; echo $c)\` 2> /dev/null`
  fi

  content=`ls ${new}/${tag} 2> /dev/null`
  if [ $? -eq 0 -a "$content" ]; then
    new_vers=`basename \`ls ${new}/${tag} 2> /dev/null | sort -rf | (read c; echo $c)\` 2> /dev/null`
  fi
  set -e

  if [ "$installed_vers" = "." -o ! "$installed_vers" ]; then
    return 0
  elif expr "$new_vers" \> "$installed_vers" > /dev/null; then
    return 0
  else
    return 1
  fi
}
