/****************************************************************************
 * Copyright(c) 2000 Broadcom Corporation, all rights reserved
 * Proprietary and Confidential Information.
 *
 * This source file is the property of Broadcom Corporation, and
 * may not be copied or distributed in any isomorphic form without 
 * the prior written consent of Broadcom Corporation. 
 *
 * Name:        pal.c
 *
 * Description: Platform Abstraction Layer implementation
 *
 * Author:      Frankie Fan
 *
 * $Log:: /Source/basplnx/pal.c                                     $
 * 
 * 21    5/08/03 6:10p Ffan
 * Fixed recurrsive multicast filter set and "buggy protocol"
 * warning.
 * 
 * 20    5/05/03 7:20p Ffan
 * Fixed the multicast leakage bug caused by not removing the
 * multicast address on unbind.
 * 
 * 19    5/04/03 8:38p Ffan
 * Added NAPI support.
 * Added support of new MII ioctl.
 * 
 * 18    2/01/02 9:27a Ffan
 * 
 * 17    12/17/01 4:45p Ffan
 * Added support of IEEE 802.3ad Link Aggregation.
 * 
 * 16    8/03/01 9:45a Ffan
 * Cleaned up un-necessary user space resources in the spawned kernel
 * thread.
 * 
 * 15    6/28/01 3:31p Ffan
 * Fixed compilation problems with Red Hat 2.4.3 kernel for IA-64.
 * 
 * 14    5/24/01 6:34p Ffan
 * Added proc fs support.
 * Fixed server hang problem when the physical NIC driver is removed
 * before detaching the team.
 * 
 * 13    5/11/01 4:32p Ffan
 * Added change_mtu handler.
 * 
 * 12    2/10/01 8:06a Ffan
 * Adapted to the different path of the spinlock.h on different Linux
 * kernel.
 * 
 * 11    2/01/01 10:40a Ffan
 * Revert back to "asm/spinlock.h" for 2.2 kernel.
 * 
 * 10    2/01/01 10:14a Ffan
 * Used "linux/spinlock.h" instead of "asm/spinlock.h".
 * 
 * 9     1/15/01 2:50p Ffan
 * Ready for 2.4 kernel.
 * 
 * 8     1/10/01 5:44p Ffan
 * Getting ready for 2.4.0 kernel.
 * Added intercept of set_mac_address request.
 * 
 * 7     12/19/00 12:02p Ffan
 * Added NET_BH handler to send pending packets. This is necessary
 * because dev_queue_xmit() is only synchronized with bottom half.
 * 
 * 6     12/15/00 3:26p Ffan
 * 
 * 5     12/12/00 7:09p Ffan
 * Added pal_register_notifier, pal_unregister_notifier and
 * pal_stricmp functions.
 * 
 * 4     12/08/00 3:00p Ffan
 * Fixed zero size of spinlock_t problem with non-SMP build.
 * Fixed pal_read_mii() to correctly handle both older version of
 * bcm5700.o and third party drivers.
 * 
 * 3     12/05/00 6:30p Ffan
 * Fixed the IOCTL command to read MII register.
 * 
 * 2     12/04/00 6:26p Ffan
 * Changed pal_device_ioctl() to be more generic.
 * Added pal_read_mii() and pal_nice_cmd() to address dependency
 * problem on linux/if.h.
 * 
 * 1     12/01/00 5:55p Ffan
 * Initial check-in.
 *
 ****************************************************************************/
#include <linux/config.h>

#if defined(CONFIG_SMP) && !defined(__SMP__)
#define __SMP__
#endif  // defined(CONFIG_SMP) && !defined(__SMP__)

#include <linux/version.h>

// only one C file can include module headers with __NO_VERSION__
#if defined(MODULE)
#define __NO_VERSION__
#if defined(CONFIG_MODVERSIONS) && !defined(MODVERSIONS)
#define MODVERSIONS
#endif  // defined(CONFIG_MODVERSIONS) && !defined(MODVERSIONS)
#if defined(MODVERSIONS)
#include <linux/modversions.h>
#include <linux/module.h>
#else   // defined(CONFIG_MODVERSIONS)
#include <linux/module.h>
#endif  // defined(CONFIG_MODVERSIONS)
#endif  // defined(MODULE)

#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/types.h>
#include <linux/fcntl.h>
#include <linux/interrupt.h>        // ???
#include <linux/ptrace.h>           // ???
#include <linux/ioport.h>           // ???
#include <linux/in.h>
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,0)
#include <linux/slab.h>
#else
#include <linux/malloc.h>
#endif
#include <linux/string.h>
#include <linux/init.h>
#include <asm/system.h>
#include <asm/bitops.h>
#include <asm/io.h>                 // ???
#include <asm/dma.h>                // ???
#include <asm/uaccess.h>            // copy_[from|to]_user
#include <asm/atomic.h>
#include <linux/errno.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/inetdevice.h>
#include <linux/skbuff.h>
#include <linux/notifier.h>
#include <linux/string.h>
#include <linux/rtnetlink.h>
#include <linux/proc_fs.h>
#include <linux/smp_lock.h>

// version dependent includes
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,0)
#include <linux/spinlock.h>
#else
#include <asm/spinlock.h>
#endif

#include "nicext.h"
#include "pal.h"

//
// linux kernel version check
//
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,0)
#undef KERNEL_2_2
#elif LINUX_VERSION_CODE >= KERNEL_VERSION(2,2,0)
#define KERNEL_2_2 1
#else
#error BASP requires at least Linux 2.2.0 kernel
#endif

//
// abstracted types
//
#if defined(KERNEL_2_2)
typedef struct device netdevice_t;
typedef struct wait_queue* event_t;
#else
typedef struct net_device netdevice_t;
typedef wait_queue_head_t event_t;
#endif // defined(KERNEL_2_2)

//
// compatibility macros
//
#if defined(KERNEL_2_2)
#define init_waitqueue_head(ev)     *(ev) = NULL
#define dev_get_by_name(dev)        dev_get(dev)
#endif // defined(KERNEL_2_2)

//
// NAPI support
//
//#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,20)
#if HAVE_NETIF_RECEIVE_SKB
#define NAPI_SUPPORT 1
#endif

//
// Resource Pool
//
#define PALI_NUM_DESCRIPTOR_PER_POOL        16
typedef struct _PALI_RESOURCE_POOL
{
    U32 rp_signature;                       // id for debugging purpose
    U32 rp_num;                             // number of resource in this pool
    struct _PALI_RESOURCE_POOL* rp_next;    // singly link-list for recycle
    PAL_RESOURCE_DESCRIPTOR rp_desc[PALI_NUM_DESCRIPTOR_PER_POOL];
    union                                   // array of resource descriptor
    {
        spinlock_t              f_spinlock[PALI_NUM_DESCRIPTOR_PER_POOL];
        atomic_t                f_atomic[PALI_NUM_DESCRIPTOR_PER_POOL];
        netdevice_t             f_device[PALI_NUM_DESCRIPTOR_PER_POOL];
        struct net_device_stats f_stat[PALI_NUM_DESCRIPTOR_PER_POOL];
        struct sk_buff_head     f_head[PALI_NUM_DESCRIPTOR_PER_POOL];
    } u;
} PALI_RESOURCE_POOL;

// macro to get the base address of the actual memory allocated for 
// the resources
#define PALI_RP_BASE( _prp )                                    \
    (U8*)(&((_prp)->u.f_spinlock[0]))

// macro to get the offset of element
#define PALI_RP_OFFSET( _array, _i )                            \
    (U32)( (U8*)(&(((PALI_RESOURCE_POOL*)0)->u.f_##_array[_i])) - \
           PALI_RP_BASE((PALI_RESOURCE_POOL*)0) )

// macro to get the address of the pool given the base address
#define PALI_RP_POOL( _base )                                   \
    ( (PALI_RESOURCE_POOL*)((U8*)(_base) -                      \
      (U8*)(&((PALI_RESOURCE_POOL*)0)->u.f_spinlock[0])) )

//
// Resource Bundle Pool
typedef struct _PALI_RESOURCE_BUNDLE_POOL
{
    U32 rp_signature;                       // id for debugging purpose
    U32 rp_num;                             // number of resource in this pool
    struct _PALI_RESOURCE_POOL* rp_next;    // singly link-list for recycle
    PAL_RESOURCE_DESCRIPTOR rp_desc[1360];  // this must be greater than the 
                                            // total of below
    spinlock_t              f_spinlock[350];
    atomic_t                f_atomic[700];
    netdevice_t             f_device[70];
    struct net_device_stats f_stat[70];
    struct sk_buff_head     f_head[10];
    event_t                 f_event[150];
} PALI_RESOURCE_BUNDLE_POOL;

// macro to get the base address of the actual memory allocated for 
// the resources
#define PALI_RBP_BASE( _prp )                                   \
    (U8*)(&((_prp)->f_spinlock[0]))

// macro to get the offset of element
#define PALI_RBP_OFFSET( _array, _i )                           \
    (U32)( (U8*)(&(((PALI_RESOURCE_BUNDLE_POOL*)0)->f_##_array[_i])) - \
           PALI_RBP_BASE((PALI_RESOURCE_BUNDLE_POOL*)0) )

//
// PALI_NOTIFIER
//
typedef struct _PALI_NOTIFIER
{
    // 1st member must be struct notifier_block
    struct notifier_block nb;
    // our stuff follows
    int (*fp_notify)( PAL_DEVICE dev, PAL_NOTIFYING_EVENT evt );
} PALI_NOTIFIER;

//
// global variables
//
static PALI_RESOURCE_POOL* pools = NULL;
static spinlock_t pool_lock = SPIN_LOCK_UNLOCKED;
static spinlock_t print_lock = SPIN_LOCK_UNLOCKED;
static char msg[4096];

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
void 
pal_init( 
    void 
    )
{
    spin_lock_init( &pool_lock );
}

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
void 
pal_fini( 
    void 
    )
{
    PALI_RESOURCE_POOL* rp;

    while( pools )
    {
        rp = pools;
        pools = rp->rp_next;
        kfree( rp );
    }   // while( *prp )
}

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
int
pal_alloc_resource( 
    U32 res_type,
    U8** base,
    PAL_RESOURCE_DESCRIPTOR** desc,
    U32* num
    )
{
    PALI_RESOURCE_POOL* rp;
    U32 i;
    unsigned long flags;

    if( res_type >= PRT_MAXIMUM )
    {
        return -EINVAL;
    }   // if( res_type >= PRT_MAXIMUM )

    else if( res_type == PRT_BUNDLE )
    {
        return pal_alloc_bundle_resource( base, desc, num );
    }

    rp = (PALI_RESOURCE_POOL*)kmalloc( sizeof(*rp), GFP_KERNEL );
    if( rp == NULL )
    {
        return -ENOMEM;
    }   // if( rp == NULL )
    memset( rp, 0, sizeof(*rp) );

    // initialize
    rp->rp_signature = 0x414C5000;
    rp->rp_num = PALI_NUM_DESCRIPTOR_PER_POOL;

    // fill the offset
    for( i = 0; i < PALI_NUM_DESCRIPTOR_PER_POOL; ++i )
    {
        rp->rp_desc[i].rd_type = res_type;

        switch( res_type )
        {
            case PRT_SPINLOCK:
                rp->rp_desc[i].rd_offset = PALI_RP_OFFSET(spinlock, i);
                rp->rp_desc[i].rd_size = sizeof(spinlock_t);
                break;

            case PRT_ATOMIC:
                rp->rp_desc[i].rd_offset = PALI_RP_OFFSET(atomic, i);
                rp->rp_desc[i].rd_size = sizeof(atomic_t);
                break;

            case PRT_DEVICE:
                rp->rp_desc[i].rd_offset = PALI_RP_OFFSET(device, i);
                rp->rp_desc[i].rd_size = sizeof(netdevice_t);
                break;

            case PRT_DEVICE_STAT:
                rp->rp_desc[i].rd_offset = PALI_RP_OFFSET(stat, i);
                rp->rp_desc[i].rd_size = sizeof(struct net_device_stats);
                break;

            case PRT_PACKET_LIST:
                rp->rp_desc[i].rd_offset = PALI_RP_OFFSET(head, i);
                rp->rp_desc[i].rd_size = sizeof(struct sk_buff_head);
                break;
        }   // switch( res_type )
    }   // for( i = 0; 

    // add this pool into our list
    spin_lock_irqsave( &pool_lock, flags );

    rp->rp_next = pools;
    pools = rp;

    spin_unlock_irqrestore( &pool_lock, flags );

    // set return parameters
    *base = PALI_RP_BASE( rp );
    *desc = rp->rp_desc;
    *num = PALI_NUM_DESCRIPTOR_PER_POOL;

    return 0;
}

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
int
pal_alloc_bundle_resource( 
    U8** base,
    PAL_RESOURCE_DESCRIPTOR** desc,
    U32* num
    )
{
    PALI_RESOURCE_BUNDLE_POOL* rbp;
    U32 i;
    U32 ridx;
    U32 total;
    unsigned long flags;

    rbp = (PALI_RESOURCE_BUNDLE_POOL*)kmalloc( sizeof(*rbp), GFP_KERNEL );
    if( rbp == NULL )
    {
        return -ENOMEM;
    }   // if( rp == NULL )
    memset( rbp, 0, sizeof(*rbp) );

    // initialize
    rbp->rp_signature = 0x414C5000;
    rbp->rp_num = sizeof(rbp->rp_desc) / sizeof(rbp->rp_desc[0]);

    // zero index
    ridx = 0;

    // fill in the offsets of spinlock array
    if( sizeof(rbp->f_spinlock[0]) )
    {
        total = sizeof(rbp->f_spinlock) / sizeof(rbp->f_spinlock[0]);
    }
    else
    {
        total = 300;
    }

    for( i = 0; i < total; ++i, ++ridx )
    {
        rbp->rp_desc[ridx].rd_type = PRT_SPINLOCK;
        rbp->rp_desc[ridx].rd_offset = PALI_RBP_OFFSET(spinlock, i);
        rbp->rp_desc[ridx].rd_size = sizeof(spinlock_t);
    }

    // fill in the offsets of atomic array
    total = sizeof(rbp->f_atomic) / sizeof(rbp->f_atomic[0]);
    for( i = 0; i < total; ++i, ++ridx )
    {
        rbp->rp_desc[ridx].rd_type = PRT_ATOMIC;
        rbp->rp_desc[ridx].rd_offset = PALI_RBP_OFFSET(atomic, i);
        rbp->rp_desc[ridx].rd_size = sizeof(atomic_t);
    }

    // fill in the offsets of device array
    total = sizeof(rbp->f_device) / sizeof(rbp->f_device[0]);
    for( i = 0; i < total; ++i, ++ridx )
    {
        rbp->rp_desc[ridx].rd_type = PRT_DEVICE;
        rbp->rp_desc[ridx].rd_offset = PALI_RBP_OFFSET(device, i);
        rbp->rp_desc[ridx].rd_size = sizeof(netdevice_t);
    }

    // fill in the offsets of device stat array
    total = sizeof(rbp->f_stat) / sizeof(rbp->f_stat[0]);
    for( i = 0; i < total; ++i, ++ridx )
    {
        rbp->rp_desc[ridx].rd_type = PRT_DEVICE_STAT;
        rbp->rp_desc[ridx].rd_offset = PALI_RBP_OFFSET(stat, i);
        rbp->rp_desc[ridx].rd_size = sizeof(struct net_device_stats);
    }

    // fill in the offsets of device stat array
    total = sizeof(rbp->f_head) / sizeof(rbp->f_head[0]);
    for( i = 0; i < total; ++i, ++ridx )
    {
        rbp->rp_desc[ridx].rd_type = PRT_PACKET_LIST;
        rbp->rp_desc[ridx].rd_offset = PALI_RBP_OFFSET(head, i);
        rbp->rp_desc[ridx].rd_size = sizeof(struct sk_buff_head);
    }

    // fill in the offsets of wait_queue_head_t array
    total = sizeof(rbp->f_event) / sizeof(rbp->f_event[0]);
    for( i = 0; i < total; ++i, ++ridx )
    {
        rbp->rp_desc[ridx].rd_type = PRT_EVENT;
        rbp->rp_desc[ridx].rd_offset = PALI_RBP_OFFSET(event, i);
        rbp->rp_desc[ridx].rd_size = sizeof(event_t);
    }

    // add this pool into our list
    spin_lock_irqsave( &pool_lock, flags );

    rbp->rp_next = pools;
    pools = (PALI_RESOURCE_POOL*)rbp;

    spin_unlock_irqrestore( &pool_lock, flags );

    // set return parameters
    *base = PALI_RBP_BASE( rbp );
    *desc = rbp->rp_desc;
    *num = rbp->rp_num;

    return 0;
}

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
int
pal_free_resource(
    U8* base
    )
{
    PALI_RESOURCE_POOL* rp;
    PALI_RESOURCE_POOL** prp;
    int found;
    unsigned long flags;

    // validate the resource 
    if( base == NULL )
    {
        return -EINVAL;
    }   // if( base == NULL )

    rp = PALI_RP_POOL( base );

    if( rp == NULL || rp->rp_signature != 0x414C5000 )
    {
        return -EINVAL;
    }   // if( rp == NULL )

    // unchain from the pool   
    spin_lock_irqsave( &pool_lock, flags );

    prp = &pools;
    found = 0;

    while( *prp )
    {
        if( *prp == rp )
        {
            found = 1;
            *prp = rp->rp_next;
            break;
        }   // if( *prp == rp )

        prp = &((*prp)->rp_next);
    }   // while( *prp )

    spin_unlock_irqrestore( &pool_lock, flags );

    if( found == 0 )
    {
        return -EINVAL;
    }

    kfree( rp );

    return 0;
}

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
long 
pal_wait_event(
    PAL_EVENT* event,
    signed long ms
    )
{
    if( sleep_on_timeout( (event_t*)event, ms ) == 0 )
        return 0;
    else
        return 1;
}

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
void
pal_set_event(
    PAL_EVENT* event
    )
{
    wake_up( (event_t*)event );
}

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
void
pal_init_event(
    PAL_EVENT* event
    )
{
    init_waitqueue_head( (event_t*)event );
}

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
PAL_TIME
pal_get_current_time(
    void
    )
{
    return jiffies;
}

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
U32
pal_diff_hms(
    PAL_TIME tstart,
    PAL_TIME tend
    )
{
    if( tend > tstart )
    {
        return (U32)( (tend - tstart) * 10 / HZ );
    }
    else
    {
        return (U32)( ((PAL_TIME)(-1) - tstart + tend) * 10 / HZ );
    }
}

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
PAL_TIME
pal_advance_second(
    PAL_TIME tstart,
    unsigned long second
    )
{
    return tstart + (PAL_TIME)(second) * HZ;
}

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
void*
pal_memcpy(
    void* dst,
    void* src,
    U32 len
    )
{
    return memcpy( dst, src, len );
}


/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
int
pal_memcmp(
    void* dst,
    void* src,
    U32 len
    )
{
    return memcmp( dst, src, len );
}


/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
void*
pal_memset(
    void* dst,
    int val,
    U32 len
    )
{
    return memset( dst, val, len );
}

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
void*
pal_malloc(
    U32 len
    )
{
    return kmalloc( len, GFP_KERNEL );
}

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
void
pal_mfree(
    void* p
    )
{
    kfree( p );
}

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
void
pal_copy_from_user(
    void* to,
    void* from,
    U32 len
    )
{
    copy_from_user( to, from, len );
}

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
void
pal_copy_to_user(
    void* to,
    void* from,
    U32 len
    )
{
    copy_to_user( to, from, len );
}

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
int
pal_stricmp(
    char* str1,
    char* str2
    )
{
    int rc;

    rc = strlen( str1 ) - strlen( str2 );
    if( rc == 0 )
    {
        rc = strcmp( str1, str2 );
    }

    return rc;
}

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
int
pal_atomic_read(
    PAL_ATOMIC atom
    )
{
    return atomic_read( (atomic_t*)atom );
}

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
void
pal_atomic_set(
    PAL_ATOMIC atom,
    int val
    )
{
    atomic_set( (atomic_t*)atom, val );
}

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
void
pal_atomic_inc(
    PAL_ATOMIC atom
    )
{
    atomic_inc( (atomic_t*)atom );
}

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
void
pal_atomic_dec(
    PAL_ATOMIC atom
    )
{
    atomic_dec( (atomic_t*)atom );
}

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
void
pal_atomic_add(
    PAL_ATOMIC atom,
    int val
    )
{
    atomic_add( val, (atomic_t*)atom );
}

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
PAL_PACKET
pal_alloc_packet(
    U32 size,
    PAL_DEVICE dev
    )
{
    struct sk_buff* skb;
    skb = dev_alloc_skb( size );
    skb->dev = (netdevice_t*)dev;
    skb->mac.raw = skb->data;
    skb->nh.raw = skb->mac.raw + ETH_HLEN;
    return (PAL_PACKET)skb;
}

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
void
pal_free_packet(
    PAL_PACKET pkt
    )
{
    dev_kfree_skb( (struct sk_buff*)pkt );
}

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
void
pal_init_packet_list(
    PAL_PACKET_LIST list
    )
{
    skb_queue_head_init( (struct sk_buff_head*)list );
}

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
void
pal_enqueue_packet(
    PAL_PACKET_LIST list,
    PAL_PACKET pkt
    )
{
    skb_queue_head( (struct sk_buff_head*)list, (struct sk_buff*)pkt );
}

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
PAL_PACKET
pal_dequeue_packet(
    PAL_PACKET_LIST list
    )
{
    struct sk_buff* skb;

    skb = skb_dequeue( (struct sk_buff_head*)list );
    return (PAL_PACKET)skb;
}

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
void
pal_set_packet_device(
    PAL_PACKET pkt,
    PAL_DEVICE dev
    )
{
    ((struct sk_buff*)pkt)->dev = (netdevice_t*)dev;
}

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
PAL_DEVICE
pal_get_packet_device(
    PAL_PACKET pkt
    )
{
    return (PAL_DEVICE)(((struct sk_buff*)pkt)->dev);
}

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
void
pal_set_packet_protocol(
    PAL_PACKET pkt,
    U16 type
    )
{
    ((struct sk_buff*)pkt)->protocol = type;
}

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
U16
pal_get_packet_protocol(
    PAL_PACKET pkt
    )
{
    return ((struct sk_buff*)pkt)->protocol;
}


/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
void
pal_send_packet(
    PAL_PACKET pkt
    )
{
    dev_queue_xmit( (struct sk_buff*)pkt );
}

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
U8*
pal_get_packet_reserved(
    PAL_PACKET pkt
    )
{
    return ((struct sk_buff*)pkt)->cb;
}

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
void
pal_indicate_receive_packet(
    PAL_PACKET pkt
    )
{
    netif_rx( (struct sk_buff*)pkt );
}


/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
void
pal_indicate_receive_napi(
    PAL_PACKET pkt
    )
{
#ifdef NAPI_SUPPORT
    netif_receive_skb( (struct sk_buff*)pkt );
#else
    netif_rx( (struct sk_buff*)pkt );
#endif
}


/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
U32
pal_get_packet_len(
    PAL_PACKET pkt
    )
{
    return ((struct sk_buff*)pkt)->len;
}

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
U8*
pal_get_packet_data(
    PAL_PACKET pkt
    )
{
    return ((struct sk_buff*)pkt)->data;
}

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
U8*
pal_get_packet_raw(
    PAL_PACKET pkt
    )
{
    return ((struct sk_buff*)pkt)->mac.raw;
}

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
U8*
pal_packet_put(
    PAL_PACKET pkt,
    unsigned int len
    )
{
    return skb_put( (struct sk_buff*)pkt, len );
}

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
void
pal_init_spinlock(
    PAL_SPINLOCK lock
    )
{
    spin_lock_init( (spinlock_t*)lock );
}

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
void
pal_acquire_spinlock(
    PAL_SPINLOCK lock,
    unsigned long* flags
    )
{
    spin_lock_irqsave( (spinlock_t*)lock, *flags );
}

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
void
pal_release_spinlock(
    PAL_SPINLOCK lock,
    unsigned long* flags
    )
{
    spin_unlock_irqrestore( (spinlock_t*)lock, *flags );
}

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
void
pal_start_bh_atomic(
    void
    )
{
#if defined(KERNEL_2_2)
    start_bh_atomic();
#endif	// defined(KERNEL_2_2)
}

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
void
pal_end_bh_atomic(
    void
    )
{
#if defined(KERNEL_2_2)
    end_bh_atomic();
#endif // defined(KERNEL_2_2)
}

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
void
pal_hook_net_bh(
    fp_hook* orig_net_bh,
    fp_hook new_net_bh
    )
{
#if defined(KERNEL_2_2)
    *orig_net_bh = bh_base[NET_BH];
    init_bh( NET_BH, new_net_bh );
#endif // defined(KERNEL_2_2)
}

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
void
pal_mark_net_bh(
    void
    )
{
#if defined(KERNEL_2_2)
    mark_bh( NET_BH );
#endif // defined(KERNEL_2_2)
}

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
void 
pal_print(
    const char* fmt,
    ...
    )
{
    va_list ap;
    unsigned long flags;

    va_start( ap, fmt );
    spin_lock_irqsave( &print_lock, flags );
    vsprintf( msg, fmt, ap );
    printk( KERN_INFO "%s", msg );
    spin_unlock_irqrestore( &print_lock, flags );
    va_end( ap );
}

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
void
pal_panic(
    const char* msg
    )
{
    panic( msg );
}

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
int
pal_sprintf(
    char* buf,
    const char* fmt,
    ...
    )
{
    int len;
    va_list ap;
    va_start( ap, fmt );
    len = vsprintf( buf, fmt, ap );
    va_end( ap );
    return len;
}

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
int
pal_device_ioctl(
    PAL_DEVICE dev,
    void* buf,
    int buflen,
    int cmd
    )
{
    struct ifreq ifr;
    int rc;

    if( ((netdevice_t*)dev)->do_ioctl == NULL )
    {
        return -EOPNOTSUPP;
    }

    if( buflen > sizeof(ifr.ifr_ifru) )
    {
        return -E2BIG;
    }

    memcpy( &ifr.ifr_ifru, buf, buflen );

    rc = ((netdevice_t*)dev)->do_ioctl( (netdevice_t*)dev, buf, cmd );

    if( rc == 0 )
    {
        memcpy( buf, &ifr.ifr_ifru, buflen );
    }

    return rc;
}

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
int
pal_is_device_up(
    PAL_DEVICE dev
    )
{
    if( ( ((netdevice_t*)dev)->flags & IFF_UP ) == 
        IFF_UP )
    {
        return 1;
    }
    else
    {
        return 0;
    }
}

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
void
pal_set_device_flags(
    PAL_DEVICE dev
    )
{
    ((netdevice_t*)dev)->flags |= ( IFF_MASTER | IFF_MULTICAST );
}

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
void
pal_set_device_tx_queue_len(
    PAL_DEVICE dev,
    int len
    )
{
    ((netdevice_t*)dev)->tx_queue_len = len;
}

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
char*
pal_get_device_name(
    PAL_DEVICE dev
    )
{
    return ((netdevice_t*)dev)->name;
}

char*
pal_get_if_name(
    PAL_DEVICE dev
    )
{
    return ((struct in_ifaddr*)dev)->ifa_label;
}

U32
pal_get_ip_addr(
    PAL_DEVICE dev
    )
{
    return ((struct in_ifaddr*)dev)->ifa_address;
}

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
void
pal_set_device_name(
    PAL_DEVICE dev,
    char* name
    )
{
#if defined(KERNEL_2_2)
    ((netdevice_t*)dev)->name = name;
#else
    strncpy( ((netdevice_t*)dev)->name, name, IFNAMSIZ-1 );
#endif // defined(KERNEL_2_2)
}

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
int
pal_register_device(
    PAL_DEVICE dev
    )
{
    int rc;

#if defined(KERNEL_2_2)

    rc = register_netdevice( (netdevice_t*)dev );

#else

    rtnl_lock();
    rc = register_netdevice( (netdevice_t*)dev );
    rtnl_unlock();

#endif // !defined(KERNEL_2_2)

    return rc;
}

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
int
pal_unregister_device(
    PAL_DEVICE dev
    )
{
    int rc;

#if defined(KERNEL_2_2)

    rc = unregister_netdevice( (netdevice_t*)dev );

#else

    rtnl_lock();
    rc = unregister_netdevice( (netdevice_t*)dev );
    rtnl_unlock();

#endif // !defined(KERNEL_2_2)

    return rc;
}

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
PAL_DEVICE
pal_get_device_by_name(
    char* name
    )
{
    return (PAL_DEVICE)dev_get_by_name( name );
}

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
int
pal_is_device_slave(
    PAL_DEVICE dev
    )
{
    if( ((netdevice_t*)dev)->flags & IFF_SLAVE ) 
    {
        return 1;
    }
    else
    {
        return 0;
    }
}

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
void
pal_mark_device_slave(
    PAL_DEVICE dev,
    PAL_DEVICE master
    )
{
#if defined(KERNEL_2_2)

    ((netdevice_t*)dev)->slave = (netdevice_t*)master;
    ((netdevice_t*)dev)->flags |= IFF_SLAVE;

#else

    rtnl_lock();
    netdev_set_master( (netdevice_t*)dev, (netdevice_t*)master );
    rtnl_unlock();

#endif // !defined(KERNEL_2_2)

}

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
void
pal_unmark_device_slave(
    PAL_DEVICE dev,
    int hold_lock
    )
{
#if defined(KERNEL_2_2)

    ((netdevice_t*)dev)->flags &= ~IFF_SLAVE;
    ((netdevice_t*)dev)->slave = NULL;

#else

    if( !hold_lock )
    {
        rtnl_lock();
    }

    netdev_set_master( (netdevice_t*)dev, (netdevice_t*)NULL );

    if( !hold_lock )
    {
        rtnl_unlock();
    }

#endif // !defined(KERNEL_2_2)
}

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
void
pal_zero_device(
    PAL_DEVICE dev
    )
{
    memset( dev, 0 , sizeof(netdevice_t) );
}

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
void
pal_zero_device_stat(
    PAL_DEVICE_STAT stat
    )
{
    memset( (void*)stat, 0, sizeof(struct net_device_stats) );
}

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
void
pal_accumulate_device_stat(
    PAL_DEVICE dev_in,
    PAL_DEVICE_STAT stat_in
    )
{
    netdevice_t* dev = (netdevice_t*)dev_in;
    struct net_device_stats* stat = (struct net_device_stats*)stat_in;
    struct net_device_stats* mif_stat;

    if( dev->get_stats == NULL )
    {
        return;
    }

    mif_stat = dev->get_stats( dev );

    // add them up
    stat->rx_packets += mif_stat->rx_packets;
    stat->rx_bytes += mif_stat->rx_bytes;
    stat->rx_errors += mif_stat->rx_errors;
    stat->rx_dropped += mif_stat->rx_dropped;

    stat->tx_packets += mif_stat->tx_packets;
    stat->tx_bytes += mif_stat->tx_bytes;
    stat->tx_errors += mif_stat->tx_errors;
    stat->tx_dropped += mif_stat->tx_dropped;

    stat->multicast += mif_stat->multicast;
    stat->collisions += mif_stat->collisions;

    stat->rx_length_errors += mif_stat->rx_length_errors;
    stat->rx_over_errors += mif_stat->rx_over_errors;
    stat->rx_crc_errors += mif_stat->rx_crc_errors;
    stat->rx_frame_errors += mif_stat->rx_frame_errors;
    stat->rx_fifo_errors += mif_stat->rx_fifo_errors;	
    stat->rx_missed_errors += mif_stat->rx_missed_errors;

    stat->tx_aborted_errors += mif_stat->tx_aborted_errors;
    stat->tx_carrier_errors += mif_stat->tx_carrier_errors;
    stat->tx_fifo_errors += mif_stat->tx_fifo_errors;
    stat->tx_heartbeat_errors += mif_stat->tx_heartbeat_errors;
    stat->tx_window_errors += mif_stat->tx_window_errors;
}

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
void
pal_set_device_stat(
    PAL_DEVICE_STAT stat_in,
    int tx_packets,
    int rx_packets
    )
{
    struct net_device_stats* stat = (struct net_device_stats*)stat_in;
    stat->tx_packets = tx_packets;
    stat->rx_packets = rx_packets;
}

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
void
pal_set_device_private(
    PAL_DEVICE dev,
    void* ptr
    )
{
    ((netdevice_t*)dev)->priv = ptr;
}

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
void*
pal_get_device_private(
    PAL_DEVICE dev
    )
{
    return ((netdevice_t*)dev)->priv;
}

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
void
pal_set_device_mac_address(
    PAL_DEVICE dev,
    U8* mac_addr
    )
{
    memcpy( ((netdevice_t*)dev)->dev_addr, mac_addr, 6 );
}

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
U8*
pal_get_device_mac_address(
    PAL_DEVICE dev
    )
{
    return &((netdevice_t*)dev)->dev_addr[0];
}

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
int
pal_invoke_device_set_mac_address(
    PAL_DEVICE dev,
    U8* mac_addr,
    void* set_mac_address
    )
{
    int (* fp)(netdevice_t*, void*);
    struct sockaddr addr;

    if( set_mac_address == NULL )
    {
        fp = ((netdevice_t*)dev)->set_mac_address;
    }
    else
    {
        fp = (int (*)(netdevice_t*, void*))set_mac_address;
    }

    if( fp )
    {
        memcpy( addr.sa_data, mac_addr, 6 );
        return fp( (netdevice_t*)dev, &addr );
    }
    else
    {
        return -EOPNOTSUPP;
    }
}

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
int
pal_set_mac_address_callback(
    netdevice_t* dev,
    void* addr
    )
{
    return -EOPNOTSUPP;
}

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
int
pal_intercept_device_set_mac_address(
    PAL_DEVICE dev,
    void* new_set_mac_address,
    void** orig_set_mac_address
    )
{
    *orig_set_mac_address = ((netdevice_t*)dev)->set_mac_address;
    if( new_set_mac_address )
    {
        ((netdevice_t*)dev)->set_mac_address = new_set_mac_address;
    }
    else
    {
        ((netdevice_t*)dev)->set_mac_address = pal_set_mac_address_callback;
    }

    return 0;
}

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
int
pal_invoke_device_open(
    PAL_DEVICE dev
    )
{
    return dev_open( (netdevice_t*)dev );
}

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
int
pal_invoke_device_close(
    PAL_DEVICE dev
    )
{
    return dev_close( (netdevice_t*)dev );
}

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
void
pal_set_device_handlers(
    PAL_DEVICE dev,
    int (*init)(PAL_DEVICE),
    int (*open)(PAL_DEVICE),
    int (*stop)(PAL_DEVICE),
    int (*hard_start_xmit)(PAL_PACKET, PAL_DEVICE),
    void (*set_multicast_list)(PAL_DEVICE),
    int (*do_ioctl)(PAL_DEVICE, void*, int),
    PAL_DEVICE_STAT (*get_stats)(PAL_DEVICE),
    int (*change_mtu)(PAL_DEVICE, int)
    )
{
    ((netdevice_t*)dev)->init = (int (*)(netdevice_t*))init;
    ((netdevice_t*)dev)->open = (int (*)(netdevice_t*))open;
    ((netdevice_t*)dev)->stop = (int (*)(netdevice_t*))stop;
    ((netdevice_t*)dev)->hard_start_xmit = 
        (int (*)(struct sk_buff*, netdevice_t*))hard_start_xmit;
    ((netdevice_t*)dev)->set_multicast_list = 
        (void (*)(netdevice_t*))set_multicast_list;
    ((netdevice_t*)dev)->set_mac_address = pal_set_mac_address_callback;
    ((netdevice_t*)dev)->do_ioctl = 
        (int (*)(netdevice_t*, struct ifreq*, int))do_ioctl;
    ((netdevice_t*)dev)->get_stats = 
        (struct net_device_stats* (*)(netdevice_t*))get_stats;   
    ((netdevice_t*)dev)->change_mtu = 
        (int (*)(netdevice_t*, int))change_mtu;   
}

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
void
pal_dec_device_reference(
    PAL_DEVICE dev
    )
{
#if !defined(KERNEL_2_2)
    dev_put( (netdevice_t*)dev );
#endif  // !defined(KERNEL_2_2)
}

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
void
pal_inc_device_reference(
    PAL_DEVICE dev
    )
{
#if !defined(KERNEL_2_2)
    dev_hold( (netdevice_t*)dev );
#endif  // !defined(KERNEL_2_2)
}

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
void
pal_ether_setup(
    PAL_DEVICE dev
    )
{
    ether_setup( (netdevice_t*)dev );
}


/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
int
pal_get_device_mtu(
    PAL_DEVICE dev
    )
{
    return ((netdevice_t*)dev)->mtu;
}


/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
void
pal_set_device_mtu(
    PAL_DEVICE dev,
    int mtu
    )
{
    ((netdevice_t*)dev)->mtu = mtu;
}


/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
int
pal_set_device_allmulticast(
    PAL_DEVICE dev,
    int inc
    )
{
    netdevice_t* nd = (netdevice_t*)dev;

    dev_set_allmulti( nd, inc );

    if( (inc>0) && !(nd->flags & IFF_ALLMULTI) )
        return -EOPNOTSUPP;
    else
        return 0;
}


/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
int
pal_add_device_multicast(
    PAL_DEVICE dev,
    char* addr,
    int addr_len
    )
{
    return dev_mc_add( (netdevice_t*)dev, addr, addr_len, 1 );
}


/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
int
pal_del_device_multicast(
    PAL_DEVICE dev,
    char* addr,
    int addr_len
    )
{
    return dev_mc_delete( (netdevice_t*)dev, addr, addr_len, 1 );
}


/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
int
pal_read_mii(
    PAL_DEVICE dev,
    U16 reg,
    U16* value
    )
{
    int rc;   
    struct ifreq ifr;
    U16* data;

    if( ((netdevice_t*)dev)->do_ioctl == NULL )
    {
        return -EOPNOTSUPP;
    }

    // setup the pointer
    data = (U16*)&ifr.ifr_data;

    // get MII register
    data[1] = reg;

#ifdef SIOCGMIIPHY
    // get PHY address using new IOCTL
    rc = ((netdevice_t*)dev)->do_ioctl( ((netdevice_t*)dev), &ifr, 
                                          SIOCGMIIPHY );
    if( rc != 0 )
#endif // SIOCGMIIPHY
    {
        // get PHY address
        rc = ((netdevice_t*)dev)->do_ioctl( ((netdevice_t*)dev), &ifr, 
                                              SIOCDEVPRIVATE );
        if( rc )
        {
            *value = 0;
            return rc;
        }
    }

#ifdef SIOCGMIIPHY
    // get register value using new IOCTL
    rc = ((netdevice_t*)dev)->do_ioctl( ((netdevice_t*)dev), &ifr, 
                                          SIOCGMIIREG );
    if( rc != 0 )
#endif // SIOCGMIIPHY
    {
        // get register value
        rc = ((netdevice_t*)dev)->do_ioctl( ((netdevice_t*)dev), &ifr, 
                                              SIOCDEVPRIVATE+1 );
    }

    if( rc == 0 )
    {
        *value = data[3];
    }
    else
    {
        *value = 0;
    }

    return rc;
}

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
int
pal_nice_cmd(
    PAL_DEVICE dev,
    void* buf,
    int buflen
    )
{
    int rc;   
    struct ifreq ifr;

    if( ((netdevice_t*)dev)->do_ioctl == NULL )
    {
        return -EOPNOTSUPP;
    }

    // copy the buffer data
    if( buflen > sizeof(ifr.ifr_ifru) )
    {
        return -E2BIG;
    }

    memcpy( &ifr.ifr_ifru, buf, buflen );

    rc = ((netdevice_t*)dev)->do_ioctl( (netdevice_t*)dev, &ifr, 
                                          SIOCNICE );

    if( rc == 0 )
    {
        memcpy( buf, &ifr.ifr_ifru, buflen );
    }

    return rc;
}

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
int
pal_notifier_callback(
    struct notifier_block* nb,
    unsigned long sysevt,
    void* arg
    )
{
    PAL_NOTIFYING_EVENT evt;

    switch( sysevt )
    {
        case NETDEV_UP:
            evt = PNV_UP;
            break;

        case NETDEV_DOWN:
            evt = PNV_DOWN;
            break;

        case NETDEV_REGISTER:
            evt = PNV_REGISTER;
            break;

        case NETDEV_UNREGISTER:
            evt = PNV_UNREGISTER;
            break;

        case NETDEV_CHANGE:
            evt = PNV_CHANGE;
            break;

        case NETDEV_CHANGEMTU:
            evt = PNV_CHANGEMTU;
            break;

       default:
            // discard
            return 0;
    }
 
    return ((PALI_NOTIFIER*)nb)->fp_notify( (PAL_DEVICE)arg, evt );
}

int
pal_inetaddr_notifier_callback(
    struct notifier_block* nb,
    unsigned long sysevt,
    void* arg
    )
{
    PAL_NOTIFYING_EVENT evt;

    switch( sysevt )
    {
        case NETDEV_DOWN:
            evt = PNV_IFDOWN;
            break;

       default:
            // discard
            return 0;
    }
 
    /* arg is in_ifaddr in this case */
    return ((PALI_NOTIFIER*)nb)->fp_notify( (PAL_DEVICE)arg, evt );
}

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
int
pal_register_notifier(
    PAL_HANDLE* pnof1,
    PAL_HANDLE* pnof2,
    PAL_NOTIFYING_FP fp
    )
{
    PALI_NOTIFIER *nf1, *nf2;

    // zero the output pointer
    *pnof1 = NULL;
    *pnof2 = NULL;

    // memory
    nf1 = (PALI_NOTIFIER*)kmalloc( sizeof(*nf1), GFP_KERNEL );
    if( nf1 == NULL )
    {
        return -ENOMEM;
    }

    nf2 = (PALI_NOTIFIER*)kmalloc( sizeof(*nf2), GFP_KERNEL );
    if( nf2 == NULL )
    {
        return -ENOMEM;
    }

    // initialize
    memset( nf1, 0, sizeof(*nf1) );
    nf1->fp_notify = fp;
    nf1->nb.notifier_call = pal_notifier_callback;

    register_netdevice_notifier( &nf1->nb );

    memset( nf2, 0, sizeof(*nf2) );
    nf2->fp_notify = fp;
    nf2->nb.notifier_call = pal_inetaddr_notifier_callback;

    register_inetaddr_notifier( &nf2->nb );

    *pnof1 = (PAL_HANDLE)nf1;
    *pnof2 = (PAL_HANDLE)nf2;
    
    return 0;
}

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
int
pal_unregister_notifier(
    PAL_HANDLE nof1,
    PAL_HANDLE nof2
    )
{
    unregister_netdevice_notifier( (struct notifier_block*)nof1 );
    unregister_inetaddr_notifier( (struct notifier_block*)nof2 );

    kfree( nof1 );
    kfree( nof2 );

    return 0;
}

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
int
pal_start_thread(
    int (*fp)(void*),
    void* arg
    )
{
    return kernel_thread( fp, arg, 0 );
}

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
int
pal_daemonize_thread(
    char* thread_name
    )
{
    lock_kernel();

#if !defined(KERNEL_2_2)
    daemonize();
#else  // !defined(KERNEL_2_2)
    current->session = 1;
    current->pgrp = 1;
    exit_mm(current);
    exit_files(current);
    exit_fs(current);
#endif // defined(KERNEL_2_2)

    strcpy( current->comm, thread_name );
    unlock_kernel();

    return 0;
}

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
int 
pal_is_kernel_2_2(
    void
    )
{
#if defined(KERNEL_2_2)

    return 1;

#else

    return 0;

#endif // !defined(KERNEL_2_2)
}

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
int
pal_is_kernel_2_4(
    void
    )
{
#if defined(KERNEL_2_2)

    return 0;

#else

    return 1;

#endif // !defined(KERNEL_2_2)
}


/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
int
pal_proc_add_dir(
    char* dir
    )
{
#if defined(CONFIG_PROC_FS) && !defined(KERNEL_2_2)
    proc_mkdir( dir, NULL );
    return 0;
#else
    return -1;
#endif // defined(KERNEL_2_2)
}

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
int
pal_proc_remove_dir(
    char* dir
    )
{
#if defined(CONFIG_PROC_FS) && !defined(KERNEL_2_2)
    remove_proc_entry( dir, NULL );
    return 0;
#else
    return -1;
#endif // defined(KERNEL_2_2)
}

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
int
pal_proc_add_read_entry(
    char* path,
    void* ctx
    )
{
#if defined(CONFIG_PROC_FS) && !defined(KERNEL_2_2)
    create_proc_read_entry( path, 0, NULL, pal_proc_read_handler, ctx );
    return 0;
#else
    return -1;
#endif // defined(KERNEL_2_2)
}

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
int
pal_proc_remove_read_entry(
    char* path
    )
{
#if defined(CONFIG_PROC_FS) && !defined(KERNEL_2_2)
    remove_proc_entry( path, NULL );
    return 0;
#else
    return -1;
#endif // defined(KERNEL_2_2)
}

/****************************************************************************
 * 
 *
 * 
 *
 ****************************************************************************/
PAL_PROC_READ_FP proc_read_handler = NULL;

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
int
pal_proc_read_handler(
    char* page,
    char** start,
    off_t off,
    int count,
    int* eof,
    void* ctx
    )
{
#if defined(CONFIG_PROC_FS) && !defined(KERNEL_2_2)
    if( proc_read_handler )
    {
        return proc_read_handler( page, start, off, count, eof, ctx );
    }
    else
    {
        return 0;
    }
#else
    return -1;
#endif // defined(KERNEL_2_2)
}

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
void
pal_set_proc_read_handler(
    PAL_PROC_READ_FP fp
    )
{
    proc_read_handler =fp;
}

/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
void
pal_propagate_mc_list(
    void* mc_list_head,
    PAL_DEVICE dev_dst
    )
{
    netdevice_t*        dd = (netdevice_t*)dev_dst;
    struct dev_mc_list* dmi;
    struct dev_mc_list* dm2;

    for( dmi = (struct dev_mc_list*)(mc_list_head); dmi; dmi = dmi->next )
    {
        for( dm2 = dd->mc_list; dm2; dm2 = dm2->next )
        {
            if( (memcmp(dmi->dmi_addr, dm2->dmi_addr, dmi->dmi_addrlen) == 0) &&
                (dmi->dmi_addrlen == dm2->dmi_addrlen) )
                break;
        }
        if( !dm2 )
            dev_mc_add( dd, dmi->dmi_addr, dmi->dmi_addrlen, dmi->dmi_gusers );
    }
}


/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
void
pal_remove_mc_list(
    void* mc_list_head,
    PAL_DEVICE dev_dst
    )
{
    netdevice_t*        dd = (netdevice_t*)dev_dst;
    struct dev_mc_list* dmi;

    for( dmi = (struct dev_mc_list*)(mc_list_head); dmi; dmi = dmi->next )
    {
        dev_mc_delete( dd, dmi->dmi_addr, dmi->dmi_addrlen, dmi->dmi_gusers );
    }
}


/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
int
pal_append_mc_list(
    PAL_DEVICE dev_src,
    void** p_mc_list_head
    )
{
    netdevice_t*        ds = (netdevice_t*)dev_src;
    struct dev_mc_list* dmi;
    struct dev_mc_list* dm2;
    int    total = 0;

    for( dm2 = (struct dev_mc_list*)(*p_mc_list_head); dm2; dm2 = dm2->next )
        ++total;

    for( dmi = ds->mc_list; dmi; dmi = dmi->next )
    {
        for( dm2 = (struct dev_mc_list*)(*p_mc_list_head); dm2; dm2 = dm2->next )
        {
            if( (memcmp(dmi->dmi_addr, dm2->dmi_addr, dmi->dmi_addrlen) == 0) &&
                (dmi->dmi_addrlen == dm2->dmi_addrlen) )
                break;
        }
        if( dm2 == NULL )
        {
            dm2 = (struct dev_mc_list*)kmalloc( sizeof(*dm2), GFP_KERNEL );
            if( dm2 == NULL )
                break;
            *dm2 = *dmi;
            dm2->next = (struct dev_mc_list*)(*p_mc_list_head);
            *p_mc_list_head = dm2;
            ++total;
        }
    }
    return total;
}


/****************************************************************************
 * 
 *
 * 
 *
 * Input;
 *      
 *
 * Output:
 *      
 *
 ****************************************************************************/
void
pal_free_mc_list(
    void** p_mc_list_head
    )
{
    struct dev_mc_list* dmi;

    while( (dmi = (struct dev_mc_list*)(*p_mc_list_head)) != NULL )
    {
        *p_mc_list_head = dmi->next;
        kfree( dmi ); 
    }
}


