/*
 * Copyright 2001 Compaq Information Technologies Group, L.P.
 * 
 * Compaq and the Compaq logo are trademarks of Compaq Information
 * Technologies Group, L.P. in the U.S. and/or other countries.
 *
 * Confidential computer software. Valid license from Compaq required
 * for possession, use or copying. Consistent with FAR 12.211 and 12.212,
 * Commerical Computer Software, Computer Software Documentation, and
 * Technical Data for Commerical Items are licensed to the
 * U.S. Government under vendor's standard commercial license.
 */

/*
 *   Filename: cmhp.c 
 *   Provides the interfaces for all the standard driver to kernel
 */

#include <linux/config.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/module.h>
#include <linux/slab.h>
#include <linux/vmalloc.h>
#include <linux/fs.h>
#include <linux/errno.h>
#include <linux/types.h>
#include <linux/pci.h>
#include <linux/proc_fs.h>
#include <linux/fcntl.h>
#include <linux/smp_lock.h>
#include <linux/spinlock.h>
#include <linux/interrupt.h>
#include <linux/devfs_fs.h>
#include <linux/devfs_fs_kernel.h>

#include <asm/softirq.h>
#include <asm/system.h>
#include <asm/delay.h>
#include <asm/io.h>
#include <asm/uaccess.h>
#include <asm/processor.h>

#include "hptypes.h"
#include "hpproto.h"
#include "unixproto.h"


/* PROTOTYPES */
int _cmhp_ioctl(struct inode *i, struct file *f, unsigned int cmd, 
                unsigned long arg);

VOID cmhpInit_inj_err_mutex();
/*
 * Module information
 */
#define MODULE_NAME         "cmhp"
#define MODULE_VERSION      "0.1"

MODULE_AUTHOR("Compaq Technologies Group, L.P.");
MODULE_DESCRIPTION(MODULE_NAME " v." MODULE_VERSION 
      ".\nThis module provides Compaq Memory Hot Plug Controller Driver support.");

#if ( LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,9))
MODULE_LICENSE("Proprietary - Compaq Technologies Group, L.P.");
#endif /* LINUX_VERSION_CODE */

/* Major device number for /dev/cmhp device */
static uint cmhp_device_major;

/* Devfs handle for the CMHP device */
static devfs_handle_t   cmhp_devfs_handle;

/* File operations permitted on this device */
static struct file_operations cmhp_fops = {   
   owner:   THIS_MODULE,
   ioctl:   _cmhp_ioctl,
};

/* If HotAdd is enabled in the system */
BOOLEAN               cmhp_hotadd_enabled = FALSE;

///* Memory rebuild rate */
//sgsULONG               cmhp_rebuild_rate = RIV_RATE_SLOW;

/* Mutual exclusion from cmhp_interrupt() */
static   spinlock_t      cmhp_intr_slock = SPIN_LOCK_UNLOCKED;

/* Proctect CMHP global variables */
static   spinlock_t      cmhp_var_slock = SPIN_LOCK_UNLOCKED;

/* Mutex for cmhp_thread_exit */
static rwlock_t				cmhp_thread_slock = RW_LOCK_UNLOCKED;

/* Function interface with the HAM module */
ham_interface_t ham_hot_add_func = NULL;

/* Function to do Event log to the IML */
ULONG (*cmhpEventHandler)(LONG, LONG, LONG, PVOID) = NULL;

/* Flag for inter_module_register() - Set if HAM has been loaded */
static int cmhp_registered = 0;

/* Flag to pass to cmhp_exit() - Set on failure during insmod  */
static int cmhp_error = 0;

ULONG cmhpErrUnixtoKern(ULONG ulUnixErr)
{
   ULONG retval = 0;
   
   // convert the cmhp return code to the kernel equiv.
   switch(retval)
   {
      case CMHP_INVAL:
         retval = -EINVAL;
         break;
      case CMHP_NOMEM:
	 retval = -ENOMEM;
	 break;
      case CMHP_FAULT:
	 retval = -EFAULT;
	 break;
      case CMHP_EAGAIN:
	 retval = -EAGAIN;
	 break;
      case CMHP_EIO:
	 retval = -EIO;
	 break;
      default:
         break;
   }

   return(retval);
}
/*
 * Provide ioctl interfase
 */
int _cmhp_ioctl(struct inode *i, struct file *f, unsigned int cmd, 
                unsigned long arg)
{
   int retval = cmhp_ioctl(cmd, arg);

   return cmhpErrUnixtoKern(retval);
}
/* 
 * Protection from interrupts. Use it to access
 * oscommon functions/variables.
 */
void
cmhp_intr_lock(ULONG irq_flags)
{
   spin_lock_irqsave(&cmhp_intr_slock, irq_flags);
   return;
}

void
cmhp_intr_unlock(ULONG irq_flags)
{
   spin_unlock_irqrestore(&cmhp_intr_slock, irq_flags);
   return;
}

/* Mutex for hot-event lists */
static spinlock_t         cmhp_event_slock = SPIN_LOCK_UNLOCKED;
/* Lock cmhp_event_list */
void cmhp_event_lock(void)
{
   spin_lock_bh(&cmhp_event_slock);
   return;
}

/* Unlock cmhp_event_list */
void cmhp_event_unlock(void)
{
   spin_unlock_bh(&cmhp_event_slock);
   return;
}


/* Sync between kcmhpd and insmod/rmmod */
static DECLARE_MUTEX_LOCKED(cmhp_thread_mutex);

/* Mutex for syncing hot-events */
static DECLARE_MUTEX_LOCKED(cmhp_event_mutex);
void cmhp_event_mutex_up()
{
   up(&cmhp_event_mutex);
}

void cmhp_event_mutex_down()
{
   down(&cmhp_event_mutex);
}

/* If this is set, no events are queued and kcmhpd should exit */
static int   cmhp_thread_exit = 0;

/*
 * kcmhpd daemon that handles all hot-events.
 */
static int cmhp_thread(void *arg)
{
   /* Make this thread a daemon */
   lock_kernel();

   sprintf(current->comm, cmhp_thread_name);
   daemonize();

#if ( LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,10))
   reparent_to_init();
#endif /* LINUX_VERSION_CODE */

#if ( LINUX_VERSION_CODE == KERNEL_VERSION(2,4,20) && defined(RED_HAT_LINUX_KERNEL) )
   // Redhat 8.1/9 did chanded the task_struct and the recalc_sigpending call.
   spin_lock_irq(&current->sighand->siglock);
   siginitsetinv(&current->blocked, 0);
   recalc_sigpending();
   spin_unlock_irq(&current->sighand->siglock);
#else
   // this is the orignal tha works with everything else
   spin_lock_irq(&current->sigmask_lock);
   siginitsetinv(&current->blocked, 0);
   recalc_sigpending(current);
   spin_unlock_irq(&current->sigmask_lock);
#endif
      
   /* we might get involved when memory gets low */
   current->flags |= PF_MEMALLOC;
#if ( LINUX_VERSION_CODE == KERNEL_VERSION(2,4,20) && defined(RED_HAT_LINUX_KERNEL) )
   current->policy = SCHED_NORMAL;
#else
   current->policy = SCHED_OTHER;
#endif

   unlock_kernel();

   cmhpDebug("%s: up & running. Task id %d \n", cmhp_thread_name, current->pid);

   /* Initialize hot-event queues */
   cmhp_init_events(1);

   /* Indicate CMHP driver that we are prepared */
//sgs   printk("cmhp_thread: before up(cmhp_thread_mutex)\n");
   up(&cmhp_thread_mutex);
//sgs   printk("cmhp_thread: after up(cmhp_thread_mutex)\n");

   for (;;) 
   {
      cmhpDebug("%s: Sleeping on wait queue. \n", cmhp_thread_name);

      /* Wait for events */
//sgs      down(&cmhp_event_mutex);
      cmhp_event_mutex_down();

      read_lock_bh(&cmhp_thread_slock);
      if (cmhp_thread_exit) {
         read_unlock_bh(&cmhp_thread_slock);
         break;
      }
      read_unlock_bh(&cmhp_thread_slock);

      process_next_event();

   }

   /* Flush all pending events */
   cmhp_flush_events();

   /* Indicate rmmod that we are done */
   up(&cmhp_thread_mutex);
   
   return(0);
}


/* Process id of kcmhpd */
static pid_t cmhp_thread_id;

/*
 * Start the kcmhpd event thread.
 */
LONG cmhp_start_thread(VOID)
{
   cmhp_thread_id = kernel_thread(cmhp_thread, NULL, 0); 
   if (cmhp_thread_id < 0) {
      printk(KERN_ERR "%s: Unable to create kernel thread %s \n",
            cmhp_modname, cmhp_thread_name);
      cmhp_thread_id = 0;
      return(-1);
   }

   /* Wait for the thread to start */
   down(&cmhp_thread_mutex);

   cmhpDebug("%s: Event thread %s started successfully. \n",
         cmhp_modname, cmhp_thread_name);
         
   return(0);
}
/*
 * Kill kcmhpd and prevent queuing of further events.
 */
void
cmhp_end_thread(void)
{
   if (!cmhp_thread_id) {
      return;
   }

   write_lock_bh(&cmhp_thread_slock);
   cmhp_thread_exit = 1;

   /* Deliver the bad news to kcmhpd */
   up(&cmhp_event_mutex);
   write_unlock_bh(&cmhp_thread_slock);

   /* Wait for the cmhpd event thread to exit */
   down(&cmhp_thread_mutex);

   cmhpDebug( "%s: kernel thread %s killed. \n", cmhp_modname, cmhp_thread_name); 

   return;
}

/* 
 * Mutual exclusion for CMHP global variables 
 */
void cmhp_var_lock(void)
{
   spin_lock_bh(&cmhp_var_slock);
   return;
}

void cmhp_var_unlock(void)
{
   spin_unlock_bh(&cmhp_var_slock);
   return;
}


/*
 * Register the cmhp device driver.
 * Create devfs entry for cmhp device. If devfs is mounted, the device
 * appears under the devfs tree. If devfs is not mounted at boot time, 
 * the /dev/cmhp entry needs to be created from userland.
 */
static int cmhp_register_device(void)
{
   SET_MODULE_OWNER(&cmhp_fops);
   cmhp_device_major = devfs_register_chrdev(0, cmhp_modname, &cmhp_fops); 
   if (cmhp_device_major < 0) 
   {
      cmhp_device_major = 0;
      return(-EINVAL);
   }

   cmhp_devfs_handle = devfs_register(NULL, cmhp_modname, 
               DEVFS_FL_DEFAULT|DEVFS_FL_AUTO_DEVNUM, 
               cmhp_device_major, 0,  
               S_IFCHR | S_IRUSR | S_IWUSR | 
               S_IRGRP | S_IWGRP, 
               &cmhp_fops, NULL);

   /* Devfs mounted at boot time */
   if (cmhp_devfs_handle) {
         devfs_get_maj_min(cmhp_devfs_handle, &cmhp_device_major, NULL);
   }

   /* We should have a major number by now */
   if (cmhp_device_major == 0) {
      printk(KERN_ERR "%s: Unable to register cmhp device.\n", cmhp_modname);
      return(-EINVAL);
   }

   cmhpDebug("%s: CMHP device registered, major number: %d, device : 0x%x \n", 
         cmhp_modname, cmhp_device_major, MKDEV(cmhp_device_major, 0));

   return(0);
}

/*
 * Remove devfs entry for this device.
 */
static int cmhp_unregister_device(void)
{
   if (cmhp_device_major) {
      if (devfs_unregister_chrdev(cmhp_device_major, cmhp_modname)) {
            printk(KERN_WARNING
               "%s: Error unregistering cmhp device. \n", cmhp_modname);
      }
      cmhp_device_major = 0;
   }

   if (cmhp_devfs_handle) {
      devfs_unregister(cmhp_devfs_handle);
      cmhp_devfs_handle = NULL;
   }

   return(0);
}

void cmhp_ham_load()
{
   /* Already loaded */
   if (ham_hot_add_func) {
      return;
   }

   ham_hot_add_func = (ham_interface_t)inter_module_get(HAM_HOT_ADD);
   if (ham_hot_add_func == NULL) {
      printk(KERN_WARNING
            "%s: WARNING: %s not found. HAM module may not be loaded."
            " Hot-add events will be ignored.\n",
            cmhp_modname, HAM_HOT_ADD);
   }

   return;
}

void cmhpLoadImlHandler()
{
   /* Already loaded */
   if (cmhpEventHandler) 
   {
      return;
   }

   cmhpEventHandler = (ULONG(*)(LONG,LONG,LONG, PVOID))inter_module_get("cpqwEventHandler");

   if (cmhpEventHandler == NULL) 
   {
      printk(KERN_WARNING
         "%s: WARNING: %s not found. cpqevt module may not be loaded."
          " events will be not be logged.\n", cmhp_modname, "cpqwEventHandler");
   }

   return;
}

void cmhpUnloadImlHandler()
{
   if (cmhpEventHandler) 
   {
      cmhpEventHandler = NULL;
      inter_module_put("cpqwEventHandler");
   }
}

void cmhp_ham_unload()
{
   if (ham_hot_add_func) 
   {
      ham_hot_add_func = NULL;
      inter_module_put(HAM_HOT_ADD);
   }
   return;
}

static int cmhp_register_module(void)
{
   cmhp_var_lock();

   if (inter_module_get(cmhp_modname)) 
   {
      inter_module_put(cmhp_modname);
      cmhp_var_unlock();
      printk(KERN_ERR
            "%s: Found another instance of %s module.\n",
            cmhp_modname, cmhp_modname);
      return(1);
   }

   inter_module_register(cmhp_modname, THIS_MODULE, (void *)cmhp_modname);
   cmhp_registered = 1;

   cmhp_var_unlock();
   return(0);
}

static void cmhp_unregister_module(void)
{
   if (! cmhp_registered)
      return;

   inter_module_unregister(cmhp_modname);
   cmhp_registered = 0;
   return;
}

static void __exit cmhp_mod_exit(void)
{
   ULONG   irq_flags = 0;

   cmhp_intr_lock(irq_flags);

   cmhp_unregister_module();
   cmhp_function_exit();
   cmhp_unregister_device();
   cmhp_end_thread();
   cmhp_ham_unload();
   cmhpUnloadImlHandler();

   if (!cmhp_error)
      printk(KERN_INFO "%s: Module unloaded.\n", cmhp_modname);

   return;
}

static int __init cmhp_mod_init(void)
{
   int rc = 0;

   cmhpDebug("%s: Init Compaq Memory Controller\n", cmhp_modname);

   spin_lock_init(&cmhp_intr_slock);
   spin_lock_init(&cmhp_var_slock);
   cmhpInit_inj_err_mutex();


   rc = cmhp_register_module();
   if (rc) 
   {
      printk(KERN_ERR "%s: Unable to load module.\n", cmhp_modname);
      goto init_failed;
   }

   /*
    * init the function, but don't enable interrupts yet
    */
   rc = cmhp_function_init(); 
   if( rc ) {
      cmhpDebug("%s: cmhp_function_init failed!", cmhp_modname);
      goto init_failed;
   }

   if (cmhp_register_device()) {
      printk(KERN_ERR
            "%s: Could not register %s device.\n",
            cmhp_modname, cmhp_modname);
      goto init_failed;
   }

   if (cmhp_start_thread()) {
      printk(KERN_ERR "%s: ERROR: Unable to load module\n", cmhp_modname);
      goto init_failed;
   }

   cmhp_ham_load();

   // load the Event Handler entry point
   cmhpLoadImlHandler();

#ifdef   _MHP_ADD_
   cmhp_hotadd_enabled = TRUE;
#endif   /* MHP_ADD */

   printk(KERN_INFO "%s: hp Proliant Memory Hot Plug Driver (rev %s)\n", cmhp_modname, EFS_VER);
   return (0);

init_failed:
   cmhp_error = 1;
   cmhp_mod_exit();
   return(-1);
}

/**************************************************************
 * Misc counters and Semaphores
 *************************************************************/
/* Count for number of error injections */
static atomic_t inj_err_count[5 /*MAX_MRM*/];
/* Mutual exclusion for injecting errors on dimms */
static struct semaphore inj_err_mutex[5 /*MAX_MRM*/];


VOID cmhpInit_inj_err_mutex()
{
   int i;
   /* Initialize the inject error semaphores */
   for (i = 0; i < 5 /*MAX_MRM*/; i++) 
   {
      init_MUTEX(&inj_err_mutex[i]);
   }
}

VOID cmph_inc_inj_err_count(ULONG mrm)
{
   atomic_inc(&inj_err_count[mrm]);
}
ULONG cmph_read_inj_err_count(ULONG mrm)
{
   atomic_read(&inj_err_count[mrm]);
}
VOID cmph_dec_inj_err_count(ULONG mrm)
{
   atomic_dec(&inj_err_count[mrm]);
}



/**************************************************************
 * Memory Access
 *************************************************************/
PVOID cmhp_get_free_DMA_page()
{
  return (PVOID)__get_free_page(GFP_DMA); 
}

ULONG cmhp_copy_from_user(PVOID dest, PVOID src, ULONG size)
{
   return copy_from_user(dest, src, size);
}
ULONG cmhp_copy_to_user(PVOID dest, PVOID src, ULONG size)
{
   return copy_to_user(dest, src, size);
}

/***************************************************************
 * cmhpMapPhysUnCache                                         *
 *                                                             *
 * This function accepts a 32 bit physical address             *
 * and returns a virtual address.  The virtual address         *
 * will be in the Uncacheable region of the Kernel.            *
 * This function is mainly used when mapping in                *
 * memoey mapped IO.                                           *
 * return NULL if something goes wrong                         *
 *                                                             *
 ***************************************************************/
PVOID cmhpMapPhysUnCache(ULONG ulPhysAddr, ULONG ulLength)
{

   PVOID VirtAddr = NULL;
   
   if(ulPhysAddr)
      VirtAddr = ioremap_nocache(ulPhysAddr, ulLength);
  
   cmhpDebug("\ncmhpMapPhys32UnCache: Phys 0x%x -> Virt 0x%x Len 0x%x\n", ulPhysAddr, VirtAddr, ulLength);
   return (VirtAddr);
}
/***************************************************************
 * cpqMapPhysCache                                             *
 *                                                             *
 * This function accepts a 64 bit physical address             *
 * and returns a virtual address.  The virtual address         *
 * will be in the cacheable region of the Kernel.              *
 * This function is mainly used when mapping in                *
 * Rom data and functions.                                     *
 * return NULL if something goes wrong                         *
 *                                                             *
 ***************************************************************/
PVOID cmhpMapPhysCache(ULONG PhysAddr, ULONG ulLength)
{
   PVOID VirtAddr = NULL;
   
   if(PhysAddr)
     VirtAddr = ioremap(PhysAddr, ulLength);

   cmhpDebug("\ncpqMapPhysCache: Phys 0x%x -> Virt 0x%x Len 0x%x\n", PhysAddr, VirtAddr, ulLength);
   return (VirtAddr);
}
/***************************************************************
 * cmhpUnMapPhys                                               *
 *                                                             *
 * This function unmaps the Virtual address from above         *
 *                                                             *
 ***************************************************************/
VOID cmhpUnMapPhys(PVOID VirtAddr, ULONG size)
{
   if(VirtAddr)
      iounmap((PVOID)VirtAddr);
}

/***************************************************************
 * cmhpMalloc                                                   *
 *                                                             *
 * This function allocates a buffer of size ulSize             *
 * return NULL if something goes wrong                         *
 *                                                             *
 ***************************************************************/
PVOID cmhpMalloc(ULONG ulSize)
{
   PUCHAR p=(PUCHAR)kmalloc(ulSize, GFP_KERNEL);
   ULONG i;

   //go ahead and zero out the memory
   if(p)
   {
      for(i = 0; i < ulSize; i++)
      {
         p[i] = 0;
      }
   }
   
   return (PVOID)p;
}  /* end cmhpMalloc */
/***************************************************************
 * cmhpAtomicMalloc                                                   *
 *                                                             *
 * This function allocates a buffer of size ulSize             *
 * return NULL if something goes wrong                         *
 *                                                             *
 ***************************************************************/
PVOID cmhpAtomicMalloc(ULONG ulSize)
{
   PUCHAR p=(PUCHAR)kmalloc(ulSize, GFP_ATOMIC);
   ULONG i;

   //go ahead and zero out the memory
   if(p)
   {
      for(i = 0; i < ulSize; i++)
      {
         p[i] = 0;
      }
   }
   
   return (PVOID)p;
}  /* end cmhpMalloc */


/***************************************************************
 * cmphFree                                                    *
 *                                                             *
 * This function frees the memory allocated by cmhpMalloc      *
 *                                                             *
 ***************************************************************/
VOID cmhpFree(PVOID pVAddr, ULONG ulSize)
{
   if(pVAddr)
      kfree (pVAddr);
}  /* end cmhpMalloc */

ULONG cmhpGetPageSize()
{
   return PAGE_SIZE;
}

/********************************************************
 * PCI interfaces
 ********************************************************/
PPCISTRUCTLIST cmhpFindPCIDevice(ULONG ulVendorID, ULONG ulDeviceID)
{
   struct pci_dev *pdev; 
   PPCISTRUCTLIST pPciHead = NULL;
   PPCISTRUCTLIST pPciTail = NULL;
   PPCISTRUCTLIST pPciTmp = NULL;


   for(pdev = pci_find_device(ulVendorID, ulDeviceID, NULL);
       pdev != NULL; 
       pdev = pci_find_device(ulVendorID, ulDeviceID, pdev))
   {
      pPciTmp = cmhpMalloc(sizeof(PCISTRUCTLIST));
      if(pPciTmp != NULL)
      {
         pPciTmp->pNext=NULL;
	 pPciTmp->sysdata = pdev;   // OS kernel data
         pPciTmp->ulMemAddr=pci_resource_start(pdev, 0); 
	 pPciTmp->ulSize = pci_resource_end(pdev,0) - pPciTmp->ulMemAddr;
         pPciTmp->ucBusNumber=pdev->bus->number;
         pPciTmp->ucDeviceNumber=PCI_SLOT(pdev->devfn);
         pPciTmp->ucFunctionNumber=PCI_FUNC(pdev->devfn);
         pPciTmp->ucIrq=pdev->irq;
	  
         //Now add this on to the list
	 if(pPciHead == NULL)
	 {
            //this is the first one found
            pPciHead = pPciTmp;
	    pPciTail = pPciTmp;
	 }
	 else
	 {
            //put it at the end of the list
            pPciTail->pNext = pPciTmp;
	    pPciTail = pPciTmp;
	 }
      }
   }//end for

   cmhpDebug("MemAddr = 0x%x\n", pPciTmp->ulMemAddr);
   return pPciHead;

}
/***************************************************************************
*****************************************************************************/
BOOLEAN FreeDeviceData(PPCISTRUCTLIST pPciList)
{
   if(pPciList)
   {
      // first free the next one
      FreeDeviceData(pPciList->pNext);

      // Then free this one
      cmhpFree(pPciList, sizeof(PCISTRUCTLIST));
   }

   return TRUE;
}

VOID cmhpPciWriteConfigWord(PVOID dev, ULONG addr, WORD val)
{
   if((pci_write_config_word(dev, addr, val)) != 0) 
   {
      cmhpDebug(KERN_WARNING "%s: pci_write_config_dword failed!", 
                cmhp_modname);
   }
}

/*****************************************************************
 * ISR
 *****************************************************************/

/* Tasklet that performs bottom half processing for cmhp_interrupt() */
DECLARE_TASKLET(cmhp_tasklet, cmhp_interrupt_bh, 0);

void _cmhp_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
   cmhp_interrupt(irq);
}

VOID cmhpRegisterIRQ(ULONG irq, PVOID arg)
{
  if(request_irq(irq, _cmhp_interrupt, SA_SHIRQ, cmhp_modname, arg)) 
  {
     printk(KERN_WARNING "%s: request irq %d failed.\n", cmhp_modname, irq);
  }
}

VOID cmhpUnRegisterIRQ(ULONG irq, PVOID arg)
{
   free_irq(irq, arg);
}

void cmhpStartDFP()
{
   tasklet_schedule(&cmhp_tasklet);
}

/**************
 * Pause      *
 **************/
ULONG cmhpUsecWait(ULONG microsecs)
{
   unsigned long usecs_per_jiffie = ( 1000000 / HZ );

/* Added a function to allow a spin delay in excess of 1 clock tick.  If we use
 * udelay to delay for more than a few milliseconds, there is a danger of overflow
 * on some of the faster processors.  Therfore, we need to use the time_after macro
 * to assure this doesn't happen.  We will allow other interrupts, and processes to
 * proceed during this delay, and prevent the spin from totally occupying the CPU
 * by calling the scheduler. - jgl 11/00
 */

   if( microsecs <= usecs_per_jiffie )
   {
      udelay(microsecs);
   }
   else 
   {
      signed long interval = ( microsecs / usecs_per_jiffie );
//      unsigned long old_state = current->state;
printk("Long delay %ld jiffies----\n", interval);
//      current->state = TASK_INTERRUPTIBLE;
      set_current_state(TASK_INTERRUPTIBLE);
      schedule_timeout(interval);
      //current->state = TASK_RUNNING;
//printk("Long delay Done\n");
   }
   return( microsecs );
}

void cmhpPrint(ULONG ulSev, PCHAR str, ...)
{
   va_list args;
   char buf[1024];
   int i;

   va_start(args, str);
   i = vsprintf(buf, str, args); /* hopefully i < sizeof(buf)-4 */
   va_end(args);

   switch(ulSev)
   {
      case CMHP_INFO:
      {
         printk(KERN_INFO "%s", buf);
         break;
      }
      case CMHP_WARNING:
      {
         printk(KERN_WARNING "%s", buf);
         break;
      }
      case CMHP_ERR:
      {
         printk(KERN_ERR "%s", buf);
         break;
      }
      case CMHP_DBG:
      {
         printk(KERN_DEBUG "%s", buf);
         break;
      }
      default:
      {
         printk("%s", buf);
      }
   }//switch
}


VOID lock_module()
{
   MOD_INC_USE_COUNT;
}

VOID unlock_module()
{
   MOD_DEC_USE_COUNT;
}

/*
 * Get the seconds since EPOC
 */
ULONG cmhpGetCurrentTimeInSec()
{
    return CURRENT_TIME;
}


module_init(cmhp_mod_init);
module_exit(cmhp_mod_exit);
