#!/bin/sh
#
###############################################################################
##  (C) Copyright 2005 Hewlett-Packard Development Company, L.P.
###############################################################################
#
KERNEL_MAJOR_VERSION=`uname -r | cut -b1,2,3`
if [ `expr ${KERNEL_MAJOR_VERSION} \>= 2.6` -eq 1 ];then
   MOD_EXT="ko"
else
   MOD_EXT="o" 
fi

###############################################################################
## ValidateKernel()
## This routine will check to see if the appropriate kernel environment is in
## place if we need to build.  
###############################################################################
ValidateKernel() {


       # we need to do a test load because RHEL 3 has several
       # different kernels with the same uname -r and these
       # modules will not load on all of them
       if [ -d /opt/hp/hp-OpenIPMI/bin/`uname -r` ]; then
          insmod  /opt/hp/hp-OpenIPMI/bin/`uname -r`/ipmi_msghandler.${MOD_EXT} 2>/tmp/hptest
          if [ $? -ne 0 ]; then
              if [ "`sed -e 's/.*\(Invalid module format\)/\1/' /tmp/hptest`" != "Invalid module format" ]; then
                  echo "There is an issue loading:  /opt/hp/hp-OpenIPMI/bin/`uname -r`/ipmi_msghandler.${MOD_EXT} "
                  echo "Attempting a rebuild of the hp-OpenIPMI driver . . ."
                  rm -rf /opt/hp/hp-OpenIPMI/bin/`uname -r`
              fi
          else
             rmmod ipmi_msghandler
          fi
       fi

       if [ ! -d /opt/hp/hp-OpenIPMI/bin/`uname -r` ]; then
		echo "This kernel requires a rebuild.  Seeking source files"
		if [ -d /lib/modules/`uname -r`/source -a \
                     ! -f /lib/modules/`uname -r`/source/include/linux/ipmi.h ]; then
			echo "The file:  /lib/modules/`uname -r`/source/include/linux/ipmi.h"
			echo "does not exist.  This is a strong indication that the"
			echo "necessary Linux Kernel Development packages are not"
			echo "installed."
			return 2
		elif [ -d /lib/modules/`uname -r`/build -a \
                       ! -f /lib/modules/`uname -r`/build/include/linux/ipmi.h ]; then
			echo "The file:  /lib/modules/`uname -r`/build/include/linux/ipmi.h"
			echo "does not exist.  This is a strong indication that the"
			echo "necessary Linux Kernel Development packages are not"
			echo "installed."
			return 2
		
		else
			echo "Basic source files appear to be installed.  Investigating further."

			if [ ! -f /lib/modules/`uname -r`/build/.config ]; then
				echo "The file: /lib/modules/`uname -r`/build/.config"
				echo "does NOT exist.  There must be a Linux kernel"
				echo "'.config' file with IPMI enabled to build this"
				echo "package."
				return 2
			fi

			return 1
		fi
	fi

	return 0
}

##############################################################################
## CheckOsFlavor
## This routine will check the Linux OS Flavor.  The following values will be
## returned.
## 1 = RHEL 4
## 2 = RHEL 3
## 3 = SLES 9
## 4 = SLES 8
## 5 = SLES 10
## 6 = DEBIAN
## 99 = UNKOWN LINUX
#############################################################################
CheckOsFlavor() {

RetCode=99
#
# Check for RHEL 4
#
	cat /etc/issue | grep -e "Red Hat Enterprise Linux .. release 4" > /dev/null
	if [ $? = 0 ] ; then
		return 1
	fi


#
# Check for RHEL 3
#
	cat /etc/issue | grep -e "Red Hat Enterprise Linux .. release 3" > /dev/null
	if [ $? = 0 ] ; then
		return 2
	fi

#
# Check for SLES 9
#
	cat /etc/issue | grep -e "SUSE LINUX Enterprise Server 9" > /dev/null
	if [ $? = 0 ] ; then
		return 3
	fi

#
# Check for SLES 10 
#
	cat /etc/issue | grep -e "SUSE Linux Enterprise Server 10" > /dev/null
	if [ $? = 0 ] ; then
		return 5
	fi

#
# Check for DEBIAN
#
	cat /etc/issue | grep -e "Debian GNU/Linux 3.1" > /dev/null
	if [ $? = 0 ] ; then
		return 6
	fi


#
# Check for VMware ESX 3.0.x
#
	cat /etc/issue | grep -e "VMware ESX Server 3" > /dev/null
	if [ $? = 0 ] ; then
		return 7
	fi


#
# Check for SLES 8
#
	cat /etc/issue | grep -e "UnitedLinux 1.0" > /dev/null
	if [ $? = 0 ] ; then
		return 4
	fi

	return 99

}


#
##############################################################################
#                           ###### M A I N ######
##############################################################################
#

ValidateKernel

case $? in
	0)
		exit 0
		;;
	1)
		src_found=Y
		;;
	2)
		src_found=N
		;;
	*)
		src_found=N
		echo "Unknown return code from ValidateKernel"
		exit 1
		;;
esac

#
# If we get this far, then we need to perform a deeper search for
# the Linux kernel source files.  This is very dependent on the
# particular flavor of Linux loaded.
#

CheckOsFlavor

case $? in
	0)
		echo "Invalid return code (0) from CheckOsFlavor"
		src_found=N
		exit 1
		;;
	1)
		echo "Checking Sources for `cat /etc/issue`"
		LINUX_BASE=`uname -r | sed -e 's/smp//g' -e 's/hugemem//g'`
		RetCode=0
		src_found=N
		rpm -q kernel-utils > /dev/null 2>&1
		if [ $? != 0 ]; then
			found="is missing"
			RetCode=1
		else
			found="has been found"
		fi
		echo "The package kernel-utils $found"

		if [ RetCode = 1 ]; then
			exit 1;
		fi

		rpm -q glibc-kernheaders > /dev/null 2>&1
		if [ $? != 0 ]; then
			found="is missing"
			RetCode=1
		else
			found="has been found"
		fi
		echo "The package glibc-kernheaders $found"

		if [ RetCode = 1 ]; then
			exit 1;
		fi

		uname -r | grep smp
		if [ $? = 0 ]; then
			rpm -q kernel-smp-${LINUX_BASE} > /dev/null 2>&1
			if [ $? != 0 ]; then
				found="is missing"
				RetCode=1
			else
				found="has been found"
			fi
			echo "The package kernel-smp-${LINUX_BASE} $found"

			if [ RetCode = 1 ]; then
				exit 1;
			fi

			rpm -q kernel-smp-devel-${LINUX_BASE} > /dev/null 2>&1
			if [ $? != 0 ]; then
				found="is missing"
				RetCode=1
			else
				found="has been found"
			fi
			echo "The package kernel-smp-devel-${LINUX_BASE} $found"


			if [ RetCode = 1 ]; then
				exit 1;
			fi
		else
			RetCode=1
		fi

		if [ RetCode = 1 ]; then
			RetCode=0
			uname -r | grep hugemem
			if [ $? = 0 ]; then
				rpm -q kernel-hugemem-${LINUX_BASE} > /dev/null 2>&1
				if [ $? != 0 ]; then
					found="is missing"
					RetCode=1
				else
					found="has been found"
				fi
				echo "The package kernel-hugemem-${LINUX_BASE} $found"

				if [ RetCode = 1 ]; then
					exit 1;
				fi

				rpm -q kernel-hugemem-devel-${LINUX_BASE} > /dev/null 2>&1
				if [ $? != 0 ]; then
					found="is missing"
					RetCode=1
				else
					found="has been found"
				fi
				echo "The package kernel-hugemem-devel-${LINUX_BASE} $found"

				if [ RetCode = 1 ]; then
					exit 1;
				fi
			fi

		fi


		if [ RetCode = 1 ]; then
			RetCode=0
			rpm -q kernel-devel-${LINUX_BASE} > /dev/null 2>&1
			if [ $? != 0 ]; then
				found="is missing"
				RetCode=1
			else
				found="has been found"
			fi
			echo "The package kernel-devel-${LINUX_BASE} $found"

			if [ RetCode = 1 ]; then
				exit 1;
			fi

		fi

		src_found="Y"

		;;

	2)
		echo "Checking Sources for `cat /etc/issue`"
		LINUX_BASE=`uname -r | sed -e 's/smp//g' -e 's/hugemem//g'`
		RetCode=0
		src_found=N
		rpm -q kernel-utils > /dev/null 2>&1
		if [ $? != 0 ]; then
			found="is missing"
			RetCode=1
		else
			found="has been found"
		fi
		echo "The package kernel-utils $found"

		if [ RetCode = 1 ]; then
			exit 1
		fi

		rpm -q glibc-kernheaders > /dev/null 2>&1
		if [ $? != 0 ]; then
			found="is missing"
			RetCode=1
		else
			found="has been found"
		fi
		echo "The package glibc-kernheaders $found"

		if [ RetCode = 1 ]; then
			exit 1
		fi

		rpm -q kernel-source-${LINUX_BASE} > /dev/null 2>&1
		if [ $? != 0 ]; then
			found="is missing"
			RetCode=1
		else
			found="has been found"
		fi
		echo "The package kernel-source-${LINUX_BASE} $found"

		if [ RetCode = 1 ]; then
			exit 1
		else
			src_found="Y"
		fi

		;;

	3)
		echo "Checking Sources for SUSE LINUX Enterprise Server 9"
		LINUX_BASE=`uname -r | sed -e 's/-smp//g' -e 's/-default//g' -e 's/-bigsmp//g'`
		RetCode=0
		src_found=N
		rpm -q kernel-syms-${LINUX_BASE} > /dev/null 2>&1
		if [ $? != 0 ]; then
			found="is missing"
			RetCode=1
		else
			found="has been found"
		fi
		echo "The package kernel-syms-${LINUX_BASE} $found"

		if [ RetCode = 1 ]; then
			exit 1;
		fi

		rpm -q kernel-source-${LINUX_BASE} > /dev/null 2>&1
		if [ $? != 0 ]; then
			found="is missing"
			RetCode=1
		else
			found="has been found"
		fi
		echo "The package kernel-source-${LINUX_BASE} $found"

		if [ RetCode = 1 ]; then
			exit 1;
		fi

		src_found="Y"

		;;

	5)
		echo "Checking Sources for SUSE Linux Enterprise Server 10"
		LINUX_BASE=`uname -r | sed -e 's/-smp//g' -e 's/-default//g' -e 's/-bigsmp//g' -e 's/-xenpae//g' -e 's/-xen//g'`
		RetCode=0
		src_found=N

		rpm -q kernel-source-${LINUX_BASE} > /dev/null 2>&1
		if [ $? != 0 ]; then
			found="is missing"
			RetCode=1
		else
			found="has been found"
		fi
		echo "The package kernel-source-${LINUX_BASE} $found"

		if [ RetCode = 1 ]; then
			exit 1;
		fi

		src_found="Y"

		;;

#          ******   DEBIAN  *****
	6)
		echo "Checking Sources for `cat /etc/issue`"
		LINUX_BASE=`uname -r | sed -e 's/-smp//g' `
		RetCode=0

		;;

#          ******   VMWARE  ******
	7)
		echo "Checking Sources for `cat /etc/issue`"
		RetCode=1
		src_found=N
		if [ ! -f /lib/modules/`uname -r`/build/Makefile ]; then
			echo "There does not appear to be source files availableto rebuild"
		elif [ ! -f /lib/modules/`uname -r`/build/.config ]; then
			echo "No '.config' file available. Seeking vmnix.config"
			if [ -f /lib/modules/`uname -r`/build/configs/vmnix.config ]; then
				echo " /lib/modules/`uname -r`/build/configs/vmnix.config found."
				echo "Using as the default kernel configuration file."
				cp -p /lib/modules/`uname -r`/build/configs/vmnix.config /lib/modules/`uname -r`/build/.config
				RetCode=0
				src_found="Y"
			fi
		else
			RetCode=0
			src_found="Y"
		fi

		if [ RetCode = 1 ]; then
			exit 1;
		fi

		;;


	4)
		echo "Checking Sources for SUSE LINUX Enterprise Server 8 / UnitedLinux 1.0"
		RetCode=0
		src_found=N

		rpm -q kernel-source > /dev/null 2>&1
		if [ $? != 0 ]; then
			found="is missing"
			RetCode=1
		else
			found="has been found"
		fi
		echo "The package kernel-source $found"

		if [ RetCode = 1 ]; then
			exit 1;
		fi

		src_found="Y"

		;;

	*)
		if [ "$src_found" = "N" ]; then
			echo "Unknown Linux Distribution - No Source to build"
			exit 1
		else
			echo "Unknown Linux Distribution - will attempt to build"
			echo "The hp-OpenIPMI(4) and hpasm(4) man pages have additional"
                        echo "information to resolve this issue. The HP Advanced System"
                        echo "Management software is currently disabled on this system."
		fi
		;;
esac

if [ "$src_found" != "N" ]; then
	cd /opt/hp/hp-OpenIPMI
	make clean
	make
	if [ $? != 0 ]; then
		echo ""
		echo "WARNING:  Build did not complete successfully!"
		echo ""
		exit 1
	fi
fi


