/* Copyright (c) 2006  Hewlett-Packard Development Company, L.P. */
/*
** High Performance Mouse Helper Program
**
** Finds the HP mouse and directs it's events to a named pipe
** so the XInput driver can receive them.
**
*/

#include <stdio.h>

#include <stdlib.h>
#include <string.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/fcntl.h>
#include <unistd.h>
#include <errno.h>
#include <signal.h>
#include <sys/utsname.h>


#include <linux/input.h>


typedef unsigned char UINT8;
typedef unsigned short UINT16;
typedef unsigned int UINT32;

int verbose=0;
FILE *logfp = NULL;

#define DEBUGOUT(x...) if (verbose) {fprintf(logfp, x); fflush(logfp);}

int ilo_init_mouse(int mode)
{
   if(mode == 1)
   {
      system("if [ -f /opt/hp/hpmouse/hpmouse ]; then sh /opt/hp/hpmouse/hpmouse enable-ilo-hpmouse >/dev/null 2>&1; fi");
   }
   return 0;
}

int is_device(int fd)
{
   unsigned short id[4]={0,0,0,0};
   unsigned long check, want;
   unsigned long bits;
  
   ioctl(fd, EVIOCGID, id);
   /* iLO2 is vendor 0x03f0, product 0x1027 */
   /* iLO1 is vendor 0x049f, product 0x0048 - 0x004f */
   if ((id[ID_VENDOR] == 0x03F0 && id[ID_PRODUCT] == 0x1027) ||
      (id[ID_VENDOR] == 0x049F && (id[ID_PRODUCT] & 0xFFFE) == 0x48)) 
   {
      DEBUGOUT("ven=%04x prod=%04x\n", id[ID_VENDOR], id[ID_PRODUCT]);

      /* Check if it's in absolute mode */
      ioctl(fd, EVIOCGBIT(0, sizeof(bits)), &bits);
      DEBUGOUT("    bits=%08lx\n", bits);
      check = (1UL<<EV_ABS) | (1UL<<EV_REL);
      want  = (1UL<<EV_ABS);
      if ((bits & check) == want) 
      {
         /*
          ** Now to be doubly sure, check the KEY
          ** bits and see if it supports ESC.  If
          ** it does, it's really the keyboard and
          ** we should skip it
          */
          bits = 0;

          ioctl(fd, EVIOCGBIT(EV_KEY, 1), &bits);

          DEBUGOUT("    EV_KEY bits = %08lx\n", bits);

          // No ESC key, then it's a mouse
          if ((bits & 2) == 0) return 1;
       }
   }
   return 0;
}

int hp_open(void)
{
   char buf[64];
   int fd;
   int i;

   for(i=0; i<100; i++)
   {
      /*
       * Would be better to use readdir() and friends, but
       * this is easy
       */
       sprintf(buf, "/dev/input/event%d", i);
       fd = open(buf, O_RDONLY);
       if (fd < 0) continue;

       DEBUGOUT( "Checking %s\n", buf);
 
       if(is_device(fd)) break;

       close(fd);
       fd = -1;
   }

   DEBUGOUT( "hpmouse: %s\n", (fd==-1) ? "unknown" : buf);

   return fd;
}


int main(int argc, char **argv)
{
   sigset_t sigset;
   int rc;
   int devicefd=-1, pipefd=-1;
   unsigned char buf[1024];
   int i;
   int pid = 0;
   int nofork = 0;
   int killer=0, die = 0;
   char *pn = "/dev/hpmouse";

   for(i=1; i<argc; i++) 
   {
      if (!strcmp(argv[i], "-v")) 
      {
         verbose++;
         continue;
      }
      if (!strcmp(argv[i], "-n")) 
      {
         nofork = 1;
         continue;
      }
      if (!strcmp(argv[i], "-p")) 
      {
         pn = argv[++i];
         continue;
      }
      if (!strcmp(argv[i], "-d")) 
      {
         killer=1;
         die = 1;
         break;
      }
         fprintf(stderr, "Unknown option: %s\n", argv[i]);
         die = 1;
   }

   //fix the wierd ps listing
//   bzero(argv[0], strlen(argv[0]));
//   if(killer)
//      sprintf(argv[0], "/opt/hp/hpmouse/hpmouse-killer");
//   else
//      sprintf(argv[0], "/opt/hp/hpmouse/hpmouse-helper");

   //if this is the killer
   if (killer)
   {
      signal(SIGTERM, SIG_IGN);
      system("killall hpmouse-helper >/dev/null 2>&1");
   }

   if (die) return -1;

   ilo_init_mouse(1);

   if(verbose)
   {
       logfp = fopen("/opt/hp/hpmouse/hpmouse-helper.log", "w");
       if(logfp == NULL) return -2;
       DEBUGOUT("Starting %s\n", argv[0]);
   }

   /* Create the named pipe */
   rc = mkfifo(pn, 0644);
   if (rc < 0 && errno != EEXIST) 
   {
      DEBUGOUT( "mkfifo failed rc=%d errno = %d", rc, errno);
      if(verbose) fclose(logfp);
      return -3;
   }
   DEBUGOUT("Pipe Created\n");

   /* Become a daemon */
   if(!nofork) daemon(0,0);

   DEBUGOUT("Child Process\n");

   /* Child */
   /* Close all files except stderr */
   if(!verbose)
   {
      for(i=0; i<255; i++) 
      {
         if (i == 2) continue;
	   close(i);
      }
   }
   
   signal(SIGPIPE, SIG_IGN);

   for(;;) 
   {
      int rsize=0, wsize=0;

      if(devicefd != -1)
      {
         if(!is_device(devicefd))
         {
            close(devicefd);
            devicefd=-1;
         }
      }
       
      //If we don't have a fd open to the device, try to get one
      //No point in leaving here unless we have a device
      while (devicefd == -1) 
      {
         devicefd = hp_open();
         if(devicefd == -1) sleep(1);
      }

      rsize = read(devicefd, buf, sizeof(buf));
      if (rsize < 0) 
      {
         /* If the device goes away, try to re-open it */
         DEBUGOUT("read Failed rsize = %d errno = %d", rsize, errno);
         close(devicefd);
         devicefd = -1;
         continue;
      }

//                if (verbose) 
//                {
//                   for(rc=0; rc<sizeof(buf); rc++)
//                   {
//                      DEBUGOUT( "%x ", buf[rc]);
//                   }
//                   DEBUGOUT( "\n\n");
//                }

      //if we haven't yet opened the pipe or it got closed, open it
      if(pipefd == -1)
      {
         do
         {
            pipefd = open(pn, O_WRONLY | O_NONBLOCK);
            DEBUGOUT( "Opening named pipe pipefd=%d errno = %d\n",pipefd,errno);
            if(pipefd == -1) sleep(1); //don't baug the cpu
         }while(pipefd == -1 && errno == ENXIO); //wait for a listener
      }
                
      //if we got a listener, start talken
      if(pipefd != -1)
      {
         wsize = write(pipefd, buf, rsize);
         if(wsize<0) 
            DEBUGOUT("Write Failed wsize = %d errno = %d\n", wsize, errno);

         if(wsize < 0 && (errno == EBADF || errno == EPIPE))
         {
            close(pipefd);
            pipefd = -1;
            DEBUGOUT("Pipe Closed - forcing reopen\n");
         }
      }
   }
   return 0;
}
