/*******************************************************************
 * This file is part of the Emulex Linux Device Driver for         *
 * Fibre Channel Host Bus Adapters.                                *
 * Copyright (C) 2003-2005 Emulex.  All rights reserved.           *
 * EMULEX and SLI are trademarks of Emulex.                        *
 * www.emulex.com                                                  *
 *                                                                 *
 * This program is free software; you can redistribute it and/or   *
 * modify it under the terms of version 2 of the GNU General       *
 * Public License as published by the Free Software Foundation.    *
 * This program is distributed in the hope that it will be useful. *
 * ALL EXPRESS OR IMPLIED CONDITIONS, REPRESENTATIONS AND          *
 * WARRANTIES, INCLUDING ANY IMPLIED WARRANTY OF MERCHANTABILITY,  *
 * FITNESS FOR A PARTICULAR PURPOSE, OR NON-INFRINGEMENT, ARE      *
 * DISCLAIMED, EXCEPT TO THE EXTENT THAT SUCH DISCLAIMERS ARE HELD *
 * TO BE LEGALLY INVALID.  See the GNU General Public License for  *
 * more details, a copy of which can be found in the file COPYING  *
 * included with this package.                                     *
 *******************************************************************/

/*
 * $Id: lpfc_cdev.c 2720 2005-11-28 18:58:48Z sf_support $
 */

#include <linux/blkdev.h>
#include <linux/list.h>
#include <linux/mempool.h>
#include <linux/pci.h>
#include <linux/spinlock.h>
#include <linux/mutex.h>
#include <linux/module.h>
#include <scsi/scsi_host.h>
#include <scsi/scsi_transport_fc.h>
#include <linux/ioctl32.h>
#include <linux/kthread.h>

#include "lpfcdfc_version.h"
#include "lpfc_version.h"
#include "lpfc_hw.h"
#include "lpfc_sli.h"
#include "lpfc_disc.h"
#include "lpfc.h"
#include "lpfc_logmsg.h"
#include "lpfc_ioctl.h"
#include "lpfc_hbaapi_ioctl.h"
#include "lpfc_diag.h"
#include "lpfc_misc.h"
#include "lpfc_util_ioctl.h"
#include "lpfc_debug_ioctl.h"

#define LPFCDFC_MODULE_DESC \
	"Emulex LightPulse FC SCSI IOCTL " LPFCDFC_DRIVER_VERSION

DEFINE_MUTEX(lpfcdfc_lock);

struct list_head lpfcdfc_hosts = LIST_HEAD_INIT(lpfcdfc_hosts);
#ifndef HBA_EVENTS_SUPPORTED
struct task_struct * lpfcdfc_ev_poller;
#endif

static int lpfcdfc_major = 0;
unsigned long lpfcdfc_loadtime;

#define LPFC_SCSI_REQ_TMO_DEFAULT 30
int lpfc_scsi_req_tmo = LPFC_SCSI_REQ_TMO_DEFAULT;
module_param(lpfc_scsi_req_tmo, int, 0);
MODULE_PARM_DESC(lpfc_scsi_req_tmo, "ioctl scsi timeout value");

#define lpfcdfc_find_phba(b) lpfcdfc_get_phba_by_inst(b,FIND_HBA)
#define lpfcdfc_validate_phba(b) lpfcdfc_get_phba_by_inst(b,VALIDATE_HBA)

static int lpfc_diag_match_version (struct Scsi_Host   * host)
{

	struct class_device * host_cdev;
	struct class_device_attribute ** attributes;
	char * buf = NULL;
	int    ret_val = 1;

	buf = kmalloc (PAGE_SIZE, GFP_KERNEL);
	if (buf == NULL)
		return 0;

	spin_lock_irq(host->host_lock);
	host_cdev = &(host->shost_classdev);
	attributes = host->hostt->shost_attrs;

	while ((*attributes != NULL)
	       && (strcmp((*attributes)->attr.name, "lpfc_drvr_version") != 0))
		attributes++;

	spin_unlock_irq(host->host_lock);

	if (*attributes != NULL) {
		((*attributes)->show)(host_cdev, buf);
		if (strncmp(buf, LPFC_MODULE_DESC,
			    (sizeof(LPFC_MODULE_DESC) - 1))) {
			printk(KERN_ERR "Wrong version of the lpfc driver: %s\n"
			       "Required version: %s\n", buf, LPFC_MODULE_DESC);
			ret_val = 0;
		}

	} else
		ret_val = 0;

	kfree (buf);

	return ret_val;
}


/* The only reason we need that reverce find, is because we
 * are bent on keeping original calling conventions.
 */
struct lpfcdfc_host * lpfcdfc_host_from_hba(struct lpfc_hba * phba)
{
	struct lpfcdfc_host * dfchba;

	mutex_lock (&lpfcdfc_lock);
	list_for_each_entry(dfchba, &lpfcdfc_hosts, node) {
		if (dfchba->phba == phba)
			break;
	}
	mutex_unlock (&lpfcdfc_lock);

	return dfchba;
}

static inline struct lpfcdfc_host * lpfcdfc_host_add (struct pci_dev * dev,
						      struct Scsi_Host * host,
						      struct lpfc_hba * phba)
{
	struct lpfcdfc_host * dfchba = NULL;
	struct lpfc_sli_ring_mask * prt = NULL;

	dfchba = kzalloc(sizeof(*dfchba), GFP_KERNEL);
	if (dfchba == NULL)
		return NULL;

	pci_dev_get(dev);

	dfchba->inst = phba->brd_no;
	dfchba->phba = phba;
	dfchba->host = host;
	dfchba->dev = dev;
	dfchba->last_hba_state = phba->hba_state;
	dfchba->last_fc_flag = phba->fc_flag;

	spin_lock_irq(phba->host->host_lock);
	prt = phba->sli.ring[LPFC_ELS_RING].prt;
	dfchba->base_els_unsol_event = prt[0].lpfc_sli_rcv_unsol_event;
	dfchba->base_ct_unsol_event = prt[2].lpfc_sli_rcv_unsol_event;
#ifdef HBA_EVENTS_SUPPORTED
	phba->hba_event = lpfcdfc_host_event;
#endif
	prt[0].lpfc_sli_rcv_unsol_event = lpfcdfc_els_unsol_event;
	prt[1].lpfc_sli_rcv_unsol_event = lpfcdfc_els_unsol_event;
	prt[2].lpfc_sli_rcv_unsol_event = lpfcdfc_ct_unsol_event;
	prt[3].lpfc_sli_rcv_unsol_event = lpfcdfc_ct_unsol_event;
	spin_unlock_irq(phba->host->host_lock);
	mutex_lock (&lpfcdfc_lock);
	list_add_tail(&dfchba->node, &lpfcdfc_hosts);
	INIT_LIST_HEAD(&dfchba->ev_waiters);
	mutex_unlock (&lpfcdfc_lock);

	return dfchba;
}


static inline void lpfcdfc_host_del (struct lpfcdfc_host *  dfchba)
{
	struct Scsi_Host * host;
	struct lpfc_hba * phba = NULL;
	struct lpfc_sli_ring_mask * prt = NULL;

	if (dfchba->dev->driver &&
	    try_module_get(dfchba->dev->driver->driver.owner)) {
		host = pci_get_drvdata(dfchba->dev);
		if ((host != NULL) &&
		    (struct lpfc_hba*)host->hostdata == dfchba->phba) {
			phba = dfchba->phba;
			mutex_unlock(&lpfcdfc_lock);
			spin_lock_irq(phba->host->host_lock);
#ifdef HBA_EVENTS_SUPPORTED
			phba->hba_event = NULL;
#endif
			prt = phba->sli.ring[LPFC_ELS_RING].prt;
			prt[0].lpfc_sli_rcv_unsol_event =
				dfchba->base_els_unsol_event;
			prt[1].lpfc_sli_rcv_unsol_event =
				dfchba->base_els_unsol_event;
			prt[2].lpfc_sli_rcv_unsol_event =
				dfchba->base_ct_unsol_event;
			prt[3].lpfc_sli_rcv_unsol_event =
				dfchba->base_ct_unsol_event;
			spin_unlock_irq(phba->host->host_lock);
			mutex_lock(&lpfcdfc_lock);
		}
		module_put(dfchba->dev->driver->driver.owner);
	}
	list_del_init(&dfchba->node);
	pci_dev_put(dfchba->dev);
	kfree (dfchba);
}

/*
 * Retrieve lpfc_hba * matching instance (board no)
 * If found return lpfc_hba *
 * If not found return NULL
 */
struct lpfcdfc_host *
lpfcdfc_get_phba_by_inst(int inst, enum get_phba_flag val_flag)
{
	struct lpfc_hba * phba = NULL;
	struct Scsi_Host * host = NULL;
	struct pci_dev * dev = NULL;
	struct lpfcdfc_host * dfchba;

	mutex_lock (&lpfcdfc_lock);
	list_for_each_entry(dfchba, &lpfcdfc_hosts, node) {
		if (dfchba->inst == inst) {
			if (dfchba->dev->driver &&
			    try_module_get(dfchba->dev->driver->driver.owner)) {
				host = pci_get_drvdata(dfchba->dev);
				if ((host != NULL) &&
				    (struct lpfc_hba*)host->hostdata ==
				    dfchba->phba) {
					mutex_unlock (&lpfcdfc_lock);
					BUG_ON(dfchba->phba->brd_no != inst);
					return dfchba;
				}
				module_put(dfchba->dev->driver->driver.owner);
			}
			lpfcdfc_host_del (dfchba);
			break;
		}
	}
	mutex_unlock (&lpfcdfc_lock);

	if (val_flag == VALIDATE_HBA)
		return NULL;

	dfchba = NULL;

	while ((dev = pci_get_device(PCI_VENDOR_ID_EMULEX, PCI_ANY_ID, dev))
	       != NULL) {
		if (dev->driver && try_module_get(dev->driver->driver.owner)) {
			host = pci_get_drvdata(dev);
			if(host != NULL) {
				if (!lpfc_diag_match_version (host)) {
					module_put(dev->driver->driver.owner);
					pci_dev_put(dev);
					break;
				}

				phba = (struct lpfc_hba*)host->hostdata;

				if(phba->brd_no == inst) {
					dfchba = lpfcdfc_host_add(dev, host,
								  phba);
					break;
				}
			}
			module_put(dev->driver->driver.owner);
		}
	}

	return dfchba;
}

static void
lpfcdfc_init(void)
{
	struct Scsi_Host * host = NULL;
	struct pci_dev * dev = NULL;
	struct lpfcdfc_host * dfchba;
	struct lpfc_hba * phba = NULL;


	while ((dev = pci_get_device(PCI_VENDOR_ID_EMULEX, PCI_ANY_ID, dev))
	       != NULL) {
		if (dev->driver && try_module_get(dev->driver->driver.owner)) {
			host = pci_get_drvdata(dev);

			if (host == NULL) {
				module_put(dev->driver->driver.owner);
				continue;
			}

			phba = (struct lpfc_hba*)host->hostdata;
			if (phba == NULL) {
				module_put(dev->driver->driver.owner);
				continue;
			}

			if (!lpfc_diag_match_version (host)) {
				module_put(dev->driver->driver.owner);
				pci_dev_put(dev);
				break;
			}

			dfchba = lpfcdfc_host_add(dev, host, phba);
			module_put(dev->driver->driver.owner);
		}
	}

	lpfcdfc_ev_poller = kthread_run(lpfcdfc_do_ev_poll, &lpfcdfc_hosts,
					"lpfcdfc_event_poll");

}

static void
lpfcdfc_uninit(void)
{
	struct lpfcdfc_host * dfchba;
	struct lpfcdfc_host * next_dfchba;

	kthread_stop(lpfcdfc_ev_poller);

	mutex_lock (&lpfcdfc_lock);

	list_for_each_entry_safe(dfchba, next_dfchba, &lpfcdfc_hosts, node)
		lpfcdfc_host_del (dfchba);

	mutex_unlock (&lpfcdfc_lock);
}

static int
lpfcdfc_do_ioctl(LPFCCMDINPUT_t *cip)
{
	struct lpfcdfc_host * dfchba = NULL;
	struct lpfc_hba *phba = NULL;
	int rc;
	uint32_t total_mem;
	void   *dataout;

	/* Some ioctls are per module and do not need phba */
	switch (cip->lpfc_cmd) {
	case LPFC_INITBRDS:
		dfchba = lpfcdfc_find_phba(cip->lpfc_brd);
		if (dfchba == NULL)
			return EINVAL;
		phba = dfchba->phba;
		break;
	case LPFC_GET_DFC_REV:
		break;
	default:
		dfchba = lpfcdfc_validate_phba(cip->lpfc_brd);
		if (dfchba == NULL)
			return EINVAL;
		phba = dfchba->phba;
		break;
	};

	if (phba)
		lpfc_printf_log(phba,
			KERN_INFO,
			LOG_LIBDFC,
			"%d:1601 libdfc ioctl entry Data: x%x x%lx x%lx x%x\n",
			phba->brd_no, cip->lpfc_cmd,
			(ulong) cip->lpfc_arg1,(ulong) cip->lpfc_arg2,
			cip->lpfc_outsz);

	if (cip->lpfc_outsz >= 4096) {

		/* Allocate memory for ioctl data. If buffer is bigger than 64k,
		 * then we allocate 64k and re-use that buffer over and over to
		 * xfer the whole block. This is because Linux kernel has a
		 * problem allocating more than 120k of kernel space memory. Saw
		 * problem with GET_FCPTARGETMAPPING...
		 */
		if (cip->lpfc_outsz <= (64 * 1024))
			total_mem = cip->lpfc_outsz;
		else
			total_mem = 64 * 1024;
	} else {
		/* Allocate memory for ioctl data */
		total_mem = 4096;
	}

	dataout = kmalloc(total_mem, GFP_KERNEL);
	if (!dataout && dfchba != NULL) {
		module_put(dfchba->dev->driver->driver.owner);
		return ENOMEM;
	}

	switch (cip->lpfc_cmd) {

	/* Diagnostic Interface Library Support - util */
	case LPFC_WRITE_PCI:
		rc = lpfc_ioctl_write_pci(phba, cip);
		break;

	case LPFC_READ_PCI:
		rc = lpfc_ioctl_read_pci(phba, cip, dataout);
		break;

	case LPFC_WRITE_MEM:
		rc = lpfc_ioctl_write_mem(phba, cip);
		break;

	case LPFC_READ_MEM:
		rc = lpfc_ioctl_read_mem(phba, cip, dataout);
		break;

	case LPFC_WRITE_CTLREG:
		rc = lpfc_ioctl_write_ctlreg(phba, cip);
		break;

	case LPFC_READ_CTLREG:
		rc = lpfc_ioctl_read_ctlreg(phba, cip, dataout);
		break;

	case LPFC_GET_DFC_REV:
		((DfcRevInfo *) dataout)->a_Major = DFC_MAJOR_REV;
		((DfcRevInfo *) dataout)->a_Minor = DFC_MINOR_REV;
		cip->lpfc_outsz = sizeof (DfcRevInfo);
		rc = 0;
		break;

	case LPFC_INITBRDS:
		rc = lpfc_ioctl_initboard(phba, cip, dataout);
		break;

	case LPFC_SETDIAG:
		rc = lpfc_ioctl_setdiag(phba, cip, dataout);
		break;

	case LPFC_HBA_SEND_SCSI:
	case LPFC_HBA_SEND_FCP:
		rc = lpfc_ioctl_send_scsi_fcp(phba, cip);
		break;

	case LPFC_SEND_ELS:
		rc = lpfc_ioctl_send_els(phba, cip, dataout);
		break;

	case LPFC_HBA_SEND_MGMT_RSP:
		rc = lpfc_ioctl_send_mgmt_rsp(phba, cip);
		break;

	case LPFC_HBA_SEND_MGMT_CMD:
	case LPFC_CT:
		rc = lpfc_ioctl_send_mgmt_cmd(phba, cip, dataout);
		break;

	case LPFC_MBOX:
		rc = lpfc_ioctl_mbox(phba, cip, dataout);
		break;

	case LPFC_LINKINFO:
		rc = lpfc_ioctl_linkinfo(phba, cip, dataout);
		break;

	case LPFC_IOINFO:
		rc = lpfc_ioctl_ioinfo(phba, cip, dataout);
		break;

	case LPFC_NODEINFO:
		rc = lpfc_ioctl_nodeinfo(phba, cip, dataout, total_mem);
		break;

	case LPFC_GETCFG:
		rc = lpfc_ioctl_getcfg(phba, cip, dataout);
		break;

	case LPFC_SETCFG:
		rc = lpfc_ioctl_setcfg(phba, cip);
		break;

	case LPFC_HBA_GET_EVENT:
		rc = lpfc_ioctl_hba_get_event(phba, cip, dataout, total_mem);
		break;

	case LPFC_HBA_SET_EVENT:
		rc = lpfc_ioctl_hba_set_event(phba, cip);
		break;

	case LPFC_LIST_BIND:
		rc = lpfc_ioctl_list_bind(phba, cip, dataout);
		break;

	case LPFC_GET_VPD:
		rc = lpfc_ioctl_get_vpd(phba, cip, dataout);
		break;

	case LPFC_GET_DUMPREGION:
		rc = lpfc_ioctl_get_dumpregion(phba, cip, dataout);
		break;

	case LPFC_GET_LPFCDFC_INFO:
		rc = lpfc_ioctl_get_lpfcdfc_info(phba, cip, dataout);
		break;

	case LPFC_LOOPBACK_MODE:
		rc = lpfc_ioctl_loopback_mode(phba, cip, dataout);
		break;

	case LPFC_LOOPBACK_TEST:
		rc = lpfc_ioctl_loopback_test(phba, cip, dataout);
		break;

	/* Debug Interface Support - dfc */
	case LPFC_LIP:
		rc = lpfc_ioctl_lip(phba, cip, dataout);
		break;

	case LPFC_INST:
		rc = lpfc_ioctl_inst(phba, cip, dataout);
		break;

	case LPFC_RESET:
		rc = lpfc_ioctl_reset(phba, cip);
		break;

	case LPFC_READ_HBA:
		rc = lpfc_ioctl_read_hba(phba, cip, dataout, total_mem);
		break;

	/* Diagnostic Interface Library Support - hbaapi */
	case LPFC_HBA_ADAPTERATTRIBUTES:
		rc = lpfc_ioctl_hba_adapterattributes(phba, cip, dataout);
		break;

	case LPFC_HBA_PORTATTRIBUTES:
		rc = lpfc_ioctl_hba_portattributes(phba, cip, dataout);
		break;

	case LPFC_HBA_PORTSTATISTICS:
		rc = lpfc_ioctl_hba_portstatistics(phba, cip, dataout);
		break;

	case LPFC_HBA_WWPNPORTATTRIBUTES:
		rc = lpfc_ioctl_hba_wwpnportattributes(phba, cip, dataout);
		break;

	case LPFC_HBA_DISCPORTATTRIBUTES:
		rc = lpfc_ioctl_hba_discportattributes(phba, cip, dataout);
		break;

	case LPFC_HBA_INDEXPORTATTRIBUTES:
		rc = lpfc_ioctl_hba_indexportattributes(phba, cip, dataout);
		break;

	case LPFC_HBA_SETMGMTINFO:
		rc = lpfc_ioctl_hba_setmgmtinfo(phba, cip);
		break;

	case LPFC_HBA_GETMGMTINFO:
		rc = lpfc_ioctl_hba_getmgmtinfo(phba, cip, dataout);
		break;

	case LPFC_HBA_REFRESHINFO:
		rc = lpfc_ioctl_hba_refreshinfo(phba, cip, dataout);
		break;

	case LPFC_HBA_RNID:
		rc = lpfc_ioctl_hba_rnid(phba, cip, dataout);
		break;

	case LPFC_HBA_GETEVENT:
		rc = lpfc_ioctl_hba_getevent(phba, cip, dataout);
		break;

	default:
		rc = EINVAL;
		break;
	}

	if (phba)
		lpfc_printf_log(phba,
			KERN_INFO,
			LOG_LIBDFC,
			"%d:1602 libdfc ioctl exit Data: x%x x%x x%x\n",
			cip->lpfc_brd,
			rc,
			cip->lpfc_outsz,
			(uint32_t) ((ulong) cip->lpfc_dataout));

	/* Copy data to user space config method */
	if (rc == 0) {
		if (cip->lpfc_outsz) {
			if (copy_to_user
			    ((uint8_t *) cip->lpfc_dataout,
			     (uint8_t *) dataout, (int)cip->lpfc_outsz)) {
				rc = EIO;
			}
		}
	}

	kfree(dataout);
	if (dfchba != NULL) {
		module_put(dfchba->dev->driver->driver.owner);
	}
	return rc;
}

static int
lpfcdfc_ioctl(struct inode *inode,
	      struct file *file, unsigned int cmd, unsigned long arg)
{
	int rc;
	LPFCCMDINPUT_t *ci;

	if (!arg)
		return -EINVAL;

	ci = (LPFCCMDINPUT_t *) kmalloc(sizeof (LPFCCMDINPUT_t), GFP_KERNEL);

	if (!ci)
		return -ENOMEM;

	if ((rc = copy_from_user
	       ((uint8_t *) ci, (uint8_t *) arg, sizeof (LPFCCMDINPUT_t)))) {
		kfree(ci);
		return -EIO;
	}

	rc = lpfcdfc_do_ioctl(ci);

	kfree(ci);
	return -rc;
}

#ifdef CONFIG_COMPAT
static long
lpfcdfc_compat_ioctl(struct file * file, unsigned int cmd, unsigned long arg)
{
	LPFCCMDINPUT32_t arg32;
	LPFCCMDINPUT_t arg64;
	int ret;

	if(copy_from_user(&arg32, (void*)arg, sizeof(LPFCCMDINPUT32_t)))
		return -EFAULT;

	arg64.lpfc_brd = arg32.lpfc_brd;
	arg64.lpfc_ring = arg32.lpfc_ring;
	arg64.lpfc_iocb = arg32.lpfc_iocb;
	arg64.lpfc_flag = arg32.lpfc_flag;
	arg64.lpfc_arg1 = (void *)(unsigned long) arg32.lpfc_arg1;
	arg64.lpfc_arg2 = (void *)(unsigned long) arg32.lpfc_arg2;
	arg64.lpfc_arg3 = (void *)(unsigned long) arg32.lpfc_arg3;
	arg64.lpfc_dataout = (void *)(unsigned long) arg32.lpfc_dataout;
	arg64.lpfc_cmd = arg32.lpfc_cmd;
	arg64.lpfc_outsz = arg32.lpfc_outsz;
	arg64.lpfc_arg4 = arg32.lpfc_arg4;
	arg64.lpfc_arg5 = arg32.lpfc_arg5;

	ret = lpfcdfc_do_ioctl(&arg64);

	arg32.lpfc_brd = arg64.lpfc_brd;
	arg32.lpfc_ring = arg64.lpfc_ring;
	arg32.lpfc_iocb = arg64.lpfc_iocb;
	arg32.lpfc_flag = arg64.lpfc_flag;
	arg32.lpfc_arg1 = (u32)(unsigned long) arg64.lpfc_arg1;
	arg32.lpfc_arg2 = (u32)(unsigned long) arg64.lpfc_arg2;
	arg32.lpfc_arg3 = (u32)(unsigned long) arg64.lpfc_arg3;
	arg32.lpfc_dataout = (u32)(unsigned long) arg64.lpfc_dataout;
	arg32.lpfc_cmd = arg64.lpfc_cmd;
	arg32.lpfc_outsz = arg64.lpfc_outsz;
	arg32.lpfc_arg4 = arg64.lpfc_arg4;
	arg32.lpfc_arg5 = arg64.lpfc_arg5;

	if(copy_to_user((void*)arg, &arg32, sizeof(LPFCCMDINPUT32_t)))
		return -EFAULT;

	return -ret;
}
#endif

static struct file_operations lpfc_fops = {
	.owner        = THIS_MODULE,
	.ioctl        = lpfcdfc_ioctl,
#ifdef CONFIG_COMPAT
	.compat_ioctl = lpfcdfc_compat_ioctl,
#endif
};

static int __init
lpfcdfc_cdev_init(void)
{
	printk(LPFCDFC_MODULE_DESC "\n");
	printk(LPFCDFC_COPYRIGHT "\n");

	lpfcdfc_major = register_chrdev(0, LPFCDFC_DRIVER_NAME, &lpfc_fops);
	if (lpfcdfc_major < 0) {
		printk("%s: unable to register \"%s\" device.\n",
		       __FUNCTION__, LPFCDFC_DRIVER_NAME);
		return lpfcdfc_major;
	}

	lpfcdfc_loadtime = jiffies;

	if (lpfc_scsi_req_tmo > LPFC_MAX_SCSI_REQ_TMO) {
		lpfc_scsi_req_tmo =  LPFC_DFT_SCSI_REQ_TMO;
	}

	if (lpfc_scsi_req_tmo != LPFC_SCSI_REQ_TMO_DEFAULT) {
		printk("%s: setting scsi request timeout, lpfc_scsi_req_tmo, "
		       "to %d secs\n", __FUNCTION__, lpfc_scsi_req_tmo);
	}

	lpfcdfc_init();

	return 0;
}

static void __exit
lpfcdfc_cdev_exit(void)
{
	lpfcdfc_uninit();
	unregister_chrdev(lpfcdfc_major, LPFCDFC_DRIVER_NAME);
}

module_init(lpfcdfc_cdev_init);
module_exit(lpfcdfc_cdev_exit);

MODULE_DESCRIPTION("Emulex LightPulse Fibre Channel driver IOCTL support");
MODULE_AUTHOR("Emulex Corporation - tech.support@emulex.com");
MODULE_LICENSE("GPL");
