/*
 * (C) Copyright 2000-2005 Hewlett-Packard Development Company, L.P.
 */
#include "../CpqCiHlx.h"
#include "../CpqCiWrp.h"
#include "../CpqCiTyp.h"
#include "CpqCiSem.h"

void hexdump(void *base, void *data, unsigned len);

//BEGIN Semaphore implementation
//Sem array
static struct semaphore* global_sem = NULL;
//Sem array size
static int global_sem_size = 0;
//global sem timer list 
LIST_HEAD(global_sem_timer);
//a spinlock when operating on global_sem_timer
spinlock_t global_sem_timer_lock = SPIN_LOCK_UNLOCKED;

//Read Write lock for ALL semaphore operations
DECLARE_RWSEM(cpqci_rwlock);

static int cpqci_del_sem_timeout_ex(sem_timeout* rc, int timer_lock, int timer_free);
static int cpqci_del_sem_timeout(sem_timeout* rc) ;
static int cpqci_del_all_sem_timeout(void);
static void cpqci_handle_sem_timeout(unsigned long data);
static int cpqci_add_sem_timeout(struct task_struct* notify_task, int ms, sem_timeout** rc);
static int cpqci_global_sem_create(int size) ;

void hexdump(void *base, void *data, unsigned len)
{
	unsigned char *d = data;
	unsigned char pbuf[17];
	unsigned i = 0;
	while (i < len) {
		if (i % 16 == 0) {
			if (i)
				printk("  %s\n%08lx  ", pbuf, i+((unsigned long)data-(unsigned long)base));

			else
				printk("%08lx  ", i+((unsigned long)data-(unsigned long)base));
			memset(pbuf, 0, sizeof(pbuf));
		}
		printk("%02x ", *d);
		pbuf[i % 16] = (*d > 32 && *d < 128) ? *d : '.';
		d++;
		i++;
	}
	printk("  %s\n", pbuf);
}


static int cpqci_del_sem_timeout_ex(sem_timeout* rc, int timer_lock, int timer_free)
{
	unsigned long flags=0;
	if (rc == NULL) {
		return -1;
	}

	DBG("del_sem_timeout_ex %x %d %d\n", rc, timer_lock, timer_free);
	if (timer_lock) {
		//DBG("timer_lock\n");
		spin_lock_irqsave(&global_sem_timer_lock, flags);
		//DBG("done\n");
	}
#if defined (CPQCI_TIMER_FLAG) || LINUX_VERSION_CODE >= 0x020500
	if ((rc->timer.entry.next != NULL) && (rc->timer.entry.prev != NULL)) {
		//DBG("del_timer %x\n", &(rc->timer));
		del_timer(&(rc->timer));
		//DBG("reset pointers\n");
		rc->timer.entry.next = rc->timer.entry.prev = NULL;
		//DBG("done\n");
	}
#else
	if ((rc->timer.list.next != NULL) && (rc->timer.list.prev != NULL)) {
		//DBG("del_timer %x\n", &(rc->timer));
		del_timer(&(rc->timer));
		//DBG("reset pointers\n");
		rc->timer.list.next = rc->timer.list.prev = NULL;
		//DBG("done\n");
	}
#endif

	if ((rc->next != NULL) && (rc->prev != NULL)) {
		//DBG("list_del %x\n", rc);
		list_del((struct list_head*)rc);
		rc->next = rc->prev = NULL;
		//DBG("done\n");
	}
	if (timer_free) {
		//DBG("kfree %x\n", rc);
		kfree(rc);
		//DBG("done\n");
	}
	if (timer_lock) {
		//DBG("timer_unlock\n");
		spin_unlock_irqrestore(&global_sem_timer_lock, flags);
		//DBG("done\n");
	}
	DBG("end\n");

	return 0;

}

static int cpqci_del_sem_timeout(sem_timeout* rc) 
{
	DBG("del_sem_timeout\n");
	return cpqci_del_sem_timeout_ex(rc, 1, 1);
}

static int cpqci_del_all_sem_timeout()
{
	unsigned long flags;
	struct list_head *curr, *next;

	DBG("del_all_sem_timeout\n");
	spin_lock_irqsave(&global_sem_timer_lock, flags);
	//DBG("timer_lock\n");
	curr = global_sem_timer.next;
	while (curr != &global_sem_timer) {
		next = curr->next;
		cpqci_del_sem_timeout_ex((sem_timeout*)curr, 0, 1); 
		curr = next;
	}
	//DBG("timer_unlock\n");
	spin_unlock_irqrestore(&global_sem_timer_lock, flags);

	return 0;
}

static void cpqci_handle_sem_timeout(unsigned long data)
{
	sem_timeout* rc = (sem_timeout*) data;
	struct task_struct *p;
	DBG("handle_sem_timeout %x\n", rc);

	p = rc->notify_task;
	if (p != NULL) {
		DBG("force_sig SIGWINCH pid %d %x\n", p->pid, p);
		force_sig(SIGWINCH, p);
	}

#if defined (CPQCI_TIMER_FLAG) || LINUX_VERSION_CODE >= 0x020500
	rc->timer.entry.next = rc->timer.entry.prev = NULL;
#else
	rc->timer.list.next = rc->timer.list.prev = NULL;
#endif

	cpqci_del_sem_timeout_ex(rc, 0, 0);
}

static int cpqci_add_sem_timeout(struct task_struct* notify_task, int ms, sem_timeout** rc)
{
	unsigned long flags;
	sem_timeout* st;

	DBG("add_sem_timeout task %x pid %d ms %d\n", notify_task, notify_task->pid, ms);
	st = (sem_timeout*) kmalloc(sizeof(sem_timeout), GFP_KERNEL);
	if (st == NULL) {
		return -ENOMEM;
	}
	DBG("sem_timeout %x\n", st);

	INIT_LIST_HEAD((struct list_head*)st);
	//DBG("init_timer\n");
	init_timer(&(st->timer));
	st->timer.function= cpqci_handle_sem_timeout;
	st->timer.expires = jiffies + ms * HZ / 1000;
	st->timer.data = (unsigned long)st;
	st->notify_task = notify_task;

	//DBG("lock_timer\n");
	spin_lock_irqsave(&global_sem_timer_lock, flags);
	//DBG("add to timer_list\n");
	list_add_tail((struct list_head*) st, &global_sem_timer);
	//DBG("add_timer\n");
	add_timer(&(st->timer));
	spin_unlock_irqrestore(&global_sem_timer_lock, flags);
	//DBG("unlock_timer\n");

	if (rc != NULL) {
		*rc = st;
	}

	return 0;
}

int cpqci_global_sem_destroy(void) 
{
	if (global_sem == NULL) return 0;

	//abort the timer list
	cpqci_del_all_sem_timeout();
	//clean up array
	kfree(global_sem);
	global_sem = NULL;
	global_sem_size = 0;
	return 0;
}

static int cpqci_global_sem_create(int size) 
{
	int i;

	cpqci_global_sem_destroy(); 
	global_sem = kmalloc(sizeof(struct semaphore) * size, GFP_KERNEL);
	if (!global_sem) return -EFAULT;
	global_sem_size = size;
	for (i=0; i<size; i++) sema_init(global_sem + i, 0);
	return 0;
}

int down_timeout(struct semaphore* sem, int ms)
{
	int rc;
	sem_timeout* st;
	rc = cpqci_add_sem_timeout(current, ms, &st);
	if (rc == 0) {
		rc = down_interruptible(sem);
		cpqci_del_sem_timeout(st);
		if (rc != 0) DBG("down_timeout %p failure rc = %d\n", sem, rc);
		if (signal_pending(current)) {
			DBG("down_timeout %p signal pending\n", sem);
			rc = -EINTR;
		}
	}
	return rc;
}

//END SEMAPHORE implementation

int sem_ioctl(struct inode *inod, struct file *f, unsigned int cmd, unsigned long arg)
{
	sem_timeout* st;
	sem_value_pair data;
	struct semaphore* sem = NULL;
	data.rc = 0;

	if (arg) {
		DBG("copy_from_user %p %p\n", &data, arg);
		if (copy_from_user(&data, (sem_value_pair*)arg, sizeof(sem_value_pair)))  return -EFAULT; 
	}

	if(cmd != SEM_DESTROY && cmd != SEM_CREATE) {
		down_read(&cpqci_rwlock);
	} else {
		down_write(&cpqci_rwlock);
	}

	DBG("check if sem is NULL\n");
	if (data.sem == NULL && cmd != SEM_CREATE) return -EFAULT;
	DBG("check if sem passed in %p is global %p\n", data.sem, (int)global_sem);
	if (data.sem != NULL && (int)data.sem != (int)global_sem) return -EFAULT;
	sem = global_sem;

	DBG("sem_ioctl start: %p, value = %x, rc = %x\n", sem, data.value, data.rc);

	switch (cmd) {
	case SEM_CREATE:
		data.rc = cpqci_global_sem_create(data.value);
		data.sem = global_sem;
		//flush_signals(current);
		//recalc_sigpending(current);
		DBG("SEM_CREATE %p\n", data.sem);
		break;
	case SEM_DESTROY:
		cpqci_global_sem_destroy();
		DBG("SEM_DESTROY %p\n", sem);
		break;
	case SEM_UP:
		up(sem + data.value);
		DBG("SEM_UP %p\n", sem + data.value);
		break;
	case SEM_DOWN:
		DBG("SEM_DOWN start %p\n", sem + data.value);
		data.rc = down_interruptible(sem + data.value);
		if (data.rc != 0) DBG("SEM_DOWN %d failure rc = %d\n", data.value, data.rc);
		if (signal_pending(current)) {
			DBG("SEM_DOWN signal pending\n");
			data.rc = -EINTR;
		}
		DBG("SEM_DOWN stop %p\n", sem + data.value, data.rc);
		break;
	case SEM_DOWN_TIMER:
		DBG("SEM_DOWN_TIMER start %p ms %d\n", sem + data.value, data.rc);
		data.rc = cpqci_add_sem_timeout(current, data.rc, &st);
		if (data.rc == 0) {
			data.rc = down_interruptible(sem + data.value);
			cpqci_del_sem_timeout(st);
			if (data.rc != 0) DBG("SEM_DOWN_TIMER %d failure rc = %d\n", data.value, data.rc);
			if (signal_pending(current)) {
				DBG("SEM_DOWN_TIMER signal pending\n");
				data.rc = -EINTR;
			}
		}
		DBG("SEM_DOWN_TIMER stop %p %d\n", sem + data.value, data.rc);
		break;
	case SEM_GET_VALUE:
		data.rc= atomic_read(&((sem + data.value)->count));
		DBG("SEM_GET_VALUE %p %d\n", sem + data.value, data.rc);
		break;
	case SEM_SET_VALUE:
		sema_init(sem + data.value, data.rc);
		DBG("SEM_SET_VALUE %p %d\n", sem + data.value, data.rc);
		break;
	default:
		data.rc= -EFAULT;
	}

	if(cmd != SEM_DESTROY && cmd != SEM_CREATE) {
		up_read(&cpqci_rwlock);
	} else {
		up_write(&cpqci_rwlock);
	}
	

	DBG("sem_ioctl end: %p, value = %x, rc = %x\n", sem, data.value, data.rc);

	if (arg) {
		DBG("copy_to_user %p %p\n", arg, &data);
		if (copy_to_user((sem_value_pair*)arg, &data, sizeof(sem_value_pair)))  return -EFAULT; 
	}
	return 0;
}
