//
// Name: ScsiInfo.h
// Copyright: (c) Copyright 2005 Hewlett-Packard Development Company, L.P.
//
// Description: header file which contains the implementation of the ScsiInfo
//              object
//
// CD		07/07/05 Initial Development
// CD		04/20/06 Fix WWID parsing for XP arrays
// CD		05/22/06 Check for string length of 0 when trimming whitespace
//                       to avoid segmentation fault when SCSI device is in 
//                       odd state

#include <iostream>
#include <unistd.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <scsi/sg.h>
#include <scsi/scsi.h>

using namespace std;

// generic defines

#define DEVICESTRLEN 255 // max length of a device name that can
                         // be passed in
#define INQ_REPLY_LEN 255 // size of buffer used to hold inquiry return
			  // data
#define INQ_CMD_CODE 0x12 // SCSI opcode for inquiry
#define INQ_CMD_LEN 6 // length in bytes of SCSI CDB for inquiry command
#define SENSE_DATA_LEN 32 // length of sense data buffer
#define XP_MODEL_NAME "OPEN-"

// error defines for ScsiInfo class

#define SCINSUCCESS 0 // good status
#define SCINSTRTOLONG -1 // name of device is too long
#define SCINNOEXIST -2 // device does not exist 
#define SCINDEVNOOPEN -3 // open failed
#define SCINIOCTLFAIL -4 // ioctl call failed 
#define SCINRETURNERROR -5 // ioctl succeeded but the actual SCSI command returned an error
#define SCINGENERICERROR -6 // error code for non specific errors

// structure definition for SCSI_IOCTL_GET_IDLUN ioctl call

typedef struct scsi_idlun {
    int four_in_one;    /* 4 separate bytes of info compacted into 1 int */
    int host_unique_id; /* distinguishes adapter cards from same supplier */
} scin_scsi_idlun_t;

// ScsiInfo class - performs inquiries on page 0x0 and page 0x83
// of a SCSI device to get the following data:
//
// SCSI vendor string
// SCSI product string
// firmware version
// World Wide Name of the port that the device is attached to
// 128 bit UUID 

class ScsiInfo {
 // public functions and data
 public:
 	int Host; // SCSI host adapter number
	int Bus; // bus number
	int Target; // target number
	int Lun; // LUN number
	string Vendor; // Vendor string returned from inquiry command
 	string Device; // Device string returned from inquiry command 
 	string Firmware; // firmware version returned from inquiry command
 	string WWN; // world wide ID in hex returned from inquiry command
 	string WWLID; // UUID of LUN in hex returned from inquiry command
	ScsiInfo();
	int GetDeviceInfo();
 	void PrintDeviceInfo();
 	int SetDeviceName(char *device);
 	int DoesDeviceExist();
	void PrintError(int error_num, char error_msg[]);
 // private functions and data
 private: 
	char device_name[DEVICESTRLEN]; // name of device to open
	sg_io_hdr_t io_hdr; // SG_IO structure to send a SCSI command
	unsigned char sense_data_buf[SENSE_DATA_LEN]; // buffer to hold sense data on error
	int GetAddress();
	int DoInquiry0();
	int DoInquiry83();
	void InitializeData();
	int isXP(string VendorString, string ModelString);
};


// Initializes Date to NULL or 0
//
// In: none
// Out: Vendor, Device, Firmware, WWN, WWLID, device, sense_data_buf all to null to 0's

void ScsiInfo::InitializeData()
{
 // initialize class strings
 
 Host = 0;
 Bus = 0;
 Target = 0;
 Lun = 0;
 Vendor = "";
 Device = "";
 Firmware = "";
 WWN = "";
 WWLID = "";
 memset (&sense_data_buf, 0, SENSE_DATA_LEN); // set sense data buffer to all zeroes
}


// Constructor for ScsiInfo class
//
// In: none
// Out: none

ScsiInfo::ScsiInfo ()
{
 InitializeData();
}


// Sets the device_name member variable of the ScsiInfo
// class to whatever is passed into the function
// 
// In: device name
// Out: device_name private variable set to the name of the
//      device to open
//      

int ScsiInfo::SetDeviceName(char *device)
{
 if ( strlen(device) > DEVICESTRLEN )
 {
	// return an error if the length of the string is greater
	// than DEVICESTRLEN
 	return (SCINSTRTOLONG);
 }
 else
 {
 	strcpy (device_name, device);
 }

 return (SCINSUCCESS);
}


// Takes the device that is in the device_name and uses the
// stat system call to see if the file exists
//
// In: none
// Out: return status of stat command (0 for success, otherwise
//      nonzero for failure)

int ScsiInfo::DoesDeviceExist()
{
 int rc;
 struct stat buf;

 rc = stat (device_name, &buf);
 
 // if the stat function call failed, return SCINDEVNOOPEN
 if ( rc != 0 )
 {
 	return (SCINNOEXIST); 
 }
 
 return (SCINSUCCESS);
}


// Sends an INQUIRY to page 0x0 and page 0x83 to the device in
// device_name
//
// In: none
// Out: Sets the Vendor, Device, Firmware, WWN, WWLID variables
//      to the values sent back from the INQUIRY command

int ScsiInfo::GetDeviceInfo ()
{
 int rc;

 // Reinitialize all data
 
 InitializeData();

 // Get SCSI address

 rc = GetAddress();

 if ( rc != SCINSUCCESS)
 {
	return (rc);
 }

 // Do Inquiry(s)

 rc = DoInquiry0();

 if ( rc != SCINSUCCESS)
 {
	return (rc);
 }

 rc = DoInquiry83();

 if ( rc != SCINSUCCESS)
 {
	return (rc);
 }

 return (SCINSUCCESS);
}


// Opens the device and uses the SCSI_IOCTL_GET_IDLUN ioctl to get
// the SCSI address of the device
// 
// In: none
// Out: Host, Bus, Target, and Lun set to the address of the
//      device we are looking at

int ScsiInfo::GetAddress ()
{
 int sg_fd;
 scin_scsi_idlun_t address;

 if ((sg_fd = open(device_name, O_RDONLY)) < 0)
	return (SCINDEVNOOPEN);

 // send the SCSI_IOCTL_GET_IDLUN ioctl command to the device

 if ((ioctl(sg_fd, SCSI_IOCTL_GET_IDLUN, &address)) < 0)
 {
 	return (SCINIOCTLFAIL);
 }

 // close device

 close (sg_fd);

 // get SCSI address info from sg_scsi_id structure

 Host = ((address.four_in_one >> 24) & 0xff);
 Bus = ((address.four_in_one >> 16) & 0xff);
 Target = (address.four_in_one & 0xff);
 Lun = ((address.four_in_one >> 8) & 0xff);
 
 return (SCINSUCCESS);
}


// Opens the device and performs a page 0x0 SCSI inquiry
// to get information from the device
//
// In: none
// Out: Sets the Vendor, Device, Firmware, WWN based on data
//      from the inquiry

int ScsiInfo::DoInquiry0 ()
{
 unsigned char inqCmdBlk[INQ_CMD_LEN] = {INQ_CMD_CODE, 0, 0x0, 0,INQ_REPLY_LEN, 0}; // Inquiry page 0x0 Command Descriptor Block
 int sg_fd; // file descriptor for SCSI device
 int i, k, cnt, done;
 unsigned char inqBuff[INQ_REPLY_LEN]; // buffer for data returned from inquiry
 char tmpstr[255]; // used to get information from the inquiry buffer
 
 // open device for reading 
 
 if ((sg_fd = open(device_name, O_RDONLY)) < 0)
 {
        return (SCINDEVNOOPEN);
 }

 // Prepare INQUIRY command
 
 memset(&io_hdr, 0, sizeof(sg_io_hdr_t)); // set all elements of SG_IO structure to 0
 io_hdr.interface_id = 'S'; // S for SCSI?
 io_hdr.cmd_len = sizeof(inqCmdBlk); // size of CDB
 io_hdr.mx_sb_len = sizeof(sense_data_buf); // pointer to buffer for sense data
 io_hdr.dxfer_direction = SG_DXFER_FROM_DEV; // i/o mode; in this case from the device
 io_hdr.dxfer_len = INQ_REPLY_LEN; // size of SCSI command reply buffer
 io_hdr.dxferp = inqBuff; // pointer to SCSI command reply buffer
 io_hdr.cmdp = inqCmdBlk; // pointer to buffer with CDB in it
 io_hdr.sbp = sense_data_buf; // pointer to sense data buffer
 io_hdr.timeout = 20000; //20000 millisecs == 20 seconds

 // send command to device using IOCTL
 
 if (ioctl(sg_fd, SG_IO, &io_hdr) < 0) 
 {
	return (SCINIOCTLFAIL);
 }

 // close device

 close (sg_fd);

 // check status returned from command

 if ((io_hdr.info & SG_INFO_OK_MASK) != SG_INFO_OK)
 	return (SCINRETURNERROR);
 else if (io_hdr.masked_status)
	return (SCINRETURNERROR);
 else if (io_hdr.host_status)
	return (SCINRETURNERROR);
 else if (io_hdr.driver_status)
	return (SCINRETURNERROR);            
 
 // get vendor ID (bytes 8-15)

 cnt = 0;
 for (i = 8;i <= 15;i++)
 {
 	tmpstr[cnt] = (char)inqBuff[i];
	cnt++;
 }
 tmpstr[cnt] = 0;

 // Remove trailing whitespace
 //
 // Note that we have to check the length of tmpstr first before we do anything
 // to make sure we don't get a seg fault

 if (strlen(tmpstr) > 0)
 {
 	done = 0;
 	for (i = (strlen(tmpstr) - 1); (done == 0 || i < 0);i--)
 	{
		if (tmpstr[i] != ' ')
		{
			done = 1;
		} 
		else
		{
			tmpstr[i] = 0;
		}
 	}
 
 	Vendor.assign (tmpstr);
 }
 else
 {
	Vendor.assign ("");
 }

 // get product ID (bytes 16-31)

 cnt = 0;
 for (i = 16;i <= 31;i++)
 {
 	tmpstr[cnt] = (char)inqBuff[i];
	cnt++;
 }
 tmpstr[cnt] = 0;

 // remove trailing whitespace
 //
 // Note that we have to check the length of tmpstr first before we do anything
 // to make sure we don't get a seg fault

 if (strlen(tmpstr) > 0)
 {

 	done = 0;
 	for (i = (strlen(tmpstr) - 1); (done == 0 || i < 0);i--)
 	{
		if (tmpstr[i] != ' ')
		{
			done = 1;
		} 
		else
		{
			tmpstr[i] = 0;
		}
 	}
 
 	Device.assign (tmpstr);
 }
 else
 {
	Device.assign ("");
 }

 // get firmware version (bytes 32-35)

 cnt = 0;
 for (i = 32;i <= 35;i++)
 {
 	tmpstr[cnt] = (char)inqBuff[i];
	cnt++;
 }
 tmpstr[cnt] = 0;

 // remove trailing whitespace
 //
 // Note that we have to check the length of tmpstr first before we do anything
 // to make sure we don't get a seg fault

 if (strlen(tmpstr) > 0)
 {
 	done = 0;
 	for (i = (strlen(tmpstr) - 1); (done == 0 || i < 0);i--)
 	{
		if (tmpstr[i] != ' ')
		{
			done = 1;
		} 
		else
		{
			tmpstr[i] = 0;
		}
 	}
 
 	Firmware.assign (tmpstr);
 }
 else
 {
	Firmware.assign ("");
 }

 // If we are not an XP device, the WWN is returned in the page 0x0 data
 
 if (isXP(Vendor, Device) != SCINSUCCESS)
 {
 	for (i = 130; i <= 137; i++)
 	{
 		sprintf (tmpstr, "%02x", inqBuff[i]); // need to convert hex to string
		WWN.append (tmpstr);
 	}
 }

 return (SCINSUCCESS);
}


// Opens the device and performs a page 0x83 SCSI inquiry
// to get information from the device
// In: none
// Out: Sets the World Wide LUN ID based on data obtained from INQUIRY command

int ScsiInfo::DoInquiry83 ()
{
 // INQUIRY page 0x83 CDB; differs from page 0x0 in that the page code is 0x83 and
 // the EVPD (extended vital product data) bit is set

 unsigned char inqCmdBlk[INQ_CMD_LEN] = {INQ_CMD_CODE, 1, 0x83, 0,INQ_REPLY_LEN, 0};
 int sg_fd; // file descriptor for SCSI device
 int i, k, cnt, done;
 unsigned char inqBuff[INQ_REPLY_LEN]; // buffer for data returned from inquiry
 char tmpstr[255]; // used to get information from the inquiry buffer
 
 // open device for reading 
 
 if ((sg_fd = open(device_name, O_RDONLY)) < 0)
 {
        return (SCINDEVNOOPEN);
 }

 // Prepare INQUIRY command
 
 memset(&io_hdr, 0, sizeof(sg_io_hdr_t)); // set all elements of SG_IO structure to 0
 io_hdr.interface_id = 'S'; // S for SCSI?
 io_hdr.cmd_len = sizeof(inqCmdBlk); // size of CDB
 io_hdr.mx_sb_len = sizeof(sense_data_buf); // pointer to buffer for sense data
 io_hdr.dxfer_direction = SG_DXFER_FROM_DEV; // i/o mode; in this case from the device
 io_hdr.dxfer_len = INQ_REPLY_LEN; // size of SCSI command reply buffer
 io_hdr.dxferp = inqBuff; // pointer to SCSI command reply buffer
 io_hdr.cmdp = inqCmdBlk; // pointer to buffer with CDB in it
 io_hdr.sbp = sense_data_buf; // pointer to sense data buffer
 io_hdr.timeout = 20000; //20000 millisecs == 20 seconds

 // send command to device using IOCTL
 
 if (ioctl(sg_fd, SG_IO, &io_hdr) < 0) 
 {
	return (SCINIOCTLFAIL);
 }

 // close device

 close (sg_fd);

 // check status returned from command

 if ((io_hdr.info & SG_INFO_OK_MASK) != SG_INFO_OK)
 	return (SCINRETURNERROR);
 else if (io_hdr.masked_status)
	return (SCINRETURNERROR);
 else if (io_hdr.host_status)
	return (SCINRETURNERROR);
 else if (io_hdr.driver_status)
	return (SCINRETURNERROR);            

 // What we do with the page 0x83 INQUIRY data is different
 // if we are an XP as opposed to an EVA or MSA
 
 if (isXP(Vendor, Device) == SCINSUCCESS)
 {
 	//
	// XP
	//
	// For the XP, the WWN and UUID are both contained in the page 0x83 INQ data

	// First get the WWN
	
 	for (i = 58; i <= 65; i++)
 	{
        	sprintf (tmpstr, "%02x", inqBuff[i]); // need to convert hex to string
        	WWN.append (tmpstr);
 	}

	//
	// Now get the UUID
	//
	// get first 8 bytes of UUID

 	for (i = 38; i <= 45; i++)
 	{
 		sprintf (tmpstr, "%02x", inqBuff[i]); // need to convert hex to string
		WWLID.append (tmpstr);
 	}

 	WWLID.append ("-");

 	// get last 8 bytes of UUID

 	for (i = 46; i <= 53; i++)
 	{
 		sprintf (tmpstr, "%02x", inqBuff[i]); // need to convert hex to string
		WWLID.append (tmpstr);
 	}	
 }
 else
 {
 	//
	// EVA/MSA
	//
	
	// get first 8 bytes of UUID

 	for (i = 8; i <= 15; i++)
 	{
 		sprintf (tmpstr, "%02x", inqBuff[i]); // need to convert hex to string
		WWLID.append (tmpstr);
 	}

 	WWLID.append ("-");

 	// get last 8 bytes of UUID

 	for (i = 16; i <= 23; i++)
 	{
 		sprintf (tmpstr, "%02x", inqBuff[i]); // need to convert hex to string
		WWLID.append (tmpstr);
 	}
 }
	
 return (SCINSUCCESS);
}


// prints out information gathered from inquiry
//
// In: none
// Out: none

void ScsiInfo::PrintDeviceInfo ()
{
 cout << "SCSI_ID=\"" << Host << "," << Bus << "," << Target << "," << Lun << "\"";
 cout << ":VENDOR=\"" << Vendor << "\"";
 cout << ":MODEL=\"" << Device << "\"";
 cout << ":FW_REV=\"" << Firmware << "\"";
 cout << ":WWN=\"" << WWN << "\"";
 cout << ":LUN=\"" << WWLID << "\"";
 cout << "\n";
}

// prints out the error messages returns from a function
//
// In: error number, message to display before error
// Out: none

void ScsiInfo::PrintError (int error_num, char error_msg[])
{
 int k;

 cout << error_msg << ": ";

 switch (error_num) {
  case SCINSUCCESS:
	cout << "Success\n";
	break;  
  case SCINSTRTOLONG:
	cout << "device name exceeds maximum length\n";
	break;
  case SCINNOEXIST:
	cout << device_name << " does not exist\n";
	break;
  case SCINDEVNOOPEN:
	cout << "open failed\n";
	break;
  case SCINIOCTLFAIL:
	cout << "ioctl call failed\n";
	break;
  case SCINRETURNERROR:
	cout << "an error occurred during the command\n";

	if ((io_hdr.info & SG_INFO_OK_MASK) != SG_INFO_OK)
	{
		if (io_hdr.sb_len_wr > 0) 
		{
            		printf("INQUIRY sense data: ");
            
			for (k = 0; k < io_hdr.sb_len_wr; ++k)
			{
                		if ((k > 0) && (0 == (k % 10)))
                    			printf("\n  ");
                		printf("0x%02x ", sense_data_buf[k]);
            		}
            		printf("\n");
        	}
        
		if (io_hdr.masked_status)
            		printf("INQUIRY SCSI status=0x%x\n", io_hdr.status);
        	if (io_hdr.host_status)
            		printf("INQUIRY host_status=0x%x\n", io_hdr.host_status);
        	if (io_hdr.driver_status)
            		printf("INQUIRY driver_status=0x%x\n", io_hdr.driver_status);
	}
	break;
  default:
	cout << "unknown error\n";
 }
}

// returns whether the vendor, device combination is that of an XP array'
//
// In: Vendor string from INQUIRY, Model string from INQUIRY
// Out: TRUE if combination is an XP; FALSE if not

int ScsiInfo::isXP (string VendorString, string ModelString)
{
 if (VendorString == "HP" && ModelString.find(XP_MODEL_NAME) != string::npos)
 {
 	return SCINSUCCESS;
 }
 else
 {
 	return SCINGENERICERROR;
 }
}
