/*======================================================================

    Utility to look up information about SCSI devices

    Written by: David Hinds, dhinds@allegro.stanford.edu
                Andrew Kreitser, Andrew.Kreitser@compaq.com

    scsi_info.c 1.7 1997/12/18 18:24:03
    scsi_info.c 2.2 2000/12/18 - AMK Alpha, Instance support
    scsi_info.c 2.4 2001/07/25 - AMK Linux Kernel 2.4.x support 
    scsi_info.c 2.5 2002/01/12 - AMK Serial Number & WWN support
    scsi_info.c 2.6 2002/04/04 - AMK hex - dec /proc LUN presentation 
    scsi_info.c 2.7 2002/04/44 - AMK hex - RAID LUN id presentation
     
======================================================================*/

#include <sys/types.h>
#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <dirent.h>

#include <linux/version.h>
#include <linux/config.h>

#include <linux/major.h>
#define VERSION(v,p,s) (((v)<<16)+(p<<8)+s)
#define CODEVER 2.7

#if (LINUX_VERSION_CODE > VERSION(1,3,97))
#include <scsi/scsi.h>
#include <scsi/sg.h>
#else
#if (LINUX_VERSION_CODE > VERSION(1,3,0))
#include <linux/scsi.h>
#include <linux/sg.h>
#else
#include <linux/fs.h>
#endif
#endif
/*====================================================================*/
static float kernel_version()
{
  FILE *f;
  float v=0;
  f=popen("uname -r|awk -F. '{printf(\"%s.%s\\n\", $1,$2);}'", "r");
  if(f) fscanf(f,"%f",&v);
  pclose(f);

  return v;
}
/*====================================================================*/
static char * get_hba(int host)
{
    DIR *dir;
    struct dirent *ent;
    static char s[128];
    strcpy( s, "Unknown" );

    dir = opendir("/proc/scsi");
    if (dir == NULL)
        return s;
    while ((ent = readdir(dir)) != NULL)
    {
//        printf("%2d [%s] %d\n", ent->d_ino, ent->d_name, (ent->d_ino & 0xff));
        if ((ent->d_ino & 0xff) == host)
            break;
    }
    closedir(dir);
    if (!ent) {
        perror("could not find SCSI host in /proc/scsi");
        return s;
    }
    else
    { 
     sprintf( s, "/proc/scsi/%s", ent->d_name);
     return s;
    }
}
/*====================================================================*/
static int check_inst( char *fn, int inst )
{
    int n;
    DIR *dir;
    struct dirent *ent;

    dir = opendir(fn);
    if (dir == NULL) {
        return -1;
    }
    
    while(1) {
        ent = readdir(dir);
	if(!ent) break;
//	printf("D [%s]\n", ent->d_name);
	if(ent->d_name[0]=='.') continue;
        if(atoi(ent->d_name) == inst)
	{
	 closedir(dir);
	 return inst;
	} 
    } 

    closedir(dir);
    dir = opendir(fn);

   
    while(1) {
        ent = readdir(dir);
        if(!ent) break;
       // printf("D [%s]\n", ent->d_name);
        if(ent->d_name[0]!='.')
	{
         n=atoi(ent->d_name);
	 closedir(dir);
	 return n;
        }
    }

    return -1;
}
/*====================================================================*/
unsigned char buffer[1024];
/*====================================================================*/
char* getstr (int shift, int start, int stop)
{
   int ln = stop-start+1; 
   int i;
   unsigned char* page= buffer+shift;
   char *str = (char*) malloc (ln+1);
   memcpy (str, page+start, ln);
   str[ln] = '\0';
   for (i = ln-1; i >= 0 && str[i] == ' '; i--) str[i] = '\0';
   return str;
}
/*====================================================================*/
char* gethex (int shift, int start, int stop)
{
   int ln = stop-start; 
   int j=0;
   unsigned char* page= buffer+shift;
   char *str = (char*) malloc (2*ln+1);

   for(j=0;j<ln;j++)
   { //printf( "%02x", (unsigned char) page[start+j]); 
    sprintf( &str[2*j], "%02x", (unsigned char) page[start+j]); }

   return str;
}
/*====================================================================*/
void dumppage (int start, int ln)
{
   int i;
   unsigned char* page= buffer+start;
 printf("=========\n");
   for (i = 0; i <= ln; i++)
     {
	printf ("%02x ", page[i]);
	if (!((i+1)%16)) printf ("  -%d\n",i+1);
     }
   if (i%16) printf ("  -%d\n",i);
 printf("=========\n");
}
/*====================================================================*/
int inquiry2buffer (char *path, int mode, int lun_mask, int page)
{
  int fd;
  int status;
  unsigned char *cmd;
  unsigned char * pagestart;

  memset (buffer, 0, sizeof(buffer));
  
  *( (int *)  buffer )		= 0;	                /* length of input data */
  *( ((int *) buffer) + 1 )	= sizeof(buffer)-16;	/* length of output buffer */

  cmd = (char *) ( ((int *) buffer) + 2 );
  
  cmd[0] = 0x12;			/* INQUIRY */
  cmd[1] = lun_mask;                    /* lun mask */
  cmd[2] = page;			/* page */
  cmd[3] = 0x00;			/* (reserved) */
  cmd[4] = 0xff;			/* allocation length */
  cmd[5] = 0x00;			/* control */

  fd = open(path, mode);
  if( fd < 0 ) return 1;

  status = ioctl( fd, 1, buffer );
  close(fd);

  return 0;
}
/*====================================================================*/
int main(int argc, char *argv[])
{
    int i, fd, host, inst, channel, id, lun;
    u_long arg[2], arx[2];
    char *p, *p1, *p2, match[128], s[128], vendor[9], model[17], rev[5], *wwn_ptr;
    FILE *f;

//printf(" LK Version: %f\n", kernel_version());

    if (argc != 2) {
        fprintf(stderr, "usage(CODEVER=%3.1f): %s [device]\n", CODEVER, argv[0]);
        exit(EXIT_FAILURE);
    }
    
    fd = open(argv[1], O_RDONLY);
    if (fd < 0) {
        perror("open() failed");
        exit(1);
    }
    if (ioctl(fd, SCSI_IOCTL_GET_BUS_NUMBER, arx) != 0) {
        perror("could not get SCSI device info");
        close(fd);
        exit(1);
    }
//printf("%d X %8.8x %8.8x\n", sizeof(u_long), arx[1], arx[0]);

    
    if (ioctl(fd, SCSI_IOCTL_GET_IDLUN, arg) != 0) {
        perror("could not get SCSI device info");
        close(fd);
        exit(1);
    }
    close(fd);
//printf("%d X %8.8x %8.8x\n", sizeof(u_long), arg[1], arg[0]);
    id = arg[0] & 0xff;
    lun = (arg[0] >> 8) & 0xff;
    channel = (arg[0] >> 16) & 0xff;
    host = ((arg[0] >> 24) & 0xff) ;
    if( kernel_version() >= 2.4 )
      inst=host;
    else
    {
      p = get_hba(host);
      inst = (arx[0] % 256);
      inst = check_inst( p, inst );
    }

    sprintf(match, "Host: scsi%d Channel: %02d Id: %02d Lun: %02d\n",
            inst, channel, id, lun);

   // printf("HBA: %s Instance: %d %s", p, inst, match);

    printf("SCSI_ID=\"%d,%d,%d,%d\":", inst, channel, id, lun);
    
    f = fopen("/proc/scsi/scsi", "r");
    if (f == NULL) {
        printf("VENDOR=\"\":MODEL=\"Unknown\":FW_REV=\"Unknown\":\n");
        exit(0);
    }

    while (fgets(s, 128, f) != NULL)
        if (strcmp(s, match) == 0) break;
    fgets(s, 128, f);
    strncpy(vendor, s+10, 8); vendor[8] = '\0';
    for (i = 7; (i >= 0) && (vendor[i] == ' '); i--)
        vendor[i] = '\0';
    strncpy(model, s+26, 16); model[16] = '\0';
    for (i = 15; (i >= 0) && (model[i] == ' '); i--)
        model[i] = '\0';
    strncpy(rev, s+48, 4); rev[4] = '\0';
    for (i = 3; (i >= 0) && (rev[i] == ' '); i--)
        rev[i] = '\0';
    if( inquiry2buffer( argv[1], O_RDWR, lun << 5 , 0x00) ) // page 0x00
    {
      // dumppage ( 8, 0xff );
      wwn_ptr=NULL;
    }
    else
      wwn_ptr= gethex(8, 130, 138);
    if( inquiry2buffer( argv[1], O_RDWR, (lun << 5) | 0x01, 0x83 ) ) // page 0x83
    {
      // dumppage ( 8, 0xff );
      p1=NULL;
      p2=NULL;
    }
    else
    {
      p1=gethex(8, 8, 16);
      p2=gethex(8, 16, 24);     
    }
 
    printf("VENDOR=\"%s\":MODEL=\"%s\":FW_REV=\"%s\":WWN=\"%s\":LUN=\"%s-%s\":\n", vendor, model, rev,
    wwn_ptr ? wwn_ptr : "", p1 ? p1 : "", p2 ? p2 : "");
    fflush(stdout);
    free(wwn_ptr);
    free(p1);
    free(p2);
    
    exit(0);
    return 0;
}

/*
  inquiry2buffer( argv[1], O_RDWR, (lun << 5) | 0x01, 0x80 );
  dumppage (8, 0xff); 

  printf("AMK SN %s %d\n", getstr (8, 4, 3+buffer[8+3]), buffer[8+3] );

  inquiry2buffer( argv[1], O_RDWR, lun << 5 , 0x00);
  st.manufacturer = getstr (8, 8, 15); 
  st.model = getstr (8, 16, 31);
  st.rev = getstr (8, 32, 35);
  printf("AMK WWN %s\n", gethex( 8, 130,138));

  inquiry2buffer( argv[1], O_RDWR | O_NDELAY, (lun << 5) | 0x01, 0x83 );
  dumppage (8, 0xff);   
  printf("AMK WWN %s\n", gethex( 8, 8, 16));
*/
