/*
 * Copyright 2000 by Compaq Computer Corporation
 *
 * UCD SNMP agent extension for Compaq Management Agents (cma)
 *
 */ 

#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <signal.h>
#include <dirent.h>
#include <unistd.h>
#include <setjmp.h>
#include <dlfcn.h>
#include <pthread.h>
#include <sys/socket.h>
#include <netdb.h>
#include <netinet/in.h>

#include <config.h>
#include "mibincl.h"
#include "util_funcs.h"
#include "snmp_logging.h"

#include "cmaX.h"

static SubAgentType *FirstSubAgent;
static int ListenDescriptor;
static struct sockaddr_in ListenSockAddr;
static RequestType GlobalRequest;
static jmp_buf JmpBuf;

static char cmaXversion[] = "Compaq Management Agents eXtension (cmaX) Version  4.90 11/22/2000";
static oid CpqMib[MAX_OID] = { 1,3,6,1,4,1,232 };
static int ColdStartInProgress=1;
static pid_t snmpd_pid;
static char TimeOut=FALSE;

extern int errno;

static int GapManager(int OpCode, oid *FromOid, size_t FromOidLength, oid *ToOid, size_t ToOidLength)
{
  int i;
  static int  Inited=0;
  static time_t TimeStamp;
  static oid  StartOid[MAX_OID], EndOid[MAX_OID];
  static size_t      StartOidLength, EndOidLength;

  switch (OpCode) {
    case 1:  // Set StartOid and EndOid
      TimeStamp = time(NULL);
      StartOidLength = FromOidLength;
      for (i=0; i<StartOidLength; i++) StartOid[i]=FromOid[i];
      if (ToOid) {
        EndOidLength = ToOidLength;
        for (i=0; i<EndOidLength; i++) EndOid[i]=ToOid[i];
      }
      else {
        EndOidLength = -1; // EndOid is the End of the MIB branch
      }
      Inited=1;
/********
   {
    int i;
    fprintf(stderr,"GapManager: TimeStamp=%lu\n", TimeStamp);
    fprintf(stderr,"GapManager: StartOid = ");
    for (i=0; i<StartOidLength; i++) fprintf(stderr,"%d ", StartOid[i]);
    fprintf(stderr, "EndOid = ");
    if (EndOidLength == -1)
      fprintf(stderr, "NULL ");
    else {
    for (i=0; i<EndOidLength; i++) fprintf(stderr,"%d ", EndOid[i]);
    }
    fprintf(stderr, "\n");
   }
 ********/
      return 1;  // OK
      break;
    case 2:  // Check if FromOid is in the Gap
      {
        int compare=0; // -1: less than 0: equal, 1: greater than
        if (!Inited) return 0;
        if ((time(NULL) - TimeStamp) > 2) return 0; // Cached around 2 seconds
        for (i=0; i<StartOidLength && i<FromOidLength; i++) {
          if (FromOid[i] < StartOid[i]) return 0;
          if (FromOid[i] > StartOid[i]) {
            compare=1;
            break;
          }
        }
        if  (compare == 0 && FromOidLength < StartOidLength) return 0;

        if (EndOidLength == -1) {
          if (StartOidLength < 8) return 0; // "1.3.6.1.4.1.232" ==> "enterprises.232"
          if (FromOid[7] == StartOid[7]) return 1; // Yes, FromOid is IN the gap
          return 0;
        }
        // EndOid exists
        for (i=0; i<EndOidLength && i<FromOidLength; i++) {
          if (FromOid[i] > EndOid[i]) return 0;
          if (FromOid[i] < EndOid[i]) return 1;
        }
        return 0; // Not in Gap.
      }
      break;
    default:
      return 0;
      break;
  }
}

static void SendCommand(RequestType *Request)
{
	struct sockaddr_in sa;
	unsigned char LocalHost[4] = { 127,0,0,1 };
        int ret, retry=10;

	sa.sin_family = PF_INET;
	sa.sin_port = htons(Request->Port);
	memcpy(&sa.sin_addr.s_addr,LocalHost,4);
//fprintf(stderr, "SendCommand: command=%d, sin_port=%d\n", Request->Command, sa.sin_port);

        while (retry--) {
	  ret=sendto(ListenDescriptor,Request,sizeof(RequestType),0,&sa,sizeof(struct sockaddr_in));
          if (ret != -1) break;
          //fprintf(stderr, "SendCommand failed: command=%d, sin_port=%d\n", Request->Command, sa.sin_port);
        }
}

static int ConvertObjectType(int CpqType)
{
	switch(CpqType)
	{
		case	CPQ_ASN_INTEGER		:
										return(ASN_INTEGER);
		case	CPQ_ASN_STRING		:
										return(ASN_OCTET_STR);
		case	CPQ_ASN_OBJECT_ID	:
										return(ASN_OBJECT_ID);
		case	CPQ_ASN_OCTET		:
										return(ASN_OCTET_STR);
		case	CPQ_ASN_COUNTER		:
										return(ASN_COUNTER);
		case	CPQ_ASN_IPADDRESS	:
										return(ASN_IPADDRESS);
		case	CPQ_ASN_GAUGE		:
										return(ASN_GAUGE);
		default						:
										return(ASN_INTEGER);
	}
}

static void TimesUp(int signal_received)
{
	TimeOut = FALSE;
	longjmp(JmpBuf,TRUE);
}

/* This is called when a SET is received */
static int WriteAction(	int Action,
					u_char *var_val,
					u_char var_val_type,
					size_t var_val_len,
					u_char *statP,
					oid *name,
					size_t name_len)
{
	SubAgentType *SubAgent;
	RequestType Request;
	int x;
	int RetError;

	SubAgent = FirstSubAgent;
	while(SubAgent != NULL)
	{
		if(SubAgent->MibTreeNumber == (int)name[7])
			break;
		SubAgent = SubAgent->Next;
	}
	if(SubAgent == NULL)
		return(SNMP_ERR_NOSUCHNAME);

	Request.Command = COMMAND_REQUEST_OBJECT;
	Request.Port = SubAgent->PortNumber;

	switch(Action)
	{
		case	RESERVE1:	
			Request.Data.AgentRequest.Request = SNMP_RESERVE1;
			break;
		case	RESERVE2:	
			Request.Data.AgentRequest.Request = SNMP_RESERVE2;
			break;
		case	ACTION:	
			Request.Data.AgentRequest.Request = SNMP_ACTION;
			break;
		case	COMMIT:	
			Request.Data.AgentRequest.Request = SNMP_COMMIT;
			break;
		case	UNDO:	
			Request.Data.AgentRequest.Request = SNMP_UNDO;
			break;
		case	FREE:	
			Request.Data.AgentRequest.Request = SNMP_FREEUP;
			break;
		default:
			Request.Data.AgentRequest.Request = SNMP_SET;
			break;
	}

	for(x=0;x<name_len;x++)
		Request.Data.AgentRequest.Oid[x] = (int)name[x];

	Request.Data.AgentRequest.OidLength = name_len;
	/* Fill in value fields for SET */
	if(var_val_type == ASN_INTEGER)
	{
		Request.Data.AgentRequest.IntRet = (unsigned long)*var_val;
		Request.Data.AgentRequest.DataLength = sizeof(unsigned long);
	}
	else
	{
		memcpy(Request.Data.AgentRequest.DataRet,var_val,var_val_len);
		Request.Data.AgentRequest.DataLength = var_val_len;
	}

	TimeOut = TRUE;
	signal(SIGALRM,TimesUp);
	if(setjmp(JmpBuf)==0)
	{
		alarm(SUBAGENT_TIMEOUT_PERIOD);
		SendCommand(&Request);
		sigpause(SIGALRM);
	}
	alarm(0);
 
 	if(TimeOut)
		return(SNMP_ERR_NOSUCHNAME);

	if(GlobalRequest.Data.AgentRequest.Oid[0] == 0x00)
	{
		switch(Action)
		{
			case	RESERVE1:	
				RetError = SNMP_ERR_RESOURCEUNAVAILABLE;
				break;
			case	RESERVE2:	
				RetError = SNMP_ERR_RESOURCEUNAVAILABLE;
				break;
			case	ACTION:	
				RetError = SNMP_ERR_RESOURCEUNAVAILABLE;
				break;
			case	COMMIT:	
				RetError = SNMP_ERR_COMMITFAILED;
				break;
			case	UNDO:	
				RetError = SNMP_ERR_UNDOFAILED;
				break;
			case	FREE:	
				RetError = SNMP_ERR_NOERROR;
				break;
			default:
				RetError = SNMP_ERR_NOSUCHNAME;
				break;
		}
		return(RetError);
	}
	else
		return(SNMP_ERR_NOERROR);
}

static void DeleteSubAgentEntryByMibTreeNumber(int MibTreeNumber)
{
	SubAgentType *SubAgent,*LastSubAgent=NULL;

	SubAgent = FirstSubAgent;

	while(SubAgent != NULL)
	{
		if(SubAgent->MibTreeNumber == MibTreeNumber)
			break;

		LastSubAgent = SubAgent;
		SubAgent = SubAgent->Next;
	}

	if(SubAgent != NULL)
	{
		if(LastSubAgent == NULL)
			FirstSubAgent = FirstSubAgent->Next;
		else
			LastSubAgent->Next = SubAgent->Next;

		free(SubAgent);
	}
}

/* This routine is called when the UCD agent receives a request from a table */
static unsigned char *MasterAgentRequestReceived( struct variable *vp,
					oid *name,
					size_t *length,
					int exact,
					size_t *var_len,
					WriteMethod **write_method)
{
	RequestType Request;
	SubAgentType *SubAgent;
	static unsigned long long_ret;
	static unsigned char string[SPRINT_MAX_LEN];
	int x;
	void (*OldHandler)(int);
	pid_t sid;

        // Send Cold Start packets
        {
          static int inited=1;
          if (!inited) {
            int mib_branch = 1;
  	    RequestType CSRequest;

            inited = 1;
            for (; mib_branch <= MAX_MIB_BRANCH; mib_branch++) {
              CSRequest.Command = COMMAND_COLD_START;
              CSRequest.Port = SOCKET_NUMBER + mib_branch;
              SendCommand(&CSRequest);
            }
          }
        }

	if(vp->acl == RWRITE)
		*write_method = WriteAction;
/************ 
  {
    int i;
    fprintf(stderr,"MasterAgentRequestReceived: receive vp->name= ");
    for (i=0; i< vp->namelen; i++) fprintf(stderr,"%d ", vp->name[i]);
    fprintf(stderr,"\n");
    fprintf(stderr,"                                receive name= ");
    for (i=0; i< *length; i++) fprintf(stderr,"%d ", name[i]);
    fprintf(stderr,"\n");
    fprintf(stderr,"                                       exact= %d\n",exact);
  }
 ************/

	SubAgent = FirstSubAgent;
	while(SubAgent != NULL)
	{
		/* if(SubAgent->MibTreeNumber == (int)name[7]) */
		if(SubAgent->MibTreeNumber == (int)vp->name[7])
			break;
		SubAgent = SubAgent->Next;
	}
	if(SubAgent == NULL) {
		return(NULL);
        }

	/* Check to see if the peer process is still running */
	sid = getsid(SubAgent->PID);
	if(sid == -1)
	{
		/* Doh! - the peer must have died, purge it from the agent list */
		DeleteSubAgentEntryByMibTreeNumber(SubAgent->MibTreeNumber);
		return(NULL);
	}

	Request.Command = COMMAND_REQUEST_OBJECT;
	Request.Port = SubAgent->PortNumber;

	Request.Data.AgentRequest.Request = exact?SNMP_GET:SNMP_GETNEXT;

	for(x=0;x<vp->namelen;x++)
		if(name[x] != vp->name[x])
			break;

	if(x == vp->namelen)
	{
		for(x=0;x<*length;x++)
			Request.Data.AgentRequest.Oid[x] = (int)name[x];
		Request.Data.AgentRequest.OidLength = *length;
	}
	else
	{
		for(x=0;x<vp->namelen;x++)
			Request.Data.AgentRequest.Oid[x] = (int)vp->name[x];
		Request.Data.AgentRequest.OidLength = vp->namelen; 
	}

/******** 
  {
    int i;
    fprintf(stderr,"MasterAgentRequestReceived: Request.Data.AgentRequest.Oid: ");
    for (i=0; i< Request.Data.AgentRequest.OidLength; i++) fprintf(stderr,"%d ",
         Request.Data.AgentRequest.Oid[i]);
  }
 ********/

        if (GapManager(2, (oid*)Request.Data.AgentRequest.Oid, Request.Data.AgentRequest.OidLength, NULL, -1)) {
		return(NULL);
        }

	TimeOut = TRUE;
	signal(SIGALRM,TimesUp);
	if(setjmp(JmpBuf)==0)
	{
		alarm(SUBAGENT_TIMEOUT_PERIOD);
		SendCommand(&Request);
		sigpause(SIGALRM);
	}
	alarm(0);

	if(TimeOut) {
		return(NULL);
        }

	/* Pick back up here when request has been received */
	/* Receive request will be in GlobalRequest */

	for(x=0;x<GlobalRequest.Data.AgentRequest.OidLength && x < MAX_OID;x++)
		name[x] = (oid)GlobalRequest.Data.AgentRequest.Oid[x];
	*length = GlobalRequest.Data.AgentRequest.OidLength;

	/* First check returned OID against original registered OID to make sure */
	/* that they somewhat match.  This is for the peer agent, which is */
	/* not quite aware of its own scope.  If they don't match, this means */
	/* the peer returned something out of its scope */

	if(GlobalRequest.Data.AgentRequest.Oid[0] == 0) {
                if (!exact) GapManager(1, (oid*)Request.Data.AgentRequest.Oid, Request.Data.AgentRequest.OidLength, NULL, -1);
		return(NULL);
        }

       	if (!exact) GapManager(1, (oid*)Request.Data.AgentRequest.Oid, Request.Data.AgentRequest.OidLength, name, *length);

	for(x=0;x<vp->namelen;x++)
	{
		if(vp->name[x] != (oid)GlobalRequest.Data.AgentRequest.Oid[x]) {
			return(NULL);
		}
	}


/******** 
  {
    int i;
    fprintf(stderr,"MasterAgentRequestReceived: GlobalRequest.Data.AgentRequest.Oid: ");
    for (i=0; i< GlobalRequest.Data.AgentRequest.OidLength; i++) fprintf(stderr,"%d ",
         GlobalRequest.Data.AgentRequest.Oid[i]);
    fprintf(stderr,"\n");
  }
 ********/

	vp->type = ConvertObjectType(GlobalRequest.Data.AgentRequest.VarClass);
	/* if(vp->type == ASN_INTEGER) */
	if((GlobalRequest.Data.AgentRequest.VarClass == CPQ_ASN_INTEGER)||
		(GlobalRequest.Data.AgentRequest.VarClass == CPQ_ASN_COUNTER)||
		(GlobalRequest.Data.AgentRequest.VarClass == CPQ_ASN_GAUGE))
	{
		*var_len = sizeof(unsigned long);
		long_ret = GlobalRequest.Data.AgentRequest.IntRet;
		return((unsigned char *)&long_ret);
	}
	else
	{
		*var_len = GlobalRequest.Data.AgentRequest.DataLength <= SPRINT_MAX_LEN?
                           GlobalRequest.Data.AgentRequest.DataLength: SPRINT_MAX_LEN;
		memcpy(string,GlobalRequest.Data.AgentRequest.DataRet,
                       GlobalRequest.Data.AgentRequest.DataLength <= SPRINT_MAX_LEN?
                           GlobalRequest.Data.AgentRequest.DataLength: SPRINT_MAX_LEN);
		return(string);
	}
}

static int MibCmp(const void *One, const void *Two)
{
	struct variable7 *v1,*v2;
	int x;

	v1 = (struct variable7 *)One;
	v2 = (struct variable7 *)Two;

	for(x=0;x<7;x++)
	{
		if(v1->name[x]!=v2->name[x])
			return(v1->name[x]<v2->name[x]?-1:1);
	}
	return(0);
}

/* Go through all the SNMP variables and register them with the UCD stack now */
static int RegisterMibsWithUCDStack()
{
	struct variable7 var[CPQ_MAX_MIB_ITEMS];
	FILE *MibFile;
	FILE *MasterMibFile;
	char Buf[MAX_STRING];
	char Filename[MAX_STRING];
	int x,MibItemCnt=0;
	char *Ptr,*Oid,*VarType,*Access;
	int iOid[MAX_OID],OidLength;

	MasterMibFile = fopen(MASTER_CONF_FILE,"r");
	if(MasterMibFile==NULL)
		return(FALSE);

	fgets(Filename,MAX_STRING,MasterMibFile);
	if(Filename[0] != 0x00)
		Filename[strlen(Filename)-1] = 0x00;

	while(!feof(MasterMibFile))
	{
		MibFile = fopen(Filename,"r");
		if(MibFile != NULL)
		{
			fgets(Buf,MAX_STRING,MibFile);
			if(Buf[0] != 0x00)
				Buf[strlen(Buf)-1] = 0x00;
	
			while(!feof(MibFile))
			{
				Oid = strtok(Buf," ");
				if(Oid)
				{
					VarType = strtok(NULL," ");
					if(VarType)
					{
						Access = strtok(NULL," ");
						if(Access)
						{
							memset(&iOid[0],0,MAX_OID*sizeof(int));
							OidLength = 0;
							Ptr = strtok(Oid,".");
							while(Ptr!=NULL)
							{
								iOid[OidLength++] = atoi(Ptr);
								Ptr = strtok(NULL,".");
							}
							if(strcmp(VarType,"INTEGER")==0)
								var[MibItemCnt].type = ASN_INTEGER;
							else
							if(strcmp(VarType,"STRING")==0)
								var[MibItemCnt].type = ASN_OCTET_STR;
							else
							if(strcmp(VarType,"OCTET")==0)
								var[MibItemCnt].type = ASN_OCTET_STR;
							else
							if(strcmp(VarType,"BOOLEAN")==0)
								var[MibItemCnt].type = ASN_BOOLEAN;
							else
							if(strcmp(VarType,"OID")==0)
								var[MibItemCnt].type = ASN_OBJECT_ID;
							else
							if(strcmp(VarType,"IPADDRESS")==0)
								var[MibItemCnt].type = ASN_IPADDRESS;
							else
							if(strcmp(VarType,"COUNTER")==0)
								var[MibItemCnt].type = ASN_COUNTER;
							else
							if(strcmp(VarType,"GAUGE")==0)
								var[MibItemCnt].type = ASN_GAUGE;
		
							if(strcmp(Access,"READWRITE")==0)
								var[MibItemCnt].acl = RWRITE;
							else
								var[MibItemCnt].acl = RONLY;
						}
					}
		
					var[MibItemCnt].magic = MibItemCnt;
	
					var[MibItemCnt].findVar = MasterAgentRequestReceived;
		
					for(x=0;x<7;x++)
						var[MibItemCnt].name[x] = iOid[x];
		
					var[MibItemCnt].namelen = OidLength <= 7? OidLength: 7;
			
					MibItemCnt++;
				}
				fgets(Buf,MAX_STRING,MibFile);
				if(Buf[0] != 0x00)
					Buf[strlen(Buf)-1] = 0x00;
			}
			fclose(MibFile);
		}
		fgets(Filename,MAX_STRING,MasterMibFile);
		if(Filename[0] != 0x00)
			Filename[strlen(Filename)-1] = 0x00;
	}

	fclose(MasterMibFile);

	qsort(var,MibItemCnt,sizeof(struct variable7),MibCmp);

//   {
//    int i, j;
//    fprintf(stderr,"RegisterMibsWithUCDStack: MIB item count = %d\n", MibItemCnt);
//    for (i=0; i<MibItemCnt; i++) {
//      for (j=0; j<var[i].namelen; j++) {
//        if (j==0)
//          fprintf(stderr,"RegisterMibsWithUCDStack: MIB item %d, OID = %d", i, var[i].name[j]);
//        else
//          fprintf(stderr,".%d", var[i].name[j]);
//      }
//      fprintf(stderr,"\n");
//    }
//   }
	register_mib("CompaqInsight",(struct variable *)var,(int)sizeof(struct variable7),MibItemCnt,CpqMib,7);

	return(TRUE);
}

static void DeleteSubAgentEntry(RequestType *Request)
{
	DeleteSubAgentEntryByMibTreeNumber(Request->Data.DisConnect.MibTreeNumber);
}

static void CreateSubAgentEntry(RequestType *Request)
{
	SubAgentType *SubAgent;

	if(FirstSubAgent == NULL)
		FirstSubAgent = SubAgent = (SubAgentType *)malloc(sizeof(SubAgentType));
	else
	{
		SubAgent = FirstSubAgent;
		while(SubAgent->Next != NULL)
			SubAgent = SubAgent->Next;

		SubAgent->Next = (SubAgentType *)malloc(sizeof(SubAgentType));
		SubAgent = SubAgent->Next;
	}
	SubAgent->Next = NULL;

	SubAgent->PortNumber = Request->Data.Connect.PortNumber;
	SubAgent->MibTreeNumber = Request->Data.Connect.MibTreeNumber;
	SubAgent->PID = Request->Data.Connect.PID;
    // fprintf(stderr,"CreateSubAgentEntry: Request->Data.Connect.Description=%s\n",
    //                 Request->Data.Connect.Description);
	strcpy(SubAgent->Description,Request->Data.Connect.Description);
}

static void ServiceResponse(RequestType *Request)
{
	memcpy(&GlobalRequest,Request,sizeof(RequestType));
	kill(snmpd_pid,SIGALRM);
}

static void *ReceiveUDPRequests(void *Param)
{
	RequestType Request;
	struct sockaddr_in sa;
	int BytesReceived;
	char Packet[PACKET_SIZE];
	char Up = TRUE;
	int Size;

	while(Up)
	{
		Size = sizeof(struct sockaddr_in);
		memset(&sa,0,Size);
		BytesReceived = recvfrom(ListenDescriptor,Packet,PACKET_SIZE,0,&sa,&Size);
		if(BytesReceived != -1)
			memcpy(&Request,Packet,sizeof(RequestType));
		else {
                        if (!ColdStartInProgress)
        		   snmp_log(LOG_NOTICE, "\ncmaX: recvfrom() failed.  Error = %d.  Peer may have been terminated.\n",errno);
                        continue;
                }

		switch(Request.Command)
		{
			case	COMMAND_CONNECT:
				CreateSubAgentEntry(&Request);
				break;
			case	COMMAND_DISCONNECT:
				DeleteSubAgentEntry(&Request);
				break;
			case	COMMAND_REQUEST_RESPONSE:
				ServiceResponse(&Request);
				break;
			case	COMMAND_SEND_TRAP:
				/* SendTrap(&Request); */
				break;
		}
	}
	return(NULL);
}

static void SetUpListenSocket()
{
	pthread_t Thread;
	unsigned char LocalHost[4] = { 127,0,0,1 };

	ListenDescriptor = socket(PF_INET,SOCK_DGRAM,0);

	ListenSockAddr.sin_family = PF_INET;
	ListenSockAddr.sin_port = htons(SOCKET_NUMBER);
	memcpy(&ListenSockAddr.sin_addr.s_addr,LocalHost,4);

	if(bind(ListenDescriptor,&ListenSockAddr,sizeof(struct sockaddr_in))==-1)
	{
        	snmp_log(LOG_NOTICE, "\ncmaX: %s, %s (%d): bind() failed!\n",
                         __FILE__,"SetUpListenSocket",__LINE__);
		return;
	}

	pthread_create(&Thread,NULL,ReceiveUDPRequests,(void *)NULL);
        // Send Cold Start packets
        {
          static int inited=0;
          if (!inited) {
            int mib_branch = 1;
  	    RequestType CSRequest;

            inited = 1;
            for (; mib_branch <= MAX_MIB_BRANCH; mib_branch++) {
              CSRequest.Command = COMMAND_COLD_START;
              CSRequest.Port = SOCKET_NUMBER + mib_branch;
              SendCommand(&CSRequest);
            }
          }
        }
        sleep(1);
        ColdStartInProgress=0;
}

void init_cmaX(void)
{
        // snmp_log(LOG_NOTICE, "cmaX: Entering init_cmaX %d\n", 88);
        snmpd_pid = getpid();
	if(RegisterMibsWithUCDStack())
		SetUpListenSocket();
}
