/*
    Re3002 Bootloader - A Bootloader for the M3002 RTC replacement 'Re3002'
    Copyright (C) 2013  Oliver Tscherwitschke <oliver@tscherwitschke.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/


.include <m88PAdef.inc>

.EQU BL_VERSION     = 0x0100
.EQU BL_SIGNATURE   = 0x6C62

.EQU F_CPU          = 20000000

.EQU BOOT_RST_ADDR  = FOURTHBOOTSTART
.EQU FW_START_ADDR  = 0x0

.EQU USE_DEBUG_PINS = 0

.EQU DEBUG_OUTPUT   = 1

.EQU INPORT         = PINC          ; input port for address/data bus (bits 3..0)
.EQU OUTPORT        = PORTD         ; output port for data (bits 7..4)

.MACRO BUSY_PIN_H
    sbi PORTB, 2                    ; set /BUSY pin to high level (inacative)
.ENDMACRO

.MACRO BUSY_PIN_L
    cbi PORTB, 2                    ; set /BUSY pin to low level (acative)
.ENDMACRO


.MACRO BUSY
    ldi temp, 0xF0
    out OUTPORT, temp
    BUSY_PIN_L
.ENDMACRO

.MACRO NOT_BUSY
    ldi temp, 0x00
    out OUTPORT, temp
    BUSY_PIN_H
.ENDMACRO



.EQU T1_OCR_1 = 19530       ; 3 x this value and...
.EQU T1_OCR_2 = 19531       ; 1 x this value results in exacly 1 Hz in average at 20 MHz.


.IF USE_DEBUG_PINS
    .MACRO TESTPIN_H   
        sbi PORTB, 5
    .ENDMACRO

    .MACRO TESTPIN_L   
        cbi PORTB, 5
    .ENDMACRO
.ELSE
    .MACRO TESTPIN_H   
    .ENDMACRO

    .MACRO TESTPIN_L   
    .ENDMACRO
.ENDIF



; M3002 Control reg bit definitions:
.EQU M3002_RUN  = 0     ; RUN bit in control reg of M3002
.EQU M3002_TEST = 7     ; TEST bit in control reg of M3002

.EQU M3002_CONTROL_ADDR = 15


.EQU SIGRD = 5

.MACRO RJMPNE
    breq _b1
    rjmp @0
_b1:
.ENDMACRO

.MACRO RJMPEQ
    brne _b2
    rjmp @0
_b2:
.ENDMACRO



; ---------------------------------------------------
; Command definitions

.EQU CMD_READ_SIG       = 0x81
.EQU CMD_SET_ADDR       = 0x82
.EQU CMD_WRITE_WORD     = 0x83
.EQU CMD_READ_WORD      = 0x84
.EQU CMD_PAGE_ERASE     = 0x85
.EQU CMD_PAGE_WRITE     = 0x86
.EQU CMD_WRITE_EE_BYTE  = 0x87
.EQU CMD_READ_EE_BYTE   = 0x88

.EQU CMD_START_FW       = 0x99


; ---------------------------------------------------
; Register definitions
.DEF zero_reg   = R0            // the code relies on this register being always zero

.DEF temp       = R18
.DEF temp2      = R19


; Pointer registers
; .DEF XL   = R26
; .DEF XH   = R27

; .DEF ZL   = R30
; .DEF ZH   = R31



; -----------------------------------------------------------------------
; These regs are exclusively reserved for the ISRs, don't use elsewhere!
.DEF bus_data   = R14
.DEF sreg_buf   = R15
.DEF itemp      = R16
.DEF bus_state  = R17

; .DEF YL   = R28
; .DEF YH   = R29
; -----------------------------------------------------------------------



;==========================================================
.DSEG

; ---------------------------------------------------------
bl_data:
bl_sig_l:               .BYTE 1
bl_sig_h:               .BYTE 1
bl_ver_l:               .BYTE 1
bl_ver_h:               .BYTE 1
bl_cmd:                 .BYTE 1
bl_arg0:                .BYTE 1
bl_arg1:                .BYTE 1
bl_arg2:                .BYTE 1
bl_arg3:                .BYTE 1
bl_dummy:               .BYTE 6
bl_status:              .BYTE 1


bl_dirty:               .BYTE 16



; ---------------------------------------------------------
; other variables (all variables up to 'data_end' will be initialized to zero on start-up):


t1_cycle:               .BYTE 1 ; timer 1 cycle counter
test_mode:              .BYTE 1 ; = 1 if test mode active

addr_l:                 .BYTE 1
addr_h:                 .BYTE 1

data_end:






;==========================================================
; Flash
.CSEG
.org BOOT_RST_ADDR
rjmp RESET          ; Reset Handler
rjmp EXT_INT0       ; IRQ0 Handler
rjmp EXT_INT1       ; IRQ1 Handler
rjmp RESET          ; PCINT0 Handler
rjmp RESET          ; PCINT1 Handler
rjmp RESET          ; PCINT2 Handler
rjmp RESET          ; Watchdog Timer Handler
rjmp RESET          ; TIM2_COMPA ; Timer2 Compare A Handler
rjmp RESET          ; TIM2_COMPB ; Timer2 Compare B Handler
rjmp RESET          ; TIM2_OVF   ; Timer2 Overflow Handler
rjmp RESET          ; TIM1_CAPT  ; Timer1 Capture Handler
rjmp RESET          ; TIM1_COMPA ; Timer1 Compare A Handler
rjmp RESET          ; TIM1_COMPB ; Timer1 Compare B Handler
rjmp RESET          ; TIM1_OVF   ; Timer1 Overflow Handler
rjmp RESET          ; TIM0_COMPA ; Timer0 Compare A Handler
rjmp RESET          ; TIM0_COMPB ; Timer0 Compare B Handler
rjmp RESET          ; TIM0_OVF   ; Timer0 Overflow Handler
rjmp RESET          ; SPI Transfer Complete Handler
rjmp RESET          ; USART, RX Complete Handler
rjmp RESET          ; USART, UDR Empty Handler
rjmp RESET          ; USART, TX Complete Handler
rjmp RESET          ; ADC Conversion Complete Handler
rjmp RESET          ; EEPROM Ready Handler
rjmp RESET          ; Analog Comparator Handler
rjmp RESET          ; 2-wire Serial Interface Handler
rjmp RESET          ; Store Program Memory Ready Handler





;==========================================================
; Main program start
;
RESET:
    ; Set Stack Pointer to top of RAM
    ldi temp, high(RAMEND)
    out SPH, temp 
    ldi temp, low(RAMEND)
    out SPL, temp
 
    rcall init_system       ; Init I/O ports, timers, interrupts, clear RAM...


    ; Send 'B' and wait until it has been sent, then check if it was received again (loop-back), 
    ; if yes, stay in bootloader

    lds temp, UDR0

    ldi R24, 'B'
    rcall putchar
    rcall waitsend

    ; test if char received
    lds temp, UCSR0A
    sbrs temp, RXC0
    rjmp reset1
    
    ; test if received char is 'B'
    lds temp, UDR0
    cpi temp, 'B'
    brne reset1

    ; stay in bootloader
    rjmp start_bootloader


reset1:
    ; Test Reset Flags. If reset was triggerd by brown-out, external or power-on-reset, start firmware
    ; else we entered bootloader by a jump from the firmware and we stay in bootloader.
    in temp, MCUSR
    out MCUSR, zero_reg     ; clear reset flags
    andi temp, (1 << BORF) | (1 << EXTRF) | (1 << PORF)
    breq start_bootloader
    
    
    ldi R24, 'F'
    rcall putchar
    

    ; move vector table to beginning of flash
    ldi temp, (1 << IVCE)
    out MCUCR, temp
    out MCUCR, zero_reg

    ; start FW
    rjmp FW_START_ADDR



start_bootloader:
    sei



;==========================================================
; The main loop that runs continuously
;
main_loop:
    

    ; Only handle protocol if bus_state == 0
    cpi bus_state, 0
    RJMPNE main_end

    ; switch (bl_cmd)
    lds R24, bl_cmd
    cpi R24, 0          ; case 0:   no command
    RJMPEQ main_end

    ; --- Read Signature bytes ---
    cpi R24, CMD_READ_SIG
    brne main1

    BUSY
    cli

        clr ZH
        clr ZL
    
        ldi R24, (1 << SIGRD) | (1 << SELFPRGEN)
        out SPMCSR, R24
        LPM temp, Z
        sts bl_arg0, temp    
    
        adiw ZL, 2
        out SPMCSR, R24
        LPM temp, Z
        sts bl_arg1, temp    
    
        adiw ZL, 2
        out SPMCSR, R24
        LPM temp, Z
        sts bl_arg2, temp    

    sts bl_cmd, zero_reg
    sei
    NOT_BUSY
    rjmp main_end


main1:
    ; --- Set Address ---
    cpi R24, CMD_SET_ADDR
    brne main2

    BUSY
    cli

        ; Copy address to Z-pointer and separate variable
        lds ZL, bl_arg0
        lds ZH, bl_arg1
        sts addr_l, ZL      
        sts addr_h, ZH       

        .IF DEBUG_OUTPUT
        ldi R24, 'a'
        rcall putchar

        mov R24, ZH
        rcall print_hex_byte
        mov R24, ZL
        rcall print_hex_byte
        .ENDIF

    sts bl_cmd, zero_reg
    sei
    NOT_BUSY
    rjmp main_end


main2:
    ; --- Write Word ---
    cpi R24, CMD_WRITE_WORD
    brne main3

    BUSY
    cli
        push R0
        push R1

        ; Copy data to R1:R0
        lds R0, bl_arg0
        lds R1, bl_arg1

        ldi R24, (1 << SELFPRGEN)
        rcall do_spm
        
        adiw ZL, 2

        .IF DEBUG_OUTPUT
        ldi R24, 'W'
        rcall putchar

        ;mov R24, ZH
        ;rcall print_hex_byte
        ;mov R24, ZL
        ;rcall print_hex_byte
        .ENDIF

        pop R1
        pop R0

    sts bl_cmd, zero_reg
    sei
    NOT_BUSY
    rjmp main_end


main3:
    ; --- Read Word ---
    cpi R24, CMD_READ_WORD
    brne main4

    BUSY
    cli
    
        .IF DEBUG_OUTPUT
        ldi R24, 'w'
        rcall putchar

        lpm temp, Z+
        sts bl_arg0, temp
        lpm temp, Z+
        sts bl_arg1, temp
        .ENDIF

    sts bl_cmd, zero_reg
    sei
    NOT_BUSY
    rjmp main_end


main4:
    ; --- Page Erase ---
    cpi R24, CMD_PAGE_ERASE
    brne main5

    BUSY
    cli
    
        lds ZL, addr_l
        lds ZH, addr_h

        .IF DEBUG_OUTPUT
        ldi R24, 'e'
        rcall putchar

        mov R24, ZH
        rcall print_hex_byte
        mov R24, ZL
        rcall print_hex_byte
        .ENDIF
        
        ldi R24, (1 << PGERS) | (1 << SELFPRGEN)
        rcall do_spm

main4a:
        ; re-enable the RWW section
        ldi R24, (1 << RWWSRE) | (1 << SELFPRGEN)
        rcall do_spm

        in temp, SPMCSR
        sbrc temp, RWWSB    ; If RWWSB is set, the RWW section is not ready yet
        rjmp main4a

    sts bl_cmd, zero_reg
    sei
    NOT_BUSY
    rjmp main_end


main5:
    ; --- Page Write ---
    cpi R24, CMD_PAGE_WRITE
    brne main6

    BUSY
    cli
    
        lds ZL, addr_l
        lds ZH, addr_h

        .IF DEBUG_OUTPUT
        ldi R24, 'P'
        rcall putchar

        mov R24, ZH
        rcall print_hex_byte
        mov R24, ZL
        rcall print_hex_byte

        ldi R24, 0x0D
        rcall putchar
        ldi R24, 0x0A
        rcall putchar
        .ENDIF

        ldi R24, (1 << PGWRT) | (1 << SELFPRGEN)
        rcall do_spm

main5a:
        ; re-enable the RWW section
        ldi R24, (1 << RWWSRE) | (1 << SELFPRGEN)
        rcall do_spm

        in temp, SPMCSR
        sbrc temp, RWWSB    ; If RWWSB is set, the RWW section is not ready yet
        rjmp main5a

    sts bl_cmd, zero_reg
    sei
    NOT_BUSY
    rjmp main_end


main6:
    ; --- Write Byte to EEPROM ---
    cpi R24, CMD_WRITE_EE_BYTE
    brne main7

    BUSY
    cli

        .IF DEBUG_OUTPUT
        ldi R24, 'M'
        rcall putchar
        .ENDIF

        mov R24, ZL
        mov R25, ZH
        lds R22, bl_arg0
        rcall ee_write

    sts bl_cmd, zero_reg
    sei
    NOT_BUSY
    rjmp main_end


main7:
    ; --- Read Byte from EEPROM ---
    cpi R24, CMD_READ_EE_BYTE
    brne main8

    BUSY
    cli

        .IF DEBUG_OUTPUT
        ldi R24, 'm'
        rcall putchar
        .ENDIF

        mov R24, ZL
        mov R25, ZH
        rcall ee_read
        sts bl_arg0, R24


    sts bl_cmd, zero_reg
    sei
    NOT_BUSY
    rjmp main_end


main8:
    ; --- Start Firmware ---
    cpi R24, CMD_START_FW
    brne main9
    
    cli

        ; move vector table to beginning of flash
        ldi temp, (1 << IVCE)
        out MCUCR, temp
        out MCUCR, zero_reg

        .IF DEBUG_OUTPUT
        ldi R24, 'f'
        rcall putchar
        .ENDIF

        ; Start firmware
        rjmp FW_START_ADDR


main9:



main_end:

    rjmp main_loop




;--------------------------------------------------------
; IN: R24 = SPMCSR value
;
;
do_spm:
    
    ; Wait until any previous SPM command is finished
spm_wait1:
    in temp, SPMCSR
    sbrc temp, SELFPRGEN
    rjmp spm_wait1
    
    ; disable interrupts
    in temp2, SREG
    cli
    
    ; Wait until any EEPROM write access is finished
spm_wait_ee:
    sbic EECR, EEPE
    rjmp spm_wait_ee
    
    ; execute the actual SPM instruction
    out SPMCSR, R24
    spm

    ; Wait until SPM command is finished
spm_wait2:
    in temp, SPMCSR
    sbrc temp, SELFPRGEN
    rjmp spm_wait2
    
    ; restore SREG
    out SREG, temp2
    ret     
            







;==================================================================
; Init I/O ports, timers, interrupts etc
;
init_system:
    ; Init register variables
    clr bus_state
    clr zero_reg

    ldi YH, high(bl_data)  ; preset YH with address high byte, the IRQ code only changes YL
    
    ; move vector table to beginning of bootloader    
    ldi temp, (1 << IVCE)
    out MCUCR, temp
    ldi temp, (1 << IVSEL)
    out MCUCR, temp


    ; Init Ports
    ldi temp, 0x2C
    out DDRB, temp
    ldi temp, 0xFF
    out PORTB, temp

    ldi temp, 0x00
    out DDRC, temp
    ldi temp, 0x3F
    out PORTC, temp

    ldi temp, 0xF0
    out DDRD, temp
    ldi temp, 0xFF
    out PORTD, temp

    ; Init RAM
    ldi XL, low(bl_data)
    ldi XH, high(bl_data)
    clr temp
clear_ram:
    st X+, temp
    cpi XL, low(data_end)
    brne clear_ram
    cpi XH, high(data_end)
    brne clear_ram


	; Init variables
	ldi temp, high(BL_VERSION)
	sts bl_ver_h, temp
	ldi temp, low(BL_VERSION)
	sts bl_ver_l, temp

    ldi temp, high(BL_SIGNATURE)
    sts bl_sig_h, temp
    ldi temp, low(BL_SIGNATURE)
    sts bl_sig_l, temp

    ldi temp, 0x80
    sts bl_status, temp



    ; Init External INTs
    ldi temp, (1 << ISC11) | (1 << ISC10) | (1 << ISC01)    ; trigger on falling edge on INT0 (/WR) and rising edge on INT1 (/RD)
    sts EICRA, temp

    ldi temp, (1 << INT1) | (1 << INT0)
    out EIMSK, temp


    ; Init Timer 1 for 1 Hz interval
    ldi temp, (1 << WGM12) | 5 << CS10      ; CTC mode, CLK / 1024
    sts TCCR1B, temp 

    ldi temp, high(T1_OCR_1)
    sts OCR1AH, temp
    ldi temp, low(T1_OCR_1)
    sts OCR1AL, temp

    ; Init UART
    ldi temp, high(10)
    sts UBRR0H, temp
    ldi temp, low(10)
    sts UBRR0L, temp

    ldi temp, (1 << RXEN0) | (1 << TXEN0)
    sts UCSR0B, temp


    NOT_BUSY

    ret



;==================================================================
; IN: R24
print_hex_byte:
    push R24
    swap R24
    andi R24, 0x0F
    rcall nibble_to_hex
    rcall putchar
    pop R24        
    andi R24, 0x0F
    rcall nibble_to_hex
    rcall putchar        
    ret


;==================================================================
; IN:  R24 = nibble in bit 0..3
; OUT: R24 = hex char
nibble_to_hex:
    cpi R24, 10
    brsh nib1
    subi R24,  -'0'
    ret
nib1:
    subi R24, -('A' - 10)
    ret


;==================================================================
; IN: R24 character to be printed
putchar:
    lds temp, UCSR0A
    sbrs temp, UDRE0
    rjmp putchar

    sts UDR0, R24
    ret

;==================================================================
; 
waitsend:
    lds temp, UCSR0A
    sbrs temp, TXC0
    rjmp waitsend

    sts UCSR0A, temp    ; clear flags
    ret


;==================================================================
; Write ISR. Gets called, when /CS and RD-/WR are getting pulled low, 
; i.e. at the start of a write cycle to read the address or data.
;
EXT_INT0:
    in sreg_buf, SREG

    ; Read address or data from I/O bus
    in itemp, INPORT


    TESTPIN_L

    ; -----------------------------------------
    ; State 0 (address)
    cpi bus_state, 0
    brne wr_st1

    mov YL, itemp
    andi YL, 0x0F
.if low(bl_data) <= 63
    ldd bus_data, Y+low(bl_data)       ; read data from bl_data[address], in case a read cycle follows
.else
    subi YL, -low(bl_data)
    ld bus_data, Y
.endif

    ; output the high nibble, in case a read cycle follows
    out OUTPORT, bus_data

    inc bus_state
    rjmp wr_end


wr_st1:
    ; -----------------------------------------
    ; State 1 (high nibble)
    cpi bus_state, 1
    brne wr_st2

    andi itemp, 0x0F
    swap itemp
    mov bus_data, itemp

    inc bus_state
    rjmp wr_end


wr_st2:
    ; -----------------------------------------
    ; State 2 (low nibble)
    cpi bus_state, 2
    brne wr_end

    andi itemp, 0x0F
    or bus_data, itemp

    st Y, bus_data                              ; write data to bl_data[address]
    ldi itemp, 3
    std Y+(bl_dirty - bl_data), itemp         ; set bits 0 and 1 in the corresponding dirty byte


    NOT_BUSY
    clr bus_state

wr_end:
    TESTPIN_H
    out SREG, sreg_buf
    reti



;==================================================================
; Read ISR. Gets called at the end of a read cycle to prepare the next cycle.
;
EXT_INT1:
    in sreg_buf, SREG
    TESTPIN_L

    ; -----------------------------------------
    ; State 0 (busy state was just read) 
    cpi bus_state, 0
    brne rd_st1
    
    ;out OUTPORT, busy
    rjmp rd_end
   
rd_st1:
    ; -----------------------------------------
    ; State 1 (high nibble was just read) 
    cpi bus_state, 1
    brne rd_st2

    swap bus_data           ; prepare low nibble
    out OUTPORT, bus_data
    inc bus_state
    rjmp rd_end  

rd_st2:
    ; -----------------------------------------
    ; State 2  (low nibble was just read)
    cpi bus_state, 2
    brne rd_end
    
    NOT_BUSY
    clr bus_state


rd_end:
    TESTPIN_H
    out SREG, sreg_buf
    reti




;==================================================================
; Include other source files
;
.include "eeprom.asm"
