/*
    RTC Flash - Firmware updater for the M3002 RTC replacement 'Re3002'.
    Copyright (C) 2013  Oliver Tscherwitschke <oliver@tscherwitschke.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/


#include <string.h>

#include "ihex.h"



/****************************************************************/
/* convert a hex digit to a value                               */
/****************************************************************/
unsigned char hex_to_nibble(unsigned char hex)
{
	if ((hex >= '0') && (hex <= '9'))
		return hex - 0x30;
	else		/* assume A - F */
		return hex - 'A' + 10;
}


/****************************************************************/
/* convert a string to an unsigned char value                   */
/****************************************************************/
unsigned char _atoc(char* buf)
{
	unsigned char val = 0;
	unsigned char i;
	
	for (i = 0; i < 2; i++)
	{
		val = val << 4;
		val |= hex_to_nibble(buf[i]);
	}
	
	return val;
}

/****************************************************************/
/* convert a string to an unsigned short value                  */
/****************************************************************/
unsigned short _atos(char* buf)
{
	unsigned short val = 0;
	unsigned char i;
	
	for (i = 0; i < 4; i++)
	{
		val = val << 4;
		val |= hex_to_nibble(buf[i]);
	}
	
	return val;
}


/****************************************************************/
/* ihex_checksum(): checks the Checksum of one I-Hex record     */
/* Input: rec:   pointer to record string                       */
/* Return:       0:     Checksum OK                             */
/*               1:     Bad checksum                            */
/****************************************************************/
unsigned char ihex_checksum(char rec[])
{
	unsigned char i;
	unsigned char reclen;
	unsigned char bufflen;
	unsigned char checksum = 0;
	
	bufflen = strlen(rec);

	if (bufflen < 11)			/* string too short */
		return 1;
	
	if (rec[0] != ':')			/* no i-hex line */
		return 1;
	
	reclen = _atoc(rec + 1);
	
	if (11 + reclen * 2 != bufflen)
		return 1;				/* wrong length */

	for (i = 0; i < 5 + reclen; i++)
	{
		checksum += _atoc(rec + 1 + 2 * i);
	}
	
	if (checksum != 0)			/* bad checksum */
		return 1;
		
	return 0;					/* everything OK */
}


/****************************************************************/
/* ihex_load_record(): load one i-hex record to buffer          */
/* Input:        sBuff: string                                  */
/* Return:       0:              OK                             */
/*               1:             End of Intel-Hex record         */
/*               2:             unsupported hex record          */
/*               3:             address out of range            */
/****************************************************************/
unsigned char ihex_load_record(char rec[], unsigned char *buffer, unsigned long flashsize, unsigned long *maxaddr)
{
	unsigned short i;
	
	unsigned short ucRecLen;
	unsigned short usiOffset;
	unsigned char ucRecType;
	unsigned char ucData;
    static unsigned long ulSBA = 0;    /* Segment Base Address (SBA) */
    unsigned long ulAdr;

	ucRecLen = _atoc(rec + 1);
	usiOffset = _atos(rec + 3);
	ucRecType = _atoc(rec + 7);

	if (ucRecType == 1)					/* end record */
		return 1;

    if (ucRecType == 2)                 /* Extended Segment Address Record */
    {
        ulSBA = _atos(rec + 9) << 4;
        *maxaddr = 0;
        return 0;
    }

	if (ucRecType != 0)					/* no data record */
		return 2;

	for (i = 0; i < ucRecLen; i++)
	{
		ucData = _atoc(rec + 9 + i * 2);
        ulAdr = ulSBA + (unsigned short) (usiOffset + i);

        if (ulAdr < flashsize)
            buffer[(unsigned short) ulAdr] = ucData;
        else
            return 3;       /* address out of range */
	}

	*maxaddr = ulAdr;		/* return max addr */

	return 0;
}


/****************************************************************/
/* read the binary file into the buffer                         */
/* return: size of file or -1 if size to big                    */
/****************************************************************/
long read_bin_file(unsigned char *buffer, unsigned long flashsize, FILE *infile)
{
    long int filesize;

    /* read file into buffer */
    fseek(infile, 0, SEEK_END);
    filesize = ftell(infile);
    fseek(infile, 0, SEEK_SET);

    if (filesize > flashsize)           /* file too big */
        return -1;
    else
        fread(buffer, 1, (size_t) filesize, infile);

    return filesize;
}


/****************************************************************/
/* read the intel hex file into the buffer                      */
/* return: size of file or -1 if size to big or other error     */
/****************************************************************/
long read_hex_file(unsigned char *buffer, unsigned long flashsize, FILE *infile)
{
	char sBuff[1024];
	unsigned char rc;
	unsigned long usiMaxAddr = 0;
	unsigned long usiActAddr;
    char *c;

	do
	{
		/* read one record from infile */
		if (fgets(sBuff, sizeof sBuff, infile) == NULL)
		    break;

        /* remove newline */
        c = sBuff;
        while ((*c != '\n') && (*c != '\r') && (*c != '\0'))
            c++;
        *c = '\0';            
        
		
		if (ihex_checksum(sBuff) == 1)	    /* checksum error */
		{
			printf("Checksum error\n");
			return -1;
		}
		else								/* checksum is OK */
		{
			rc = ihex_load_record(sBuff, buffer, flashsize, &usiActAddr);	/* load record to buffer */
            if (rc == 0)
            {
			    if (usiActAddr > usiMaxAddr)				    /* remember max address */
				    usiMaxAddr = usiActAddr;
            }				
			else if (rc == 2)                                   /* ERR_UNSUPPORTED */
			{
				printf("Unsupported Hex record\n");
				return -1;
			} 
			else if (rc == 3)
			{
				printf("Address out of range\n");
				return -1;
			}
			
		}

	} 
	while (rc != 1);

	return usiMaxAddr + 1;
}



