/* 
 *  Locomotive BASIC lister 
 * 
 *  Will list Amstrad CPC BASIC programs.
 *
 *  (c) Copyright, Kevin Thacker 2002,2003
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/* keywords without any prefix token byte */
const char *Keywords[]=
{
"AFTER",
"AUTO",
"BORDER",
"CALL",
"CAT",
"CHAIN",
"CLEAR",
"CLG",
"CLOSEIN",
"CLOSEOUT",
"CLS",
"CONT",
"DATA",
"DEF",
"DEFINT",
"DEFREAL",
"DEFSTR",
"DEG",
"DELETE",
"DIM",
"DRAW",
"DRAWR",
"EDIT",
"ELSE",
"END",
"ENT",
"ENV",
"ERASE",
"ERROR",
"EVERY",
"FOR",
"GOSUB",
"GOTO",
"IF",
"INK",
"INPUT",
"KEY",
"LET",
"LINE",
"LIST",
"LOAD",
"LOCATE",
"MEMORY",
"MERGE",
"MID$",
"MODE",
"MOVE",
"MOVER",
"NEXT",
"NEW",
"ON",
"ON BREAK",
"ON ERROR GOTO",
"SQ",
"OPENIN",
"OPENOUT",
"ORIGIN",
"OUT",
"PAPER",
"PEN",
"PLOT",
"PLOTR",
"POKE",
"PRINT",
"'",
"RAD",
"RANDOMIZE",
"READ",
"RELEASE",
"REM",
"RENUM",
"RESTORE",
"RESUME",
"RETURN",
"RUN",
"SAVE",
"SOUND",
"SPEED",
"STOP",
"SYMBOL",
"TAG",
"TAGOFF",
"TROFF",
"TRON",
"WAIT",
"WEND",
"WHILE",
"WIDTH",
"WINDOW",
"WRITE",
"ZONE",
"DI",
"EI",
"FILL",
"GRAPHICS",
"MASK",
"FRAME",
"CURSOR",
NULL,
"ERL",
"FN",
"SPC",
"STEP",
"SWAP",
NULL,
NULL,
"TAB",
"THEN",
"TO",
"USING",
">",
"=",
">=",
"<",
"<>",
"<=",
"+",
"-",
"*",
"/",
"^",
"\\",
"AND",
"MOD",
"OR",
"XOR",
"NOT",
NULL
};

/* keywords with prefix token of &ff */
const char *AdditionalKeywords[]=
{
"ABS",
"ASC",
"ATN",
"CHR$",
"CINT",
"COS",
"CREAL",
"EXP",
"FIX",
"FRE",
"INKEY",
"INP",
"INT",
"JOY",
"LEN",
"LOG",
"LOG10",
"LOWER$",
"PEEK",
"REMAIN",
"SGN",
"SIN",
"SPACE$",
"SQ",
"SQR",
"STR$",
"TAN",
"UNT",
"UPPER$",
"VAL",
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
"EOF",
"ERR",
"HIMEM",
"INKEY$",
"PI",
"RND",
"TIME",
"XPOS",
"YPOS",
"DERR",
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
"BIN$",
"DEC$",
"HEX$",
"INSTR",
"LEFT$",
"MAX",
"MIN",
"POS",
"RIGHT$",
"ROUND",
"STRING$",
"TEST",
"TESTR",
"COPYCHR$",
"VPOS",
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
NULL,
};

/*-----------------------*/
/* display variable name */

/* last character of variable name has bit 7 set */
static unsigned char *DisplayVariable(unsigned char *pLinePtr)
{
	char ch;

	do
	{
		/* get character */
		ch = pLinePtr[0];
		/* increment byte position in tokenised BASIC program */
		pLinePtr++;
		/* output character */
		printf("%c",ch&0x07f);
	}
	while ((ch&0x080)==0);

	/* return updated position in tokenised BASIC program */
	return pLinePtr;
}

/*-------------------------------------*/
/* translate a line of tokenised BASIC */
static void TranslateLine(unsigned char *pLinePtr, unsigned long LineLength)
{
	unsigned char *pLocalLinePtr = pLinePtr;
	unsigned char Token;

	do
	{
		Token = pLocalLinePtr[0];
		pLocalLinePtr++;

		switch (Token)
		{
			/* end of line marker */
			case 0x00:
				break;

			/* instruction seperator */
			case 0x01:
			{
				/* : is the BASIC instruction seperator */
				/* this also occurs:
				
				1. immediatly before a ELSE instruction,
				 e.g. in the IF THEN .. ELSE .. instruction sequence 
				2. immediatly before a comment using ' character
				*/

				/* is next instruction the BASIC command "ELSE"? */
				if ((pLocalLinePtr[0]!=0x097) && (pLocalLinePtr[0]!=0x0c0))
				{
					/* no, display the seperator */
					printf(":");
				}
			}
			break;

			/* integer variable definition with "%" suffix */
			case 0x02:
			{
				pLocalLinePtr+=2;

				pLocalLinePtr=DisplayVariable(pLocalLinePtr);
				printf("%%");
			}
			break;

			/* string variable definition with "$" suffix */
			case 0x03:
			{
			
				pLocalLinePtr+=2;

				pLocalLinePtr=DisplayVariable(pLocalLinePtr);
				printf("$");
			}
			break;

			/* floating point variable definition with "!" suffix */
			case 0x04:
			{
				pLocalLinePtr+=2;

				pLocalLinePtr=DisplayVariable(pLocalLinePtr);
				printf("!");
			}
			break;

			/* variable with no suffix */
			case 0x05:
			case 0x06:
			case 0x07:
			case 0x08:
			case 0x09:
			case 0x0a:
			case 0x0b:
			case 0x0c:
			case 0x0d:
			{
				pLocalLinePtr+=2;

				pLocalLinePtr=DisplayVariable(pLocalLinePtr);
			}
			break;

			/* number constant */
			case 0x0e:
			case 0x0f:
			case 0x10:
			case 0x11:
			case 0x12:
			case 0x13:
			case 0x14:
			case 0x15:
			case 0x16:
			case 0x17:
			case 0x18:
			{
				int Number;

				Number = (Token-0x0e);
				printf("%d",Number);
			}
			break;

			/* 8-bit integer decimal number */
			case 0x019:
			{			
				unsigned short Number;

				Number = (pLocalLinePtr[0]&0x0ff);

				printf("%d",Number);

				pLocalLinePtr++;
			}
			break;

			/* 16-bit integer decimal number */
			case 0x01a:
			{			
				unsigned short Number;

				Number = (pLocalLinePtr[0]&0x0ff)|((pLocalLinePtr[1]&0x0ff)<<8);

				printf("%d",Number);

				pLocalLinePtr+=2;
			}
			break;

			case 0x01b:
			{
				int j;
				int bIgnoreDigit = 1;

				/* 16-bit binary number */
				unsigned short Number;

				Number = (pLocalLinePtr[0]&0x0ff)|
					((pLocalLinePtr[1]&0x0ff)<<8);

				/* number is displayed with preceeding 0 digits removed */

				printf("&X");
				
				for (j=0; j<16; j++)
				{
					int bd;

					bd = (Number>>(15-j))&0x01;

					if (bd!=0)
					{
						bIgnoreDigit = 0;
					}

					if (!bIgnoreDigit)
					{
						printf("%c",'0'+bd);
					}
				}
			
				pLocalLinePtr+=2;
			}
			break;

			/* 16-bit hexidecimal integer number */
			case 0x01c:
			{
				unsigned short Number;

				Number = (pLocalLinePtr[0]&0x0ff)|
					((pLocalLinePtr[1]&0x0ff)<<8);

				/* hexidecimal numbers are displayed in upper case,
				with preceeding 0 digits removed */
				printf("&%X",Number);
				
				pLocalLinePtr+=2;
			}
			break;

			/* 16-bit line memory address pointer */
			case 0x01d:
				break;

			/* decimal line number */
			case 0x01e:
			{
				unsigned short Number;

				Number = (pLocalLinePtr[0]&0x0ff)|
					((pLocalLinePtr[1]&0x0ff)<<8);

				printf("%d",Number);

				pLocalLinePtr+=2;
			}
			break;

			case 0x01f:
			{
				unsigned char exponent;
				unsigned char sign;
				unsigned long mantissa;

				exponent = pLocalLinePtr[4];
				mantissa = ((pLocalLinePtr[3]&0x07f)<<24)|
							((pLocalLinePtr[2]&0x0ff)<<16)|
							((pLocalLinePtr[1]&0x0ff)<<8)|
							((pLocalLinePtr[0]&0x0ff));
				sign = pLocalLinePtr[3]&0x080;

				if (sign!=0)
				{
					printf("-");
				}

				pLocalLinePtr+=5;
			}
			break;

			case 0x022:
			{
				/* string; terminated with a quote, or end of line */
				char ch;

				ch = Token;

				printf("%c",ch);
				do
				{
					ch = pLocalLinePtr[0];
					pLocalLinePtr++;
					if (ch==0)
					{
						pLocalLinePtr--;
					}
					else
					{
						printf("%c",ch);
					}
				}
				while ((ch!='"') && (ch!='\0'));
			}
			break;

			case 0x0ff:
			{
				unsigned char Keyword;

				Keyword = pLocalLinePtr[0];
				pLocalLinePtr++;

				printf("%s",AdditionalKeywords[Keyword&0x0ff]);
			}
			break;

			/* | symbol; character prefix for RSX command */
			case 0x07c:
			{
				printf("|");
				/* first byte following '|' symbol is a byte offset */
				pLocalLinePtr++;
				/* following this is the RSX name. Last character of name has bit 7 set to 1 */
				pLocalLinePtr = DisplayVariable(pLocalLinePtr);
			}
			break;


			default:
			{
				if ((Token>=0x080) && (Token<=0x0fe))
				{
					unsigned char NextToken;

					printf("%s",Keywords[(Token-0x080)]);
			
					/* is this token PRINT and the next token is:
						- extended keyword 
						- variable name
						- keyword
					? */
					
					if (Token==0x0bf)
					{
						NextToken = pLocalLinePtr[0];

						if (
							/* extended keyword */
							(NextToken==0x0ff) ||
							((NextToken>=0x080) && (NextToken<=0x0fe)) ||
							/* variable name */
							((NextToken>=2) && (NextToken<=0x0d))
							)
						{
							/* output a space */
							printf(" ");
						}
					}
				}
				else if ((Token>=0x020) && (Token<=0x07f))
				{
					printf("%c",Token);
				}
				else
				{
					printf("--%02x--",Token);
				}
			}
			break;
		}
	}
	while (Token!=0);

}


/*--------------------*/
/* calculate checksum */
static unsigned short AMSDOSHeader_CalculateChecksum(const char *pData)
{
	unsigned short CalculatedChecksum = 0;
	unsigned long i;

	/* generate checksum */
	for (i=0; i<66; i++)
	{
		CalculatedChecksum+=(unsigned short)(pData[i]&0x0ff);
	}

	return CalculatedChecksum;
}

/*------------------------------------------------*/
/* Detect if there is a AMSDOS header on the file */
static int AMSDOSHeader_Checksum(const char *pData)
{
	unsigned short StoredChecksum;
	unsigned short CalculatedChecksum;

	CalculatedChecksum = AMSDOSHeader_CalculateChecksum(pData);

	/* get the stored checksum */
	StoredChecksum = (unsigned short)((pData[67]&0x0ff)|((pData[68]&0x0ff)<<8));

	return (CalculatedChecksum==StoredChecksum);
}


/*-------------------------------------*/
int main(int argc, char **argv)
{
	if (argc!=2)
	{
		printf("BASLIST <filename>\n\n");
	}
	else
	{
		FILE *fh;

		fh = fopen(argv[1],"rb");

		if (fh!=NULL)
		{
			unsigned char *pBuffer;
			unsigned long length;

			fseek(fh,0, SEEK_END);
			length = ftell(fh);
			fseek(fh,0, SEEK_SET);

			pBuffer = (unsigned char *)malloc(length);

			if (pBuffer)
			{
				if (fread(pBuffer, sizeof(unsigned char), length, fh)==length)
				{
					unsigned long DataOffset = 0;

					if (length>128)
					{
						/* has a AMSDOS header? */
						if (AMSDOSHeader_Checksum(pBuffer))
						{
							/* yes */
							DataOffset=128;
						}
					}

					{
						unsigned long LineLength;
						unsigned long LineNumber;

						unsigned char *pProgramPtr = pBuffer+DataOffset;

						do
						{
							LineLength = (pProgramPtr[0]&0x0ff)+((pProgramPtr[1]&0x0ff)<<8);

							/* if LineLength==0, then end of BASIC program found, else
							contains length of tokenised line in bytes */
							if (LineLength!=0)
							{
								/* get decimal line number */
								LineNumber = (pProgramPtr[2]&0x0ff)+((pProgramPtr[3]&0x0ff)<<8);

								/* display line number and space */
								printf("%d ",LineNumber);

								/* decode tokenised data and display */
								TranslateLine(&pProgramPtr[4],LineLength);
							
								/* display ending cr */
								printf("\n");

								/* updated pointer for next basic line */
								pProgramPtr+=LineLength;
							}
						}
						while (LineLength!=0);
					}
				}

				free(pBuffer);
			}

			fclose(fh);
		}
	}

	exit(0);
}
