/*
 * @(#)Amstrad.java 1.0 26 Aug 2001 Kevin Thacker
 *
 *  ArnoldJ emulator (c) Copyright, Kevin Thacker
 *  
 *  This file is part of the ArnoldJ emulator source code distribution.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *  The Z80 emulation is not covered by this license and is covered by the
 *  original license with the Jasper emulator. Z80 emulation has been
 *  distributed with this emulator with permission.*/

import java.awt.*;
import java.util.*;
import java.io.*;
import java.net.*;
import java.awt.image.*;

/**
 * The Amstrad class extends the Z80 class implementing the Amstrad CPC
 * hardware.
 *
 * @version 1.0 26 Aug 2001
 * @author <A HREF="http://arnold.emuunlim.com/">Arnold Homepage</A>
 *
 * @see arnoldj
 * @see Z80
 */

public class amstrad extends z80 
{
	final int SCREEN_WIDTH = 640;
	final int SCREEN_HEIGHT = 200;
	final int BORDER_WIDTH = 8;
	final int BORDER_HEIGHT = 8;

	//public Graphics		parentGraphics = null;
	public Graphics		canvasGraphics = null;
	public Graphics		bufferGraphics = null;
	public Image		bufferImage = null;
    public MemoryImageSource MemImage = null;
    //public MemoryImageSource source = null;

	public Container	parent      = null; // SpecApp usually (where border is drawn)
	public Canvas		canvas      = null; // Main screen
	public TextField	urlField    = null; // ESC shows a URL popup
    public int W;
    public int H;

    /* array holding disk image data */
    private int diskimage_data[];
    /* disk image type: 0 = standard CPCEMU disk image, 1 = extended CPCEMU disk image */
    private int diskimage_type;
    /* disk image track offsets */
    private int diskimage_track_offsets[] = new int [84*2];
    /* sector offsets for current track */
    private int diskimage_sector_offsets[] = new int [21];

    int pix[];
	public amstrad( Container _parent ) throws Exception 
	{
		// Amstrad runs at 4.0Mhz
		super( 3.5 );

		parent = _parent;

//		nPixelsWide = 800;
//		nPixelsHigh = 600;
//		pixelScale = 1;

		parent.add( canvas = new Canvas() );
		canvas.resize( this.SCREEN_WIDTH+this.BORDER_WIDTH, this.SCREEN_HEIGHT+this.BORDER_HEIGHT);
		canvas.show();

        W = 800;
        H = 600;

    	pix =  new  int[W  *  H];

	    int value = parent.getBackground().getRGB();

	   for (int i=0; i<W*H; i++)
	    {
	        pix[i] = value;
	    }


		MemImage = new MemoryImageSource(W,H,pix,0,W);
        MemImage.setAnimated(true);
        //MemImage.addConsumer(parent);

        bufferImage = parent.createImage(MemImage);
        canvas.setForeground(new Color(255,0,0));
		//bufferImage = parent.createImage( nPixelsWide*pixelScale, nPixelsHigh*pixelScale );
		//bufferGraphics = bufferImage.getGraphics();
		//parentGraphics = parent.getGraphics();
		canvasGraphics = canvas.getGraphics();
      //  canvasGraphics.setColor(new Color(255,0,0));
//		parent.add( progressBar = new AMDProgressBar() );
//		progressBar.setBarColor( new Color( 192, 52, 4 ) );
//		progressBar.show();
//		progressBar.hide();

		parent.add( urlField = new TextField() );
		urlField.show();
		urlField.hide();
	}

	public void setBorderWidth( int width ) {
		borderWidth = width;

		canvas.move( borderWidth, borderWidth );

		urlField.reshape( 0, 0,
			parent.preferredSize().width,
			urlField.preferredSize().height );

//		progressBar.reshape(
//			1, (borderWidth+nPixelsHigh*pixelScale)+2,
//			nPixelsWide*pixelScale+borderWidth*2-2, borderWidth-2 );

//		progressBar.setFont( urlField.getFont() );
	}

	/**** GRAPHICS DECODE ****/
	/* tables to convert graphics byte to pen index */
	private int amstrad_mode0_gfx_decode[] = new int [256];
	private int amstrad_mode1_gfx_decode[] = new int [256];
	private int amstrad_mode2_gfx_decode[] = new int [256];

	private final void graphics_decode_setup()
	{
		int i;
		int Pixel;

		for (i=0; i<256; i++)
		{
			/* mode 0 pixel */
			Pixel = ((i & 0x02)<<2) |
				((i & 0x020)>>(5-2)) |
				((i & 0x08)>>(3-1)) |
				((i & 0x080)>>(7-0));

			amstrad_mode0_gfx_decode[i] = Pixel;

			/* mode 1 pixel */
	        Pixel = ((i & 0x08)>>(3-1)) | ((i & 0x080)>>7);
			amstrad_mode1_gfx_decode[i] = Pixel;

			/* mode 2 pixel */
			Pixel = (i>>7) & 1;
			amstrad_mode2_gfx_decode[i] = Pixel;
		}
	}



	/* keyboard */

	private int amstrad_keyboard[] = new int [16];
	private int amstrad_keyboard_line;

	private final void amstrad_reset_keyboard()
	{
		for (int i=0; i<16; i++)
		{
			amstrad_keyboard[i] = 255;
		}
	}

	private final int amstrad_read_keyboard()
	{
		return amstrad_keyboard[amstrad_keyboard_line];
	}


	/**
		CRTC 6845

		over simplified
	*/

	private int crtc_registers[] = new int [32];
	private int crtc_regindex;

	private final void crtc_write_regindex(int outByte)
	{
		crtc_regindex = outByte & 0x01f;
	}

	private final void crtc_write_reg(int outByte)
	{
		if (crtc_regindex == 12)
		{
			if ((crtc_registers[12] & 0x030)!=(outByte & 0x030))
			{
				int base = (outByte & 0x030)<<(2+8);

				for (int i=0; i<16384; i++)
				{
					if (amstrad_scr_bytes[i]!=amstrad_memory[base+i])
					{
						plot(i, amstrad_memory[base+i]);
					}
				}
			}
		}

		crtc_registers[crtc_regindex] = outByte;
	}

	private final int crtc_read_reg()
	{
		return crtc_registers[crtc_regindex];
	}

	/**
		PSG

		over simplified
		*/

	private int psg_registers[] = new int [16];
	private int psg_regindex;

	private final void	psg_write_regindex(int outByte)
	{
		psg_regindex = outByte & 0x0f;
	}

	private final void psg_write_reg(int outByte)
	{
		psg_registers[psg_regindex] = outByte;
	}

	private final int psg_read_reg()
	{
		if (psg_regindex == 14)
		{
			if ((psg_registers[7] & 0x040)==0x00)
			{
				/* read keyboard */
				return amstrad_read_keyboard();
			}
			else
				return 0x0ff;
		}
		else
		{
			return psg_registers[psg_regindex];
		}
	}

	/**
		PPI

	*/

    private final int PPI_CONTROL_FUNCTION = 0x080;
    private final int PPI_CONTROL_PORT_A_CU_MODE1 = 0x040;
    private final int PPI_CONTROL_PORT_A_CU_MODE0 = 0x020;
    private final int PPI_CONTROL_PORT_A_IO = 0x010;
    private final int PPI_CONTROL_PORT_C_UPPER_IO = 0x08;
    private final int PPI_CONTROL_PORT_B_CL_MODE = 0x04;
    private final int PPI_CONTROL_PORT_B_IO = 0x02;
    private final int PPI_CONTROL_PORT_C_LOWER_IO = 0x01;

    private int ppi_latches[] = new int [3];
    private int ppi_io[] = new int [3];
    private int ppi_control;
    private int ppi_outputs[] = new int [3];
    private int ppi_inputs[] = new int [3];

	private final void amstrad_handle_port_c()
	{
		amstrad_keyboard_line = ppi_outputs[2] & 0x0f;

		switch (ppi_outputs[2] & 0x0c0)
		{
			case 0x000:
			{
			}
			break;

			case 0x040:
			{
				ppi_inputs[0] = psg_read_reg();
			}
			break;

			case 0x080:
			{
				psg_write_reg(ppi_outputs[0]);
			}
			break;

			case 0x0c0:
			{
				psg_write_regindex(ppi_outputs[0]);
			}
			break;
		}
	}

	private final int ppi_read_port(int port_index)
	{
	    return ((ppi_inputs[port_index] & ppi_io[port_index]) |
	            (ppi_latches[port_index] & (~ppi_io[port_index])));
	}

    private final void ppi_write_port(int port_index, int data)
    {
        ppi_latches[port_index] = data;

        ppi_outputs[port_index] = ((ppi_latches[port_index] & (~ppi_io[port_index])) |
                                    (0x0ff & ppi_io[port_index]));
    }

	private final int ppi_read_port_a()
	{
		amstrad_handle_port_c();

        return ppi_read_port(0);
    }

	private final int ppi_read_port_b()
	{
	    ppi_inputs[0] = ((0x0f)<<1) | 0x01;

	    return ppi_read_port(1);
	}

	private final int ppi_read_port_c()
	{
	    ppi_inputs[0] = 0x02f;

	    return ppi_read_port(2);
	}

	private final void ppi_write_port_a(int outByte)
	{
	    ppi_write_port(0,outByte);

		amstrad_handle_port_c();
	}

	private final void ppi_write_port_b(int outByte)
	{
	    ppi_write_port(1,outByte);
	}

	private final void ppi_write_port_c(int outByte)
	{
	    ppi_write_port(2,outByte);

		amstrad_handle_port_c();
	}

	private final void ppi_write_control(int outByte)
	{
	    if ((outByte & 0x080)!=0)
	    {
	        ppi_control = outByte;

	        if ((ppi_control & PPI_CONTROL_PORT_A_IO)!=0)
	        {
	            /* input */
	            ppi_io[0] = 0x0ff;
	        }
	        else
	        {
	            /* output */
	            ppi_io[0] = 0x00;
	        }

	        if ((ppi_control & PPI_CONTROL_PORT_B_IO)!=0)
	        {
	            /* input */
	            ppi_io[1] = 0x0ff;
	        }
	        else
	        {
	            /* output */
	            ppi_io[1] = 0x00;
	        }

	        ppi_io[2] = 0;

	        if ((ppi_control & PPI_CONTROL_PORT_C_UPPER_IO)!=0)
	        {
	            /* upper is input */
	            ppi_io[2] |= 0x0f0;
	        }

	        if ((ppi_control & PPI_CONTROL_PORT_C_LOWER_IO)!=0)
	        {
	            /* lower is input */
	            ppi_io[2] |= 0x0f;
	        }

            /* clear latches */
	        ppi_write_port_a(0);
	        ppi_write_port_b(0);
	        ppi_write_port_c(0);
	    }
	    else
	    {
	        int BitIndex = (outByte>>1) & 7;

	        if ((outByte & 1)==1)
	        {
	            ppi_latches[2] |= 1<<BitIndex;
	        }
	        else
	        {
	            ppi_latches[2] &= ~(1<<BitIndex);
	        }

	        ppi_write_port_c(ppi_latches[2]);
	    }
	}

	/**
		Amstrad ROMS
		*/

	public int amstrad_urom_location;
	public int amstrad_urom_offset;

    public int amstrad_urom_selected;

	/* select a upper rom */
	public void	amstrad_rom_select(int outByte)
	{
	    amstrad_urom_selected = outByte;

		switch (outByte)
		{
			case 7:
			{
				/* amsdos selected */
				amstrad_urom_location = 2;
				amstrad_urom_offset = 0;
			}
			break;

			default:
			{
				/* basic selected, or any other rom */
				amstrad_urom_location = 1;
				amstrad_urom_offset = 16384;
			}
			break;

		}

		amstrad_update_mem();
	}


	/**
	 *
		Amstrad GA
	*/
	/* this array holds the colour settings */
	public int amstrad_ga_colour_pal[] = new int [18];

    private int amstrad_ga_render_colour[] = new int [18];

	/* pen select register */
	public int amstrad_ga_penselect;
	/* pen data register */
	public int amstrad_ga_pendata;
	/* rom config/mode register */
	public int amstrad_ga_rom_mode;
	/* ram config register */
	public int amstrad_ga_ram_config;

	/* defines ram config selected */
	public int amstrad_ram_location[] = new int [4];
	public int amstrad_ram_offset[] = new int [4];

	public int     amstrad_ram_configs[] = new int [8*4];

	public void amstrad_update_mem()
	{
		/* write */
		amstrad_write_location[3] = amstrad_ram_location[3];
		amstrad_write_offset[3] = amstrad_ram_offset[3];
		amstrad_write_location[2] = amstrad_ram_location[2];
		amstrad_write_offset[2] = amstrad_ram_offset[2];
		amstrad_write_location[1] = amstrad_ram_location[1];
		amstrad_write_offset[1] = amstrad_ram_offset[1];
		amstrad_write_location[0] = amstrad_ram_location[0];
		amstrad_write_offset[0] = amstrad_ram_offset[0];

		/* read */

		if ((amstrad_ga_rom_mode & 0x08)==0x00)
		{
			/* ROM enabled: upper rom enabled - Basic, Amsdos ... */
			amstrad_read_location[3] = amstrad_urom_location;
			amstrad_read_offset[3] = amstrad_urom_offset;
		}
		else
		{
			amstrad_read_location[3] = amstrad_ram_location[3];
			amstrad_read_offset[3] = amstrad_ram_offset[3];
		}


		amstrad_read_location[2] = amstrad_ram_location[2];
		amstrad_read_offset[2] = amstrad_ram_offset[2];

		amstrad_read_location[1] = amstrad_ram_location[1];
		amstrad_read_offset[1] = amstrad_ram_offset[1];


		if ((amstrad_ga_rom_mode & 0x04)==0x00)
		{
			/* ROM enabled: lower rom enabled - OS */
			amstrad_read_location[0] = 1;
			amstrad_read_offset[0] = 0;
		}
		else
		{
			amstrad_read_location[0] = amstrad_ram_location[0];
			amstrad_read_offset[0] = amstrad_ram_offset[0];
		}
	}


	public void	amstrad_ga_write(int outByte)
	{
		switch (outByte & 0x0c0)
		{
			/* set pen index */
			case 0:
			{
				amstrad_ga_penselect = outByte;
			}
			break;

			/* set pen colour */
			case 0x040:
			{
				int pen;

				amstrad_ga_pendata = outByte;

				if ((amstrad_ga_penselect & 0x010)!=0)
				{
					/* border */
					pen = 16;
				}
				else
				{
					pen = amstrad_ga_penselect & 0x0f;
				}

                if (amstrad_ga_colour_pal[pen] != (amstrad_ga_pendata & 0x01f))
                {
                    refreshscreen = true;
					//for (int i=0; i<8192; i++)
					//{
					//    //plot_update(i<<1);
					//}
					/*
						switch (amstrad_ga_rom_mode & 0x03)
						{
							case 1:
							{
								int byte0, byte1;


								byte0 = amstrad_scr_bytes[(i<<1)];
								byte1 = amstrad_scr_bytes[(i<<1)+1];

								for (int p=0; p<3; p++)
								{
									int thisPen;

									thisPen = amstrad_mode1_gfx_decode[byte0 & 0x0ff];

									if (thisPen == pen)
									{
										plot_update((i<<1));
										break;
									}


									thisPen = amstrad_mode1_gfx_decode[byte1 & 0x0ff];

									if (thisPen == pen)
									{
										plot_update((i<<1));
										break;
									}

									byte0 = byte0<<1;
									byte1 = byte1<<1;
								}
							}
							break;
						}
					}
					*/
				}
                
				amstrad_ga_colour_pal[pen] = amstrad_ga_pendata & 0x01f;

			    amstrad_ga_render_colour[pen] = (amstrad_colours[amstrad_ga_colour_pal[pen]].getRed()<<16) |
			                                    (amstrad_colours[amstrad_ga_colour_pal[pen]].getGreen()<<8) |
			                                    (amstrad_colours[amstrad_ga_colour_pal[pen]].getBlue()) |
			                                    (0x0ff<<24);

			}
			break;

			case 0x080:
			{
			    if ((outByte & 0x03)!=(amstrad_ga_rom_mode & 0x03))
			    {
			             refreshscreen = true;
	            }

				amstrad_ga_rom_mode = outByte;

				amstrad_update_mem();

			}
			break;

			case 0x0c0:
			{
				int config_index;

				amstrad_ga_ram_config = outByte;

				config_index = outByte & 0x07;

				amstrad_ram_location[0] = 0;
				amstrad_ram_location[1] = 0;
				amstrad_ram_location[2] = 0;
				amstrad_ram_location[3] = 0;

				amstrad_ram_offset[0] =
					amstrad_ram_configs[(config_index<<2)]<<14;

				amstrad_ram_offset[1] =
					amstrad_ram_configs[(config_index<<2)+1]<<14;

				amstrad_ram_offset[2] =
					amstrad_ram_configs[(config_index<<2)+2]<<14;

				amstrad_ram_offset[3] =
					amstrad_ram_configs[(config_index<<2)+3]<<14;

				amstrad_update_mem();
			}
			break;
		}
	}

	public void amstrad_ga_reset()
	{
		for (int i=0; i<18; i++)
		{
		    amstrad_ga_write(0x040 | i);
		    amstrad_ga_write(0x054);
		}

		/* selection 0 */
		amstrad_ram_configs[0] = 0;
		amstrad_ram_configs[1] = 1;
		amstrad_ram_configs[2] = 2;
		amstrad_ram_configs[3] = 3;
		/* selection 1 */
		amstrad_ram_configs[4] = 0;
		amstrad_ram_configs[5] = 1;
			amstrad_ram_configs[6] = 2;
		amstrad_ram_configs[7] = 7;
		/* selection 2 */
		amstrad_ram_configs[8] = 4;
		amstrad_ram_configs[9] = 5;
		amstrad_ram_configs[10] = 6;
		amstrad_ram_configs[11] = 7;
		/* selection 3 */
		amstrad_ram_configs[12] = 0;
		amstrad_ram_configs[13] = 3;
		amstrad_ram_configs[14] = 2;
		amstrad_ram_configs[15] = 7;
		/* selection 4 */
		amstrad_ram_configs[16] = 0;
		amstrad_ram_configs[17] = 4;
		amstrad_ram_configs[18] = 2;
		amstrad_ram_configs[19] = 3;
		/* selection 5 */
		amstrad_ram_configs[20] = 0;
		amstrad_ram_configs[21] = 5;
		amstrad_ram_configs[22] = 2;
		amstrad_ram_configs[23] = 3;
		/* selection 6 */
		amstrad_ram_configs[24] = 0;
		amstrad_ram_configs[25] = 6;
		amstrad_ram_configs[26] = 2;
		amstrad_ram_configs[27] = 3;
		/* selection 7 */
		amstrad_ram_configs[28] = 0;
		amstrad_ram_configs[29] = 7;
		amstrad_ram_configs[30] = 2;
		amstrad_ram_configs[31] = 3;

		amstrad_ga_write(0x089);
		amstrad_ga_write(0x0c0);

	}

	/**
	 * NEC765 emulation
	 */

    /* drive */

	 private final int DRIVE_STATUS_DRIVE_PRESENT = 0x001;
	 private final int DRIVE_STATUS_DISK_INSERTED = 0x0002;


     private int drive_id_index[] = new int[2];
     private int drive_status[] = new int[2];
     private int drive_current_track[] = new int [2];

     private int drive_track_last_accessed[] = new int [2];
     private int drive_side_last_accessed[] = new int [2];

     private void drive_seek(int drive, int track)
     {
        drive_current_track[(drive & 0x01)] = track;
     }

     private int drive_get_spt(int drive, int side)
     {
      //  if (diskimage_type==0)
        //{
            return dsk_get_spt(drive_current_track[drive], side);
        //}

//        return 0;
     }

    private void drive_get_id(int drive, int side)
    {
        drive_id_index[drive]++;

        dsk_get_id(drive_current_track[drive], side,drive_id_index[drive]);
    }

    private void drive_get_id2(int drive, int side, int index)
    {
        dsk_get_id(drive_current_track[drive], side, index);
    }

    private void drive_fetch_sector(int drive, int side, int index, int buffer[])
    {
        dsk_get_sector(drive_current_track[drive], side, index, buffer);
    }

    private int fdc_id_buffer[] = new int[6];

	 	// number of bytes transfered in command phase, execution,
	// or result phase
	private int fdc_transfer_count;
	// current position in transfer data
	private int fdc_transfer_position;
	// fdc phase
	private int fdc_phase;
	// command bytes
	private int fdc_command_bytes[] = new int [12];
	// result bytes
	private int fdc_result_bytes[] = new int [12];

    private int fdc_execution_buffer[] = new int [8096];

	private int fdc_status_registers[] = new int [4];

	// main status register
	private int fdc_main_status_register;

    private int fdc_dest_track;
    private int fdc_id_index;
    private int fdc_drive;
    private int fdc_side;

    private final int FDC_PHASE_WRITE_FIRST_COMMAND_BYTE = 0;
	private final int FDC_PHASE_WRITE_COMMAND_BYTES = 1;
	private final int FDC_PHASE_EXECUTION_PHASE_WRITE = 2;
	private final int FDC_PHASE_EXECUTION_PHASE_READ = 3;
	private final int FDC_PHASE_RESULT_PHASE = 4;

	private final int FDC_MSR_DATA_REQUEST  = 0x080;
	private final int FDC_MSR_DATA_DIRECTION = 0x040;
    private final int FDC_MSR_EXECUTION_PHASE = 0x020;
    private final int FDC_MSR_BUSY = 0x010;
    private final int FDC_MSR_FDD_3_BUSY = 0x008;
    private final int FDC_MSR_FDD_2_BUSY = 0x004;
    private final int FDC_MSR_FDD_1_BUSY = 0x002;
    private final int FDC_MSR_FDD_0_BUSY = 0x001;

	// returns number of bytes required by command written
	private int fdc_get_command_size(int command)
	{
		switch (command)
		{
			// 0,1,11,14,16,18,19,20,21,22,23,24,26,27,28,30,31 invalid
			// 8: sense interrupt status
			default:
			case 0:
			case 1:
				return 1;

			// 2: read a track
			// 5: write data
			// 6: read data
			// 9: write deleted data
			// 12: read deleted data
			// 17: scan equal
			// 25: scan low or equal
			// 29: scan high or equal
			case 2:
			case 5:
			case 6:
			case 9:
			case 12:
			case 17:
			case 25:
			case 29:
				return 9;

			// 3: specify
			// 15: seek
			case 3:
			case 15:
				return 3;


			// 4: sense drive status
			// 7: recalibrate
			// 10: read id
			case 4:
			case 7:
			case 10:
				return 2;

			// 13: format a track
			case 13:
				return 6;
		}
	}


	// returns number of bytes required by command written
	private int fdc_get_result_size(int command)
	{
		switch (command)
		{
			// 8: sense interrupt status
		    case 8:
		        return 2;

			// 0,1,11,14,16,18,19,20,21,22,23,24,26,27,28,30,31 invalid
			default:
			    return 1;

			// 2: read a track
			// 13: format a track
			// 9: write deleted data
			// 12: read deleted data
			// 10: read id
			// 5: write data
			// 6: read data

			// 17: scan equal
			// 25: scan low or equal
			// 29: scan high or equal
			case 2:
			case 13:
			case 12:
			case 9:
		    case 10:
			case 5:
			case 6:
			    return 7;

			// 4: sense drive status
			case 4:
                return 1;
		}
	}

	private void fdc_idle()
	{
	    fdc_main_status_register &= ~(
	        FDC_MSR_DATA_DIRECTION |
	        FDC_MSR_EXECUTION_PHASE |
	        FDC_MSR_BUSY);

	    fdc_main_status_register |=
	        FDC_MSR_DATA_REQUEST;

		fdc_phase = FDC_PHASE_WRITE_FIRST_COMMAND_BYTE;
	}

    private final void fdc_reset()
    {
        drive_status[0] = 0;
        drive_status[1] = 0;


        fdc_idle();
    }



	private void fdc_setup_result_phase()
	{
	    fdc_main_status_register |= FDC_MSR_DATA_REQUEST | FDC_MSR_DATA_DIRECTION;
		fdc_main_status_register &= ~FDC_MSR_EXECUTION_PHASE;
		fdc_transfer_count = fdc_get_result_size(fdc_command_bytes[0] & 0x01f);
		fdc_transfer_position = 0;
		fdc_phase = FDC_PHASE_RESULT_PHASE;
	}

    private void fdc_setup_execution_phase_read(int transfer_count, int transfer_position)
	{
	    fdc_main_status_register |= FDC_MSR_DATA_REQUEST | FDC_MSR_DATA_DIRECTION;
		fdc_transfer_count = transfer_count;
		fdc_transfer_position = transfer_position;
		fdc_phase = FDC_PHASE_EXECUTION_PHASE_READ;
	}


    private void fdc_do_seek()
    {
        fdc_drive = fdc_command_bytes[1] & 0x01;

        if ((drive_status[fdc_drive] & DRIVE_STATUS_DISK_INSERTED)!=0)
        {
            drive_seek(fdc_drive,fdc_dest_track);

            fdc_status_registers[0] = 0x020 | fdc_drive;
        }
        else
        {
            fdc_status_registers[0] = 0x010 | 0x08 | 0x040;
        }
    }

    private final int fdc_get_sector_index()
    {
        int spt;

        spt = drive_get_spt(fdc_drive,fdc_side);

        if (spt!=0)
        {
            for (int i=0; i<spt; i++)
            {
                drive_get_id2(fdc_drive, fdc_side, i);

                if (
                    (fdc_result_bytes[3] == fdc_command_bytes[2]) &&
                    (fdc_result_bytes[4] == fdc_command_bytes[3]) &&
                    (fdc_result_bytes[5] == fdc_command_bytes[4]) &&
                    (fdc_result_bytes[6] == fdc_command_bytes[5]))
                    {
                        return i;
                    }
             }
        }

        return -1;
    }



    private void fdc_read_data()
    {
        int sector_index = fdc_get_sector_index();

        drive_fetch_sector(fdc_drive, fdc_side, sector_index, fdc_execution_buffer);
    
        fdc_setup_execution_phase_read((1<<(fdc_command_bytes[5]+7)), 0);
    }


	private void fdc_continue_command()
	{
		switch (fdc_command_bytes[0] & 0x01f)
		{
            // read data
            case 6:
            {
                if (fdc_command_bytes[4]==fdc_command_bytes[6])
                {
                    fdc_result_bytes[0] = fdc_status_registers[0];
                    fdc_result_bytes[1] = fdc_status_registers[1];
                    fdc_result_bytes[2] = fdc_status_registers[2];
                    fdc_result_bytes[3] = fdc_command_bytes[2];
                    fdc_result_bytes[4] = fdc_command_bytes[3];
                    fdc_result_bytes[5] = fdc_command_bytes[4];
                    fdc_result_bytes[6] = fdc_command_bytes[5];
                    
                    fdc_setup_result_phase();
                }
                else
                {
                    fdc_result_bytes[4]++;
                    
                    fdc_read_data();
                }
            }
            break;

			default:
				break;
		}
	}


	private void fdc_command_setup()
	{
	    fdc_main_status_register &= ~FDC_MSR_DATA_REQUEST;
	    fdc_main_status_register |= FDC_MSR_EXECUTION_PHASE;

		switch (fdc_command_bytes[0] & 0x01f)
		{
			// 0,1,11,14,16,18,19,20,21,22,23,24,26,27,28,30,31 invalid
			case 0:
			case 1:
			case 11:
			case 14:
			case 16:
			case 18:
			case 19:
			case 20:
			case 21:
			case 22:
			case 23:
			case 24:
			case 26:
			case 27:
			case 28:
			case 30:
			case 31:
			{
				fdc_result_bytes[0] = 0x080;

				fdc_setup_result_phase();
			}
			break;

			case 3:
			{
				// specify
				fdc_idle();

			}
			break;

			case 8:
			{
				// sense interrupt status
		        if (fdc_status_registers[0] == 0x080)
		        {
		           fdc_command_bytes[0] = 0;
		        }
                else
                {
		            fdc_result_bytes[0] = fdc_status_registers[0];
		            fdc_status_registers[0] = 0x080;
		            fdc_result_bytes[1] =  fdc_dest_track;

		        }

		        fdc_setup_result_phase();
		    }
			break;

            // recalibrate
            case 7:
            {
                fdc_dest_track = 0;

                fdc_do_seek();

                fdc_idle();
            }
            break;

            // seek
            case 15:
            {
                fdc_dest_track = fdc_command_bytes[2];

                fdc_do_seek();

                fdc_idle();
            }
            break;

            // read id
            case 10:
            {
                int spt;

                fdc_drive = fdc_command_bytes[1] & 0x01;
                fdc_side = (fdc_command_bytes[1]>>2) & 0x01;

                fdc_status_registers[0] = 0;
                fdc_status_registers[1] = 0;
                fdc_status_registers[2] = 0;

                spt = drive_get_spt(fdc_drive, fdc_side);

                if (spt!=0)
                {
                    fdc_id_index++;
                    fdc_id_index = fdc_id_index % spt;

                    drive_get_id(fdc_drive, fdc_side);
                    //drive_get_c(fdc_drive, fdc_side, fdc_id_index);
                    //drive_get_c(fdc_drive, fdc_side, fdc_id_index);
                    //drive_get_c(fdc_drive, fdc_side, fdc_id_index);
                    //drive_get_c(fdc_drive, fdc_side, fdc_id_index);

                }
                else
                {
                }

                fdc_result_bytes[0] = fdc_status_registers[0];
                fdc_result_bytes[1] = fdc_status_registers[0];
                fdc_result_bytes[2] = fdc_status_registers[0];

                fdc_setup_result_phase();
            }
            break;

            // read data
            case 6:
            {
                fdc_drive = (fdc_command_bytes[1] & 03);
                fdc_side = (fdc_command_bytes[1]>>2) & 0x01;

                fdc_status_registers[0] = 0;
                fdc_status_registers[1] = 0;
                fdc_status_registers[2] = 0;

                fdc_read_data();


            }
            break;





			default:
				break;








		}
	}

	public void fdc_write_data_register(int Data)
	{
		switch (fdc_phase)
		{
			// write the first command byte
			case FDC_PHASE_WRITE_FIRST_COMMAND_BYTE:
			{
				// store command byte
				fdc_command_bytes[0] = Data;

                fdc_main_status_register |= FDC_MSR_BUSY;

				// number of bytes remaining.
				fdc_transfer_count = fdc_get_command_size(Data & 0x01f);
				fdc_transfer_count--;
				fdc_transfer_position=1;

				// more command bytes required?
				if (fdc_transfer_count!=0)
				{
					// yes.
	    			fdc_phase = FDC_PHASE_WRITE_COMMAND_BYTES;
				}
				else
				{
					// no.
					fdc_command_setup();
				}

			}
			break;

			// write remaining command bytes
			case FDC_PHASE_WRITE_COMMAND_BYTES:
			{
				fdc_command_bytes[fdc_transfer_position] = Data;
				fdc_transfer_count--;
				fdc_transfer_position++;

				if (fdc_transfer_count==0)
				{
					// command setup
					fdc_command_setup();
				}
			}
			break;

			default:
				break;
		}
	}

	public int fdc_read_data_register()
	{
		switch (fdc_phase)
		{
			case FDC_PHASE_EXECUTION_PHASE_READ:
			{
			    int Data;
			    
			    Data = fdc_execution_buffer[fdc_transfer_position];
			    fdc_transfer_count--;
			    fdc_transfer_position++;

			    if (fdc_transfer_count==0)
			    {
			        fdc_continue_command();
                }
                
                return Data;
    		}

			case FDC_PHASE_RESULT_PHASE:
			{
				int Data;

				// return result data.
				Data = fdc_result_bytes[fdc_transfer_position];
				fdc_transfer_count--;
				fdc_transfer_position++;

				if (fdc_transfer_count==0)
				{
					fdc_idle();
				}

				return Data;
			}

			default:
				break;
		}
		return 0x0ff;
	}

	public int	fdc_read_main_status_register()
	{
		return fdc_main_status_register;
	}


	/**
	 * Z80 hardware interface
	 */
	public int inb( int port )
	{
	    int Data=0x0ff;

        if ((port & 0x04000)==0)
        {
            int            Index;

            Index = (port & 0x0300)>>8;

			switch (Index)
			{
				case 3:
				{
					Data = crtc_read_reg();
				}
				break;
			}
        }

        if ((port & 0x0800)==0)
        {
            int            Index;

            Index = (port & 0x0300)>>8;

			switch (Index)
			{
				case 0:
				{
					Data = ppi_read_port_a();
				}
				break;

				case 1:
				{
					Data = ppi_read_port_b();
				}
				break;

				case 2:
				{
					Data = ppi_read_port_c();
				}
				break;
			}
        }

        if ((port & 0x0480)==0)
        {
            int            Index;

            Index = ((port & 0x0100)>>(8-1)) | (port & 0x01);

			switch (Index)
			{
				case 2:
				{
					Data = fdc_read_main_status_register();
				}
				break;

				case 3:
				{
					Data = fdc_read_data_register();
				}
				break;

				default:
					break;
			}
		}


		return(Data);
	}

	public void outb( int port, int outByte, int tstates )
	{
		if ((port & 0x0c000)==0x04000)
		{
			amstrad_ga_write(outByte);
		}

        if ((port & 0x04000)==0)
        {
            int            Index;

            Index = (port>>8) & 0x03;

			switch (Index)
			{
				case 0:
				{
					crtc_write_regindex(outByte);
				}
				break;

				case 1:
				{
					crtc_write_reg(outByte);
				}
				break;

				default:
					break;
			}
		}

		if ((port & 0x02000)==0)
		{
			amstrad_rom_select(outByte);
		}

		if ((port & 0x0800)==0)
        {
			int            Index;

            Index = (port & 0x0300)>>8;

			switch (Index)
			{
				case 0:
				{
					ppi_write_port_a(outByte);
				}
				break;

				case 1:
				{
					ppi_write_port_b(outByte);
				}
				break;

				case 2:
				{
					ppi_write_port_c(outByte);
				}
				break;

				case 3:
				{
					ppi_write_control(outByte);
				}
				break;
			}
	    }

        if ((port & 0x0480)==0)
        {
            int            Index;

            Index = ((port & 0x0100)>>(8-1)) | (port & 0x01);

			switch (Index)
			{
				case 3:
				{
					fdc_write_data_register(outByte);
				}
				break;

				default:
					break;
			}
        }
   	}


	/** 128k ram **/
	public int	amstrad_memory[] = new int [(128*1024)];
	/** 32k rom (os + basic) **/
	public int	amstrad_os_rom[] = new int [32768];
	/** 16k rom (amsdos **/
	public int  amstrad_amsdos_rom[] = new int [16384];

	/** Ram/Rom paging **/
	/** arrays below are used for memory paging. Memory is
	split into 16k blocks **/

	/** location values:

	0 = ram
	1 = os rom
	2 = amsdos rom

     **/

	/** location defines what memory above to read from **/
	public int amstrad_read_location[] = new int[4];
	/** offset in memory to read **/
	public int	amstrad_read_offset[] = new int[4];

	/** location defines what memory above to write to **/
	public int	amstrad_write_location[] = new int[4];
	/** offset in memory to read **/
	public int	amstrad_write_offset[] = new int[4];

	/** Byte access */
	public void pokeb( int addr, int newByte )
	{
		int Bank;
		int offset;

		Bank = (addr>>14) & 0x03;

		offset = amstrad_write_offset[Bank];

		switch (amstrad_write_location[Bank])
		{
			/* write to ram */
			case 0:
			{
				amstrad_memory[offset+(addr & 0x03fff)] = newByte;
			}
			break;
		}

		if (Bank==((crtc_registers[12]>>4) & 0x03))
		{
		    plot((addr & 0x03fff), newByte);
		}
	}

	public int peekb(int addr)
	{
		int Bank;
		int offset;

		Bank = (addr>>14) & 0x03;

		offset = amstrad_read_offset[Bank];

		switch (amstrad_read_location[Bank])
		{
			/* read from ram */
			case 0:
			{
				return amstrad_memory[offset+(addr & 0x03fff)];
			}

			/* read from os rom */
			case 1:
			{
				return amstrad_os_rom[offset+(addr & 0x03fff)];
			}

			/* read from amsdos rom */
			case 2:
			{
				return amstrad_amsdos_rom[offset+(addr & 0x03fff)];
			}
		}

		return 0x0ff;
	}


	// Word access
	public void pokew( int addr, int word )
	{
		pokeb(addr, word & 0x0ff);
		pokeb(addr+1, word>>8);
	}

	public int peekw(int addr)
	{
		int word;

		word = peekb(addr);
		word |= (peekb(addr+1)<<8);

		return word;
	}

	public void	amstrad_reset()
	{
	    /* reset GA */
		amstrad_ga_reset();

		/* reset keyboard inputs */
		amstrad_reset_keyboard();

	    fdc_reset();

	    InitialiseChanges();
	}


	/** Since execute runs as a tight loop, some Java VM implementations
	 *  don't allow any other threads to get a look in. This give the
	 *  GUI time to update. If anyone has a better solution please
	 *  email us at mailto:spectrum@odie.demon.co.uk
	 */
	public  int     sleepHack = 0;
	public  int     refreshRate = 1;  // refresh every 'n' interrupts

	private int     interruptCounter = 0;
	private boolean resetAtNextInterrupt = false;
	private boolean pauseAtNextInterrupt = false;
	private boolean refreshNextInterrupt = true;
	private boolean loadFromURLFieldNextInterrupt = false;

	public  Thread  pausedThread = null;
	public  long    timeOfLastInterrupt = 0;
	private long    timeOfLastSample = 0;


	public final int interrupt()
	{
        paintBuffer();
    	interruptCounter++;

        if ((interruptCounter % 6)==0)
	    {
    		if ( pauseAtNextInterrupt ) {
    			urlField.show();

    			pausedThread = Thread.currentThread();
    			while ( pauseAtNextInterrupt ) {
    				showMessage( "Adam Davidson & Andrew Pollard" );
    				if ( refreshNextInterrupt ) {
    					refreshNextInterrupt = false;
//    					oldBorder = -1;
    					paintBuffer();
    				}
    				if ( loadFromURLFieldNextInterrupt ) {
    					loadFromURLFieldNextInterrupt = false;
    //					loadFromURLField();
    				}
    				else {
    					try { Thread.sleep( 500 ); } catch ( Exception ignored ) {}
    				}
    			}
    			pausedThread = null;

    			urlField.hide();
    		}

    		if ( refreshNextInterrupt ) {
    			refreshNextInterrupt = false;
//    			oldBorder = -1;
    			paintBuffer();
    		}

    		if ( resetAtNextInterrupt ) {
    			resetAtNextInterrupt = false;
    			reset();
    		}


    //		// Characters flash every 1/2 a second
    //		if ( (interruptCounter % 25) == 0 ) {
    //			refreshFlashChars();
    //		}

    		// Update speed indicator every 2 seconds of 'Spectrum time'
    		if ( (interruptCounter % 100) == 0 ) {
    			refreshSpeed();
    		}

    		// Refresh every interrupt by default
    		if ( (interruptCounter % 1)==0) {   //refreshRate) == 0 ) {
    			screenPaint();
    			paintBuffer();
    		}

    		timeOfLastInterrupt = System.currentTimeMillis();

    		// Trying to slow to 100%, browsers resolution on the system
    		// time is not accurate enough to check every interrurpt. So
    		// we check every 4 interrupts.
    		if ( (interruptCounter % 4) == 0 ) {
    			long durOfLastInterrupt = timeOfLastInterrupt - timeOfLastSample;
    			timeOfLastSample = timeOfLastInterrupt;
    			if ( !runAtFullSpeed && (durOfLastInterrupt < 40) ) {
    				try { Thread.sleep( 50 - durOfLastInterrupt ); }
    				catch ( Exception ignored ) {}
    			}
    		}

    		// This was put in to handle Netscape 2 which was prone to
    		// locking up if one thread never gave up its timeslice.
    		if ( sleepHack > 0 ) {
    			try { Thread.sleep( sleepHack ); }
    			catch ( Exception ignored ) {}
    		}
    	}

		return super.interrupt();
	}

	public void pauseOrResume() {
		// Pause
		if ( pausedThread == null ) {
			pauseAtNextInterrupt = true;
		}
		// Resume
		else {
			pauseAtNextInterrupt = false;

		}
	}

	public void repaint() 
	{
		refreshNextInterrupt = true;
		paintBuffer();
	}

	public void reset() 
	{
		super.reset();

		amstrad_reset();

		graphics_decode_setup();
	}

	/**
	 * Screen stuff
	 */
    /* these are used to speed up screen updates */

    /* the crtc usually accesses 16k of ram at a time. e.g. in most
    games - which we are only really interested in running with this
    applet. this keeps track of changes and the last byte written */
    private int amstrad_scr_bytes[]= new int [16384];

    private int scr_bytes_linked_list[] = new int [16384];
    private int scr_bytes_used[] = new int [16384];

    private int first = -1;

    private void    InitialiseChanges()
    {
        for (int i=0; i<16384; i++)
        {
            // mark these as not used yet.
            scr_bytes_used[i] = 0;
        }

        // null out start of list
        first = -1;
    }

	private void plot_update(int offset)
	{
	    // have we already put this pixel into the list?
	    if (scr_bytes_used[offset]==0)
	    {
	        // no

            // it is now...
            scr_bytes_used[offset] = -1;

    	    // has list started?
    	    if (first==-1)
    	    {
    	        // no items in list

    	        // the first item is the offset for this byte
    	        first = offset;

    	        // mark this as the last in the list.
    	        scr_bytes_linked_list[offset] = -1;
    	    }
    	    else
    	    {
    	        // set this offset to point to offset which was the
    	        // previous start of the list
    	        scr_bytes_linked_list[offset] = first;
    	        first = offset;
    	    }
    	}
	}

    private final void plot( int offset, int newByte )
    {
        int oldByte;

        // get old byte
        oldByte = amstrad_scr_bytes[offset];

        // compare to new...
        if (newByte!=oldByte)
        {
            // different. (We want to update this pixel)
            amstrad_scr_bytes[offset] = newByte;

			plot_update(offset);
	    }
    }

    private final void amstrad_render_mode_0(int plot_offset, int byte0, int byte1)
    {
        int offset;
        int Pen, PackedRGB;

        offset = plot_offset;

        Pen = amstrad_mode0_gfx_decode[byte0];
        PackedRGB = amstrad_ga_render_colour[Pen];

        pix[offset] = PackedRGB;
        offset++;
        pix[offset] = PackedRGB;
        offset++;

        byte0 = (byte0<<1) & 0x0ff;

        Pen = amstrad_mode0_gfx_decode[byte0];
        PackedRGB = amstrad_ga_render_colour[Pen];

        pix[offset] = PackedRGB;
        offset++;
        pix[offset] = PackedRGB;
        offset++;

        Pen = amstrad_mode0_gfx_decode[byte1];
        PackedRGB = amstrad_ga_render_colour[Pen];

        pix[offset] = PackedRGB;
        offset++;
        pix[offset] = PackedRGB;
        offset++;

        byte1 = (byte1<<1) & 0x0ff;

        Pen = amstrad_mode0_gfx_decode[byte1];
        PackedRGB = amstrad_ga_render_colour[Pen];

        pix[offset] = PackedRGB;
        offset++;
        pix[offset] = PackedRGB;
        offset++;
    }

    private final void amstrad_render_mode_1(int plot_offset, int byte0, int byte1)
    {
        int Pen, PackedRGB;
        int offset;

        offset = plot_offset;

        Pen = amstrad_mode1_gfx_decode[byte0];
        PackedRGB = amstrad_ga_render_colour[Pen];

        pix[offset] = PackedRGB;
        offset++;

        byte0 = (byte0<<1) & 0x0ff;

        Pen = amstrad_mode1_gfx_decode[byte0];
        PackedRGB = amstrad_ga_render_colour[Pen];

        pix[offset] = PackedRGB;
        offset++;

        byte0 = (byte0<<1) & 0x0ff;

        Pen = amstrad_mode1_gfx_decode[byte0];
        PackedRGB = amstrad_ga_render_colour[Pen];

        pix[offset] = PackedRGB;
        offset++;

        byte0 = (byte0<<1) & 0x0ff;

        Pen = amstrad_mode1_gfx_decode[byte0];
        PackedRGB = amstrad_ga_render_colour[Pen];

        pix[offset] = PackedRGB;
        offset++;

        Pen = amstrad_mode1_gfx_decode[byte1];
        PackedRGB = amstrad_ga_render_colour[Pen];

        pix[offset] = PackedRGB;
        offset++;

        byte1 = (byte1<<1) & 0x0ff;

        Pen = amstrad_mode1_gfx_decode[byte1];
        PackedRGB = amstrad_ga_render_colour[Pen];

        pix[offset] = PackedRGB;
        offset++;

        byte1 = (byte1<<1) & 0x0ff;

        Pen = amstrad_mode1_gfx_decode[byte1];
        PackedRGB = amstrad_ga_render_colour[Pen];

        pix[offset] = PackedRGB;
        offset++;

        byte1 = (byte1<<1) & 0x0ff;

        Pen = amstrad_mode1_gfx_decode[byte1];
        PackedRGB = amstrad_ga_render_colour[Pen];

        pix[offset] = PackedRGB;
        offset++;
    }

    private final void amstrad_render_mode_2(int plot_offset, int byte0, int byte1)
    {
        int Pen0, Pen1,Pen;
        int PixelData;
        int offset;

        offset = plot_offset;

        Pen0 = amstrad_ga_render_colour[0];
        Pen1 = amstrad_ga_render_colour[1];

        PixelData = (byte0<<8) | byte1;

        for (int p=0; p<8; p++)
        {
            if ((PixelData & 0x08000)!=0)
            {
                Pen = Pen1;
            }
            else
            {
                Pen = Pen0;
            }

            pix[offset] = Pen;
            offset++;

            PixelData = PixelData<<2;
        }
    }

    private final void amstrad_render_mode_x(int plot_offset, int Addr)
    {
        int byte0, byte1;

        /* get bytes */
        byte0 = amstrad_memory[Addr & 0x0ffff] & 0x0ff;
    	byte1 = amstrad_memory[(Addr+1) & 0x0ffff] & 0x0ff;

		switch (amstrad_ga_rom_mode & 0x03)
        {
            case 0:
            {
                amstrad_render_mode_0(plot_offset, byte0, byte1);
            }
            break;

            case 1:
            {
                amstrad_render_mode_1(plot_offset, byte0, byte1);
            }
            break;

            case 2:
            {
                amstrad_render_mode_2(plot_offset, byte0, byte1);
            }
            break;

            case 3:
                break;

            default:
                break;
        }

        MemImage.newPixels(plot_offset % W,plot_offset/W,8,1, false);
    }




    /* update changes stored in the linked list */
    public final void update_changes()
    {
            int cnt = 0;
            int current = first;

            // list is empty.
            if (current<0)
                return;

            // list has items.

            do
            {
                // plot this pixel

                int MA = ((crtc_registers[12] & 0x03)<<8) | (crtc_registers[13] & 0x0ff);

                // calc RA, MA
                MA = (((current>>1) & 0x03ff) - MA) & 0x03ff;
                int RA = (current>>11) & 0x07;
                int X,line;

                // x pos
                X = (MA % crtc_registers[1])*8;
                // y pos
                line = (MA/crtc_registers[1])*(crtc_registers[9]+1);
                line+= RA;

                int Addr;
    			int byte0;
    			int byte1;

    			Addr = (current & 0x3ffe) | ((crtc_registers[12] & 0x030)<<(2+8));


    			int plot_offset = (X+(line*W));

    			amstrad_render_mode_x(plot_offset, Addr);
    			X+=8;

                cnt++;

                // mark this as not used.
                scr_bytes_used[current] = 0;

                // get next
                current = scr_bytes_linked_list[current];

                if (cnt>=16384)
                {
                    current = -1;
                }

            }
            while (current!=-1);

            first = -1;

    }






	private static final Color amstrad_colours []=
	{
		new Color(128,128,128),
		new Color(128,128,128),
		new Color(0,255,128),
		new Color(255,255,128),
		new Color(0,0,128),
		new Color(255, 0, 128),
		new Color(0,128,128),
		new Color(255,128,128),
		new Color(255,0,128),
		new Color(255,255,128),
		new Color(255,255,0),
		new Color(255,255,255),
		new Color(255,0,0),
		new Color(255,0,255),
		new Color(255,128,0),
		new Color(255,128,255),
		new Color(0,0,128),
		new Color(0,255,128),
		new Color(0,255,0),
		new Color(0,255,255),
		new Color(0,0,0),
		new Color(0,0,255),
		new Color(0,128,0),
		new Color(0,128,255),
		new Color(128,0,128),
		new Color(128,255,128),
		new Color(128,255,0),
		new Color(128,255,255),
		new Color(128,0,0),
		new Color(128,0,255),
		new Color(128,128,0),
		new Color(128,128,255)
	};

	/** first screen line in linked list to be redrawn */
	//private int first = -1;
	/** first attribute in linked list to be redrawn */
	//private int FIRST = -1;
	//private final int last[] = new int[ (nPixelsHigh+nCharsHigh)*nCharsWide ];
	//private final int next[] = new int[ (nPixelsHigh+nCharsHigh)*nCharsWide ];

	//public int newBorder = 7;  // White border on startup
	//public int oldBorder = -1; // -1 mean update screen

    public boolean refreshscreen = false;
	public boolean refreshcolours = false;

	public long oldTime = 0;
	public int oldSpeed = -1; // -1 mean update progressBar
	public int newSpeed = 0;
	public boolean showStats = true;
	public String statsMessage = null;
	private static final String cancelMessage = new String( "Click here at any time to cancel sleep" );

	//private boolean flashInvert = false;


	public final void refreshSpeed() {
		long newTime = timeOfLastInterrupt;

		if ( oldTime != 0 ) {
			newSpeed = (int) (200000.0 / (newTime - oldTime));
		}

		oldTime = newTime;
		if ( (statsMessage != null) && (sleepHack > 0) && (statsMessage != cancelMessage) ) {
			showMessage( cancelMessage );
		}
		else {
			showMessage( null );
		}
	}


//	public final void refreshWholeScreen() {
//		showMessage( "Drawing Off-Screen Buffer" );
//
//		for ( int i = 0; i < firstAttr; i++ ) {
//			next[ i ] = i-1;
//			last[ i ] = (~mem[ i+16384 ]) & 0xff;
//		}
//
//		for ( int i = firstAttr; i < lastAttr; i++ ) {
//			next[ i ] = -1;
//			last[ i ] = mem[ i+16384 ];
//		}
//
//		first = firstAttr - 1;
//		FIRST = -1;
//
//		oldBorder = -1;
//		oldSpeed  = -1;
//	}
//
//	private final void plot( int addr, int newByte ) {
//		int     offset = addr - 16384;
//
//		if ( next[ offset ] == -1 ) {
//			if ( offset < firstAttr ) {
//				next[ offset ] = first;
//				first = offset;
//			}
//			else {
//				next[ offset ] = FIRST;
//				FIRST = offset;
//			}
//		}
//	}
//
//	public final void borderPaint() {
//		if ( oldBorder == newBorder ) {
//			return;
//		}
//		oldBorder = newBorder;
//
//		if ( borderWidth == 0 ) {
//			return;
//		}
//
//		parentGraphics.setColor( brightColors[ newBorder ] );
//		parentGraphics.fillRect( 0, 0,
//			(nPixelsWide*pixelScale) + borderWidth*2,
//			(nPixelsHigh*pixelScale) + borderWidth*2 );
//	}

	private static final String fullSpeed = "Full Speed: ";
	private static final String slowSpeed = "Slow Speed: ";
 	public boolean	runAtFullSpeed = true;

	private final void toggleSpeed() {
		runAtFullSpeed = !runAtFullSpeed;
		showMessage( statsMessage );
	}

	public void showMessage( String m ) {
		statsMessage = m;
		oldSpeed = -1; // Force update of progressBar
		statsPaint();
	}

	public final void statsPaint() {
		if ( oldSpeed == newSpeed ) {
			return;
		}
		oldSpeed = newSpeed;

		if ( (!showStats) || (borderWidth < 10) ) {
			return;
		}

		String stats = statsMessage;
		if ( stats == null ) {
			String	speedString = runAtFullSpeed ? fullSpeed : slowSpeed;

			if ( newSpeed > 0 ) {
				stats = speedString + String.valueOf( newSpeed ) + "%";
			}
			else {
				stats = "Speed: calculating";
			}
			if ( sleepHack > 0 ) {
				stats = stats + ", Sleep: " + String.valueOf( sleepHack );
			}
		}
//		progressBar.setText( stats );
//		progressBar.show();
	}

//	public static synchronized Image getImage( Component comp, int attr, int pattern ) {
//		try {
//			return tryGetImage( comp, attr, pattern );
//		}
//		catch ( OutOfMemoryError e ) {
//			imageMap   = null;
//			patternMap = null;
//
//			System.gc();
//
//			patternMap = new Hashtable();
//			imageMap   = new Image[ 1<<11 ];
//
//			return tryGetImage( comp, attr, pattern );
//		}
//	}


	public final void screenPaint()
	{


	    if (refreshscreen!=true)
	   {
	        update_changes();
	       return;
	    }

        for (int i=0; i<16384; i++)
        {
            scr_bytes_used[i] = 0;
        }

	    refreshscreen = false;
        first = -1;

		int MA;
		int CurMA;

		int line = 0;
		int X;

		MA = ((crtc_registers[12] & 0x0ff)<<8) | (crtc_registers[13] & 0x0ff);

        CurMA = MA;
		for (int v=0; v<(crtc_registers[6]); v++)
		{
			for (int r=0; r<(crtc_registers[9]&0x01f)+1; r++)
			{
			    X = 0;

				CurMA = MA;
				for (int h=0; h<(crtc_registers[1] & 0x03f); h++)
				{
					int Addr;

					Addr = ((CurMA & 0x03000)<<2) |
							((r & 7)<<11) | ((CurMA & 0x03ff)<<1);

	                int plot_offset = (X+(line*W));

	                amstrad_render_mode_x(plot_offset, Addr);
	                X+=8;

					CurMA++;
				}

				line++;

			}
			MA = CurMA;
		}

	}

	public void paintBuffer() {
		//MemImage.newPixels();   //(0,0,800,600);
		//canvasGraphics.drawImage(MemImage, 0,0, null);
		//parent.prepareimage(bufferImage, MemImage);

		//canvasGraphics.setPaintMode();
		canvasGraphics.drawImage( bufferImage,0, 0, null );
 		//borderPaint();

	}

	/** Process events from UI */
	public boolean handleEvent( Event e ) {
//		if ( e.target == progressBar ) {
//			if ( e.id == Event.MOUSE_DOWN ) {
//				if ( sleepHack > 0 ) {
//					sleepHack = 0;
//					showMessage( "Sleep Cancelled" );
//				}
//				else {
//					toggleSpeed();
//				}
//				canvas.requestFocus();
//				return true;
//			}
//			return false;
//		}

//		if ( e.target == urlField ) {
//			if ( e.id == Event.ACTION_EVENT ) {
//				loadFromURLFieldNextInterrupt = true;
//				return true;
//			}
//			return false;
//		}

		switch ( e.id ) {
		case Event.MOUSE_DOWN:
			canvas.requestFocus();
			return true;
		case Event.KEY_ACTION:
		case Event.KEY_PRESS:
			return doKey( true, e.key, e.modifiers );
		case Event.KEY_ACTION_RELEASE:
		case Event.KEY_RELEASE:
			return doKey( false, e.key, e.modifiers );
		case Event.GOT_FOCUS:
		case Event.LOST_FOCUS:
			amstrad_reset_keyboard();
			return true;
		}

		return false;
	}

	/** Handle Keyboard */

	/** Key IDs **/
    private final int CPC_KEY_CURSOR_UP = 0;
    private final int CPC_KEY_CURSOR_RIGHT = 1;
    private final int CPC_KEY_CURSOR_DOWN = 2;
    private final int CPC_KEY_F9 = 3;
    private final int CPC_KEY_F6 = 4;
    private final int CPC_KEY_F3 = 5;
    private final int CPC_KEY_SMALL_ENTER = 6;
    private final int CPC_KEY_FDOT = 7;
    private final int CPC_KEY_CURSOR_LEFT = 8;
    private final int CPC_KEY_COPY = 9;
    private final int CPC_KEY_F7 = 10;
    private final int CPC_KEY_F8 = 11;
    private final int CPC_KEY_F5 = 12;
    private final int CPC_KEY_F1 = 13;
    private final int CPC_KEY_F2 = 14;
    private final int CPC_KEY_F0 = 15;
    private final int CPC_KEY_CLR = 16;
    private final int CPC_KEY_OPEN_BRACE = 17;
    private final int CPC_KEY_RETURN = 18;
    private final int CPC_KEY_CLOSE_BRACE = 19;
    private final int CPC_KEY_F4 = 20;
	private final int CPC_KEY_SHIFT = 21;
	private final int CPC_KEY_BACK_SLASH = 22;
	private final int CPC_KEY_CTRL = 23;
	private final int CPC_KEY_POUND = 24;
	private final int CPC_KEY_EQUALS = 25;
	private final int CPC_KEY_AT = 26;
	private final int CPC_KEY_P = 27;
	private final int CPC_KEY_SEMICOLON = 28;
	private final int CPC_KEY_COLON = 29;
	private final int CPC_KEY_FWD_SLASH = 30;
	private final int CPC_KEY_DOT = 31;
	private final int CPC_KEY_0 = 32;
	private final int CPC_KEY_9 = 33;
	private final int CPC_KEY_O = 34;
	private final int CPC_KEY_I = 35;
	private final int CPC_KEY_L = 36;
	private final int CPC_KEY_K = 37;
	private final int CPC_KEY_M = 38;
	private final int CPC_KEY_COMMA = 39;
	private final int CPC_KEY_8 = 40;
	private final int CPC_KEY_7 = 41;
	private final int CPC_KEY_U = 42;
	private final int CPC_KEY_Y = 43;
	private final int CPC_KEY_H = 44;
	private final int CPC_KEY_J = 45;
	private final int CPC_KEY_N = 46;
	private final int CPC_KEY_SPACE = 47;
	private final int CPC_KEY_6 = 48;
	private final int CPC_KEY_5 = 49;
	private final int CPC_KEY_R = 50;
	private final int CPC_KEY_T = 51;
	private final int CPC_KEY_G = 52;
	private final int CPC_KEY_F = 53;
	private final int CPC_KEY_B = 54;
	private final int CPC_KEY_V = 55;
	private final int CPC_KEY_4 = 56;
	private final int CPC_KEY_3 = 57;
	private final int CPC_KEY_E = 58;
	private final int CPC_KEY_W = 59;
	private final int CPC_KEY_S = 60;
	private final int CPC_KEY_D = 61;
	private final int CPC_KEY_C = 62;
	private final int CPC_KEY_X = 63;
	private final int CPC_KEY_1 = 64;
	private final int CPC_KEY_2 = 65;
	private final int CPC_KEY_ESC = 66;
	private final int CPC_KEY_Q = 67;
	private final int CPC_KEY_TAB = 68;
	private final int CPC_KEY_A = 69;
	private final int CPC_KEY_CAPS_LOCK = 70;
	private final int CPC_KEY_Z = 71;
	private final int CPC_KEY_JOY_UP = 72;
	private final int CPC_KEY_JOY_DOWN = 73;
	private final int CPC_KEY_JOY_LEFT = 74;
	private final int CPC_KEY_JOY_RIGHT = 75;
	private final int CPC_KEY_JOY_FIRE1 = 76;
	private final int CPC_KEY_JOY_FIRE2 = 77;
	private final int CPC_KEY_JOY_FIRE3 = 78;
	private final int CPC_KEY_DEL = 79;
	  /** keyindex: low 3 bits defines the bit index for key, upper bits contain keyboard
    line index */

    /* if a key is pressed, the bit is 0 */
    private final void KeySet(int KeyID, boolean down)
    {
        int Bit = 1<<(KeyID & 7);
        int KeyboardLine = (KeyID>>3) & 0x0f;

        if (down)
        {
            /* pressed */
            amstrad_keyboard[KeyboardLine] &= ~Bit;
        }
        else
        {
            /* released */
            amstrad_keyboard[KeyboardLine] |= Bit;
        }
    }

	public final boolean doKey( boolean down, int ascii, int mods ) 
	{
		boolean    CAPS = ((mods & Event.CTRL_MASK) != 0);
		boolean   SHIFT = ((mods & Event.SHIFT_MASK) != 0);

        KeySet(CPC_KEY_SHIFT, SHIFT);
        
		switch ( ascii ) {
		case 'a':    KeySet(CPC_KEY_A, down);    break;
		case 'b':    KeySet(CPC_KEY_B, down);    break;
		case 'c':    KeySet(CPC_KEY_C, down);    break;
		case 'd':    KeySet(CPC_KEY_D, down);    break;
		case 'e':    KeySet(CPC_KEY_E, down);    break;
		case 'f':    KeySet(CPC_KEY_F, down);    break;
		case 'g':    KeySet(CPC_KEY_G, down);    break;
		case 'h':    KeySet(CPC_KEY_H, down);    break;
		case 'i':    KeySet(CPC_KEY_I, down);    break;
		case 'j':    KeySet(CPC_KEY_J, down);    break;
		case 'k':    KeySet(CPC_KEY_K, down);    break;
		case 'l':    KeySet(CPC_KEY_L, down);    break;
		case 'm':    KeySet(CPC_KEY_M, down);    break;
		case 'n':    KeySet(CPC_KEY_N, down);    break;
		case 'o':    KeySet(CPC_KEY_O, down);    break;
		case 'p':    KeySet(CPC_KEY_P, down);    break;
		case 'q':    KeySet(CPC_KEY_Q, down);    break;
		case 'r':    KeySet(CPC_KEY_R, down);    break;
		case 's':    KeySet(CPC_KEY_S, down);    break;
		case 't':    KeySet(CPC_KEY_T, down);    break;
		case 'u':    KeySet(CPC_KEY_U, down);    break;
		case 'v':    KeySet(CPC_KEY_V, down);    break;
		case 'w':    KeySet(CPC_KEY_W, down);    break;
		case 'x':    KeySet(CPC_KEY_X, down);    break;
		case 'y':    KeySet(CPC_KEY_Y, down);    break;
		case 'z':    KeySet(CPC_KEY_Z, down);    break;

		case '0':    KeySet(CPC_KEY_0, down);    break;
		case '1':    KeySet(CPC_KEY_1, down);    break;
		case '2':    KeySet(CPC_KEY_2, down);    break;
		case '3':    KeySet(CPC_KEY_3, down);    break;
		case '4':    KeySet(CPC_KEY_4, down);    break;
		case '5':    KeySet(CPC_KEY_5, down);    break;
		case '6':    KeySet(CPC_KEY_6, down);    break;
		case '7':    KeySet(CPC_KEY_7, down);    break;
		case '8':    KeySet(CPC_KEY_8, down);    break;
		case '9':    KeySet(CPC_KEY_9, down);    break;

		case ' ':    KeySet(CPC_KEY_SPACE, down);  break;

		case 'A':    CAPS = true;   KeySet(CPC_KEY_A, down);   break;
		case 'B':    CAPS = true;   KeySet(CPC_KEY_B, down);   break;
		case 'C':    CAPS = true;   KeySet(CPC_KEY_C, down);   break;
		case 'D':    CAPS = true;   KeySet(CPC_KEY_D, down);   break;
		case 'E':    CAPS = true;   KeySet(CPC_KEY_E, down);   break;
		case 'F':    CAPS = true;   KeySet(CPC_KEY_F, down);   break;
		case 'G':    CAPS = true;   KeySet(CPC_KEY_G, down);   break;
		case 'H':    CAPS = true;   KeySet(CPC_KEY_H, down);   break;
		case 'I':    CAPS = true;   KeySet(CPC_KEY_I, down);   break;
		case 'J':    CAPS = true;    KeySet(CPC_KEY_J, down);   break;
		case 'K':    CAPS = true;   KeySet(CPC_KEY_K, down);   break;
		case 'L':    CAPS = true;   KeySet(CPC_KEY_L, down);   break;
		case 'M':    CAPS = true;   KeySet(CPC_KEY_M, down);   break;
		case 'N':    CAPS = true;   KeySet(CPC_KEY_N, down);   break;
		case 'O':    CAPS = true;   KeySet(CPC_KEY_O, down);   break;
		case 'P':    CAPS = true;   KeySet(CPC_KEY_P, down);   break;
		case 'Q':    CAPS = true;   KeySet(CPC_KEY_Q, down);   break;
		case 'R':    CAPS = true;   KeySet(CPC_KEY_R, down);   break;
		case 'S':    CAPS = true;   KeySet(CPC_KEY_S, down);   break;
		case 'T':    CAPS = true;   KeySet(CPC_KEY_T, down);   break;
		case 'U':    CAPS = true;   KeySet(CPC_KEY_U, down);   break;
		case 'V':    CAPS = true;   KeySet(CPC_KEY_V, down);   break;
		case 'W':    CAPS = true;   KeySet(CPC_KEY_W, down);   break;
		case 'X':    CAPS = true;   KeySet(CPC_KEY_X, down);   break;
		case 'Y':    CAPS = true;   KeySet(CPC_KEY_Y, down);   break;
		case 'Z':    CAPS = true;   KeySet(CPC_KEY_Z, down);   break;

		case ',':    KeySet(CPC_KEY_COMMA, down);  break;
		case '\n':
		case '\r':   KeySet(CPC_KEY_RETURN,down);    break;

		case '\b':
		case 127:    KeySet(CPC_KEY_DEL, down);  break;

		case Event.F1: KeySet(CPC_KEY_F1, down); break;
		case Event.F2: KeySet(CPC_KEY_F2, down); break;
		case Event.F3: KeySet(CPC_KEY_F3, down); break;
		case Event.F4: KeySet(CPC_KEY_F4, down); break;
		case Event.F5: KeySet(CPC_KEY_F5, down); break;
		case Event.F6: KeySet(CPC_KEY_F6, down); break;
		case Event.F7: KeySet(CPC_KEY_F7, down); break;
		case Event.F8: KeySet(CPC_KEY_F8, down); break;
		case Event.F9: KeySet(CPC_KEY_F9, down); break;

		case Event.LEFT:    /*CAPS = SHIFT;*/ KeySet(CPC_KEY_CURSOR_LEFT,down);       break;
		case Event.DOWN:    /*CAPS = SHIFT;*/ KeySet(CPC_KEY_CURSOR_DOWN,down);   break;
		case Event.UP:      /*CAPS = SHIFT;*/ KeySet(CPC_KEY_CURSOR_UP, down);    break;
		case Event.RIGHT:   /*CAPS = SHIFT;*/ KeySet(CPC_KEY_CURSOR_RIGHT,down );    break;

		case Event.END: {
				if ( down ) {
					resetAtNextInterrupt = true;
				}
				break;
			}
		case '\033': // ESC
		case Event.HOME: {
				if ( down ) {
					pauseOrResume();
				}
				break;
			}

		default:
			return false;
		}

		return true;
	}

	public void loadOSROM( String name, InputStream is ) throws Exception {
//		startProgress( "Loading " + name, 32768 );

		readBytes( is, amstrad_os_rom, 0, 32768 );
	}

	public void loadAMSDOSROM( String name, InputStream is ) throws Exception {
//		startProgress( "Loading " + name, 16384 );

		readBytes( is, amstrad_amsdos_rom, 0, 16384);

	    // patch amsdos rom to remove pauses etc.

	    amstrad_amsdos_rom[0x07e0] = 0x0c9;
	    amstrad_amsdos_rom[0x07d0] = 0;
	    amstrad_amsdos_rom[0x07d1] = 0;
	    amstrad_amsdos_rom[0x0931] = 0;
	    amstrad_amsdos_rom[0x0932] = 0;
	    amstrad_amsdos_rom[0x0933] = 0;
	    amstrad_amsdos_rom[0x0934] = 0;
	    amstrad_amsdos_rom[0x0935] = 0;
	    amstrad_amsdos_rom[0x096e] = 0;
	    amstrad_amsdos_rom[0x096f] = 0;
	    amstrad_amsdos_rom[0x0970] = 0;
	    amstrad_amsdos_rom[0x0972] = 0;
	    amstrad_amsdos_rom[0x0973] = 0;
	    amstrad_amsdos_rom[0x05d4] = 1;
	    amstrad_amsdos_rom[0x05d5] = 0;
	    amstrad_amsdos_rom[0x05d6] = 1;
	    amstrad_amsdos_rom[0x05d7] = 0;
	}

     /* load a CPCEMU .SNA file */
     public void loadSnapshot(String name, InputStream is, int snapshotLength)
        throws Exception
     {
        int snapshot_data[];

        int bytesLeft = snapshotLength;

        // Linux  JDK doesn't always know the size of files
		if ( snapshotLength < 0 ) {
			ByteArrayOutputStream os = new ByteArrayOutputStream();
			is = new BufferedInputStream( is, 4096 );

			int byteOrMinus1;
			int i;

			for ( i = 0; (byteOrMinus1 = is.read()) != -1; i++ ) {
				os.write( (byte) byteOrMinus1 );
			}

			is = new ByteArrayInputStream( os.toByteArray() );
			snapshotLength = i;
		}

        // allocate some space for it
        snapshot_data = new int [snapshotLength];

//   		startProgress( "Loading " + name, snapshotLength);

        bytesLeft -= readBytes( is, snapshot_data, 0, snapshotLength);


		A( snapshot_data[0x011] );
		F( snapshot_data[0x012] );
        C( snapshot_data[0x013] );
		B( snapshot_data[0x014] );
		E( snapshot_data[0x015] );
		D( snapshot_data[0x016] );
		L( snapshot_data[0x017] );
		H( snapshot_data[0x018] );

        R( snapshot_data[0x019] );
        I( snapshot_data[0x01a] );

      	IFF1( snapshot_data[0x01b] != 0 );
		IFF2( snapshot_data[0x01c] != 0 );

        IX( snapshot_data[0x01d] | (snapshot_data[0x01e]<<8) );
        IY( snapshot_data[0x01f] | (snapshot_data[0x020]<<8) );

		SP( snapshot_data[0x021] | (snapshot_data[0x022]<<8) );
        PC( snapshot_data[0x023] | (snapshot_data[0x024]<<8) );

		switch ( snapshot_data[0x025] & 0x03 ) {
		case 0:
			IM( IM0 );
			break;
		case 1:
			IM( IM1 );
			break;
		default:
			IM( IM2 );
			break;
		}

		ex_af_af();
		exx();

		A( snapshot_data[0x026] );
		F( snapshot_data[0x027] );
		C( snapshot_data[0x028] );
		B( snapshot_data[0x029] );
		E( snapshot_data[0x02a] );
		D( snapshot_data[0x02b] );
		L( snapshot_data[0x02c] );
		H( snapshot_data[0x02d] );

		ex_af_af();
		exx();

        /* Gate Array */

        for (int i=0; i<17; i++)
        {
            amstrad_ga_write(i);
            amstrad_ga_write((snapshot_data[0x02f + i] & 0x01f) | 0x040);
        }

        amstrad_ga_write(snapshot_data[0x02e]);

        amstrad_ga_write(snapshot_data[0x040]);
        amstrad_ga_write(snapshot_data[0x041]);

         /* crtc */

         for (int i=0; i<18; i++)
         {
            crtc_registers[i] = snapshot_data[0x043 + i];
         }

         crtc_regindex = snapshot_data[0x042];

         /* rom */
         amstrad_rom_select(snapshot_data[0x055]);

        /* PPI */

        ppi_write_control(snapshot_data[0x059]);
        ppi_write_port_a(snapshot_data[0x056]);
        ppi_write_port_b(snapshot_data[0x057]);
        ppi_write_port_c(snapshot_data[0x058]);

        /* PSG */
        for (int i=0; i<16; i++)
        {
            psg_registers[i] = snapshot_data[0x05b + i];
        }

        psg_regindex = snapshot_data[0x05a];

        // copy ram data
        for (int i=0; i<65536; i++)
        {
            amstrad_memory[i] = snapshot_data[0x0100 + i];
        }

		int base = (crtc_registers[12] & 0x030)<<(2+8);

		for (int i=0; i<16384; i++)
		{
			amstrad_scr_bytes[i] = 	amstrad_memory[base+i];
		}


     }

    private final int dsk_header[] =
    {
        'M',
        'V',
        ' ',
        '-',
        ' ',
        'C',
        'P',
        'C'
    };

    private final int extended_dsk_header[] =
    {
        'E',
        'X',
        'T',
        'E',
        'N',
        'D',
        'E',
        'D'
    };

    private boolean array_cmp(int array1[], int array2[], int length)
    {
        for (int i=0; i<8; i++)
        {
            if (array1[i] != array2[i])
            {
                return false;
            }
        }

        return true;
    }

    private boolean diskimage_is_dsk()
    {
        return array_cmp(diskimage_data, dsk_header, 8);
    }

    private void dsk_setup_track_offsets()
    {
        int offset;
        int tracks;
        int sides;
        int track_size;
        int step;
        int count;
        int offs;

        tracks = diskimage_data[0x030];
        sides = diskimage_data[0x031];

        track_size = diskimage_data[0x032] | (diskimage_data[0x033]<<8);

        step = 2-(sides-1);
        count = tracks*sides;

        offs = 0;

        offset = 0x0100;

        for (int i=0; i<count; i++)
        {
            diskimage_track_offsets[offs] = offset;
            offs+=step;
            offset+=track_size;
        }
    }

    private void dsk_setup_sector_offsets(int track, int side)
    {
        int spt;
        int track_offset;
        int sector_size;
        int offset;

        track_offset = diskimage_track_offsets[(track<<1) + side];

        if (track_offset==0)
            return;

        spt = diskimage_data[track_offset + 0x015];

        sector_size = (1<<(diskimage_data[track_offset + 0x014]+7));

        offset = 0x0100;

        for (int i=0; i<spt; i++)
        {
            diskimage_sector_offsets[i] = offset;
            offset+=sector_size;
        }
    }

    private int dsk_get_spt(int track, int side)
    {
        int spt;
        int track_offset;
        int sector_size;

        track_offset = diskimage_track_offsets[(track<<1) + side];

        if (track_offset==0)
            return 0;

        spt = diskimage_data[track_offset + 0x015];

        return spt;
    }

    private void dsk_get_id(int track, int side, int index)
    {
        int spt;
        int track_offset;
        int sector_size;

        track_offset = diskimage_track_offsets[(track<<1) + side];

        if (track_offset==0)
            return;

        spt = diskimage_data[track_offset + 0x015];

        if (spt!=0)
        {
            int id_offset;

            index = index % spt;

            id_offset = track_offset + 0x018 + (index<<3);

            fdc_result_bytes[3] = diskimage_data[id_offset];
            id_offset++;
            fdc_result_bytes[4] = diskimage_data[id_offset];
            id_offset++;
            fdc_result_bytes[5] = diskimage_data[id_offset];
            id_offset++;
            fdc_result_bytes[6] = diskimage_data[id_offset];
            id_offset++;

        }
    }

    private void dsk_get_sector(int track, int side, int index, int buffer[])
    {
        int track_offset;
        int sector_offset;
        int sector_size;

        track_offset = diskimage_track_offsets[(track<<1) + side];

        if (track_offset==0)
            return;

        dsk_setup_sector_offsets(track, side);

        sector_size =  (1<<(diskimage_data[track_offset + 0x014]+7));

        sector_offset = diskimage_sector_offsets[index];

        for (int i=0; i<sector_size; i++)
        {
            buffer[i] = diskimage_data[track_offset + sector_offset + i];
        }
    }




    private void extended_dsk_setup_track_offsets()
    {
    }


    private boolean diskimage_is_extended_dsk()
    {
        return array_cmp(diskimage_data, extended_dsk_header, 8);
    }

     /* load a CPCEMU .DSK file */
     public void loadDiskImage(String name, InputStream is, int diskimageLength)
        throws Exception
     {
        int bytesLeft = diskimageLength;

        // Linux  JDK doesn't always know the size of files
		if ( diskimageLength < 0 ) {
			ByteArrayOutputStream os = new ByteArrayOutputStream();
			is = new BufferedInputStream( is, 4096 );

			int byteOrMinus1;
			int i;

			for ( i = 0; (byteOrMinus1 = is.read()) != -1; i++ ) {
				os.write( (byte) byteOrMinus1 );
			}

			is = new ByteArrayInputStream( os.toByteArray() );
			diskimageLength = i;
		}

        // allocate some space for it
        diskimage_data = new int [diskimageLength];

//   		startProgress( "Loading " + name, diskimageLength);

        bytesLeft -= readBytes( is, diskimage_data, 0, diskimageLength);

        // disk image is loaded.
        if (diskimage_is_dsk())
        {
            diskimage_type = 0;
            drive_status[0] = DRIVE_STATUS_DISK_INSERTED;

            dsk_setup_track_offsets();
        }
        else if (diskimage_is_extended_dsk())
        {
            drive_status[0] = DRIVE_STATUS_DISK_INSERTED;
            diskimage_type = 1;

            extended_dsk_setup_track_offsets();
        }
        else
        {
            diskimage_type = -1;
        }
     }



     /*******************************/

	public int bytesReadSoFar = 0;
	public int bytesToReadTotal = 0;

//	private void startProgress( String text, int nBytes ) {
//		progressBar.setText( text );
//
//		bytesReadSoFar = 0;
//		bytesToReadTotal = nBytes;
//		updateProgress( 0 );
//
//		if ( showStats ) {
//			progressBar.show();
//			Thread.yield();
//		}
//	}

//	private void stopProgress() {
//		bytesReadSoFar = 0;
//		bytesToReadTotal = 0;
//		progressBar.setPercent( 0.0 );
//
//		if ( showStats ) {
//			progressBar.show();
//			Thread.yield();
//		}
//	}

//	private void updateProgress( int bytesRead ) {
//		bytesReadSoFar += bytesRead;
//		if ( bytesReadSoFar >= bytesToReadTotal ) {
//			stopProgress();
//			return;
//		}
//		progressBar.setPercent( (double)bytesReadSoFar / (double)bytesToReadTotal );
//		Thread.yield();
//	}


	private int readBytes( InputStream is, int a[], int off, int n ) throws Exception {
		try {
			BufferedInputStream bis = new BufferedInputStream( is, n );

			byte buff[] = new byte[ n ];
			int toRead = n;
			while ( toRead > 0 ) {
				int	nRead = bis.read( buff, n-toRead, toRead );
				toRead -= nRead;
//				updateProgress( nRead );
			}

			for ( int i = 0; i < n; i++ ) {
				a[ i+off ] = (buff[i]+256)&0xff;
			}

			return n;
		}
		catch ( Exception e ) {
			System.err.println( e );
			e.printStackTrace();
//			stopProgress();
			throw e;
		}
	}
}
