/* 
 *  BUILDCPR (c) Kevin Thacker 1999-2001
 *  
 *  This file is part of the BUILDCPR source code distribution.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
/* BUILDCPR (c) Kevin Thacker 1999.

  This program will build a cartridge image (.CPR) suitable for WinAPE32, Multi-Machine
  and Arnold CPC+ Emulators.

  Input is a binary file without any headers, and output is the .CPR file.
  Any file between 1 byte and 512k in size can be used. (512k is the maximum
  size of a cartridge for the CPC+).

*/

#include <stdio.h>
#include <stdlib.h>
#include "endian.h"

// define for Intel systems, or any where low byte is written first
#define LSB_FIRST

typedef struct RIFF_CHUNK
{
	unsigned long ChunkName;
	unsigned long ChunkLength;
} RIFF_CHUNK;


/* get size of an opened file */
int	GetFileSize(FILE *fh)
{
	int CurrentPosition;
	int FileSize;

	/* get current position - so we can restore it */
	CurrentPosition = ftell(fh);

	/* seek to end of file */
	fseek(fh, 0, SEEK_END);

	/* report position in file = File size */
	FileSize = ftell(fh);

	/* seek back to original position */
	fseek(fh, CurrentPosition, SEEK_SET);

	return FileSize;
}

int Endian_CheckEndianness(void)
{
	int EndianCorrect;
	unsigned long TestLong = 0;
	unsigned char *pLongPtr = (unsigned char *)&TestLong;

	pLongPtr[0] = 1;

#ifdef LSB_FIRST
	EndianCorrect = 0;
	if (TestLong == 1)
	{	
		/* on little endian, long = 1 */
		EndianCorrect = 1;
	}
#else
	EndianCorrect = 1;
	if (TestLong == 1)
	{
		/* on little endian, long = 1 */
		EndianCorrect = 0;
	}
#endif

	return EndianCorrect;
}




void	main(int argc, char *argv[])
{

	printf("BUILDCPR (c) Kevin Thacker.\r\n");

	if (Endian_CheckEndianness()==0)
	{
		printf("Program compiled with wrong Endianness setting.\r\n");
		exit(1);
	}
	
	if (argc!=3)
	{
		/* number of parameters is incorrect */
		printf("Usage: %s <filename.bin> <filename.cpr>\r\n",argv[0]);
		exit(1);
	}
	else
	{
		/* number of parameters is correct */

		FILE *fhInput;

		printf("Input Filename: %s\r\n", argv[1]);
		printf("Output Filename: %s\r\n", argv[2]);

		/* open input file */
		fhInput = fopen(argv[1],"rb");

		if (fhInput==NULL)
		{
			printf("Failed to open input file\r\n");
			exit(1);
		}

		if (fhInput!=NULL)
		{
			FILE *fhOutput;
			int InputFileSize = GetFileSize(fhInput);

			/* check input file size is valid */
			if ((InputFileSize<0) || (InputFileSize>(512*1024)))
			{
				/* input file size is invalid */
				printf("Input File Size (%d bytes) is invalid. This utility requires \
					an input file size between 1 byte and 512k.\r\n",InputFileSize);
				
				/* close the input file */
				fclose(fhInput);

				/* exit */
				exit(1);
			}

			/* input file size is valid */

			/* open output file */
			fhOutput = fopen(argv[2],"wb");
			
			if (fhOutput==NULL)
			{
				printf("Failed to open output file\r\n");

				fclose(fhInput);
				exit(1);
			}

			if (fhOutput!=NULL)
			{
				char	ChunkData[16384];
				int RiffTotalSize;
				int NoOfBlocks;
				int i;

				/* get size of input file in 16k blocks, if file
				doesn't fit exactly into 16k blocks, then this value
				will be enough to contain it, plus some extra filler
				data */
				NoOfBlocks = InputFileSize>>14;

				if ((InputFileSize & 0x03fff)!=0)
				{
					NoOfBlocks++;
				}

				/* calc size of output file */
				RiffTotalSize = (NoOfBlocks<<14) +
							(NoOfBlocks*sizeof(RIFF_CHUNK)) +
							sizeof(RIFF_CHUNK) + 4;

				
				{
					unsigned char *MainHeader = "RIFF";
					int RiffChunkSize;
					char	*RiffType = "AMS!";

					/* write main header ("RIFF") */
					fwrite(MainHeader, 4, 1, fhOutput);
					
					/* size of main chunk is size of total file - size
					of chunk header */
					RiffChunkSize = RiffTotalSize-sizeof(RIFF_CHUNK);

#ifndef LSB_FIRST
					RiffChunkSize = SwapEndianLong(RiffChunkSize);
#endif

					/* write RIFF chunk size */
					fwrite(&RiffChunkSize, 4, 1, fhOutput);
				
					/* write RIFF type */
					fwrite(RiffType, 4,1, fhOutput);
				}
				
				/* write data for all blocks - last one written seperatly */
				for (i=0; i<NoOfBlocks-1; i++)
				{
					char	ChunkName[6];
					unsigned long	ChunkSize = 16384;

					sprintf(ChunkName,"cb%02d",i);

					/* write chunk header */
					fwrite(ChunkName, 4, 1, fhOutput);

#ifndef LSB_FIRST
					ChunkSize = SwapEndianLong(ChunkSize);
#endif

					/* chunk size */
					fwrite(&ChunkSize, 4, 1, fhOutput);

					/* read data from input file */
					fread(ChunkData, 16384, 1, fhInput);
	
					/* write data to output file */
					fwrite(ChunkData, 16384, 1, fhOutput);
				}


				/* write last block */
				{
					char ChunkName[6];

					int ChunkSize = 16384;

					sprintf(ChunkName, "cb%02d",i);

					/* write chunk header */
					fwrite(ChunkName, 4, 1, fhOutput);
#ifndef LSB_FIRST
					ChunkSize = SwapEndianLong(ChunkSize);
#endif

					/* write chunk size */
					fwrite(&ChunkSize, 4, 1, fhOutput);

					ChunkSize = InputFileSize - (i<<14);

					fread(ChunkData, ChunkSize, 1, fhInput);

					fwrite(ChunkData, ChunkSize, 1, fhOutput);

					/* pad rest of chunk to 16384 bytes */
					for (i=0; i<16384-ChunkSize; i++)
					{
						fputc(0,fhOutput);
					}

				}	

				printf("Output File Written.\r\n");

				fclose(fhOutput);
			}

			fclose(fhInput);
		}

	}

	exit(0);
}


