/* 
 *  (c) Copyright, Kevin Thacker 2002
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <memory.h>

void	write_dummy_custom_block(FILE *fh, char id, unsigned long length)
{
	unsigned long i;

	fputc(id, fh);
	fputc((length&0x0ff),fh);
	fputc(((length>>8)&0x0ff),fh);
	fputc(((length>>16)&0x0ff),fh);
	fputc(((length>>24)&0x0ff),fh);

	/* fill block with data */
	for (i=0; i<length; i++)
	{
		fputc(0x0ff,fh);
	}
}

void	write_description_block(FILE *fh, const char *description)
{
	unsigned long length = 0;
	unsigned long description_length;
	unsigned long i;

	/* crop length to 255 characters if the description is longer */
	if (description)
	{
		description_length = strlen(description);

		if (description_length<255)
		{
			length = description_length;
		}
		else
		{
			length = 255;
		}
	}

	fputc(0x030,fh);
	fputc((length&0x0ff),fh);
	
	if (description)
	{
		for (i=0; i<length; i++)
		{
			fputc(description[i], fh);
		}
	}
}

void	write_group_start(FILE *fh, const char *description)
{
	unsigned long length = 0;
	unsigned long description_length;
	unsigned long i;

	/* crop length to 255 characters if the description is longer */
	if (description)
	{
		description_length = strlen(description);

		if (description_length<255)
		{
			length = description_length;
		}
		else
		{
			length = 255;
		}
	}

	fputc(0x021,fh);
	fputc((length&0x0ff),fh);
	
	if (description)
	{
		for (i=0; i<length; i++)
		{
			fputc(description[i], fh);
		}
	}
}

void	write_group_end(FILE *fh)
{
	fputc(0x022,fh);
}


void	write_message_block(FILE *fh, const char *message, int time)
{
	unsigned long length = 0;
	unsigned long message_length;
	unsigned long i;

	/* crop length to 255 characters if the description is longer */
	if (message)
	{
		message_length = strlen(message);

		if (message_length<255)
		{
			length = message_length;
		}
		else
		{
			length = 255;
		}
	}

	fputc(0x031,fh);
	fputc((time&0x0ff),fh);
	fputc((length&0x0ff),fh);

	if (message)
	{
		for (i=0; i<length; i++)
		{
			fputc(message[i], fh);
		}
	}
}


void	write_dummy_snapshot_block(FILE *fh, unsigned long snapshot_length)
{
	unsigned long i;

	/* 1 byte snapshot block id */
	fputc(0x040,fh);
	/* snapshot type */
	fputc(0x000,fh);
	/* 3 byte snapshot length */
	fputc((snapshot_length&0x0ff),fh);
	fputc(((snapshot_length>>8)&0x0ff),fh);
	fputc(((snapshot_length>>16)&0x0ff),fh);
	/* length of data */
	for (i=0; i<snapshot_length; i++)
	{
		fputc((i&0x0ff),fh);
	}
}

void	write_emulation_info(FILE *fh, unsigned short Flags, unsigned short IntFrequency, unsigned char ScrRefresh)
{
	fputc(0x034,fh);
	fputc((Flags&0x0ff),fh);
	fputc(((Flags>>8)&0x0ff),fh);
	fputc((ScrRefresh&0x0ff),fh);
	fputc((IntFrequency&0x0ff),fh);
	fputc(((IntFrequency>>8)&0x0ff),fh);
	fputc(1,fh);
	fputc(2,fh);
	fputc(3,fh);
}

void	write_dummy_custom_info(FILE *fh, const char *description, unsigned long custom_info_length)
{
	unsigned long i;
	unsigned long length;

	/* 1 byte snapshot block id */
	fputc(0x035,fh);

	/* crop length to 16 characters */
	length = strlen(description);
	if (length>16)
		length = 16;

	/* write description */
	for (i=0; i<length; i++)
	{
		fputc(description[i],fh);
	}
	
	/* 3 byte snapshot length */
	fputc((custom_info_length&0x0ff),fh);
	fputc(((custom_info_length>>8)&0x0ff),fh);
	fputc(((custom_info_length>>16)&0x0ff),fh);
	fputc(((custom_info_length>>32)&0x0ff),fh);
	/* length of data */
	for (i=0; i<custom_info_length; i++)
	{
		fputc((i&0x0ff),fh);
	}
}

char *tzx_signature="ZXTape!\x1a\x1\xd";

void	write_signature(FILE *fh)
{
	fwrite(tzx_signature, 10, 1, fh);
}


void	write_header(FILE *fh, unsigned char MajorVersion, unsigned char MinorVersion)
{
	fwrite(tzx_signature, 8, 1, fh);
	fputc(MajorVersion, fh);
	fputc(MinorVersion, fh);
}

typedef struct tzx_archive_info_string
{
	unsigned char text_id;
	const char *text;
} tzx_archive_info_string;

typedef struct tzx_hardware_type
{
	unsigned char hardware_type;
	unsigned char hardware_id;
	unsigned char value;
} tzx_hardware_type;

typedef struct tzx_select_block_item
{
	signed short relative_offset;
	const char *description;
} tzx_select_block_item;

signed short test_call_sequence[]=
{
	2,3,4,5
};

static struct tzx_hardware_type test_hardware_type[]=
{
	{0,0x0fe,0},
	{0,0x017,0},
	{0x0f3,0x0fe,0},
	{0x0c,0x01,0},
};

static struct tzx_archive_info_string test_archive_info[]=
{
	{00,"this is a test tape to test the tzx/cdt format"},
	{01,"publisher is kev"},
	{02,"author is kev"},
	{03,"the year of this prog is 2002"},
	{04,"language is blah"},
	{05,"type of this is test"},
	{06,"price of this is a lot!"},
	{07,"protection scheme is standard or none"},
	{8,"origin is the centre of the earth"},
	{0x0ff,"this is just a comment"},
	{80,"this type is not defined!!!!!!!!!"},
	{0x079,NULL},
};


/* these were added in v1.12 */
static struct tzx_archive_info_string test3_archive_info[]=
{
	{05,"type of this is test"},
	{06,"price of this is a lot!"},
	{07,"protection scheme is standard or none"},
	{8,"origin is the centre of the earth"},
};

static struct tzx_select_block_item test_selections[]=
{
	{-1,"this is selection 1"},
	{-30000,"this is selection 2"},
	{2,"this is selection 3"},
	{0,"this is selection 4"},
	{1,"this is selection 5"},
	{0,NULL},
};


static struct tzx_select_block_item test5_selections[]=
{
	{-1,"this is an invalid selection!"},
	{0,"this is a valid selection and displays this selection again"},
	{1,"this is test file 1"},
	{8,"this is test file 2"},
	{15,"this is test file 3"},
	{22,"is this the joined block header or the block after??"},
	{30000,"this is another invalid selection!"},
};


void	write_hardware_type(FILE *fh, unsigned long count, tzx_hardware_type *hardware_type_list)
{
	unsigned long i;


	/* block id */
	fputc(0x033,fh);
	if (hardware_type_list)
	{
		/* number of hardware type's */
		fputc((count&0x0ff),fh);
	}
	else
	{
		/* if list is not specified, do not store any hardware type's */
		fputc(0,fh);
	}

	if (hardware_type_list)
	{
		for (i=0; i<(count&0x0ff); i++)
		{
			fputc(hardware_type_list[i].hardware_type, fh);
			fputc(hardware_type_list[i].hardware_id, fh);
			fputc(hardware_type_list[i].value, fh);
		}
	}
}


/* length is zero */
void	write_empty_selection1(FILE *fh)
{
	fputc(0x028,fh);
	fputc(0x0,fh);
	fputc(0x0,fh);
}

/* length is 1, and there are no strings */
void	write_empty_selection2(FILE *fh)
{
	fputc(0x028,fh);
	fputc(0x1,fh);
	fputc(0x0,fh);
	fputc(0x0,fh);
}

/* length is zero */
void	write_empty_archive_info1(FILE *fh)
{
	fputc(0x032,fh);
	fputc(0x0,fh);
	fputc(0x0,fh);
}

/* length is 1, and there are no strings */
void	write_empty_archive_info2(FILE *fh)
{
	fputc(0x032,fh);
	fputc(0x1,fh);
	fputc(0x0,fh);
	fputc(0x0,fh);
}


void	write_return_from_sequence(FILE *fh)
{
	fputc(0x027,fh);
}

void	write_loop_start(FILE *fh, unsigned long repetitions)
{
	fputc(0x024,fh);
	fputc((repetitions&0x0ff),fh);
	fputc(((repetitions>>8)&0x0ff),fh);
}

void	write_loop_end(FILE *fh)
{
	fputc(0x025,fh);
}

void	write_relative_jump(FILE *fh, signed short offset)
{
	fputc(0x023,fh);
	fputc((offset&0x0ff),fh);
	fputc(((offset>>8)&0x0ff),fh);
}

void	write_pure_tone(FILE *fh, unsigned long NumPulses, unsigned short PulseLength)
{
	fputc(0x012,fh);
	fputc((PulseLength&0x0ff),fh);
	fputc(((PulseLength>>8)&0x0ff),fh);
	fputc((NumPulses&0x0ff),fh);
	fputc(((NumPulses>>8)&0x0ff),fh);
}

unsigned short test_pulses[]=
{
	100,200,300
};

/* length is 1, and there are no strings */
void	write_empty_archive_info3(FILE *fh)
{
	fputc(0x032,fh);
	fputc(0x3,fh);		// length of data in block excluding these two bytes
	fputc(0x0,fh);
	fputc(0x1,fh);		// num strings
	fputc(0x0,fh);		// id
	fputc(0x0,fh);		// length
}

void	write_archive_info(FILE *fh, unsigned long count, tzx_archive_info_string *archive_info_list)
{
	unsigned long i;
	unsigned long string_total_length = 0;
	unsigned long number_of_strings = 0;
	unsigned long block_total_length = 0;
	unsigned long num_strings = 0;

	/* block id */
	fputc(0x032,fh);

	/* TODO: limit length of block to 64k */
	/* TODO: limit length of block if number of strings exceeds 255 */
	if (archive_info_list)
	{
		for (i=0; i<count; i++)
		{
			if (archive_info_list[i].text!=NULL)
			{
				unsigned long string_length;

				/* crop length to 255 characters */
				string_length = strlen(archive_info_list[i].text);

				if (string_length>255)
					string_length=255;

				string_total_length+=string_length;

	
			}
			num_strings++;
			/* include size of id and length bytes */
			block_total_length+=2;
		}
	}

	/* include size of num strings byte */
	block_total_length++;

	/* include length of all strings */
	block_total_length+=string_total_length;

	fputc((block_total_length&0x0ff),fh);
	fputc(((block_total_length>>8)&0x0ff),fh);
	
	fputc((num_strings&0x0ff),fh);

	if (archive_info_list)
	{
		for (i=0; i<count; i++)
		{
			/* id of string */
			fputc(archive_info_list[i].text_id,fh);

			if (archive_info_list[i].text!=NULL)
			{
				unsigned long j;
				unsigned long string_length;

				/* crop length to 255 characters */
				string_length = strlen(archive_info_list[i].text);

				if (string_length>255)
					string_length=255;

				string_total_length+=string_length;

				/* length of string */
				fputc(string_length&0x0ff,fh);

				for (j=0; j<string_length; j++)
				{
					fputc(archive_info_list[i].text[j],fh);
				}
			}
			else
			{
				fputc(0,fh);
			}
		}
	}
}

void	write_selections(FILE *fh, unsigned long count, tzx_select_block_item *selection_list)
{
	unsigned long i;
	unsigned long string_total_length = 0;
	unsigned long number_of_strings = 0;
	unsigned long block_total_length = 0;
	unsigned long num_selections= 0;

	/* block id */
	fputc(0x028,fh);

	/* TODO: limit length of block to 64k */
	/* TODO: limit length of block if number of selections exceeds 255 */
	if (selection_list)
	{
		for (i=0; i<count; i++)
		{
			if (selection_list[i].description!=NULL)
			{
				unsigned long string_length;

				/* crop length to 255 characters */
				string_length = strlen(selection_list[i].description);

				if (string_length>255)
					string_length=255;

				string_total_length+=string_length;
			}
			/* include size of offset and length */
			block_total_length+=3;

			num_selections++;
		}
	}

	/* include number of selections */
	block_total_length++;

	/* include length of all strings */
	block_total_length+=string_total_length;

	fputc((block_total_length&0x0ff),fh);
	fputc(((block_total_length>>8)&0x0ff),fh);
	
	fputc((num_selections&0x0ff),fh);

	if (selection_list)
	{
		for (i=0; i<count; i++)
		{
			/* offset of selection */
			fputc((selection_list[i].relative_offset&0x0ff),fh);
			fputc(((selection_list[i].relative_offset>>8)&0x0ff),fh);

			if (selection_list[i].description!=NULL)
			{
				unsigned long j;
				unsigned long string_length;

				/* crop length to 255 characters */
				string_length = strlen(selection_list[i].description);

				if (string_length>255)
					string_length=255;

				string_total_length+=string_length;

				/* length of string */
				fputc(string_length&0x0ff,fh);

				for (j=0; j<string_length; j++)
				{
					fputc(selection_list[i].description[j],fh);
				}
			}
			else
			{
				fputc(0,fh);
			}

		}
	}
}

void	write_call_sequence(FILE *fh, unsigned long Count, signed short *call_sequence)
{
	unsigned long i;

	fputc(0x026,fh);

	if (call_sequence==NULL)
	{
		Count = 0;
	}

	fputc((Count&0x0ff),fh);
	fputc(((Count>>8)&0x0ff),fh);

	for (i=0; i<Count; i++)
	{
		fputc((call_sequence[i]&0x0ff),fh);
		fputc(((call_sequence[i]>>8)&0x0ff),fh);
	}


}

void	write_pause(FILE *fh, unsigned long Pause)
{
	fputc(0x020,fh);
	fputc((Pause&0x0ff),fh);
	fputc(((Pause>>8)&0x0ff),fh);
}

void	write_sequence_of_pulses(FILE *fh, unsigned long Count, unsigned short *pulses)
{
	unsigned long i;
	unsigned long NumPulses;

	fputc(0x013,fh);
	
	if (Count>255)
		NumPulses = 255;
	else 
		NumPulses = Count;

	if (pulses==NULL)
		NumPulses = 0;

	fputc((NumPulses&0x0ff), fh);

	for (i=0; i<NumPulses; i++)
	{
		fputc((pulses[i]&0x0ff),fh);
		fputc(((pulses[i]>>8)&0x0ff),fh);
	}
}

void	write_standard_speed(FILE *fh, unsigned long PauseLength, unsigned long DataLength)
{
	unsigned long ActualLength;
	unsigned long i;

	fputc(0x010, fh);
	fputc((PauseLength&0x0ff),fh);
	fputc(((PauseLength>>8)&0x0ff),fh);

	if (DataLength>65535)
	{
		ActualLength = 65535;
	}
	else
	{
		ActualLength = DataLength;
	}

	fputc((ActualLength&0x0ff),fh);
	fputc(((ActualLength>>8)&0x0ff),fh);

	for (i=0; i<ActualLength; i++)
	{
		fputc((i&0x0ff),fh);
	}
}

void	write_pure_data(FILE *fh, unsigned long ZeroPulseLength, unsigned long OnePulseLength,unsigned long BitsUsedInLastByte, unsigned long Pause, unsigned long DataLength)
{
	unsigned long i;

	fputc(0x014,fh);
	fputc((ZeroPulseLength&0x0ff),fh);
	fputc(((ZeroPulseLength>>8)&0x0ff),fh);
	fputc((OnePulseLength&0x0ff),fh);
	fputc(((OnePulseLength>>8)&0x0ff),fh);
	fputc(BitsUsedInLastByte,fh);
	fputc((Pause&0x0ff),fh);
	fputc(((Pause>>8)&0x0ff),fh);
	fputc((DataLength&0x0ff),fh);
	fputc(((DataLength>>8)&0x0ff),fh);
	fputc(((DataLength>>16)&0x0ff),fh);

	for (i=0; i<DataLength; i++)
	{
		fputc((i&0x0ff),fh);
	}
}


const char *InstructionsString="Instructions    ";

void	WriteInstructionsBlock(FILE *fh, const char *pInstructions)
{
	fwrite(InstructionsString,16,1,fh);
	
	if (pInstructions==NULL)
	{
		fputc(0,fh);
		fputc(0,fh);
		fputc(0,fh);
		fputc(0,fh);
	}
	else
	{
		unsigned long InstructionLength;

		InstructionLength = strlen(pInstructions);

		fputc((InstructionLength&0x0ff),fh);
		fputc(((InstructionLength>>8)&0x0ff),fh);
		fputc(((InstructionLength>>16)&0x0ff),fh);
		fputc(((InstructionLength>>24)&0x0ff),fh);

		fwrite(pInstructions, InstructionLength, 1, fh);
	}
}


const char *PokesString="POKEs            ";

/* check format of pokes block! */

void	WritePokesBlock(FILE *fh)
{
	fwrite(PokesString,16,1,fh);
	fputc(0,fh);
	fputc(0,fh);
	fputc(0,fh);
	fputc(0,fh);
}


/* in this example, the two blocks should be ignored as they were
not supported by this version */
static void WriteTestTape2(char *pFilename)
{
	FILE *fh;

	fh = fopen(pFilename,"wb");

	if (fh!=NULL)
	{
		/* write v1.12 cdt */
		write_header(fh, 1, 12);
	
		/* new in v1.13 are the C64 blocks */
		write_dummy_custom_block(fh, 0x016, 0);
		write_dummy_custom_block(fh, 0x016, 8);
		write_dummy_custom_block(fh, 0x017, 0);
		write_dummy_custom_block(fh, 0x017, 8);

		fclose(fh);
	}
}

static void WriteTestTape3(char *pFilename)
{
	FILE *fh;

	fh = fopen(pFilename,"wb");

	if (fh!=NULL)
	{
		/* write v1.11 cdt */
		write_header(fh, 1, 11);
	
		/* new in v1.13 are the C64 blocks */
		write_dummy_custom_block(fh, 0x016, 0);
		write_dummy_custom_block(fh, 0x016, 8);
		write_dummy_custom_block(fh, 0x017, 0);
		write_dummy_custom_block(fh, 0x017, 8);

		/* new in v1.12 are some archive info blocks */

		write_archive_info(fh, (sizeof(test3_archive_info)/sizeof(tzx_archive_info_string)), test3_archive_info);

		fclose(fh);
	}
}



static void WriteTestTape(char *pFilename)
{
	FILE *fh;

	fh = fopen(pFilename,"wb");

	if (fh!=NULL)
	{
		int i;

		write_signature(fh);

		for (i=0; i<0x0ff; i++)
		{
			switch (i)
			{
				case 0x010:
				{
					write_standard_speed(fh, 0, 0);
					write_standard_speed(fh, 0, 3000);
					write_standard_speed(fh, 100,3000);
				}
				break;
				case 0x011:
					break;
				case 0x012:
				{
					write_pure_tone(fh, 0, 0);
					write_pure_tone(fh, 0, 1000);
					write_pure_tone(fh, 1000, 0);
					write_pure_tone(fh, 100,2000);
				}
				break;
				case 0x013:
				{
					write_sequence_of_pulses(fh, 0, NULL);
					write_sequence_of_pulses(fh, sizeof(test_pulses)/sizeof(unsigned short), test_pulses);
				}
				break;
				case 0x014:
				{
					write_pure_data(fh,0,0,0,0,0);
					write_pure_data(fh,0,0,0,0,1);					
					write_pure_data(fh,0,0,0,1,0);					
					write_pure_data(fh,0,0,0,1,1);					
					write_pure_data(fh,0,0,1,0,0);					
					write_pure_data(fh,0,0,1,0,1);					
					write_pure_data(fh,0,0,1,1,0);					
					write_pure_data(fh,0,0,1,1,1);					
					write_pure_data(fh,0,103,0,0,0);					
					write_pure_data(fh,0,103,0,0,1);					
					write_pure_data(fh,0,103,0,1,0);					
					write_pure_data(fh,0,103,0,1,1);					
					write_pure_data(fh,0,103,1,0,0);					
					write_pure_data(fh,0,103,1,0,1);					
					write_pure_data(fh,0,103,1,1,0);					
					write_pure_data(fh,0,103,1,1,1);					
					write_pure_data(fh,103,103,0,0,0);					
					write_pure_data(fh,103,103,0,0,1);					
					write_pure_data(fh,103,103,0,1,0);					
					write_pure_data(fh,103,103,0,1,1);					
					write_pure_data(fh,103,103,1,0,0);					
					write_pure_data(fh,103,103,1,0,1);					
					write_pure_data(fh,103,103,1,1,0);					
					write_pure_data(fh,103,103,1,1,1);					
				}
				break;
				case 0x015:
					break;
				case 0x016:
					break;
				case 0x017:
					break;
				case 0x020:
				{
					write_pause(fh, 0);
					write_pause(fh, 500);
				}
				break;

				case 0x021:
				{
					write_group_start(fh,NULL);
					write_group_end(fh);

					write_group_start(fh,"a dummy group");
					write_group_end(fh);
				}
				break;
				case 0x022:
				{
					write_group_start(fh,"another dummy group");
					write_group_end(fh);
				}
				break;
				case 0x023:
				{
					write_relative_jump(fh, 1);
				}
				break;
				case 0x024:
				{
					write_loop_start(fh, 0);
					write_loop_end(fh);
					write_loop_start(fh, 1);
					write_loop_end(fh);
					write_loop_start(fh, 2);
					write_loop_end(fh);
				}
				break;
				case 0x025:
				{
					write_loop_start(fh,1);
					write_loop_end(fh);

					write_loop_end(fh);
				}
				break;
				case 0x026:
				{
					unsigned long j;

					write_call_sequence(fh,0, NULL);
					write_call_sequence(fh,sizeof(test_call_sequence)/sizeof(signed short), test_call_sequence);
					write_relative_jump(fh, sizeof(test_call_sequence)/sizeof(signed short));				
				
					for (j=0; j<sizeof(test_call_sequence)/sizeof(signed short); j++)
					{
						write_return_from_sequence(fh);
					}
					
				}
				break;
				case 0x027:
				{

					write_return_from_sequence(fh);
				}
				break;
				case 0x028:
				{
					write_empty_selection1(fh);
					write_empty_selection2(fh);
					write_selections(fh, sizeof(test_selections)/sizeof(tzx_select_block_item), test_selections);
				}
				break;
				case 0x031:
				{
					int j;
					char longmessage[512];

					write_message_block(fh, NULL,0);
					write_message_block(fh, "this is a test message",0);
					write_message_block(fh, "this is another test message",1);
					for (j=0; j<sizeof(longmessage); j++)
					{
						longmessage[j] = 'a'+(j % 26);
					}
					longmessage[sizeof(longmessage)-1] = '\0';
					write_message_block(fh, longmessage,0);
					write_message_block(fh, longmessage,1);				
				}
				break;
				
				case 0x032:
				{

					write_empty_archive_info1(fh);
					write_empty_archive_info2(fh);
					write_empty_archive_info3(fh);
					write_archive_info(fh, (sizeof(test_archive_info)/sizeof(tzx_archive_info_string)), test_archive_info);
				}
				break;
				
				case 0x033:
				{
					write_hardware_type(fh, 0, NULL);
					write_hardware_type(fh, (sizeof(test_hardware_type)/sizeof(tzx_hardware_type)), test_hardware_type);
				}
				break;
				case 0x034:
				{
					write_emulation_info(fh, 0x01234, 50,3);
				}
				break;

				case 0x035:
				{
					write_dummy_custom_info(fh, "dummy custom info",0);
					write_dummy_custom_info(fh, "another dummy custom info block",512);					
				}
				break;
				
				case 0x040:
				{
					write_dummy_snapshot_block(fh, 0);
					write_dummy_snapshot_block(fh, 49152);
				}
				break;

				case 0x05a:
				{
					write_signature(fh);
				}
				break;

				case 0x030:
				{
					int j;
					char longdescription[512];

					write_description_block(fh, NULL);
					for (j=0; j<sizeof(longdescription); j++)
					{
						longdescription[j] = 'a'+(j % 26);
					}
					longdescription[sizeof(longdescription)-1] = '\0';
					write_description_block(fh, longdescription);
					write_description_block(fh, "This is a test description");
				
				}
				break;

				case 0x02a:
				default:
				{
					write_dummy_custom_block(fh, i&0x0ff, 0);
					write_dummy_custom_block(fh, i&0x0ff, 8);
				}
				break;
			}
		}

		WriteInstructionsBlock(fh,NULL);
		WriteInstructionsBlock(fh,"This is a dummy instructions block!");
		fclose(fh);
	}
}

int	LoadFile(const char *pFilename, char **ppData, unsigned long *pLength)
{
	FILE *fh;
	int success = 0;

	*ppData = NULL;
	*pLength = 0;

	fh = fopen(pFilename,"rb");

	if (fh!=NULL)
	{
		char *pData;
		unsigned long length;

		fseek(fh, 0, SEEK_END);
		length = ftell(fh);
		fseek(fh, 0, SEEK_SET);

		pData = malloc(length);

		if (pData!=NULL)
		{
			fread(pData, 1, length, fh);
			*ppData = pData;
			*pLength = length;
			success = 1;
		}

		fclose(fh);
	}
	
	return success;
}

void	put_test_program(FILE *fh, char *pData, int length)
{
	write_group_start(fh, "test program");
	
	fwrite(pData, 1, length, fh);

	write_group_end(fh);
}

typedef struct
{
	int BlockActive;

	// position of num pulses variable in file
	size_t BlockPos;
	// file handle
	FILE *fh;
	// num pulses
	int NumPulses;
} Block13Writer;

void Block13Writer_End(Block13Writer *writer)
{
	if (writer->BlockActive)
	{
		size_t CurPos;

		// get cur pos
		CurPos = ftell(writer->fh);
		// seek back
		fseek(writer->fh, writer->BlockPos, SEEK_SET);
		// write number of pulses
		fputc(writer->NumPulses, writer->fh);
		// seek forwards
		fseek(writer->fh, CurPos, SEEK_SET);
	}

	writer->BlockActive = 0;

}

void Block13Writer_Begin(Block13Writer *writer)
{
	if (writer->BlockActive)
	{
		Block13Writer_End(writer);
	}

	// write block id
	fputc(0x013, writer->fh);
	// get pos
	writer->BlockPos = ftell(writer->fh);
	// write fake num pulses
	fputc(0x00, writer->fh);
	writer->NumPulses = 0;
	writer->BlockActive = 1;
}

void Block13Writer_Init(Block13Writer *writer, FILE *fh)
{
	writer->BlockActive = 0;
	writer->fh = fh;
	writer->NumPulses = 0;
	writer->BlockPos = 0;
}


void Block13Writer_AddPulse(Block13Writer *writer, unsigned short Pulse)
{
	// block active?
	if (!writer->BlockActive)
	{
		// no

		// begin a new block
		Block13Writer_Begin(writer);
		// write this pulse
		fputc((Pulse&0x0ff),writer->fh);
		fputc(((Pulse>>8)&0x0ff),writer->fh);
		writer->NumPulses++;
	}
	else
	{
		// yes

		// end previous block
		if (writer->NumPulses==255)
		{
			Block13Writer_End(writer);
			// begin new block
			Block13Writer_Begin(writer);
		}
		// write this pulse
		fputc((Pulse&0x0ff),writer->fh);
		fputc(((Pulse>>8)&0x0ff),writer->fh);
		writer->NumPulses++;

	}
}

void	write_0x011_as_pulses(FILE *fh, char *pBlock)
{
	int i;
	Block13Writer writer;

	Block13Writer_Init(&writer,fh);

	for (i=0; i<5; i++)
	{
		switch (i)
		{
			case 0:
			{
				unsigned short PilotPulseLength;
				unsigned short NumPilotPulses;

				PilotPulseLength = (pBlock[0]&0x0ff)|((pBlock[1]&0x0ff)<<8);
				NumPilotPulses = (pBlock[0x0a]&0x0ff)|((pBlock[0x0b]&0x0ff)<<8);

				if ((PilotPulseLength!=0) && (NumPilotPulses!=0))
				{
					int p;

					for (p=0; p<NumPilotPulses; p++)
					{
						Block13Writer_AddPulse(&writer,PilotPulseLength);
					}
				}
			}
			break;

			case 1:
			{
				unsigned short Sync1;

				Sync1 = (pBlock[2]&0x0ff)|((pBlock[3]&0x0ff)<<8);

				if (Sync1!=0)
				{
					Block13Writer_AddPulse(&writer,Sync1);
				}
			}
			break;

			case 2:
			{
				unsigned short Sync2;

				Sync2 = (pBlock[4]&0x0ff)|((pBlock[5]&0x0ff)<<8);

				if (Sync2!=0)
				{
					Block13Writer_AddPulse(&writer,Sync2);
				}
			}
			break;

			case 3:
			{
				unsigned long Length;

				Length = (pBlock[0x0f]&0x0ff)|((pBlock[0x010]&0x0ff)<<8)|((pBlock[0x011]&0x0ff)<<16);

				if (Length!=0)
				{
					unsigned long p;
					unsigned short Bit1,Bit0;
					unsigned char *pData;

					Bit1= (pBlock[8]&0x0ff)|((pBlock[9]&0x0ff)<<8);
					Bit0= (pBlock[6]&0x0ff)|((pBlock[7]&0x0ff)<<8);

					pData = &pBlock[0x012];

					for (p=0; p<Length; p++)
					{
						int b;

						if (p==(Length-1))
						{
							b = pBlock[0x0c]&0x0ff;
						}
						else
						{
							b = 8;
						}

						for (b=0; b<8; b++)
						{
							unsigned short Pulse;

							if (pData[p]&(1<<(7-b)))
							{
								Pulse = Bit1;
							}
							else
							{
								Pulse = Bit0;
							}
							Block13Writer_AddPulse(&writer,Pulse);
						}
					}
				}
			}
			break;

			case 4:
			{
				unsigned short Pause;

				Pause = (pBlock[0x0d]&0x0ff)|((pBlock[0x0e]&0x0ff)<<8);
		
				Block13Writer_End(&writer);
				
				if (Pause!=0)
				{
					write_pause(fh, Pause);
				}
			}
			break;

		}
	}
}

unsigned long GetBlockLength(char *pBlock)
{
	unsigned long ID;
	unsigned char *pBlockData = pBlock+1;
	unsigned long Length;

	ID = pBlock[0];

	switch (ID)
	{
		case 'Z':
			Length = 0x0a;
			break;

		case 0x011:
			Length = ((pBlockData[0x0f]&0x0ff)|((pBlockData[0x010]&0x0ff)<<8)|((pBlockData[0x011]&0x0ff)<<16)) + 0x012 + 1;
			break;

	}

	return Length;
}

signed short Test2CallSequence[]=
{
	3,		/* pause + relative jump + return */
	7,		/* pause + program + return */
	15,		/* return from sequence */
	16,	
	28		/* pause + program as pulses + return */
};

int main(int argc, char **argv)
{
	char *pData;
	char **pBlockPtrs;
	int BlockCount;

	unsigned long DataLen;

	/* a empty CDT/TZX */
	{
		FILE *fh;

		fh = fopen("test1.cdt","wb");

		if (fh!=NULL)
		{
			int i;

			write_signature(fh);

			fclose(fh);
		}
	}
	
	{
		/* has all blocks */
		WriteTestTape("test2.cdt");
		/* writes a v1.12 cdt but with some blocks that were new in v1.13 */
		/* should be ignored by emu */
		WriteTestTape2("test3.cdt");
		/* writes a v1.11 cdt but with some blocks that were new in v1.12 and v1.13 */
		/* should be ignored by emu */
		WriteTestTape3("test4.cdt");
	}

	/* a special CDT using repeat etc */

	LoadFile("data.cdt",&pData,&DataLen);

	{
		char *pBlockPtr;

		pBlockPtr = pData;

		pBlockPtr += GetBlockLength(pBlockPtr);

		BlockCount = 0;

		while (pBlockPtr<(pData+DataLen))
		{
			BlockCount++;

			pBlockPtr += GetBlockLength(pBlockPtr);
		}
	
		pBlockPtrs = malloc(sizeof(char *)*BlockCount);

		pBlockPtr = pData;

		pBlockPtr += GetBlockLength(pBlockPtr);
		BlockCount = 0;

		while (pBlockPtr<(pData+DataLen))
		{
			pBlockPtrs[BlockCount] = pBlockPtr;
			BlockCount++;

			pBlockPtr += GetBlockLength(pBlockPtr);
		}
	}

	{
		FILE *fh;

		fh = fopen("test5.cdt","wb");

		if (fh!=NULL)
		{
			int i;

			write_signature(fh);

			// block 0
			write_selections(fh, sizeof(test5_selections)/sizeof(test5_selections[0]), test5_selections);

			// block 1
			write_message_block(fh, "this is test file 1",0);
			// block 2,3,4,5,6
			put_test_program(fh, pData, DataLen);
			// block 7
			write_relative_jump(fh, 0);

			// block 8
			write_message_block(fh, "this is test file 2",0);
			// block 9,10,11,12,13
			put_test_program(fh, pData, DataLen);
			// block 14
			write_relative_jump(fh, 0);

			// block 15
			write_message_block(fh, "this is test file 3",0);
			// block 16,17,18,19,20
			put_test_program(fh, pData, DataLen);
			// block 21
			write_relative_jump(fh, 0);

			// block 22
			write_signature(fh);
			write_relative_jump(fh, 0);

			fclose(fh);
		}
	}
	
	{
		FILE *fh;

		fh = fopen("test6.cdt","wb");

		if (fh!=NULL)
		{
			int i;

			write_signature(fh);

			// block 0
			write_call_sequence(fh, sizeof(Test2CallSequence)/sizeof(signed short), Test2CallSequence);
			// block 1
			write_message_block(fh, "this is the end of the test",0);
			// block 2
			write_relative_jump(fh, 0);
			// block 3
			write_pause(fh, 2000);
			// block 4
			write_relative_jump(fh, 1);
			// block 5
			write_message_block(fh, "this message should never appear!",0);
			// block 6
			write_return_from_sequence(fh);
			// block 7
			write_description_block(fh, "original test program");
			// block 8
			write_pause(fh, 2000);
			// block 9,10,11,12,13
			put_test_program(fh, pData, DataLen);
			// block 14
			write_return_from_sequence(fh);
			// block 15
			write_return_from_sequence(fh);
			// block 16			
			write_message_block(fh, "3 repetitions of test program follows",0);
			// block 17
			write_description_block(fh, "3 repetitions of test program");
			// block 18
			write_loop_start(fh, 3);
			// block 19
			write_message_block(fh, "test program",1);
			// block 20
			write_pause(fh, 1000);
			// block 21,22,23,24,25
			put_test_program(fh, pData, DataLen);
			// block 26
			write_loop_end(fh);
			// block 27
			write_return_from_sequence(fh);		
			// block 28
			write_pause(fh, 2000);
			// block 25
			write_description_block(fh, "test program as pulses!");
			// block 26	
			for (i=0; i<BlockCount; i++)
			{
				char *pBlock;

				pBlock = pBlockPtrs[i];

				if (pBlock[0]==0x011)
				{
					write_0x011_as_pulses(fh, pBlock+1);
				}
				// block ?
				write_return_from_sequence(fh);

			}

			fclose(fh);
		}
	}

	exit(0);
}