/* (c) Kevin Thacker, May 2005 */
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "aslink.h"
#include "lkrrel.h"

struct relrechead
{
	int count;
	struct relrec *head;
};

/* 16-bit values, add full 16-bit base address */

static struct relrechead head16= {0,NULL};
/* 8-bit, but add high byte of 16-bit base address */
static struct relrechead head8hi = {0,NULL};
/* 8-bit, but add low byte of 16-bit base address */
static struct relrechead head8lo = {0,NULL};

/* internal; free's list */
static void relrecfree(struct relrechead *head)
{
	struct relrec *cur = head->head;
	
	while (cur!=NULL)
	{
		struct relrec *next = cur->next;
		free(cur);

		cur = next;
	}
}

/* free relocation record list; e.g. on exit */
void freerelrec()
{
	relrecfree(&head16);
	relrecfree(&head8hi);
	relrecfree(&head8lo);
}

/* internal; allocate a relocation record */
static void newrelrec(struct relrechead *head, Addr_T addr, Addr_T symaddr)
{
	struct relrec *rec;
	
	rec  = (struct relrec *) malloc(sizeof(struct relrec));
	
	/* error if allocation failed */
	if (rec==NULL)
		return;

	rec->addr = addr;
	rec->symaddr = symaddr;
	
	/* add to start of list */
	rec->next = head->head;
	head->head = rec;
	head->count++;
}
	
/* add item to 16-bit relocation record list */
void relrec16(Addr_T addr)
{
	newrelrec(&head16, addr,0);
}

/* add item to 8-bit (high) relocation record list */
void relrec8hi(Addr_T addr, Addr_T symaddr)
{
	newrelrec(&head8hi, addr,symaddr);
}

/* add address to 8-bit (low) relocation list */
void relrec8lo(Addr_T addr, Addr_T symaddr)
{
	newrelrec(&head8lo, addr,symaddr);
}

/* internal; compare two addresses within two relocate records */ 
static int relreccompare(const void *a, const void *b)
{
	const struct relrec *relreca = (const struct relrec *)a;
	const struct relrec *relrecb = (const struct relrec *)b;

	return (relreca->addr-relrecb->addr);
}

/* sort a list of relocate records and return an array of the records */
static struct relrec *relrecsort(struct relrechead *head)
{
	int count;
	struct relrec *reclist;
	struct relrec *cur;

	if (head->count==0)
		return NULL;

	/* allocate list to hold all items */
	reclist = (struct relrec *)malloc(sizeof(struct relrec)*head->count);	

	if (reclist==NULL)
		return NULL;

	/* fill list */
	count = 0;
	cur = head->head;
	while (cur!=NULL)
	{
		memcpy(&reclist[count],cur,sizeof(struct relrec));
		count++;
		cur = cur->next;
	}

	/* sort into ascending address order */
	qsort(reclist, count, sizeof(struct relrec),relreccompare);

	return reclist;
}

int outbyte(int addr, char byte)
{
	rtval[0] = addr&0x0ff;
	rtval[1] = (addr>>8)&0x0ff;
	rtval[2] = byte&0x0ff;
	addr+=1;		
	rtflg[0] = 1;
	rtflg[1] = 1;
	rtflg[2] = 1;
	rtcnt = 3;
	ihx(1);
	return addr;
}

void relrecwritelist1(struct relrechead *list,int *addrptr)
{
	int i;
	struct relrec *relreclist;
	int addr;

	addr = *addrptr;

	relreclist = relrecsort(list);

	if (relreclist!=NULL)
	{
		int prevaddr = 0;
		for (i=0; i<list->count; i++)
		{
			struct relrec *cur = &relreclist[i];

			int delta = cur->addr-prevaddr;
			
			if (delta>254)
			{
				int largedelta = delta-1;
				addr = outbyte(addr,0x0ff);
				addr = outbyte(addr,largedelta&0x0ff);
				addr = outbyte(addr,(largedelta>>8)&0x0ff);
				delta = 1;
			}
			prevaddr = cur->addr;
			addr = outbyte(addr,delta&0x0ff);
		}	
	
		free(relreclist);
	}
	
	addr = outbyte(addr, 0);
	
	*addrptr = addr;

}

void relrecwritelist2(struct relrechead *list,int *addrptr)
{
	int i;
	struct relrec *relreclist;
	int addr;

	addr = *addrptr;

	relreclist = relrecsort(list);

	if (relreclist!=NULL)
	{
		int prevaddr = 0;
		for (i=0; i<list->count; i++)
		{
			struct relrec *cur = &relreclist[i];
			int delta = cur->addr-prevaddr;
			prevaddr = cur->addr;
			
			if (delta>254)
			{
				int largedelta = delta-1;
				addr = outbyte(addr,0x0ff);
				addr = outbyte(addr,largedelta&0x0ff);
				addr = outbyte(addr,(largedelta>>8)&0x0ff);
				delta = 1;
			}

			addr = outbyte(addr,delta&0x0ff);
			addr = outbyte(addr,cur->symaddr&0x0ff);
		}	
	
		free(relreclist);
	}

	addr = outbyte(addr,0);
	
	*addrptr = addr;

}


void relrecwrite()
{
	int areasize = 0;

	struct area *cur = areap;
	while (cur!=NULL)
	{
		areasize+=cur->a_size;
		cur=cur->a_ap;
	}

	//printf("total length before relocation records: %04x\n",areasize);

	/* re-write offset to relocation records */
	rtval[0] = 0;
	rtval[1] = 0;
	rtval[2] = areasize&0x0ff;
	rtval[3] = (areasize>>8)&0x0ff;
	rtflg[0] = 1;
	rtflg[1] = 1;
	rtflg[2] = 1;
	rtflg[3] = 1;
	rtcnt = 4;
	ihx(1);
	

	int addr = areasize;


	relrecwritelist1(&head16,&addr);
	relrecwritelist1(&head8lo,&addr);
	relrecwritelist2(&head8hi,&addr);
}
