/* 
 *
 *  Standard loader testing tool (c) Kevin Thacker, July 2002 
 *
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/* This tool will check if a file which uses the STANDARD loader
is correct:

  What this tool can identify:
  - missing header or data blocks
  - checksum problems
  - malformed files
  - header block lengths are correct
  - data block lengths are correct based on previous header
  - will attempt to recognise headerless files.
    will only suggest these if the crc checks out for
	a complete number of records!

  What this tool can't identify:
  - complete missing files

  Bugs & to do:
  - doesn't check timings (e.g. sync) to ensure they are correct
  - check pauses between blocks (if pause is not correct then Amstrad will miss the file!)
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <memory.h>

typedef int BOOL;
#define TRUE (1==1)
#define FALSE (1==0)

char *Messages[]=
{
	"Header block missing!",
	"Data block missing!",
	"Block not recognised. Ignored.",
	"Checksum failed!",
	"Warning: Block has no length",
	"Ok ",
	"BAD",
	"Header block has bad length!. Unable to verify",
	"First block of file not found!",
	"Missing block(s)!",
	"Missing block(s)! Found block of another file!",
	"Sync byte is not recognised!",
	"Data block has bad length!. Unable to verify",
	"Insufficient pause",
	"Block may not be complete!",
	"Header indicates that first data block exceeds 2K in length! This block will not be readable on CPC!",
	"This data block is too long!",
	"Header block may be missing!",
	"Previous file was not complete!",
};

void OutputError(const char *pMessage)
{
	fprintf(stdout,"*** ");
	fprintf(stdout,pMessage);
	fprintf(stdout,"\r\n");
}

void	OutputText(const char *pMessage)
{
	fprintf(stdout, pMessage);
}

BOOL ReadTZXFile(const char *pFilename, unsigned char **ppData, unsigned long *pLength)
{
	BOOL bSuccess = FALSE;
	FILE *fh;

	/* open file */
	fh = fopen(pFilename,"rb");

	if (fh!=NULL)
	{
		unsigned long length;
		unsigned long currentPosition;
		/* get current position */
		currentPosition = ftell(fh);
		/* seek to end */
		fseek(fh, 0, SEEK_END);
		/* report position = length */
		length = ftell(fh);
		/* seek to original position */
		fseek(fh, currentPosition, SEEK_SET);

		if (length!=0)
		{
			unsigned char *pData;

			/* allocate memory to load tzx into */
			pData = (unsigned char *)malloc(length);

			if (pData!=NULL)
			{
				/* read data */
				if (fread(pData, sizeof(unsigned char), length, fh)==length)
				{
					bSuccess = TRUE;					
					*pLength = length;
					*ppData = pData;
				}

				/* if not successful, free memory allocated for file */
				if (!bSuccess)
				{
					free(pData);
				}
			}

		}

		/* close the file */
		fclose(fh);
	}

	return bSuccess;
}

unsigned short TZX_ReadShort(unsigned char **pPtr)
{
	unsigned short Data;
	unsigned char *pLocalPtr = *pPtr;

	Data = pLocalPtr[0] & 0x0ff;
	pLocalPtr++;
	Data |= ((pLocalPtr[0] & 0x0ff)<<8);
	pLocalPtr++;

	*pPtr = pLocalPtr;

	return Data;
}

unsigned char TZX_ReadByte(unsigned char **pPtr)
{
	unsigned char Data;
	unsigned char *pLocalPtr = *pPtr;

	Data = pLocalPtr[0] & 0x0ff;
	pLocalPtr++;

	*pPtr = pLocalPtr;

	return Data;
}

unsigned long TZX_ReadTriple(unsigned char **pPtr)
{
	unsigned long Data;
	unsigned char *pLocalPtr = *pPtr;

	Data = pLocalPtr[0] & 0x0ff;
	pLocalPtr++;
	Data |= ((pLocalPtr[0] & 0x0ff)<<8);
	pLocalPtr++;
	Data |= ((pLocalPtr[0] & 0x0ff)<<16);
	pLocalPtr++;

	*pPtr = pLocalPtr;

	return Data;
}

unsigned long TZX_ReadLong(unsigned char **pPtr)
{
	unsigned long Data;
	unsigned char *pLocalPtr = *pPtr;

	Data = pLocalPtr[0] & 0x0ff;
	pLocalPtr++;
	Data |= ((pLocalPtr[0] & 0x0ff)<<8);
	pLocalPtr++;
	Data |= ((pLocalPtr[0] & 0x0ff)<<16);
	pLocalPtr++;
	Data |= ((pLocalPtr[0] & 0x0ff)<<24);
	pLocalPtr++;

	*pPtr = pLocalPtr;

	return Data;
}

unsigned long TotalPause;

/* filename from current */
char prevHeader[64];
/* number of previous block or -1 if waiting for first block of new file (expectingBlockType is &2c), 
 or fetching data block of last file (expectingBlockType is &16) */
int prevBlockNumber;
/* type of the expected block (&2c is header, &16 is data) */
unsigned long expectingBlockType;
/* number of files encountered */
int NumFiles;

/* CRC code shamelessly taken from Pierre Guerrier's AIFF decoder! */
#define kCRCpoly  4129  /* used for binary long division in CRC */

/* CRC polynomial: X^16+X^12+X^5+1 */
unsigned int CRCupdate(unsigned int CRC, unsigned char new)
{
 unsigned int aux = CRC ^ (new << 8);
 int i;

 for(i=0; i<8; i++)
   if (aux & 0x8000)
       aux = (aux <<= 1) ^ kCRCpoly;
     else
       aux <<= 1;

 return(aux);
}

/* I am using a enum, so that I can poke data into structures without
worrying how the compiler has aligned it */
enum
{
	CPC_TAPE_HEADER_FILENAME_BYTE0 = 0,
	CPC_TAPE_HEADER_FILENAME_BYTE1,
	CPC_TAPE_HEADER_FILENAME_BYTE2,
	CPC_TAPE_HEADER_FILENAME_BYTE3,
	CPC_TAPE_HEADER_FILENAME_BYTE4,
	CPC_TAPE_HEADER_FILENAME_BYTE5,
	CPC_TAPE_HEADER_FILENAME_BYTE6,
	CPC_TAPE_HEADER_FILENAME_BYTE7,
	CPC_TAPE_HEADER_FILENAME_BYTE8,
	CPC_TAPE_HEADER_FILENAME_BYTE9,
	CPC_TAPE_HEADER_FILENAME_BYTE10,
	CPC_TAPE_HEADER_FILENAME_BYTE11,
	CPC_TAPE_HEADER_FILENAME_BYTE12,
	CPC_TAPE_HEADER_FILENAME_BYTE13,
	CPC_TAPE_HEADER_FILENAME_BYTE14,
	CPC_TAPE_HEADER_FILENAME_BYTE15,
	CPC_TAPE_HEADER_BLOCK_NUMBER,
	CPC_TAPE_HEADER_LAST_BLOCK_FLAG,
	CPC_TAPE_HEADER_FILE_TYPE,
	CPC_TAPE_HEADER_DATA_LENGTH_LOW,
	CPC_TAPE_HEADER_DATA_LENGTH_HIGH,
	CPC_TAPE_HEADER_DATA_LOCATION_LOW,
	CPC_TAPE_HEADER_DATA_LOCATION_HIGH,
	CPC_TAPE_HEADER_FIRST_BLOCK_FLAG,
	CPC_TAPE_HEADER_DATA_LOGICAL_LENGTH_LOW,
	CPC_TAPE_HEADER_DATA_LOGICAL_LENGTH_HIGH,
	CPC_TAPE_HEADER_DATA_EXECUTION_ADDRESS_LOW,
	CPC_TAPE_HEADER_DATA_EXECUTION_ADDRESS_HIGH,
} CPC_TAPE_HEADER_ENUM;

/* get number of records for data block from stored header block */
unsigned long Standard_NumRecords(const char *pHeader)
{
	unsigned long BlockLength;
	unsigned long NumRecords;

	/* length from header block */
	BlockLength = pHeader[CPC_TAPE_HEADER_DATA_LENGTH_LOW] & 0x0ff;
	BlockLength |= (pHeader[CPC_TAPE_HEADER_DATA_LENGTH_HIGH] & 0x0ff)<<8;

	/* a record is defined to be 256 bytes */
	/* all files are padded to a integer number of blocks in length */
	NumRecords =(BlockLength+255)>>8;

	return NumRecords;
}

/* get number of records for data block from stored header block */
unsigned long Standard_GetDataBlockLength(const char *pHeader)
{
	unsigned long BlockLength;

	/* length from header block */
	BlockLength = pHeader[CPC_TAPE_HEADER_DATA_LENGTH_LOW] & 0x0ff;
	BlockLength |= (pHeader[CPC_TAPE_HEADER_DATA_LENGTH_HIGH] & 0x0ff)<<8;

	return BlockLength;
}

BOOL	Standard_CheckLength(unsigned long TZXBlockLength, unsigned long ExpectedLength)
{
	/* a record is defined to be 256 bytes */
	/* all files are padded to a integer number of blocks in length */
	unsigned long NumRecords =(ExpectedLength+255)>>8;

	/* 256 data bytes and 2 crc bytes per record */
	unsigned long ExpectedTZXBlockLength = (NumRecords<<8)+(NumRecords<<1);

	/* if block length is less, then there is a definite error */
	/* first byte of TZX Block is sync byte which is not included in calculation */
	if ((TZXBlockLength-1)<ExpectedTZXBlockLength)
		return FALSE;

	/* if length is greater or equal, then tzx block length is valid */

	return TRUE;
}

BOOL	Standard_ChecksumBlock(unsigned char *pPtr, unsigned long NumRecords)
{
	/* initialise checksum */
	unsigned char DataByte;
	unsigned char *pLocalPtr = pPtr;
	unsigned int StoredCRC;
	int i,r;

	for (r=0; r<NumRecords; r++)
	{
		unsigned int CRC = 0x0ffff;
		for (i=0; i<256; i++)
		{
			DataByte = TZX_ReadByte(&pLocalPtr);
			CRC = CRCupdate(CRC, DataByte);
		}

		/* stored big endian and inverted */
		StoredCRC = ((TZX_ReadByte(&pLocalPtr)&0x0ff)^0x0ff)<<8;
		StoredCRC |= ((TZX_ReadByte(&pLocalPtr)&0x0ff)^0x0ff);

		if ((CRC & 0x0ffff)!=StoredCRC)
		{
			return FALSE;
		}
	}

	return TRUE;

}


unsigned char Standard_ToUpperCase(unsigned char ch)
{
	/* convert to upper case */
	if ((ch>='a') && (ch<='z'))
	{
		ch-=('a'-'A');
	}

	return ch;
}

void	Standard_DumpFilename(unsigned char *pPtr)
{
	unsigned char DataByte;
	unsigned char *pLocalPtr = pPtr;
	char Filename[18];
	int FilenameLength;
	int i;

	/* check for invalid characters in filename? */
	FilenameLength = 0;
	for (i=0; i<16; i++)
	{
		DataByte = TZX_ReadByte(&pLocalPtr);
		
		if (DataByte==0)
			break;

		/* printable character? */
		if ((DataByte<' ') || ((DataByte & 0x080)!=0))
		{
			/* not printable; display '.' instead */
			Filename[FilenameLength]='.';
		}
		else
		{
			/* display printable character */
			Filename[FilenameLength] = Standard_ToUpperCase(DataByte);
		}
		FilenameLength++;
	}

	Filename[FilenameLength] = '\0';

	if (FilenameLength==0)
	{
		OutputText("Unnamed file    ");
	}
	else
	{
		OutputText(Filename);

		for (i=0; i<16-FilenameLength; i++)
		{
			printf(" ");
		}
	}
}

/* compare two filenames; returns TRUE if they are correct; FALSE if they are different */
BOOL Standard_CompareFilenames(unsigned char *pPtrA, unsigned char *pPtrB)
{
	int index = 0;
	char chA, chB;

	do
	{
		/* get character */
		chA = pPtrA[index];
		chB = pPtrB[index];
		index++;

		chA = Standard_ToUpperCase(chA);
		chB = Standard_ToUpperCase(chB);

		/* this will exit if the characters are different. includes comparison against
		end of filename for one name, but not the other */
		if (chA!=chB) 
			return FALSE;

		/* gets to here if filenames are the same */
	}
	while ((chA!=0) && (index!=16));

	return TRUE;
}

BOOL Headerless_Detect(unsigned char *pPtr, unsigned long BlockLength, unsigned char BlockType)
{
	BOOL bHeaderless = FALSE;

	/* test for headerless; headerless uses same loader format as standard
	but is often stored as a continuous block */
	{
		BOOL bPotentiallyCut;

		/* calculate the number of possible records */
		/* each record is 256 bytes + 2 bytes CRC */
		unsigned long NumRecords = ((BlockLength-1)/258);
		/* length of all the records */
		unsigned long LengthForAllRecords = NumRecords*258;
		/* size */
		unsigned long EstimatedSize = NumRecords<<8;

		/* must have at least one record to qualify as headerless */
		if (NumRecords!=0)
		{
			bPotentiallyCut = FALSE;
			/* often there is 32 1's stored at the end, which equals 4 bytes */
			if ((BlockLength-LengthForAllRecords)>4)
			{
				/* potential problem */
				bPotentiallyCut = TRUE;
			}

			/* checksum the blocks */
			if (Standard_ChecksumBlock(pPtr, NumRecords))
			{
				bHeaderless = TRUE;
				/* checksum is ok */
				printf("Type: HEADERLESS ");
				printf("Sync: %02x ",BlockType);
				printf("Estimated Size: %04x ",EstimatedSize);
				printf(" Checksum: ");
				OutputText(Messages[5]);
				printf("\r\n");

				if (bPotentiallyCut)
				{
					OutputError(Messages[14]);
				}
				printf("---------------------------------------------------------\r\n");
			}
		}
	}

	return bHeaderless;
}


void	Standard_AnalyzeBlock(unsigned char *pPtr, unsigned long BlockLength)
{
	unsigned char BlockType;

	/* read block type */
	BlockType = TZX_ReadByte(&pPtr);

	/* if no prev block is defined... */
	if (expectingBlockType==0x016)
	{
		/* data block? */
		if (BlockType==0x02c)
		{
			/* yes. Data block is missing! */
			OutputError(Messages[1]);
		}
	}

	switch (BlockType)
	{
		case 0x02c:
		{
			if (!Standard_CheckLength(BlockLength, 64))
			{
				OutputError(Messages[7]);
			}
			else
			{
				BOOL bSeperator = FALSE;

				unsigned char *pLocalPtr = pPtr;
				char TapeHeader[64];
				int i;

				for (i=0; i<64; i++)
				{
					TapeHeader[i] = TZX_ReadByte(&pLocalPtr);
				}

				expectingBlockType = 0x016;

				if (prevBlockNumber!=-1)
				{
					/* not first block */

					/* compare filenames */
					if (Standard_CompareFilenames(&prevHeader[0], &TapeHeader[0]))
					{
						/* filenames are the same */

						/* seen other blocks; check index etc */
						if (TapeHeader[CPC_TAPE_HEADER_BLOCK_NUMBER]!=((prevBlockNumber+1) & 0x0ff))
						{
							OutputError(Messages[9]);

							/* attempt a recovery */
							/* if subsequent blocks of the next file exist, allow them to proceed without errors, so that
							the only errors are those where blocks are missing */
							prevBlockNumber = TapeHeader[CPC_TAPE_HEADER_BLOCK_NUMBER] & 0x0ff;
							memcpy(prevHeader, TapeHeader, 64);

						}
						else
						{
							/* last block of file? */
							if (TapeHeader[CPC_TAPE_HEADER_LAST_BLOCK_FLAG]!=0)
							{
								/* no longer expecting more blocks from this file */
								prevBlockNumber = -1;
								memcpy(prevHeader, TapeHeader, 64);
							}
							else
							{
								prevBlockNumber = (TapeHeader[CPC_TAPE_HEADER_BLOCK_NUMBER]&0x0ff);
								memcpy(prevHeader, TapeHeader, 64);
							}
						}
					}
					else
					{
						bSeperator = TRUE;

						OutputError(Messages[10]);
			
						/* attempt a recovery */
						/* if subsequent blocks of the next file exist, allow them to proceed without errors, so that
						the only errors are those where blocks are missing */
						prevBlockNumber = TapeHeader[CPC_TAPE_HEADER_BLOCK_NUMBER] & 0x0ff;
						memcpy(prevHeader, TapeHeader, 64);
					}

				}
				else
				{
					/* copy header */
					memcpy(prevHeader, TapeHeader, 64);
					/* no. expecting more blocks from this file */
					prevBlockNumber = TapeHeader[CPC_TAPE_HEADER_BLOCK_NUMBER] & 0x0ff;

					if (TapeHeader[CPC_TAPE_HEADER_FIRST_BLOCK_FLAG]==0) 
					{
						/* first block flag not set! */

						/* error */
						OutputError(Messages[8]);			
					}
					else
					{
						/* check length of data block is less than or equal to 2K */
						if (Standard_GetDataBlockLength(TapeHeader)>2048)
						{
							/* greater than 2048! Error. CPC would give a 'read error d' in this case */
							OutputError(Messages[14]);							
						}
					}
				}
				/* last block? */
				if (TapeHeader[CPC_TAPE_HEADER_LAST_BLOCK_FLAG]!=0)
				{
					prevBlockNumber = -1;
				}

				/* last block of file */
//				if (bSeperator)
//				{
//					printf("- - - - - - - - - - - - - - - - - - - - - - - - - - - - -\r\n");
//				}

				/* header block */
				OutputText("Type: HEADER ");

				OutputText("Filename: ");
				Standard_DumpFilename(pPtr);

				printf("  ");

				OutputText("Block: ");
				printf("%3d",TapeHeader[CPC_TAPE_HEADER_BLOCK_NUMBER]&0x0ff);
				printf(" ");

				OutputText("Checksum: ");

				if (Standard_ChecksumBlock(pPtr, 1))
				{ 
					OutputText(Messages[5]);
				}
				else
				{
					OutputText(Messages[6]);
				}
				OutputText("\r\n");



//				if (TotalPause<)
//				{
//					OutputText(Messages[12]);
//				}

			}
		}
		break;

		case 0x016:
		{

			if (expectingBlockType==0x016)
			{
				/* this block was expected */

				/* check against the size defined by a previous header */
				if (!Standard_CheckLength(BlockLength, Standard_GetDataBlockLength(prevHeader)))
				{
					/* report the error */
					OutputError(Messages[12]);
				}
				else
				{
					unsigned long NumRecords;

					if (prevHeader[CPC_TAPE_HEADER_FIRST_BLOCK_FLAG]!=0)
					{
						if (Standard_GetDataBlockLength(prevHeader)>2048)
						{
							OutputError(Messages[15]);							
						}
					}

					/* data block */
					OutputText("Type: DATA   ");

					/* calculate the number of records from the previous header */
					NumRecords = Standard_NumRecords(prevHeader);
					
					OutputText("Checksum: ");

					if (Standard_ChecksumBlock(pPtr, NumRecords))
					{ 
						OutputText(Messages[5]);
					}
					else
					{
						OutputText(Messages[6]);
					}
					OutputText("\r\n");
				}

				/* last block of file */
				if (prevBlockNumber==-1)
				{
					printf("---------------------------------------------------------\r\n");
				}

			}
			else
			{
				/* if no prev block is defined... */
				if (expectingBlockType==0x02c)
				{
					/* if block index is defined then a header must be missing! */
					if (prevBlockNumber!=-1)
					{
						/* yes. Header block is missing! */
						OutputError(Messages[0]);
					}
					else
					{
						/* no block index defined, the code is looking for the first block 
						of a file, but could equally find a headerless block with a sync byte
						of 0x016!! */

						/* therefore there is a potential that a header block is missing */
						OutputError(Messages[0]);
					}
				}

				/* attempt to detect headerless */
				if (!Headerless_Detect(pPtr, BlockLength,BlockType))
				{
					/* block not recognised */
					OutputError(Messages[16]);					
				}
			}

			expectingBlockType = 0x02c;

		}
		break;

		default:
		{
			
			/* attempt to detect headerless */
			if (!Headerless_Detect(pPtr, BlockLength,BlockType))
			{
				/* sync not recognised */
				OutputError(Messages[11]);
			}

			expectingBlockType = 0x02c;
		}
		break;
	}
}



int main(int argc, char **argv)
{
	if (argc==2)
	{
		unsigned char *pData;
		unsigned long DataLength;

		if (ReadTZXFile(argv[1], &pData, &DataLength))
		{
			// header matches
			if (memcmp(pData, "ZXTape!\x1a",8)==0)
			{
				unsigned char *pPtr;

				pPtr = pData+10;

				/* searching for first block */
				NumFiles = 0;

				prevBlockNumber = -1;
				TotalPause = 0;
				expectingBlockType = 0x02c;

				do
				{
					unsigned char BlockID;

					BlockID = TZX_ReadByte(&pPtr);

					switch (BlockID)
					{
						case 0x010:
						{
							unsigned short PauseAfterBlock;
							unsigned long Length;

							PauseAfterBlock = TZX_ReadShort(&pPtr);
							Length = TZX_ReadShort(&pPtr);

							Standard_AnalyzeBlock(pPtr,Length);

							pPtr+=Length;

							// set new pause
							TotalPause = Length;

						}
						break;

						case 0x011:
						{
							unsigned short PilotPulse;
							unsigned short Sync0Pulse;
							unsigned short Sync1Pulse;
							unsigned short ZeroPulse;
							unsigned short OnePulse;
							unsigned short PilotTone;
							unsigned char UsedBitsInLastByte;
							unsigned short PauseAfterBlock;
							unsigned long Length;

							PilotPulse= TZX_ReadShort(&pPtr);
							Sync0Pulse= TZX_ReadShort(&pPtr);
							Sync1Pulse= TZX_ReadShort(&pPtr);
							ZeroPulse= TZX_ReadShort(&pPtr);
							OnePulse= TZX_ReadShort(&pPtr);
							PilotTone= TZX_ReadShort(&pPtr);
							UsedBitsInLastByte= TZX_ReadByte(&pPtr);
							PauseAfterBlock= TZX_ReadShort(&pPtr);
							Length = TZX_ReadTriple(&pPtr);

							if (Length==0)
							{
								OutputText(Messages[4]);
							}
							else
							{
								Standard_AnalyzeBlock(pPtr,Length);
							}
							
							pPtr+=Length;

							// set new pause
							TotalPause = PauseAfterBlock;
						}
						break;

						case 0x020:
						{
							unsigned short Pause;

							Pause = TZX_ReadShort(&pPtr);

							TotalPause += Pause;
						}
						break;

					}
				}
				while (((unsigned long)pPtr-(unsigned long)pData)<DataLength);

				/* was expecting a block! */
				if (prevBlockNumber!=-1)
				{
					if (expectingBlockType==0x02c)
					{
						/* yes. Header block is missing! */
						OutputError(Messages[0]);
					}
					/* if no prev block is defined... */
					if (expectingBlockType==0x016)
					{
						/* yes. Data block is missing! */
						OutputError(Messages[1]);
					}
				}
				else
				{

					/* if no prev block is defined... */
					if (expectingBlockType==0x016)
					{
						/* yes. Header block is missing! */
						OutputError(Messages[1]);
					}
				}
			}

			free(pData);
		}			
	}
}

