/*
    This program will process a .map file generated by SDCC, and create an 
    output as a AS compatible assembler file or an ASMLNK compatible OBJ file 
    with the global labels defined in it.
    
    This file can then be assembled/linked into other projects.
    
    The idea is that the main kernel of an operating system is built, and the
    corresponding .map file is converted with this tool.
    
    Any utility which uses the kernel then links to the output file, generated 
    by this program.
    
    (c) Copyright Kevin Thacker 2005.
    
    Released under the GPL V2 or later.

    Assumes hex radix.

    Will only accept AREAS named '_CODE' and '_DATA'.
    This will limit the labels that are output.
    */
#ifndef NULL
#define NULL 0
#endif

/* Map file has a structure defined by tab characters.*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <getopt.h>

const char *AREA_TOKEN="AREA";
const char *GLOBALS_TOKEN="GLOBALS";

int num_valid_areas;
const char *valid_areas[256];
int num_excluded_labels;
const char *excluded_labels[256];

int num_files;
const char *files[256];

enum
{
    /* looking for 'AREA' token in first column */
    STATE_LOOKING_FOR_AREA = 0,
    /* looking for 'GLOBALS' token within an AREA */
    STATE_LOOKING_FOR_GLOBALS = 1,
    /* reading labels from within a 'GLOBALS' */
    STATE_READING_LABELS = 2,
};

int main(int argc, char **argv)
{
    FILE *fhIn, *fhOut;
    char c;
    int i;

    /* areas which we will accept; none defined means we include all */
    num_valid_areas = 0;
    /* labels to exclude; none defined means we include all */
    num_excluded_labels = 0;
	
    num_files = 0;
    do
      {
	c = getopt(argc, argv,"a:x:");
	switch (c)
	  {
	  case 'a':
	    {
	      valid_areas[num_valid_areas] = optarg;
	      num_valid_areas++;
	    }
	    break;
	  case 'x':
	    {
	      excluded_labels[num_excluded_labels] = optarg;
	      num_excluded_labels++;
	    }
	    break;
	  default:
	    break;
	  }

      }
    while (c!=-1);
    for (i=optind; i<argc; i++)
    {
      files[num_files] = argv[i];
      num_files++;
    }
    
    if (num_files!=2)
    {
        printf("Usage: make_labels [.map file] [.output file]\n");
    }
    else
    {
        int current_state = STATE_LOOKING_FOR_AREA;
        int output_type = 0;
        
        /* open map file */
        fhIn = fopen(files[0],"r");    
        
        if (fhIn!=NULL)
        {
			char * chrpos;
			
			chrpos = strrchr(files[1],'.');
			if (chrpos!=NULL)
			{
				/* if output filename has s/asm extension, then output as assembler */
				/* if output filename has obj/rel extension, then output as obj/rel for linking */
				if (
					(strcmp(chrpos+1,"s")==0) ||
					(strcmp(chrpos+1,"asm")==0))
					{
						output_type = 0;	
						
					}
				else
				if (
					(strcmp(chrpos+1,"rel")==0) ||
					(strcmp(chrpos+1,"o")==0) ||
				    (strcmp(chrpos+1,"obj")==0))
				    {
					    output_type = 1;
				    }
				    			
			}
			
	        fhOut = fopen(files[1],"w");
            
            if (fhOut!=NULL)
            {
                /* holds a line from the file */
                char line[256];
                char *result;
                int global_accepted = 0;
                int area_accepted = 0;
             
                if (output_type==1)
                {
	            	fprintf(fhOut,"XL\n");    	                
                }
                   
                do
                {
                    int line_length;
                    int tab_count;
                    int i;
                    
                    /* try to read a line from the input file */
                    result = fgets(line, sizeof(line), fhIn);
                
                    if (result==NULL)
                        break;
                         
                    line_length = strlen(line);
 
                    /* remove CR,LF characters from end of line,
                    and adjust line length appropiatly */                   
                    for (i = line_length-1; i>=0; i--)
                    {
                        if ((line[i]=='\r') || (line[i]=='\n'))
                        {
                            line[i] = '\0';
                            line_length--;                               
                        }    
                        else
                        {
                            break;
                        }
                    }
                    

                    /* count tabs at start of line */
                    tab_count = 0;
                    for (i=0; i<line_length; i++)
                    {
                        if (line[i]=='\t')
                        {
                            tab_count++;
                        }
                        else
                        {
                            break;                        
                        }
                    }
                      
                    switch (current_state)
                    {
                        case STATE_LOOKING_FOR_AREA:
                        {
			
                            /* previously found an AREA marker. Next line has
                            a single tab so could be a GLOBALS marker. */
                            if ((area_accepted==1) && (tab_count==1))
                            {
	                            global_accepted = 0;
                                current_state = STATE_LOOKING_FOR_GLOBALS;    
                            }  
                            /*Otherwise remain in state looking for area 
                            marker */
                        }
                        break;
                        
                        case STATE_LOOKING_FOR_GLOBALS:
                        {
                            /* previously found a GLOBALS marker. Next line has
                            2 tabs so could be first line of labels list */
                            if ((global_accepted==1) && (tab_count==2))
                            {
                                current_state = STATE_READING_LABELS;                       
                            }
                            else
                            {
                                /* if this line has no tabs, then revert
                                back to looking for AREA marker */
                                if (tab_count==0)
                                {
                                    area_accepted = 0;
                                    current_state = STATE_LOOKING_FOR_AREA; 
                                }
                            }
                            
                        }
                        break;
                        
                        case STATE_READING_LABELS:
                        {
                            /* depending on tab count, change state */
                            /* if line has 2 tabs then keep reading labels */
                            global_accepted = 0;
                            
                            if (tab_count==0)
                            {
                                area_accepted = 0;
                                current_state = STATE_LOOKING_FOR_AREA;
                            }
                            else
                            if (tab_count==1)
                            {
	                            global_accepted = 0;
                                current_state = STATE_LOOKING_FOR_GLOBALS;    
                            }                            
                        }
                        break;
                        
                    }
                    
                    switch (current_state)
                    {
                        case STATE_LOOKING_FOR_AREA:
                        {
                            int area_token_length = strlen(AREA_TOKEN);

			/* check that the length of the line is large
                            enough for the area token, plus a space, plus at
                            least one character identifying the area */
                            if (line_length>(area_token_length+2))  
                            {
	                            if (line[area_token_length]==' ')
	                            {
	                                /* area token? */
	                                if (strncmp(line, AREA_TOKEN, area_token_length)==0)
	                                {                         
	                                    /* extract area name */
	                                    int i;
					    char *spacepos;
	                                    char *area_name;
	 
	                                    area_name = line+area_token_length+1;
	 
					    spacepos = strchr(area_name, ' ');
	 
	                                    if (spacepos!=NULL)
	                                    {
	                                        /* area name has something after it */
	                                        spacepos[0] = '\0';                                           
	                                    }

					    /* compare area name against list of areas we accept */
					    for (i=0; i<num_valid_areas; i++)
					    {
						if (strcmp(valid_areas[i], area_name)==0)
						  {
						    area_accepted = 1;
						    break;


						  }
					    }

					}                               
	                            }                         
                            }
                        }
                        break;
                        
                        case STATE_LOOKING_FOR_GLOBALS:
                        {
                            /* precondition: must have 1 tab */
                            if (line_length==(strlen(GLOBALS_TOKEN)+tab_count))
                            {
                                if (strcmp(line+tab_count, GLOBALS_TOKEN)==0)
                                {
                                    global_accepted = 1;
                                }
                            }                           
                        }
                        break;
                        
                        case STATE_READING_LABELS:
                        {
                            /* precondition: tab count = 2 */
                            
                            /* must have at least two characters on the line */
                            if (line_length>2)
                            {
                                char *chrpos;
                                
                                /* find first occurance of tab; this will
                                find tab character which seperates label
                                and it's value */
                                chrpos= strchr(line+tab_count,'\t');
                                
                                if (chrpos!=NULL)
                                {
				  int label_excluded;
                                    char *label;
                                    char *value;
                                    
                                    chrpos[0] = '\0';
 
                                    label = line+tab_count;
                                    value = chrpos+1;

                                    /* look for spaces in label name; usually where
                                    map file contains a label type marker e.g. .ABS. */
                                    chrpos = strchr(label,' ');
                                    if (chrpos!=NULL)
                                    {
	                                 	chrpos[0] = '\0';   
                                    }

#if 0
                                    /* for debugging */
                                    printf("%s = 0x%s\n",label,value); 
#endif
				    label_excluded = 0;

				    for (i=0; i<num_excluded_labels; i++)
				      {
					if (strcmp(excluded_labels[i],label)==0)
					  {
					    label_excluded = 1;
					    break;
					  }


				      }

#if 0
				    /* debugging */
				    if (label_excluded)
				      printf("label excluded\n");
#endif
				    
				    if (!label_excluded)
				      {
                                    if (output_type==0)
                                    {
										fprintf(fhOut,"%s == 0x%s\n",label,value);
									}
									else
									{
										fprintf(fhOut,"S %s Def%s\n",label,value);

									}
				      }
                                }
                            }
                        }
                        break;
                    }
                                      
                }
                while (result!=NULL);
                
                fclose(fhOut);    
            }
            else
            {
                printf("failed to open output file\n");               
            }
        
            fclose(fhIn);    
        }
        else
        {
           printf("failed to open input file\n");    
        }
    }
}

