/*
 *  ADDHEAD utility (c) Copyright, Kevin Thacker 2001-2005
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/* This utility can be used to add a AMSDOS header to a file. The file will
 * be treated by the Amstrad operating system as binary. The resulting file
 * can be injected into a disk image.
 *
 * This utility is ideal as part of a cross-development tool.
*/
#include "opth.h"
#include <stdio.h>
#include <stdlib.h>
#include <memory.h>

/* I am using a enum, so that I can poke data into structures without
worrying how the compiler has aligned it */
enum
{
	CPC_DISC_HEADER_FILENAME_BYTE0 = 0,
	CPC_DISC_HEADER_FILENAME_BYTE1,
	CPC_DISC_HEADER_FILENAME_BYTE2,
	CPC_DISC_HEADER_FILENAME_BYTE3,
	CPC_DISC_HEADER_FILENAME_BYTE4,
	CPC_DISC_HEADER_FILENAME_BYTE5,
	CPC_DISC_HEADER_FILENAME_BYTE6,
	CPC_DISC_HEADER_FILENAME_BYTE7,
	CPC_DISC_HEADER_FILENAME_BYTE8,
	CPC_DISC_HEADER_FILENAME_EXTENSION0,
	CPC_DISC_HEADER_FILENAME_EXTENSION1,
	CPC_DISC_HEADER_FILENAME_EXTENSION2,
	CPC_DISC_HEADER_FILE_TYPE = 18,
	CPC_DISC_HEADER_LENGTH_LOW,
	CPC_DISC_HEADER_LENGTH_HIGH,
	CPC_DISC_HEADER_LOCATION_LOW,
	CPC_DISC_HEADER_LOCATION_HIGH,
	CPC_DISC_HEADER_FIRST_BLOCK_FLAG,
	CPC_DISC_HEADER_LOGICAL_LENGTH_LOW,
	CPC_DISC_HEADER_LOGICAL_LENGTH_HIGH,
	CPC_DISC_HEADER_EXECUTION_ADDRESS_LOW,
	CPC_DISC_HEADER_EXECUTION_ADDRESS_HIGH,
	CPC_DISC_HEADER_DATA_LENGTH_LOW = 64,
	CPC_DISC_HEADER_DATA_LENGTH_MID,
	CPC_DISC_HEADER_DATA_LENGTH_HIGH,
	CPC_DISC_HEADER_CHECKSUM_LOW,
	CPC_DISC_HEADER_CHECKSUM_HIGH
} CPC_DISC_HEADER_ENUM;

/*--------------------*/
/* calculate checksum */
static unsigned short AMSDOSHeader_CalculateChecksum(const char *pData)
{
	unsigned short CalculatedChecksum = 0;
	unsigned long i;

	/* generate checksum */
	for (i=0; i<66; i++)
	{
		CalculatedChecksum+=(unsigned short)(pData[i]&0x0ff);
	}

	return CalculatedChecksum;
}

/*------------------------------------------------*/
/* Detect if there is a AMSDOS header on the file */
static int AMSDOSHeader_Checksum(const char *pData)
{
	unsigned short StoredChecksum;
	unsigned short CalculatedChecksum;

	CalculatedChecksum = AMSDOSHeader_CalculateChecksum(pData);

	/* get the stored checksum */
	StoredChecksum = (unsigned short)((pData[67]&0x0ff)|((pData[68]&0x0ff)<<8));

    printf("%04x %04x\n",CalculatedChecksum, StoredChecksum);


	return (CalculatedChecksum==StoredChecksum);
}

/* buffer to hold generated header data */
static char HeaderBuffer[128];
static int nExecutionAddress = 0;
static int nStartAddress = 0;
static int nFileType = 2;

/*------------------------------------------------*/
static void AMSDOSHeader_Initialise(char *pData, unsigned long Length)
{
	unsigned short CalculatedChecksum;

	memset(pData, 0, 128);

	switch (nFileType)
	{
		case 2:
		{
			pData[CPC_DISC_HEADER_FILE_TYPE] = 2;
		}
		break;

		default:
            break;
	}
	pData[CPC_DISC_HEADER_FIRST_BLOCK_FLAG] = (char)(0x0ff);

	pData[CPC_DISC_HEADER_DATA_LENGTH_LOW] = (char)(Length&0x0ff);
	pData[CPC_DISC_HEADER_DATA_LENGTH_MID] = (char)((Length>>8)&0x0ff);
	pData[CPC_DISC_HEADER_DATA_LENGTH_HIGH] = (char)((Length>>16)&0x0ff);

	pData[CPC_DISC_HEADER_LOCATION_LOW] = (char)(nStartAddress&0x0ff);
	pData[CPC_DISC_HEADER_LOCATION_HIGH] = (char)((nStartAddress>>8)&0x0ff);

	pData[CPC_DISC_HEADER_LENGTH_LOW] = pData[CPC_DISC_HEADER_DATA_LENGTH_LOW];
	pData[CPC_DISC_HEADER_LENGTH_HIGH] = pData[CPC_DISC_HEADER_DATA_LENGTH_MID];

	pData[CPC_DISC_HEADER_LOGICAL_LENGTH_LOW] = pData[CPC_DISC_HEADER_DATA_LENGTH_LOW];
	pData[CPC_DISC_HEADER_LOGICAL_LENGTH_HIGH] = pData[CPC_DISC_HEADER_DATA_LENGTH_MID];

	pData[CPC_DISC_HEADER_EXECUTION_ADDRESS_LOW] = (nExecutionAddress&0x0ff);
	pData[CPC_DISC_HEADER_EXECUTION_ADDRESS_HIGH] = (nExecutionAddress>>8)&0x0ff;

	CalculatedChecksum = AMSDOSHeader_CalculateChecksum(pData);

	pData[CPC_DISC_HEADER_CHECKSUM_LOW] = (char)(CalculatedChecksum&0x0ff);
	pData[CPC_DISC_HEADER_CHECKSUM_HIGH] = (char)((CalculatedChecksum>>8)&0x0ff);
}

static int NumFiles = 0;
static char *Filenames[2];

int	NonOptionHandler(const char *pOption)
{
	if (NumFiles<2)
	{
		Filenames[NumFiles] = pOption;
		NumFiles++;
	}

	return OPTION_OK;
}

static int bAddHeader;

int AddHeaderOption(ARGUMENT_DATA *pData)
{
	bAddHeader = 1;

	return OPTION_OK;
}

int RemoveHeaderOption(ARGUMENT_DATA *pData)
{
	bAddHeader = 0;

	return OPTION_OK;
}

int SetFileTypeOption(ARGUMENT_DATA *pData)
{
	const char *pAddress = ArgumentList_GetNext(pData);

	if (pAddress==NULL)
		return OPTION_MISSING_PARAMETER;

	if (strcmp(pAddress,"binary")==0)
		nFileType=2;
	else if (strcmp(pAddress,"basic")==0)
		nFileType=1;

	return OPTION_OK;
}

int SetExecutionAddressOption(ARGUMENT_DATA *pData)
{
	const char *pAddress = ArgumentList_GetNext(pData);

	if (pAddress==NULL)
		return OPTION_MISSING_PARAMETER;

	nExecutionAddress = atoi(pAddress);

	return OPTION_OK;
}

int SetStartAddressOption(ARGUMENT_DATA *pData)
{
	const char *pAddress = ArgumentList_GetNext(pData);

	if (pAddress==NULL)
		return OPTION_MISSING_PARAMETER;

	nStartAddress = atoi(pAddress);

	return OPTION_OK;
}

OPTION OptionTable[]=
{
	{"a",AddHeaderOption},
	{"r",RemoveHeaderOption},
	{"s",SetStartAddressOption},
	{"t",SetFileTypeOption},
	{"x",SetExecutionAddressOption},
	{NULL, NULL},
};

void	OutputDetails(void)
{
	fprintf(stdout,"ADDHEAD\n");
	fprintf(stdout,"\n");
	fprintf(stdout,"(c) Kevin Thacker 2001-2005\n");
	fprintf(stdout,"A utility to add/remove a AMSDOS header to a file.\n");
	fprintf(stdout,"\n");
	fprintf(stdout,"Usage: ADDHEAD <switches> <input filename> <output filename>\n");
	fprintf(stdout,"\n");
	fprintf(stdout,"-a     - add a header to file\n");
	fprintf(stdout,"-r     - remove a existing header\n");
	fprintf(stdout,"-t     - set file type (\"binary\" (default), \"basic\"");
	fprintf(stdout,"-x     - set execution address (decimal)\n");
	fprintf(stdout,"-s     - set start address (decimal)\n");
	fprintf(stdout,"\n");
}

int	main(int argc, char *argv[])
{
	bAddHeader = 1;
	nFileType = 2;
	nExecutionAddress = 0;

	NumFiles = 0;

	if (ArgumentList_Execute(argc, argv, OptionTable, printf, NonOptionHandler)==OPTION_OK)
	{
		if (NumFiles!=2)
		{
			fprintf(stderr,"Not enough files specified!\n");
		}
		else
		{

			FILE *fh_in, *fh_out;
			unsigned long filesize;

			/* open input binary file */
			fh_in = fopen(Filenames[0],"rb");

			if (fh_in==NULL)
			{
				printf("Failed to open input file\n");
			}
			else
			{
				unsigned char *file_data;

				/* seek to end of file */
				fseek(fh_in, 0, SEEK_END);

				/* report position in file */
				filesize = ftell(fh_in);

				/* seek to beginning of file */
				fseek(fh_in, 0,SEEK_SET);

				file_data = (unsigned char *)malloc(filesize);

				if (file_data!=NULL)
				{
					fread(file_data, 1, filesize, fh_in);
				}
				else
				{
					printf("Failed to allocate memory for input file.\n");
				}

				fclose(fh_in);

				if (file_data!=NULL)
				{
					if (!bAddHeader)
					{
						/* removing a existing header */

						/* file must have at least the size of the header */
						if (filesize>=128)
						{
							/* calculate checksum and compare against stored checksum */
							/* if valid, then file is assumed to have a header */
							if (AMSDOSHeader_Checksum((const char *)file_data))
							{
								printf("AMSDOS header detected.\n");

								/* now open output file */
								fh_out = fopen(Filenames[1],"wb");

								if (fh_out==NULL)
								{
									printf("Failed to open output file.\n");
								}
								else
								{
									/* copy remaining data to output */
									fwrite(file_data+128, 1, filesize-128, fh_out);

									fclose(fh_out);

									printf("Output file written.\n");
								}

							}
							else
							{
								printf("The input file does not have a AMSDOS header.\n");
							}
						}
						else
						{
							printf("The input file is too small to have a AMSDOS header.\n");
						}
					}
					else
					{
						/* adding a header */

						/* now open output file */
						fh_out = fopen(Filenames[1],"wb");

						if (fh_out==NULL)
						{
							printf("Failed to open output file.\n");
						}
						else
						{
							AMSDOSHeader_Initialise(HeaderBuffer, filesize);

							/* write header to output file */
							fwrite(HeaderBuffer, 1, 128, fh_out);

							/* copy remaining data to output */
							fwrite(file_data, 1, filesize, fh_out);

							/* close output file */
							fclose(fh_out);

							printf("Output file written.\n");

						}

						/* free memory allocated for input file */
						free(file_data);
					}
				}
			}
		}
	}

	exit(0);
}

