/* 
 *  Art Studio Picture Viewer (c) Copyright, Kevin Thacker 2003
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#include <stdio.h>
#include <stdlib.h>
#include <memory.h>
#include "image.h"
#include "filehelp.h"
#include "artstud.h"
#include "amsdos.h"

int ArtStudioViewer_GetMode(const char *pPalette);

typedef struct
{
	int  InkIndexs[8];
} PixelGroup;

char DefaultPalette[]=
{
	1, 
	0x0ff, 
	0x019,
	0x04b,0x04b,0x04b,0x04b,0x04b,0x04b,0x04b,0x04b,0x04b,0x04b,0x04b,0x04b,
	0x054,0x054,0x054,0x054,0x054,0x054,0x054,0x054,0x054,0x054,0x054,0x054,
	0x04c,0x04c,0x04c,0x04c,0x04c,0x04c,0x04c,0x04c,0x04c,0x04c,0x04c,0x04c,
	0x055,0x055,0x055,0x055,0x055,0x055,0x055,0x055,0x055,0x055,0x055,0x055,
	0x044,0x044,0x044,0x044,0x044,0x044,0x044,0x044,0x044,0x044,0x044,0x044,
	0x04a,0x04a,0x04a,0x04a,0x04a,0x04a,0x04a,0x04a,0x04a,0x04a,0x04a,0x04a,
	0x053,0x053,0x053,0x053,0x053,0x053,0x053,0x053,0x053,0x053,0x053,0x053,
	0x04d,0x04d,0x04d,0x04d,0x04d,0x04d,0x04d,0x04d,0x04d,0x04d,0x04d,0x04d,
	0x046,0x046,0x046,0x046,0x046,0x046,0x046,0x046,0x046,0x046,0x046,0x046,
	0x05e,0x05e,0x05e,0x05e,0x05e,0x05e,0x05e,0x05e,0x05e,0x05e,0x05e,0x05e,
	0x047,0x047,0x047,0x047,0x047,0x047,0x047,0x047,0x047,0x047,0x047,0x047,
	0x052,0x052,0x052,0x052,0x052,0x052,0x052,0x052,0x052,0x052,0x052,0x052,
	0x045,0x045,0x045,0x045,0x045,0x045,0x045,0x045,0x045,0x045,0x045,0x045,
	0x057,0x057,0x057,0x057,0x057,0x057,0x057,0x057,0x057,0x057,0x057,0x057,
	0x05c,0x05c,0x05c,0x05c,0x05c,0x05c,0x05c,0x05c,0x05c,0x05c,0x05c,0x05c,
	0x04b,0x04b,0x04b,0x04b,0x04b,0x04b,0x04b,0x04b,0x04b,0x04b,0x04b,0x04b,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
};



/* base hardware colours */
const unsigned char AmstradPalette[32*3] =
{
		0x060,0x060,0x060,
        0x060,0x060,0x060,
		0x00,0x0ff,0x060, 
        0x0ff,0x0ff,0x060,
        0x00,0x00,0x060,  
        0x0ff,0x00,0x060, 
        0x00,0x060,0x060, 
        0x0ff,0x060,0x060,
        0x0ff,0x00,0x060, 
        0x0ff,0x0ff,0x060,
        0x0ff,0x0ff,0x00, 
        0x0ff,0x0ff,0x0ff,
        0x0ff,0x00,0x00,  
        0x0ff,0x00,0x0ff, 
        0x0ff,0x060,0x00, 
        0x0ff,0x060,0x0ff,
        0x00,0x00,0x060,  
        0x00,0x0ff,0x060, 
        0x00,0x0ff,0x00,  
        0x00,0x0ff,0x0ff, 
        0x00,0x00,0x00,   
        0x00,0x00,0x0ff,  
        0x00,0x060,0x00,  
        0x00,0x060,0x0ff, 
        0x060,0x00,0x060, 
        0x060,0x0ff,0x060,
        0x060,0x0ff,0x00, 
        0x060,0x0ff,0x0ff,
        0x060,0x00,0x00,  
        0x060,0x00,0x0ff, 
        0x060,0x060,0x00, 
        0x060,0x060,0x0ff 
};


/* loaded .SCR file */
static char *pLoadedImage;
static unsigned long LoadedImageLength;
/* loaded .PAL file */
static char *pLoadedPalette;
static unsigned long LoadedPaletteLength;

static int CurrentMode;

static char *pDecompressedImage;

static char *pActivePalette;

void ArtStudioViewer_Init(void)
{

	pDecompressedImage = NULL;

	pLoadedImage = NULL;
	LoadedImageLength = 0;
	pLoadedPalette = NULL;
	LoadedPaletteLength = 0;
}

void ArtStudioViewer_Free(void)
{
	if (pDecompressedImage!=NULL)
	{
		free(pDecompressedImage);
		pDecompressedImage = NULL;
	}

	if (pLoadedImage!=NULL)
	{
		free(pLoadedImage);
		pLoadedImage = NULL;
	}
	LoadedImageLength = 0;

	if (pLoadedPalette!=NULL)
	{
		free(pLoadedPalette);
		pLoadedPalette = NULL;
	}
	LoadedPaletteLength = 0;
}

void ArtStudioViewer_GetDimensions(int *pWidth, int *pHeight)
{
	*pWidth = 640;
	*pHeight = 200;
}





/* decompress art studio image data from source buffer into destination buffer */
void	DecompressData(const unsigned char *pSource, const unsigned long SourceDataLength, unsigned char *pDestination, const unsigned long DestDataLength)
{
	unsigned char databyte;
	unsigned char *pSrc = pSource;
	unsigned char *pDest = pDestination;
	unsigned long SrcBytesRemaining = SourceDataLength;
	unsigned long DestBytesRemaining = DestDataLength;
	int State;
	int Count;
	int Byte;
	int dataLength;
	int count;
	
	dataLength = 0;
	State = 0;
	while ((SrcBytesRemaining!=0) && (DestBytesRemaining!=0))
	{

		databyte = pSrc[0];
		pSrc++;
		SrcBytesRemaining--;

		switch (State)
		{
			case 0:
			{
				if (databyte=='M')
				{
					State++;
				}
			}
			break;

			case 1:
			{
				if (databyte=='J')
				{
					State++;
				}
			}
			break;

			case 2:
			{
				if (databyte=='H')
				{
					State++;
				}
			}
			break;

			case 3:
			{
				dataLength &=~0x0ff;
				dataLength|=databyte & 0x0ff;
				State++;
			}
			break;

			case 4:
			{
				dataLength&=~0x0ff00;
				dataLength|=((databyte & 0x0ff)<<8);
				State++;
			}
			break;

			case 5:
			{
				if (databyte==0x01)
				{
					State++;
				}
				else
				{
					pDest[0] = databyte;
					pDest++;
					DestBytesRemaining--;

					dataLength--;

					if (dataLength==0)
						State = 0;

				}
			}
			break;

			case 6:
			{
				count = databyte&0x0ff;
				if (count==0)
					count = 256;
				State++;
			}
			break;

			case 7:
			{
				int i;

				for (i=0; i<count; i++)
				{
					pDest[0] = databyte;
					pDest++;
					DestBytesRemaining--;
					dataLength--;

					if (dataLength==0)
						break;

					if (DestBytesRemaining==0)
						break;
				}

				State = 5;

				if (dataLength==0)
					State = 0;

			}
			break;

		}
	}
}

int ArtStudioViewer_GetMode(const char *pPalette)
{
	int Mode;

	Mode = pPalette[0]&0x0ff;

	if ((Mode<0) || (Mode>2))
		Mode = 0;

	return Mode;
}


BOOL ArtStudioViewer_SetScreen(const char *pScreenFilename)
{
	ArtStudioViewer_Free();

	/* attempt to load the screen */
	if (LoadFile(pScreenFilename, &pLoadedImage, &LoadedImageLength))
	{	
		pDecompressedImage = malloc(16384);

		if (pScreenFilename!=NULL)
		{
			unsigned long Length = LoadedImageLength;
			unsigned long DataOffset = 0;

			/* has a amsdos header? */
			if (AMSDOS_HasAmsdosHeader(pLoadedImage))
			{
				Length-=AMSDOS_HEADER_LENGTH;
				DataOffset+=AMSDOS_HEADER_LENGTH;
			}

			if (Length>16384)
				Length = 16384;

			memset(pDecompressedImage, 0, 16384);

			if (Length==16384)
			{
				memcpy(pDecompressedImage, pLoadedImage+DataOffset, Length);
			}
			else
			{
				DecompressData(pLoadedImage+DataOffset, Length, pDecompressedImage, 16384);
			}
		}

		/* the active palette will be the default palette if no palette file
		is found, or there is a problem loading the palette */
		pActivePalette = DefaultPalette;

		if (pLoadedImage!=NULL)
		{
			/* screen loaded, now attempt to load the palette  */

			/* change the filename extension, and try to load this file */
			char *pPaletteFilename = ChangeExtension(pScreenFilename,"pal");

			if (pPaletteFilename!=NULL)
			{
				if (LoadFile(pPaletteFilename, &pLoadedPalette, &LoadedPaletteLength))
				{
					if (pLoadedPalette!=NULL)
					{
						/* the active palette is the loaded palette */

						if (AMSDOS_HasAmsdosHeader(pLoadedPalette))
						{
							pActivePalette = pLoadedPalette + AMSDOS_HEADER_LENGTH;
						}
						else
						{
							pActivePalette = pLoadedPalette;
						}

					}		
				}
			}
		}

		CurrentMode = ArtStudioViewer_GetMode(pActivePalette);
		
		return TRUE;
	}

	return FALSE;
}



void	ArtStudio_InitialiseLUT(const char *pPalette, int *OutputLUT, const int FrameIndex)
{
	int i;
	unsigned long Offset = 3+FrameIndex;

	for (i=0; i<17; i++)
	{
		OutputLUT[i] = pPalette[Offset];

		Offset+=12;
	}
}


/* initialise translation */
void	InitialisePixelTranslationTable(const int Mode, PixelGroup *pTranslationTable)
{
	switch (Mode)
	{
		/* mode 0 */
		case 0:
		{
			int i;

			for (i=0; i<256; i++)
			{
				int InkIndex[2];

				InkIndex[0] = ((i & 0x02)<<2) | ((i & 0x020)>>3) | ((i & 0x08)>>2) | ((i & 0x080)>>7);

				InkIndex[1] = ((i & 0x01)<<3) | ((i & 0x010)>>2) | ((i & 0x04)>>1) | ((i & 0x040)>>6);

				pTranslationTable[i].InkIndexs[0] = InkIndex[0];
				pTranslationTable[i].InkIndexs[1] = InkIndex[0];
				pTranslationTable[i].InkIndexs[2] = InkIndex[0];
				pTranslationTable[i].InkIndexs[3] = InkIndex[0];
				pTranslationTable[i].InkIndexs[4] = InkIndex[1];
				pTranslationTable[i].InkIndexs[5] = InkIndex[1];
				pTranslationTable[i].InkIndexs[6] = InkIndex[1];
				pTranslationTable[i].InkIndexs[7] = InkIndex[1];
			}
		}
		break;

		/* mode 1 */
		case 1:
		{
			int i;

			for (i=0; i<256; i++)
			{
				int InkIndex[4];

				InkIndex[0] = ((i & 0x08)>>2) | ((i & 0x080)>>7);
				InkIndex[1] = ((i & 0x04)>>1) | ((i & 0x040)>>6);
				InkIndex[2] = ((i & 0x02)>>0) | ((i & 0x020)>>5);
				InkIndex[3] = ((i & 0x01)<<1) | ((i & 0x010)>>4);

				pTranslationTable[i].InkIndexs[0] = InkIndex[0];
				pTranslationTable[i].InkIndexs[1] = InkIndex[0];
				pTranslationTable[i].InkIndexs[2] = InkIndex[1];
				pTranslationTable[i].InkIndexs[3] = InkIndex[1];
				pTranslationTable[i].InkIndexs[4] = InkIndex[2];
				pTranslationTable[i].InkIndexs[5] = InkIndex[2];
				pTranslationTable[i].InkIndexs[6] = InkIndex[3];
				pTranslationTable[i].InkIndexs[7] = InkIndex[3];
			}
		}
		break;

		/* mode 2 */
		case 2:
		{
			int i;

			for (i=0; i<256; i++)
			{
				pTranslationTable[i].InkIndexs[0] = (i&0x080);
				pTranslationTable[i].InkIndexs[1] = (i&0x040);
				pTranslationTable[i].InkIndexs[2] = (i&0x020);
				pTranslationTable[i].InkIndexs[3] = (i&0x010);
				pTranslationTable[i].InkIndexs[4] = (i&0x008);
				pTranslationTable[i].InkIndexs[5] = (i&0x004);
				pTranslationTable[i].InkIndexs[6] = (i&0x002);
				pTranslationTable[i].InkIndexs[7] = (i&0x001);
			}
		}
		break;
	}
}

/* set palette of image */
void	SetImagePalette(struct Image *image, const char *pPalette, int FrameIndex)
{
	int LUT[17];
	int i;

	ArtStudio_InitialiseLUT(pPalette, LUT, FrameIndex);
	
	for (i=0; i<17; i++)
	{
		char Red,Green,Blue;
		int CPCColourIndex;

		CPCColourIndex = LUT[i]&0x01f;

		Red = AmstradPalette[(CPCColourIndex*3)+0];
		Green = AmstradPalette[(CPCColourIndex*3)+1];
		Blue = AmstradPalette[(CPCColourIndex*3)+2];

		Image_SetPaletteEntry(image, i, Red, Green, Blue);
	}

}


BOOL	ArtStudioViewer_CanSetMode(void)
{
	return (pActivePalette==&DefaultPalette);
}

/* set the view mode only if the default palette */
void	ArtStudioViewer_SetMode(int Mode)
{
	if (pActivePalette == &DefaultPalette)
	{
		CurrentMode = Mode;
	}
}


void	ArtStudioViewer_PutToImage(struct Image *image)
{
	unsigned long BaseOffset;
	unsigned long LineOffset;
	unsigned long CRTCCharsWidth = 40;
	unsigned long CRTCCharsTall = 25;
	int y;
	int h;

	PixelGroup PixelTranslation[256];

	BaseOffset = 0;
	LineOffset = 0;

	InitialisePixelTranslationTable(CurrentMode,PixelTranslation);

	SetImagePalette(image, pActivePalette, 0);

	h = 0;
	for (y=0; y<(CRTCCharsTall<<3); y++)
	{
		int x;
		unsigned char *pPixelData;

		pPixelData = Image_GetLineStart(image, y);

		for (x=0; x<(CRTCCharsWidth<<1); x++)
		{
			int i;
			unsigned char PixelByte;
			PixelGroup *pPixelGroup;

			PixelByte = pDecompressedImage[LineOffset+x];
			pPixelGroup = &PixelTranslation[PixelByte&0x0ff];

			for (i=0; i<8; i++)
			{
				pPixelData[0] = pPixelGroup->InkIndexs[i];
				pPixelData++;
			}
		}

		/* update offset */
		h++;

		if (h==8)
		{
			h = 0;
			BaseOffset += (CRTCCharsWidth<<1);
			LineOffset = BaseOffset;
		}
		else
		{
			LineOffset += 0x0800;
		}
	}
}